# -*- coding: utf-8 -*-
"""
直播间统计计算模块
"""

from typing import List, Dict
from collections import Counter
import re


class LiveStats:
    """直播间统计工具类"""
    
    @staticmethod
    def extract_keywords(messages: List[str], top_n: int = 10) -> List[tuple]:
        """
        提取弹幕关键词
        
        Args:
            messages: 弹幕内容列表
            top_n: 返回前N个关键词
        
        Returns:
            [(关键词, 次数), ...]
        """
        # 简单分词（按空格和标点）
        words = []
        for msg in messages:
            # 移除表情符号和特殊字符
            clean = re.sub(r'[^\u4e00-\u9fa5a-zA-Z0-9]', ' ', msg)
            # 分割
            parts = clean.split()
            for part in parts:
                if len(part) >= 2:  # 至少2个字符
                    words.append(part)
        
        # 统计频次
        counter = Counter(words)
        return counter.most_common(top_n)
    
    @staticmethod
    def calculate_gift_value_rmb(douyin_coins: int) -> float:
        """
        将抖币转换为人民币
        抖币与人民币比例约为 10:1
        
        Args:
            douyin_coins: 抖币数量
        
        Returns:
            人民币金额
        """
        return douyin_coins / 10.0
    
    @staticmethod
    def format_gift_value(douyin_coins: int) -> str:
        """格式化礼物价值显示"""
        if douyin_coins >= 10000:
            rmb = douyin_coins / 10.0
            return f"¥{rmb:.0f}"
        elif douyin_coins >= 100:
            return f"{douyin_coins}抖币"
        else:
            return f"{douyin_coins}抖币"
    
    @staticmethod
    def format_number(num: int) -> str:
        """格式化数字显示"""
        if num >= 10000:
            return f"{num/10000:.1f}万"
        elif num >= 1000:
            return f"{num/1000:.1f}千"
        return str(num)
    
    @staticmethod
    def calculate_heat_level(chat_freq: float, online: int) -> tuple:
        """
        计算直播间热度等级
        
        Args:
            chat_freq: 弹幕频率（条/分钟）
            online: 在线人数
        
        Returns:
            (等级名称, 等级值0-100)
        """
        # 简单热度算法
        score = min(100, int(chat_freq * 2 + online / 10))
        
        if score >= 80:
            return ("🔥 火爆", score)
        elif score >= 60:
            return ("🌟 热门", score)
        elif score >= 40:
            return ("👍 活跃", score)
        elif score >= 20:
            return ("😊 一般", score)
        else:
            return ("😴 冷清", score)
