# -*- coding: utf-8 -*-
"""
直播场次管理
每场直播独立统计，支持历史查看
"""

import json
import sqlite3
import threading
from typing import Dict, List, Optional
from dataclasses import dataclass, field, asdict
from datetime import datetime
from pathlib import Path
import uuid


@dataclass
class SessionSummary:
    """场次汇总数据"""
    duration_seconds: int = 0
    peak_online: int = 0
    total_entrance: int = 0
    total_follow: int = 0
    total_chat: int = 0
    total_like: int = 0
    total_gift_value: int = 0
    unique_users: int = 0


@dataclass
class LiveSession:
    """直播场次"""
    session_id: str
    room_id: str
    started_at: datetime
    ended_at: Optional[datetime] = None
    status: str = "running"  # running, ended
    summary: SessionSummary = field(default_factory=SessionSummary)
    
    def to_dict(self) -> dict:
        """转换为字典"""
        return {
            'session_id': self.session_id,
            'room_id': self.room_id,
            'started_at': self.started_at.isoformat(),
            'ended_at': self.ended_at.isoformat() if self.ended_at else None,
            'status': self.status,
            'summary': asdict(self.summary),
        }
    
    @classmethod
    def from_dict(cls, data: dict) -> 'LiveSession':
        """从字典创建"""
        return cls(
            session_id=data['session_id'],
            room_id=data['room_id'],
            started_at=datetime.fromisoformat(data['started_at']),
            ended_at=datetime.fromisoformat(data['ended_at']) if data.get('ended_at') else None,
            status=data.get('status', 'ended'),
            summary=SessionSummary(**data.get('summary', {})),
        )


class SessionManager:
    """场次管理器"""
    
    def __init__(self, data_dir: Path = None):
        if data_dir is None:
            data_dir = Path.home() / ".douyin_live_assistant" / "analytics"
        
        self._data_dir = data_dir
        self._data_dir.mkdir(parents=True, exist_ok=True)
        
        self._db_path = self._data_dir / "sessions.db"
        self._lock = threading.Lock()
        
        self._current_session: Optional[LiveSession] = None
        
        self._init_db()
    
    def _init_db(self):
        """初始化数据库"""
        with sqlite3.connect(self._db_path) as conn:
            conn.execute('''
                CREATE TABLE IF NOT EXISTS sessions (
                    session_id TEXT PRIMARY KEY,
                    room_id TEXT,
                    started_at TEXT,
                    ended_at TEXT,
                    status TEXT,
                    summary_json TEXT
                )
            ''')
            
            conn.execute('''
                CREATE TABLE IF NOT EXISTS user_stats (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    session_id TEXT,
                    user_id TEXT,
                    nickname TEXT,
                    chat_count INTEGER DEFAULT 0,
                    gift_total INTEGER DEFAULT 0,
                    gift_count INTEGER DEFAULT 0,
                    like_count INTEGER DEFAULT 0,
                    entrance_count INTEGER DEFAULT 0,
                    first_seen TEXT,
                    last_seen TEXT,
                    FOREIGN KEY (session_id) REFERENCES sessions(session_id)
                )
            ''')
            
            conn.execute('''
                CREATE TABLE IF NOT EXISTS gift_records (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    session_id TEXT,
                    user_id TEXT,
                    nickname TEXT,
                    gift_id TEXT,
                    gift_name TEXT,
                    gift_count INTEGER,
                    gift_price INTEGER,
                    timestamp TEXT,
                    FOREIGN KEY (session_id) REFERENCES sessions(session_id)
                )
            ''')
            
            conn.execute('CREATE INDEX IF NOT EXISTS idx_session_id ON user_stats(session_id)')
            conn.execute('CREATE INDEX IF NOT EXISTS idx_gift_session ON gift_records(session_id)')
            conn.commit()
    
    def start_session(self, room_id: str) -> LiveSession:
        """开始新场次"""
        with self._lock:
            # 结束之前的场次
            if self._current_session and self._current_session.status == "running":
                self.end_session()
            
            session = LiveSession(
                session_id=str(uuid.uuid4())[:8],
                room_id=room_id,
                started_at=datetime.now(),
            )
            
            self._current_session = session
            self._save_session(session)
            
            print(f"[Session] 新场次开始: {session.session_id}")
            return session
    
    def end_session(self) -> Optional[LiveSession]:
        """结束当前场次"""
        with self._lock:
            if not self._current_session:
                return None
            
            self._current_session.ended_at = datetime.now()
            self._current_session.status = "ended"
            
            # 计算时长
            if self._current_session.started_at:
                delta = self._current_session.ended_at - self._current_session.started_at
                self._current_session.summary.duration_seconds = int(delta.total_seconds())
            
            self._save_session(self._current_session)
            
            print(f"[Session] 场次结束: {self._current_session.session_id}")
            
            ended = self._current_session
            self._current_session = None
            return ended
    
    def _save_session(self, session: LiveSession):
        """保存场次到数据库"""
        with sqlite3.connect(self._db_path) as conn:
            conn.execute('''
                INSERT OR REPLACE INTO sessions 
                (session_id, room_id, started_at, ended_at, status, summary_json)
                VALUES (?, ?, ?, ?, ?, ?)
            ''', (
                session.session_id,
                session.room_id,
                session.started_at.isoformat(),
                session.ended_at.isoformat() if session.ended_at else None,
                session.status,
                json.dumps(asdict(session.summary)),
            ))
            conn.commit()
    
    def update_summary(self, summary: SessionSummary):
        """更新当前场次汇总"""
        with self._lock:
            if self._current_session:
                self._current_session.summary = summary
    
    def save_user_stats(self, session_id: str, user_id: str, nickname: str, stats: dict):
        """保存用户统计"""
        with sqlite3.connect(self._db_path) as conn:
            conn.execute('''
                INSERT OR REPLACE INTO user_stats
                (session_id, user_id, nickname, chat_count, gift_total, gift_count, 
                 like_count, entrance_count, first_seen, last_seen)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ''', (
                session_id,
                user_id,
                nickname,
                stats.get('chat_count', 0),
                stats.get('gift_total', 0),
                stats.get('gift_count', 0),
                stats.get('like_count', 0),
                stats.get('entrance_count', 0),
                stats.get('first_seen', ''),
                stats.get('last_seen', ''),
            ))
            conn.commit()
    
    def save_gift_record(self, session_id: str, user_id: str, nickname: str,
                         gift_id: str, gift_name: str, count: int, price: int):
        """保存礼物记录"""
        with sqlite3.connect(self._db_path) as conn:
            conn.execute('''
                INSERT INTO gift_records
                (session_id, user_id, nickname, gift_id, gift_name, gift_count, gift_price, timestamp)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ''', (
                session_id,
                user_id,
                nickname,
                gift_id,
                gift_name,
                count,
                price,
                datetime.now().isoformat(),
            ))
            conn.commit()
    
    @property
    def current_session(self) -> Optional[LiveSession]:
        """获取当前场次"""
        return self._current_session
    
    def get_session(self, session_id: str) -> Optional[LiveSession]:
        """获取指定场次"""
        with sqlite3.connect(self._db_path) as conn:
            cursor = conn.execute(
                'SELECT * FROM sessions WHERE session_id = ?',
                (session_id,)
            )
            row = cursor.fetchone()
            if row:
                return self._row_to_session(row)
        return None
    
    def get_recent_sessions(self, limit: int = 10) -> List[LiveSession]:
        """获取最近的场次列表"""
        sessions = []
        with sqlite3.connect(self._db_path) as conn:
            cursor = conn.execute(
                'SELECT * FROM sessions ORDER BY started_at DESC LIMIT ?',
                (limit,)
            )
            for row in cursor:
                sessions.append(self._row_to_session(row))
        return sessions
    
    def get_session_user_stats(self, session_id: str) -> List[dict]:
        """获取场次的用户统计"""
        stats = []
        with sqlite3.connect(self._db_path) as conn:
            cursor = conn.execute(
                'SELECT * FROM user_stats WHERE session_id = ? ORDER BY gift_total DESC',
                (session_id,)
            )
            for row in cursor:
                stats.append({
                    'user_id': row[2],
                    'nickname': row[3],
                    'chat_count': row[4],
                    'gift_total': row[5],
                    'gift_count': row[6],
                    'like_count': row[7],
                    'entrance_count': row[8],
                })
        return stats
    
    def get_session_gifts(self, session_id: str) -> List[dict]:
        """获取场次的礼物记录"""
        gifts = []
        with sqlite3.connect(self._db_path) as conn:
            cursor = conn.execute(
                'SELECT * FROM gift_records WHERE session_id = ? ORDER BY timestamp DESC',
                (session_id,)
            )
            for row in cursor:
                gifts.append({
                    'user_id': row[2],
                    'nickname': row[3],
                    'gift_id': row[4],
                    'gift_name': row[5],
                    'gift_count': row[6],
                    'gift_price': row[7],
                    'timestamp': row[8],
                })
        return gifts
    
    def _row_to_session(self, row) -> LiveSession:
        """数据库行转场次对象"""
        summary_data = json.loads(row[5]) if row[5] else {}
        return LiveSession(
            session_id=row[0],
            room_id=row[1],
            started_at=datetime.fromisoformat(row[2]),
            ended_at=datetime.fromisoformat(row[3]) if row[3] else None,
            status=row[4],
            summary=SessionSummary(**summary_data),
        )
    
    def delete_session(self, session_id: str):
        """删除场次及相关数据"""
        with sqlite3.connect(self._db_path) as conn:
            conn.execute('DELETE FROM gift_records WHERE session_id = ?', (session_id,))
            conn.execute('DELETE FROM user_stats WHERE session_id = ?', (session_id,))
            conn.execute('DELETE FROM sessions WHERE session_id = ?', (session_id,))
            conn.commit()
