# -*- coding: utf-8 -*-
"""
直播间数据采集器
实时收集直播间各类事件数据
"""

from typing import Dict, List, Optional, Callable
from dataclasses import dataclass, field
from datetime import datetime, timedelta
from collections import defaultdict
import threading
import time

from src.utils.events import (
    EventType, Event, User, Gift,
    ChatEvent, GiftEvent, EntranceEvent, FollowEvent, LikeEvent, RoomStatsEvent
)
from .session import SessionManager, SessionSummary


@dataclass
class UserStats:
    """用户统计数据"""
    user: User
    chat_count: int = 0
    gift_total: int = 0  # 抖币总额
    gift_count: int = 0  # 礼物次数
    like_count: int = 0
    entrance_count: int = 0
    first_seen: datetime = field(default_factory=datetime.now)
    last_seen: datetime = field(default_factory=datetime.now)
    
    @property
    def total_score(self) -> int:
        """综合互动分数"""
        return self.chat_count * 2 + self.gift_total // 10 + self.like_count + self.entrance_count


@dataclass
class GiftRecord:
    """礼物记录"""
    user: User
    gift: Gift
    timestamp: datetime = field(default_factory=datetime.now)
    
    @property
    def value(self) -> int:
        """礼物价值（抖币）"""
        return self.gift.price * self.gift.count


@dataclass
class TimeSlotStats:
    """时段统计"""
    slot_start: datetime
    chat_count: int = 0
    gift_value: int = 0
    entrance_count: int = 0
    follow_count: int = 0
    like_count: int = 0
    online_peak: int = 0


class LiveDataCollector:
    """直播间数据采集器"""
    
    def __init__(self, session_manager: SessionManager = None):
        self._lock = threading.Lock()
        self._started_at: Optional[datetime] = None
        self._is_collecting = False
        self._room_id: str = ""
        
        # 场次管理器
        self._session_manager = session_manager or SessionManager()
        
        # 用户数据
        self._user_stats: Dict[str, UserStats] = {}
        
        # 弹幕记录
        self._chat_messages: List[tuple] = []  # (timestamp, user_id, content)
        
        # 礼物记录
        self._gift_records: List[GiftRecord] = []
        
        # 时段统计（每分钟）
        self._time_slots: Dict[str, TimeSlotStats] = {}
        
        # 实时计数器
        self._total_entrance = 0
        self._total_follow = 0
        self._total_like = 0
        self._total_chat = 0
        self._total_gift_value = 0
        
        # 在线人数历史
        self._online_history: List[tuple] = []  # (timestamp, count)
        self._current_online = 0
        self._peak_online = 0
        
        # 回调函数
        self._on_update: Optional[Callable] = None
    
    def start(self, room_id: str = ""):
        """开始采集"""
        with self._lock:
            self._started_at = datetime.now()
            self._is_collecting = True
            self._room_id = room_id
            self._reset_data()
            
            # 创建新场次
            self._session_manager.start_session(room_id)
    
    def stop(self):
        """停止采集"""
        with self._lock:
            self._is_collecting = False
            
            # 更新场次汇总并保存用户数据
            self._save_session_data()
    
    def reset(self):
        """重置数据"""
        with self._lock:
            self._reset_data()
    
    def _reset_data(self):
        """内部重置"""
        self._user_stats.clear()
        self._chat_messages.clear()
        self._gift_records.clear()
        self._time_slots.clear()
        self._online_history.clear()
        self._total_entrance = 0
        self._total_follow = 0
        self._total_like = 0
        self._total_chat = 0
        self._total_gift_value = 0
        self._current_online = 0
        self._peak_online = 0
    
    def _save_session_data(self):
        """保存场次数据到数据库"""
        session = self._session_manager.current_session
        if not session:
            return
        
        # 更新汇总数据
        summary = SessionSummary(
            duration_seconds=self.duration_seconds,
            peak_online=self._peak_online,
            total_entrance=self._total_entrance,
            total_follow=self._total_follow,
            total_chat=self._total_chat,
            total_like=self._total_like,
            total_gift_value=self._total_gift_value,
            unique_users=len(self._user_stats),
        )
        self._session_manager.update_summary(summary)
        
        # 保存用户统计
        for user_id, stats in self._user_stats.items():
            self._session_manager.save_user_stats(
                session.session_id,
                user_id,
                stats.user.nickname,
                {
                    'chat_count': stats.chat_count,
                    'gift_total': stats.gift_total,
                    'gift_count': stats.gift_count,
                    'like_count': stats.like_count,
                    'entrance_count': stats.entrance_count,
                    'first_seen': stats.first_seen.isoformat(),
                    'last_seen': stats.last_seen.isoformat(),
                }
            )
        
        # 结束场次
        self._session_manager.end_session()
        print(f"[Collector] 场次数据已保存")
    
    def set_update_callback(self, callback: Callable):
        """设置数据更新回调"""
        self._on_update = callback
    
    def _notify_update(self):
        """通知数据更新"""
        if self._on_update:
            try:
                self._on_update()
            except:
                pass
    
    def _get_or_create_user(self, user: User) -> UserStats:
        """获取或创建用户统计"""
        if user.id not in self._user_stats:
            self._user_stats[user.id] = UserStats(user=user)
        stats = self._user_stats[user.id]
        stats.last_seen = datetime.now()
        return stats
    
    def _get_time_slot(self, dt: datetime = None) -> TimeSlotStats:
        """获取时段统计（按分钟）"""
        if dt is None:
            dt = datetime.now()
        slot_key = dt.strftime("%Y%m%d%H%M")
        if slot_key not in self._time_slots:
            slot_start = dt.replace(second=0, microsecond=0)
            self._time_slots[slot_key] = TimeSlotStats(slot_start=slot_start)
        return self._time_slots[slot_key]
    
    # ========== 事件处理 ==========
    
    def on_chat(self, event: ChatEvent):
        """处理弹幕事件"""
        if not self._is_collecting:
            return
        
        with self._lock:
            user_stats = self._get_or_create_user(event.user)
            user_stats.chat_count += 1
            
            self._chat_messages.append((
                event.timestamp,
                event.user.id,
                event.content
            ))
            
            self._total_chat += 1
            self._get_time_slot().chat_count += 1
        
        self._notify_update()
    
    def on_gift(self, event: GiftEvent):
        """处理礼物事件"""
        if not self._is_collecting or not event.gift:
            return
        
        with self._lock:
            user_stats = self._get_or_create_user(event.user)
            value = event.gift.price * event.gift.count
            user_stats.gift_total += value
            user_stats.gift_count += 1
            
            record = GiftRecord(
                user=event.user,
                gift=event.gift,
                timestamp=event.timestamp
            )
            self._gift_records.append(record)
            
            self._total_gift_value += value
            self._get_time_slot().gift_value += value
        
        self._notify_update()
    
    def on_entrance(self, event: EntranceEvent):
        """处理进场事件"""
        if not self._is_collecting:
            return
        
        with self._lock:
            user_stats = self._get_or_create_user(event.user)
            user_stats.entrance_count += 1
            
            self._total_entrance += 1
            self._get_time_slot().entrance_count += 1
        
        self._notify_update()
    
    def on_follow(self, event: FollowEvent):
        """处理关注事件"""
        if not self._is_collecting:
            return
        
        with self._lock:
            self._get_or_create_user(event.user)
            self._total_follow += 1
            self._get_time_slot().follow_count += 1
        
        self._notify_update()
    
    def on_like(self, event: LikeEvent):
        """处理点赞事件"""
        if not self._is_collecting:
            return
        
        with self._lock:
            user_stats = self._get_or_create_user(event.user)
            user_stats.like_count += event.count
            
            self._total_like += event.count
            self._get_time_slot().like_count += event.count
        
        self._notify_update()
    
    def on_room_stats(self, event: RoomStatsEvent):
        """处理直播间统计事件"""
        if not self._is_collecting:
            return
        
        with self._lock:
            self._current_online = event.online_count
            if event.online_count > self._peak_online:
                self._peak_online = event.online_count
            
            self._online_history.append((datetime.now(), event.online_count))
            
            # 保留最近60分钟的数据
            if len(self._online_history) > 3600:
                self._online_history = self._online_history[-3600:]
            
            slot = self._get_time_slot()
            if event.online_count > slot.online_peak:
                slot.online_peak = event.online_count
        
        self._notify_update()
    
    # ========== 数据获取 ==========
    
    @property
    def is_collecting(self) -> bool:
        return self._is_collecting
    
    @property
    def started_at(self) -> Optional[datetime]:
        return self._started_at
    
    @property
    def duration_seconds(self) -> int:
        """采集时长（秒）"""
        if not self._started_at:
            return 0
        return int((datetime.now() - self._started_at).total_seconds())
    
    @property
    def duration_str(self) -> str:
        """采集时长字符串"""
        secs = self.duration_seconds
        hours = secs // 3600
        mins = (secs % 3600) // 60
        secs = secs % 60
        if hours > 0:
            return f"{hours}时{mins}分{secs}秒"
        elif mins > 0:
            return f"{mins}分{secs}秒"
        return f"{secs}秒"
    
    def get_summary(self) -> dict:
        """获取汇总数据"""
        with self._lock:
            return {
                'duration': self.duration_str,
                'duration_seconds': self.duration_seconds,
                'current_online': self._current_online,
                'peak_online': self._peak_online,
                'total_entrance': self._total_entrance,
                'total_follow': self._total_follow,
                'total_like': self._total_like,
                'total_chat': self._total_chat,
                'total_gift_value': self._total_gift_value,
                'unique_users': len(self._user_stats),
            }
    
    def get_chat_rank(self, limit: int = 10) -> List[UserStats]:
        """获取发言排行"""
        with self._lock:
            users = list(self._user_stats.values())
            users.sort(key=lambda x: x.chat_count, reverse=True)
            return users[:limit]
    
    def get_gift_rank(self, limit: int = 10) -> List[UserStats]:
        """获取礼物排行"""
        with self._lock:
            users = [u for u in self._user_stats.values() if u.gift_total > 0]
            users.sort(key=lambda x: x.gift_total, reverse=True)
            return users[:limit]
    
    def get_interaction_rank(self, limit: int = 10) -> List[UserStats]:
        """获取互动排行"""
        with self._lock:
            users = list(self._user_stats.values())
            users.sort(key=lambda x: x.total_score, reverse=True)
            return users[:limit]
    
    def get_online_history(self, minutes: int = 30) -> List[tuple]:
        """获取在线人数历史"""
        with self._lock:
            cutoff = datetime.now() - timedelta(minutes=minutes)
            return [(t, c) for t, c in self._online_history if t >= cutoff]
    
    def get_recent_gifts(self, limit: int = 20) -> List[GiftRecord]:
        """获取最近礼物记录"""
        with self._lock:
            return self._gift_records[-limit:][::-1]
    
    def get_chat_frequency(self, minutes: int = 5) -> float:
        """获取弹幕频率（条/分钟）"""
        with self._lock:
            cutoff = datetime.now() - timedelta(minutes=minutes)
            recent = [m for m in self._chat_messages if m[0] >= cutoff]
            return len(recent) / minutes if minutes > 0 else 0
