# -*- coding: utf-8 -*-
import logging
import json
from typing import Generator, Optional

# 我们尽量不引入庞大的依赖，如果用户没有安装openai库，我们用requests作为备选？
# 为了最好的兼容性和流式支持，建议使用 openai 库。
# 如果用户打包，我们需要确保 openai 库被包含。

try:
    from openai import OpenAI, APIError
except ImportError:
    OpenAI = None
    print("Warning: openai library not found. AI features will be disabled.")

logger = logging.getLogger(__name__)

class LLMClient:
    def __init__(self, api_key: str, base_url: str = "https://api.deepseek.com", model: str = "deepseek-chat", system_prompt: str = ""):
        self.api_key = api_key
        self.base_url = base_url
        self.model = model
        self.system_prompt = system_prompt
        self.client: Optional[OpenAI] = None
        self.history = []  # 对话历史
        self.max_history = 10 # 保留最近10轮对话
        
        if OpenAI and api_key:
            try:
                self.client = OpenAI(api_key=api_key, base_url=base_url)
                logger.info(f"LLM Client initialized: {base_url} | {model}")
            except Exception as e:
                logger.error(f"Failed to initialize LLM Client: {e}")

    def update_config(self, api_key: str, base_url: str, model: str, system_prompt: str):
        """动态更新配置"""
        self.api_key = api_key
        self.base_url = base_url
        self.model = model
        self.system_prompt = system_prompt
        self.history = [] # 清空历史
        
        if OpenAI and api_key:
            self.client = OpenAI(api_key=api_key, base_url=base_url)

    def chat(self, user_input: str, stream: bool = False) -> str:
        """
        发送消息并获取回复
        :param user_input: 用户输入
        :param stream: 是否流式返回 (暂未实现流式处理逻辑，保留接口)
        :return: AI回复文本
        """
        if not self.api_key:
            return "错误：AI未配置API Key"

        # 构建消息列表
        messages = [{"role": "system", "content": self.system_prompt}]
        messages.extend(self.history)
        messages.append({"role": "user", "content": user_input})

        # 优先使用 OpenAI 客户端
        if self.client:
            try:
                response = self.client.chat.completions.create(
                    model=self.model,
                    messages=messages,
                    stream=False,
                    temperature=1.3
                )
                reply = response.choices[0].message.content.strip()
                self._update_history(user_input, reply)
                return reply
            except Exception as e:
                logger.warning(f"OpenAI客户端调用失败，尝试requests备用: {e}")
        
        # 备用方案：使用 requests 直接调用 API
        try:
            import requests
            headers = {
                "Authorization": f"Bearer {self.api_key}",
                "Content-Type": "application/json"
            }
            data = {
                "model": self.model,
                "messages": messages,
                "temperature": 1.3
            }
            url = f"{self.base_url}/chat/completions"
            response = requests.post(url, headers=headers, json=data, timeout=30)
            
            if response.status_code == 200:
                result = response.json()
                reply = result["choices"][0]["message"]["content"].strip()
                self._update_history(user_input, reply)
                return reply
            else:
                return f"AI请求失败: HTTP {response.status_code}"
        except Exception as e:
            logger.error(f"LLM Chat Error: {e}")
            return f"AI思考短路了: {str(e)}"
    
    def _update_history(self, user_input: str, reply: str):
        """更新对话历史"""
        self.history.append({"role": "user", "content": user_input})
        self.history.append({"role": "assistant", "content": reply})
        if len(self.history) > self.max_history * 2:
            self.history = self.history[-self.max_history * 2:]

    def clear_history(self):
        self.history = []
    
    def test_connection(self) -> tuple[bool, str]:
        """
        测试 API 连接是否正常 (使用 requests，带超时)
        :return: (是否成功, 消息)
        """
        if not self.api_key:
            return False, "未配置 API Key"
        
        try:
            import requests
            headers = {
                "Authorization": f"Bearer {self.api_key}",
                "Content-Type": "application/json"
            }
            data = {
                "model": self.model,
                "messages": [{"role": "user", "content": "hi"}],
                "max_tokens": 10
            }
            url = f"{self.base_url}/chat/completions"
            response = requests.post(url, headers=headers, json=data, timeout=10)
            
            if response.status_code == 200:
                result = response.json()
                reply = result["choices"][0]["message"]["content"].strip()
                return True, f"连接成功！AI: {reply}"
            else:
                return False, f"HTTP {response.status_code}: {response.text[:100]}"
        except requests.Timeout:
            return False, "连接超时 (10秒)"
        except Exception as e:
            return False, f"连接失败: {str(e)}"
