# -*- coding: utf-8 -*-
"""
更新弹窗 UI（Flet 版本）
显示更新信息、下载进度、安装确认

使用方法:
    from gui.update_dialog import show_update_dialog
    show_update_dialog(page, update_info, on_confirm, on_skip)
"""

import flet as ft
from typing import Callable, Optional
from dataclasses import dataclass


@dataclass
class UpdateDialogInfo:
    """更新弹窗信息"""
    version: str = ""
    changelog: str = ""
    total_size: int = 0
    force_update: bool = False
    current_version: str = ""


def format_size(size_bytes: int) -> str:
    """格式化文件大小"""
    if size_bytes < 1024:
        return f"{size_bytes} B"
    elif size_bytes < 1024 * 1024:
        return f"{size_bytes / 1024:.1f} KB"
    elif size_bytes < 1024 * 1024 * 1024:
        return f"{size_bytes / 1024 / 1024:.1f} MB"
    else:
        return f"{size_bytes / 1024 / 1024 / 1024:.1f} GB"


class UpdateDialog:
    """更新弹窗"""
    
    def __init__(self, page: ft.Page):
        self.page = page
        self.dialog: Optional[ft.AlertDialog] = None
        self.progress_bar: Optional[ft.ProgressBar] = None
        self.progress_text: Optional[ft.Text] = None
        self.status_text: Optional[ft.Text] = None
        self.confirm_button: Optional[ft.ElevatedButton] = None
        self.skip_button: Optional[ft.TextButton] = None
        
        self._on_confirm: Optional[Callable] = None
        self._on_skip: Optional[Callable] = None
        self._on_cancel: Optional[Callable] = None
    
    def show(self, 
             info: UpdateDialogInfo,
             on_confirm: Callable = None,
             on_skip: Callable = None,
             on_cancel: Callable = None):
        """显示更新弹窗"""
        self._on_confirm = on_confirm
        self._on_skip = on_skip
        self._on_cancel = on_cancel
        
        # 构建更新日志内容
        changelog_lines = info.changelog.split('\n') if info.changelog else ["暂无更新说明"]
        changelog_items = []
        for line in changelog_lines[:10]:  # 最多显示10行
            if line.strip():
                changelog_items.append(
                    ft.Text(f"• {line.strip()}", size=13, color=ft.Colors.GREY_700)
                )
        
        # 进度条
        self.progress_bar = ft.ProgressBar(
            width=400,
            value=0,
            visible=False,
            color=ft.Colors.BLUE_400,
            bgcolor=ft.Colors.GREY_200,
        )
        
        self.progress_text = ft.Text(
            "准备下载...",
            size=12,
            color=ft.Colors.GREY_600,
            visible=False,
        )
        
        self.status_text = ft.Text(
            "",
            size=12,
            color=ft.Colors.GREEN_600,
            visible=False,
        )
        
        # 确认按钮
        self.confirm_button = ft.ElevatedButton(
            "立即更新",
            icon=ft.Icons.SYSTEM_UPDATE_ALT,
            on_click=self._handle_confirm,
            style=ft.ButtonStyle(
                color=ft.Colors.WHITE,
                bgcolor=ft.Colors.BLUE_600,
            ),
        )
        
        # 跳过按钮（非强制更新时显示）
        self.skip_button = ft.TextButton(
            "稍后提醒",
            on_click=self._handle_skip,
            visible=not info.force_update,
        )
        
        # 构建弹窗内容
        content = ft.Container(
            content=ft.Column([
                # 标题区域
                ft.Container(
                    content=ft.Row([
                        ft.Icon(ft.Icons.SYSTEM_UPDATE, size=32, color=ft.Colors.BLUE_600),
                        ft.Column([
                            ft.Text("发现新版本", size=18, weight=ft.FontWeight.BOLD),
                            ft.Text(
                                f"v{info.current_version} → v{info.version}",
                                size=13,
                                color=ft.Colors.GREY_600
                            ),
                        ], spacing=2),
                    ], spacing=12),
                    padding=ft.padding.only(bottom=16),
                ),
                
                # 更新说明
                ft.Container(
                    content=ft.Column([
                        ft.Text("更新内容:", size=14, weight=ft.FontWeight.W_500),
                        ft.Container(
                            content=ft.Column(changelog_items, spacing=4),
                            bgcolor=ft.Colors.GREY_100,
                            border_radius=8,
                            padding=12,
                            height=120,
                        ),
                    ], spacing=8),
                ),
                
                # 文件大小
                ft.Container(
                    content=ft.Row([
                        ft.Icon(ft.Icons.FOLDER_OUTLINED, size=16, color=ft.Colors.GREY_600),
                        ft.Text(
                            f"更新大小: {format_size(info.total_size)}",
                            size=13,
                            color=ft.Colors.GREY_600
                        ),
                    ], spacing=8),
                    padding=ft.padding.only(top=8),
                ),
                
                # 进度区域
                ft.Container(
                    content=ft.Column([
                        self.progress_bar,
                        self.progress_text,
                        self.status_text,
                    ], spacing=4),
                    padding=ft.padding.only(top=16),
                ),
                
            ], spacing=0, tight=True),
            width=420,
            padding=ft.padding.all(8),
        )
        
        # 创建弹窗
        self.dialog = ft.AlertDialog(
            modal=info.force_update,  # 强制更新时为模态
            title=None,
            content=content,
            actions=[
                self.skip_button,
                self.confirm_button,
            ],
            actions_alignment=ft.MainAxisAlignment.END,
            shape=ft.RoundedRectangleBorder(radius=12),
        )
        
        self.page.dialog = self.dialog
        self.dialog.open = True
        self.page.update()
    
    def update_progress(self, progress: float, status: str = ""):
        """更新下载进度"""
        if self.progress_bar:
            self.progress_bar.visible = True
            self.progress_bar.value = progress / 100
        
        if self.progress_text:
            self.progress_text.visible = True
            self.progress_text.value = f"下载进度: {progress:.1f}%"
        
        if status and self.status_text:
            self.status_text.visible = True
            self.status_text.value = status
        
        # 禁用按钮
        if self.confirm_button:
            self.confirm_button.disabled = True
            self.confirm_button.text = "下载中..."
        
        if self.skip_button:
            self.skip_button.disabled = True
        
        self.page.update()
    
    def show_complete(self, success: bool, message: str = ""):
        """显示完成状态"""
        if self.progress_bar:
            self.progress_bar.value = 1.0 if success else 0
            self.progress_bar.color = ft.Colors.GREEN_400 if success else ft.Colors.RED_400
        
        if self.progress_text:
            self.progress_text.visible = False
        
        if self.status_text:
            self.status_text.visible = True
            self.status_text.value = message or ("更新完成！" if success else "更新失败")
            self.status_text.color = ft.Colors.GREEN_600 if success else ft.Colors.RED_600
        
        if self.confirm_button:
            self.confirm_button.disabled = False
            if success:
                self.confirm_button.text = "立即重启"
                self.confirm_button.icon = ft.Icons.RESTART_ALT
            else:
                self.confirm_button.text = "重试"
                self.confirm_button.icon = ft.Icons.REFRESH
        
        if self.skip_button:
            self.skip_button.disabled = False
            self.skip_button.visible = True
            self.skip_button.text = "稍后重启" if success else "取消"
        
        self.page.update()
    
    def close(self):
        """关闭弹窗"""
        if self.dialog:
            self.dialog.open = False
            self.page.update()
    
    def _handle_confirm(self, e):
        """处理确认按钮"""
        if self._on_confirm:
            self._on_confirm()
    
    def _handle_skip(self, e):
        """处理跳过按钮"""
        self.close()
        if self._on_skip:
            self._on_skip()
    
    def _handle_cancel(self, e):
        """处理取消按钮"""
        self.close()
        if self._on_cancel:
            self._on_cancel()


def show_update_dialog(
    page: ft.Page,
    version: str,
    changelog: str,
    total_size: int,
    current_version: str = "1.0.0",
    force_update: bool = False,
    on_confirm: Callable = None,
    on_skip: Callable = None,
) -> UpdateDialog:
    """
    显示更新弹窗（便捷函数）
    
    Args:
        page: Flet 页面对象
        version: 新版本号
        changelog: 更新日志
        total_size: 更新大小（字节）
        current_version: 当前版本号
        force_update: 是否强制更新
        on_confirm: 确认回调
        on_skip: 跳过回调
    
    Returns:
        UpdateDialog 实例（用于更新进度）
    """
    dialog = UpdateDialog(page)
    info = UpdateDialogInfo(
        version=version,
        changelog=changelog,
        total_size=total_size,
        current_version=current_version,
        force_update=force_update,
    )
    dialog.show(info, on_confirm, on_skip)
    return dialog
