# -*- coding: utf-8 -*-
"""
专业主题配色系统
参考 GitHub / Figma / Linear 设计语言
"""

import flet as ft
from dataclasses import dataclass
from typing import Optional


@dataclass
class ThemeColors:
    """主题颜色定义"""
    # 背景层级
    bg_base: str           # 最底层背景
    bg_surface: str        # 表面/卡片背景
    bg_elevated: str       # 提升层背景
    bg_overlay: str        # 覆盖层背景
    
    # 边框
    border_default: str    # 默认边框
    border_muted: str      # 弱化边框
    border_emphasis: str   # 强调边框
    
    # 文字
    text_primary: str      # 主要文字
    text_secondary: str    # 次要文字
    text_muted: str        # 弱化文字
    text_placeholder: str  # 占位符文字
    
    # 主色调
    accent: str            # 强调色
    accent_emphasis: str   # 强调色加深
    accent_muted: str      # 强调色减淡
    accent_subtle: str     # 强调色极淡（用于背景）
    
    # 语义色
    success: str           # 成功
    success_emphasis: str  # 成功加深
    warning: str           # 警告
    warning_emphasis: str  # 警告加深
    error: str             # 错误
    error_emphasis: str    # 错误加深
    info: str              # 信息
    
    # 交互状态
    hover: str             # 悬停背景
    active: str            # 激活背景
    selected: str          # 选中背景


# 深色主题 - GitHub Dark 风格
DARK_THEME = ThemeColors(
    # 背景层级
    bg_base="#0d1117",
    bg_surface="#161b22",
    bg_elevated="#21262d",
    bg_overlay="#30363d",
    
    # 边框
    border_default="#30363d",
    border_muted="#21262d",
    border_emphasis="#8b949e",
    
    # 文字
    text_primary="#e6edf3",
    text_secondary="#c9d1d9",
    text_muted="#8b949e",
    text_placeholder="#6e7681",
    
    # 主色调 - 蓝色
    accent="#58a6ff",
    accent_emphasis="#1f6feb",
    accent_muted="#388bfd",
    accent_subtle="#0d419d",
    
    # 语义色
    success="#3fb950",
    success_emphasis="#238636",
    warning="#d29922",
    warning_emphasis="#9e6a03",
    error="#f85149",
    error_emphasis="#da3633",
    info="#58a6ff",
    
    # 交互状态
    hover="#21262d",
    active="#30363d",
    selected="#161b22",
)


# 亮色主题 - GitHub Light 风格
LIGHT_THEME = ThemeColors(
    # 背景层级
    bg_base="#ffffff",
    bg_surface="#f6f8fa",
    bg_elevated="#ffffff",
    bg_overlay="#ffffff",
    
    # 边框
    border_default="#d0d7de",
    border_muted="#d8dee4",
    border_emphasis="#1f2328",
    
    # 文字
    text_primary="#1f2328",
    text_secondary="#24292f",
    text_muted="#57606a",
    text_placeholder="#6e7781",
    
    # 主色调 - 蓝色
    accent="#0969da",
    accent_emphasis="#0550ae",
    accent_muted="#218bff",
    accent_subtle="#ddf4ff",
    
    # 语义色
    success="#1a7f37",
    success_emphasis="#116329",
    warning="#9a6700",
    warning_emphasis="#7d4e00",
    error="#cf222e",
    error_emphasis="#a40e26",
    info="#0969da",
    
    # 交互状态
    hover="#f3f4f6",
    active="#ebecf0",
    selected="#ddf4ff",
)


class ThemeManager:
    """主题管理器"""
    
    def __init__(self):
        self._is_dark = True
        self._colors = DARK_THEME
        self._on_change_callbacks = []
    
    @property
    def is_dark(self) -> bool:
        return self._is_dark
    
    @property
    def colors(self) -> ThemeColors:
        return self._colors
    
    def toggle(self):
        """切换主题"""
        self._is_dark = not self._is_dark
        self._colors = DARK_THEME if self._is_dark else LIGHT_THEME
        for callback in self._on_change_callbacks:
            callback(self._is_dark)
    
    def set_dark(self, is_dark: bool):
        """设置主题"""
        self._is_dark = is_dark
        self._colors = DARK_THEME if is_dark else LIGHT_THEME
        for callback in self._on_change_callbacks:
            callback(self._is_dark)
    
    def on_change(self, callback):
        """注册主题变化回调"""
        self._on_change_callbacks.append(callback)
    
    def get_flet_theme(self) -> ft.Theme:
        """获取 Flet 主题对象"""
        c = self._colors
        return ft.Theme(
            color_scheme_seed=c.accent,
            color_scheme=ft.ColorScheme(
                primary=c.accent,
                on_primary="#ffffff" if self._is_dark else "#ffffff",
                primary_container=c.accent_subtle,
                secondary=c.text_secondary,
                surface=c.bg_surface,
                background=c.bg_base,
                error=c.error,
                on_surface=c.text_primary,
                on_background=c.text_primary,
            ),
        )


# 全局主题管理器实例
theme = ThemeManager()
