# -*- coding: utf-8 -*-
"""工具模块 Tab - 调用原版独立程序"""

import flet as ft
import subprocess
import sys
import os
from pathlib import Path
from typing import Callable, Optional
from ..themes import theme
from ..components import Card


class ToolsTab(ft.Container):
    """工具模块 Tab - 调用原版独立程序"""
    
    def __init__(self, page: ft.Page = None, on_open_timer=None, on_open_camera=None, **kwargs):
        """初始化工具模块"""
        self._page = page
        
        # 进程管理
        self._timer_process = None
        self._camera_process = None
        
        # 获取工具路径
        self._root_dir = self._get_root_dir()
        
        c = theme.colors

        self._tools = [
            {
                "key": "timer",
                "name": "翻页时钟",
                "icon": ft.Icons.ACCESS_TIME_ROUNDED,
                "desc": "精美翻页时钟，支持正/倒计时。",
                "details": "支持时钟 / 正计时 / 倒计时模式；可自定义配色。",
                "features": [
                    "时钟模式：实时显示当前时间",
                    "正计时：秒表功能",
                    "倒计时：到时自动提醒",
                    "多套配色方案",
                ],
                "tips": [
                    "右键打开菜单切换模式",
                    "双击全屏/退出全屏",
                    "拖动调整位置和大小",
                ],
                "on_click": self._toggle_timer,
            },
            {
                "key": "camera",
                "name": "摄像头窗口",
                "icon": ft.Icons.VIDEOCAM_ROUNDED,
                "desc": "透明摄像头窗口，适合直播露脸。",
                "details": "支持多种形状，可调节大小和透明度。",
                "features": [
                    "多种形状：圆形、心形、方形等",
                    "美颜滤镜、大眼瘦脸",
                    "人脸贴纸",
                    "始终置顶",
                ],
                "tips": [
                    "右键打开设置菜单",
                    "滚轮调整大小",
                    "双击隐藏窗口",
                ],
                "on_click": self._toggle_camera,
            },
        ]

        tool_items = [self._build_tool_item(t) for t in self._tools]

        tools_grid = ft.GridView(
            expand=True,
            max_extent=170,
            child_aspect_ratio=1.25,
            spacing=12,
            run_spacing=12,
            padding=ft.padding.all(6),
        )

        tools_grid.controls = tool_items

        tools_card = Card(
            "工具",
            ft.Container(content=tools_grid, expand=True),
            expand=True,
        )

        super().__init__(
            content=tools_card,
            padding=15,
            expand=True,
        )

    def _get_root_dir(self) -> Path:
        """获取项目根目录"""
        if getattr(sys, 'frozen', False):
            # 打包后
            return Path(sys.executable).parent
        else:
            # 开发环境
            return Path(__file__).parent.parent.parent.parent

    def _build_tool_item(self, tool: dict) -> ft.Control:
        c = theme.colors

        def on_click(e):
            cb = tool.get("on_click")
            if cb:
                cb(e)

        features = tool.get("features", []) or []
        tips = tool.get("tips", []) or []
        tooltip_text = f"{tool.get('desc', '')}\n\n{tool.get('details', '')}"
        if features:
            tooltip_text += "\n\n功能:\n" + "\n".join(f"  - {f}" for f in features)
        if tips:
            tooltip_text += "\n\n提示:\n" + "\n".join(f"  - {t}" for t in tips)

        return ft.Container(
            content=ft.Column([
                ft.Icon(tool.get("icon"), size=34, color=c.accent),
                ft.Text(tool.get("name", ""), size=12, color=c.text_primary, 
                       weight=ft.FontWeight.W_500, text_align=ft.TextAlign.CENTER),
            ], spacing=8, horizontal_alignment=ft.CrossAxisAlignment.CENTER, 
               alignment=ft.MainAxisAlignment.CENTER),
            padding=12,
            bgcolor=c.bg_elevated,
            border=ft.border.all(1, c.border_default),
            border_radius=10,
            alignment=ft.alignment.center,
            ink=True,
            on_click=on_click,
            tooltip=tooltip_text,
        )
    
    def _toggle_timer(self, e=None):
        """切换翻页时钟"""
        if self._timer_process and self._timer_process.poll() is None:
            # 进程正在运行，关闭它
            self._timer_process.terminate()
            self._timer_process = None
            print("[工具] 翻页时钟已关闭")
        else:
            self._open_timer()
    
    def _open_timer(self):
        """打开翻页时钟"""
        # 查找可执行文件
        exe_paths = []
        
        if getattr(sys, 'frozen', False):
            # 打包后环境
            base = Path(sys.executable).parent
            exe_paths = [
                base / "FlipClock.exe",
                base / "_internal" / "FlipClock.exe",
            ]
            if hasattr(sys, '_MEIPASS'):
                exe_paths.insert(0, Path(sys._MEIPASS) / "FlipClock.exe")
        else:
            # 开发环境 - 直接运行 Python 脚本
            timer_script = self._root_dir / "tools" / "timer" / "timer.py"
            if timer_script.exists():
                try:
                    self._timer_process = subprocess.Popen(
                        [sys.executable, str(timer_script)],
                        cwd=str(timer_script.parent),
                        creationflags=subprocess.CREATE_NO_WINDOW if sys.platform == 'win32' else 0,
                    )
                    print(f"[工具] 翻页时钟已启动 (脚本)")
                    return
                except Exception as ex:
                    print(f"[工具] 启动翻页时钟失败: {ex}")
                    return
        
        # 查找并运行 exe
        for exe_path in exe_paths:
            if exe_path.exists():
                try:
                    self._timer_process = subprocess.Popen(
                        [str(exe_path)],
                        cwd=str(exe_path.parent),
                        creationflags=subprocess.CREATE_NO_WINDOW if sys.platform == 'win32' else 0,
                    )
                    print(f"[工具] 翻页时钟已启动: {exe_path}")
                    return
                except Exception as ex:
                    print(f"[工具] 启动翻页时钟失败: {ex}")
        
        print(f"[工具] 找不到翻页时钟程序")
    
    def _toggle_camera(self, e=None):
        """切换摄像头窗口"""
        if self._camera_process and self._camera_process.poll() is None:
            self._camera_process.terminate()
            self._camera_process = None
            print("[工具] 摄像头窗口已关闭")
        else:
            self._open_camera()
    
    def _open_camera(self):
        """打开摄像头窗口"""
        exe_paths = []
        
        if getattr(sys, 'frozen', False):
            base = Path(sys.executable).parent
            exe_paths = [
                base / "CameraWindow.exe",
                base / "_internal" / "CameraWindow.exe",
            ]
            if hasattr(sys, '_MEIPASS'):
                exe_paths.insert(0, Path(sys._MEIPASS) / "CameraWindow.exe")
        else:
            # 开发环境 - 优先运行精简版（无需 OpenCV）
            camera_script = self._root_dir / "tools" / "camera" / "camera_window_lite.py"
            if not camera_script.exists():
                camera_script = self._root_dir / "tools" / "camera" / "camera_window.py"
            if camera_script.exists():
                try:
                    self._camera_process = subprocess.Popen(
                        [sys.executable, str(camera_script)],
                        cwd=str(camera_script.parent),
                        creationflags=subprocess.CREATE_NO_WINDOW if sys.platform == 'win32' else 0,
                    )
                    print(f"[工具] 摄像头窗口已启动 (脚本)")
                    return
                except Exception as ex:
                    print(f"[工具] 启动摄像头窗口失败: {ex}")
                    return
        
        for exe_path in exe_paths:
            if exe_path.exists():
                try:
                    self._camera_process = subprocess.Popen(
                        [str(exe_path)],
                        cwd=str(exe_path.parent),
                        creationflags=subprocess.CREATE_NO_WINDOW if sys.platform == 'win32' else 0,
                    )
                    print(f"[工具] 摄像头窗口已启动: {exe_path}")
                    return
                except Exception as ex:
                    print(f"[工具] 启动摄像头窗口失败: {ex}")
        
        print(f"[工具] 找不到摄像头窗口程序")
    
    def set_page(self, page: ft.Page):
        """设置页面引用"""
        self._page = page
    
    def dispose(self):
        """清理资源"""
        if self._timer_process and self._timer_process.poll() is None:
            self._timer_process.terminate()
        if self._camera_process and self._camera_process.poll() is None:
            self._camera_process.terminate()
