# -*- coding: utf-8 -*-
"""首页 Tab"""

import flet as ft
from typing import Callable, Optional
from ..themes import theme
from ..components import Card, StatusChip, StatusIndicator
from ..services import StreamServiceManager, EnvironmentChecker


class HomeTab(ft.Container):
    """首页 Tab"""
    
    def __init__(self, 
                 on_start_stream: Callable = None,
                 on_stop_stream: Callable = None,
                 on_close_companion: Callable = None,
                 on_switch_obs: Callable = None,
                 on_copy_url: Callable = None,
                 on_copy_key: Callable = None,
                 on_browse_companion: Callable = None,
                 on_refresh_interfaces: Callable = None,
                 on_danmu_login: Callable = None,
                 on_send_danmu: Callable[[str], None] = None,
                 config_manager = None):
        
        self.config_manager = config_manager
        self._on_start_stream = on_start_stream
        self._on_stop_stream = on_stop_stream
        self._on_close_companion = on_close_companion
        self._on_switch_obs = on_switch_obs
        self._on_copy_url = on_copy_url
        self._on_copy_key = on_copy_key
        self._on_browse_companion = on_browse_companion
        self._on_refresh_interfaces = on_refresh_interfaces
        self._on_danmu_login = on_danmu_login
        self._on_send_danmu = on_send_danmu
        
        c = theme.colors
        
        # ========== 环境检测状态 ==========
        self.npcap_status = StatusChip("Npcap", "检测中...")
        self.companion_status = StatusChip("直播伴侣", "检测中...")
        self.obs_status = StatusChip("OBS", "检测中...")
        
        # ========== 网卡选择 ==========
        self.interface_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option("all", "所有网卡 (推荐)")],
            value="all",
            width=400,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
        )
        
        # ========== 自动连接 ==========
        self.auto_connect_check = ft.Checkbox(label="自动连接直播间", value=True)
        
        # ========== 进度条 ==========
        self.progress_bar = ft.ProgressBar(
            value=0, bgcolor=c.bg_elevated, color=c.accent, height=6
        )
        
        # ========== 推流状态 ==========
        self.stream_status = StatusIndicator("就绪")
        
        # ========== 操作按钮 ==========
        self.start_btn = ft.ElevatedButton(
            "▶ 开播",
            bgcolor=c.success, color="#ffffff",
            on_click=self._handle_start,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        self.stop_btn = ft.OutlinedButton(
            "停止", disabled=True,
            on_click=self._handle_stop,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        # ========== 推流信息显示 ==========
        self.stream_info_text = ft.Text(
            "等待抓取推流信息...",
            size=11, color=c.text_muted
        )
        
        self.stream_info_container = ft.Container(
            content=self.stream_info_text,
            bgcolor=c.bg_elevated,
            padding=10,
            border_radius=6,
        )
        
        # ========== 推流操作按钮 ==========
        self.copy_url_btn = ft.OutlinedButton(
            "复制地址", icon=ft.Icons.CONTENT_COPY, disabled=True,
            on_click=self._handle_copy_url,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        self.copy_key_btn = ft.OutlinedButton(
            "复制密钥", icon=ft.Icons.KEY, disabled=True,
            on_click=self._handle_copy_key,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        self.switch_obs_btn = ft.OutlinedButton(
            "切换OBS", icon=ft.Icons.SWITCH_VIDEO, disabled=True,
            on_click=self._handle_switch_obs,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        self.close_companion_btn = ft.ElevatedButton(
            "关闭伴侣", icon=ft.Icons.POWER_SETTINGS_NEW,
            bgcolor=c.error, color="#ffffff", disabled=False,
            on_click=self._handle_close_companion,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
            tooltip="关闭直播伴侣进程（切换OBS后使用）",
        )
        
        # ========== 一键开播卡片 ==========
        stream_card = Card("一键开播", ft.Column([
            ft.Row([
                self.npcap_status,
                self.companion_status,
                self.obs_status,
            ], spacing=12),
            ft.Row([
                ft.Text("网卡:", size=13, color=c.text_muted, width=50),
                self.interface_dropdown,
                ft.IconButton(icon=ft.Icons.REFRESH, icon_color=c.text_muted, tooltip="刷新", on_click=self._handle_refresh_interfaces),
            ], spacing=10),
            self.auto_connect_check,
            self.progress_bar,
            ft.Row([
                self.stream_status,
                ft.Container(expand=True),
                self.start_btn,
                self.stop_btn,
            ], spacing=10),
            self.stream_info_container,
            ft.Row([
                self.copy_url_btn,
                self.copy_key_btn,
                self.switch_obs_btn,
                self.close_companion_btn,
            ], spacing=10),
        ], spacing=14))
        
        # ========== 直播间配置 ==========
        self.room_id_input = ft.TextField(
            hint_text="输入直播间ID",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_change=self._on_room_id_change,
        )
        
        self.mode_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option("direct", "直连"), ft.dropdown.Option("proxy", "代理")],
            value="direct",
            width=150,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._on_mode_change,
        )
        
        # 主播昵称（用于过滤自己的弹幕）
        self.anchor_name_input = ft.TextField(
            hint_text="主播昵称（自动检测或手动输入）",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_change=self._on_anchor_name_change,
        )
        
        # 忽略主播弹幕选项
        self.ignore_anchor_check = ft.Checkbox(
            label="忽略主播弹幕（AI不回复自己）",
            value=True,
            on_change=self._on_ignore_anchor_change,
        )
        
        self.room_card = Card("直播间配置", ft.Column([
            ft.Row([
                ft.Text("直播间ID:", size=13, color=c.text_muted, width=80),
                self.room_id_input,
                ft.TextButton("如何查找?", style=ft.ButtonStyle(color=c.accent), on_click=self._show_room_id_help),
            ], spacing=10),
            ft.Row([
                ft.Text("主播昵称:", size=13, color=c.text_muted, width=80),
                self.anchor_name_input,
            ], spacing=10),
            self.ignore_anchor_check,
        ], spacing=14), show_save_button=True, on_save=self._save_room_config)
        
        # ========== 弹幕发送 ==========
        self._original_chrome_path = ""  # 保存原始 Chrome 路径用于变更检测
        self.chrome_path_input = ft.TextField(
            hint_text="Chrome浏览器路径 (留空自动查找)",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_change=self._on_chrome_path_change,
        )
        
        self.danmu_status = StatusIndicator("未登录")
        
        self.danmu_login_btn = ft.ElevatedButton(
            "登录抖音", bgcolor=c.accent, color="#ffffff",
            on_click=self._handle_danmu_login,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        self.danmu_input = ft.TextField(
            hint_text="输入要发送的弹幕...",
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_submit=self._handle_send_danmu,
        )
        
        self.danmu_send_btn = ft.ElevatedButton(
            "发送", bgcolor=c.accent, color="#ffffff", disabled=True,
            on_click=self._handle_send_danmu,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        self.danmu_card = Card("弹幕发送 (以你的身份)", ft.Column([
            ft.Row([
                ft.Text("浏览器:", size=13, color=c.text_muted, width=70),
                self.chrome_path_input,
                ft.OutlinedButton("浏览", style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6))),
            ], spacing=10),
            ft.Row([
                self.danmu_status,
                ft.Container(expand=True),
                self.danmu_login_btn,
            ], spacing=10),
            ft.Row([
                self.danmu_input,
                self.danmu_send_btn,
            ], spacing=10),
            ft.Text("支持 Chrome/Edge 浏览器，首次需扫码登录，之后自动登录", size=11, color=c.text_muted),
        ], spacing=12), show_save_button=True, on_save=self._save_danmu_config)
        
        # ========== 主容器 ==========
        super().__init__(
            content=ft.Column([
                stream_card,
                self.room_card,
                self.danmu_card,
            ], spacing=15, scroll=ft.ScrollMode.AUTO, expand=True),
            padding=15,
            expand=True,
        )
    
    # ========== 事件处理 ==========
    
    def _handle_start(self, e):
        # 立即禁用按钮，防止重复点击
        self.start_btn.disabled = True
        self.start_btn.text = "启动中..."
        self.stop_btn.disabled = False
        if self.page:
            self.page.update()
        
        if self._on_start_stream:
            self._on_start_stream()
    
    def _handle_stop(self, e):
        # 立即禁用停止按钮，防止重复点击
        self.stop_btn.disabled = True
        self.stop_btn.text = "停止中..."
        if self.page:
            self.page.update()
        
        if self._on_stop_stream:
            self._on_stop_stream()
    
    def _handle_close_companion(self, e):
        print("[HomeTab] 关闭伴侣按钮被点击")
        if self._on_close_companion:
            print("[HomeTab] 调用 _on_close_companion 回调")
            self._on_close_companion()
        else:
            print("[HomeTab] 警告: _on_close_companion 回调未设置")
    
    def _handle_copy_url(self, e):
        if self._on_copy_url:
            self._on_copy_url()
    
    def _handle_copy_key(self, e):
        if self._on_copy_key:
            self._on_copy_key()
    
    def _handle_browse_companion(self, e):
        if self._on_browse_companion:
            self._on_browse_companion()
    
    def _handle_switch_obs(self, e):
        if self._on_switch_obs:
            self._on_switch_obs()
    
    def _handle_refresh_interfaces(self, e):
        if self._on_refresh_interfaces:
            self._on_refresh_interfaces()
    
    def _handle_danmu_login(self, e):
        if self._on_danmu_login:
            self._on_danmu_login()
    
    def _handle_send_danmu(self, e):
        text = self.danmu_input.value.strip()
        if text and self._on_send_danmu:
            self._on_send_danmu(text)
            self.danmu_input.value = ""
    
    def _on_room_id_change(self, e):
        """UI 变化时只标记待保存，不立即更新 config_manager"""
        self.room_card.set_has_changes(True)
    
    def _on_mode_change(self, e):
        """UI 变化时只标记待保存，不立即更新 config_manager"""
        self.room_card.set_has_changes(True)
    
    def _on_anchor_name_change(self, e):
        """UI 变化时只标记待保存，不立即更新 config_manager"""
        self.room_card.set_has_changes(True)
    
    def _on_ignore_anchor_change(self, e):
        """UI 变化时只标记待保存，不立即更新 config_manager"""
        self.room_card.set_has_changes(True)
    
    def _save_room_config(self, e):
        """保存直播间配置 - 保存时才更新 config_manager"""
        if self.config_manager:
            # 保存时才更新 config_manager
            self.config_manager.set_room_id(self.room_id_input.value or "")
            self.config_manager.set_mode(self.mode_dropdown.value or "direct")
            self.config_manager.set_anchor_name(self.anchor_name_input.value or "")
            self.config_manager.set_ignore_anchor(self.ignore_anchor_check.value)
            
            success = self.config_manager.save()
            if success:
                self.room_card.set_has_changes(False)
            self._show_save_result(success, "直播间配置已保存" if success else None)
    
    def _show_save_result(self, success: bool, custom_message: str = None):
        """显示保存结果弹窗"""
        c = theme.colors
        
        def close_dialog(e):
            dialog.open = False
            if self.page:
                self.page.update()
        
        if success:
            title = "保存成功"
            message = custom_message or "配置已保存"
            btn_color = "#4CAF50"
        else:
            title = "保存失败"
            message = custom_message or "保存配置时出错，请重试"
            btn_color = "#E53935"
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text(title, weight=ft.FontWeight.BOLD),
            content=ft.Text(message, size=14),
            actions=[
                ft.ElevatedButton("确定", bgcolor=btn_color, color="#ffffff", on_click=close_dialog),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    def _on_chrome_path_change(self, e):
        """Chrome 路径变化"""
        current_path = self.chrome_path_input.value or ""
        has_changes = current_path != self._original_chrome_path
        self.danmu_card.set_has_changes(has_changes)
    
    def _save_danmu_config(self, e):
        """保存弹幕发送配置"""
        import os
        chrome_path = self.chrome_path_input.value or ""
        
        # 如果路径不为空，检查是否存在
        if chrome_path and not os.path.exists(chrome_path):
            self._show_path_not_found_dialog(chrome_path)
            return
        
        if self.config_manager:
            self.config_manager.set_chrome_path(chrome_path)
            self.config_manager.save()
        
        # 更新原始配置
        self._original_chrome_path = chrome_path
        self.danmu_card.set_has_changes(False)
        
        self._show_save_result(True)
    
    def _show_path_not_found_dialog(self, path: str):
        """显示路径不存在提示对话框"""
        def close_dialog(e):
            dialog.open = False
            if self.page:
                self.page.update()
        
        def auto_detect(e):
            dialog.open = False
            self.chrome_path_input.value = ""  # 清空路径，使用自动检测
            self._original_chrome_path = ""
            self.danmu_card.set_has_changes(False)
            if self.config_manager:
                self.config_manager.set_chrome_path("")
                self.config_manager.save()
            if self.page:
                self.page.update()
            self._show_save_result(True, "已设置为自动检测")
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text("路径不存在", weight=ft.FontWeight.BOLD),
            content=ft.Text(f"指定的浏览器路径不存在：\n{path}\n\n是否使用自动检测？", size=14),
            actions=[
                ft.TextButton("取消", on_click=close_dialog),
                ft.ElevatedButton("自动检测", bgcolor="#4CAF50", color="#ffffff", on_click=auto_detect),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    # ========== 状态更新 ==========
    
    def update_env_status(self, npcap: bool, companion: bool, obs: bool):
        """更新环境检测状态"""
        if npcap:
            self.npcap_status.set_ok("已安装")
        else:
            self.npcap_status.set_error("未安装")
        
        if companion:
            self.companion_status.set_ok("已安装")
        else:
            self.companion_status.set_error("未安装")
        
        if obs:
            self.obs_status.set_ok("已安装")
        else:
            self.obs_status.set_error("未安装")
    
    def update_stream_status(self, state: str, message: str = ""):
        """更新推流状态"""
        c = theme.colors
        
        if state == "idle":
            self.stream_status.set_status("就绪", c.text_muted)
            self.start_btn.disabled = False
            self.start_btn.text = "▶ 开播"
            self.stop_btn.disabled = True
            self.stop_btn.text = "停止"
            self.progress_bar.value = 0  # 重置进度条
        elif state == "starting":
            self.stream_status.set_status("启动中...", c.warning)
            self.start_btn.disabled = True
            self.start_btn.text = "启动中..."
            self.stop_btn.disabled = False
            self.stop_btn.text = "停止"
        elif state == "capturing":
            self.stream_status.set_status(message or "抓取中...", c.warning)
            self.start_btn.disabled = True
            self.start_btn.text = "抓取中..."
            self.stop_btn.disabled = False
            self.stop_btn.text = "停止"
        elif state == "connected":
            self.stream_status.set_status("已连接", c.success)
            self.start_btn.disabled = True
            self.start_btn.text = "直播中"
            self.stop_btn.disabled = False
            self.stop_btn.text = "停止"
            self.copy_url_btn.disabled = False
            self.copy_key_btn.disabled = False
            self.switch_obs_btn.disabled = False
            # 停止进度条动画
            self.progress_bar.value = 1.0  # 完成
        elif state == "error":
            self.stream_status.set_error(message or "错误")
            self.start_btn.disabled = False
            self.start_btn.text = "▶ 开播"
            self.stop_btn.disabled = True
            self.stop_btn.text = "停止"
    
    def update_progress(self, value: int):
        """更新进度条"""
        self.progress_bar.value = value / 100.0
    
    def update_stream_info(self, url: str, key: str):
        """更新推流信息"""
        c = theme.colors
        self.stream_info_text.value = f"【推流信息】\n服务器: {url}\n密钥: {key[:30]}..."
        self.stream_info_text.color = c.text_secondary
    
    def set_danmu_logged_in(self, logged_in: bool):
        """设置弹幕登录状态"""
        c = theme.colors
        if logged_in:
            self.danmu_status.set_status("已登录", c.success)
            self.danmu_login_btn.text = "已登录"
            self.danmu_login_btn.disabled = True
            self.danmu_send_btn.disabled = False
        else:
            self.danmu_status.set_status("未登录", c.text_muted)
            self.danmu_login_btn.text = "登录抖音"
            self.danmu_login_btn.disabled = False
            self.danmu_send_btn.disabled = True
    
    def load_config(self, config_manager):
        """从配置加载数据"""
        import os
        self.config_manager = config_manager
        self.room_id_input.value = config_manager.get_room_id()
        self.mode_dropdown.value = config_manager.get_mode()
        self.anchor_name_input.value = config_manager.get_anchor_name()
        self.ignore_anchor_check.value = config_manager.get_ignore_anchor()
        
        # 加载 Chrome 路径并检查是否存在
        chrome_path = config_manager.get_chrome_path()
        if chrome_path and not os.path.exists(chrome_path):
            # 路径不存在，延迟显示提示对话框
            self.chrome_path_input.value = chrome_path
            self._original_chrome_path = chrome_path
            self._pending_path_check = True  # 标记需要检查路径
        else:
            self.chrome_path_input.value = chrome_path
            self._original_chrome_path = chrome_path
            self._pending_path_check = False
        
        # 重置卡片状态
        self.room_card.set_has_changes(False)
        self.danmu_card.set_has_changes(False)
    
    def check_pending_path(self):
        """检查待处理的路径（需要在页面加载后调用）"""
        if getattr(self, '_pending_path_check', False) and self.chrome_path_input.value:
            self._pending_path_check = False
            self._show_saved_path_invalid_dialog(self.chrome_path_input.value)
    
    def _show_saved_path_invalid_dialog(self, path: str):
        """显示保存的路径无效提示对话框"""
        def close_dialog(e):
            dialog.open = False
            if self.page:
                self.page.update()
        
        def auto_detect(e):
            dialog.open = False
            self.chrome_path_input.value = ""  # 清空路径，使用自动检测
            self._original_chrome_path = ""
            self.danmu_card.set_has_changes(False)
            if self.config_manager:
                self.config_manager.set_chrome_path("")
                self.config_manager.save()
            if self.page:
                self.page.update()
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text("浏览器路径无效", weight=ft.FontWeight.BOLD),
            content=ft.Text(f"之前保存的浏览器路径不存在：\n{path}\n\n是否切换为自动检测？", size=14),
            actions=[
                ft.TextButton("保留路径", on_click=close_dialog),
                ft.ElevatedButton("自动检测", bgcolor="#4CAF50", color="#ffffff", on_click=auto_detect),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    def _show_room_id_help(self, e):
        """显示如何查找直播间ID的帮助"""
        c = theme.colors
        
        dialog = ft.AlertDialog(
            title=ft.Row([
                ft.Icon(ft.Icons.HELP_OUTLINE, color=c.accent),
                ft.Text("如何查找直播间ID", weight=ft.FontWeight.BOLD),
            ], spacing=8),
            content=ft.Column([
                ft.Text("方法一：从直播链接获取", size=14, weight=ft.FontWeight.BOLD),
                ft.Text("1. 在抖音 App 或网页打开直播间", size=13),
                ft.Text("2. 点击分享，复制链接", size=13),
                ft.Text("3. 链接中的数字就是直播间ID", size=13),
                ft.Text("   例如: live.douyin.com/123456789", size=12, color=c.text_muted, italic=True),
                ft.Divider(height=16),
                ft.Text("方法二：从直播伴侣获取", size=14, weight=ft.FontWeight.BOLD),
                ft.Text("1. 打开抖音直播伴侣，开始直播", size=13),
                ft.Text("2. 在伴侣界面找到直播间链接", size=13),
                ft.Text("3. 链接中的数字就是直播间ID", size=13),
            ], tight=True, spacing=6),
            actions=[
                ft.TextButton("知道了", on_click=lambda e: self._close_help_dialog(dialog)),
            ],
        )
        
        if self.page:
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
    
    def _close_help_dialog(self, dialog):
        """关闭帮助对话框"""
        dialog.open = False
        if self.page:
            self.page.update()
