# -*- coding: utf-8 -*-
"""说明 Tab - 参照 voice.py 结构重写"""

import flet as ft
from typing import Callable, Optional
from ..themes import theme
from ..components import Card

# 配置信息
CONFIG = {
    "qq": "1622068165",
    "buy_url": "https://pay.ldxp.cn/shop/wutong/n2tps8",
}


class HelpTab(ft.Container):
    """说明 Tab"""
    
    def __init__(self, on_feedback: Callable = None):
        self._on_feedback = on_feedback
        c = theme.colors
        
        # ========== 快捷操作卡片 ==========
        self.buy_btn = ft.ElevatedButton(
            "购买卡密",
            icon=ft.Icons.SHOPPING_CART,
            bgcolor=ft.Colors.BLUE_600,
            color=ft.Colors.WHITE,
            on_click=self._on_buy,
        )
        
        self.trial_btn = ft.OutlinedButton(
            "免费试用",
            icon=ft.Icons.CARD_GIFTCARD,
            on_click=self._on_trial,
        )
        
        self.qq_btn = ft.OutlinedButton(
            f"客服QQ: {CONFIG['qq']}",
            icon=ft.Icons.SUPPORT_AGENT,
            on_click=self._on_copy_qq,
        )
        
        self.action_card = Card("快捷操作", ft.Column([
            ft.Row([
                self.buy_btn,
                self.trial_btn,
                self.qq_btn,
            ], spacing=10),
        ], spacing=10))
        
        # ========== 留言反馈卡片 ==========
        self.feedback_type = ft.Dropdown(
            label="反馈类型",
            width=150,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            options=[
                ft.dropdown.Option("FEEDBACK", "功能建议"),
                ft.dropdown.Option("BUG", "问题报告"),
                ft.dropdown.Option("CONSULT", "咨询"),
            ],
            value="FEEDBACK",
        )
        
        self.contact_field = ft.TextField(
            label="联系方式（QQ/微信）*",
            hint_text="必填",
            width=200,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
        )
        
        self.feedback_field = ft.TextField(
            label="留言内容",
            hint_text="请描述您的问题或建议...",
            multiline=True,
            min_lines=3,
            max_lines=5,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
        )
        
        self.submit_btn = ft.ElevatedButton(
            "提交留言",
            icon=ft.Icons.SEND,
            bgcolor=ft.Colors.GREEN_600,
            color=ft.Colors.WHITE,
            on_click=self._on_submit_feedback,
        )
        
        self.feedback_status = ft.Text("", size=12, color=c.text_muted)
        
        self.feedback_card = Card("留言反馈", ft.Column([
            ft.Row([
                self.feedback_type,
                self.contact_field,
            ], spacing=10),
            self.feedback_field,
            ft.Row([
                self.submit_btn,
                self.feedback_status,
            ], spacing=10),
        ], spacing=10))
        
        # ========== 功能说明卡片 ==========
        features = [
            ("🏠 首页", "一键开播、弹幕监控、弹幕发送"),
            ("🎤 语音回复", "入场欢迎、礼物感谢、关注感谢、朗读弹幕"),
            ("🤖 AI智能", "智能回复、暖场模式、弹幕发送"),
            ("👤 虚拟形象", "口型同步、OBS集成"),
            ("🛡️ 防检测", "画面防检测、打字防检测"),
            ("🔧 工具", "翻页时钟、正计时、倒计时"),
            ("📊 统计", "实时数据、热度指示、排行榜"),
        ]
        
        feature_rows = []
        for title, desc in features:
            feature_rows.append(
                ft.Row([
                    ft.Text(title, size=13, weight=ft.FontWeight.BOLD, color=c.text_primary, width=100),
                    ft.Text(desc, size=12, color=c.text_secondary),
                ], spacing=10)
            )
        
        self.feature_card = Card("功能说明", ft.Column(feature_rows, spacing=8))
        
        # ========== 激活说明卡片 ==========
        activation_steps = [
            "1. 点击右上角授权状态区域，打开激活窗口",
            "2. 申请试用：点击「申请试用」获取7天免费体验",
            "3. 购买卡密：点击上方「购买卡密」按钮前往购买",
            "4. 激活授权：将卡密粘贴到输入框，点击「激活」",
        ]
        
        activation_rows = [ft.Text(step, size=12, color=c.text_secondary) for step in activation_steps]
        
        self.activation_card = Card("如何激活", ft.Column(activation_rows, spacing=6))
        
        # ========== 常见问题卡片 ==========
        faq = [
            ("Q: 抓取不到推流地址？", "A: 确保已安装 Npcap，并以管理员权限运行"),
            ("Q: OBS 连接失败？", "A: 检查 OBS WebSocket 是否启用"),
            ("Q: 语音没有声音？", "A: 检查音量设置，点击试听测试"),
        ]
        
        faq_rows = []
        for q, a in faq:
            faq_rows.append(ft.Text(q, size=12, weight=ft.FontWeight.W_500, color=c.text_primary))
            faq_rows.append(ft.Text(a, size=12, color=c.text_secondary))
        
        self.faq_card = Card("常见问题", ft.Column(faq_rows, spacing=4))
        
        # ========== 主容器 ==========
        super().__init__(
            content=ft.Column([
                self.action_card,
                self.feedback_card,
                self.feature_card,
                self.activation_card,
                self.faq_card,
            ], spacing=15, scroll=ft.ScrollMode.AUTO, expand=True),
            padding=15,
            expand=True,
        )
    
    def _on_buy(self, e):
        """打开购买链接"""
        if e.page:
            e.page.launch_url(CONFIG["buy_url"])
    
    def _on_trial(self, e):
        """申请试用"""
        if self._on_feedback:
            self._on_feedback("trial")
    
    def _on_copy_qq(self, e):
        """复制客服QQ"""
        if e.page:
            e.page.set_clipboard(CONFIG["qq"])
            e.page.snack_bar = ft.SnackBar(content=ft.Text(f"已复制客服QQ: {CONFIG['qq']}"))
            e.page.snack_bar.open = True
            e.page.update()
    
    def _on_submit_feedback(self, e):
        """提交留言"""
        c = theme.colors
        contact = self.contact_field.value
        if not contact or not contact.strip():
            self.feedback_status.value = "请输入联系方式"
            self.feedback_status.color = ft.Colors.RED_400
            if e.page:
                e.page.update()
            return
        
        content = self.feedback_field.value
        if not content or not content.strip():
            self.feedback_status.value = "请输入留言内容"
            self.feedback_status.color = ft.Colors.RED_400
            if e.page:
                e.page.update()
            return
        
        # 调用回调提交留言
        if self._on_feedback:
            self._on_feedback("feedback", {
                "type": self.feedback_type.value,
                "contact": self.contact_field.value,
                "content": content,
            })
        
        # 清空表单
        self.feedback_field.value = ""
        self.contact_field.value = ""
        self.feedback_status.value = "✓ 留言已提交，感谢您的反馈！"
        self.feedback_status.color = ft.Colors.GREEN_400
        if e.page:
            e.page.update()
    
    def refresh_theme(self):
        """刷新主题"""
        pass
