# -*- coding: utf-8 -*-
"""形象 Tab - 点击即时生效，无需保存"""

import flet as ft
import webbrowser
from pathlib import Path
from typing import Callable, List
from ..themes import theme
from ..components import Card


class AvatarTab(ft.Container):
    """形象 Tab"""
    
    def __init__(self, 
                 on_avatar_toggle: Callable[[bool], None] = None,
                 on_desktop_toggle: Callable[[bool], None] = None,
                 on_model_change: Callable[[str], None] = None,
                 on_unlock_toggle: Callable = None,
                 models: List[str] = None,
                 config_manager = None):
        
        self._on_avatar_toggle = on_avatar_toggle
        self._on_desktop_toggle = on_desktop_toggle
        self._on_model_change = on_model_change
        self._on_unlock_toggle = on_unlock_toggle
        self.config_manager = config_manager
        
        c = theme.colors
        
        # 模型列表
        self._models = models or ["Hiyori", "Mao", "Rice"]
        
        # ========== 启用开关 ==========
        self.avatar_check = ft.Checkbox(
            label="启用虚拟形象服务 (Web Server)",
            value=False,
            on_change=self._on_avatar_change,
        )
        
        # ========== 模型选择 ==========
        self.model_dropdown = ft.Dropdown(
            options=[ft.dropdown.Option(m) for m in self._models],
            value=self._models[0] if self._models else None,
            width=180,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            on_change=self._on_model_select,
        )
        
        # ========== 桌面模式 ==========
        self.desktop_check = ft.Checkbox(
            label="桌面透明悬浮窗 (桌宠模式)",
            value=False,
            on_change=self._on_desktop_change,
        )
        
        # ========== 解锁按钮 ==========
        self.unlock_btn = ft.OutlinedButton(
            "🔓 解锁形象 (可移动)",
            disabled=True,
            on_click=self._handle_unlock,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
        )
        
        # ========== OBS 地址 ==========
        self.url_input = ft.TextField(
            value="http://127.0.0.1:8080/live2d/index.html?model=Hiyori",
            read_only=True,
            border_color=c.border_default,
            bgcolor=c.bg_elevated,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
        )
        
        # ========== 卡片（无保存按钮，点击即时生效） ==========
        self.avatar_card = Card("Live2D 虚拟形象", ft.Column([
            self.avatar_check,
            ft.Row([
                ft.Text("选择模型:", size=13, color=c.text_muted, width=80),
                self.model_dropdown,
            ], spacing=10),
            self.desktop_check,
            self.unlock_btn,
            ft.Text("OBS 浏览器源地址:", size=13, color=c.text_muted),
            self.url_input,
            ft.Row([
                ft.OutlinedButton(
                    "复制", icon=ft.Icons.CONTENT_COPY,
                    on_click=self._copy_url,
                    style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
                ),
                ft.OutlinedButton(
                    "在浏览器打开", icon=ft.Icons.OPEN_IN_NEW,
                    on_click=self._open_url,
                    style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=6)),
                ),
            ], spacing=10),
            ft.Text(
                "提示: Web 服务在系统启动时自动运行。\n如需OBS透明背景，请使用浏览器源。",
                size=11, color=c.text_muted
            ),
        ], spacing=12))  # 移除保存按钮
        
        super().__init__(
            content=ft.Column([self.avatar_card], spacing=15, scroll=ft.ScrollMode.AUTO, expand=True),
            padding=15,
            expand=True,
        )
    
    def _on_avatar_change(self, e):
        """勾选变化 - 即时生效"""
        enabled = e.control.value
        # 保存配置
        self._save_config_silent()
        # 触发回调
        if self._on_avatar_toggle:
            self._on_avatar_toggle(enabled)
        if self.page:
            self.page.update()
    
    def _on_desktop_change(self, e):
        """桌面模式变化 - 即时生效"""
        enabled = e.control.value
        self.unlock_btn.disabled = not enabled
        # 保存配置
        self._save_config_silent()
        # 触发回调
        if self._on_desktop_toggle:
            self._on_desktop_toggle(enabled)
        if self.page:
            self.page.update()
    
    def _on_model_select(self, e):
        """模型选择 - 即时生效"""
        model = e.control.value
        # 更新 URL（使用当前端口）
        # 注：端口由 app.py 在 Web Server 启动后更新
        current_url = self.url_input.value
        # 从当前 URL 提取端口
        import re
        port_match = re.search(r':(\d+)/', current_url)
        port = port_match.group(1) if port_match else "8080"
        self.url_input.value = f"http://127.0.0.1:{port}/live2d/index.html?model={model}"
        # 保存配置
        self._save_config_silent()
        # 触发回调
        if self._on_model_change:
            self._on_model_change(model)
        if self.page:
            self.page.update()
    
    def _save_config_silent(self):
        """静默保存配置（不弹窗）"""
        if self.config_manager:
            config = {
                "enabled": self.avatar_check.value,
                "model": self.model_dropdown.value,
                "desktop_mode": self.desktop_check.value,
            }
            self.config_manager.set_avatar_config(config)
            self.config_manager.save()
    
    def _handle_unlock(self, e):
        if self._on_unlock_toggle:
            self._on_unlock_toggle()
    
    def _copy_url(self, e):
        e.page.set_clipboard(self.url_input.value)
        e.page.show_snack_bar(ft.SnackBar(content=ft.Text("已复制到剪贴板")))
    
    def _open_url(self, e):
        webbrowser.open(self.url_input.value)
    
    def set_models(self, models: List[str]):
        """设置模型列表"""
        self._models = models
        self.model_dropdown.options = [ft.dropdown.Option(m) for m in models]
        if models and not self.model_dropdown.value:
            self.model_dropdown.value = models[0]
    
    def set_unlock_state(self, unlocked: bool):
        """设置解锁状态"""
        if unlocked:
            self.unlock_btn.text = "🔓 解锁形象 (可移动)"
        else:
            self.unlock_btn.text = "🔒 锁定形象 (点击穿透)"
    
    def load_config(self, config_manager):
        """从配置加载数据"""
        self.config_manager = config_manager
        
        config = config_manager.get_avatar_config()
        self.avatar_check.value = config.get("enabled", False)
        
        model = config.get("model", "Hiyori")
        if model in self._models:
            self.model_dropdown.value = model
        elif self._models:
            self.model_dropdown.value = self._models[0]
        
        self.desktop_check.value = config.get("desktop_mode", False)
        # URL 会在 Web Server 启动后由 app.py 更新为实际端口
        self.url_input.value = f"http://127.0.0.1:8080/live2d/index.html?model={self.model_dropdown.value}"
        
        # 更新解锁按钮状态
        self.unlock_btn.disabled = not self.desktop_check.value
        # 注：不自动启动桌面悬浮窗，用户需要手动勾选才会启动
