# -*- coding: utf-8 -*-
"""工作线程服务 - 弹幕监听、语音回复等"""

import sys
import asyncio
import threading
from pathlib import Path
from typing import Callable, Optional, Dict, Any
from dataclasses import dataclass

ROOT_DIR = Path(__file__).parent.parent.parent.parent
if getattr(sys, 'frozen', False):
    ROOT_DIR = Path(sys._MEIPASS)
sys.path.insert(0, str(ROOT_DIR))


@dataclass
class MessageEvent:
    """消息事件"""
    type: str  # entrance, gift, chat, follow
    nickname: str
    content: str


class WorkerManager:
    """工作线程管理器"""
    
    def __init__(self):
        self._thread: Optional[threading.Thread] = None
        self._running = False
        self._loop: Optional[asyncio.AbstractEventLoop] = None
        self._client = None
        self._handler = None
        self._player = None
        self._tts = None
        self._tts_lock = None
        self._data_collector = None  # 数据采集器
        
        # 配置
        self.room_id: str = ""
        self.config: Dict[str, Any] = {}
        
        # 自动重连
        self._has_connected_before = False
        self._auto_reconnect_enabled = True
        self._reconnect_interval = 5
        self._max_reconnect_attempts = 10
        self._reconnect_count = 0
        self._is_manual_stop = False
        self._is_connected = False  # 连接状态，用于控制语音播报
        
        # 回调
        self._on_message: Optional[Callable[[MessageEvent], None]] = None
        self._on_status_change: Optional[Callable[[str], None]] = None
        self._on_error: Optional[Callable[[str], None]] = None
        self._on_voice_played: Optional[Callable[[str], None]] = None
        self._on_danmu_request: Optional[Callable[[str], None]] = None
    
    def on_message(self, callback: Callable[[MessageEvent], None]):
        self._on_message = callback
    
    def on_status_change(self, callback: Callable[[str], None]):
        self._on_status_change = callback
    
    def on_error(self, callback: Callable[[str], None]):
        self._on_error = callback
    
    def on_voice_played(self, callback: Callable[[str], None]):
        self._on_voice_played = callback
    
    def on_danmu_request(self, callback: Callable[[str], None]):
        self._on_danmu_request = callback
    
    def set_data_collector(self, collector):
        """设置数据采集器"""
        self._data_collector = collector
    
    def start(self, room_id: str, config: Dict[str, Any]):
        """启动工作线程"""
        # 如果已在运行，先停止旧连接
        if self._running or self._thread is not None:
            print(f"[DEBUG] 检测到旧连接，先清理...")
            self.stop(manual=False)
            # 等待旧线程结束
            if self._thread and self._thread.is_alive():
                self._thread.join(timeout=2)
            self._thread = None
            self._loop = None
        
        self.room_id = room_id
        self.config = config
        self._is_manual_stop = False
        self._reconnect_count = 0  # 重置重连计数
        
        self._thread = threading.Thread(target=self._run, daemon=True)
        self._thread.start()
        return True
    
    def stop(self, manual: bool = True):
        """停止工作线程"""
        print(f"[DEBUG] WorkerManager.stop() called, manual={manual}")
        self._is_manual_stop = manual
        self._running = False
        
        # 先停止客户端
        if self._client:
            try:
                self._client._running = False
                if self._client._ws:
                    self._client._ws.close()
            except Exception as e:
                print(f"[DEBUG] 关闭客户端异常: {e}")
        
        # 停止播放器
        if self._player:
            try:
                self._player.stop()
                self._player.clear_queue()
            except:
                pass
        
        # 停止 pygame
        try:
            import pygame
            pygame.mixer.music.stop()
        except:
            pass
        
        # 停止事件循环（简化处理，避免递归）
        if self._loop:
            try:
                if self._loop.is_running():
                    self._loop.call_soon_threadsafe(self._loop.stop)
            except Exception as e:
                print(f"[DEBUG] 停止事件循环异常: {e}")
        
        # 清理引用（不清理 _player 和 _tts，避免 pygame 状态问题）
        self._client = None
        self._handler = None
        self._is_connected = False
        print(f"[DEBUG] WorkerManager.stop() 完成")
    
    def _schedule_reconnect(self):
        """调度自动重连"""
        if self._is_manual_stop or not self._has_connected_before:
            return
        
        if not self._auto_reconnect_enabled:
            return
        
        if self._reconnect_count >= self._max_reconnect_attempts:
            self._notify_error(f"已达到最大重连次数 ({self._max_reconnect_attempts})")
            return
        
        self._reconnect_count += 1
        wait_time = min(self._reconnect_interval * self._reconnect_count, 60)
        
        def do_reconnect():
            import time
            time.sleep(wait_time)
            if not self._is_manual_stop and not self._running:
                self._notify_status("reconnecting")
                self.start(self.room_id, self.config)
        
        threading.Thread(target=do_reconnect, daemon=True).start()
    
    def update_config(self, config: Dict[str, Any]):
        """更新配置（支持热更新 TTS 引擎）"""
        self.config = config
        
        # 更新 TTS 引擎（支持音色热切换）
        if self._running and self._tts:
            try:
                from src.tts import create_tts_engine
                
                tts_config = config.get("tts", {})
                engine_type = tts_config.get("engine", "edge")
                
                tts_kwargs = {}
                if engine_type == "edge":
                    edge_config = tts_config.get("edge", {})
                    tts_kwargs = {
                        "voice": edge_config.get("voice", "zh-CN-XiaoxiaoNeural"),
                        "rate": edge_config.get("rate", "+0%")
                    }
                elif engine_type == "volcengine":
                    volc_config = tts_config.get("volcengine", {})
                    tts_kwargs = {
                        "app_id": volc_config.get("app_id"),
                        "access_token": volc_config.get("access_token"),
                        "voice_type": volc_config.get("voice_type", "BV001_streaming"),
                        "cluster": volc_config.get("cluster", "volcano_tts")
                    }
                
                # 重新创建 TTS 引擎
                new_tts = create_tts_engine(
                    engine=engine_type,
                    output_dir=str(ROOT_DIR / "output"),
                    **tts_kwargs
                )
                
                # 替换旧引擎
                self._tts = new_tts
                
                # 更新 ReplyHandler 中的 TTS 引擎
                if self._handler and hasattr(self._handler, '_handler'):
                    self._handler._handler.tts = new_tts
                
                print(f"[DEBUG] TTS 引擎已热更新: {engine_type}, voice={tts_kwargs.get('voice', tts_kwargs.get('voice_type'))}")
            except Exception as e:
                print(f"[DEBUG] TTS 热更新失败: {e}")
        
        # 更新音量
        if self._player:
            try:
                audio_config = config.get("audio", {})
                volume = audio_config.get("volume", 1.0)
                self._player.set_volume(volume)
            except:
                pass
        
        # 更新回复处理器配置
        if self._handler:
            try:
                self._handler.update_config(config)
            except:
                pass
    
    def _run(self):
        """运行工作线程"""
        from src.utils.events import EventType
        from src.danmaku import create_danmaku_client
        from src.tts import create_tts_engine
        from src.audio import init_audio_player
        from src.handler import ReplyHandler
        
        self._running = True
        self._loop = asyncio.new_event_loop()
        asyncio.set_event_loop(self._loop)
        
        try:
            # 初始化 TTS
            tts_config = self.config.get("tts", {})
            engine_type = tts_config.get("engine", "edge")
            
            tts_kwargs = {}
            if engine_type == "edge":
                edge_config = tts_config.get("edge", {})
                tts_kwargs = {
                    "voice": edge_config.get("voice", "zh-CN-XiaoxiaoNeural"),
                    "rate": edge_config.get("rate", "+0%")
                }
            elif engine_type == "volcengine":
                volc_config = tts_config.get("volcengine", {})
                tts_kwargs = {
                    "app_id": volc_config.get("app_id"),
                    "access_token": volc_config.get("access_token"),
                    "voice_type": volc_config.get("voice_type", "BV001_streaming"),
                    "cluster": volc_config.get("cluster", "volcano_tts")
                }
            
            self._tts_lock = threading.Lock()
            
            self._tts = create_tts_engine(
                engine=engine_type,
                output_dir=str(ROOT_DIR / "output"),
                **tts_kwargs
            )
            
            # 初始化播放器
            audio_config = self.config.get("audio", {})
            self._player = init_audio_player(
                volume=audio_config.get("volume", 1.0),
                queue_max_size=20
            )
            
            # 初始化回复处理器
            self._handler = ReplyHandlerWithCallback(
                tts_engine=self._tts,
                audio_player=self._player,
                config=self.config,
                voice_callback=self._handle_voice_played,
                danmu_callback=self._handle_danmu_request
            )
            
            # 创建弹幕客户端
            mode = self.config.get("live", {}).get("mode", "direct")
            proxy_url = self.config.get("live", {}).get("proxy_ws_url")
            
            self._notify_status("connecting")
            print(f"[DEBUG] 开始创建弹幕客户端, room_id={self.room_id}, mode={mode}")
            
            try:
                self._client = create_danmaku_client(
                    room_id=self.room_id,
                    mode=mode,
                    proxy_ws_url=proxy_url
                )
                print(f"[DEBUG] 弹幕客户端创建成功: {type(self._client).__name__}")
            except Exception as e:
                print(f"[DEBUG] 创建弹幕客户端失败: {e}")
                import traceback
                traceback.print_exc()
                raise
            
            # 注册事件处理
            print(f"[DEBUG] 开始注册事件处理器")
            self._register_handlers()
            print(f"[DEBUG] 事件处理器注册完成")
            
            # 启动主任务
            async def main_task():
                warm_task = asyncio.create_task(self._warm_up_loop())
                client_task = asyncio.create_task(self._client.start())
                await asyncio.gather(client_task, warm_task, return_exceptions=True)
            
            self._loop.run_until_complete(main_task())
            
        except Exception as e:
            self._notify_error(str(e))
        finally:
            self._running = False
            self._notify_status("disconnected")
            if self._loop:
                self._loop.close()
            # 尝试自动重连
            self._schedule_reconnect()
    
    async def _warm_up_loop(self):
        """暖场检查循环"""
        while self._running:
            try:
                if self._handler:
                    await self._handler.check_warm_up()
            except Exception as e:
                print(f"暖场检查异常: {e}")
            await asyncio.sleep(10)
    
    def _register_handlers(self):
        """注册事件处理器"""
        from src.utils.events import EventType
        
        async def on_entrance(event):
            if event.user:
                self._notify_message("entrance", event.user.nickname, "进入")
                # 数据采集
                if self._data_collector:
                    self._data_collector.on_entrance(event)
                # 仅在已连接状态下播报语音
                if self._is_connected:
                    await self._handler.handle_event(event)
        
        async def on_gift(event):
            if event.user and event.gift:
                self._notify_message("gift", event.user.nickname, f"{event.gift.name}x{event.gift.count}")
                # 数据采集
                if self._data_collector:
                    self._data_collector.on_gift(event)
                if self._is_connected:
                    await self._handler.handle_event(event)
        
        async def on_chat(event):
            if event.user:
                self._notify_message("chat", event.user.nickname, event.content)
                # 数据采集
                if self._data_collector:
                    self._data_collector.on_chat(event)
                if self._is_connected:
                    # 检查是否忽略主播弹幕
                    if self._should_ignore_anchor(event.user.nickname):
                        return
                    await self._handler.handle_event(event)
        
        async def on_follow(event):
            if event.user:
                self._notify_message("follow", event.user.nickname, "关注")
                # 数据采集
                if self._data_collector:
                    self._data_collector.on_follow(event)
                if self._is_connected:
                    await self._handler.handle_event(event)
        
        async def on_like(event):
            # 数据采集
            if self._data_collector:
                self._data_collector.on_like(event)
        
        async def on_room_stats(event):
            # 数据采集
            if self._data_collector:
                self._data_collector.on_room_stats(event)
        
        async def on_connected(event):
            self._has_connected_before = True
            self._reconnect_count = 0
            self._is_connected = True
            # 启动数据采集
            if self._data_collector:
                self._data_collector.start(room_id=self.room_id)
            self._notify_status("connected")
        
        async def on_disconnected(event):
            self._is_connected = False
            # 停止数据采集
            if self._data_collector:
                self._data_collector.stop()
            # 断开连接时停止语音播报
            self._stop_voice_playback()
            self._notify_status("disconnected")
        
        self._client.on(EventType.ENTRANCE, on_entrance)
        self._client.on(EventType.GIFT, on_gift)
        self._client.on(EventType.CHAT, on_chat)
        self._client.on(EventType.FOLLOW, on_follow)
        self._client.on(EventType.LIKE, on_like)
        self._client.on(EventType.ROOM_STATS, on_room_stats)
        self._client.on(EventType.CONNECTED, on_connected)
        self._client.on(EventType.DISCONNECTED, on_disconnected)
    
    def _notify_message(self, msg_type: str, nickname: str, content: str):
        if self._on_message:
            self._on_message(MessageEvent(msg_type, nickname, content))
    
    def _notify_status(self, status: str):
        if self._on_status_change:
            self._on_status_change(status)
    
    def _notify_error(self, msg: str):
        if self._on_error:
            self._on_error(msg)
    
    def _handle_voice_played(self, text: str):
        if self._on_voice_played:
            self._on_voice_played(text)
    
    def _handle_danmu_request(self, text: str):
        if self._on_danmu_request:
            self._on_danmu_request(text)
    
    def _stop_voice_playback(self):
        """停止语音播报"""
        if self._player:
            try:
                self._player.stop()
                self._player.clear_queue()
            except:
                pass
    
    def _should_ignore_anchor(self, nickname: str) -> bool:
        """检查是否应该忽略该用户的弹幕（主播自己的弹幕）"""
        # 检查是否启用忽略主播弹幕
        if not self.config.get("live", {}).get("ignore_anchor", True):
            return False
        
        # 获取主播昵称
        anchor_name = self.config.get("live", {}).get("anchor_name", "")
        if not anchor_name:
            return False
        
        # 比较昵称（忽略大小写和空格）
        return nickname.strip().lower() == anchor_name.strip().lower()


class ReplyHandlerWithCallback:
    """带回调的回复处理器"""
    
    def __init__(self, tts_engine, audio_player, config, voice_callback=None, danmu_callback=None):
        from src.handler import ReplyHandler
        
        self._handler = ReplyHandler(
            tts_engine=tts_engine,
            audio_player=audio_player,
            config=config
        )
        self._voice_callback = voice_callback
        self._danmu_callback = danmu_callback
    
    async def handle_event(self, event):
        result = await self._handler.handle_event(event)
        
        # 如果有语音播报，通知回调
        if result and hasattr(result, 'voice_text') and result.voice_text:
            if self._voice_callback:
                self._voice_callback(result.voice_text)
        
        # 如果需要发送弹幕，通知回调
        if result and hasattr(result, 'danmu_text') and result.danmu_text:
            if self._danmu_callback:
                self._danmu_callback(result.danmu_text)
        
        return result
    
    async def check_warm_up(self):
        return await self._handler.check_warm_up()
    
    def update_config(self, config):
        self._handler.update_config(config)


class PreviewManager:
    """预览管理器 - 用于试听语音"""
    
    def __init__(self):
        self._thread: Optional[threading.Thread] = None
        self._stop = False
    
    def preview(self, engine_type: str, params: Dict[str, Any], volume: float,
                on_finished: Callable = None, on_error: Callable[[str], None] = None):
        """预览语音"""
        self._stop = False
        self._thread = threading.Thread(
            target=self._run_preview,
            args=(engine_type, params, volume, on_finished, on_error),
            daemon=True
        )
        self._thread.start()
    
    def stop(self):
        """停止预览"""
        self._stop = True
        try:
            import pygame
            pygame.mixer.music.stop()
        except:
            pass
    
    def _run_preview(self, engine_type, params, volume, on_finished, on_error):
        import os
        try:
            loop = asyncio.new_event_loop()
            asyncio.set_event_loop(loop)
            
            from src.tts import create_tts_engine
            
            tts = create_tts_engine(
                engine=engine_type,
                output_dir=str(ROOT_DIR / "output"),
                **params
            )
            
            async def synthesize():
                return await tts.synthesize("欢迎来到直播间")
            
            temp_path = loop.run_until_complete(synthesize())
            
            if temp_path and os.path.exists(temp_path) and not self._stop:
                try:
                    import pygame
                    pygame.mixer.init()
                    pygame.mixer.music.set_volume(volume)
                    pygame.mixer.music.load(temp_path)
                    pygame.mixer.music.play()
                    while pygame.mixer.music.get_busy() and not self._stop:
                        import time
                        time.sleep(0.1)
                    pygame.mixer.music.stop()
                except Exception as e:
                    if on_error:
                        on_error(f"播放失败: {e}")
            
            loop.close()
            if on_finished:
                on_finished()
                
        except Exception as e:
            if on_error:
                on_error(str(e))
