# -*- coding: utf-8 -*-
"""环境检测服务"""

import os
import sys
import subprocess
from pathlib import Path
from typing import Callable, Optional
from dataclasses import dataclass
import threading


@dataclass
class EnvStatus:
    """环境状态"""
    npcap_installed: bool = False
    companion_installed: bool = False
    companion_path: Optional[str] = None
    obs_installed: bool = False
    obs_path: Optional[str] = None


class EnvironmentChecker:
    """环境检测器"""
    
    def __init__(self):
        self.status = EnvStatus()
        self._callbacks = []
    
    def on_status_change(self, callback: Callable[[EnvStatus], None]):
        """注册状态变化回调"""
        self._callbacks.append(callback)
    
    def _notify(self):
        """通知状态变化"""
        for cb in self._callbacks:
            try:
                cb(self.status)
            except:
                pass
    
    def check_all_async(self):
        """异步检测所有环境"""
        threading.Thread(target=self._check_all, daemon=True).start()
    
    def _check_all(self):
        """检测所有环境"""
        self.check_npcap()
        self.check_companion()
        self.check_obs()
        self._notify()
    
    def check_npcap(self) -> bool:
        """检测 Npcap 是否安装"""
        try:
            npcap_path = Path(os.environ.get('SystemRoot', r'C:\Windows')) / 'System32' / 'Npcap'
            if npcap_path.exists():
                self.status.npcap_installed = True
            else:
                # 检查注册表
                import winreg
                try:
                    key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, r"SOFTWARE\Npcap")
                    winreg.CloseKey(key)
                    self.status.npcap_installed = True
                except:
                    self.status.npcap_installed = False
        except:
            self.status.npcap_installed = False
        
        return self.status.npcap_installed
    
    def check_companion(self) -> bool:
        """检测直播伴侣是否安装"""
        try:
            import os
            # 原系统检测路径
            companion_paths = [
                r"D:\soft\zhibobanlv\webcast_mate",
                r"C:\Program Files\webcast_mate",
                r"C:\Program Files (x86)\webcast_mate",
                os.path.expanduser(r"~\AppData\Local\webcast_mate"),
            ]
            
            for base_path in companion_paths:
                if os.path.isdir(base_path):
                    # 查找子目录中的 exe
                    try:
                        for item in os.listdir(base_path):
                            item_path = os.path.join(base_path, item)
                            if os.path.isdir(item_path):
                                exe = os.path.join(item_path, "直播伴侣.exe")
                                if os.path.exists(exe):
                                    self.status.companion_installed = True
                                    self.status.companion_path = exe
                                    return True
                            # 直接在目录下
                            exe = os.path.join(base_path, "直播伴侣.exe")
                            if os.path.exists(exe):
                                self.status.companion_installed = True
                                self.status.companion_path = exe
                                return True
                    except:
                        pass
            
            self.status.companion_installed = False
            return False
            
        except:
            self.status.companion_installed = False
            return False
    
    def check_obs(self) -> bool:
        """检测 OBS 是否安装"""
        try:
            # 使用 ObsManager 检测
            import sys
            from pathlib import Path
            ROOT_DIR = Path(__file__).parent.parent.parent.parent
            sys.path.insert(0, str(ROOT_DIR))
            
            try:
                from src.stream import ObsManager
                obs_manager = ObsManager(logger=lambda msg: None)
                if obs_manager.is_installed:
                    self.status.obs_installed = True
                    self.status.obs_path = getattr(obs_manager, '_obs_path', None)
                    return True
            except:
                pass
            
            # 备选：常见安装路径
            possible_paths = [
                Path(r"C:\Program Files\obs-studio\bin\64bit\obs64.exe"),
                Path(r"C:\Program Files (x86)\obs-studio\bin\64bit\obs64.exe"),
                Path(os.environ.get('LOCALAPPDATA', '')) / "Programs" / "obs-studio" / "bin" / "64bit" / "obs64.exe",
            ]
            
            for path in possible_paths:
                if path.exists():
                    self.status.obs_installed = True
                    self.status.obs_path = str(path)
                    return True
            
            self.status.obs_installed = False
            return False
            
        except:
            self.status.obs_installed = False
            return False
    
    def set_companion_path(self, path: str) -> bool:
        """手动设置直播伴侣路径"""
        if Path(path).exists():
            self.status.companion_installed = True
            self.status.companion_path = path
            self._notify()
            return True
        return False
