# -*- coding: utf-8 -*-
"""配置管理服务"""

from pathlib import Path
from typing import Dict, Any, Optional
import sys
import shutil

# 获取项目根目录（用于读取默认配置模板）
ROOT_DIR = Path(__file__).parent.parent.parent.parent
if getattr(sys, 'frozen', False):
    ROOT_DIR = Path(sys._MEIPASS)

sys.path.insert(0, str(ROOT_DIR))

from src.utils.config_manager import config as global_config, load_config

# 用户配置目录（敏感信息存这里，不会被打包）
USER_CONFIG_DIR = Path.home() / ".douyin_live_assistant"
USER_CONFIG_FILE = USER_CONFIG_DIR / "config.yaml"

# 默认配置模板路径（打包时只包含模板）
DEFAULT_CONFIG_FILE = ROOT_DIR / "config" / "config.yaml"


class ConfigManager:
    """配置管理器"""
    
    def __init__(self):
        self.config: Dict[str, Any] = {}
        # 用户配置存储在用户目录，而非项目目录
        self.config_path = USER_CONFIG_FILE
        # 确保用户配置目录存在
        USER_CONFIG_DIR.mkdir(parents=True, exist_ok=True)
    
    def load(self) -> Dict[str, Any]:
        """加载配置
        
        优先级：
        1. 用户目录的配置文件 (~/.douyin_live_assistant/config.yaml)
        2. 如果不存在，从默认模板复制
        """
        try:
            if not self.config_path.parent.exists():
                self.config_path.parent.mkdir(parents=True, exist_ok=True)
            
            # 如果用户配置不存在，从默认模板复制
            if not self.config_path.exists() and DEFAULT_CONFIG_FILE.exists():
                shutil.copy(DEFAULT_CONFIG_FILE, self.config_path)
                print(f"[Config] 已从模板创建用户配置: {self.config_path}")
            
            if self.config_path.exists():
                loaded = load_config(str(self.config_path))
                self.config = loaded.config if hasattr(loaded, 'config') else loaded
                print(f"[Config] 已加载用户配置: {self.config_path}")
            
            return self.config
        except Exception as e:
            print(f"[Config] 加载失败: {e}")
            return {}
    
    def save(self) -> bool:
        """保存配置"""
        try:
            if not self.config_path.parent.exists():
                self.config_path.parent.mkdir(parents=True, exist_ok=True)
            
            global_config._config.update(self.config)
            global_config.save(str(self.config_path))
            return True
        except:
            return False
    
    # ========== 直播配置 ==========
    
    def get_room_id(self) -> str:
        return self.config.get("live", {}).get("room_id", "")
    
    def set_room_id(self, room_id: str):
        if "live" not in self.config:
            self.config["live"] = {}
        self.config["live"]["room_id"] = room_id
    
    def get_mode(self) -> str:
        return self.config.get("live", {}).get("mode", "direct")
    
    def set_mode(self, mode: str):
        if "live" not in self.config:
            self.config["live"] = {}
        self.config["live"]["mode"] = mode
    
    # ========== TTS 配置 ==========
    
    def get_tts_engine(self) -> str:
        return self.config.get("tts", {}).get("engine", "edge")
    
    def set_tts_engine(self, engine: str):
        if "tts" not in self.config:
            self.config["tts"] = {}
        self.config["tts"]["engine"] = engine
    
    def get_edge_voice(self) -> str:
        return self.config.get("tts", {}).get("edge", {}).get("voice", "zh-CN-XiaoxiaoNeural")
    
    def set_edge_voice(self, voice: str):
        if "tts" not in self.config:
            self.config["tts"] = {}
        if "edge" not in self.config["tts"]:
            self.config["tts"]["edge"] = {}
        self.config["tts"]["edge"]["voice"] = voice
    
    def get_edge_rate(self) -> str:
        return self.config.get("tts", {}).get("edge", {}).get("rate", "+0%")
    
    def set_edge_rate(self, rate: str):
        if "tts" not in self.config:
            self.config["tts"] = {}
        if "edge" not in self.config["tts"]:
            self.config["tts"]["edge"] = {}
        self.config["tts"]["edge"]["rate"] = rate
    
    def get_volc_config(self) -> Dict[str, str]:
        """获取火山引擎配置"""
        volc = self.config.get("tts", {}).get("volcengine", {})
        return {
            "app_id": volc.get("app_id", ""),
            "access_token": volc.get("access_token", ""),
            "voice_type": volc.get("voice_type", "BV001_streaming"),
        }
    
    def set_volc_config(self, app_id: str, access_token: str, voice_type: str):
        """设置火山引擎配置"""
        if "tts" not in self.config:
            self.config["tts"] = {}
        if "volcengine" not in self.config["tts"]:
            self.config["tts"]["volcengine"] = {}
        self.config["tts"]["volcengine"]["app_id"] = app_id
        self.config["tts"]["volcengine"]["access_token"] = access_token
        self.config["tts"]["volcengine"]["voice_type"] = voice_type
    
    def get_volume(self) -> float:
        return self.config.get("audio", {}).get("volume", 0.8)
    
    def set_volume(self, volume: float):
        if "audio" not in self.config:
            self.config["audio"] = {}
        self.config["audio"]["volume"] = volume
    
    # ========== AI 配置 ==========
    
    def get_ai_enabled(self) -> bool:
        return self.config.get("ai", {}).get("enabled", False)
    
    def set_ai_enabled(self, enabled: bool):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["enabled"] = enabled
    
    def get_ai_provider(self) -> str:
        return self.config.get("ai", {}).get("provider", "deepseek")
    
    def set_ai_provider(self, provider: str):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["provider"] = provider
    
    def get_ai_model(self) -> str:
        return self.config.get("ai", {}).get("model", "deepseek-chat")
    
    def set_ai_model(self, model: str):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["model"] = model
    
    def get_ai_api_key(self) -> str:
        return self.config.get("ai", {}).get("api_key", "")
    
    def set_ai_api_key(self, key: str):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["api_key"] = key
    
    def get_ai_prompt(self) -> str:
        return self.config.get("ai", {}).get("system_prompt", "")
    
    def set_ai_prompt(self, prompt: str):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["system_prompt"] = prompt
    
    def get_ai_reply_mode(self) -> str:
        return self.config.get("ai", {}).get("reply_mode", "tts")
    
    def set_ai_reply_mode(self, mode: str):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["reply_mode"] = mode
    
    def get_ai_reply_rate(self) -> float:
        return self.config.get("ai", {}).get("reply_rate", 1.0)
    
    def set_ai_reply_rate(self, rate: float):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["reply_rate"] = rate
    
    # ========== 暖场配置 ==========
    
    def get_warm_enabled(self) -> bool:
        return self.config.get("ai", {}).get("warm_enabled", False)
    
    def set_warm_enabled(self, enabled: bool):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["warm_enabled"] = enabled
    
    def get_warm_interval(self) -> int:
        return self.config.get("ai", {}).get("warm_interval", 60)
    
    def set_warm_interval(self, interval: int):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["warm_interval"] = interval
    
    def get_warm_category(self) -> str:
        return self.config.get("ai", {}).get("warm_category", "random")
    
    def set_warm_category(self, category: str):
        if "ai" not in self.config:
            self.config["ai"] = {}
        self.config["ai"]["warm_category"] = category
    
    # ========== 回复规则配置 ==========
    
    def get_reply_config(self) -> Dict[str, Any]:
        return self.config.get("reply", {
            "entrance_enabled": True,
            "entrance_cd": 3,
            "gift_enabled": True,
            "gift_cd": 1,
            "follow_enabled": True,
            "follow_cd": 2,
            "read_chat_enabled": True,
        })
    
    def set_reply_config(self, reply_config: Dict[str, Any]):
        self.config["reply"] = reply_config
    
    # ========== 弹幕配置 ==========
    
    def get_chrome_path(self) -> str:
        return self.config.get("danmu", {}).get("chrome_path", "")
    
    def set_chrome_path(self, path: str):
        if "danmu" not in self.config:
            self.config["danmu"] = {}
        self.config["danmu"]["chrome_path"] = path
    
    # ========== 直播伴侣配置 ==========
    
    def get_companion_path(self) -> str:
        return self.config.get("stream", {}).get("companion_path", "")
    
    def set_companion_path(self, path: str):
        if "stream" not in self.config:
            self.config["stream"] = {}
        self.config["stream"]["companion_path"] = path
    
    # ========== 构建完整配置 ==========
    
    def get_anchor_name(self) -> str:
        return self.config.get("live", {}).get("anchor_name", "")
    
    def set_anchor_name(self, name: str):
        if "live" not in self.config:
            self.config["live"] = {}
        self.config["live"]["anchor_name"] = name
    
    def get_ignore_anchor(self) -> bool:
        return self.config.get("live", {}).get("ignore_anchor", True)
    
    def set_ignore_anchor(self, ignore: bool):
        if "live" not in self.config:
            self.config["live"] = {}
        self.config["live"]["ignore_anchor"] = ignore
    
    # ========== 变声器配置 ==========
    
    def get_voice_changer_config(self) -> Dict[str, Any]:
        """获取变声器配置"""
        return self.config.get("voice_changer", {})
    
    def set_voice_changer_config(self, config: Dict[str, Any]):
        """设置变声器配置"""
        self.config["voice_changer"] = config
    
    def get_vc_last_model(self) -> str:
        """获取上次使用的模型"""
        return self.config.get("voice_changer", {}).get("last_model", "")
    
    def set_vc_last_model(self, model_name: str):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["last_model"] = model_name
    
    def get_vc_pitch(self) -> int:
        return self.config.get("voice_changer", {}).get("pitch", 0)
    
    def set_vc_pitch(self, pitch: int):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["pitch"] = pitch
    
    def get_vc_index_rate(self) -> float:
        return self.config.get("voice_changer", {}).get("index_rate", 0.0)
    
    def set_vc_index_rate(self, rate: float):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["index_rate"] = rate
    
    def get_vc_rms_mix(self) -> float:
        return self.config.get("voice_changer", {}).get("rms_mix", 0.0)
    
    def set_vc_rms_mix(self, rate: float):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["rms_mix"] = rate
    
    def get_vc_f0method(self) -> str:
        return self.config.get("voice_changer", {}).get("f0method", "fcpe")
    
    def set_vc_f0method(self, method: str):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["f0method"] = method
    
    def get_vc_input_device(self) -> str:
        return self.config.get("voice_changer", {}).get("input_device", "")
    
    def set_vc_input_device(self, device: str):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["input_device"] = device
    
    def get_vc_output_device(self) -> str:
        return self.config.get("voice_changer", {}).get("output_device", "")
    
    def set_vc_output_device(self, device: str):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["output_device"] = device
    
    def get_vc_hostapi(self) -> str:
        return self.config.get("voice_changer", {}).get("hostapi", "")
    
    def set_vc_hostapi(self, hostapi: str):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["hostapi"] = hostapi
    
    def get_vc_noise_reduce(self) -> Dict[str, bool]:
        return self.config.get("voice_changer", {}).get("noise_reduce", {"input": False, "output": False})
    
    def set_vc_noise_reduce(self, input_nr: bool, output_nr: bool):
        if "voice_changer" not in self.config:
            self.config["voice_changer"] = {}
        self.config["voice_changer"]["noise_reduce"] = {"input": input_nr, "output": output_nr}
    
    # ========== 防检测配置 ==========
    
    def get_anti_detect_config(self) -> Dict[str, Any]:
        """获取防检测配置"""
        return self.config.get("anti_detect", {
            "visual_enabled": False,
            "visual_strength": "medium",
            "typing_enabled": False,
            "typing_speed": "normal",
        })
    
    def set_anti_detect_config(self, config: Dict[str, Any]):
        """设置防检测配置"""
        self.config["anti_detect"] = config
    
    def get_visual_enabled(self) -> bool:
        return self.config.get("anti_detect", {}).get("visual_enabled", False)
    
    def set_visual_enabled(self, enabled: bool):
        if "anti_detect" not in self.config:
            self.config["anti_detect"] = {}
        self.config["anti_detect"]["visual_enabled"] = enabled
    
    def get_visual_strength(self) -> str:
        return self.config.get("anti_detect", {}).get("visual_strength", "medium")
    
    def set_visual_strength(self, strength: str):
        if "anti_detect" not in self.config:
            self.config["anti_detect"] = {}
        self.config["anti_detect"]["visual_strength"] = strength
    
    def get_typing_enabled(self) -> bool:
        return self.config.get("anti_detect", {}).get("typing_enabled", False)
    
    def set_typing_enabled(self, enabled: bool):
        if "anti_detect" not in self.config:
            self.config["anti_detect"] = {}
        self.config["anti_detect"]["typing_enabled"] = enabled
    
    def get_typing_speed(self) -> str:
        return self.config.get("anti_detect", {}).get("typing_speed", "normal")
    
    def set_typing_speed(self, speed: str):
        if "anti_detect" not in self.config:
            self.config["anti_detect"] = {}
        self.config["anti_detect"]["typing_speed"] = speed
    
    # ========== 虚拟形象配置 ==========
    
    def get_avatar_config(self) -> Dict[str, Any]:
        """获取虚拟形象配置"""
        return self.config.get("avatar", {
            "enabled": False,
            "model": "Hiyori",
            "desktop_mode": False,
        })
    
    def set_avatar_config(self, config: Dict[str, Any]):
        """设置虚拟形象配置"""
        self.config["avatar"] = config
    
    def get_avatar_enabled(self) -> bool:
        return self.config.get("avatar", {}).get("enabled", False)
    
    def set_avatar_enabled(self, enabled: bool):
        if "avatar" not in self.config:
            self.config["avatar"] = {}
        self.config["avatar"]["enabled"] = enabled
    
    def get_avatar_model(self) -> str:
        return self.config.get("avatar", {}).get("model", "Hiyori")
    
    def set_avatar_model(self, model: str):
        if "avatar" not in self.config:
            self.config["avatar"] = {}
        self.config["avatar"]["model"] = model
    
    def get_avatar_desktop_mode(self) -> bool:
        return self.config.get("avatar", {}).get("desktop_mode", False)
    
    def set_avatar_desktop_mode(self, enabled: bool):
        if "avatar" not in self.config:
            self.config["avatar"] = {}
        self.config["avatar"]["desktop_mode"] = enabled
    
    # ========== 弹幕发送配置 ==========
    
    def get_chrome_path(self) -> str:
        """获取 Chrome 浏览器路径"""
        return self.config.get("danmu", {}).get("chrome_path", "")
    
    def set_chrome_path(self, path: str):
        """设置 Chrome 浏览器路径"""
        if "danmu" not in self.config:
            self.config["danmu"] = {}
        self.config["danmu"]["chrome_path"] = path
    
    def build_full_config(self) -> Dict[str, Any]:
        """构建完整配置供 Worker 使用"""
        return {
            "live": {
                "room_id": self.get_room_id(),
                "mode": self.get_mode(),
                "anchor_name": self.get_anchor_name(),
                "ignore_anchor": self.get_ignore_anchor(),
            },
            "tts": {
                "engine": self.get_tts_engine(),
                "edge": {
                    "voice": self.get_edge_voice(),
                    "rate": self.get_edge_rate(),
                },
            },
            "audio": {
                "volume": self.get_volume(),
            },
            "ai": {
                "enabled": self.get_ai_enabled(),
                "provider": self.get_ai_provider(),
                "model": self.get_ai_model(),
                "api_key": self.get_ai_api_key(),
                "system_prompt": self.get_ai_prompt(),
                "reply_mode": self.get_ai_reply_mode(),
                "reply_rate": self.get_ai_reply_rate(),
                "warm_enabled": self.get_warm_enabled(),
                "warm_interval": self.get_warm_interval(),
                "warm_category": self.get_warm_category(),
            },
            "reply": self.get_reply_config(),
        }
