# -*- coding: utf-8 -*-
"""右侧弹幕面板"""

import flet as ft
from typing import Callable
from datetime import datetime
from ..themes import theme


class RightPanel(ft.Container):
    """右侧弹幕面板"""
    
    def __init__(self, on_float_window: Callable = None, on_send_danmu: Callable = None):
        self._on_float_window = on_float_window
        self._on_send_danmu = on_send_danmu
        
        c = theme.colors
        
        # 消息列表
        self.msg_list = ft.ListView(
            expand=True,
            spacing=2,
            auto_scroll=True,
        )
        
        # 发送输入框
        self.danmu_input = ft.TextField(
            hint_text="输入弹幕...",
            hint_style=ft.TextStyle(color=c.text_muted, size=12),
            border=ft.InputBorder.OUTLINE,
            border_color=c.border_default,
            focused_border_color=c.accent,
            bgcolor=c.bg_elevated,
            text_size=12,
            color=c.text_primary,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
            expand=True,
            on_submit=self._handle_send,
        )
        
        self.send_btn = ft.IconButton(
            icon=ft.Icons.SEND_ROUNDED,
            icon_color=c.accent,
            icon_size=18,
            on_click=self._handle_send,
        )
        
        # 添加欢迎消息
        self.add_message("系统", "欢迎使用梧桐抖音运营伴侣")
        
        super().__init__(
            content=ft.Column([
                ft.Row([
                    ft.Text("消息", size=13, weight=ft.FontWeight.W_500, color=c.text_secondary),
                    ft.Container(expand=True),
                    ft.TextButton(
                        "清空",
                        on_click=self._clear_messages,
                        style=ft.ButtonStyle(color=c.text_muted, padding=ft.padding.all(4)),
                    ),
                    ft.TextButton(
                        "独立窗口",
                        on_click=self._handle_float,
                        style=ft.ButtonStyle(color=c.accent, padding=ft.padding.all(4)),
                    ),
                ], spacing=0),
                ft.Divider(height=1, color=c.border_muted),
                self.msg_list,
                ft.Divider(height=1, color=c.border_muted),
                # 发送弹幕区域
                ft.Row([self.danmu_input, self.send_btn], spacing=4),
            ], spacing=8, expand=True),
            width=280,
            bgcolor=c.bg_surface,
            padding=12,
            border=ft.border.only(left=ft.BorderSide(1, c.border_default)),
        )
    
    def _handle_float(self, e):
        if self._on_float_window:
            self._on_float_window()
    
    def _handle_send(self, e):
        """发送弹幕"""
        text = self.danmu_input.value.strip()
        if text and self._on_send_danmu:
            self._on_send_danmu(text)
            self.danmu_input.value = ""
            self.danmu_input.update()
    
    def _clear_messages(self, e):
        self.msg_list.controls.clear()
        self.add_message("系统", "消息已清空")
        self.update()
    
    def add_message(self, user: str, content: str, msg_type: str = "chat"):
        """添加消息"""
        c = theme.colors
        
        # 根据类型选择颜色
        type_colors = {
            "entrance": c.success,
            "gift": c.warning,
            "follow": c.accent,
            "chat": c.accent,
            "system": c.text_muted,
        }
        user_color = type_colors.get(msg_type, c.accent)
        
        # 类型图标
        type_icons = {
            "entrance": "→",
            "gift": "🎁",
            "follow": "❤",
            "chat": "",
            "system": "●",
        }
        icon = type_icons.get(msg_type, "")
        
        self.msg_list.controls.append(
            ft.Container(
                content=ft.Row([
                    ft.Text(icon, size=10, color=user_color) if icon else ft.Container(width=0),
                    ft.Text(f"{user}:", size=12, color=user_color, weight=ft.FontWeight.W_500),
                    ft.Text(content, size=12, color=c.text_secondary),
                ], spacing=4, wrap=True),
                padding=ft.padding.symmetric(vertical=4, horizontal=8),
            )
        )
    
    def add_entrance(self, user: str):
        """入场消息"""
        self.add_message(user, "进入直播间", "entrance")
    
    def add_gift(self, user: str, gift_name: str, count: int):
        """礼物消息"""
        self.add_message(user, f"送出 {gift_name}x{count}", "gift")
    
    def add_follow(self, user: str):
        """关注消息"""
        self.add_message(user, "关注了你", "follow")
    
    def add_chat(self, user: str, content: str):
        """聊天消息"""
        self.add_message(user, content, "chat")
    
    def add_system(self, content: str):
        """系统消息 - 只写入日志文件，不显示在界面"""
        import sys
        from datetime import datetime
        from pathlib import Path
        
        try:
            # 确定日志目录
            if getattr(sys, 'frozen', False):
                # 打包后的环境：使用可执行文件所在目录
                if hasattr(sys, '_MEIPASS'):
                    # PyInstaller 目录模式
                    log_dir = Path(sys.executable).parent / "_internal" / "logs"
                else:
                    log_dir = Path(sys.executable).parent / "logs"
            else:
                # 开发环境
                log_dir = Path(__file__).parent.parent.parent.parent / "logs"
            
            log_dir.mkdir(parents=True, exist_ok=True)
            log_file = log_dir / "system.log"
            
            # 写入日志
            timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            with open(log_file, "a", encoding="utf-8") as f:
                f.write(f"[{timestamp}] {content}\n")
        except Exception as e:
            # 日志写入失败不影响主功能
            print(f"Log write failed: {e}")
