# -*- coding: utf-8 -*-
"""
Flet UI 入口 - 统一使用无边框窗口
"""

import sys
import threading
from pathlib import Path
import flet as ft
from .services.auth_service import get_auth_service
from src.core.startup_update import check_update_on_startup, get_update_url_from_config

# 获取根目录
ROOT_DIR = Path(__file__).parent.parent.parent
if getattr(sys, 'frozen', False):
    ROOT_DIR = Path(sys._MEIPASS)


def main(page: ft.Page):
    # 设置统一字体
    page.fonts = {
        "default": "Microsoft YaHei",  # 微软雅黑
    }
    page.theme = ft.Theme(font_family="Microsoft YaHei")
    
    # 设置页面基础属性
    page.title = "梧桐抖音运营伴侣"
    page.padding = 0
    page.spacing = 0
    page.bgcolor = "#1a1a2e"  # 深色背景
    
    # 设置窗口图标路径
    icon_path = ROOT_DIR / "static" / "icons" / "icon.ico"
    
    def set_login_window():
        """设置登录窗口属性 - 小窗口模式"""
        if hasattr(page, 'window') and hasattr(page.window, 'width'):
            page.window.width = 450
            page.window.height = 520
            page.window.min_width = 400
            page.window.min_height = 480
            page.window.center()
            page.window.frameless = True
            page.window.title_bar_hidden = True
            if icon_path.exists():
                page.window.icon = str(icon_path)
        else:
            page.window_width = 450
            page.window_height = 520
            page.window_min_width = 400
            page.window_min_height = 480
            page.window_center()
            page.window_frameless = True
            page.window_title_bar_hidden = True
            if icon_path.exists():
                page.window_icon = str(icon_path)
    
    def set_main_window():
        """设置主应用窗口属性 - 大窗口模式"""
        if hasattr(page, 'window') and hasattr(page.window, 'width'):
            page.window.width = 1200
            page.window.height = 850
            page.window.min_width = 1000
            page.window.min_height = 750
            page.window.center()
            page.window.frameless = True
            page.window.title_bar_hidden = True
            if icon_path.exists():
                page.window.icon = str(icon_path)
        else:
            page.window_width = 1200
            page.window_height = 850
            page.window_min_width = 1000
            page.window_min_height = 750
            page.window_center()
            page.window_frameless = True
            page.window_title_bar_hidden = True
            if icon_path.exists():
                page.window_icon = str(icon_path)
    
    def show_loading():
        """显示加载界面"""
        set_main_window()  # 先设置窗口大小
        
        loading_view = ft.Container(
            content=ft.Column([
                ft.ProgressRing(width=40, height=40, stroke_width=3, color="#4fc3f7"),
                ft.Text("正在加载...", size=14, color="#aaaaaa"),
            ], alignment=ft.MainAxisAlignment.CENTER, horizontal_alignment=ft.CrossAxisAlignment.CENTER, spacing=15),
            alignment=ft.alignment.center,
            expand=True,
        )
        page.add(loading_view)
        page.update()
    
    def show_main_app():
        """显示主应用"""
        set_main_window()
        page.controls.clear()
        from .app import MainApp
        MainApp(page)
        page.update()
        
        # 启动时检查更新（从配置文件读取服务器地址）
        update_url = get_update_url_from_config()
        check_update_on_startup(page, update_url)
    
    def show_login_page():
        """显示登录页面 - 小窗口模式"""
        set_login_window()
        page.controls.clear()
        from .login_page import LoginPage
        from .themes import theme
        
        c = theme.colors
        page.bgcolor = c.bg_surface
        
        login_page = LoginPage(page, on_login_success=show_main_app_async)
        page.add(login_page)
        page.update()
    
    def show_main_app_async():
        """异步显示主应用（带加载界面）"""
        show_loading()
        
        def load():
            import time
            time.sleep(0.1)  # 让加载界面先显示
            show_main_app()
        
        threading.Thread(target=load, daemon=True).start()
    
    # 检查登录状态
    auth_service = get_auth_service()
    
    if auth_service.is_logged_in():
        # 已登录 - 直接进入主界面（信任本地缓存，启动更快）
        # token 有效性会在后续 API 调用时自动验证
        show_loading()
        
        def load_main():
            import time
            time.sleep(0.05)  # 让加载界面先渲染
            show_main_app()
        
        threading.Thread(target=load_main, daemon=True).start()
    else:
        show_login_page()


def run():
    """运行应用"""
    ft.app(target=main)


if __name__ == "__main__":
    run()
