# -*- coding: utf-8 -*-
"""
登录注册页面 - Flet UI
参考 wapp/electron-app 的实现逻辑，使用 DouyinLiveAssistant 的主题风格
"""

import flet as ft
import base64
import asyncio
from typing import Callable, Optional

# Flet兼容层 - 支持0.21.x和0.28.x
if not hasattr(ft, 'Icons') and hasattr(ft, 'icons'):
    ft.Icons = ft.icons
if not hasattr(ft, 'Colors') and hasattr(ft, 'colors'):
    ft.Colors = ft.colors

from .services.auth_service import get_auth_service
from .themes import theme


class LoginPage(ft.Container):
    """登录注册页面 - 使用 DouyinLiveAssistant 主题风格"""
    
    def __init__(self, page: ft.Page, on_login_success: Callable[[], None]):
        """
        Args:
            page: Flet 页面对象
            on_login_success: 登录成功回调
        """
        super().__init__()
        self.page = page
        self.on_login_success = on_login_success
        self.auth_service = get_auth_service()
        
        # 验证码相关状态
        self._pic_captcha_reqno = ""
        self._email_code_reqno = ""
        self._countdown = 0
        self._countdown_task = None
        
        # 构建 UI
        self._build_ui()
        
        # 加载图片验证码
        self.page.run_task(self._load_pic_captcha)
    
    def _build_ui(self):
        """构建登录界面 - 使用 DouyinLiveAssistant 主题"""
        c = theme.colors
        
        # 邮箱输入
        self.email_input = ft.TextField(
            label="邮箱",
            hint_text="请输入邮箱",
            prefix_icon=ft.Icons.EMAIL,
            keyboard_type=ft.KeyboardType.EMAIL,
            width=320,
            border_color=c.border_default,
            focused_border_color=c.accent,
            label_style=ft.TextStyle(color=c.text_muted),
            hint_style=ft.TextStyle(color=c.text_placeholder),
            text_style=ft.TextStyle(color=c.text_primary),
            cursor_color=c.accent,
        )
        
        # 图片验证码输入
        self.pic_captcha_input = ft.TextField(
            label="图片验证码",
            hint_text="请输入验证码",
            prefix_icon=ft.Icons.SECURITY,
            width=180,
            border_color=c.border_default,
            focused_border_color=c.accent,
            label_style=ft.TextStyle(color=c.text_muted),
            hint_style=ft.TextStyle(color=c.text_placeholder),
            text_style=ft.TextStyle(color=c.text_primary),
            cursor_color=c.accent,
        )
        
        # 图片验证码图片
        self.captcha_image = ft.Image(
            src_base64="",
            width=120,
            height=40,
            fit=ft.ImageFit.CONTAIN,
            tooltip="点击刷新",
        )
        self.captcha_container = ft.Container(
            content=self.captcha_image,
            on_click=lambda _: self.page.run_task(self._load_pic_captcha),
            border_radius=5,
            bgcolor="#ffffff",
            padding=2,
            border=ft.border.all(1, c.border_default),
        )
        
        # 邮箱验证码输入
        self.email_code_input = ft.TextField(
            label="邮箱验证码",
            hint_text="请输入验证码",
            prefix_icon=ft.Icons.VERIFIED,
            width=180,
            border_color=c.border_default,
            focused_border_color=c.accent,
            label_style=ft.TextStyle(color=c.text_muted),
            hint_style=ft.TextStyle(color=c.text_placeholder),
            text_style=ft.TextStyle(color=c.text_primary),
            cursor_color=c.accent,
            on_submit=lambda _: self.page.run_task(self._do_login),
        )
        
        # 发送验证码按钮
        self.send_code_btn = ft.ElevatedButton(
            text="发送验证码",
            on_click=lambda _: self.page.run_task(self._send_email_code),
            width=120,
            height=48,
            bgcolor=c.bg_elevated,
            color=c.text_primary,
            style=ft.ButtonStyle(
                shape=ft.RoundedRectangleBorder(radius=8),
                side=ft.BorderSide(1, c.border_default),
            ),
        )
        
        # 错误提示
        self.error_text = ft.Text(
            value="",
            color=c.error,
            size=13,
            visible=False,
        )
        
        # 登录按钮
        self.login_btn = ft.ElevatedButton(
            text="登录 / 注册",
            icon=ft.Icons.LOGIN,
            on_click=lambda _: self.page.run_task(self._do_login),
            width=320,
            height=48,
            bgcolor=c.accent,
            color="#ffffff",
            style=ft.ButtonStyle(
                shape=ft.RoundedRectangleBorder(radius=8),
            ),
        )
        
        # 提示文字
        tip_text = ft.Text(
            value="新用户自动注册，老用户直接登录",
            size=12,
            color=c.text_muted,
        )
        
        # 登录表单
        login_form = ft.Column(
            controls=[
                # 标题
                ft.Container(
                    content=ft.Column([
                        ft.Row([
                            ft.Icon(ft.Icons.LIVE_TV_ROUNDED, size=36, color=c.accent),
                            ft.Text("梧桐抖音运营伴侣", size=24, weight=ft.FontWeight.W_600, color=c.text_primary),
                        ], alignment=ft.MainAxisAlignment.CENTER, spacing=10),
                        ft.Text("企业级直播运营工具", size=13, color=c.text_muted),
                    ], horizontal_alignment=ft.CrossAxisAlignment.CENTER, spacing=5),
                    margin=ft.margin.only(bottom=25),
                ),
                
                # 邮箱
                self.email_input,
                ft.Container(height=12),
                
                # 图片验证码
                ft.Row([
                    self.pic_captcha_input,
                    self.captcha_container,
                ], alignment=ft.MainAxisAlignment.CENTER, spacing=10),
                ft.Container(height=12),
                
                # 邮箱验证码
                ft.Row([
                    self.email_code_input,
                    self.send_code_btn,
                ], alignment=ft.MainAxisAlignment.CENTER, spacing=10),
                ft.Container(height=8),
                
                # 错误提示
                self.error_text,
                ft.Container(height=15),
                
                # 登录按钮
                self.login_btn,
                ft.Container(height=12),
                
                # 提示
                tip_text,
            ],
            horizontal_alignment=ft.CrossAxisAlignment.CENTER,
            spacing=0,
        )
        
        # 关闭按钮
        def on_close(e):
            if hasattr(self.page, 'window') and hasattr(self.page.window, 'close'):
                self.page.window.close()
            elif hasattr(self.page, 'window_close'):
                self.page.window_close()
        
        close_btn = ft.IconButton(
            icon=ft.Icons.CLOSE,
            icon_color=c.text_muted,
            icon_size=18,
            tooltip="关闭",
            on_click=on_close,
        )
        
        # 标题栏（用于拖动窗口）
        title_bar = ft.Container(
            content=ft.Row([
                ft.Container(expand=True),  # 占位
                close_btn,
            ], alignment=ft.MainAxisAlignment.END),
            padding=ft.padding.only(right=5, top=5),
        )
        
        # 设置页面内容 - 小窗口模式
        self.content = ft.Container(
            content=ft.Column([
                ft.WindowDragArea(
                    content=title_bar,
                    maximizable=False,
                ),
                ft.Container(
                    content=login_form,
                    padding=ft.padding.only(left=35, right=35, bottom=30),
                ),
            ], spacing=0),
            expand=True,
            bgcolor=c.bg_surface,
            border_radius=12,
        )
        self.expand = True
    
    def _show_error(self, message: str):
        """显示错误信息"""
        self.error_text.value = message
        self.error_text.visible = True
        self.page.update()
        
        # 5秒后自动隐藏
        async def hide_error():
            await asyncio.sleep(5)
            self.error_text.visible = False
            self.page.update()
        
        self.page.run_task(hide_error)
    
    async def _load_pic_captcha(self):
        """加载图片验证码"""
        try:
            success, result = self.auth_service.get_pic_captcha()
            if success and result:
                # 提取 base64 数据（去掉 data:image/png;base64, 前缀）
                base64_str = result.get('validCodeBase64', '')
                if base64_str.startswith('data:'):
                    base64_str = base64_str.split(',', 1)[1]
                
                self.captcha_image.src_base64 = base64_str
                self._pic_captcha_reqno = result.get('validCodeReqNo', '')
                self.page.update()
            else:
                self._show_error(f"获取验证码失败: {result}")
        except Exception as e:
            self._show_error(f"获取验证码失败: {str(e)}")
    
    async def _send_email_code(self):
        """发送邮箱验证码"""
        email = self.email_input.value.strip()
        pic_captcha = self.pic_captcha_input.value.strip()
        
        if not email:
            self._show_error("请先输入邮箱")
            return
        if not pic_captcha:
            self._show_error("请先输入图片验证码")
            return
        
        # 禁用按钮
        self.send_code_btn.disabled = True
        self.send_code_btn.text = "发送中..."
        self.page.update()
        
        try:
            success, result = self.auth_service.get_email_valid_code(
                email, pic_captcha, self._pic_captcha_reqno
            )
            
            if success:
                self._email_code_reqno = result
                # 开始倒计时
                self._countdown = 60
                self.page.run_task(self._countdown_timer)
            else:
                self.send_code_btn.disabled = False
                self.send_code_btn.text = "发送验证码"
                self._show_error(result or "发送验证码失败")
                # 如果是验证码错误，刷新图片验证码
                if result and "验证码" in str(result):
                    await self._load_pic_captcha()
                self.page.update()
                
        except Exception as e:
            self.send_code_btn.disabled = False
            self.send_code_btn.text = "发送验证码"
            self._show_error(f"发送失败: {str(e)}")
            self.page.update()
    
    async def _countdown_timer(self):
        """倒计时"""
        while self._countdown > 0:
            self.send_code_btn.text = f"{self._countdown}s"
            self.page.update()
            await asyncio.sleep(1)
            self._countdown -= 1
        
        self.send_code_btn.disabled = False
        self.send_code_btn.text = "发送验证码"
        self.page.update()
    
    async def _do_login(self):
        """执行登录/注册"""
        email = self.email_input.value.strip()
        email_code = self.email_code_input.value.strip()
        
        if not email:
            self._show_error("请输入邮箱")
            return
        if not email_code:
            self._show_error("请输入邮箱验证码")
            return
        
        # 禁用登录按钮
        self.login_btn.disabled = True
        self.login_btn.text = "登录中..."
        self.page.update()
        
        try:
            success, result = self.auth_service.login_or_register(
                email, email_code, self._email_code_reqno
            )
            
            if success:
                # 登录成功，调用回调
                if self.on_login_success:
                    self.on_login_success()
            else:
                self.login_btn.disabled = False
                self.login_btn.text = "登录 / 注册"
                self._show_error(result or "登录失败")
                # 如果是验证码错误，刷新图片验证码
                if result and "验证码" in str(result):
                    await self._load_pic_captcha()
                self.page.update()
                
        except Exception as e:
            self.login_btn.disabled = False
            self.login_btn.text = "登录 / 注册"
            self._show_error(f"登录失败: {str(e)}")
            self.page.update()
