# -*- coding: utf-8 -*-
"""状态指示组件"""

import flet as ft
from ..themes import theme


class StatusIndicator(ft.Container):
    """状态指示器 (圆点 + 文字)"""
    
    def __init__(self, text: str = "未连接", color: str = None):
        c = theme.colors
        self._dot = ft.Icon(ft.Icons.CIRCLE, color=color or c.text_muted, size=10)
        self._text = ft.Text(text, size=12, color=color or c.text_muted)
        
        super().__init__(
            content=ft.Row([self._dot, self._text], spacing=6),
        )
    
    def set_status(self, text: str, color: str):
        """设置状态"""
        self._dot.color = color
        self._text.value = text
        self._text.color = color
    
    def set_connected(self):
        """设置为已连接状态"""
        c = theme.colors
        self.set_status("已连接", c.success)
    
    def set_disconnected(self):
        """设置为未连接状态"""
        c = theme.colors
        self.set_status("未连接", c.text_muted)
    
    def set_connecting(self):
        """设置为连接中状态"""
        c = theme.colors
        self.set_status("连接中...", c.warning)
    
    def set_error(self, msg: str = "错误"):
        """设置为错误状态"""
        c = theme.colors
        self.set_status(msg, c.error)


class StatusChip(ft.Container):
    """状态标签 (带图标)"""
    
    def __init__(self, label: str, status: str = "检测中...", ok: bool = False):
        c = theme.colors
        icon = ft.Icons.CHECK_CIRCLE if ok else ft.Icons.CIRCLE
        color = c.success if ok else c.text_muted
        
        self._icon = ft.Icon(icon, color=color, size=14)
        self._label = ft.Text(f"{label} {status}", size=11, color=c.text_secondary)
        
        super().__init__(
            content=ft.Row([self._icon, self._label], spacing=4),
        )
        
        self._label_text = label
    
    def set_ok(self, status: str = "已安装"):
        """设置为成功状态"""
        c = theme.colors
        self._icon.name = ft.Icons.CHECK_CIRCLE
        self._icon.color = c.success
        self._label.value = f"{self._label_text} {status}"
        self._label.color = c.success
    
    def set_error(self, status: str = "未安装"):
        """设置为错误状态"""
        c = theme.colors
        self._icon.name = ft.Icons.ERROR
        self._icon.color = c.error
        self._label.value = f"{self._label_text} {status}"
        self._label.color = c.error
    
    def set_checking(self):
        """设置为检测中状态"""
        c = theme.colors
        self._icon.name = ft.Icons.CIRCLE
        self._icon.color = c.text_muted
        self._label.value = f"{self._label_text} 检测中..."
        self._label.color = c.text_muted
