# -*- coding: utf-8 -*-
"""个人中心弹窗组件"""

import flet as ft
import time
import threading
import requests
from typing import Callable
from ..themes import theme
from ..services.auth_service import get_auth_service, AuthConfig
from ..services.api_logger import log_request, log_response


class ProfileDialog(ft.AlertDialog):
    """个人中心弹窗"""
    
    def __init__(self, page: ft.Page, on_save: Callable = None, on_close: Callable = None):
        self.page = page
        self._on_save = on_save
        self._on_close = on_close
        self._auth_service = get_auth_service()
        
        c = theme.colors
        user_info = self._auth_service.get_user_info() or {}
        
        username = (user_info.get('nickname') or user_info.get('username') or '').strip() or '用户'
        email = user_info.get('email') or ''
        phone = user_info.get('phone') or ''
        avatar_url = user_info.get('avatar') or ''
        
        # 当前头像 URL（用于保存时发送到服务器）
        self._current_avatar_url = avatar_url
        
        # 头像
        if avatar_url:
            try:
                self.avatar = ft.CircleAvatar(foreground_image_url=avatar_url, radius=24)
            except TypeError:
                try:
                    self.avatar = ft.CircleAvatar(foreground_image_src=avatar_url, radius=24)
                except TypeError:
                    self.avatar = ft.CircleAvatar(content=ft.Text(username[0].upper(), size=16, color="#fff"), radius=24, bgcolor=c.accent)
        else:
            self.avatar = ft.CircleAvatar(
                content=ft.Text(username[0].upper() if username else 'U', size=16, weight=ft.FontWeight.W_600, color="#fff"),
                radius=24, bgcolor=c.accent,
            )
        
        avatar_stack = ft.Stack([
            self.avatar,
            ft.Container(
                content=ft.Icon(ft.Icons.EDIT, size=10, color="#fff"),
                width=16, height=16, border_radius=8, bgcolor=c.accent,
                alignment=ft.alignment.center, right=0, bottom=0,
            ),
        ], width=48, height=48)
        
        self._file_picker = None
        
        # 用户信息行
        user_row = ft.Row([
            ft.Container(content=avatar_stack, on_click=self._on_avatar_click, tooltip="点击修改头像"),
            ft.Column([
                ft.Text(username, size=14, weight=ft.FontWeight.W_500, color=c.text_primary),
                ft.Text(email or "未绑定邮箱", size=11, color=c.text_muted),
            ], spacing=2, alignment=ft.MainAxisAlignment.CENTER),
        ], spacing=12)
        
        # 输入框
        input_style = {
            "border_radius": 6, "filled": True, "bgcolor": c.bg_elevated,
            "border_color": c.border_default, "focused_border_color": c.accent,
            "label_style": ft.TextStyle(color=c.text_muted, size=11),
            "text_style": ft.TextStyle(color=c.text_primary, size=13),
            "content_padding": ft.padding.symmetric(horizontal=10, vertical=8),
            "dense": True,
        }
        
        self.nickname_input = ft.TextField(label="昵称", value=user_info.get('nickname') or '', **input_style)
        self.phone_input = ft.TextField(label="手机号", value=phone, **input_style)
        
        # 消息提示
        self.message_text = ft.Text("", size=11, visible=False)
        
        # 按钮
        btn_row = ft.Row([
            ft.TextButton("取消", on_click=self._on_cancel_click, style=ft.ButtonStyle(color=c.text_muted)),
            ft.OutlinedButton(
                "保存", 
                on_click=self._on_save_click,
                style=ft.ButtonStyle(
                    color=c.accent,
                    side=ft.BorderSide(1, c.accent),
                ),
            ),
        ], alignment=ft.MainAxisAlignment.END, spacing=8)
        
        # 内容 - 紧凑布局
        content = ft.Container(
            content=ft.Column([
                user_row,
                ft.Divider(height=1, color=c.border_muted),
                self.nickname_input,
                self.phone_input,
                self.message_text,
                btn_row,
            ], spacing=10, tight=True),
            width=280, padding=4,
        )
        
        super().__init__(
            modal=True,
            title=ft.Text("个人中心", size=14, weight=ft.FontWeight.W_600, color=c.text_primary),
            content=content,
            bgcolor=c.bg_surface,
            shape=ft.RoundedRectangleBorder(radius=10),
            title_padding=ft.padding.only(left=16, top=12, bottom=6),
            content_padding=ft.padding.only(left=12, right=12, bottom=12),
        )
    
    def _on_avatar_click(self, e):
        """点击头像"""
        try:
            if not self._file_picker:
                self._file_picker = ft.FilePicker(on_result=self._on_file_picked)
                self.page.overlay.append(self._file_picker)
                self.page.update()
            self._file_picker.pick_files(file_type=ft.FilePickerFileType.IMAGE, allow_multiple=False)
        except Exception as exc:
            self._show_message(f"文件选择器错误", is_error=True)

    def _on_file_picked(self, e: ft.FilePickerResultEvent):
        """选择头像文件后上传"""
        if not e.files or not e.files[0].path:
            return
        file_path = e.files[0].path
        self._show_message("头像上传中...", is_error=False)

        def do_upload():
            success, url_or_msg = self._upload_avatar(file_path)
            def update_ui():
                if success and url_or_msg:
                    self._current_avatar_url = url_or_msg  # 更新当前头像 URL
                    try:
                        self.avatar.foreground_image_url = self._current_avatar_url
                    except:
                        try:
                            self.avatar.foreground_image_src = self._current_avatar_url
                        except:
                            pass
                    self._show_message("头像已更新", is_error=False)
                else:
                    self._show_message(url_or_msg or "上传失败", is_error=True)
                self.page.update()
            self.page.run_thread(update_ui)

        threading.Thread(target=do_upload, daemon=True).start()

    def _upload_avatar(self, file_path: str):
        """上传头像到服务器"""
        api_base = AuthConfig.API_BASE_URL
        token = self._auth_service.token or ""
        upload_url = f"{api_base}/dev/file/uploadDynamicReturnUrl"
        headers = {'token': token}
        
        # 记录请求日志
        log_request("POST", upload_url, headers, {"file": f"<binary:{file_path}>"})
        start_time = time.time()
        
        try:
            with open(file_path, 'rb') as f:
                resp = requests.post(upload_url, files={'file': f}, headers=headers, timeout=30)
            
            duration_ms = (time.time() - start_time) * 1000
            
            if resp.status_code != 200:
                error_msg = f"HTTP {resp.status_code}"
                log_response(upload_url, status_code=resp.status_code, success=False, error=error_msg, duration_ms=duration_ms)
                return False, error_msg
            
            data = resp.json()
            if data.get('code') != 200:
                error_msg = data.get('msg', '上传失败')
                log_response(upload_url, status_code=data.get('code'), success=False, error=error_msg, duration_ms=duration_ms)
                return False, error_msg
            
            payload = data.get('data')
            log_response(upload_url, status_code=200, success=True, data=payload, duration_ms=duration_ms)
            
            if isinstance(payload, str):
                return True, payload
            if isinstance(payload, dict):
                return True, payload.get('url') or payload.get('fileUrl') or ""
            return False, "未返回URL"
        except Exception as e:
            duration_ms = (time.time() - start_time) * 1000
            error_msg = str(e)
            log_response(upload_url, success=False, error=error_msg, duration_ms=duration_ms)
            return False, error_msg
    
    def _on_save_click(self, e):
        """保存 - 同时更新服务器和本地缓存"""
        nickname = self.nickname_input.value.strip()
        phone = self.phone_input.value.strip()
        
        if not nickname:
            self._show_message("请输入昵称", is_error=True)
            return
        
        self._show_message("保存中...", is_error=False)
        
        def do_save():
            # 调用服务器接口更新用户信息（使用当前头像 URL）
            success, result = self._auth_service.update_user_info(
                nickname=nickname,
                phone=phone if phone else None,
                avatar=self._current_avatar_url if self._current_avatar_url else None
            )
            
            def update_ui():
                if success:
                    self._show_message("保存成功", is_error=False)
                    
                    if self._on_save:
                        self._on_save({'nickname': nickname, 'phone': phone, 'avatar': self._current_avatar_url})
                    
                    # 延迟关闭
                    def close():
                        import time
                        time.sleep(0.5)
                        self.page.run_thread(self._close_dialog)
                    threading.Thread(target=close, daemon=True).start()
                else:
                    self._show_message(f"保存失败: {result}", is_error=True)
                self.page.update()
            
            self.page.run_thread(update_ui)
        
        threading.Thread(target=do_save, daemon=True).start()
    
    def _on_cancel_click(self, e):
        self._close_dialog()
    
    def _close_dialog(self):
        self.open = False
        self.page.update()
        if self._on_close:
            self._on_close()
    
    def _show_message(self, message: str, is_error: bool = True):
        c = theme.colors
        self.message_text.value = message
        self.message_text.color = c.error if is_error else c.success
        self.message_text.visible = True
        self.page.update()
