# -*- coding: utf-8 -*-
"""独立悬浮消息窗口 - 使用 Flet 实现，保持UI一致"""

import subprocess
import sys
import os
import json
import threading
import tempfile
from datetime import datetime
from typing import Callable, Optional
from pathlib import Path


class FloatingMessageWindow:
    """悬浮消息窗口 - 启动独立 Flet 进程"""
    
    def __init__(self, on_send: Callable = None, on_close: Callable = None):
        self._on_send = on_send
        self._on_close = on_close
        self._process = None
        self._running = False
        self._visible = False
        self._msg_counter = 0
        
        # 创建临时文件用于进程间通信
        self._temp_dir = tempfile.mkdtemp(prefix="flet_float_")
        self._msg_file = os.path.join(self._temp_dir, "messages.json")
        self._cmd_file = os.path.join(self._temp_dir, "commands.json")
        
        # 清空文件
        open(self._msg_file, 'w').close()
        open(self._cmd_file, 'w').close()
    
    def show(self):
        """启动独立窗口进程"""
        if self._running:
            return
        
        self._running = True
        self._visible = True
        
        # 启动独立 Flet 应用
        if getattr(sys, 'frozen', False):
            # 打包后的环境：查找并运行 FloatWindow.exe
            float_exe = None
            
            # 查找 FloatWindow.exe 的可能位置
            search_paths = [
                Path(sys.executable).parent / "FloatWindow.exe",  # 主程序同目录
                Path(sys.executable).parent / "_internal" / "FloatWindow.exe",  # _internal 目录
            ]
            
            if hasattr(sys, '_MEIPASS'):
                search_paths.insert(0, Path(sys._MEIPASS) / "FloatWindow.exe")  # PyInstaller 临时目录
            
            for path in search_paths:
                if path.exists():
                    float_exe = path
                    break
            
            if not float_exe:
                print(f"错误：找不到 FloatWindow.exe，搜索路径：{search_paths}")
                self._running = False
                self._visible = False
                return
            
            cmd = [str(float_exe), self._msg_file, self._cmd_file]
            
            self._process = subprocess.Popen(
                cmd,
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                creationflags=subprocess.CREATE_NO_WINDOW if sys.platform == 'win32' else 0,
            )
        else:
            # 开发环境：使用子进程
            app_path = Path(__file__).parent / "float_window_app.py"
            cmd = [sys.executable, str(app_path), self._msg_file, self._cmd_file]
            
            self._process = subprocess.Popen(
                cmd,
                stdout=subprocess.DEVNULL,
                stderr=subprocess.DEVNULL,
                creationflags=subprocess.CREATE_NO_WINDOW if sys.platform == 'win32' else 0,
            )
        
        # 启动命令监听线程
        self._cmd_thread = threading.Thread(target=self._poll_commands, daemon=True)
        self._cmd_thread.start()
    
    def _poll_commands(self):
        """监听来自独立窗口的命令"""
        import time
        last_content = ""
        
        while self._running:
            try:
                if os.path.exists(self._cmd_file):
                    with open(self._cmd_file, 'r', encoding='utf-8') as f:
                        content = f.read()
                    
                    if content and content != last_content:
                        last_content = content
                        data = json.loads(content)
                        
                        if data.get("type") == "send" and self._on_send:
                            self._on_send(data.get("text", ""))
                        elif data.get("type") == "close":
                            self._visible = False
                            self._running = False
                            if self._on_close:
                                self._on_close()
                            break
                
                # 检查进程是否还在运行
                if self._process and self._process.poll() is not None:
                    self._visible = False
                    self._running = False
                    if self._on_close:
                        self._on_close()
                    break
                    
            except Exception as e:
                pass
            
            time.sleep(0.1)
    
    def _send_to_window(self, data: dict):
        """发送数据到独立窗口"""
        try:
            self._msg_counter += 1
            data["_id"] = self._msg_counter
            with open(self._msg_file, 'w', encoding='utf-8') as f:
                json.dump(data, f, ensure_ascii=False)
        except Exception as e:
            pass
    
    def append_message(self, msg_type: str, user: str, content: str):
        """添加消息"""
        self._send_to_window({
            "type": "message",
            "msg_type": msg_type,
            "user": user,
            "content": content,
        })
    
    def update_status(self, status: str, connected: bool = False):
        """更新状态"""
        self._send_to_window({
            "type": "status",
            "status": status,
            "connected": connected,
        })
    
    def set_send_enabled(self, enabled: bool):
        """设置发送功能状态"""
        self._send_to_window({
            "type": "send_enabled",
            "enabled": enabled,
        })
    
    def isVisible(self) -> bool:
        """检查窗口是否可见"""
        return self._visible and self._running
    
    def raise_(self):
        """置顶窗口"""
        pass
    
    def close(self):
        """关闭窗口"""
        self._running = False
        self._visible = False
        if self._process:
            try:
                import os
                import signal
                
                # Windows 上使用 taskkill 强制终止进程树
                if os.name == 'nt':
                    try:
                        os.system(f'taskkill /F /T /PID {self._process.pid} >nul 2>&1')
                    except:
                        pass
                else:
                    # Unix 系统使用 SIGTERM
                    try:
                        os.killpg(os.getpgid(self._process.pid), signal.SIGTERM)
                    except:
                        pass
                
                # 备用方法：直接 kill
                try:
                    self._process.kill()
                except:
                    pass
                
                self._process = None
            except:
                pass
        
        # 清理临时文件
        try:
            import shutil
            shutil.rmtree(self._temp_dir, ignore_errors=True)
        except:
            pass


def is_available() -> bool:
    """始终可用"""
    return True
