# -*- coding: utf-8 -*-
"""卡片组件"""

import flet as ft
from typing import Optional, Callable
from ..themes import theme


class Card(ft.Container):
    """统一的卡片组件，支持右上角保存按钮"""
    
    def __init__(self, title: str, content: ft.Control, 
                 on_save: Optional[Callable] = None,
                 show_save_button: bool = False,
                 **kwargs):
        c = theme.colors
        self._on_save = on_save
        self._has_changes = False
        
        # 标题文本
        self._title_text = ft.Text(title, size=14, weight=ft.FontWeight.W_600, color=c.text_secondary)
        
        # 保存按钮（默认隐藏）
        self._save_btn = ft.Container(
            content=ft.Row([
                ft.Icon(ft.Icons.SAVE_OUTLINED, size=14, color=c.text_muted),
                ft.Text("保存", size=12, color=c.text_muted),
            ], spacing=4),
            bgcolor=c.bg_elevated,
            border=ft.border.all(1, c.border_default),
            border_radius=4,
            padding=ft.padding.symmetric(horizontal=10, vertical=4),
            on_click=self._handle_save,
            visible=show_save_button,
            tooltip="保存当前设置",
        )
        
        # 标题行（标题 + 保存按钮）
        title_row = ft.Row([
            self._title_text,
            ft.Container(expand=True),  # 占位
            self._save_btn,
        ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN)
        
        super().__init__(
            content=ft.Column([
                title_row,
                ft.Divider(height=1, color=c.border_muted),
                content,
            ], spacing=12),
            bgcolor=c.bg_surface,
            border=ft.border.all(1, c.border_default),
            border_radius=10,
            padding=18,
            **kwargs
        )
    
    def _handle_save(self, e):
        """处理保存点击"""
        if self._on_save:
            self._on_save(e)
        self.set_has_changes(False)
    
    def set_has_changes(self, has_changes: bool):
        """设置是否有未保存的更改"""
        c = theme.colors
        self._has_changes = has_changes
        
        if has_changes:
            # 待保存状态 - 高亮显示
            self._save_btn.bgcolor = c.accent
            self._save_btn.border = ft.border.all(1, c.accent)
            self._save_btn.content.controls[0].color = "#ffffff"
            self._save_btn.content.controls[1].color = "#ffffff"
            self._save_btn.content.controls[1].value = "待保存"
            self._save_btn.tooltip = "有未保存的更改，点击保存"
        else:
            # 已保存状态
            self._save_btn.bgcolor = c.bg_elevated
            self._save_btn.border = ft.border.all(1, c.border_default)
            self._save_btn.content.controls[0].color = c.text_muted
            self._save_btn.content.controls[1].color = c.text_muted
            self._save_btn.content.controls[1].value = "已保存"
            self._save_btn.tooltip = "设置已保存"
        
        if self.page:
            self._save_btn.update()
    
    def show_save_button(self, show: bool = True):
        """显示/隐藏保存按钮"""
        self._save_btn.visible = show
        if self.page:
            self._save_btn.update()
    
    def refresh_theme(self):
        """刷新主题"""
        c = theme.colors
        self.bgcolor = c.bg_surface
        self.border = ft.border.all(1, c.border_default)
        self._title_text.color = c.text_secondary
        if not self._has_changes:
            self._save_btn.bgcolor = c.bg_elevated
            self._save_btn.border = ft.border.all(1, c.border_default)
