# -*- coding: utf-8 -*-
"""
主应用 - Flet UI
"""

import flet as ft

# Flet兼容层 - 支持0.21.x和0.28.x
if not hasattr(ft, 'Icons') and hasattr(ft, 'icons'):
    ft.Icons = ft.icons
if not hasattr(ft, 'Colors') and hasattr(ft, 'colors'):
    ft.Colors = ft.colors

import sys
import subprocess
import webbrowser
from pathlib import Path
from typing import Optional

ROOT_DIR = Path(__file__).parent.parent.parent
if getattr(sys, 'frozen', False):
    ROOT_DIR = Path(sys._MEIPASS)
sys.path.insert(0, str(ROOT_DIR))

from .themes import theme
from .components import TitleBar
from .tabs import HomeTab, VoiceTab, AITab, AvatarTab, AntiDetectTab, ToolsTab, HelpTab, AnalyticsTab
from src.analytics.collector import LiveDataCollector
from .panels import RightPanel
from .services import StreamServiceManager, WorkerManager, ConfigManager, EnvironmentChecker
from .services.api_logger import log_request, log_response

# 授权模块
try:
    from gui.license_manager import (
        get_online_license_manager,
        check_online_license,
        apply_online_trial,
        activate_online_license,
        generate_device_id,
        check_feature,
        get_feature_status,
        Features,
        APP_VERSION
    )
    HAS_LICENSE_MODULE = True
except ImportError:
    HAS_LICENSE_MODULE = False
    Features = None
    APP_VERSION = "1.0.0"


class MainApp:
    """主应用"""
    
    def __init__(self, page: ft.Page):
        self.page = page
        
        # 数据采集器（需要在 worker_manager 之前初始化）
        self._data_collector = LiveDataCollector()
        
        # 服务
        self.config_manager = ConfigManager()
        self.env_checker = EnvironmentChecker()
        self.stream_manager = StreamServiceManager()
        self.worker_manager = WorkerManager()
        self.worker_manager.set_data_collector(self._data_collector)
        
        # UI 组件
        self.title_bar: Optional[TitleBar] = None
        self.home_tab: Optional[HomeTab] = None
        self.voice_tab: Optional[VoiceTab] = None
        # self.voice_changer_tab 已移至独立软件
        self.ai_tab: Optional[AITab] = None
        self.avatar_tab: Optional[AvatarTab] = None
        self.anti_detect_tab: Optional[AntiDetectTab] = None
        self.tools_tab: Optional[ToolsTab] = None
        self.analytics_tab: Optional[AnalyticsTab] = None
        self.help_tab: Optional[HelpTab] = None
        self.right_panel: Optional[RightPanel] = None
        
        # 状态
        self._current_tab = 0
        self._overlay_process = None
        self._timer_process = None  # 翻页时钟进程
        self.danmu_sender = None
        self._capture_timeout_active = False
        self._packet_capture = None  # 网络抓包服务引用
        self._stream_captured = False  # 推流码是否已捕获
        
        # 授权状态
        self._license_valid = False
        self._license_info = None
        
        # 初始化
        self._setup_page()
        self._init_services()
        self._build_ui()
        self._load_config()
        self._start_env_check()
        
        # 检查授权
        self._check_license()
    
    def _setup_page(self):
        """设置页面属性 - 兼容flet 0.24.x和0.28.x"""
        self.page.title = "梧桐抖音运营伴侣"
        
        # 兼容新旧版本flet API
        if hasattr(self.page, 'window') and hasattr(self.page.window, 'width'):
            # flet 0.28+
            self.page.window.width = 1200
            self.page.window.height = 850
            self.page.window.min_width = 1000
            self.page.window.min_height = 750
            self.page.window.center()
            self.page.window.frameless = True
            self.page.window.title_bar_hidden = True
            icon_path = ROOT_DIR / "static" / "icons" / "icon.ico"
            if icon_path.exists():
                self.page.window.icon = str(icon_path)
        else:
            # flet 0.24.x及更早版本
            self.page.window_width = 1200
            self.page.window_height = 850
            self.page.window_min_width = 1000
            self.page.window_min_height = 750
            self.page.window_center()
            self.page.window_frameless = True
            self.page.window_title_bar_hidden = True
            icon_path = ROOT_DIR / "static" / "icons" / "icon.ico"
            if icon_path.exists():
                self.page.window_icon = str(icon_path)
        
        self.page.padding = 0
        self.page.spacing = 0
        
        self._apply_theme()
    
    def _apply_theme(self):
        """应用主题"""
        c = theme.colors
        self.page.bgcolor = c.bg_base
        self.page.theme_mode = ft.ThemeMode.DARK if theme.is_dark else ft.ThemeMode.LIGHT
        self.page.theme = theme.get_flet_theme()
    
    def _init_services(self):
        """初始化服务"""
        # 注册回调
        self.env_checker.on_status_change(self._on_env_status)
        self.stream_manager.on_state_change(self._on_stream_state)
        self.stream_manager.on_stream_captured(self._on_stream_captured)
        self.stream_manager.on_room_captured(self._on_room_captured)
        self.stream_manager.on_error(self._on_stream_error)
        self.stream_manager.on_progress(self._on_stream_progress)

        # 注册抓包日志回调（新增）
        if hasattr(self.stream_manager, 'on_log'):
            self.stream_manager.on_log(self._on_capture_log)

        self.worker_manager.on_message(self._on_worker_message)
        self.worker_manager.on_status_change(self._on_worker_status)
        self.worker_manager.on_error(self._on_worker_error)

        # 启动 Web 服务器（自动选择可用端口）
        self._web_server = None
        self._web_server_port = 8080
        self._start_web_server_background()
    
    def _toggle_theme(self, e):
        """切换主题"""
        theme.toggle()
        self._apply_theme()
        self._rebuild_ui()
    
    def _rebuild_ui(self):
        """重建 UI"""
        self._build_ui()
        self.page.update()
    
    def _create_tab_with_help(self, name: str, icon, content, help_text: str, colors):
        """创建带帮助提示的 Tab"""
        # 使用原生 Tab，tooltip 放在 content 的顶部容器
        return ft.Tab(text=name, icon=icon, content=content)
    
    def _build_ui(self):
        """构建界面"""
        c = theme.colors
        
        # 标题栏
        self.title_bar = TitleBar(
            on_theme_toggle=self._toggle_theme,
            on_minimize=self._on_minimize,
            on_close=lambda _: self._on_close(),
            on_license_click=lambda _: self._show_license_dialog(),
            on_logout=lambda _: self._on_logout(),
            on_profile_click=lambda _: self._show_profile_dialog(),
        )
        
        # 创建 Tabs
        self.home_tab = HomeTab(
            on_start_stream=self._on_start_stream,
            on_stop_stream=self._on_stop_stream,
            on_close_companion=self._on_close_companion,
            on_switch_obs=self._on_switch_obs,
            on_copy_url=self._on_copy_url,
            on_copy_key=self._on_copy_key,
            on_browse_companion=self._on_browse_companion,
            on_refresh_interfaces=lambda: self._refresh_interfaces(),
            on_danmu_login=self._on_danmu_login,
            on_send_danmu=self._on_send_danmu,
            config_manager=self.config_manager,
        )
        
        self.voice_tab = VoiceTab(
            on_preview=self._on_preview_voice,
            on_config_change=self._on_voice_config_change,
            on_save=self._on_voice_config_save,
            config_manager=self.config_manager,
        )
        
        # 变声器已移至独立软件 VoiceChanger
        
        self.ai_tab = AITab(
            on_test=self._on_test_ai,
            on_config_change=self._on_ai_config_change,
            on_save=self._on_ai_config_save,
            config_manager=self.config_manager,
        )
        
        self.avatar_tab = AvatarTab(
            on_avatar_toggle=self._on_avatar_toggle,
            on_desktop_toggle=self._on_desktop_toggle,
            on_model_change=self._on_model_change,
            on_unlock_toggle=self._on_unlock_toggle,
            models=self._scan_models(),
            config_manager=self.config_manager,
        )
        
        self.anti_detect_tab = AntiDetectTab(
            on_visual_toggle=self._on_visual_toggle,
            on_visual_strength=self._on_visual_strength,
            on_typing_toggle=self._on_typing_toggle,
            on_typing_speed=self._on_typing_speed,
            config_manager=self.config_manager,
        )
        
        self.tools_tab = ToolsTab(
            on_open_timer=self._on_open_timer,
        )
        
        self.analytics_tab = AnalyticsTab(collector=self._data_collector)
        
        self.help_tab = HelpTab(on_feedback=self._on_help_feedback)
        
        # Tab 配置：(名称, 图标, 内容, 帮助说明)
        tab_configs = [
            ("首页", ft.Icons.HOME_ROUNDED, self.home_tab, "直播间流捕获、推流设置、弹幕监控"),
            ("语音回复", ft.Icons.RECORD_VOICE_OVER, self.voice_tab, "自动语音播报弹幕、礼物、关注等消息"),
            ("AI", ft.Icons.SMART_TOY, self.ai_tab, "AI智能回复弹幕，支持多种大模型"),
            ("形象", ft.Icons.FACE, self.avatar_tab, "Live2D虚拟形象，自动口型同步"),
            ("防检测", ft.Icons.SHIELD, self.anti_detect_tab, "OBS虚拟摄像头防检测"),
            ("工具", ft.Icons.BUILD_ROUNDED, self.tools_tab, "翻页时钟、正计时、倒计时"),
            ("统计", ft.Icons.ANALYTICS_ROUNDED, self.analytics_tab, "直播间数据统计分析"),
            ("说明", ft.Icons.HELP_OUTLINE, self.help_tab, None),  # 说明页不需要帮助
        ]
        
        # Tab 导航 - 禁用滑动切换
        tabs = ft.Tabs(
            selected_index=self._current_tab,
            on_change=self._on_tab_change,
            animation_duration=0,  # 禁用动画
            scrollable=False,  # 禁用滚动
            label_color=c.text_muted,
            indicator_color=c.accent,
            divider_color=c.border_default,
            tabs=[self._create_tab_with_help(name, icon, content, help_text, c) 
                  for name, icon, content, help_text in tab_configs],
            expand=True,
        )
        
        # 底部连接按钮
        self.connect_btn = ft.ElevatedButton(
            "连接", icon=ft.Icons.LINK_ROUNDED,
            bgcolor=c.success, color="#ffffff", height=40,
            on_click=self._on_connect,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=8)),
        )
        
        self.stop_btn = ft.OutlinedButton(
            "停止", height=40, disabled=True,
            on_click=self._on_stop,
            style=ft.ButtonStyle(shape=ft.RoundedRectangleBorder(radius=8)),
        )
        
        self.status_indicator = ft.Container(
            content=ft.Row([
                ft.Icon(ft.Icons.CIRCLE, color=c.text_muted, size=10),
                ft.Text("未连接", size=12, color=c.text_muted),
            ], spacing=6),
        )
        
        bottom_bar = ft.Container(
            content=ft.Column([
                ft.Divider(height=1, color=c.border_default),
                ft.Row([self.connect_btn, self.stop_btn], spacing=12),
                self.status_indicator,
            ], spacing=10),
            padding=ft.padding.all(15),
        )
        
        # 左侧面板 - expand 填充剩余空间
        left_panel = ft.Container(
            content=ft.Column([tabs, bottom_bar], spacing=0),
            expand=True,
            bgcolor=c.bg_surface,
        )
        
        # 右侧面板 - 固定宽度
        self.right_panel = RightPanel(
            on_float_window=self._on_float_window,
            on_send_danmu=self._on_send_danmu
        )
        
        # 主布局
        self.page.controls.clear()
        self.page.add(
            ft.Column([
                ft.WindowDragArea(self.title_bar, maximizable=False),
                ft.Row([
                    left_panel,
                    self.right_panel,
                ], expand=True, spacing=0),
            ], expand=True, spacing=0)
        )
    
    def _load_config(self):
        """加载配置"""
        self.config_manager.load()
        
        if self.home_tab:
            self.home_tab.load_config(self.config_manager)
            # 延迟检查 Chrome 路径是否有效
            self.page.run_task(self._check_pending_paths)
        if self.voice_tab:
            self.voice_tab.load_config(self.config_manager)
        if self.ai_tab:
            self.ai_tab.load_config(self.config_manager)
        if self.anti_detect_tab:
            self.anti_detect_tab.load_config(self.config_manager)
        if self.avatar_tab:
            self.avatar_tab.load_config(self.config_manager)
    
    async def _check_pending_paths(self):
        """检查待处理的路径"""
        import asyncio
        await asyncio.sleep(0.5)  # 等待页面完全加载
        if self.home_tab:
            self.home_tab.check_pending_path()
    
    def _save_config(self):
        """保存配置"""
        self.config_manager.save()
    
    def _start_env_check(self):
        """启动环境检测"""
        self.env_checker.check_all_async()
    
    def _scan_models(self):
        """扫描模型"""
        try:
            model_dir = ROOT_DIR / "static" / "live2d" / "live2d-model"
            models = [p.name for p in model_dir.iterdir() if p.is_dir()]
            return sorted(models) if models else ["Hiyori"]
        except:
            return ["Hiyori"]
    
    def _refresh_interfaces(self):
        """
        刷新网卡列表 - 增强版
        支持有线以太网、WiFi等所有类型的真实网卡
        """
        if not self.home_tab:
            return
        
        try:
            from scapy.arch.windows import get_windows_if_list
            interfaces = get_windows_if_list()
            
            # "所有网卡" 选项（使用 None 进行全局监听）
            options = [ft.dropdown.Option("all", "🌐 所有网卡 (推荐，支持有线/无线)")]
            
            # 需要过滤的关键词（虚拟网卡、驱动等）
            skip_keywords = [
                'loopback', 'virtual', 'vmware', 'virtualbox', 'hyper-v',
                'wan miniport', 'npcap loopback', 'filter', 'qos', 'scheduler',
                'lightweight', 'ndis', 'zerotier', 'tunnel', 'vpn',
                'teredo', 'isatap', '6to4', 'pseudo', 'microsoft wi-fi direct',
                'bluetooth'
            ]
            
            valid_count = 0
            ethernet_found = False
            wifi_found = False
            
            for iface in interfaces:
                name = iface.get('name', '')
                description = iface.get('description', '')
                ips = iface.get('ips', [])
                
                if not name:
                    continue
                
                # 过滤虚拟网卡
                desc_lower = description.lower()
                name_lower = name.lower()
                
                # 注意：不过滤 npcap 相关的真实网卡，只过滤 npcap loopback
                should_skip = False
                for kw in skip_keywords:
                    if kw in desc_lower or kw in name_lower:
                        # 特殊处理：不跳过 "Npcap Loopback" 以外的 npcap 相关项
                        if kw == 'npcap loopback' and 'loopback' not in desc_lower:
                            continue
                        should_skip = True
                        break
                
                if should_skip:
                    print(f"[网卡] 跳过虚拟网卡: {description}")
                    continue
                
                # 只保留有有效 IPv4 的网卡
                ipv4_list = [ip for ip in ips if isinstance(ip, str) and '.' in ip and not ip.startswith('169.254')]
                if not ipv4_list:
                    print(f"[网卡] 跳过无IP网卡: {description}")
                    continue
                
                # 识别网卡类型
                icon = "🔌"  # 默认图标
                if 'ethernet' in desc_lower or '以太网' in description or 'realtek' in desc_lower or 'intel' in desc_lower:
                    icon = "🔌"  # 有线网卡
                    ethernet_found = True
                elif 'wi-fi' in desc_lower or 'wifi' in desc_lower or 'wireless' in desc_lower or '无线' in description:
                    icon = "📶"  # 无线网卡
                    wifi_found = True
                
                # 显示友好名称: "🔌 以太网 (192.168.1.100)"
                display_name = f"{icon} {description} ({ipv4_list[0]})"
                options.append(ft.dropdown.Option(name, display_name))
                valid_count += 1
                print(f"[网卡] 添加: {display_name}")
            
            self.home_tab.interface_dropdown.options = options
            self.home_tab.interface_dropdown.value = "all"
            self.page.update()
            
            status_msg = f"[ENV] 刷新网卡列表: {valid_count} 个有效网卡"
            if ethernet_found:
                status_msg += " (含有线)"
            if wifi_found:
                status_msg += " (含无线)"
            print(status_msg)
            
        except Exception as e:
            print(f"刷新网卡列表失败: {e}")
            import traceback
            traceback.print_exc()
    
    # ========== 环境检测回调 ==========
    
    def _on_env_status(self, status):
        """环境状态更新"""
        print(f"[ENV] npcap={status.npcap_installed}, companion={status.companion_installed}, obs={status.obs_installed}")
        if self.home_tab:
            self.home_tab.update_env_status(
                status.npcap_installed,
                status.companion_installed,
                status.obs_installed
            )
            # 同时刷新网卡列表
            if status.npcap_installed:
                self._refresh_interfaces()
            self.page.update()
    
    # ========== 推流服务回调 ==========
    
    def _on_stream_state(self, status):
        if self.home_tab:
            self.home_tab.update_stream_status(status.state, status.message)
            self.page.update()
    
    def _on_stream_captured(self, url: str, key: str):
        """推流信息捕获成功"""
        if self.home_tab:
            # 标记捕获成功，停止后续进度更新
            self._stream_captured = True
            
            self.home_tab.update_stream_info(url, key)
            # 启用按钮
            self.home_tab.copy_url_btn.disabled = False
            self.home_tab.copy_key_btn.disabled = False
            self.home_tab.switch_obs_btn.disabled = False
            self.home_tab.close_companion_btn.disabled = False
            self.home_tab.update_progress(100)
            self.home_tab.update_stream_status("connected", "已捕获推流码")
            
            # 停止网络捕获（已捕获成功，无需继续）
            if self._packet_capture:
                try:
                    self._packet_capture.stop()
                except:
                    pass
            
            self.right_panel.add_system("✅ 捕获成功！")
            self.right_panel.add_system(f"服务器: {url}")
            self.right_panel.add_system(f"密钥: {key[:40]}...")
            self.right_panel.add_system("点击「切换OBS」或「复制」使用")
            self.page.update()
    
    def _on_room_captured(self, room_id: str, web_rid: str):
        """房间ID捕获成功"""
        if self.home_tab:
            self.home_tab.room_id_input.value = room_id or web_rid
            self.config_manager.set_room_id(room_id or web_rid)
            
            self.right_panel.add_system(f"房间ID: {room_id}")
            if web_rid and web_rid != room_id:
                self.right_panel.add_system(f"web_rid: {web_rid}")
            
            # 如果勾选了自动连接，则自动连接弹幕
            if self.home_tab.auto_connect_check.value:
                self.right_panel.add_system("自动连接弹幕...")
                import threading
                threading.Timer(1.0, self._on_connect).start()
            
            self.page.update()
    
    def _on_stream_error(self, msg: str):
        self.right_panel.add_system(f"错误: {msg}")
        self.page.update()

    def _on_stream_progress(self, value: int):
        # 如果已捕获成功，忽略后续进度更新
        if getattr(self, '_stream_captured', False):
            return
        if self.home_tab:
            self.home_tab.update_progress(value)
            self.page.update()

    def _on_capture_log(self, msg: str):
        """抓包日志回调 - 实时显示在界面"""
        if self.right_panel:
            # 过滤掉过于频繁的状态日志，只显示关键信息
            if any(kw in msg for kw in ['捕获', '检测', 'RTMP', '推流', '启动', '停止', '错误', '✓', '✅', '❌', '⚠']):
                self.right_panel.add_system(f"[抓包] {msg}")
                self.page.update()

    # ========== Worker 回调 ==========
    
    def _on_worker_message(self, event):
        if self.right_panel:
            if event.type == "entrance":
                self.right_panel.add_entrance(event.nickname)
                self._sync_to_float_window("entrance", event.nickname, "进入直播间")
            elif event.type == "gift":
                self.right_panel.add_gift(event.nickname, event.content, 1)
                self._sync_to_float_window("gift", event.nickname, f"送出 {event.content}")
            elif event.type == "follow":
                self.right_panel.add_follow(event.nickname)
                self._sync_to_float_window("follow", event.nickname, "关注了你")
            else:
                self.right_panel.add_chat(event.nickname, event.content)
                self._sync_to_float_window("chat", event.nickname, event.content)
            self.page.update()
    
    def _on_worker_status(self, status: str):
        c = theme.colors
        if status == "connected":
            self.status_indicator.content.controls[0].color = c.success
            self.status_indicator.content.controls[1].value = "已连接"
            self.status_indicator.content.controls[1].color = c.success
            self.connect_btn.disabled = True
            self.stop_btn.disabled = False
            self.right_panel.add_system("已连接到直播间")
            # 自动启用 AI（如果已配置）
            self._auto_enable_ai()
            # 同步状态到独立窗口
            self._sync_float_window_status("已连接", True)
        elif status == "disconnected":
            self.status_indicator.content.controls[0].color = c.text_muted
            self.status_indicator.content.controls[1].value = "未连接"
            self.status_indicator.content.controls[1].color = c.text_muted
            self.connect_btn.disabled = False
            self.stop_btn.disabled = True
            # 同步状态到独立窗口
            self._sync_float_window_status("未连接", False)
        elif status == "connecting":
            self.status_indicator.content.controls[0].color = c.warning
            self.status_indicator.content.controls[1].value = "连接中..."
            self.status_indicator.content.controls[1].color = c.warning
            # 同步状态到独立窗口
            self._sync_float_window_status("连接中...", False)
        elif status == "reconnecting":
            self.status_indicator.content.controls[0].color = c.warning
            self.status_indicator.content.controls[1].value = "重连中..."
            self.status_indicator.content.controls[1].color = c.warning
            self.right_panel.add_system("正在自动重连...")
            # 同步状态到独立窗口
            self._sync_float_window_status("重连中...", False)
        self.page.update()
    
    def _on_worker_error(self, msg: str):
        self.right_panel.add_system(f"错误: {msg}")
        self.page.update()
    
    def _auto_enable_ai(self):
        """自动启用 AI（如果已配置）"""
        try:
            # 检查是否勾选了启用 AI
            ai_enabled = self.config_manager.get_ai_enabled()
            if not ai_enabled:
                return
            
            # 检查是否有 API Key
            api_key = self.config_manager.get_ai_api_key()
            if not api_key or not api_key.strip():
                return
            
            # 已配置，自动启用
            self.right_panel.add_system("✓ AI 回复已自动启用")
            
        except Exception as e:
            print(f"自动启用 AI 失败: {e}")
    
    # ========== UI 事件 ==========
    
    def _on_tab_change(self, e):
        """Tab 切换 - 检查功能权限"""
        target_tab = e.control.selected_index
        
        # Tab 索引 -> 功能 ID 映射
        # 0: 首页 (基础功能), 6: 说明 (无需授权)
        TAB_FEATURE_MAP = {
            1: Features.VOICE_TTS if Features else None,      # 语音回复
            2: Features.VOICE_CHANGER if Features else None,  # 变声器
            3: Features.AI_CHAT if Features else None,        # AI
            4: Features.AVATAR if Features else None,         # 形象
            5: Features.ANTI_DETECT if Features else None,    # 防检测
            6: Features.ANALYTICS if Features else None,      # 统计分析
        }
        
        feature_id = TAB_FEATURE_MAP.get(target_tab)
        
        # 需要检查权限的 Tab
        if feature_id and HAS_LICENSE_MODULE:
            has_permission, msg = check_feature(feature_id)
            if not has_permission:
                # 直接打开授权码填写窗口（不显示中间提示弹窗）
                self._show_license_dialog()
                # 还原到之前的 Tab
                e.control.selected_index = self._current_tab
                self.page.update()
                return
        
        self._current_tab = target_tab
    
    def _show_feature_locked_dialog(self, feature_id: str, msg: str):
        """显示功能锁定提示，提供激活入口"""
        info = Features.INFO.get(feature_id, {}) if Features else {}
        name = info.get("name", "此功能")
        desc = info.get("desc", "")
        
        def close_dialog(e):
            dialog.open = False
            self.page.update()
        
        def open_license_dialog(e):
            dialog.open = False
            self.page.update()
            self._show_license_dialog()
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text(f"🔒 {name}"),
            content=ft.Column([
                ft.Text(msg, size=14),
                ft.Text(f"功能说明: {desc}", size=12, color=ft.Colors.GREY_500) if desc else ft.Container(),
                ft.Container(height=10),
                ft.Text("激活授权或申请试用后即可使用此功能", size=12, color=ft.Colors.BLUE_400),
            ], tight=True, spacing=8),
            actions=[
                ft.TextButton("取消", on_click=close_dialog),
                ft.ElevatedButton("激活授权", on_click=open_license_dialog),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _on_start_stream(self):
        """开播 - 完整复刻原系统流程"""
        import threading
        import os
        import ctypes
        
        # 0. 检查管理员权限（网络抓包需要管理员权限）
        try:
            is_admin = ctypes.windll.shell32.IsUserAnAdmin()
        except:
            is_admin = False
        
        if not is_admin:
            self.right_panel.add_system("⚠ 未以管理员身份运行")
            self.right_panel.add_system("网络抓包需要管理员权限，可能无法捕获推流码")
            self.right_panel.add_system("建议：右键点击程序 → 以管理员身份运行")
            self.page.update()
            # 不阻止继续，只是警告
        
        # 1. 检查 Npcap
        npcap_dll = r"C:\Windows\System32\Npcap\wpcap.dll"
        if not os.path.exists(npcap_dll):
            self.right_panel.add_system("❌ Npcap 未安装")
            self.right_panel.add_system("请先安装 Npcap 网络驱动")
            import webbrowser
            webbrowser.open("https://npcap.com/#download")
            self.page.update()
            return
        
        # 2. 检查直播伴侣
        companion_path = self.env_checker.status.companion_path
        if not companion_path or not os.path.exists(companion_path):
            self.right_panel.add_system("❌ 直播伴侣未安装")
            self.right_panel.add_system("请先安装抖音直播伴侣或设置路径")
            import webbrowser
            webbrowser.open("https://www.douyin.com/falcon/webcast_openpc/pages/streamingtool_download/index.html")
            self.page.update()
            return
        
        # 3. 重置状态
        self.stream_manager.status.stream_url = ""
        self.stream_manager.status.stream_key = ""
        self.home_tab.copy_url_btn.disabled = True
        self.home_tab.copy_key_btn.disabled = True
        self.home_tab.switch_obs_btn.disabled = True
        self.home_tab.start_btn.disabled = True
        self.home_tab.stop_btn.disabled = False
        self.home_tab.update_progress(5)
        self.home_tab.update_stream_status("starting", "正在启动...")
        self.page.update()
        
        # 4. 获取网卡配置
        interface = self.home_tab.interface_dropdown.value
        if interface == "all":
            interface = None
        
        config = {
            'auto_start_companion': True,
            'companion_path': companion_path,
            'network_interface': interface,
            'capture_timeout': 300,
        }
        
        self.right_panel.add_system("【一键开播】启动中...")
        self.page.update()
        
        # 5. 启动网络监控
        self.home_tab.update_progress(10)
        self.home_tab.update_stream_status("capturing", "启动网络监控...")
        self.right_panel.add_system("正在启动网络流量监控...")
        self.page.update()
        
        self.stream_manager.start(config)
        
        # 6. 延迟启动自动开播（等网络监控启动）
        def delayed_auto_broadcast():
            import time
            time.sleep(1)
            self._auto_start_companion(companion_path)
        
        threading.Thread(target=delayed_auto_broadcast, daemon=True).start()
    
    def _auto_start_companion(self, companion_path: str):
        """自动启动直播伴侣并点击开播"""
        import time
        
        # 重置停止标志
        self._auto_broadcast_stop = False
        
        def is_stopped():
            """检查是否已停止"""
            return getattr(self, '_auto_broadcast_stop', False)
        
        def update_ui(progress: int, status: str, detail: str = None, error: bool = False):
            if is_stopped():
                return
            self.home_tab.update_progress(progress)
            self.home_tab.update_stream_status("error" if error else "capturing", status)
            if detail:
                self.right_panel.add_system(detail)
            self.page.update()
        
        try:
            from src.stream.auto_broadcast import AutoBroadcast, check_dependencies
            
            if is_stopped():
                return
            
            # 检查依赖
            deps = check_dependencies()
            print(f"[GUI] 自动开播依赖: {deps}")
            
            auto = AutoBroadcast(
                companion_path=companion_path,
                logger=lambda msg: print(f"[AutoBroadcast] {msg}")
            )
            
            update_ui(20, "启动直播伴侣...", "正在启动直播伴侣...")
            
            if is_stopped():
                return
            
            # 启动直播伴侣
            if not auto.start_companion():
                update_ui(0, "启动失败", "❌ 直播伴侣启动失败", error=True)
                self.home_tab.start_btn.disabled = False
                self.home_tab.stop_btn.disabled = True
                self.page.update()
                return
            
            if is_stopped():
                return
            
            update_ui(35, "等待窗口出现...", "✓ 直播伴侣已启动，等待窗口...")
            
            # 直播伴侣启动后立即开始监控和捕获（防止点击识别问题导致漏捕）
            self._start_capture_timeout_check()
            self._start_companion_monitor()
            self.right_panel.add_system("✓ 推流码捕获已启动")
            
            if is_stopped():
                return
            
            # 查找窗口（使用短间隔循环代替长超时，以便响应停止）
            hwnd = None
            for _ in range(30):  # 30秒超时
                if is_stopped():
                    return
                hwnd = auto.find_companion_window(timeout=1)
                if hwnd:
                    break
            
            if not hwnd:
                if is_stopped():
                    return
                update_ui(50, "请手动开播", "❌ 未找到直播伴侣窗口，请手动操作")
                self.right_panel.add_system("⚠ 未找到窗口，但推流码捕获仍在运行")
                self.page.update()
                return
            
            if is_stopped():
                return
            
            update_ui(50, "等待界面加载...", "✓ 窗口已出现，等待界面加载...")
            
            # 使用短间隔等待代替长 sleep
            for _ in range(10):  # 5秒 = 10 * 0.5秒
                if is_stopped():
                    return
                time.sleep(0.5)
            
            if is_stopped():
                return
            
            update_ui(65, "识别按钮位置...", "正在识别开播按钮...")
            
            # 点击开播按钮（无论成功失败，推流码捕获都已在运行）
            if is_stopped():
                return
            
            if auto.click_start_button():
                update_ui(85, "已点击开播，监听推流码...", 
                         "✓ 定位并点击成功\n正在监听推流码...")
                self.right_panel.add_system("等待捕获推流信息...")
            else:
                # 定位失败，弹窗提示但继续监听
                if is_stopped():
                    return
                update_ui(50, "请手动开播", "监听中...")
                self._show_manual_start_dialog()
                self.right_panel.add_system("⚠ 自动定位失败，请手动点击开播按钮")
                
        except ImportError as e:
            print(f"[GUI] 导入自动开播模块失败: {e}")
            self._fallback_manual_mode(companion_path)
        except Exception as e:
            print(f"[GUI] 自动开播异常: {e}")
            import traceback
            traceback.print_exc()
            # 异常时仍然启动监控，网络捕获已在运行
            self.home_tab.update_progress(30)
            self.home_tab.update_stream_status("capturing", "请手动开播")
            self.right_panel.add_system(f"⚠ 自动开播异常: {e}")
            self.right_panel.add_system("✓ 推流码捕获仍在运行，请手动点击【开始直播】")
            self._start_capture_timeout_check()
            self._start_companion_monitor()
            self.page.update()
    
    def _fallback_manual_mode(self, companion_path: str):
        """降级到手动模式 - 仅启动直播伴侣"""
        import subprocess
        import os
        
        try:
            work_dir = os.path.dirname(companion_path)
            subprocess.Popen(
                [companion_path],
                cwd=work_dir,
                shell=True
            )
            self.home_tab.update_progress(30)
            self.home_tab.update_stream_status("capturing", "请手动点击开播")
            self.right_panel.add_system("✓ 直播伴侣已启动")
            self.right_panel.add_system("⚠ 识别模块不可用，请手动点击【开始直播】")
            self.right_panel.add_system("✓ 推流码捕获已启动，开播后会自动抓取")
            # 启动超时检测和进程监控
            self._start_capture_timeout_check()
            self._start_companion_monitor()
            self.page.update()
        except Exception as e:
            self.right_panel.add_system(f"❌ 启动直播伴侣失败: {e}")
            self.home_tab.start_btn.disabled = False
            self.home_tab.stop_btn.disabled = True
            self.home_tab.update_stream_status("error", "启动失败")
            self.page.update()
    
    def _on_stop_stream(self):
        """停止推流"""
        # 设置停止标志（用于中断自动开播流程）
        self._auto_broadcast_stop = True
        self._capture_timeout_active = False  # 停止超时检测
        self._companion_monitor_active = False  # 停止进程监控
        
        self.stream_manager.stop()
        
        # 重置UI状态
        self.home_tab.start_btn.disabled = False
        self.home_tab.stop_btn.disabled = True
        self.home_tab.update_progress(0)
        self.home_tab.update_stream_status("idle", "已停止")
        self.right_panel.add_system("已停止开播流程")
        self.page.update()
    
    def _show_manual_start_dialog(self):
        """显示手动开播提示对话框"""
        c = theme.colors
        
        dialog = ft.AlertDialog(
            title=ft.Row([
                ft.Icon(ft.Icons.INFO_OUTLINE, color=ft.Colors.ORANGE_400),
                ft.Text("请手动点击开播", weight=ft.FontWeight.BOLD),
            ], spacing=8),
            content=ft.Column([
                ft.Text("自动定位开播按钮失败，请手动操作：", size=14),
                ft.Text("", size=8),
                ft.Text("1. 确认抖音直播伴侣已登录账号", size=13),
                ft.Text("2. 在伴侣界面手动点击【开始直播】", size=13),
                ft.Text("", size=8),
                ft.Container(
                    content=ft.Text("✓ 软件正在持续监听推流码，开播后会自动捕获", 
                                   size=12, color=c.success),
                    bgcolor=ft.Colors.with_opacity(0.1, c.success),
                    padding=10,
                    border_radius=6,
                ),
            ], tight=True, spacing=4),
            actions=[
                ft.TextButton("知道了", on_click=lambda e: self._close_dialog(dialog)),
            ],
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _start_capture_timeout_check(self):
        """启动推流码捕获超时检测"""
        import threading
        
        self._capture_timeout_active = True
        timeout_seconds = 60  # 60秒后提示
        
        def check_timeout():
            import time
            time.sleep(timeout_seconds)
            
            # 检查是否已捕获或已停止
            if not self._capture_timeout_active:
                return
            
            # 检查是否已捕获推流码
            if self.stream_manager.status.stream_url:
                return
            
            # 超时未捕获，弹窗提示
            self.page.run_thread(lambda: self._show_capture_timeout_dialog())
        
        threading.Thread(target=check_timeout, daemon=True).start()
    
    def _show_capture_timeout_dialog(self):
        """显示推流码捕获超时提示"""
        c = theme.colors
        
        dialog = ft.AlertDialog(
            title=ft.Row([
                ft.Icon(ft.Icons.TIMER_OFF, color=ft.Colors.ORANGE_400),
                ft.Text("等待推流码超时", weight=ft.FontWeight.BOLD),
            ], spacing=8),
            content=ft.Column([
                ft.Text("较长时间未检测到推流码，请检查：", size=14),
                ft.Text("", size=8),
                ft.Text("1. 抖音直播伴侣是否已登录账号？", size=13),
                ft.Text("2. 是否已点击【开始直播】按钮？", size=13),
                ft.Text("3. 直播是否已成功开启？", size=13),
                ft.Text("", size=8),
                ft.Container(
                    content=ft.Text("✓ 软件仍在持续监听，开播后会自动捕获推流码", 
                                   size=12, color=c.success),
                    bgcolor=ft.Colors.with_opacity(0.1, c.success),
                    padding=10,
                    border_radius=6,
                ),
            ], tight=True, spacing=4),
            actions=[
                ft.TextButton("继续等待", on_click=lambda e: self._close_dialog(dialog)),
                ft.TextButton("停止", on_click=lambda e: self._close_and_stop(dialog)),
            ],
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _close_and_stop(self, dialog):
        """关闭对话框并停止"""
        self._close_dialog(dialog)
        self._on_stop_stream()
    
    def _start_companion_monitor(self):
        """启动直播伴侣进程监控"""
        import threading
        
        self._companion_monitor_active = True
        
        def monitor_companion():
            import time
            try:
                import psutil
                PSUTIL_OK = True
            except ImportError:
                PSUTIL_OK = False
                print("[Monitor] psutil不可用，无法监控进程")
                return
            
            check_interval = 3  # 每3秒检查一次
            
            while self._companion_monitor_active:
                time.sleep(check_interval)
                
                if not self._companion_monitor_active:
                    break
                
                # 检查直播伴侣进程是否存在
                companion_running = False
                for proc in psutil.process_iter(['name', 'exe']):
                    try:
                        name = proc.info.get('name', '').lower()
                        exe = proc.info.get('exe', '') or ''
                        if '直播伴侣' in name or '直播伴侣' in exe or 'webcast' in name.lower():
                            companion_running = True
                            break
                    except (psutil.NoSuchProcess, psutil.AccessDenied):
                        pass
                
                if not companion_running:
                    # 进程已退出，通知UI
                    print("[Monitor] 直播伴侣进程已退出")
                    self._companion_monitor_active = False
                    self.page.run_thread(lambda: self._on_companion_exited())
                    break
        
        threading.Thread(target=monitor_companion, daemon=True).start()
    
    def _on_companion_exited(self):
        """直播伴侣进程退出回调"""
        c = theme.colors
        
        # 停止捕获
        self._capture_timeout_active = False
        self.stream_manager.stop()
        
        # 更新UI状态
        self.home_tab.start_btn.disabled = False
        self.home_tab.stop_btn.disabled = True
        self.home_tab.update_progress(0)
        self.home_tab.update_stream_status("error", "伴侣已退出")
        self.right_panel.add_system("⚠ 检测到直播伴侣已关闭，已停止捕获推流码")
        
        # 弹窗提示
        dialog = ft.AlertDialog(
            title=ft.Row([
                ft.Icon(ft.Icons.WARNING_AMBER_ROUNDED, color=ft.Colors.ORANGE_400),
                ft.Text("直播伴侣已关闭", weight=ft.FontWeight.BOLD),
            ], spacing=8),
            content=ft.Column([
                ft.Text("检测到抖音直播伴侣进程已退出", size=14),
                ft.Text("", size=8),
                ft.Text("已自动停止推流码捕获", size=13, color=c.text_secondary),
                ft.Text("如需继续直播，请重新点击「开播」按钮", size=13, color=c.text_secondary),
            ], tight=True, spacing=4),
            actions=[
                ft.TextButton("知道了", on_click=lambda e: self._close_dialog(dialog)),
            ],
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _on_close_companion(self):
        """关闭直播伴侣进程 - 需要用户确认"""
        import psutil
        
        def do_close():
            """执行关闭"""
            def show_result_dialog(title: str, message: str, is_success: bool):
                """显示结果弹窗"""
                def close_dialog(e):
                    result_dialog.open = False
                    self.page.update()
                
                result_dialog = ft.AlertDialog(
                    modal=True,
                    title=ft.Text(title, weight=ft.FontWeight.BOLD),
                    content=ft.Text(message, size=14),
                    actions=[
                        ft.ElevatedButton(
                            "确定", 
                            bgcolor="#4CAF50" if is_success else "#E53935", 
                            color="#ffffff", 
                            on_click=close_dialog
                        ),
                    ],
                    actions_alignment=ft.MainAxisAlignment.END,
                )
                self.page.overlay.append(result_dialog)
                result_dialog.open = True
                self.page.update()
            
            try:
                closed_count = 0
                for proc in psutil.process_iter(['pid', 'name']):
                    try:
                        name = proc.info.get('name', '').lower()
                        if '直播伴侣' in name or 'webcast' in name or 'livemate' in name:
                            proc.terminate()
                            closed_count += 1
                    except:
                        pass
                
                if closed_count > 0:
                    self.right_panel.add_system(f"✅ 已关闭 {closed_count} 个直播伴侣进程")
                    show_result_dialog("关闭成功", f"已成功关闭 {closed_count} 个直播伴侣进程", True)
                else:
                    self.right_panel.add_system("⚠ 未找到直播伴侣进程")
                    show_result_dialog("未找到进程", "未找到直播伴侣进程，可能已经关闭", False)
            except Exception as e:
                self.right_panel.add_system(f"❌ 关闭失败: {e}")
                show_result_dialog("关闭失败", f"关闭直播伴侣时出错：{e}", False)
        
        def on_confirm(e):
            dialog.open = False
            self.page.update()
            do_close()
        
        def on_cancel(e):
            dialog.open = False
            self.page.update()
        
        # 显示确认对话框
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text("关闭直播伴侣", weight=ft.FontWeight.BOLD),
            content=ft.Text("请确认您已执行「切换OBS」操作！\n\n关闭直播伴侣后，将停止直播伴侣的推流，\n由 OBS 接管推流。", size=14),
            actions=[
                ft.TextButton("取消", on_click=on_cancel),
                ft.ElevatedButton("确认关闭", bgcolor="#E53935", color="#ffffff", on_click=on_confirm),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _on_switch_obs(self):
        """切换到 OBS 直播 - 使用直播伴侣抓取的真实推流码"""
        import threading
        
        url = self.stream_manager.status.stream_url
        key = self.stream_manager.status.stream_key
        
        # 严格检查：必须有真实抓取的推流码
        if not url or not key:
            self.right_panel.add_system("❌ 还没有捕获到推流码")
            self.right_panel.add_system("请先点击「开播」通过直播伴侣抓取推流信息")
            self.page.update()
            return
        
        # 验证是真实推流码（非测试）
        if 'stream-' in key and len(key) < 50:
            self.right_panel.add_system("❌ 当前是测试推流码，请先抓取真实推流信息")
            self.page.update()
            return
        
        def do_switch():
            def on_progress(msg):
                self.right_panel.add_system(f"OBS: {msg}")
                self.page.update()
            
            self.right_panel.add_system("【切换OBS】使用真实推流码")
            self.right_panel.add_system(f"服务器: {url}")
            self.right_panel.add_system(f"密钥: {key[:30]}...")
            self.home_tab.switch_obs_btn.disabled = True
            self.page.update()
            
            try:
                from src.stream import ObsManager
                
                obs_manager = ObsManager(logger=lambda msg: print(f"[OBS] {msg}"))
                
                # 检查 OBS
                if not obs_manager.is_installed:
                    self.right_panel.add_system("❌ OBS 未安装，请先安装 OBS Studio")
                    import webbrowser
                    webbrowser.open("https://obsproject.com/download")
                    return
                
                on_progress("正在启动 OBS...")
                
                # 使用完全自动化推流（与推送OBS相同逻辑）
                success, message = obs_manager.full_auto_stream(url, key, on_progress=on_progress)
                
                if success:
                    self.right_panel.add_system("✅ OBS 推流成功!")
                    self.home_tab.update_stream_status("connected", "OBS推流中")
                else:
                    self.right_panel.add_system(f"❌ 推流失败: {message}")
                    
            except Exception as e:
                self.right_panel.add_system(f"❌ 错误: {e}")
            finally:
                self.home_tab.switch_obs_btn.disabled = False
                self.page.update()
        
        threading.Thread(target=do_switch, daemon=True).start()
    
    def _on_copy_url(self):
        url = self.stream_manager.status.stream_url
        if url:
            self.page.set_clipboard(url)
            self.right_panel.add_system("✓ 服务器地址已复制")
    
    def _on_copy_key(self):
        key = self.stream_manager.status.stream_key
        if key:
            self.page.set_clipboard(key)
            self.right_panel.add_system("✓ 推流密钥已复制")
    
    def _on_browse_companion(self):
        """浏览直播伴侣路径"""
        def on_result(e: ft.FilePickerResultEvent):
            if e.files:
                path = e.files[0].path
                self.config_manager.set_companion_path(path)
                self.env_checker.set_companion_path(path)
        
        picker = ft.FilePicker(on_result=on_result)
        self.page.overlay.append(picker)
        self.page.update()
        picker.pick_files(allowed_extensions=["exe"])
    
    def _on_connect(self, e=None):
        """连接直播间"""
        room_id = self.config_manager.get_room_id()
        if not room_id:
            self._show_need_room_id_dialog()
            return
        
        # 获取主播昵称和忽略选项
        anchor_name = self.home_tab.anchor_name_input.value or ""
        ignore_anchor = self.home_tab.ignore_anchor_check.value
        self.config_manager.set_anchor_name(anchor_name)
        self.config_manager.set_ignore_anchor(ignore_anchor)
        
        config = self.config_manager.build_full_config()
        self.worker_manager.start(room_id, config)
    
    def _on_stop(self, e):
        """停止连接"""
        self.worker_manager.stop()
        # 更新 UI 状态
        self._on_worker_status("disconnected")
        self.right_panel.add_system("已停止连接")
    
    def _on_danmu_login(self):
        """弹幕登录"""
        import threading
        import os
        import traceback
        import io
        from pathlib import Path
        
        # 导入日志函数
        from src.danmu.sender import danmu_log, get_log_file_path
        
        def log(msg, level="INFO"):
            """写入日志到文件"""
            danmu_log(msg, level, "App")
        
        log("=" * 60)
        log("_on_danmu_login 被调用")
        log(f"日志文件路径: {get_log_file_path()}")
        log("=" * 60)
        
        # 在界面上提示日志文件位置
        self.right_panel.add_system(f"📝 日志: {get_log_file_path()}")
        
        # 获取直播间ID
        room_id = self.config_manager.get_room_id()
        log(f"直播间ID: {room_id}")
        
        if not room_id:
            log("直播间ID为空，显示提示对话框", "WARN")
            self._show_need_room_id_dialog()
            return
        
        # 检查是否已有浏览器实例
        log(f"danmu_sender 实例: {self.danmu_sender}")
        if self.danmu_sender:
            log(f"danmu_sender._started: {getattr(self.danmu_sender, '_started', 'N/A')}")
        
        # 如果浏览器已启动，刷新页面（支持切换主播ID）
        if self.danmu_sender and getattr(self.danmu_sender, '_started', False):
            log("浏览器已启动，执行刷新操作")
            self.right_panel.add_system("🔄 正在刷新浏览器页面...")
            def refresh_browser():
                try:
                    log("开始刷新浏览器页面...")
                    self.danmu_sender.refresh_page(room_id)
                    if self.danmu_sender.is_logged_in:
                        log("刷新完成，已登录")
                        self.right_panel.add_system("✅ 页面已刷新，已登录")
                        self.page.run_thread(lambda: self._update_danmu_status(True))
                    else:
                        log("刷新完成，未登录，等待用户登录")
                        self.right_panel.add_system("⏳ 请在浏览器中登录抖音...")
                        self._wait_danmu_login()
                except Exception as e:
                    log(f"刷新失败: {e}", "ERROR")
                    # 将 traceback 写入日志
                    tb_str = io.StringIO()
                    traceback.print_exc(file=tb_str)
                    log(f"堆栈跟踪:\n{tb_str.getvalue()}", "ERROR")
                    self.right_panel.add_system(f"❌ 刷新失败: {e}")
            threading.Thread(target=refresh_browser, daemon=True).start()
            return
        
        # 检查并自动检索浏览器路径
        log("检查浏览器路径...")
        chrome_path = self.config_manager.get_chrome_path()
        log(f"配置中的浏览器路径: {chrome_path}")
        
        if not chrome_path or not os.path.exists(chrome_path):
            log("配置的路径为空或不存在，自动检索浏览器...")
            # 自动检索 Chrome 路径
            chrome_paths = [
                r"C:\Program Files\Google\Chrome\Application\chrome.exe",
                r"C:\Program Files (x86)\Google\Chrome\Application\chrome.exe",
                str(Path.home() / "AppData" / "Local" / "Google" / "Chrome" / "Application" / "chrome.exe"),
                # Edge 浏览器作为备选
                r"C:\Program Files (x86)\Microsoft\Edge\Application\msedge.exe",
                r"C:\Program Files\Microsoft\Edge\Application\msedge.exe",
            ]
            
            found_path = None
            for path in chrome_paths:
                exists = os.path.exists(path)
                log(f"  检查: {path} -> {'存在' if exists else '不存在'}")
                if exists and not found_path:
                    found_path = path
            
            if found_path:
                # 自动保存检索到的路径
                log(f"找到浏览器: {found_path}")
                self.config_manager.set_chrome_path(found_path)
                self.config_manager.save()
                chrome_path = found_path
                self.right_panel.add_system(f"✓ 自动检测到浏览器: {os.path.basename(found_path)}")
            else:
                # 未找到浏览器
                log("未找到任何可用的浏览器！", "ERROR")
                self._show_browser_not_found_dialog()
                return
        else:
            log(f"使用配置的浏览器路径: {chrome_path}")
        
        self.right_panel.add_system("🔄 正在启动浏览器...")
        log("准备在后台线程中启动浏览器...")
        
        def start_sender():
            log("start_sender 线程开始执行")
            try:
                log("导入 init_danmu_sender...")
                from src.danmu import init_danmu_sender
                log("init_danmu_sender 导入成功")
                
                log(f"调用 init_danmu_sender(room_id={room_id}, use_existing=True, chrome_path={chrome_path})")
                self.danmu_sender = init_danmu_sender(room_id, use_existing=True, chrome_path=chrome_path)
                log(f"init_danmu_sender 返回: {self.danmu_sender}")
                
                log("调用 danmu_sender.start()...")
                success = self.danmu_sender.start()
                log(f"danmu_sender.start() 返回: {success}")
                
                if success:
                    log("浏览器启动成功")
                    self.right_panel.add_system("✅ 浏览器已启动")
                    
                    is_logged_in = self.danmu_sender.is_logged_in
                    log(f"登录状态: {is_logged_in}")
                    
                    if is_logged_in:
                        log("已登录抖音")
                        self.right_panel.add_system("✅ 已登录抖音")
                        self.page.run_thread(lambda: self._update_danmu_status(True))
                    else:
                        log("未登录，等待用户在浏览器中登录...")
                        self.right_panel.add_system("⏳ 请在浏览器中登录抖音...")
                        self._wait_danmu_login()
                else:
                    log("浏览器启动失败", "ERROR")
                    self.right_panel.add_system("❌ 浏览器启动失败，请查看控制台日志")
                        
            except Exception as e:
                log(f"start_sender 异常: {e}", "ERROR")
                log(f"异常类型: {type(e).__name__}", "ERROR")
                # 将 traceback 写入日志
                tb_str = io.StringIO()
                traceback.print_exc(file=tb_str)
                log(f"堆栈跟踪:\n{tb_str.getvalue()}", "ERROR")
                self.right_panel.add_system(f"❌ 错误: {e}")
            
            log("start_sender 线程结束")
        
        log("创建并启动 start_sender 线程")
        thread = threading.Thread(target=start_sender, daemon=True)
        thread.start()
        log(f"线程已启动: {thread.name}, is_alive={thread.is_alive()}")
    
    def _show_browser_not_found_dialog(self):
        """显示浏览器未找到对话框"""
        c = theme.colors
        
        def on_download(e):
            import webbrowser
            webbrowser.open("https://www.google.cn/chrome/")
            self._close_dialog(dialog)
            self.right_panel.add_system("💡 请安装 Chrome 后重新点击登录")
        
        dialog = ft.AlertDialog(
            title=ft.Row([
                ft.Icon(ft.Icons.BROWSER_NOT_SUPPORTED, color=ft.Colors.ORANGE_400),
                ft.Text("未检测到浏览器", weight=ft.FontWeight.BOLD),
            ], spacing=8),
            content=ft.Column([
                ft.Text("弹幕发送功能需要浏览器支持。", size=14),
                ft.Text("", size=8),
                ft.Text("系统未检测到以下浏览器：", size=13, color=c.text_muted),
                ft.Text("• Google Chrome", size=13),
                ft.Text("• Microsoft Edge", size=13),
                ft.Text("", size=8),
                ft.Text("推荐安装 Chrome 浏览器（免费）", size=13, weight=ft.FontWeight.BOLD, color=c.accent),
            ], tight=True, spacing=4),
            actions=[
                ft.ElevatedButton("立即下载 Chrome", bgcolor=c.accent, color="#ffffff", on_click=on_download),
                ft.TextButton("稍后", on_click=lambda e: self._close_dialog(dialog)),
            ],
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _show_need_room_id_dialog(self):
        """显示需要输入直播间ID的对话框"""
        c = theme.colors
        
        dialog = ft.AlertDialog(
            title=ft.Row([
                ft.Icon(ft.Icons.INFO_OUTLINE, color=c.accent),
                ft.Text("请先输入直播间ID", weight=ft.FontWeight.BOLD),
            ], spacing=8),
            content=ft.Column([
                ft.Text("在首页的「直播间ID」输入框中输入你的直播间ID。", size=14),
                ft.Text("", size=8),
                ft.Text("如何获取直播间ID：", size=13, color=c.text_muted),
                ft.Text("1. 打开抖音直播伴侣，开始直播", size=13),
                ft.Text("2. 复制直播间链接，ID是链接中的数字", size=13),
                ft.Text("   例如: live.douyin.com/123456789", size=12, color=c.text_muted),
            ], tight=True, spacing=4),
            actions=[
                ft.TextButton("知道了", on_click=lambda e: self._close_dialog(dialog)),
            ],
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _wait_danmu_login(self):
        """等待弹幕登录"""
        import time
        for _ in range(120):
            if self.danmu_sender and self.danmu_sender.is_logged_in:
                self.page.run_thread(lambda: self._update_danmu_status(True))
                return
            time.sleep(1)
    
    def _update_danmu_status(self, logged_in: bool):
        if self.home_tab:
            self.home_tab.set_danmu_logged_in(logged_in)
            # 同步到悬浮窗口
            if hasattr(self, '_float_window') and self._float_window:
                if logged_in:
                    self._float_window.update_status("已连接", True)
                    self._float_window.set_send_enabled(True)
                else:
                    self._float_window.update_status("未连接", False)
                    self._float_window.set_send_enabled(False)
            self.page.update()
    
    def _on_send_danmu(self, text: str):
        """发送弹幕"""
        if self.danmu_sender and self.danmu_sender.is_logged_in:
            import threading
            def send():
                try:
                    self.danmu_sender.send(text)
                    self.right_panel.add_system(f"已发送: {text}")
                    self.page.update()
                except Exception as e:
                    self.right_panel.add_system(f"发送失败: {e}")
                    self.page.update()
            threading.Thread(target=send, daemon=True).start()
    
    def _on_preview_voice(self, engine: str, params: dict, volume: float):
        """预览语音"""
        from .services.worker_service import PreviewManager
        
        if self.voice_tab:
            self.voice_tab.set_preview_state(True)
        
        preview = PreviewManager()
        preview.preview(
            engine, params, volume,
            on_finished=lambda: self._on_preview_done(),
            on_error=lambda e: self._on_preview_done(e)
        )
    
    def _on_preview_done(self, error: str = None):
        if self.voice_tab:
            self.voice_tab.set_preview_state(False)
            self.page.update()
        if error:
            self.right_panel.add_system(f"❌ 预览失败: {error}")
    
    def _on_voice_config_change(self, config: dict):
        """语音配置变化 - 同时保存到本地"""
        if self.worker_manager:
            self.worker_manager.update_config(config)
        
        # 保存语音配置到本地
        tts_cfg = config.get("tts", {})
        self.config_manager.set_tts_engine(tts_cfg.get("engine", "edge"))
        
        edge_cfg = tts_cfg.get("edge", {})
        self.config_manager.set_edge_voice(edge_cfg.get("voice", "zh-CN-XiaoxiaoNeural"))
        self.config_manager.set_edge_rate(edge_cfg.get("rate", "+0%"))
        
        audio_cfg = config.get("audio", {})
        self.config_manager.set_volume(audio_cfg.get("volume", 0.8))
        
        # 保存回复规则和模板
        reply_cfg = config.get("reply", {})
        self.config_manager.set_reply_config(reply_cfg)
        
        # 持久化到磁盘
        self.config_manager.save()
    
    def _on_voice_config_save(self, config: dict):
        """语音配置保存后 - 热更新 TTS 引擎"""
        if self.worker_manager:
            # 获取完整配置并同步到 worker（触发 TTS 热更新）
            full_config = self.config_manager.config
            self.worker_manager.update_config(full_config)
            tts_cfg = config.get("tts", {})
            print(f"[DEBUG] 语音配置已保存并同步: engine={tts_cfg.get('engine')}, voice={tts_cfg.get('edge', {}).get('voice')}")
    
    def _on_test_ai(self, provider: str, model: str, api_key: str):
        """测试 AI 连接 - 异步非阻塞"""
        import threading
        import requests
        
        # 检查 API Key
        if not api_key or not api_key.strip():
            self.ai_tab.test_status.value = "请先输入 API Key"
            self.ai_tab.test_status.color = ft.Colors.ORANGE_400
            self.page.update()
            return
        
        # 立即更新 UI 状态
        self.ai_tab.test_btn.disabled = True
        self.ai_tab.test_btn.text = "测试中..."
        self.ai_tab.test_status.value = ""
        self.page.update()
        
        def test():
            import time
            success = False
            message = ""
            try:
                # 获取 base_url
                from gui.flet_app.tabs.ai import AI_PROVIDERS
                provider_info = AI_PROVIDERS.get(provider, {})
                base_urls = {
                    "deepseek": "https://api.deepseek.com",
                    "dashscope": "https://dashscope.aliyuncs.com/compatible-mode/v1",
                    "openai": "https://api.openai.com/v1",
                }
                base_url = base_urls.get(provider, "https://api.deepseek.com")
                
                # 使用 requests 快速测试，设置短超时
                headers = {
                    "Authorization": f"Bearer {api_key}",
                    "Content-Type": "application/json"
                }
                data = {
                    "model": model,
                    "messages": [{"role": "user", "content": "hi"}],
                    "max_tokens": 5
                }
                url = f"{base_url}/chat/completions"
                
                # 记录请求日志
                log_request("POST", url, headers, data)
                start_time = time.time()
                
                response = requests.post(url, headers=headers, json=data, timeout=8)
                duration_ms = (time.time() - start_time) * 1000
                
                if response.status_code == 200:
                    success = True
                    message = "连接成功!"
                    log_response(url, status_code=200, success=True, data="OK", duration_ms=duration_ms)
                else:
                    try:
                        err = response.json()
                        message = err.get("error", {}).get("message", response.text[:80])
                    except:
                        message = f"HTTP {response.status_code}"
                    log_response(url, status_code=response.status_code, success=False, error=message, duration_ms=duration_ms)
            except requests.Timeout:
                message = "连接超时"
                log_response(url, success=False, error=message, duration_ms=(time.time() - start_time) * 1000 if 'start_time' in dir() else 0)
            except Exception as e:
                message = str(e)[:80]
                log_response(url if 'url' in dir() else "unknown", success=False, error=message)
            
            # 更新结果 - 直接更新控件
            self.ai_tab.set_test_result(success, message)
            if self.page:
                self.page.update()
        
        threading.Thread(target=test, daemon=True).start()
    
    def _on_ai_config_change(self, config: dict):
        """AI 配置变化（未保存）"""
        pass
    
    def _on_ai_config_save(self, config: dict):
        """AI 配置保存后 - 同步到 worker"""
        if self.worker_manager:
            # 获取完整配置并同步到 worker
            full_config = self.config_manager.config
            self.worker_manager.update_config(full_config)
            ai_cfg = config.get("ai", {})
            print(f"[DEBUG] AI 配置已同步到 worker: ai_enabled={ai_cfg.get('enabled')}, warm_enabled={ai_cfg.get('warm_enabled')}, warm_interval={ai_cfg.get('warm_interval')}")
    
    def _on_avatar_toggle(self, enabled: bool):
        """形象开关 - Web Server 全局运行，这里只更新 UI"""
        # 更新 URL 使用实际端口
        model = self.avatar_tab.model_dropdown.value or "Hiyori"
        port = getattr(self, '_web_server_port', 8080)
        self.avatar_tab.url_input.value = f"http://127.0.0.1:{port}/live2d/index.html?model={model}"
        
        if not enabled:
            # 关闭形象服务时也关闭桌面模式
            if self.avatar_tab.desktop_check.value:
                self.avatar_tab.desktop_check.value = False
                self._on_desktop_toggle(False)
        self.page.update()
    
    def _start_web_server(self):
        """启动 Live2D Web Server"""
        if hasattr(self, '_web_server') and self._web_server:
            return
        try:
            from src.web.server import WebServerThread
            
            # 确定 static 目录路径
            if getattr(sys, 'frozen', False):
                # 打包环境 - 多个可能的位置
                base_dir = Path(sys.executable).parent
                static_paths = [
                    base_dir / "static",
                    base_dir / "_internal" / "static",
                ]
                if hasattr(sys, '_MEIPASS'):
                    static_paths.insert(0, Path(sys._MEIPASS) / "static")
                
                static_dir = None
                for p in static_paths:
                    print(f"[WebServer] 检查路径: {p}, exists={p.exists()}")
                    if p.exists():
                        static_dir = p
                        break
                
                if not static_dir:
                    self.right_panel.add_system(f"Web服务启动失败: 找不到 static 目录 (检查了: {[str(p) for p in static_paths]})")
                    return
            else:
                static_dir = ROOT_DIR / "static"
            
            print(f"[WebServer] 使用 static 目录: {static_dir}")
            
            # 检查 live2d 目录是否存在
            live2d_dir = static_dir / "live2d"
            if not live2d_dir.exists():
                self.right_panel.add_system(f"Web服务启动失败: live2d 目录不存在 ({live2d_dir})")
                return
            
            # 启动服务器（端口占用时自动切换）
            self._web_server = WebServerThread(port=8080, root_dir=str(static_dir), auto_port=True)
            self._web_server.start()
            
            # 等待服务器启动完成
            if self._web_server.wait_for_start(timeout=2.0):
                actual_port = self._web_server.get_actual_port()
                self._web_server_port = actual_port
                
                # 更新 UI 中的 URL
                model = self.avatar_tab.model_dropdown.value or "Hiyori"
                new_url = f"http://127.0.0.1:{actual_port}/live2d/index.html?model={model}"
                self.avatar_tab.url_input.value = new_url
                
                self.right_panel.add_system(f"Live2D Web 服务已启动 (端口 {actual_port})")
            else:
                self.right_panel.add_system("Web服务启动失败: 超时")
                self._web_server = None
        except Exception as e:
            import traceback
            traceback.print_exc()
            self.right_panel.add_system(f"Web服务启动失败: {e}")
    
    def _stop_web_server(self):
        """停止 Web Server"""
        if hasattr(self, '_web_server') and self._web_server:
            self._web_server.stop()
            self._web_server = None
            self.right_panel.add_system("Live2D Web 服务已停止")
    
    def _start_web_server_background(self):
        """后台启动 Web Server（系统启动时调用）"""
        import threading
        import traceback
        from datetime import datetime
        from pathlib import Path
        
        # 确定日志目录（与 server.py 使用相同逻辑）
        if getattr(sys, 'frozen', False):
            log_base = Path(sys.executable).parent
        else:
            log_base = ROOT_DIR
        log_dir = log_base / "logs"
        log_dir.mkdir(exist_ok=True)
        log_file = log_dir / "webserver.log"
        
        def _log(msg):
            """写入日志文件"""
            try:
                timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                with open(log_file, "a", encoding="utf-8") as f:
                    f.write(f"{timestamp} | APP   | {msg}\n")
                    f.flush()
            except Exception as e:
                # 最后的备用：写到用户目录
                try:
                    backup_log = Path.home() / "webserver_debug.log"
                    with open(backup_log, "a", encoding="utf-8") as f:
                        f.write(f"{datetime.now()} | {msg} | ERROR: {e}\n")
                except:
                    pass
        
        # 立即记录启动
        _log("_start_web_server_background 被调用")
        
        def start():
            try:
                _log("========== 开始启动 Web Server ==========")
                from src.web.server import WebServerThread
                
                # 确定 static 目录路径
                _log(f"frozen={getattr(sys, 'frozen', False)}")
                _log(f"ROOT_DIR={ROOT_DIR}")
                
                if getattr(sys, 'frozen', False):
                    base_dir = Path(sys.executable).parent
                    _log(f"base_dir={base_dir}")
                    static_paths = [
                        base_dir / "static",
                        base_dir / "_internal" / "static",
                    ]
                    if hasattr(sys, '_MEIPASS'):
                        static_paths.insert(0, Path(sys._MEIPASS) / "static")
                        _log(f"_MEIPASS={sys._MEIPASS}")
                    
                    static_dir = None
                    for p in static_paths:
                        exists = p.exists()
                        _log(f"检查: {p} -> exists={exists}")
                        if exists:
                            static_dir = p
                            break
                    
                    if not static_dir:
                        _log(f"错误: 找不到 static 目录，检查了: {static_paths}")
                        return
                else:
                    static_dir = ROOT_DIR / "static"
                    _log(f"开发环境，使用: {static_dir}")
                
                _log(f"使用 static 目录: {static_dir}")
                
                # 检查 live2d 目录
                live2d_dir = static_dir / "live2d"
                if not live2d_dir.exists():
                    _log(f"错误: live2d 目录不存在: {live2d_dir}")
                    return
                
                # 列出 live2d 目录内容
                try:
                    contents = list(live2d_dir.iterdir())
                    _log(f"live2d 目录内容: {[c.name for c in contents[:10]]}")
                except Exception as e:
                    _log(f"无法列出 live2d 目录: {e}")
                
                _log(f"创建 WebServerThread, port=8080, root_dir={static_dir}")
                self._web_server = WebServerThread(port=8080, root_dir=str(static_dir), auto_port=True)
                
                _log("调用 start()...")
                self._web_server.start()
                
                _log("等待启动完成...")
                if self._web_server.wait_for_start(timeout=3.0):
                    self._web_server_port = self._web_server.get_actual_port()
                    _log(f"✓ 启动成功，端口: {self._web_server_port}")
                    
                    # 延迟更新 UI（等 avatar_tab 初始化完成）
                    import time
                    time.sleep(0.5)
                    if hasattr(self, 'avatar_tab') and self.avatar_tab:
                        model = self.avatar_tab.model_dropdown.value or "Hiyori"
                        self.avatar_tab.url_input.value = f"http://127.0.0.1:{self._web_server_port}/live2d/index.html?model={model}"
                        _log("URL 已更新")
                    
                    # 添加系统消息
                    if hasattr(self, 'right_panel') and self.right_panel:
                        self.right_panel.add_system(f"Live2D Web 服务已启动 (端口 {self._web_server_port})")
                else:
                    _log("✗ 启动超时")
                    if self._web_server:
                        error = self._web_server.get_error()
                        if error:
                            _log(f"错误信息: {error}")
                    self._web_server = None
            except Exception as e:
                _log(f"✗ 启动异常: {e}")
                _log(traceback.format_exc())
        
        _log("启动后台线程...")
        threading.Thread(target=start, daemon=True).start()
    
    def _on_desktop_toggle(self, enabled: bool):
        """桌面模式开关 - 启动/停止桌面悬浮窗"""
        if enabled:
            # 先启用 Web Server
            if not self.avatar_tab.avatar_check.value:
                self.avatar_tab.avatar_check.value = True
                self._on_avatar_toggle(True)
            
            # 启动桌面悬浮窗
            self._start_avatar_window()
            self.avatar_tab.unlock_btn.disabled = False
        else:
            self._stop_avatar_window()
            self.avatar_tab.unlock_btn.disabled = True
        self.page.update()
    
    def _start_avatar_window(self):
        """启动 Live2D 桌面悬浮窗（独立进程）"""
        if hasattr(self, '_avatar_process') and self._avatar_process and self._avatar_process.poll() is None:
            return
        try:
            model = self.avatar_tab.model_dropdown.value or "Hiyori"
            
            if getattr(sys, 'frozen', False):
                # 打包后环境 - 查找 AvatarWindow.exe
                exe_paths = [
                    Path(sys.executable).parent / "AvatarWindow.exe",
                    Path(sys.executable).parent / "_internal" / "AvatarWindow.exe",
                ]
                if hasattr(sys, '_MEIPASS'):
                    exe_paths.insert(0, Path(sys._MEIPASS) / "AvatarWindow.exe")
                
                avatar_exe = next((p for p in exe_paths if p.exists()), None)
                if avatar_exe:
                    self._avatar_process = subprocess.Popen(
                        [str(avatar_exe), model],
                        cwd=str(avatar_exe.parent),
                        creationflags=subprocess.CREATE_NO_WINDOW if sys.platform == 'win32' else 0,
                    )
                    self.right_panel.add_system(f"桌面形象已启动: {model}")
                else:
                    self.right_panel.add_system("桌面形象程序未找到")
                    self.avatar_tab.desktop_check.value = False
            else:
                # 开发环境 - 运行 Python 脚本
                script_path = ROOT_DIR / "src" / "gui" / "run_avatar.py"
                self._avatar_process = subprocess.Popen(
                    [sys.executable, str(script_path), model],
                    cwd=str(ROOT_DIR)
                )
                self.right_panel.add_system(f"桌面形象已启动: {model}")
        except Exception as e:
            self.right_panel.add_system(f"桌面形象启动失败: {e}")
            self.avatar_tab.desktop_check.value = False
    
    def _stop_avatar_window(self):
        """停止桌面悬浮窗"""
        if hasattr(self, '_avatar_process') and self._avatar_process:
            try:
                self._avatar_process.terminate()
            except:
                pass
            self._avatar_process = None
            self.right_panel.add_system("桌面形象已关闭")
    
    def _on_model_change(self, model: str):
        """模型变化"""
        # 更新 URL（使用实际端口）
        port = getattr(self, '_web_server_port', 8080)
        url = f"http://127.0.0.1:{port}/live2d/index.html?model={model}"
        self.avatar_tab.url_input.value = url
        
        # 如果桌面进程在运行，重启以切换模型
        if hasattr(self, '_avatar_process') and self._avatar_process and self._avatar_process.poll() is None:
            self._stop_avatar_window()
            self._start_avatar_window()
        
        self.page.update()
    
    def _on_unlock_toggle(self):
        """解锁/锁定形象 - 独立进程模式下通过右键菜单操作"""
        self.right_panel.add_system("提示: 请在桌面形象上右键点击，选择解锁/锁定")
        self.page.update()
    
    def _stop_overlay(self):
        """停止覆盖层进程"""
        if self._overlay_process:
            try:
                self._overlay_process.kill()
                self._overlay_process.wait(timeout=2)
            except:
                pass
            finally:
                self._overlay_process = None
    
    def _start_overlay(self, strength: str = "medium"):
        """启动覆盖层进程"""
        try:
            if getattr(sys, 'frozen', False):
                args = [sys.executable, "--overlay", "--strength", strength]
            else:
                script_path = ROOT_DIR / "src" / "utils" / "desktop_overlay.py"
                args = [sys.executable, str(script_path), "--strength", strength]
            
            self._overlay_process = subprocess.Popen(args)
            self.anti_detect_tab.set_visual_status(True)
            return True
        except Exception as e:
            print(f"Overlay start failed: {e}")
            return False
    
    def _on_visual_toggle(self, enabled: bool):
        """画面防检测开关"""
        if enabled:
            strength = self.anti_detect_tab.visual_strength.value or "medium"
            self._start_overlay(strength)
        else:
            self._stop_overlay()
            self.anti_detect_tab.set_visual_status(False)
        self.page.update()
    
    def _on_visual_strength(self, strength: str):
        """画面防检测强度 - 如果正在运行则重启"""
        if self._overlay_process:
            self._stop_overlay()
            self._start_overlay(strength)
            self.page.update()
    
    def _on_typing_toggle(self, enabled: bool):
        """打字防检测开关"""
        if self.danmu_sender:
            speeds = {"slow": "slow", "normal": "normal", "fast": "fast"}
            speed = speeds.get(self.anti_detect_tab.typing_speed.value, "normal")
            self.danmu_sender.set_human_typing(enabled, speed)
    
    def _on_typing_speed(self, speed: str):
        """打字速度"""
        if self.danmu_sender:
            enabled = self.anti_detect_tab.typing_check.value
            self.danmu_sender.set_human_typing(enabled, speed)
    
    def _on_open_timer(self):
        """打开/关闭独立计时器窗口（toggle）"""
        try:
            # 检查是否有 FlipClock.exe 进程在运行
            import os
            result = os.popen('tasklist /FI "IMAGENAME eq FlipClock.exe" /NH').read()
            if 'FlipClock.exe' in result:
                # 有进程在运行，杀死它
                os.system('taskkill /F /IM FlipClock.exe >nul 2>&1')
                self._timer_process = None
                self.right_panel.add_system(f"✓ 翻页时钟已关闭")
                try:
                    self.page.show_snack_bar(ft.SnackBar(content=ft.Text("✓ 翻页时钟已关闭")))
                except Exception:
                    pass
                self.page.update()
                return
            
            if getattr(sys, 'frozen', False):
                # 打包环境：使用 FlipClock.exe
                timer_exe = ROOT_DIR / "FlipClock.exe"
                if timer_exe.exists():
                    self._timer_process = subprocess.Popen(
                        [str(timer_exe)],
                        stdout=subprocess.PIPE,
                        stderr=subprocess.PIPE
                    )
                    self.right_panel.add_system(f"✓ 翻页时钟已启动")
                    try:
                        self.page.show_snack_bar(ft.SnackBar(content=ft.Text("✓ 翻页时钟已启动")))
                    except Exception:
                        pass
                else:
                    self.right_panel.add_system(f"❌ 未找到翻页时钟: {timer_exe}")
                    try:
                        self.page.show_snack_bar(ft.SnackBar(content=ft.Text("❌ 未找到翻页时钟")))
                    except Exception:
                        pass
            else:
                # 开发环境：使用 Python 启动 timer.py
                if self._timer_process and self._timer_process.poll() is None:
                    try:
                        self._timer_process.terminate()
                    except Exception:
                        pass
                    self._timer_process = None
                    try:
                        self.page.show_snack_bar(ft.SnackBar(content=ft.Text("翻页时钟已在运行，正在重新打开...")))
                    except Exception:
                        pass
                
                timer_path = ROOT_DIR / "tools" / "timer" / "timer.py"
                if not timer_path.exists():
                    timer_path = ROOT_DIR.parent / "timer" / "timer.py"
                
                if timer_path.exists():
                    self._timer_process = subprocess.Popen(
                        [sys.executable, str(timer_path)],
                        cwd=str(timer_path.parent),
                    )
                    self.right_panel.add_system(f"✓ 翻页时钟已启动")
                    try:
                        self.page.show_snack_bar(ft.SnackBar(content=ft.Text("✓ 翻页时钟已启动")))
                    except Exception:
                        pass
                else:
                    self.right_panel.add_system(f"❌ 未找到翻页时钟")
                    try:
                        self.page.show_snack_bar(ft.SnackBar(content=ft.Text("❌ 未找到翻页时钟")))
                    except Exception:
                        pass
            self.page.update()
        except Exception as e:
            self.right_panel.add_system(f"❌ 启动翻页时钟失败: {e}")
            try:
                self.page.show_snack_bar(ft.SnackBar(content=ft.Text(f"❌ 启动翻页时钟失败: {e}")))
            except Exception:
                pass
            self.page.update()
    
    def _on_float_window(self):
        """独立窗口 - 创建置顶悬浮窗口"""
        try:
            from .components.float_window import FloatingMessageWindow, is_available
            
            if not is_available():
                self.right_panel.add_system("❌ 独立窗口需要 PyQt6，请安装: pip install PyQt6")
                self.page.update()
                return
            
            # 如果已存在窗口，直接显示
            if hasattr(self, '_float_window') and self._float_window:
                self._float_window.show()
                self._float_window.raise_()
                return
            
            # 创建独立窗口
            self._float_window = FloatingMessageWindow(
                on_send=self._on_float_send_danmu,
                on_close=self._on_float_close
            )
            
            # 先显示窗口
            self._float_window.show()
            
            # 等待窗口初始化完成后再设置状态
            import time
            time.sleep(0.3)
            
            # 更新连接状态 - 检查直播间连接和弹幕发送器
            is_connected = self.worker_manager._running if hasattr(self.worker_manager, '_running') else False
            is_danmu_logged = self.danmu_sender and self.danmu_sender.is_logged_in
            
            if is_connected:
                self._float_window.update_status("已连接", True)
                self._float_window.set_send_enabled(is_danmu_logged)
            elif is_danmu_logged:
                self._float_window.update_status("弹幕已登录", True)
                self._float_window.set_send_enabled(True)
            else:
                self._float_window.update_status("未连接", False)
                self._float_window.set_send_enabled(False)
            self.right_panel.add_system("✓ 独立窗口已打开")
            self.page.update()
            
        except Exception as e:
            self.right_panel.add_system(f"❌ 打开独立窗口失败: {e}")
            self.page.update()
    
    def _on_float_send_danmu(self, text: str):
        """从悬浮窗口发送弹幕"""
        if self.danmu_sender and self.danmu_sender.is_logged_in:
            self.danmu_sender.send(text)
            # 独立窗口已经自己添加了消息，这里只同步到主程序右侧面板
            self.right_panel.add_message("我", text, "chat")
            self.page.update()
    
    def _on_float_close(self):
        """悬浮窗口关闭"""
        self._float_window = None
    
    def _sync_to_float_window(self, msg_type: str, user: str, content: str):
        """同步消息到悬浮窗口"""
        if hasattr(self, '_float_window') and self._float_window and self._float_window.isVisible():
            self._float_window.append_message(msg_type, user, content)
    
    def _sync_float_window_status(self, status: str, connected: bool):
        """同步连接状态到独立窗口"""
        if hasattr(self, '_float_window') and self._float_window and self._float_window.isVisible():
            self._float_window.update_status(status, connected)
            self._float_window.set_send_enabled(connected and self.danmu_sender and self.danmu_sender.is_logged_in)
    
    def _on_minimize(self, e=None):
        """最小化窗口"""
        if hasattr(self.page, 'window') and hasattr(self.page.window, 'minimized'):
            self.page.window.minimized = True
        else:
            self.page.window_minimized = True
        self.page.update()
    
    # ==================== 授权管理 ====================
    
    def _check_license(self):
        """检查授权状态（静默检查，不强制弹窗）"""
        if not HAS_LICENSE_MODULE:
            self._license_valid = False  # 无授权模块时默认未授权
            self._update_title_bar_license(False)
            return
        
        # 设置授权状态变化回调
        get_online_license_manager().set_license_status_callback(self._on_license_status_changed)
        
        # 设置版本更新回调
        get_online_license_manager().set_version_update_callback(self._on_version_update_detected)
        
        valid, msg = check_online_license()
        self._license_valid = valid
        self._license_info = get_online_license_manager().get_license_info()
        
        if valid and self._license_info:
            # 已授权，启动心跳
            get_online_license_manager().start_heartbeat(interval=300)
            remaining = get_online_license_manager().get_remaining_days()
            license_type = self._license_info.get("licenseType", "未知")
            expire_at = self._license_info.get("expireAt", "")
            self.right_panel.add_system(f"✅ 授权有效 - {license_type}，剩余 {remaining} 天")
            # 更新标题栏授权状态
            self._update_title_bar_license(True, license_type, expire_at, remaining)
        else:
            # 未授权，仅提示（不弹窗，点击付费功能时才要求授权）
            self.right_panel.add_system("💡 基础功能免费使用，付费功能需激活授权")
            self._update_title_bar_license(False)
        
        # 启动反馈回复轮询（无论是否授权都启动）
        self._start_feedback_polling()
        
        # 启动时检查版本更新
        self._check_version_update_on_startup()
    
    def _update_title_bar_license(self, is_valid: bool, license_type: str = "", expire_at: str = "", remaining_days: int = 0):
        """更新标题栏授权状态"""
        if self.title_bar:
            self.title_bar.update_license_status(is_valid, license_type, expire_at, remaining_days)
            self.page.update()
    
    def _on_license_status_changed(self, is_valid: bool, reason: str):
        """授权状态变化回调"""
        if not is_valid:
            self._license_valid = False
            self.right_panel.add_system(f"⚠️ 授权已失效: {reason}")
            # 更新标题栏
            self._update_title_bar_license(False)
            # 显示授权对话框
            self._show_license_dialog()
    
    def _on_version_update_detected(self, version_info):
        """版本更新检测回调（心跳触发）"""
        self.right_panel.add_system(f"🔔 发现新版本: {version_info.version_name}")
        
        # 强制更新时直接弹窗
        if version_info.force_update:
            self._show_update_dialog(version_info)
    
    def _check_version_update_on_startup(self):
        """启动时检查版本更新"""
        import threading
        
        def check():
            import time
            time.sleep(2)  # 延迟检查，让主界面先加载完
            
            if not HAS_LICENSE_MODULE:
                return
            
            version_info = get_online_license_manager().check_version_update()
            if version_info and version_info.has_update:
                # 判断是否需要弹窗
                should_show = version_info.force_update or not self._is_update_skipped_today(version_info.latest_version_code)
                
                if should_show:
                    # 使用 page.run_thread 确保在 UI 线程执行
                    self.right_panel.add_system(f"🔔 发现新版本: {version_info.version_name}")
                    self._show_update_dialog(version_info)
        
        threading.Thread(target=check, daemon=True).start()
    
    def _is_update_skipped_today(self, version: str) -> bool:
        """检查今天是否已跳过此版本更新"""
        try:
            from pathlib import Path
            skip_file = Path.home() / ".douyin_live_assistant" / ".skip_update"
            if skip_file.exists():
                content = skip_file.read_text().strip()
                parts = content.split("|")
                if len(parts) == 2:
                    from datetime import date
                    if parts[0] == date.today().isoformat() and parts[1] == version:
                        return True
        except:
            pass
        return False
    
    def _skip_update_today(self, version: str):
        """记录今天跳过此版本更新"""
        try:
            from pathlib import Path
            from datetime import date
            skip_file = Path.home() / ".douyin_live_assistant" / ".skip_update"
            skip_file.parent.mkdir(parents=True, exist_ok=True)
            skip_file.write_text(f"{date.today().isoformat()}|{version}")
        except:
            pass
    
    def _show_update_dialog(self, version_info):
        """显示版本更新对话框"""
        c = theme.colors
        
        def on_update(e):
            """立即更新"""
            if version_info.download_url:
                import webbrowser
                webbrowser.open(version_info.download_url)
            dialog.open = False
            self.page.update()
        
        def on_later(e):
            """稍后提醒"""
            if not version_info.force_update:
                self._skip_update_today(version_info.latest_version_code)
            dialog.open = False
            self.page.update()
        
        # 更新内容
        update_content = version_info.update_content or "暂无更新说明"
        file_size = ""
        if version_info.file_size:
            size_mb = version_info.file_size / 1024 / 1024
            file_size = f"（{size_mb:.1f} MB）"
        
        # 强制更新提示
        force_tip = ft.Container() if not version_info.force_update else ft.Container(
            content=ft.Row([
                ft.Icon(ft.Icons.WARNING_AMBER, color=ft.Colors.ORANGE_400, size=16),
                ft.Text("此版本为强制更新，请立即更新后继续使用", size=12, color=ft.Colors.ORANGE_400),
            ], spacing=6),
            margin=ft.margin.only(top=10),
        )
        
        dialog = ft.AlertDialog(
            modal=version_info.force_update,  # 强制更新时不可关闭
            title=ft.Row([
                ft.Icon(ft.Icons.SYSTEM_UPDATE, color=c.accent),
                ft.Text(f"发现新版本 {version_info.version_name}", weight=ft.FontWeight.BOLD),
            ], spacing=10),
            content=ft.Container(
                width=450,
                content=ft.Column([
                    ft.Text(f"当前版本: {APP_VERSION} → 最新版本: {version_info.version_name} {file_size}", size=13),
                    ft.Divider(height=15, color=ft.Colors.TRANSPARENT),
                    ft.Text("更新内容:", size=13, weight=ft.FontWeight.BOLD),
                    ft.Container(
                        content=ft.Text(update_content, size=12, color=c.text_secondary),
                        bgcolor=c.bg_elevated,
                        padding=10,
                        border_radius=6,
                        height=150,
                    ),
                    force_tip,
                ], tight=True, spacing=6),
            ),
            actions=[
                ft.TextButton("稍后提醒", on_click=on_later, disabled=version_info.force_update),
                ft.ElevatedButton("立即更新", on_click=on_update, bgcolor=c.accent, color="#ffffff"),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _show_license_dialog(self):
        """显示授权对话框"""
        device_id = generate_device_id() if HAS_LICENSE_MODULE else "未知"
        buy_url = "https://pay.ldxp.cn/shop/wutong/n2tps8"
        
        # 授权码输入框
        license_key_field = ft.TextField(
            label="授权码",
            hint_text="请输入授权码",
            width=350,
            border_color=theme.colors.accent,
            text_size=12,
            content_padding=ft.padding.symmetric(horizontal=10, vertical=5),
        )
        
        # 状态文本
        status_text = ft.Text("", color=ft.Colors.RED_400, size=12)
        
        def on_activate(e):
            """激活授权"""
            key = license_key_field.value.strip()
            if not key:
                status_text.value = "请输入授权码"
                status_text.color = ft.Colors.RED_400
                self.page.update()
                return
            
            # 获取当前登录用户信息（激活码绑定用户，必须先登录）
            user_info = None
            try:
                from .services.auth_service import get_auth_service
                auth_service = get_auth_service()
                if auth_service.is_logged_in():
                    user_info = auth_service.get_user_info()
            except:
                pass
            
            if not user_info:
                status_text.value = "请先登录后再激活授权码"
                status_text.color = ft.Colors.RED_400
                self.page.update()
                return
            
            status_text.value = "正在激活..."
            status_text.color = ft.Colors.BLUE_400
            self.page.update()
            
            success, msg = activate_online_license(key, user_info)
            if success:
                status_text.value = msg
                status_text.color = ft.Colors.GREEN_400
                self._license_valid = True
                self._license_info = get_online_license_manager().get_license_info()
                get_online_license_manager().start_heartbeat(interval=300)
                # 更新标题栏授权状态
                remaining = get_online_license_manager().get_remaining_days()
                license_type = self._license_info.get("licenseType", "")
                expire_at = self._license_info.get("expireAt", "")
                self._update_title_bar_license(True, license_type, expire_at, remaining)
                self.page.update()
                # 延迟关闭对话框
                import time
                time.sleep(1)
                dialog.open = False
                self.right_panel.add_system(f"✅ {msg}")
            else:
                status_text.value = msg
                status_text.color = ft.Colors.RED_400
            self.page.update()
        
        def on_trial(e):
            """申请试用"""
            status_text.value = "正在申请试用..."
            status_text.color = ft.Colors.BLUE_400
            self.page.update()
            
            # 获取当前登录用户信息（用于双重试用限制）
            user_info = None
            try:
                from .services.auth_service import get_auth_service
                auth_service = get_auth_service()
                if auth_service.is_logged_in():
                    user_info = auth_service.get_user_info()
            except:
                pass
            
            success, msg = apply_online_trial(user_info)
            if success:
                status_text.value = msg
                status_text.color = ft.Colors.GREEN_400
                self._license_valid = True
                self._license_info = get_online_license_manager().get_license_info()
                get_online_license_manager().start_heartbeat(interval=300)
                # 更新标题栏授权状态
                remaining = get_online_license_manager().get_remaining_days()
                license_type = self._license_info.get("licenseType", "")
                expire_at = self._license_info.get("expireAt", "")
                self._update_title_bar_license(True, license_type, expire_at, remaining)
                self.page.update()
                import time
                time.sleep(1)
                dialog.open = False
                self.right_panel.add_system(f"✅ {msg}")
            else:
                status_text.value = msg
                status_text.color = ft.Colors.RED_400
            self.page.update()

        def on_buy(e):
            """购买卡密"""
            try:
                if hasattr(self.page, "launch_url"):
                    self.page.launch_url(buy_url)
                else:
                    webbrowser.open(buy_url)
            except Exception:
                try:
                    webbrowser.open(buy_url)
                except Exception:
                    pass
        
        dialog = ft.AlertDialog(
            modal=True,
            title=ft.Text("软件授权", weight=ft.FontWeight.BOLD),
            content=ft.Container(
                width=400,
                content=ft.Column([
                    ft.Text("请激活软件或申请试用", size=14),
                    ft.Divider(height=10, color=ft.Colors.TRANSPARENT),
                    ft.Row([
                        ft.Text("设备码:", size=12, color=ft.Colors.GREY_400),
                        ft.Text(device_id, size=12, selectable=True),
                        ft.IconButton(
                            icon=ft.Icons.COPY,
                            icon_size=16,
                            tooltip="复制设备码",
                            on_click=lambda e: self.page.set_clipboard(device_id)
                        )
                    ]),
                    ft.Divider(height=10, color=ft.Colors.TRANSPARENT),
                    license_key_field,
                    status_text,
                ], tight=True, spacing=8),
            ),
            actions=[
                ft.TextButton(
                    content=ft.Text("关闭", size=14),
                    on_click=lambda e: self._close_dialog(dialog),
                    style=ft.ButtonStyle(color=ft.Colors.GREY_400)
                ),
                ft.TextButton(
                    content=ft.Text("购买卡密", size=14),
                    on_click=on_buy,
                    style=ft.ButtonStyle(color=ft.Colors.BLUE_400)
                ),
                ft.TextButton(
                    content=ft.Text("申请试用", size=14),
                    on_click=on_trial,
                    style=ft.ButtonStyle(color=ft.Colors.BLUE_400)
                ),
                ft.ElevatedButton(
                    content=ft.Text("激活", size=14, weight=ft.FontWeight.NORMAL),
                    on_click=on_activate,
                    bgcolor=ft.Colors.BLUE_600, color=ft.Colors.WHITE
                ),
            ],
            actions_alignment=ft.MainAxisAlignment.END,
        )
        
        self.page.overlay.append(dialog)
        dialog.open = True
        self.page.update()
    
    def _close_dialog(self, dialog):
        """关闭对话框"""
        dialog.open = False
        self.page.update()
    
    def _show_profile_dialog(self):
        """显示个人中心弹窗"""
        print("[Profile] 尝试打开个人中心...")
        
        try:
            from .components import ProfileDialog
            
            # 关闭已有的个人中心弹窗
            for overlay in self.page.overlay[:]:
                if isinstance(overlay, ProfileDialog):
                    print("[Profile] 关闭已有弹窗")
                    overlay.open = False
                    self.page.overlay.remove(overlay)
            
            # 直接创建弹窗（不使用 FilePicker，简化逻辑）
            dialog = ProfileDialog(
                page=self.page,
                on_save=lambda info: self.right_panel.add_system(f"个人信息已更新"),
                on_close=lambda: self._remove_dialog(dialog),
            )
            
            self.page.overlay.append(dialog)
            dialog.open = True
            self.page.update()
            print("[Profile] 弹窗已打开")
        except Exception as e:
            print(f"[错误] 打开个人中心失败: {e}")
            import traceback
            traceback.print_exc()
            self.right_panel.add_system(f"打开个人中心失败: {e}")
    
    def _remove_dialog(self, dialog):
        """从overlay中移除弹窗"""
        if dialog in self.page.overlay:
            self.page.overlay.remove(dialog)
    
    def _on_help_feedback(self, action: str, data: dict = None):
        """处理说明页的回调"""
        if action == "trial":
            # 打开授权弹窗
            self._show_license_dialog()
        elif action == "feedback":
            # 提交留言
            self._submit_feedback(data)
    
    def _submit_feedback(self, data: dict):
        """提交留言到后端"""
        import threading
        
        def do_submit():
            import time
            try:
                import requests
                from ..license_manager import LICENSE_SERVER_URL, SOFTWARE_CODE
                
                device_id = ""
                if HAS_LICENSE_MODULE:
                    device_info = get_online_license_manager().get_device_info()
                    device_id = device_info.get("machineCode", "") if device_info else ""
                
                payload = {
                    "softwareCode": SOFTWARE_CODE,
                    "machineCode": device_id,
                    "contact": data.get("contact", ""),
                    "feedbackType": data.get("type", "FEEDBACK"),
                    "content": data.get("content", ""),
                }
                
                url = f"{LICENSE_SERVER_URL}/biz/feedback/client/submit"
                
                # 记录请求日志
                log_request("POST", url, {}, payload)
                start_time = time.time()
                
                response = requests.post(url, json=payload, timeout=10)
                duration_ms = (time.time() - start_time) * 1000
                
                log_response(url, status_code=response.status_code, success=response.status_code==200, 
                            data=response.text[:200] if response.status_code==200 else None,
                            error=response.text[:200] if response.status_code!=200 else None, 
                            duration_ms=duration_ms)
            except Exception as e:
                import traceback
                print(f"[Feedback] 提交失败: {e}")
                log_response(url if 'url' in dir() else "unknown", success=False, error=str(e))
                traceback.print_exc()
        
        threading.Thread(target=do_submit, daemon=True).start()
    
    def _start_feedback_polling(self):
        """启动反馈回复轮询"""
        import threading
        
        print("[Feedback] 启动反馈回复轮询...")
        
        def poll_replies():
            import time
            # 首次等待5秒后检查
            time.sleep(5)
            self._check_feedback_replies()
            while True:
                try:
                    time.sleep(60)  # 之后每60秒轮询一次
                    self._check_feedback_replies()
                except Exception as e:
                    print(f"[Feedback] 轮询异常: {e}")
        
        threading.Thread(target=poll_replies, daemon=True).start()
    
    def _check_feedback_replies(self):
        """检查是否有新的反馈回复"""
        import time
        url = None
        try:
            import requests
            from ..license_manager import LICENSE_SERVER_URL
            
            device_id = ""
            if HAS_LICENSE_MODULE:
                device_info = get_online_license_manager().get_device_info()
                device_id = device_info.get("machineCode", "") if device_info else ""
            
            if not device_id:
                return
            
            url = f"{LICENSE_SERVER_URL}/biz/feedback/client/replies"
            params = {"machineCode": device_id}
            
            # 记录请求日志
            log_request("GET", url, {}, params)
            start_time = time.time()
            
            response = requests.get(url, params=params, timeout=10)
            duration_ms = (time.time() - start_time) * 1000
            
            if response.status_code == 200:
                data = response.json()
                log_response(url, status_code=200, success=True, data=data, duration_ms=duration_ms)
                if data.get("code") == 200 and data.get("data"):
                    replies = data.get("data", [])
                    if replies:
                        self._show_feedback_reply_dialog(replies[0])
            else:
                log_response(url, status_code=response.status_code, success=False, error=response.text[:200], duration_ms=duration_ms)
        except Exception as e:
            log_response(url or "unknown", success=False, error=str(e))
    
    def _show_feedback_reply_dialog(self, feedback: dict):
        """显示反馈回复弹窗"""
        if not self.page:
            print("[Feedback] page为空，无法显示弹窗")
            return
        
        def show_dialog():
            try:
                content_text = feedback.get("content", "")
                if len(content_text) > 100:
                    content_text = content_text[:100] + "..."
                
                def close_dialog(e):
                    self.page.close(dialog)
                    # 标记已读
                    self._mark_feedback_read(feedback.get("id"))
                
                dialog = ft.AlertDialog(
                    modal=True,
                    title=ft.Text("📬 您有新的反馈回复"),
                    content=ft.Column([
                        ft.Text("您的反馈：", weight=ft.FontWeight.BOLD, size=13),
                        ft.Container(
                            content=ft.Text(content_text, size=12, color=ft.Colors.BLACK),
                            bgcolor=ft.Colors.GREY_300,
                            padding=10,
                            border_radius=5,
                        ),
                        ft.Text("官方回复：", weight=ft.FontWeight.BOLD, size=13),
                        ft.Container(
                            content=ft.Text(feedback.get("replyContent", ""), size=12, color=ft.Colors.BLACK),
                            bgcolor=ft.Colors.LIGHT_BLUE_100,
                            padding=10,
                            border_radius=5,
                        ),
                        ft.Text(f"回复时间：{feedback.get('replyTime', '')}", size=11, color=ft.Colors.GREY_500),
                    ], spacing=10, tight=True, width=350),
                    actions=[
                        ft.ElevatedButton("知道了", on_click=close_dialog),
                    ],
                    actions_alignment=ft.MainAxisAlignment.END,
                )
                
                self.page.open(dialog)
                print("[Feedback] 弹窗已显示")
            except Exception as e:
                import traceback
                print(f"[Feedback] 显示弹窗失败: {e}")
                traceback.print_exc()
        
        # 直接调用（Flet会自动处理线程安全）
        show_dialog()
    
    def _mark_feedback_read(self, feedback_id: str):
        """标记反馈已读"""
        if not feedback_id:
            return
        import threading
        
        def do_mark():
            import time
            url = None
            try:
                import requests
                from ..license_manager import LICENSE_SERVER_URL
                url = f"{LICENSE_SERVER_URL}/biz/feedback/client/markRead"
                params = {"id": feedback_id}
                
                log_request("POST", url, {}, params)
                start_time = time.time()
                
                response = requests.post(url, params=params, timeout=10)
                duration_ms = (time.time() - start_time) * 1000
                
                log_response(url, status_code=response.status_code, success=response.status_code==200, duration_ms=duration_ms)
            except Exception as e:
                log_response(url or "unknown", success=False, error=str(e))
        
        threading.Thread(target=do_mark, daemon=True).start()
    
    def _on_logout(self):
        """退出登录"""
        from .services.auth_service import get_auth_service
        from .login_page import LoginPage
        from .themes import theme
        
        # 调用退出登录
        auth_service = get_auth_service()
        auth_service.logout()
        
        # 清理当前页面
        self.page.controls.clear()
        
        # 设置背景色
        c = theme.colors
        self.page.bgcolor = c.bg_surface
        
        # 切换到小窗口模式 - 必须先设置 min_width/min_height，再设置 width/height
        if hasattr(self.page, 'window') and hasattr(self.page.window, 'width'):
            self.page.window.min_width = 400
            self.page.window.min_height = 480
            self.page.window.width = 450
            self.page.window.height = 520
            self.page.update()  # 先应用窗口尺寸
            self.page.window.center()
        else:
            self.page.window_min_width = 400
            self.page.window_min_height = 480
            self.page.window_width = 450
            self.page.window_height = 520
            self.page.update()  # 先应用窗口尺寸
            self.page.window_center()
        
        def on_login_success():
            """登录成功后重新显示主应用"""
            self.page.controls.clear()
            # 切换到大窗口 - 必须先设置 min_width/min_height，再设置 width/height
            if hasattr(self.page, 'window') and hasattr(self.page.window, 'width'):
                self.page.window.min_width = 1000
                self.page.window.min_height = 750
                self.page.window.width = 1200
                self.page.window.height = 850
                self.page.update()  # 先应用窗口尺寸
                self.page.window.center()
            else:
                self.page.window_min_width = 1000
                self.page.window_min_height = 750
                self.page.window_width = 1200
                self.page.window_height = 850
                self.page.update()  # 先应用窗口尺寸
                self.page.window_center()
            
            MainApp(self.page)
            self.page.update()
        
        login_page = LoginPage(self.page, on_login_success=on_login_success)
        self.page.add(login_page)
        self.page.update()
    
    def _on_close(self):
        """关闭应用"""
        self._save_config()
        self.worker_manager.stop()
        self.stream_manager.stop()
        # 停止授权心跳
        if HAS_LICENSE_MODULE:
            get_online_license_manager().stop_heartbeat()
        if self._overlay_process:
            try:
                self._overlay_process.kill()
            except:
                pass
        # 关闭悬浮窗口
        if hasattr(self, '_float_window') and self._float_window:
            try:
                self._float_window.close()
            except:
                pass
        # 关闭翻页时钟进程
        if self._timer_process:
            try:
                self._timer_process.terminate()
            except:
                pass
        # 兼容flet 0.21.x和0.28.x
        if hasattr(self.page, 'window') and hasattr(self.page.window, 'close'):
            self.page.window.close()
        elif hasattr(self.page, 'window_close'):
            self.page.window_close()


def main(page: ft.Page):
    MainApp(page)


if __name__ == "__main__":
    ft.app(target=main)
