// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Diagnostics.CodeAnalysis;
using MQTTnet.Diagnostics.Logger;
using MQTTnet.Server.EnhancedAuthentication;
using MQTTnet.Server.Internal.Adapter;

namespace MQTTnet.Server;

[SuppressMessage("Performance", "CA1822:Mark members as static")]
public sealed class MqttServerFactory
{
    public MqttServerFactory() : this(new MqttNetNullLogger())
    {
    }

    public MqttServerFactory(IMqttNetLogger logger)
    {
        DefaultLogger = logger ?? throw new ArgumentNullException(nameof(logger));
    }

    public IMqttNetLogger DefaultLogger { get; }

    public IList<Func<MqttServerFactory, IMqttServerAdapter>> DefaultServerAdapters { get; } = [_ => new MqttTcpServerAdapter()];

    public IDictionary<object, object> Properties { get; } = new Dictionary<object, object>();

    public MqttApplicationMessageBuilder CreateApplicationMessageBuilder()
    {
        return new MqttApplicationMessageBuilder();
    }

    public ExchangeEnhancedAuthenticationOptionsFactory CreateExchangeExtendedAuthenticationOptionsBuilder()
    {
        return new ExchangeEnhancedAuthenticationOptionsFactory();
    }

    public MqttServer CreateMqttServer(MqttServerOptions options)
    {
        return CreateMqttServer(options, DefaultLogger);
    }

    public MqttServer CreateMqttServer(MqttServerOptions options, IMqttNetLogger logger)
    {
        ArgumentNullException.ThrowIfNull(logger);

        var serverAdapters = DefaultServerAdapters.Select(a => a.Invoke(this));
        return CreateMqttServer(options, serverAdapters, logger);
    }

    public MqttServer CreateMqttServer(MqttServerOptions options, IEnumerable<IMqttServerAdapter> serverAdapters, IMqttNetLogger logger)
    {
        ArgumentNullException.ThrowIfNull(serverAdapters);
        ArgumentNullException.ThrowIfNull(logger);

        return new MqttServer(options, serverAdapters, logger);
    }

    public MqttServer CreateMqttServer(MqttServerOptions options, IEnumerable<IMqttServerAdapter> serverAdapters)
    {
        ArgumentNullException.ThrowIfNull(serverAdapters);

        return new MqttServer(options, serverAdapters, DefaultLogger);
    }

    public MqttServerClientDisconnectOptionsBuilder CreateMqttServerClientDisconnectOptionsBuilder()
    {
        return new MqttServerClientDisconnectOptionsBuilder();
    }


    public MqttServerStopOptionsBuilder CreateMqttServerStopOptionsBuilder()
    {
        return new MqttServerStopOptionsBuilder();
    }

    public MqttServerOptionsBuilder CreateServerOptionsBuilder()
    {
        return new MqttServerOptionsBuilder();
    }
}