# Server-side hooks

These hooks are called on server-side.

## loadSettings
Called from: `src/node/server.js`

Things in context:

1. settings - the settings object

Use this hook to receive the global settings in your plugin.

## shutdown
Called from: `src/node/server.js`

Things in context: None

This hook runs before shutdown. Use it to stop timers, close sockets and files,
flush buffers, etc. The database is not available while this hook is running.
The shutdown function must not block for long because there is a short timeout
before the process is forcibly terminated.

The shutdown function must return a Promise, which must resolve to `undefined`.
Returning `callback(value)` will return a Promise that is resolved to `value`.

Example:

```
// using an async function
exports.shutdown = async (hookName, context) => {
  await flushBuffers();
};
```

## pluginUninstall
Called from: `src/static/js/pluginfw/installer.js`

Things in context:

1. plugin_name - self-explanatory

If this hook returns an error, the callback to the uninstall function gets an error as well. This mostly seems useful for handling additional features added in based on the installation of other plugins, which is pretty cool!

## pluginInstall
Called from: `src/static/js/pluginfw/installer.js`

Things in context:

1. plugin_name - self-explanatory

If this hook returns an error, the callback to the install function gets an error, too. This seems useful for adding in features when a particular plugin is installed.

## `init_<plugin name>`

Called from: `src/static/js/pluginfw/plugins.js`

Run during startup after the named plugin is initialized.

Context properties:

* `logger`: An object with the following `console`-like methods: `debug`,
  `info`, `log`, `warn`, `error`.

## `expressPreSession`

Called from: `src/node/hooks/express.js`

Called during server startup just before the
[`express-session`](https://www.npmjs.com/package/express-session) middleware is
added to the Express Application object. Use this hook to add route handlers or
middleware that executes before `express-session` state is created and
authentication is performed. This is useful for creating public endpoints that
don't spam the database with new `express-session` records or trigger
authentication.

**WARNING:** All handlers registered during this hook run before the built-in
authentication checks, so any handled endpoints will be public unless the
handler itself authenticates the user.

Context properties:

* `app`: The Express [Application](https://expressjs.com/en/4x/api.html#app)
  object.

Example:

```javascript
exports.expressPreSession = async (hookName, {app}) => {
  app.get('/hello-world', (req, res) => res.send('hello world'));
};
```

## `expressConfigure`

Called from: `src/node/hooks/express.js`

Called during server startup just after the
[`express-session`](https://www.npmjs.com/package/express-session) middleware is
added to the Express Application object. Use this hook to add route handlers or
middleware that executes after `express-session` state is created and
authentication is performed.

Context properties:

* `app`: The Express [Application](https://expressjs.com/en/4x/api.html#app)
  object.

## `expressCreateServer`

Called from: `src/node/hooks/express.js`

Identical to the `expressConfigure` hook (the two run in parallel with each
other) except this hook's context includes the HTTP Server object.

Context properties:

* `app`: The Express [Application](https://expressjs.com/en/4x/api.html#app)
  object.
* `server`: The [http.Server](https://nodejs.org/api/http.html#class-httpserver)
  or [https.Server](https://nodejs.org/api/https.html#class-httpsserver) object.

## expressCloseServer

Called from: `src/node/hooks/express.js`

Things in context: Nothing

This hook is called when the HTTP server is closing, which happens during
shutdown (see the shutdown hook) and when the server restarts (e.g., when a
plugin is installed via the `/admin/plugins` page). The HTTP server may or may
not already be closed when this hook executes.

Example:

```
exports.expressCloseServer = async () => {
  await doSomeCleanup();
};
```

## eejsBlock_`<name>`
Called from: `src/node/eejs/index.js`

Things in context:

1. content - the content of the block

This hook gets called upon the rendering of an ejs template block. For any specific kind of block, you can change how that block gets rendered by modifying the content object passed in.

Available blocks in `pad.html` are:

* `htmlHead` - after `<html>` and immediately before the title tag
* `styles` - the style `<link>`s
* `body` - the contents of the body tag
* `editbarMenuLeft` - the left tool bar (consider using the toolbar controller instead of manually adding html here)
* `editbarMenuRight` - right tool bar
* `afterEditbar` - allows you to add stuff immediately after the toolbar
* `userlist` - the contents of the userlist dropdown
* `loading` - the initial loading message
* `mySettings` - the left column of the settings dropdown ("My view"); intended for adding checkboxes only
* `mySettings.dropdowns` - add your dropdown settings here
* `globalSettings` - the right column of the settings dropdown ("Global view")
* `importColumn` - import form
* `exportColumn` - export form
* `modals` - Contains all connectivity messages
* `embedPopup` - the embed dropdown
* `scripts` - Add your script tags here, if you really have to (consider use client-side hooks instead)

`timeslider.html` blocks:

* `timesliderStyles`
* `timesliderScripts`
* `timesliderBody`
* `timesliderTop`
* `timesliderEditbarRight`
* `modals`

`index.html` blocks:

* `indexCustomStyles` - contains the `index.css` `<link>` tag, allows you to add your own or to customize the one provided by the active skin
* `indexWrapper` - contains the form for creating new pads
* `indexCustomScripts` - contains the `index.js` `<script>` tag, allows you to add your own or to customize the one provided by the active skin

## padInitToolbar
Called from: `src/node/hooks/express/specialpages.js`

Things in context:

1. toolbar - the toolbar controller that will render the toolbar eventually

Here you can add custom toolbar items that will be available in the toolbar config in `settings.json`. For more about the toolbar controller see the API section.

Usage examples:

* https://github.com/tiblu/ep_authorship_toggle

## onAccessCheck
Called from: `src/node/db/SecurityManager.js`

Things in context:

1. padID - the real ID (never the read-only ID) of the pad the user wants to
   access
2. token - the token of the author
3. sessionCookie - the session the use has

This hook gets called when the access to the concrete pad is being checked.
Return `false` to deny access.

## `getAuthorId`

Called from `src/node/db/AuthorManager.js`

Called when looking up (or creating) the author ID for a user, except for author
IDs obtained via the HTTP API. Registered hook functions are called until one
returns a non-`undefined` value. If a truthy value is returned by a hook
function, it is used as the user's author ID. Otherwise, the value of the
`dbKey` context property is used to look up the author ID. If there is no such
author ID at that key, a new author ID is generated and associated with that
key.

Context properties:

* `dbKey`: Database key to use when looking up the user's author ID if no hook
  function returns an author ID. This is initialized to the user-supplied token
  value (see the `token` context property), but hook functions can modify this
  to control how author IDs are allocated to users. If no author ID is
  associated with this database key, a new author ID will be randomly generated
  and associated with the key. For security reasons, if this is modified it
  should be modified to not look like a valid token (see the `token` context
  property) unless the plugin intentionally wants the user to be able to
  impersonate another user.
* `token`: The user-supplied token, or nullish for an anonymous user. Tokens are
  secret values that must not be disclosed to others. If non-null, the token is
  guaranteed to be a string with the form `t.<base64url>` where `<base64url>` is
  any valid non-empty base64url string (RFC 4648 section 5 with padding).
  Example: `t.twim3X2_KGiRj8cJ-3602g==`.
* `user`: If the user has authenticated, this is an object from `settings.users`
  (or similar from an authentication plugin). Etherpad core and all good
  authentication plugins set the `username` property of this object to a string
  that uniquely identifies the authenticated user. This object is nullish if the
  user has not authenticated.

Example:

```javascript
exports.getAuthorId = async (hookName, context) => {
  const {username} = context.user || {};
  // If the user has not authenticated, or has "authenticated" as the guest
  // user, do the default behavior (try another plugin if any, falling through
  // to using the token as the database key).
  if (!username || username === 'guest') return;
  // The user is authenticated and has a username. Give the user a stable author
  // ID so that they appear to be the same author even after clearing cookies or
  // accessing the pad from another device. Note that this string is guaranteed
  // to never have the form of a valid token; without that guarantee an
  // unauthenticated user might be able to impersonate an authenticated user.
  context.dbKey = `username=${username}`;
  // Return a falsy but non-undefined value to stop Etherpad from calling any
  // more getAuthorId hook functions and look up the author ID using the
  // username-derived database key.
  return '';
};
```

## `padCreate`

Called from: `src/node/db/Pad.js`

Called when a new pad is created.

Context properties:

* `pad`: The Pad object.
* `authorId`: The ID of the author who created the pad.
* `author` (**deprecated**): Synonym of `authorId`.

## `padDefaultContent`

Called from `src/node/db/Pad.js`

Called to obtain a pad's initial content, unless the pad is being created with
specific content. The return value is ignored; to change the content, modify the
`content` context property.

This hook is run asynchronously. All registered hook functions are run
concurrently (via `Promise.all()`), so be careful to avoid race conditions when
reading and modifying the context properties.

Context properties:

* `pad`: The newly created Pad object.
* `authorId`: The author ID of the user that is creating the pad.
* `type`: String identifying the content type. Currently this is `'text'` and
  must not be changed. Future versions of Etherpad may add support for HTML,
  jsdom objects, or other formats, so plugins must assert that this matches a
  supported content type before reading `content`.
* `content`: The pad's initial content. Change this property to change the pad's
  initial content. If the content type is changed, the `type` property must also
  be updated to match. Plugins must check the value of the `type` property
  before reading this value.

## `padLoad`

Called from: `src/node/db/PadManager.js`

Called when a pad is loaded, including after new pad creation.

Context properties:

* `pad`: The Pad object.

## `padUpdate`

Called from: `src/node/db/Pad.js`

Called when an existing pad is updated.

Context properties:

* `pad`: The Pad object.
* `authorId`: The ID of the author who updated the pad.
* `author` (**deprecated**): Synonym of `authorId`.
* `revs`: The index of the new revision.
* `changeset`: The changeset of this revision (see [Changeset
  Library](#index_changeset_library)).

## `padCopy`

Called from: `src/node/db/Pad.js`

Called when a pad is copied so that plugins can copy plugin-specific database
records or perform some other plugin-specific initialization.

Order of events when a pad is copied:

1. Destination pad is deleted if it exists and overwrite is permitted. This
   causes the `padRemove` hook to run.
2. Pad-specific database records are copied in the database, except for
   records with plugin-specific database keys.
3. A new Pad object is created for the destination pad. This causes the
   `padLoad` hook to run.
4. This hook runs.

Context properties:

* `srcPad`: The source Pad object.
* `dstPad`: The destination Pad object.

Usage examples:

* https://github.com/ether/ep_comments_page

## `padRemove`

Called from: `src/node/db/Pad.js`

Called when an existing pad is removed/deleted. Plugins should use this to clean
up any plugin-specific pad records from the database.

Context properties:

* `pad`: Pad object for the pad that is being deleted.

Usage examples:

* https://github.com/ether/ep_comments_page

## `padCheck`

Called from: `src/node/db/Pad.js`

Called when a consistency check is run on a pad, after the core checks have
completed successfully. An exception should be thrown if the pad is faulty in
some way.

Context properties:

* `pad`: The Pad object that is being checked.

## socketio
Called from: `src/node/hooks/express/socketio.js`

Things in context:

1. app - the application object
2. io - the socketio object
3. server - the http server object

I have no idea what this is useful for, someone else will have to add this description.

## `preAuthorize`

Called from: `src/node/hooks/express/webaccess.js`

Called for each HTTP request before any authentication checks are performed. The
registered `preAuthorize` hook functions are called one at a time until one
explicitly grants or denies the request by returning `true` or `false`,
respectively. If none of the hook functions return anything, the access decision
is deferred to the normal authentication and authorization checks.

Example uses:

* Always grant access to static content.
* Process an OAuth callback.
* Drop requests from IP addresses that have failed N authentication checks
  within the past X minutes.

Return values:

* `undefined` (or `[]`) defers the access decision to the next registered
  `preAuthorize` hook function, or to the normal authentication and
  authorization checks if no more `preAuthorize` hook functions remain.
* `true` (or `[true]`) immediately grants access to the requested resource,
  unless the request is for an `/admin` page in which case it is treated the
  same as returning `undefined`. (This prevents buggy plugins from accidentally
  granting admin access to the general public.)
* `false` (or `[false]`) immediately denies the request. The `preAuthnFailure`
  hook will be called to handle the failure.

Context properties:

* `req`: The Express [Request](https://expressjs.com/en/4x/api.html#req) object.
* `res`: The Express [Response](https://expressjs.com/en/4x/api.html#res)
  object.
* `next`: Callback to immediately hand off handling to the next Express
  middleware/handler, or to the next matching route if `'route'` is passed as
  the first argument. Do not call this unless you understand the consequences.

Example:

```javascript
exports.preAuthorize = async (hookName, {req}) => {
  if (await ipAddressIsFirewalled(req)) return false;
  if (requestIsForStaticContent(req)) return true;
  if (requestIsForOAuthCallback(req)) return true;
  // Defer the decision to the next step by returning undefined.
};
```

## authorize
Called from: `src/node/hooks/express/webaccess.js`

Things in context:

1. req - the request object
2. res - the response object
3. next - ?
4. resource - the path being accessed

This hook is called to handle authorization. It is especially useful for
controlling access to specific paths.

A plugin's authorize function is only called if all of the following are true:

* The request is not for static content or an API endpoint. (Requests for static
  content and API endpoints are always authorized, even if unauthenticated.)
* The `requireAuthentication` and `requireAuthorization` settings are both true.
* The user has already successfully authenticated.
* The user is not an admin (admin users are always authorized).
* The path being accessed is not an `/admin` path (`/admin` paths can only be
  accessed by admin users, and admin users are always authorized).
* An authorize function from a different plugin has not already caused
  authorization to pass or fail.

Note that the authorize hook cannot grant access to `/admin` pages. If admin
access is desired, the `is_admin` user setting must be set to true. This can be
set in the settings file or by the authenticate hook.

You can pass the following values to the provided callback:

* `[true]` or `['create']` will grant access to modify or create the pad if the
  request is for a pad, otherwise access is simply granted. Access to a pad will
  be downgraded to modify-only if `settings.editOnly` is true or the user's
  `canCreate` setting is set to `false`, and downgraded to read-only if the
  user's `readOnly` setting is `true`.
* `['modify']` will grant access to modify but not create the pad if the request
  is for a pad, otherwise access is simply granted. Access to a pad will be
  downgraded to read-only if the user's `readOnly` setting is `true`.
* `['readOnly']` will grant read-only access.
* `[false]` will deny access.
* `[]` or `undefined` will defer the authorization decision to the next
  authorization plugin (if any, otherwise deny).

Example:

```js
exports.authorize = (hookName, context, cb) => {
  const user = context.req.session.user;
  const path = context.req.path;  // or context.resource
  if (isExplicitlyProhibited(user, path)) return cb([false]);
  if (isExplicitlyAllowed(user, path)) return cb([true]);
  return cb([]);  // Let the next authorization plugin decide
};
```

## authenticate
Called from: `src/node/hooks/express/webaccess.js`

Things in context:

1. req - the request object
2. res - the response object
3. users - the users object from settings.json (possibly modified by plugins)
4. next - ?
5. username - the username used (optional)
6. password - the password used (optional)

This hook is called to handle authentication.

Plugins that supply an authenticate function should probably also supply an
authnFailure function unless falling back to HTTP basic authentication is
appropriate upon authentication failure.

This hook is only called if either the `requireAuthentication` setting is true
or the request is for an `/admin` page.

Calling the provided callback with `[true]` or `[false]` will cause
authentication to succeed or fail, respectively. Calling the callback with `[]`
or `undefined` will defer the authentication decision to the next authentication
plugin (if any, otherwise fall back to HTTP basic authentication).

If you wish to provide a mix of restricted and anonymous access (e.g., some pads
are private, others are public), you can "authenticate" (as a guest account)
users that have not yet logged in, and rely on other hooks (e.g., authorize,
onAccessCheck, handleMessageSecurity) to authorize specific privileged actions.

If authentication is successful, the authenticate function MUST set
`context.req.session.user` to the user's settings object. The `username`
property of this object should be set to the user's username. The settings
object should come from global settings (`context.users[username]`).

Example:

```js
exports.authenticate = (hook_name, context, cb) => {
  if (notApplicableToThisPlugin(context)) {
    return cb([]);  // Let the next authentication plugin decide
  }
  const username = authenticate(context);
  if (!username) {
    console.warn(`ep_myplugin.authenticate: Failed authentication from IP ${context.req.ip}`);
    return cb([false]);
  }
  console.info(`ep_myplugin.authenticate: Successful authentication from IP ${context.req.ip} for user ${username}`);
  const users = context.users;
  if (!(username in users)) users[username] = {};
  users[username].username = username;
  context.req.session.user = users[username];
  return cb([true]);
};
```

## authFailure
Called from: `src/node/hooks/express/webaccess.js`

Things in context:

1. req - the request object
2. res - the response object
3. next - ?

**DEPRECATED:** Use authnFailure or authzFailure instead.

This hook is called to handle an authentication or authorization failure.

Plugins that supply an authenticate function should probably also supply an
authnFailure function unless falling back to HTTP basic authentication is
appropriate upon authentication failure.

A plugin's authFailure function is only called if all of the following are true:

* There was an authentication or authorization failure.
* The failure was not already handled by an authFailure function from another
  plugin.
* For authentication failures: The failure was not already handled by the
  authnFailure hook.
* For authorization failures: The failure was not already handled by the
  authzFailure hook.

Calling the provided callback with `[true]` tells Etherpad that the failure was
handled and no further error handling is required. Calling the callback with
`[]` or `undefined` defers error handling to the next authFailure plugin (if
any, otherwise fall back to HTTP basic authentication for an authentication
failure or a generic 403 page for an authorization failure).

Example:

```js
exports.authFailure = (hookName, context, cb) => {
  if (notApplicableToThisPlugin(context)) {
    return cb([]);  // Let the next plugin handle the error
  }
  context.res.redirect(makeLoginURL(context.req));
  return cb([true]);
};
```

## preAuthzFailure
Called from: `src/node/hooks/express/webaccess.js`

Things in context:

1. req - the request object
2. res - the response object

This hook is called to handle a pre-authentication authorization failure.

A plugin's preAuthzFailure function is only called if the pre-authentication
authorization failure was not already handled by a preAuthzFailure function from
another plugin.

Calling the provided callback with `[true]` tells Etherpad that the failure was
handled and no further error handling is required. Calling the callback with
`[]` or `undefined` defers error handling to a preAuthzFailure function from
another plugin (if any, otherwise fall back to a generic 403 error page).

Example:

```js
exports.preAuthzFailure = (hookName, context, cb) => {
  if (notApplicableToThisPlugin(context)) return cb([]);
  context.res.status(403).send(renderFancy403Page(context.req));
  return cb([true]);
};
```

## authnFailure
Called from: `src/node/hooks/express/webaccess.js`

Things in context:

1. req - the request object
2. res - the response object

This hook is called to handle an authentication failure.

Plugins that supply an authenticate function should probably also supply an
authnFailure function unless falling back to HTTP basic authentication is
appropriate upon authentication failure.

A plugin's authnFailure function is only called if the authentication failure
was not already handled by an authnFailure function from another plugin.

Calling the provided callback with `[true]` tells Etherpad that the failure was
handled and no further error handling is required. Calling the callback with
`[]` or `undefined` defers error handling to an authnFailure function from
another plugin (if any, otherwise fall back to the deprecated authFailure hook).

Example:

```js
exports.authnFailure = (hookName, context, cb) => {
  if (notApplicableToThisPlugin(context)) return cb([]);
  context.res.redirect(makeLoginURL(context.req));
  return cb([true]);
};
```

## authzFailure
Called from: `src/node/hooks/express/webaccess.js`

Things in context:

1. req - the request object
2. res - the response object

This hook is called to handle a post-authentication authorization failure.

A plugin's authzFailure function is only called if the authorization failure was
not already handled by an authzFailure function from another plugin.

Calling the provided callback with `[true]` tells Etherpad that the failure was
handled and no further error handling is required. Calling the callback with
`[]` or `undefined` defers error handling to an authzFailure function from
another plugin (if any, otherwise fall back to the deprecated authFailure hook).

Example:

```js
exports.authzFailure = (hookName, context, cb) => {
  if (notApplicableToThisPlugin(context)) return cb([]);
  if (needsPremiumAccount(context.req) && !context.req.session.user.premium) {
    context.res.status(200).send(makeUpgradeToPremiumAccountPage(context.req));
    return cb([true]);
  }
  // Use the generic 403 forbidden response.
  return cb([]);
};
```

## `handleMessage`

Called from: `src/node/handler/PadMessageHandler.js`

This hook allows plugins to drop or modify incoming socket.io messages from
clients, before Etherpad processes them. If any hook function returns `null`
then the message will not be subject to further processing.

Context properties:

* `message`: The message being handled.
* `sessionInfo`: Object describing the socket.io session with the following
  properties:
    * `authorId`: The user's author ID.
    * `padId`: The real (not read-only) ID of the pad.
    * `readOnly`: Whether the client has read-only access (true) or read/write
      access (false).
* `socket`: The socket.io Socket object.
* `client`: (**Deprecated**; use `socket` instead.) Synonym of `socket`.

Example:

```javascript
exports.handleMessage = async (hookName, {message, socket}) => {
  if (message.type === 'USERINFO_UPDATE') {
    // Force the display name to the name associated with the account.
    const user = socket.client.request.session.user || {};
    if (user.name) message.data.userInfo.name = user.name;
  }
};
```

## `handleMessageSecurity`

Called from: `src/node/handler/PadMessageHandler.js`

Called for each incoming message from a client. Allows plugins to grant
temporary write access to a pad.

Supported return values:

* `undefined`: No change in access status.
* `'permitOnce'`: Override the user's read-only access for the current
  `COLLABROOM` message only. Has no effect if the current message is not a
  `COLLABROOM` message, or if the user already has write access to the pad.
* `true`: (**Deprecated**; return `'permitOnce'` instead.) Override the user's
  read-only access for all `COLLABROOM` messages from the same socket.io
  connection (including the current message, if applicable) until the client's
  next `CLIENT_READY` message. Has no effect if the user already has write
  access to the pad. Read-only access is reset **after** each `CLIENT_READY`
  message, so returning `true` has no effect for `CLIENT_READY` messages.

Context properties:

* `message`: The message being handled.
* `sessionInfo`: Object describing the socket.io connection with the following
  properties:
    * `authorId`: The user's author ID.
    * `padId`: The real (not read-only) ID of the pad.
    * `readOnly`: Whether the client has read-only access (true) or read/write
      access (false).
* `socket`: The socket.io Socket object.
* `client`: (**Deprecated**; use `socket` instead.) Synonym of `socket`.

Example:

```javascript
exports.handleMessageSecurity = async (hookName, context) => {
  const {message, sessionInfo: {readOnly}} = context;
  if (!readOnly || message.type !== 'COLLABROOM') return;
  if (await messageIsBenign(message)) return 'permitOnce';
};
```

## clientVars
Called from: `src/node/handler/PadMessageHandler.js`

Things in context:

1. clientVars - the basic `clientVars` built by the core
2. pad - the pad this session is about
3. socket - the socket.io Socket object

This hook is called after a client connects but before the initial configuration
is sent to the client. Plugins can use this hook to manipulate the
configuration. (Example: Add a tracking ID for an external analytics tool that
is used client-side.)

You can manipulate `clientVars` in two different ways:
* Return an object. The object will be merged into `clientVars` via
  `Object.assign()`, so any keys that already exist in `clientVars` will be
  overwritten by the values in the returned object.
* Modify `context.clientVars`. Beware: Other plugins might also be reading or
  manipulating the same `context.clientVars` object. To avoid race conditions,
  you are encouraged to return an object rather than modify
  `context.clientVars`.

If needed, you can access the user's account information (if authenticated) via
`context.socket.client.request.session.user`.

Examples:

```js
// Using an async function
exports.clientVars = async (hookName, context) => {
  const user = context.socket.client.request.session.user || {};
  return {'accountUsername': user.username || '<unknown>'}
};

// Using a regular function
exports.clientVars = (hookName, context, callback) => {
  const user = context.socket.client.request.session.user || {};
  return callback({'accountUsername': user.username || '<unknown>'});
};
```

## `getLineHTMLForExport`

Called from: `src/node/utils/ExportHtml.js`

This hook will allow a plug-in developer to re-write each line when exporting to
HTML.

Context properties:

* `apool`: Pool object.
* `attribLine`: Line attributes.
* `line`:
* `lineContent`:
* `text`: Line text.
* `padId`: Writable (not read-only) pad identifier.

Example:

```javascript
const AttributeMap = require('ep_etherpad-lite/static/js/AttributeMap');
const Changeset = require('ep_etherpad-lite/static/js/Changeset');

exports.getLineHTMLForExport = async (hookName, context) => {
  if (!context.attribLine) return;
  const [op] = Changeset.deserializeOps(context.attribLine);
  if (op == null) return;
  const heading = AttributeMap.fromString(op.attribs, context.apool).get('heading');
  if (!heading) return;
  context.lineContent = `<${heading}>${context.lineContent}</${heading}>`;
};
```

## exportHTMLAdditionalContent
Called from: `src/node/utils/ExportHtml.js`

Things in context:

1. padId

This hook will allow a plug-in developer to include additional HTML content in
the body of the exported HTML.

Example:

```js
exports.exportHTMLAdditionalContent = async (hookName, {padId}) => {
  return 'I am groot in ' + padId;
};
```

## stylesForExport
Called from: `src/node/utils/ExportHtml.js`

Things in context:

1. padId - The Pad Id

This hook will allow a plug-in developer to append Styles to the Exported HTML.

Example:

```js
exports.stylesForExport = function(hook, padId, cb){
  cb("body{font-size:13.37em !important}");
}
```

## aceAttribClasses
Called from: `src/static/js/linestylefilter.js`

This hook is called when attributes are investigated on a line. It is useful if
you want to add another attribute type or property type to a pad.

An attributes object is passed to the aceAttribClasses hook functions instead of
the usual context object. A hook function can either modify this object directly
or provide an object whose properties will be assigned to the attributes object.

Example:

```js
exports.aceAttribClasses = (hookName, attrs, cb) => {
  return cb([{
    sub: 'tag:sub',
  }]);
};
```

## exportFileName
Called from `src/node/handler/ExportHandler.js`

Things in context:

1. padId

This hook will allow a plug-in developer to modify the file name of an exported pad.  This is useful if you want to export a pad under another name and/or hide the padId under export.  Note that the doctype or file extension cannot be modified for security reasons.

Example:

```js
exports.exportFileName = function(hook, padId, callback){
  callback("newFileName"+padId);
}
```

## exportHtmlAdditionalTags
Called from `src/node/utils/ExportHtml.js`

Things in context:

1. Pad object

This hook will allow a plug-in developer to include more properties and attributes to support during HTML Export. If tags are stored as `['color', 'red']` on the attribute pool, use `exportHtmlAdditionalTagsWithData` instead. An Array should be returned.

Example:
```js
// Add the props to be supported in export
exports.exportHtmlAdditionalTags = function(hook, pad, cb){
  var padId = pad.id;
  cb(["massive","jugs"]);
};
```

## exportHtmlAdditionalTagsWithData
Called from `src/node/utils/ExportHtml.js`

Things in context:

1. Pad object

Identical to `exportHtmlAdditionalTags`, but for tags that are stored with a specific value (not simply `true`) on the attribute pool. For example `['color', 'red']`, instead of `['bold', true]`. This hook will allow a plug-in developer to include more properties and attributes to support during HTML Export. An Array of arrays should be returned. The exported HTML will contain tags like `<span data-color="red">` for the content where attributes are `['color', 'red']`.

Example:
```js
// Add the props to be supported in export
exports.exportHtmlAdditionalTagsWithData = function(hook, pad, cb){
  var padId = pad.id;
  cb([["color", "red"], ["color", "blue"]]);
};
```

## `exportEtherpadAdditionalContent`

Called from `src/node/utils/ExportEtherpad.js` and
`src/node/utils/ImportEtherpad.js`.

Called when exporting to an `.etherpad` file or when importing from an
`.etherpad` file. The hook function should return prefixes for pad-specific
records that should be included in the export/import. On export, all
`${prefix}:${padId}` and `${prefix}:${padId}:*` records are included in the
generated `.etherpad` file. On import, all `${prefix}:${padId}` and
`${prefix}:${padId}:*` records are loaded into the database.

Context properties: None.

Example:

```js
// Add support for exporting comments metadata
exports.exportEtherpadAdditionalContent = () => ['comments'];
```

## `exportEtherpad`

Called from `src/node/utils/ExportEtherpad.js`.

Called when exporting to an `.etherpad` file.

Context properties:

* `pad`: The exported pad's Pad object.
* `data`: JSONable output object. This is pre-populated with records from core
  Etherpad as well as pad-specific records with prefixes from the
  `exportEtherpadAdditionalContent` hook. Registered hook functions can modify
  this object (but not replace the object) to perform any desired
  transformations to the exported data (such as the inclusion of
  plugin-specific records). All registered hook functions are executed
  concurrently, so care should be taken to avoid race conditions with other
  plugins.
* `dstPadId`: The pad ID that should be used when writing pad-specific records
  to `data` (instead of `pad.id`). This avoids leaking the writable pad ID
  when a user exports a read-only pad. This might be a dummy value; plugins
  should not assume that it is either the pad's real writable ID or its
  read-only ID.

## `importEtherpad`

Called from `src/node/utils/ImportEtherpad.js`.

Called when importing from an `.etherpad` file.

Context properties:

* `pad`: Temporary Pad object containing the pad's data read from the imported
  `.etherpad` file. The `pad.db` object is a temporary in-memory database
  whose records will be copied to the real database after they are validated
  (see the `padCheck` hook). Registered hook functions MUST NOT use the real
  database to access (read or write) pad-specific records; they MUST instead
  use `pad.db`. All registered hook functions are executed concurrently, so
  care should be taken to avoid race conditions with other plugins.
* `data`: Raw JSONable object from the `.etherpad` file. This data must not be
  modified.
* `srcPadId`: The pad ID used for the pad-specific information in `data`.

## `import`

Called from: `src/node/handler/ImportHandler.js`

Called when a user submits a document for import, before the document is
converted to HTML. The hook function should return a truthy value if the hook
function elected to convert the document to HTML.

Context properties:

* `destFile`: The destination HTML filename.
* `fileEnding`: The lower-cased filename extension from `srcFile` **with leading
  period** (examples: `'.docx'`, `'.html'`, `'.etherpad'`).
* `padId`: The identifier of the destination pad.
* `srcFile`: The document to convert.
* `ImportError`: Subclass of Error that can be thrown to provide a specific
  error message to the user. The constructor's first argument must be a string
  matching one of the [known error
  identifiers](https://github.com/ether/etherpad-lite/blob/1.8.16/src/static/js/pad_impexp.js#L80-L86).

Example:

```javascript
exports.import = async (hookName, {fileEnding, ImportError}) => {
  // Reject all *.etherpad imports with a permission denied message.
  if (fileEnding === '.etherpad') throw new ImportError('permission');
};
```

## `userJoin`

Called from: `src/node/handler/PadMessageHandler.js`

Called after users have been notified that a new user has joined the pad.

Context properties:

* `authorId`: The user's author identifier.
* `displayName`: The user's display name.
* `padId`: The real (not read-only) identifier of the pad the user joined. This
  MUST NOT be shared with any users that are connected with read-only access.
* `readOnly`: Whether the user only has read-only access.
* `readOnlyPadId`: The read-only identifier of the pad the user joined.
* `socket`: The socket.io Socket object.

Example:

```javascript
exports.userJoin = async (hookName, {authorId, displayName, padId}) => {
  console.log(`${authorId} (${displayName}) joined pad ${padId});
};
```

## `userLeave`

Called from: `src/node/handler/PadMessageHandler.js`

Called when a user disconnects from a pad. This is useful if you want to perform
certain actions after a pad has been edited.

Context properties:

* `authorId`: The user's author ID.
* `padId`: The pad's real (not read-only) identifier.
* `readOnly`: If truthy, the user only has read-only access.
* `readOnlyPadId`: The pad's read-only identifier.
* `socket`: The socket.io Socket object.

Example:

```javascript
exports.userLeave = async (hookName, {author, padId}) => {
  console.log(`${author} left pad ${padId}`);
};
```

## `chatNewMessage`

Called from: `src/node/handler/PadMessageHandler.js`

Called when a user (or plugin) generates a new chat message, just before it is
saved to the pad and relayed to all connected users.

Context properties:

* `message`: The chat message object. Plugins can mutate this object to change
  the message text or add custom metadata to control how the message will be
  rendered by the `chatNewMessage` client-side hook. The message's `authorId`
  property can be trusted (the server overwrites any client-provided author ID
  value with the user's actual author ID before this hook runs).
* `padId`: The pad's real (not read-only) identifier.
* `pad`: The pad's Pad object.
