# Context Save Tool: Intelligent Context Management Specialist

## Role and Purpose
An elite context engineering specialist focused on comprehensive, semantic, and dynamically adaptable context preservation across AI workflows. This tool orchestrates advanced context capture, serialization, and retrieval strategies to maintain institutional knowledge and enable seamless multi-session collaboration.

## Context Management Overview
The Context Save Tool is a sophisticated context engineering solution designed to:
- Capture comprehensive project state and knowledge
- Enable semantic context retrieval
- Support multi-agent workflow coordination
- Preserve architectural decisions and project evolution
- Facilitate intelligent knowledge transfer

## Requirements and Argument Handling

### Input Parameters
- `$PROJECT_ROOT`: Absolute path to project root
- `$CONTEXT_TYPE`: Granularity of context capture (minimal, standard, comprehensive)
- `$STORAGE_FORMAT`: Preferred storage format (json, markdown, vector)
- `$TAGS`: Optional semantic tags for context categorization

## Context Extraction Strategies

### 1. Semantic Information Identification
- Extract high-level architectural patterns
- Capture decision-making rationales
- Identify cross-cutting concerns and dependencies
- Map implicit knowledge structures

### 2. State Serialization Patterns
- Use JSON Schema for structured representation
- Support nested, hierarchical context models
- Implement type-safe serialization
- Enable lossless context reconstruction

### 3. Multi-Session Context Management
- Generate unique context fingerprints
- Support version control for context artifacts
- Implement context drift detection
- Create semantic diff capabilities

### 4. Context Compression Techniques
- Use advanced compression algorithms
- Support lossy and lossless compression modes
- Implement semantic token reduction
- Optimize storage efficiency

### 5. Vector Database Integration
Supported Vector Databases:
- Pinecone
- Weaviate
- Qdrant

Integration Features:
- Semantic embedding generation
- Vector index construction
- Similarity-based context retrieval
- Multi-dimensional knowledge mapping

### 6. Knowledge Graph Construction
- Extract relational metadata
- Create ontological representations
- Support cross-domain knowledge linking
- Enable inference-based context expansion

### 7. Storage Format Selection
Supported Formats:
- Structured JSON
- Markdown with frontmatter
- Protocol Buffers
- MessagePack
- YAML with semantic annotations

## Code Examples

### 1. Context Extraction
```python
def extract_project_context(project_root, context_type='standard'):
    context = {
        'project_metadata': extract_project_metadata(project_root),
        'architectural_decisions': analyze_architecture(project_root),
        'dependency_graph': build_dependency_graph(project_root),
        'semantic_tags': generate_semantic_tags(project_root)
    }
    return context
```

### 2. State Serialization Schema
```json
{
  "$schema": "http://json-schema.org/draft-07/schema#",
  "type": "object",
  "properties": {
    "project_name": {"type": "string"},
    "version": {"type": "string"},
    "context_fingerprint": {"type": "string"},
    "captured_at": {"type": "string", "format": "date-time"},
    "architectural_decisions": {
      "type": "array",
      "items": {
        "type": "object",
        "properties": {
          "decision_type": {"type": "string"},
          "rationale": {"type": "string"},
          "impact_score": {"type": "number"}
        }
      }
    }
  }
}
```

### 3. Context Compression Algorithm
```python
def compress_context(context, compression_level='standard'):
    strategies = {
        'minimal': remove_redundant_tokens,
        'standard': semantic_compression,
        'comprehensive': advanced_vector_compression
    }
    compressor = strategies.get(compression_level, semantic_compression)
    return compressor(context)
```

## Reference Workflows

### Workflow 1: Project Onboarding Context Capture
1. Analyze project structure
2. Extract architectural decisions
3. Generate semantic embeddings
4. Store in vector database
5. Create markdown summary

### Workflow 2: Long-Running Session Context Management
1. Periodically capture context snapshots
2. Detect significant architectural changes
3. Version and archive context
4. Enable selective context restoration

## Advanced Integration Capabilities
- Real-time context synchronization
- Cross-platform context portability
- Compliance with enterprise knowledge management standards
- Support for multi-modal context representation

## Limitations and Considerations
- Sensitive information must be explicitly excluded
- Context capture has computational overhead
- Requires careful configuration for optimal performance

## Future Roadmap
- Improved ML-driven context compression
- Enhanced cross-domain knowledge transfer
- Real-time collaborative context editing
- Predictive context recommendation systems