#!/usr/bin/env python3
"""
SKILL.md Enhancement Script
Uses platform AI APIs to improve SKILL.md by analyzing reference documentation.

Usage:
    # Claude (default)
    skill-seekers enhance output/react/
    skill-seekers enhance output/react/ --api-key sk-ant-...

    # Gemini
    skill-seekers enhance output/react/ --target gemini --api-key AIzaSy...

    # OpenAI
    skill-seekers enhance output/react/ --target openai --api-key sk-proj-...
"""

import os
import sys
import json
import argparse
from pathlib import Path

# Add parent directory to path for imports when run as script
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from skill_seekers.cli.constants import API_CONTENT_LIMIT, API_PREVIEW_LIMIT
from skill_seekers.cli.utils import read_reference_files

try:
    import anthropic
except ImportError:
    print("❌ Error: anthropic package not installed")
    print("Install with: pip3 install anthropic")
    sys.exit(1)


class SkillEnhancer:
    def __init__(self, skill_dir, api_key=None):
        self.skill_dir = Path(skill_dir)
        self.references_dir = self.skill_dir / "references"
        self.skill_md_path = self.skill_dir / "SKILL.md"

        # Get API key - support both ANTHROPIC_API_KEY and ANTHROPIC_AUTH_TOKEN
        self.api_key = (api_key or
                       os.environ.get('ANTHROPIC_API_KEY') or
                       os.environ.get('ANTHROPIC_AUTH_TOKEN'))
        if not self.api_key:
            raise ValueError(
                "No API key provided. Set ANTHROPIC_API_KEY or ANTHROPIC_AUTH_TOKEN "
                "environment variable or use --api-key argument"
            )

        # Support custom base URL for alternative API endpoints
        base_url = os.environ.get('ANTHROPIC_BASE_URL')
        client_kwargs = {'api_key': self.api_key}
        if base_url:
            client_kwargs['base_url'] = base_url
            print(f"ℹ️  Using custom API base URL: {base_url}")

        self.client = anthropic.Anthropic(**client_kwargs)

    def read_current_skill_md(self):
        """Read existing SKILL.md"""
        if not self.skill_md_path.exists():
            return None
        return self.skill_md_path.read_text(encoding='utf-8')

    def enhance_skill_md(self, references, current_skill_md):
        """Use Claude to enhance SKILL.md"""

        # Build prompt
        prompt = self._build_enhancement_prompt(references, current_skill_md)

        print("\n🤖 Asking Claude to enhance SKILL.md...")
        print(f"   Input: {len(prompt):,} characters")

        try:
            message = self.client.messages.create(
                model="claude-sonnet-4-20250514",
                max_tokens=4096,
                temperature=0.3,
                messages=[{
                    "role": "user",
                    "content": prompt
                }]
            )

            # Handle response content - newer SDK versions may include ThinkingBlock
            # Find the TextBlock containing the actual response
            enhanced_content = None
            for block in message.content:
                if hasattr(block, 'text'):
                    enhanced_content = block.text
                    break

            if not enhanced_content:
                print("❌ Error: No text content found in API response")
                return None

            return enhanced_content

        except Exception as e:
            print(f"❌ Error calling Claude API: {e}")
            return None

    def _build_enhancement_prompt(self, references, current_skill_md):
        """Build the prompt for Claude"""

        # Extract skill name and description
        skill_name = self.skill_dir.name

        prompt = f"""You are enhancing a Claude skill's SKILL.md file. This skill is about: {skill_name}

I've scraped documentation and organized it into reference files. Your job is to create an EXCELLENT SKILL.md that will help Claude use this documentation effectively.

CURRENT SKILL.MD:
{'```markdown' if current_skill_md else '(none - create from scratch)'}
{current_skill_md or 'No existing SKILL.md'}
{'```' if current_skill_md else ''}

REFERENCE DOCUMENTATION:
"""

        for filename, content in references.items():
            prompt += f"\n\n## {filename}\n```markdown\n{content[:30000]}\n```\n"

        prompt += """

YOUR TASK:
Create an enhanced SKILL.md that includes:

1. **Clear "When to Use This Skill" section** - Be specific about trigger conditions
2. **Excellent Quick Reference section** - Extract 5-10 of the BEST, most practical code examples from the reference docs
   - Choose SHORT, clear examples that demonstrate common tasks
   - Include both simple and intermediate examples
   - Annotate examples with clear descriptions
   - Use proper language tags (cpp, python, javascript, json, etc.)
3. **Detailed Reference Files description** - Explain what's in each reference file
4. **Practical "Working with This Skill" section** - Give users clear guidance on how to navigate the skill
5. **Key Concepts section** (if applicable) - Explain core concepts
6. **Keep the frontmatter** (---\nname: ...\n---) intact

IMPORTANT:
- Extract REAL examples from the reference docs, don't make them up
- Prioritize SHORT, clear examples (5-20 lines max)
- Make it actionable and practical
- Don't be too verbose - be concise but useful
- Maintain the markdown structure for Claude skills
- Keep code examples properly formatted with language tags

OUTPUT:
Return ONLY the complete SKILL.md content, starting with the frontmatter (---).
"""

        return prompt

    def save_enhanced_skill_md(self, content):
        """Save the enhanced SKILL.md"""
        # Backup original
        if self.skill_md_path.exists():
            backup_path = self.skill_md_path.with_suffix('.md.backup')
            self.skill_md_path.rename(backup_path)
            print(f"  💾 Backed up original to: {backup_path.name}")

        # Save enhanced version
        self.skill_md_path.write_text(content, encoding='utf-8')
        print(f"  ✅ Saved enhanced SKILL.md")

    def run(self):
        """Main enhancement workflow"""
        print(f"\n{'='*60}")
        print(f"ENHANCING SKILL: {self.skill_dir.name}")
        print(f"{'='*60}\n")

        # Read reference files
        print("📖 Reading reference documentation...")
        references = read_reference_files(
            self.skill_dir,
            max_chars=API_CONTENT_LIMIT,
            preview_limit=API_PREVIEW_LIMIT
        )

        if not references:
            print("❌ No reference files found to analyze")
            return False

        print(f"  ✓ Read {len(references)} reference files")
        total_size = sum(len(c) for c in references.values())
        print(f"  ✓ Total size: {total_size:,} characters\n")

        # Read current SKILL.md
        current_skill_md = self.read_current_skill_md()
        if current_skill_md:
            print(f"  ℹ Found existing SKILL.md ({len(current_skill_md)} chars)")
        else:
            print(f"  ℹ No existing SKILL.md, will create new one")

        # Enhance with Claude
        enhanced = self.enhance_skill_md(references, current_skill_md)

        if not enhanced:
            print("❌ Enhancement failed")
            return False

        print(f"  ✓ Generated enhanced SKILL.md ({len(enhanced)} chars)\n")

        # Save
        print("💾 Saving enhanced SKILL.md...")
        self.save_enhanced_skill_md(enhanced)

        print(f"\n✅ Enhancement complete!")
        print(f"\nNext steps:")
        print(f"  1. Review: {self.skill_md_path}")
        print(f"  2. If you don't like it, restore backup: {self.skill_md_path.with_suffix('.md.backup')}")
        print(f"  3. Package your skill:")
        print(f"     skill-seekers package {self.skill_dir}/")

        return True


def main():
    parser = argparse.ArgumentParser(
        description='Enhance SKILL.md using platform AI APIs',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Claude (default)
  export ANTHROPIC_API_KEY=sk-ant-...
  skill-seekers enhance output/react/

  # Gemini
  export GOOGLE_API_KEY=AIzaSy...
  skill-seekers enhance output/react/ --target gemini

  # OpenAI
  export OPENAI_API_KEY=sk-proj-...
  skill-seekers enhance output/react/ --target openai

  # With explicit API key
  skill-seekers enhance output/react/ --api-key sk-ant-...

  # Dry run
  skill-seekers enhance output/godot/ --dry-run
"""
    )

    parser.add_argument('skill_dir', type=str,
                       help='Path to skill directory (e.g., output/steam-inventory/)')
    parser.add_argument('--api-key', type=str,
                       help='Platform API key (or set environment variable)')
    parser.add_argument('--target',
                       choices=['claude', 'gemini', 'openai'],
                       default='claude',
                       help='Target LLM platform (default: claude)')
    parser.add_argument('--dry-run', action='store_true',
                       help='Show what would be done without calling API')

    args = parser.parse_args()

    # Validate skill directory
    skill_dir = Path(args.skill_dir)
    if not skill_dir.exists():
        print(f"❌ Error: Directory not found: {skill_dir}")
        sys.exit(1)

    if not skill_dir.is_dir():
        print(f"❌ Error: Not a directory: {skill_dir}")
        sys.exit(1)

    # Dry run mode
    if args.dry_run:
        print(f"🔍 DRY RUN MODE")
        print(f"   Would enhance: {skill_dir}")
        print(f"   References: {skill_dir / 'references'}")
        print(f"   SKILL.md: {skill_dir / 'SKILL.md'}")

        refs_dir = skill_dir / "references"
        if refs_dir.exists():
            ref_files = list(refs_dir.glob("*.md"))
            print(f"   Found {len(ref_files)} reference files:")
            for rf in ref_files:
                size = rf.stat().st_size
                print(f"     - {rf.name} ({size:,} bytes)")

        print("\nTo actually run enhancement:")
        print(f"  skill-seekers enhance {skill_dir}")
        return

    # Check if platform supports enhancement
    try:
        from skill_seekers.cli.adaptors import get_adaptor

        adaptor = get_adaptor(args.target)

        if not adaptor.supports_enhancement():
            print(f"❌ Error: {adaptor.PLATFORM_NAME} does not support AI enhancement")
            print(f"\nSupported platforms for enhancement:")
            print("  - Claude AI (Anthropic)")
            print("  - Google Gemini")
            print("  - OpenAI ChatGPT")
            sys.exit(1)

        # Get API key
        api_key = args.api_key
        if not api_key:
            api_key = os.environ.get(adaptor.get_env_var_name(), '').strip()

        if not api_key:
            print(f"❌ Error: {adaptor.get_env_var_name()} not set")
            print(f"\nSet your API key for {adaptor.PLATFORM_NAME}:")
            print(f"  export {adaptor.get_env_var_name()}=...")
            print("Or provide it directly:")
            print(f"  skill-seekers enhance {skill_dir} --target {args.target} --api-key ...")
            sys.exit(1)

        # Run enhancement using adaptor
        print(f"\n{'='*60}")
        print(f"ENHANCING SKILL: {skill_dir}")
        print(f"Platform: {adaptor.PLATFORM_NAME}")
        print(f"{'='*60}\n")

        success = adaptor.enhance(Path(skill_dir), api_key)

        if success:
            print(f"\n✅ Enhancement complete!")
            print(f"\nNext steps:")
            print(f"  1. Review: {Path(skill_dir) / 'SKILL.md'}")
            print(f"  2. If you don't like it, restore backup: {Path(skill_dir) / 'SKILL.md.backup'}")
            print(f"  3. Package your skill:")
            print(f"     skill-seekers package {skill_dir}/ --target {args.target}")

        sys.exit(0 if success else 1)

    except ImportError as e:
        print(f"❌ Error: {e}")
        print("\nAdaptor system not available. Reinstall skill-seekers.")
        sys.exit(1)
    except ValueError as e:
        print(f"❌ Error: {e}")
        sys.exit(1)
    except Exception as e:
        print(f"❌ Unexpected error: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)


if __name__ == "__main__":
    main()
