module poly1305

import rand
import encoding.hex
import encoding.binary

// see https://github.com/floodyberry/poly1305-donna/blob/master/poly1305-donna.c
fn test_incremental_update_ported_from_poly1305donna() ! {
	nacl_key := [u8(0xee), 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2, 0xcb, 0x21,
		0x4d, 0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23, 0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4,
		0xc8, 0xcf, 0xf8, 0x80]
	nacl_msg := [u8(0x8e), 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba, 0x32,
		0xfc, 0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4, 0x47, 0x6f, 0xb8, 0xc5,
		0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c, 0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7, 0xf0,
		0x11, 0xec, 0x48, 0xc9, 0x72, 0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2, 0x27, 0x0d,
		0x6f, 0xb8, 0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7, 0xcc, 0x8a, 0xb9,
		0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68, 0x51, 0x7a, 0xcf, 0xea,
		0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda, 0x99, 0x83, 0x2b, 0x61, 0xca, 0x01, 0xb6,
		0xde, 0x56, 0x24, 0x4a, 0x9e, 0x88, 0xd5, 0xf9, 0xb3, 0x79, 0x73, 0xf6, 0x22, 0xa4, 0x3d,
		0x14, 0xa6, 0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4, 0x5a, 0x74, 0xe3, 0x55, 0xa5]

	nacl_mac := [u8(0xf3), 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5, 0x2a, 0x7d, 0xfb, 0x4b, 0x3d,
		0x33, 0x05, 0xd9]
	// generates a final value of (2^130 - 2) == 3
	wrap_key := [u8(0x02), 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00]

	wrap_msg := [u8(0xff), 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff]

	wrap_mac := [u8(0x03), 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00]

	// mac of the macs of messages of length 0 to 256, where the key and messages
	// have all their values set to the length
	total_key := [u8(0x01), 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa,
		0xf9, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff]

	total_mac := [u8(0x64), 0xaf, 0xe2, 0xe8, 0xd6, 0xad, 0x7b, 0xbd, 0xd2, 0x87, 0xf9, 0x7c, 0x44,
		0x62, 0x3d, 0x39]

	// poly1305_auth(mac, nacl_msg, sizeof(nacl_msg), nacl_key);
	mut mac := []u8{len: tag_size}
	create_tag(mut mac, nacl_msg, nacl_key)!
	assert mac == nacl_mac
	// result &= poly1305_verify(nacl_mac, mac);
	mut result := verify_tag(nacl_mac, nacl_msg, nacl_key)
	assert result == true

	// poly1305_init(&ctx, nacl_key);
	// poly1305_update(&ctx, nacl_msg +   0, 32);
	// poly1305_update(&ctx, nacl_msg +  32, 64);
	// poly1305_update(&ctx, nacl_msg +  96, 16);
	// poly1305_update(&ctx, nacl_msg + 112,  8);
	// poly1305_update(&ctx, nacl_msg + 120,  4);
	// poly1305_update(&ctx, nacl_msg + 124,  2);
	// poly1305_update(&ctx, nacl_msg + 126,  1);
	// poly1305_update(&ctx, nacl_msg + 127,  1);
	// poly1305_update(&ctx, nacl_msg + 128,  1);
	// poly1305_update(&ctx, nacl_msg + 129,  1);
	// poly1305_update(&ctx, nacl_msg + 130,  1);
	// poly1305_finish(&ctx, mac);
	//	result &= poly1305_verify(nacl_mac, mac);
	mut po := new(nacl_key)!
	mut tag := []u8{len: tag_size}

	po.update(nacl_msg)
	po.finish(mut tag)
	assert tag == nacl_mac

	// lets reset and reinit poly1305 instance and performs test for incremental update
	// Note: as security notes, you should use same key twice for authenticated same messages
	po.reinit(nacl_key)
	po.update(nacl_msg[0..32])
	po.update(nacl_msg[32..96])
	po.update(nacl_msg[96..112])
	po.update(nacl_msg[112..120])
	po.update(nacl_msg[120..124])
	po.update(nacl_msg[124..126])
	po.update(nacl_msg[126..127])
	po.update(nacl_msg[127..128])
	po.update(nacl_msg[128..129])
	po.update(nacl_msg[129..130])
	po.update(nacl_msg[130..131])
	po.finish(mut tag)
	assert tag == nacl_mac

	// poly1305_auth(mac, wrap_msg, sizeof(wrap_msg), wrap_key);
	// result &= poly1305_verify(wrap_mac, mac);
	mut pi := new(wrap_key)!
	pi.update(wrap_msg)
	pi.finish(mut tag)
	assert tag == wrap_mac

	// poly1305_init(&total_ctx, total_key);
	dump(total_key.len)
	mut tkey := []u8{len: key_size}
	_ := copy(mut tkey, total_key)
	mut ptot := new(tkey)!
	mut all_key := []u8{len: 32}
	mut all_msg := []u8{len: 256}
	for i := 0; i < 256; i++ {
		// set key and message to 'i,i,i..'
		for x := 0; x < all_key.len; x++ {
			all_key[x] = u8(i)
		}
		for y := 0; y < i; y++ {
			all_msg[y] = u8(i)
		}
		create_tag(mut tag, all_msg[..i], all_key)!
		ptot.update(tag)
	}
	ptot.finish(mut tag)
	assert total_mac == tag
}

fn test_poly1305_with_smoked_messages_are_working_normally() ! {
	key := rand.bytes(32)!
	// generates smoked message with full bits is set
	msg := u8(0xff).repeat(135).bytes()
	// msg = ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff
	mut tag := []u8{len: tag_size}
	create_tag(mut tag, msg, key)!

	// lets verify its working normally
	valid := verify_tag(tag, msg, key)
	assert valid == true
}

// This is a test case from RFC 8439 vector test data.
// There are 12 cases provided.
fn test_poly1305_core_rfc_vector_tests() ! {
	for i, c in rfc_test_cases {
		mut key := hex.decode(c.key) or { panic(err.msg()) }
		msg := hex.decode(c.msg) or { panic(err.msg()) }
		expected_tag := hex.decode(c.tag) or { panic(err.msg()) }

		mut tag := []u8{len: tag_size}

		// check for instance based method
		mut po := new(key)!
		po.update(msg)
		valid := po.verify(expected_tag)
		assert valid == true

		po.finish(mut tag)
		// check tag same with expected_tag
		assert tag == expected_tag

		// verify the tag has right result
		assert verify_tag(tag, msg, key) == true

		create_tag(mut tag, msg, key)!
		assert tag == expected_tag
		assert verify_tag(tag, msg, key) == true
	}
}

fn test_poly1305_function_based_core_functionality() ! {
	for i, c in rfc_test_cases {
		mut key := hex.decode(c.key) or { panic(err.msg()) }
		mut msg := hex.decode(c.msg) or { panic(err.msg()) }

		expected_tag := hex.decode(c.tag) or { panic(err.msg()) }
		mut tag := []u8{len: tag_size}

		mut po := new(key)!

		poly1305_update_block(mut po, msg)
		// you can finalize directly, maybe there are some leftover bytes unprocessed
		// call .finish instead
		// finalize(mut tag, mut po.h, po.s)
		po.finish(mut tag)
		assert tag == expected_tag
	}
}

// its comes from golang poly1305 vector test, except minus with changed internal state test
fn test_poly1305_smoked_data_vectors() ! {
	for i, c in testdata {
		mut key := hex.decode(c.key)!
		mut msg := hex.decode(c.msg)!
		expected_tag := hex.decode(c.tag)!
		mut s := Uint192{}
		mut poly := new(key)!
		if c.state != '' {
			// state is hex encoded in big-endian byte order
			buf := hex.decode(c.state)!

			if buf.len != 3 * 8 {
				panic('incorrect state length')
			}

			s.lo = binary.big_endian_u64(buf[16..24])
			s.mi = binary.big_endian_u64(buf[8..16])
			s.hi = binary.big_endian_u64(buf[0..8])
			// set with accumulator s
			poly.h = s
		}
		mut tag := []u8{len: tag_size}

		poly.update(msg)
		poly.finish(mut tag)
		assert tag == expected_tag
		//
		poly.reinit(key)
		poly.h = s
		poly.update(msg)
		mut res := poly.verify(expected_tag)
		assert res == true
		// If the key is zero, the tag will always be zero, independent of the input.
		if msg.len > 0 && key.len != 32 {
			msg[0] ^= 0xff
			res = poly.verify(tag) // same with verify_tag(tag, msg, key)
			assert res == false
			msg[0] ^= 0xff
		}

		// If the input is empty, the tag only depends on the second half of the key.
		if msg.len > 0 {
			key[0] ^= 0xff
			res = verify_tag(tag, msg, key)
			assert res == false
			key[0] ^= 0xff
		}
		tag[0] ^= 0xff
		res = verify_tag(tag, msg, key)
		assert res == false
		tag[0] ^= 0xff
	}
}

struct RFCTestCases {
	key string
	msg string
	tag string
}

const rfc_test_cases = [
	// 0. core basic example test
	RFCTestCases{
		key: '85d6be7857556d337f4452fe42d506a80103808afb0db2fd4abff6af4149f51b'
		msg: '43727970746f6772617068696320466f72756d2052657365617263682047726f7570'
		tag: 'a8061dc1305136c6c22b8baf0c0127a9'
	},
	// https://datatracker.ietf.org/doc/html/rfc8439#appendix-A.3
	// 1. A.3.1 case
	RFCTestCases{
		key: '0000000000000000000000000000000000000000000000000000000000000000'
		msg: '00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000'
	},
	// 2. A.3.2 case
	RFCTestCases{
		key: '0000000000000000000000000000000036e5f6b5c5e06070f0efca96227a863e'
		msg: '416e79207375626d697373696f6e20746f20746865204945544620696e74656e6465642062792074686520436f6e7472696275746f7220666f72207075626c69636174696f6e20617320616c6c206f722070617274206f6620616e204945544620496e7465726e65742d4472616674206f722052464320616e6420616e792073746174656d656e74206d6164652077697468696e2074686520636f6e74657874206f6620616e204945544620616374697669747920697320636f6e7369646572656420616e20224945544620436f6e747269627574696f6e222e20537563682073746174656d656e747320696e636c756465206f72616c2073746174656d656e747320696e20494554462073657373696f6e732c2061732077656c6c206173207772697474656e20616e6420656c656374726f6e696320636f6d6d756e69636174696f6e73206d61646520617420616e792074696d65206f7220706c6163652c207768696368206172652061646472657373656420746f'
		tag: '36e5f6b5c5e06070f0efca96227a863e'
	},
	// 3. A.3.3 case
	RFCTestCases{
		key: '36e5f6b5c5e06070f0efca96227a863e00000000000000000000000000000000'
		msg: '416e79207375626d697373696f6e20746f20746865204945544620696e74656e6465642062792074686520436f6e7472696275746f7220666f72207075626c69636174696f6e20617320616c6c206f722070617274206f6620616e204945544620496e7465726e65742d4472616674206f722052464320616e6420616e792073746174656d656e74206d6164652077697468696e2074686520636f6e74657874206f6620616e204945544620616374697669747920697320636f6e7369646572656420616e20224945544620436f6e747269627574696f6e222e20537563682073746174656d656e747320696e636c756465206f72616c2073746174656d656e747320696e20494554462073657373696f6e732c2061732077656c6c206173207772697474656e20616e6420656c656374726f6e696320636f6d6d756e69636174696f6e73206d61646520617420616e792074696d65206f7220706c6163652c207768696368206172652061646472657373656420746f'
		tag: 'f3477e7cd95417af89a6b8794c310cf0'
	},
	// 4. A.3.4 case
	RFCTestCases{
		key: '1c9240a5eb55d38af333888604f6b5f0473917c1402b80099dca5cbc207075c0'
		msg: '2754776173206272696c6c69672c20616e642074686520736c6974687920746f7665730a446964206779726520616e642067696d626c6520696e2074686520776162653a0a416c6c206d696d737920776572652074686520626f726f676f7665732c0a416e6420746865206d6f6d65207261746873206f757467726162652e'
		tag: '4541669a7eaaee61e708dc7cbcc5eb62'
	},
	// Test Vector #5: If one uses 130-bit partial reduction, does the code
	// handle the case where partially reduced final result is not fully
	// reduced!
	// r := '02000000000000000000000000000000'
	// s := '00000000000000000000000000000000'
	// data := 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
	// result_tag := '03000000000000000000000000000000'
	// key := r + s
	RFCTestCases{
		key: '0200000000000000000000000000000000000000000000000000000000000000'
		msg: 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
		tag: '03000000000000000000000000000000'
	},
	// Test Vector #6: What happens if addition of s overflows modulo 2^128!
	// r := '02000000000000000000000000000000'
	// s := 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
	// data := '02000000000000000000000000000000'
	// result_tag := '03000000000000000000000000000000'
	// key := r + s
	RFCTestCases{
		key: '02000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
		msg: '02000000000000000000000000000000'
		tag: '03000000000000000000000000000000'
	},
	// Test Vector #7: What happens if data limb is all ones and there is
	// carry from lower limb!
	// r := '01000000000000000000000000000000'
	// s := '00000000000000000000000000000000'
	// data := 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF11000000000000000000000000000000'
	// result_tag := '05000000000000000000000000000000'
	RFCTestCases{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		msg: 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF11000000000000000000000000000000'
		tag: '05000000000000000000000000000000'
	},
	// Test Vector #8: What happens if final result from polynomial part is
	// exactly 2^130-5!
	// r := '01000000000000000000000000000000'
	// s := '00000000000000000000000000000000'
	// data := 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBFEFEFEFEFEFEFEFEFEFEFEFEFEFEFE01010101010101010101010101010101'
	// result_tag := '00000000000000000000000000000000'
	RFCTestCases{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		msg: 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBFEFEFEFEFEFEFEFEFEFEFEFEFEFEFE01010101010101010101010101010101'
		tag: '00000000000000000000000000000000'
	},
	// Test Vector #9: What happens if final result from polynomial part is
	//  exactly 2^130-6!
	// r := '02000000000000000000000000000000'
	// s := '00000000000000000000000000000000'
	// data := 'FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
	// result_tag := 'FAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
	RFCTestCases{
		key: '0200000000000000000000000000000000000000000000000000000000000000'
		msg: 'FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
		tag: 'FAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
	},
	// Test Vector #10: What happens if 5*H+L-type reduction produces
	//  131-bit intermediate result!
	// r := '01000000000000000400000000000000'
	// s := '00000000000000000000000000000000'
	// data := 'E33594D7505E43B900000000000000003394D7505E4379CD01000000000000000000000000000000000000000000000001000000000000000000000000000000'
	// result_tag := '14000000000000005500000000000000'
	RFCTestCases{
		key: '0100000000000000040000000000000000000000000000000000000000000000'
		msg: 'E33594D7505E43B900000000000000003394D7505E4379CD01000000000000000000000000000000000000000000000001000000000000000000000000000000'
		tag: '14000000000000005500000000000000'
	},
	// Test Vector #11: What happens if 5*H+L-type reduction produces
	//   131-bit final result!
	// r := '01000000000000000400000000000000'
	// s := '00000000000000000000000000000000'
	// data := 'E33594D7505E43B900000000000000003394D7505E4379CD010000000000000000000000000000000000000000000000'
	// result_tag := '13000000000000000000000000000000'
	RFCTestCases{
		key: '0100000000000000040000000000000000000000000000000000000000000000'
		msg: 'E33594D7505E43B900000000000000003394D7505E4379CD010000000000000000000000000000000000000000000000'
		tag: '13000000000000000000000000000000'
	},
]

struct TestCase {
	msg   string
	key   string
	tag   string
	state string
}

const testdata = [
	TestCase{
		key: '3b3a29e93b213a5c5c3b3b053a3a8c0d00000000000000000000000000000000'
		tag: '6dc18b8c344cd79927118bbe84b7f314'
		msg: '81d8b2e46a25213b58fee4213a2a28e921c12a9632516d3b73272727becf2129'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: '04000000000000000000000000000000' // (2¹³⁰-1) % (2¹³⁰-5)
		msg: 'ffffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: 'faffffffffffffffffffffffffffffff' // (2¹³⁰-6) % (2¹³⁰-5)
		msg: 'faffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000' // (2¹³⁰-5) % (2¹³⁰-5)
		msg: 'fbffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: 'f9ffffffffffffffffffffffffffffff' // (2*(2¹³⁰-6)) % (2¹³⁰-5)
		msg: 'fafffffffffffffffffffffffffffffffaffffffffffffffffffffffffffffff00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000' // (2*(2¹³⁰-5)) % (2¹³⁰-5)
		msg: 'fbfffffffffffffffffffffffffffffffbffffffffffffffffffffffffffffff00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: 'f8ffffffffffffffffffffffffffffff' // (3*(2¹³⁰-6)) % (2¹³⁰-5)
		msg: 'fafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffaffffffffffffffffffffffffffffff000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000' // (3*(2¹³⁰-5)) % (2¹³⁰-5)
		msg: 'fbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbffffffffffffffffffffffffffffff000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: 'f7ffffffffffffffffffffffffffffff' // (4*(2¹³⁰-6)) % (2¹³⁰-5)
		msg: 'fafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffaffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000' // (4*(2¹³⁰-5)) % (2¹³⁰-5)
		msg: 'fbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: 'f3ffffffffffffffffffffffffffffff' // (8*(2¹³⁰-6)) % (2¹³⁰-5)
		msg: 'fafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffaffffffffffffffffffffffffffffff00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000' // (8*(2¹³⁰-5)) % (2¹³⁰-5)
		msg: 'fbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbffffffffffffffffffffffffffffff00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: 'ebffffffffffffffffffffffffffffff' // (16*(2¹³⁰-6)) % (2¹³⁰-5)
		msg: 'fafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffafffffffffffffffffffffffffffffffaffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0100000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000' // (16*(2¹³⁰-5)) % (2¹³⁰-5)
		msg: 'fbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbfffffffffffffffffffffffffffffffbffffffffffffffffffffffffffffff0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	// original smoke tests
	TestCase{
		key: '746869732069732033322d62797465206b657920666f7220506f6c7931333035'
		tag: 'a6f745008f81c916a20dcc74eef2b2f0'
		msg: '48656c6c6f20776f726c6421'
	},
	TestCase{
		key: '746869732069732033322d62797465206b657920666f7220506f6c7931333035'
		tag: '49ec78090e481ec6c26b33b91ccc0307'
		msg: '0000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '746869732069732033322d62797465206b657920666f7220506f6c7931333035'
		tag: 'da84bcab02676c38cdb015604274c2aa'
		msg: '000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	TestCase{
		key: '0000000000000000000000000000000000000000000000000000000000000000'
		tag: '00000000000000000000000000000000'
		msg: '000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000'
	},
	// randomly generated
	TestCase{
		key: '52fdfc072182654f163f5f0f9a621d729566c74d10037c4d7bbb0407d1e2c649'
		tag: '9566c74d10037c4d7bbb0407d1e2c649'
		msg: ''
	},
	TestCase{
		key: '81855ad8681d0d86d1e91e00167939cb6694d2c422acd208a0072939487f6999'
		tag: 'eaa270caaa12faa39b797374a4b8a420'
		msg: 'eb'
	},
	TestCase{
		key: '9d18a44784045d87f3c67cf22746e995af5a25367951baa2ff6cd471c483f15f'
		tag: 'dbea66e1da48a8f822887c6162c2acf1'
		msg: 'b90b'
	},
	TestCase{
		key: 'adb37c5821b6d95526a41a9504680b4e7c8b763a1b1d49d4955c848621632525'
		tag: '6ac09aaa88c32ee95a7198376f16abdb'
		msg: '3fec73'
	},
	TestCase{
		key: '8dd7a9e28bf921119c160f0702448615bbda08313f6a8eb668d20bf505987592'
		tag: 'b1443487f97fe340b04a74719ed4de68'
		msg: '1e668a5b'
	},
	TestCase{
		key: 'df2c7fc4844592d2572bcd0668d2d6c52f5054e2d0836bf84c7174cb7476364c'
		tag: '7463be0f9d99a5348039e4afcbf4019c'
		msg: 'c3dbd968b0'
	},
	TestCase{
		key: 'f7172ed85794bb358b0c3b525da1786f9fff094279db1944ebd7a19d0f7bbacb'
		tag: '2edaee3bcf303fd05609e131716f8157'
		msg: 'e0255aa5b7d4'
	},
	TestCase{
		key: '4bec40f84c892b9bffd43629b0223beea5f4f74391f445d15afd4294040374f6'
		tag: '965f18767420c1d94a4ef657e8d15e1e'
		msg: '924b98cbf8713f'
	},
	TestCase{
		key: '8d962d7c8d019192c24224e2cafccae3a61fb586b14323a6bc8f9e7df1d92933'
		tag: '2bf4a33287dd6d87e1ed4282f7342b6a'
		msg: '3ff993933bea6f5b'
	},
	TestCase{
		key: '3af6de0374366c4719e43a1b067d89bc7f01f1f573981659a44ff17a4c7215a3'
		tag: 'c5e987b60373a48893c5af30acf2471f'
		msg: 'b539eb1e5849c6077d'
	},
	TestCase{
		key: 'bb5722f5717a289a266f97647981998ebea89c0b4b373970115e82ed6f4125c8'
		tag: '19f0f640b309d168ea1b480e6a4faee5'
		msg: 'fa7311e4d7defa922daa'
	},
	TestCase{
		key: 'e7786667f7e936cd4f24abf7df866baa56038367ad6145de1ee8f4a8b0993ebd'
		tag: 'de75e5565d97834b9fa84ad568d31359'
		msg: 'f8883a0ad8be9c3978b048'
	},
	TestCase{
		key: '83e56a156a8de563afa467d49dec6a40e9a1d007f033c2823061bdd0eaa59f8e'
		tag: 'de184a5a9b826aa203c5c017986d6690'
		msg: '4da6430105220d0b29688b73'
	},
	TestCase{
		key: '4b8ea0f3ca9936e8461f10d77c96ea80a7a665f606f6a63b7f3dfd2567c18979'
		tag: '7478f18d9684905aa5d1a34ee67e4c84'
		msg: 'e4d60f26686d9bf2fb26c901ff'
	},
	TestCase{
		key: '354cde1607ee294b39f32b7c7822ba64f84ab43ca0c6e6b91c1fd3be89904341'
		tag: '3b2008a9c52b5308f5538b789ab5506f'
		msg: '79d3af4491a369012db92d184fc3'
	},
	TestCase{
		key: '9d1734ff5716428953bb6865fcf92b0c3a17c9028be9914eb7649c6c93478009'
		tag: '71c8e76a67a505b7370b562ba15ba032'
		msg: '79d1830356f2a54c3deab2a4b4475d'
	},
	TestCase{
		key: '63afbe8fb56987c77f5818526f1814be823350eab13935f31d84484517e924ae'
		tag: '1dc895f74f866bdb3edf6c4430829c1c'
		msg: 'f78ae151c00755925836b7075885650c'
	},
	TestCase{
		key: '30ec29a3703934bf50a28da102975deda77e758579ea3dfe4136abf752b3b827'
		tag: 'afca2b3ba7b0e1a928001966883e9b16'
		msg: '1d03e944b3c9db366b75045f8efd69d22ae5411947cb553d7694267aef4ebcea406b32d6108bd68584f57e37caac6e33feaa3263a399437024ba9c9b14678a274f01a910ae295f6efbfe5f5abf44ccde263b5606633e2bf0006f28295d7d39069f01a239c4365854c3af7f6b41d631f92b9a8d12f41257325fff332f7576b0620556304a3e3eae14c28d0cea39d2901a52720da85ca1e4b38eaf3f'
	},
	TestCase{
		key: '44c6c6ef8362f2f54fc00e09d6fc25640854c15dfcacaa8a2cecce5a3aba53ab'
		tag: '6f2a09aa76c9b76774e31ec02dcf7991'
		msg: '705b18db94b4d338a5143e63408d8724b0cf3fae17a3f79be1072fb63c35d6042c4160f38ee9e2a9f3fb4ffb0019b454d522b5ffa17604193fb8966710a7960732ca52cf53c3f520c889b79bf504cfb57c7601232d589baccea9d6e263e25c27741d3f6c62cbbb15d9afbcbf7f7da41ab0408e3969c2e2cdcf233438bf1774ace7709a4f091e9a83fdeae0ec55eb233a9b5394cb3c7856b546d313c8a3b4c1c0e05447f4ba370eb36dbcfdec90b302dcdc3b9ef522e2a6f1ed0afec1f8e20faabedf6b162e717d3a748a58677a0c56348f8921a266b11d0f334c62fe52ba53af19779cb2948b6570ffa0b773963c130ad797ddea'
	},
	TestCase{
		key: 'fe4e3ad29b5125210f0ef1c314090f07c79a6f571c246f3e9ac0b7413ef110bd'
		tag: '27381e3fc2a356103fb796f107d826e7'
		msg: '58b00ce73bff706f7ff4b6f44090a32711f3208e4e4b89cb5165ce64002cbd9c2887aa113df2468928d5a23b9ca740f80c9382d9c6034ad2960c796503e1ce221725f50caf1fbfe831b10b7bf5b15c47a53dbf8e7dcafc9e138647a4b44ed4bce964ed47f74aa594468ced323cb76f0d3fac476c9fb03fc9228fbae88fd580663a0454b68312207f0a3b584c62316492b49753b5d5027ce15a4f0a58250d8fb50e77f2bf4f0152e5d49435807f9d4b97be6fb77970466a5626fe33408cf9e88e2c797408a32d29416baf206a329cfffd4a75e498320982c85aad70384859c05a4b13a1d5b2f5bfef5a6ed92da482caa9568e5b6fe9d8a9ddd9eb09277b92cef9046efa18500944cbe800a0b1527ea6'
	},
	TestCase{
		key: '4729a861d2f6497a3235c37f4192779ec1d96b3b1c5424fce0b727b03072e641'
		tag: '0173965669fb9de88d38a827a0271271'
		msg: '5a761f03abaa40abc9448fddeb2191d945c04767af847afd0edb5d8857b799acb18e4affabe3037ffe7fa68aa8af5e39cc416e734d373c5ebebc9cdcc595bcce3c7bd3d8df93fab7e125ddebafe65a31bd5d41e2d2ce9c2b17892f0fea1931a290220777a93143dfdcbfa68406e877073ff08834e197a4034aa48afa3f85b8a62708caebbac880b5b89b93da53810164402104e648b6226a1b78021851f5d9ac0f313a89ddfc454c5f8f72ac89b38b19f53784c19e9beac03c875a27db029de37ae37a42318813487685929359ca8c5eb94e152dc1af42ea3d1676c1bdd19ab8e2925c6daee4de5ef9f9dcf08dfcbd02b80809398585928a0f7de50be1a6dc1d5768e8537988fddce562e9b948c918bba3e933e5c400cde5e60c5ead6fc7ae77ba1d259b188a4b21c86fbc23d728b45347eada650af24c56d0800a8691332088a805bd55c446e25eb07590bafcccbec6177536401d9a2b7f512b54bfc9d00532adf5aaa7c3a96bc59b489f77d9042c5bce26b163defde5ee6a0fbb3e9346cef81f0ae9515ef30fa47a364e75aea9e111d596e685a591121966e031650d510354aa845580ff560760fd36514ca197c875f1d02d9216eba7627e2398322eb5cf43d72bd2e5b887d4630fb8d4747ead6eb82acd1c5b078143ee26a586ad23139d5041723470bf24a865837c'
	},
	TestCase{
		key: '9123461c41f5ff99aa99ce24eb4d788576e3336e65491622558fdf297b9fa007'
		tag: '1eb0cdad9237905250d30a24fe172a34'
		msg: '864bafd7cd4ca1b2fb5766ab431a032b72b9a7e937ed648d0801f29055d3090d2463718254f9442483c7b98b938045da519843854b0ed3f7ba951a493f321f0966603022c1dfc579b99ed9d20d573ad53171c8fef7f1f4e4613bb365b2ebb44f0ffb6907136385cdc838f0bdd4c812f042577410aca008c2afbc4c79c62572e20f8ed94ee62b4de7aa1cc84c887e1f7c31e927dfe52a5f8f46627eb5d3a4fe16fafce23623e196c9dfff7fbaff4ffe94f4589733e563e19d3045aad3e226488ac02cca4291aed169dce5039d6ab00e40f67aab29332de1448b35507c7c8a09c4db07105dc31003620405da3b2169f5a910c9d0096e5e3ef1b570680746acd0cc7760331b663138d6d342b051b5df410637cf7aee9b0c8c10a8f9980630f34ce001c0ab7ac65e502d39b216cbc50e73a32eaf936401e2506bd8b82c30d346bc4b2fa319f245a8657ec122eaf4ad5425c249ee160e17b95541c2aee5df820ac85de3f8e784870fd87a36cc0d163833df636613a9cc947437b6592835b9f6f4f8c0e70dbeebae7b14cdb9bc41033aa5baf40d45e24d72eac4a28e3ca030c9937ab8409a7cbf05ae21f97425254543d94d115900b90ae703b97d9856d2441d14ba49a677de8b18cb454b99ddd9daa7ccbb7500dae4e2e5df8cf3859ebddada6745fba6a04c5c37c7ca35036f11732ce8bc27b48868611fc73c82a491bfabd7a19df50fdc78a55dbbc2fd37f9296566557fab885b039f30e706f0cd5961e19b642221db44a69497b8ad99408fe1e037c68bf7c5e5de1d2c68192348ec1189fb2e36973cef09ff14be23922801f6eaee41409158b45f2dec82d17caaba160cd6'
	},
	TestCase{
		key: '40ff73495fe4a05ce1202ca7287ed3235b95e69f571fa5e656aaa51fae1ebdd7'
		tag: '2e619d8ea81b77484e4fddeb29844e4b'
		msg: 'aa6269c2ec7f4057b33593bc84888c970fd528d4a99a1eab9d2420134537cd6d02282e0981e140232a4a87383a21d1845c408ad757043813032a0bd5a30dcca6e3aa2df04715d879279a96879a4f3690ac2025a60c7db15e0501ebc34b734355fe4a059bd3899d920e95f1c46d432f9b08e64d7f9b38965d5a77a7ac183c3833e1a3425ead69d4f975012fd1a49ed832f69e6e9c63b453ec049c9e7a5cf944232d10353f64434abae060f6506ad3fdb1f4415b0af9ce8c208bc20ee526741539fa3203c77ecba410fd6718f227e0b430f9bcb049a3d38540dc222969120ce80f2007cd42a708a721aa29987b45d4e428811984ecad349cc35dd93515cefe0b002cee5e71c47935e281ebfc4b8b652b69ccb092e55a20f1b9f97d046296124621928739a86671cc180152b953e3bf9d19f825c3dd54ae1688e49efb5efe65dcdad34bc860010e7c8c997cd5f9e320ca7d39d4ba801a175b1c76f057832f3f36d7d893e216e4c7bbdb548d0ba48449330027368b34f9c69776b4591532da1c5be68ef4eebe8cb8fa7dc5483fb70c2c896334cb1f9cb5dfe044fa086197ff5dfd02f2ba3884c53dd718c8560da743a8e9d4aeae20ccef002d82ca352592b8d8f2a8df3b0c35f15b9b370dca80d4ca8e9a133eb52094f2dd5c08731f52315d828846e37df68fd10658b480f2ac84233633957e688e924ffe3713b52c76fd8a56da8bb07daa8eb4eb8f7334f99256e2766a4109150eed424f0f743543cdea66e5baaa03edc918e8305bb19fc0c6b4ddb4aa3886cb5090940fc6d4cabe2153809e4ed60a0e2af07f1b2a6bb5a6017a578a27cbdc20a1759f76b0889a83ce25ce3ca91a4eb5c2f8580819da04d02c41770c01746de44f3db6e3402e7873db7635516e87b33e4b412ba3df68544920f5ea27ec097710954f42158bdba66d4814c064b4112538676095467c89ba98e6a543758d7093a494df'
	},
	TestCase{
		key: '5cc36d09c7a6472a41f29c380a987b1ecdcf84765f4e5d3ceefc1c02181f570f'
		tag: '0d57b8cbea8090df0541354673dcb4e0'
		msg: '44fcd629f08dc1ef53c9ae0d8869fe67fdc7a2c67b425f13c5be8d9f630c1d063c02fd75cf64c1aec9d2e2ef6e6431d5f5ad0489078dc61f46494dccf403dad7f094170d2c3e29c198b0f341e284c4be8fa60c1a478d6bd55dd2c04dad86d2053d5d25b014e3d8b64322cdcb5004faa46cfa2d6ad2ff933bc3bd9a5a74660af3d048a9a43634c0250427d9a6219197a3f3633f841753ba7c27f3619f387b6b1a6cb9c1dc227674aa020724d137da2cb87b1615d512974fa4747dd1e17d02c9462a44fec150ca3a8f99cc1e4953365e4299565e108535b1f62e1d4ba18e17a52164418bfd1a933f7fb3a126c860830a87293d9271da736e4398c1e37fb75c4bf02786e1faf4b610cd1377fbb9ae180655a0abefbad700c09473469f1eca5a66d53fa3dc7cd3e7c3b0411d7e145f96eb9654ab94913dda503a50f9e773842f4d2a5faa60869bf365830511f2ededd03e0a73000edb60c9a29a5f5e194cf3b5667a694690384599d116f8d2fd93b2aed55b7d44b5b054f3f38e788e4fdf36e591568c41d1052cad0fcb68ca4c4bf5090d57df9db6f0d91dd8b11b804f331adb7efb087a5604e9e22b4d54db40bcbc6e272ff5eaddfc1471459e59f0554c58251342134a8daaef1498069ba581ef1da2510be92843487a4eb8111c79a6f0195fc38ad6aee93c1df2b5897eaa38ad8f47ab2fe0e3aa3e6accbfd4c16d468433185fc61c861b96ca65e34d31f24d6f56ee85092314a4d7656205c15322f1c97613c079eae292ba966e10d1e700164e518b243f424c46f9ea63db1c2c34b512c403c128ee19030a6226517b805a072512a5e4cd274b7fd1fa23f830058208ff1a063b41039c74036b5b3da8b1a0b93135a710352da0f6c31203a09d1f2329651bb3ab3984ab591f2247e71cd44835e7a1a1b66d8595f7aef9bf39d1417d2d31ea3599d405ff4b5999a86f52f3259b452909b57937d85364d6c23deb4f14e0d9fcee9184df5994fdc11f045c025c8d561adb0e7dfd4748fd4b20f84e53322471a410cdb3fd88e48b2e7eb7ae5dae994cb5eae3eaf21cf9005db560d6d22e4d9b97d7e9e488751afcd72aa176c0fcde9316f676fd527d9c42105b851639f09ea70533d26fc60cbeb4b76ed554fc99177620b28ca6f56a716f8cb384'
	},
	TestCase{
		key: '811c3e356e7c793acf114c624dc86ace38e67bff2a60e5b2a6c20723c1b9f003'
		tag: 'c6e59044cefc43ee681c3eed872d02b3'
		msg: 'e115b304c023792448794546a2474f04294d7a616215e5dd6c40a65bb6edb508c3680b14c176c327fdfb1ee21962c0006b7deb4e5de87db21989d13c3ab0462d5d2a52ef4ca0d366ae06a314f50e3a21d9247f814037798cc5e10a63de027477decdeb8a8e0c279299272490106ddf8683126f60d35772c6dfc744b0adbfd5dcf118c4f2b06cfaf077881d733a5e643b7c46976647d1c1d3f8f6237c6218fa86fb47080b1f7966137667bd6661660c43b75b63390b514bbe491aa46b524bde1c5b7456255fb214c3f74907b7ce1cba94210b78b5e68f049fcb002b96a5d38d59df6e977d587abb42d0972d5f3ffc898b3cbec26f104255761aee1b8a232d703585dd276ee1f43c8cd7e92a993eb15107d02f59ba75f8dd1442ee37786ddb902deb88dd0ebdbf229fb25a9dca86d0ce46a278a45f5517bff2c049cc959a227dcdd3aca677e96ce84390e9b9a28e0988777331847a59f1225b027a66c1421422683dd6081af95e16f248ab03da494112449ce7bdace6c988292f95699bb5e4d9c8d250aa28a6df44c0c265156deb27e9476a0a4af44f34bdf631b4af1146afe34ea988fc953e71fc21ce60b3962313000fe46d757109281f6e55bc950200d0834ceb5c41553afd12576f3fbb9a8e05883ccc51c9a1269b6d8e9d27123dce5d0bd6db649c6fea06b4e4e9dea8d2d17709dc50ae8aa38231fd409e9580e255fe2bf59e6e1b6e310610ea4881206262be76120d6c97db969e003947f08bad8fa731f149397c47d2c964e84f090e77e19046277e18cd8917c48a776c9de627b6656203b522c60e97cc61914621c564243913ae643f1c9c9e0ad00a14f66eaa45844229ecc35abb2637317ae5d5e338c68691bea8fa1fd469b7b54d0fccd730c1284ec7e6fccdec800b8fa67e6e55ac574f1e53a65ab9764c218a404184793cc9892308e296b334c85f7097edc16927c2451c4cd7e53f239aa4f4c83241bde178f692898b1ece2dbcb19a97e64c4710326528f24b099d0b674bd614fad307d9b9440adab32117f0f15b1450277b00eb366e0260fca84c1d27e50a1116d2ce16c8f5eb212c77c1a84425744ea3195edbb54c970b77e090b644942d43fe8c4546a158bad7620217a40e34b9bb84d189eff32b20ef3f015714dbb1f150015d6eeb84cbccbd3fffa63bde89'
	},
	TestCase{
		key: 'f33691f5db2dea41e1e608af3ff39f3a6988dba204ce1b09214475ae0ea864b8'
		tag: '6e50e70411201378c8d67857d7b631d2'
		msg: '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'
	},
	TestCase{
		key: 'd115b6ac998a65b48b3dae5977abaf985258d3d1cfe1616cec3d6a77f7a75785'
		tag: 'b431c9318ec2769fc8ee8f5fc3c079c3'
		msg: '7e7eb43839a6d7616b8a7b1fb7144817904342a9bd34167051162941a6b1b85db5e587f76e4a53211755d5ab29c11822d7711a97b3f1ff5b21f2485d9c86241fb56cdd6796245d3112df11ad9a7344db44d09934c4efb280ed6580cfcafb5c97a32993cbbf4917183e0b7bb38f2ce2479c28e1d39f67396217a7010448dfd39a4e7f406c8bd2d804f993bb410fffa4eb57518a531ecf259a8af068230acb826d9ffc20ee0fc43885221a321e3928971bb28615f0d9f099f5b68a80503a910fdba0bc643c60b64837900be38770b6b30c362c4580722b5dbb1b9c8cd02a18fd7b5661d2c4d28aa941c50af6655c82669037312fbf9f1cf4adb0b9400532755011b40e8252bd0e3c7a22efb0ef91221e04b4aa8316d4a4ffeaa11909d38cc264650e7ca416835ded0953f39e29b01d3a33bba454760fb0a96d9fe50b3e42c95271e57840380d1fd39a375b3e5513a31a4b80a2dad8731d4fd1ced5ff61e1fbe8ff3ff90a277e6b5631f99f046c4c3c66158554f61af2ede73aede97e94b1d1f129aaadf9b53548553cc2304103e245b77701f134d94d2a3658f2b41108c5a519c2c8f450db027824f1c0ab94010589a4139ff521938b4f0c7bf0986585f535b6e292e5b3ded23bf81cec17c8420fe67a449e508864e4cbb7eaf335975668f013e9da70b33bd52a72094a8f03762ea7440ce9fcd10e251837cfc9ccc1a8cc470c67379f6a32f16cf70ea8c19d1a67779a9b2d2b379665e0e908a88b26e78c9f94f17acefa6d5feb70a7095e0297c53e091cf98df132a23a5ce5aa7259f1154b92e079f0b6f95d2a38aa5d62a2fd97c12ee7b085e57cc46528638defacc1e70c3aceab82a9fa04e6aa70f5fbfd19de075bee4e3aac4a87d0ad0226a463a554816f1ebac08f30f4c3a93fa85d79b92f0da06348b4f008880fac2df0f768d8f9d082f5a747afb0f62eb29c89d926de9fc4919214741d8647c67d57ac55f94751389ee466bbd44dbe186f2f38abbc61a0425613e9b6a64e6bcb45a2e2bb783b9103483643d5610a7e2dcdb10b5d78423285506b42a99b00a4fb7b619b4526bb4ec78299dd01ad894fde2f053e18c55b6047f86333f2690c2cb8e87d9834ab8a5e339aa346e4d9952ed62dc083e3b11a823a67f23fec099a033f127ebe8626a89fa1a5a6b3520aa0d215a8e7dea3af37907686c16521739a95d6c532cc259c497bf397fceaea49cd46b9ad5c1b39a36fdd2f0d2225fef1b6ca2bb73fe604646c10ba4c572ab13a26559ededc98f5a34c874cc25621e65ba4852529b5a4e9c1b2bf8e1a8f8ff05a31095b84696c6381eb9ad37ac0db184fe5fccf3554e514946a33cabe6f4d617b549d28ad1cc4642dac96e0215ee1596481600d3619e8f45e2c9ae1da834d44aca216bba0efef6254503ca90339f2d7ca508b2722d50c08def8a736590fa44855cd9eb9979c743783aa26e633696739f2ae25ff7b72ceb24dff4455b85bbd675c8cb71ad18386dc58c371bdf37b4b3875b98a9423ff3becfc0d0ba2aacab3ee7683cb3b345095fefcaca5751ca793da63c89428'
	},
	TestCase{
		key: 'f3717306b9729be998cdb2c9d856306c5ae3d89da2cdcef12f86f6110c98d873'
		tag: '907dba0f4849c7cf4570b5128b5f31d5'
		msg: '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'
	},
	TestCase{
		key: '9772c1a4b74cbf53586e5df04369b35f1fdca390565872251bc6844bc81bda88'
		tag: '68eb7fc459ecc3be819485001ab438dc'
		msg: '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'
	},
	TestCase{
		key: 'd18a328b63a1d0f34e987682fe6ca3d48b4834b4312a17e99b3d88827b8d2238'
		tag: '938b43b80cb3935e39b21dd8ba133cf8'
		msg: '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'
	},
	TestCase{
		key: '46e8eb27acfdc8f4be622d8741c7bc414464c149e21da97ab4afbf3e07b98b0e'
		tag: '56b5f49be824c7a19b19faabf0787a87'
		msg: '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'
	},
	TestCase{
		key: 'b41210e5ef845bd5a8128455c4e67b533e3e2b19dffc1fb754caa528c234d6a0'
		tag: '72c9534aec8c1d883eef899f04e1c65e'
		msg: '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'
	},
	TestCase{
		key: 'd9b0dc303188756312c12d08488c29f43a72e78714560fe476703c1d9d3e20c1'
		tag: '6b9782f2a09b59653aa448348a49291b'
		msg: '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'
	},
	TestCase{
		key: 'fb70ae7ec12264ff9f51124da188e5b11dbf53cae2671363f6054b575b1ddcc1'
		tag: 'd9ab81fab28b3be96fa3331714e78c9a'
		msg: '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'
	},
	TestCase{
		key: 'f7e7affceb80a0127d9ce2f27693f447be80efc695d2e3ee9ca37c3f1b4120f4'
		tag: '41c32ced08a16bb35ac8c23868f58ac9'
		msg: '5a3607fb98eaea52e4d642e98aa35719bfce5b7d7902950995f4a87c3dc6ad6238aadc71b7884318c2b93cd24139eed13d68773f901307a90189e2726471e4bf9e786b2e4cf144764f33c3ac3e66521f845f6f0688f09eaa227fe71033b0f74295f6ddb91fe741323f2b54f420cb9b774d4291b06219f1fb4410b55900425c5e6fcabec76a5c2424d637a1641db6f0f6cad564a36a910f49894bfd598e91f38ceea65e8253c1284f210cf7b50a96e664e562f3cc01c4fc490fa6d4679fd63fbb3ed8995a8a05166b573e92d22ef4370c6aac74ae94c94177e5f71143c6f340efceefda679ae76f6ed7f26eaa4848a8de8c40894316efbb06400f9695b18ba279e8947c032a84a40ca647d9ace4576dd0082494d6bd7be4e7928e749c78110af8774a5d43e9c9479964e2fddcee51146460eac734311225d08c60706e40f298a7cb97f369ef599be097ac3bf1c275497bbd68968a235fdf8a61bc7cfeef0fe451bb04e662ca39f34ea8e3acdd0befe9762f9eeb275c0cdd43c80fc91131d1e0e790020975ab65afbea81f303ebd86760821efb4cad7cc01fd6d6fd194ac5ffe7703d890d0169e21b444cdbaf691fc741a5d99bd47357c37785755fa72582ca4754a03b4def86ded39aa6d9eb3f38801077e6d17e3cee3fb57ae83f30c79c3cf290e2739c6b7323612cec3a561ebeadb4faa642f150323aaa9d270658c907c4c1610a5e1834730c08be3379cf1abc50c30e2bf01ce903927c27d85e1353db9e216dda8860c45925e2bb791abe5c8281ee6d16607bdca87f60662dcbd6e20224e7f009a86db66fadd8e37e0a59559328385090c6953cd20bb61f28a734fb056714f5159977f18e5c5f11de75f7a00ba807e47a29e4da32d5c67ec76ce4d7b669b5e6ee17e1df7c673dd8a7c87fce665cda8adb9547d1dccbdbe7be44846b4b121b0bfa65e4ed530789510d79bc4477e50178060f2668ac8956f39ef422ecb0e4cf90b8ce508552eedeeefa6c7d1bccc077e8088bd7e0e6aaf0bda9f11c412c270ee2ad6912f9808f9344a4bb137bdacb5b9372b00b0de026a8f5d1fb13972e1290b5005689f7636c43aee2fd44393d390371ae573f0e064b2d7df552b9adf04bf173d71c621795b9fb503dc5e918536c6ad25ce4a76f70e6b752b6d44be321187269a19bcf33ec899ca40e88b4eb23217095a85057bf95d8a54812cae4a7d32e0c2966a2137611074c6c8c3dd45a553c43c675d23308709f91be0b235d0222aa5e1e1ce08f9c6b45ceb5b47bcd7d7b2d4380bcdbd6eced452d93e6d8cbe18123277889c7f86b15fb991364a501fbf5d8244f2e3332ea0ab49e833c6f765017a4006cc7cd1a0365945a8d8873cb21832b210c83e451c01ac949de2fb0f7a420e405bf64eb251c6f022181595d68174b91e503187d3b3f49b60c23e44ea40ca20311305b413047bb22e89672758b74d6bd1a06decf09e9556421087a40c1d2c44c5fb13d4d9625581ac4ccef1a1b5eeb5689aac5c0291aebda27650daf9d4396a64d02c6d58bcbd609d9a0017880ae0cbaf02ad0f1fc8d1b3ec987ffe13102d77352690c9b761bf13ea0b3a8ebad4a0823817fcaab4d09b0bf03486620761dc77a6ba007ba07153b17425c4026597473e78863cbf430c0e5e9b04a83ad11506b61b8d9be3aeb06b5114e0d53d4724863eba124f3b974bdb0d02743520409910621cd730c97ca984fe2921c38055f83ee8c4611db92e52d8ea51d89203e89df7586c574df15f3a96ed5a10bf04cb27f9656b5b11cf35fd21360b029ab26e9a741c6b3e6357aa1a41de2cac6e85f9a49e3441e60a60e74f434e1b8cd4454b11962e5507ebf904e9d6c52a7d9722300517c434758fbd6191f4550108b143eb16c0b60094fdc29327492c18a3f36737e506fda2ae48cd48691533f525acfffb619d356bf8347a8bbb4babdc2ac866e497f192e65a694d620687cfb4f631fbd6ae5d20ac2e3a124d85f9391a240b616d829ac2adceedf8f3451ee77e4835639b13c622ef8c48a181fc7598eacb419fa438d4046aa971942c86b36eb8e16eab67105783d27fc56f5b66f35451b2a407d4648a87ae70807e45bccf14983b3abcb198d661d562dfcb00ffc569ca967171746e4e36f839946bc7d2ea9a0eda85b5a5594f6a9c1b179f7230eaa7797a6aaf8628d67fd538050cf47aa654778c11dbdc149458c1ec2233c7ca5cb172356424eb79479b6a3eed1deb9f32785282a1034ba165032b0d30733912e7cd775cdb7e0f2616b05d521dc407a2ae7dfcf46fbae30547b56f14dbb0ead11b3666666c45d345cd5dbfa200ae24d5d0b747cdc29dfe7d9029a3e8c94d205c0b78b56d5e18613b3169bd441b3c31513528fe102f9bac588c400f29c515d59bbcb0725a62c2e5bfb32b5cf291d737e67f923080f52d8a79f2324e45a3bd051bd51bac2816c501af873b27f253ef9b92ba4d7a422e2fb26a35c1e99eca605acc10d2a60369d01f52bca5850299a522b3aa126f470675fa2ec84793a31e9ac0d11beab08e2c66d989a1e1b89db8d11439ad0d0e79617eafe0160e88384f936c15eb15ece4ff00e1ba80b0f9fb7a7d6138bdf0bf48d5d2ad494deae0ccf448c4bd60f0788d3f2b76de8ad1456f7572bd0ffd27bc2836d704d95e9c0df345719dab267dd805577fafda03b834dd225ad9714d2bd182b4103faa5975180f90d5d6cac1825a19b9d4c87cc825512ae9dbeb33d2759c990905050f960cdb3eb364c15b593524c882902b2a1d7fe40ea3f54fb0202fd8821463c7e34b02a1209ba0048a9805f0468a13e03d18009318ecd92042959be263a51a407f1e660632c4247419659a4e073a8e9cd4a226763a7daea464d54272707efd053cb4efc0504602c4f63e7d247b55db2ce1c07138f585d16cec97a30731d5aec2166cb4de41695feb76280cbae1af8a2e67c2d5a3ac5487ffe8640f308ace6137e83576b79d586b663122221c20aba7a6bf60f73958f43659f087f850ba6e2d7fd862249c5fa6b20e3e43d4f2aa10d4c9cebfcbdf026b8d103e4f89b93dd8af172f421001c8b162bd6d0b847a58ac108b6d6cc49c7a9ba069deee'
	},
	TestCase{
		key: 'e3d21f9674f72ae65661aebe726a8a6496dd3cc4b3319f797e75ccbc98125caa'
		tag: '3c95668130de728d24f7bca0c91588bc'
		msg: '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'
	},
	TestCase{
		key: '84373472e362a356bd5c9b50f55c588d067b939009944f02564f136c62dac36b'
		tag: '12dd5297cfcec53deae1dd5f9325d894'
		msg: '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'
	},
	TestCase{
		key: '2d0035a30d19b9cbc7a27561f3ab474c01115c4499b4adec660ea06ebaa1a14c'
		tag: 'a2c77b55cb0c076d8ea83cfe0e64f293'
		msg: '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'
	},
	TestCase{
		key: '2f29d71d73f7af98f96b34e939e1a21e2789ec6271b878bbebd14d7942d30080'
		tag: 'ec02f4953a9a63ab6f2bfc3501e4fab8'
		msg: '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'
	},
	TestCase{
		key: '1285f117bd90b70ef078ae62f37d2218419e894b7d334759ddb2d88833b287b5'
		tag: '429b2b39195a10357043c9601590a277'
		msg: '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'
	},
	TestCase{
		key: '491ebd0dddefc9f0117176772f9bab61b92a1f1de13796176091c56d1e53dfbe'
		tag: 'fbd3f884a3dc2a8be06ce03883282e1e'
		msg: '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'
	},
	TestCase{
		key: 'b41db44465a0f0d70093f0303bbd7776017bca8461c92116595ae89f1da1e95f'
		tag: 'd8a111a09db22b841fa28367ce35438b'
		msg: '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'
	},
	TestCase{
		key: 'b78d5b3019688e6ef5980c17d28d7f543ca5b8f9f360f805ee459717ca0d85a1'
		tag: 'f01babc4901e957d0c2032a7279321e1'
		msg: 'ba7d35b2ef8af1118bce1e78018c9314b0c8c320591e103d23f715acb05edc98fbc618de06627661df5842dbba9f604c2d20d664e5db06e949b11d49665088dbafdb0d39d20beaca7d723f8dcdc57e9c5583d303b6cdfdbecf957d8daf2f1c72b2a6fa27e3d18841f4841abafd334c110cd2b74efb6191dbab9b8fc8427ee17664082f31db98d30bf15dda967e20730a9ef525abe9f3f620e559ed22bf74d347c9869f0311f33da7f1a3dc858b3a8aa73a35989db055a4a2c269c95e352259c57de8b94d8de48984ecde426d3ef60ec1c7b441cc950f7764f55bd0cf52d069b9ad446d1f765f35d02ec104ffcc00bf1edc1b951ef953acd19984ff1b41041bea0e9f5326a7c9ed97e6aab42174ee971ea1dbe2fd1c1f67f977ab215962b0195417170f6b7748fd57262424d6cf7c235b34425f4047191232722932213b3eb73904cadd6a2e9c7571d7c66c2f705b5039ff75e5e71c5aa738bf4177653e6eb0b49303a4bc0e641e912691f217296a3325431d578d615afddf47784e4618a2ca40ccecb05d621da52f272b8cf84f7fd8177c83af1580d25a764cc06436d67171cb5d1e3b39367b46d9a59d849d87ab6bfcf3fb9bac2b1ebfcd1cef4459e74b0e1b7080dabd2dea79f75581a55de63c4b23ff67d986ad060102933fc6cce8d614c9c86dc84068828dd9e21ffc5665c809d83b09432fd315dfce5d7a4ebd8143181953e3f8716e47b0b30cc1f753e31a7d509f2dbd4177b6da310cf3cd025db270adf98e96259a5ae1b81f5be4d5c76f502a612ca73c76b91e0ca695aa921f9489948619482c2956205ae71fffc3aba4476ff754e4878e36c7632c935c076857c5b90cd63ea4764efbcee53e2ddc9bdce54b1cbbcf0e7544d023e7c2b79419ad92221a1f76abe31a8236e370d38e2493cc9ca2aaa81130fc713d11f500fd071d6eba6861e8b0859b372e62fe60b627a96c377f66236aedf307e1d148a61bdad072b93d7d2a73367c595b1e048f7023e727291ec508326f5424a5bbf4e010d0240b71fa9137e6642ab40c5e4fff79877db3253c663a221b49b3e77ea307c7b9f3f72a0f3a54d0112c45c64a0c0034baf2b55ae36ea6f811bbb480cee663136474dacac174c73b1e8be817916cfd4eb1876582bb3a36cfbabad91776aa676305ddf568a86e3a5eb687fa8167771fca7b5ca00e974b3cc3e322b4bd9bcee2a87d0ae7976da5e04fa18c219fa988d4f6fce62f194b05c26ed3ae1b066cd9751a2d916d53426a454d58f9c3b2fb49374e5791b412fdee1b6029144f1ca787f56fece4f64f4facbfe4cfd8ba7c807a83cf44008fe5126a283ab2631a87acd8e2a3bd10979c4b07a84a49b0687a45a4798ded0b5e9b2acce30e714d78395bfa8f33ca91e68b2138bd67d8a694cd87c88dcefcd101a3b408d7a9095cc6a4b38898ecc8b375f5a67deaaf73eb7e99b10314ca6bba824658bee85dd731d9a1475f976b7c0aed4b67b088f0db5ca5091273217f724969dff6cf184181377c455722beb23fd9d097a82ea2d8d527ba6284acc20cb30f2e52af28800c61fd1faf9f4f619550e0162a1a63758e202533889b27420fe7d0eac9a47a6e111d80054412340e0426cdddbb3c7b9b823b8db3ef58230fad7a3ac21a7805d30878d4ea78dda95c951b7a5dc552e9434c35e03e1dd88652d3714f8fbea39936cc0717c2e0335371f2a751204f5d9386baaec853f019325edfd1b0719d1fdac3fbd774a64bf957fc54039501f66df94b5b9b82c2076c597065dfcfe58b2e215a3734066aeb685ef97759c704b5f32dd672ba59b74806cfad5daeeb98d16f7332ff0ca713d541c84e4aef0750bab7477ea707e2e497e12882dbc0765106070ec6a722d08fe5c84a677817b28fa3a41a6117f2f5465c2a2f0eb2b8be4f36e676b4115008bade3573c86cfb1370c03b6b0dc4bbbb0ada4dedac10a593655068a26febc2bf10d869cac84e046c9c846ce7927431f606f07b92abdfd81260199ae05ed01dfa07088c56a6a8de9c6d51d61d6a6d3f9904c216ea8329467a006a3d2495a768a39ef99a21827d2def909bb743fed7209f7fe59ff1c1e710095b05f166c6173deef5c6ec4105c5fc3b87c8269c786bebd999af4acbf12d20453b125f338aee87e9509ee4059c9e568e336304d7be9ffe81d1700555b0800242d9b7450d7256f2b17f6ed46a39f67bb2980572ce73169e352070dbafd4c7fa5a6be78cf9b72981c0a01f1e1e30ee3736c59828b791d2373799854497a28a44bbe0e0749257234986696fbb06ef9ea83fbd49c45a583ce12ff10258ba06127c67b0f66dd109f1366d8036853973d8884f93de54fb2a12949eefc020717eff47898cef306b5de068411f1e113ffdfe2556e0faedc3e27d95a45b8afc15ba0eeeffeb86da7b4324e20af80c62bf0ceb4aee1515f5912f71c6bf2febf20123e3dd3a82dc1e58a108f1039942dcdacdeb1f0ad0b2ef34488d98d6a52311aeacbd03c12f6e775e375d5979c7c295bb049f2cfd3580e3da3841ddd8e6af4de5e6512ca79cebcab9280554524881da37984d340e8f0163fe10a02ed088682560bc6d3c4dbcf1a542ffb3dcc2ed16a2eb96896e8269697ffeb50b73f2cc354092e782a0072fc12e1eaff117c2cc8a5a1ad8b47802ac9e23fb91a0cef9e4027595e0885464e61563093ee2b1dc5f22dfd04af7de6a70d5977d3751a4b3cc0c71a71c59c0534cb1f8c0eeddcf1c0e1b3e5ad0d083b66e8b998ddf9ae9d3b365c851d42e995b9afdf8d66b2ac40bf514ce32e4560880afd38c42c08926067eb243c4b1184e667ba756c14ace5f525eb48df7ebb429d0a23d159664f8021d27dc7167081de331c7114c9c6456e1ffdb422172a81c06d8deca995e158c48df27261a83f83e0127f5e056a139be9b76e25dadf534d3d1ed6ebc0b5d77d51e5b90ff86f30d4023066115bc11b33cc827b1103098826d0bf8777176b2da6f1e5b580e407ccf7e614fdf4f5b533ef6d30b20c1bee61eab90e983b1a97173a62720ffd27abb8976a948d532d06596c23b0ef31c79831bead8f8e99ad209af3658cac0cb3c3f9c88379b9bc871d8e84171d53400902da1243f664afeaff60bd96ba2639a7644676ca79f43130af12ba2c877d67f7ec030a4217a72f5368af7c9f24e643db6ac97a04adaf57dbc53762d8dfa1afd49667c4041adcb5ec303e191b786273bbb065cd9f16a3a4a399c6a7aab9c1a6604998264e8b3dbd13d8f2228b13b2c2b9fec5055d8e9f2df1d9a25e4bfe2029776389877bbef7e2c7621f06bc0b7fc0786e2b2d042483ccd4a59d2872a6c5ac73e217123e5c8401580a8d967e0895aaa28f4d25ce68c90b4394d8113bc423e9fae46ac47bc2ac191fb97b80b5a85feb2bb54f84c493235c1408662fe253c6786fcf6fdb8be87dc66a72cc847f94dfb5214af5905b7039a7363a1b23a07853daa26862783ba08a80846fbb93ce98700a4f9961115128dd67bd7d19e0c588fdf6196c11cb0154002ae862f11421f5dc3a57b6c0870b452272be556a1d14eab1af0a91ff5b89de6bbeed6e03bc64f5efddf9e54da71c594bc5ef78e0192cfdeda36e4ad1a6b0b51110c1b24d20dea1f19e18cb1184d80189f842d4f07ac834744dd009aa3771b1e5502fe4b65a403a4bb319e1880ff6ba852e90a8f4fcb52cf374c88408428cdb1255291b04ed58c992310955198d61fa1fd9d762d48f2f65a287773efc67d549981c291b427889d3e3dfc0cc6cd68415cdbed81b516786dacf431472a7dfc99688d15bb6c1b85b1a2015a106e5de8cb9eec4c80b17d00fdcf4a9c64de4643a95dade8fa9f1bc5c839037d86c13800a244188e3b18561a74912ed72f99f2365f0126732d037dd54a3ab77f9a9f6a1c1469ea92eb707482066bd4990dec4d7614ccb4ea6dd4deb8bee22c4dc0b9b4d4cc70a500d2c8a5ac3ef88a38439b7dc254a6d920cfd317a84d7747148c65b6730709e43369d4c995b03c58b9df444f77f216944e70f66446554d8d513b8f7f28ef0a2d7ad5ca2f6110304196953247a7ac184f6861fba896c2d5a59007ec2b2c8e263957e54cdc1f3b4a145228823fdf0960c33a28f59b03ee4be21001d2f56fd49ed14db33b2c4eec2c3f41b250a62499a9b6602c1e838526a54cdcd058af1c252d56009d4c7769deace53bdb66543f5a081cdde775e61efa70956fe2a7a6019a164c6e413ded314bc928b4aebccb946ffdf3eb33e187bf421febe26112b3262a526de65678cd1fa03b83513705108fe0bb87aa99aceb28af3641c46a2c4427cc1063de01aedaeafba68155d4de494a27ff6b7fcc8f5c5c3f7d3a115c397a1a295bc55aec8f7f150cbce2a8aa4706d54ec863877bb966ad441c57e612a1b5d438b98d9efcdfe6d4f66e885f96407e038015cf974ae5a3540692b054d2ddfde59b28ede7e2f581eeb56c5b88e2779aea60c1d8ca6107b0cdda1ac93e6c7520daedc66afeed12f980e20e1e1c327d15ade4bb90de30b011a9cb33855ca3cae2'
	},
	TestCase{
		key: '2b0b0fd3347e73c2fa3a9234e2787e690a11aec97a1c6d555ff7b4047b36f372'
		tag: '81b1a6633f849ab0aa7baafa58a5d9b8'
		msg: '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'
	},
	TestCase{
		key: 'aa3a83a6843cec16ab9a02db3725654cb177e55ec9c0c4abd03ada0fbafca99a'
		tag: '719dbe5a028d634398ce98e6702a164b'
		msg: '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'
	},
	TestCase{
		key: '1793bfda9c8666f0839b4b983776735a927bdaa3da99b13c9f3d1cc57d4d6b03'
		tag: 'bc89cfec34ab2f4f2d5308b8c1a5e70a'
		msg: 'a09f661aa125471417d88912f0a4a14115df9a3a19c1de184878291acb0e89ee1f9d8213f62df442f8969a9a5a7c402fea09bdbe236fb832544e1f939cdd4873802b2bb8fc35ba06b7ff96da6dc7efddfeeda84116bc525a7fc52d84d2e63cbac00b122dc64f2d15b36595259d81a1d2a09f204c54072751dd812259df1104bb2d2ee58baee917c5d0aa2649c8a1503114501e6ed6fe239847d3d88dccd63d5f842426b600079c6bf06e80a2813b2208181163b861dca07fa4d88254e84dac1c78c38397a016b5ad55a6b58878f99036db5689871ab3c321f6ed5895f218f8fd976c348b3f1269fcdf4d38c9492b47216c45f499f5705830b33114d721f9731acf6c69fca681b74c2d82c92e145b7bab77110821d3a12cc818d7595a5c60c4b5e5219376c38a4dd52d435d41562802ff65ba2bba5c331c333d5adf194d29b2cd9ebb55927bb4ec17681a3f5574ad34fb4e964f2c756f6dbbb7a6876a21579a515263444de7a30a3315005458bc137ccfdff18a3892fc9f58f1de10d4de20bbcf860f5f036d8e8a188f18e5cf7ea3cd260710e7491befcb131d49a28dfb1ef688fd021a1ee4420d32fbfb03b47f5e85c37d91e49a1b0db85d966eb5434c4197433eb49d56f2ff999c9a72230447032dc949202468261b48b6ac212e3f651d6c6303a06c90bb2d3a755ed91ba73bcdc28e1c5b0936e51e0a9f69c3ebabd3dbadd7abab6d8f6a44daeb3126429a01815f57444fb7022a4a510f8b564ae2dd9779b3a273fef15859a33e233724846c30d89fb78a595b6ff6c834812c00a991e405806aafd0c26a788895ad00a5e43c5426197aa8247207077548ee67db4cd6f878431a2e36e952d84b5fb89d681f553198e2c066310ea6ac3a31f5b1792620616f6c41d486fb844eeacc7fd36971abf416e8d6d50985c83cc92ea46ac37da8f0026aba30c945d8bb15080d2d95e4081bad6261993f95f57ed3252822a7caa035ae22a36c35e280cbbc82d729346cacdb1794ae9a9bb2793fd1d5c47121b135c2836063367339c5151b4e35278e97f62afdd2f231d4b47812d083a829ebb9c374ff2ae8479cc4b76d55f9cef3ec6c4894f53e8caaeb0d8cd072960cedaf758e48e3640590d4f728626e0a08eeebf719c96bf8ed4d0c283be09c0ae67b609e22d3b9aa6b03642854909de05ed52b39673867bf586a632ab8072de15c637cc212cba8387515c9c9c433abd7ba6b02abd09da06a34694ad34f88515b65c0c9c247fdf9819fb05a1aea4728c1182f8a08a64b7581cd0fb2131265edcb3d4874b009aede0e87ed463a2e4392aefd55e008eb7ba931788262f56e53193122a3555d4c08133b66020154b15643fa7f4f5e9f17621d350ede3dc70be02c59e40fea74dbbd7919d1a8d4e22ef07c916fa65e7d4b89fb11a7c24ddc4ca5f43344c753b61331c3fa4558738ba7832b5b2a275bc9b7989b6e6888865793329806cd3bf0ba57c941d4428623e062f4ac05e7cd79ad5446f8838f2b247b66bddadf540845a1bb304a04b7edbbff579c8d37e2f6718f8690abd5231822c7e56569365ce532449a41ae963ec23a2a75e88307dc6b59cbb3fab913e43ed74d841ca9f6e4ef96dfd9f04e29e89361aece439c0b2e1943b30410a63d495c522ac3ec1b04ec4cb345f7f86969957ad750e5bd7dbf1d6a22eed02f70b81cb5b2b020c0694d7f63044f9de0c3de1ede52009c858992d01ebb92ff19a9e0fbea18942fbafb77746c8e9e687dd58ccc569e767528bde43b62c7c1270a5721f1212de2b29a7aae2d6ba6cd173d7fbc78aec4356ce2e8ba9164d97dec061dd0c3a0e3c520a7611ac99739049dd5825537c70b7ef660046c1785546cd99aa400da848eb7c3c91247415c8e245d0f14c30d482c5849aeaaeab2568288229b08267818dae8f76fc674c684c99eb5faf88a0783813df7298e0b50cb233f78471e5ca9cc3b04927c26a3871cf253798cc49aa717d8f18a1ddcbdc26497d188f15f86ec494dcf8f942c3e07e572385c6fa0ef40c0b625f1737543074a747a369482a0b342a08b3eccac9f9209be31aefe5a7794974f71ac0bc9a58026397ea3dd4f5e40511d58d2a3b45925c194ef13987037d736dd48b509d003a86471d5f161e0e5dd168b4f1ce32f703b8915004d8dfc708a5bb02b2e6fb67424b2cbcb31ddaa0114c4016b0917382daad11815ff5b6e37d5af48daa5ef67cee3439283712bc51b5adf2356cb2a5181b8941fd78945c7c9d61497683e44fee456ad345e12b4258f15945d45b6ca4369ee792d849112d583fdb39cd4d333ee057355f0abc8d1eea4640c128cc1617982db0394233dbd416102eec1874081247d2982bbf9fed1b1b38f4da923d68c8975c698f189a4d7840fd7aca9dceb7d91c076f85e1c546f4d5de4f60c91348455aaea30cac134c844dad93d583c139dd52b3be6346c4d2e6864125c5a2d0aed8f67930e1ebf8700ca88aacc914ea76ff17148f0777738cc126e75a2c81110faf02fefc47c91edbab7814599000ce55fe20ef313566e9b62457acf2f22e1141e220bd9d4747417d03e703d4e39282803386327fc65dd597f723ee28185c78d9195fc70a75706c36287ab9c6e00e85cecbbd6043c6af8d30df6cdd8777be0686853b7c8a55a5b1e03e4431d391725ff99875a85cae6926998723b36d13ad458220712209bfc5e8d2ca5d44ea044d5ba846b4035e7ac7e9885f55d3f85c0c1b3d09fe929a74450f5d29c9672e42d3f59be4ca9d864a4322cc454c2578493bd498a51bbe960e6573e5dd02c4a3a386d4f29e4578a39e9184024cd28d0e86ecac893b8e271bfce3f944d130817378c74d471bd20a4086f2429ed66c5c99969fd8da358ff5c3be72bf356ae49a385aa0a631b588ddb63628fd162673e915cfc4de56eae6ff7101df3b33125c9bab95928f6e61c60039b6cc07a66f9c733251447ef9c1ffefa2158a8ddf89dc08686a4cf9b86ea09914e79842d72a3236afc98a3afa0a1cac5590ab6a923e35a2ab8db6410a9d33cb84d1c48a054377e549774b25f50fbb343ecd5db095155cce9fb0c77d09752f62d4bbf16a77030452a75f6bdf73f7807d8f3a6bae16ad06b22175fee60549c22548de9c13df35ef4e7bf7b66491a62b93c2c3fb0c5edc51f60f5704b56af30f1079d7c385b99f958ef8209e030e381d1ee8d67d3cb84f32e030e8ea2c1d0c77fd6b242a9f48707557c8682a08e1127f51221a55c733ab1edd00a9c2912cb36dde85f73b524e1a4f4da6414c5e4c18d9537722b2becc8a91bcc63f2b09f32409c53c2beee0de6726dabcd6bf33118a5c23fb9c5c1810476efe6584bb6109c516b45e16b2f79f96755680374d82b91f2c519639a1815fd485ba3c00b46fbefeafcf25554ec5a6a5ae2da07c85b8a0f9fcde50263d9ed85038b2f7aadb9de765655bd201235218bfc74bcad6a9ddf4506167a649afadf400b85752d68a92b7a97f26b334dd77fce824862046b286a7c8e0adc36f713a252a673d4d995b268badf4bec8b8eefe85c25b823b6728582d35c4a60041114dab72b0623b99e2758f6a1e97365279bfba0eb1fc8952ca4f2c6fbffd9f5fd7dcad1125b18a796981b5ead0b6431141315898ace96f0d38f865698df8822ca7b65644b6b1f0a0f0d2e5850d4c93ec48ca3eba1b919e24413a46d595ffa427715e499db3b7b9ab53c64abec7302bc737a5bd124bcda756abbca132f7f67e6989e09bfb23b497da31bf156bb9c69ae54588df17420e8fe989f0472c8893b2bfe57cdae265a8cc7aeb39624167a567a6fbebb1aa30c3dcfd14f2808a070994085e6e1fa79021e77c399f90ab1f995a7baff672cb693bd39b798b4c890b7d0a57978d6b9bcdc5bf3f4d205f8f24b2b43d3ae300a96971c9182be297618b9adceebedba1ab0f324b01d23d7e635f009db3dbbc643c2d787567594bc639bfd78c4f3e6d948caf06f013423eb3c764666b58f886d5d28137c053c2a28535efcea400147e92ac67535743b47f9cb48852abed1d057647d5b1c6f334eab1a813401fccd3dae332738776bb223e359f3c459b5c573ba64fa945bdd66c5ac0fcbd53b67032a7b8025f551e8d1fd2a4291bdb7941cbabe3a09765dc263e2bbb6db7077cc8fe6790d4bed5e36bd976d1e37dfdba36aafcdaa10c5f3ed51ba973379bcb8fd203d8b7282abbd271ecf947e54486e8653b7712c9df996a8ad035f41f29cab81509f922c67dacb03f25f8f120cb1365ab3c1c286849c2722448ba9bcff42a6b8a7a52f2c79b2bfcbdd22ef8a5651c18879a9575dac35f57d8107d6bece37b15d7dfff480c01f4461ef11f22228792accda4f7936d29d4c56cbba103b6d3e6db86e39e5f1bb9e9fd955df65b8a6e44a148620f02b5b90b2be9e5bb526d0ec75b1e723e94da933a356d7ca42d0ce8349699f730b8e59bac24a6b633759c88041d29399ce60a2ca2261c7eec1acb9a56e0e65bde37653ce2cf7eb83a4d019c755bdc5d685b6394ecddb9006823182dd8138a1bf79a32d07a8e5e8ab221995c714e571b40bb255b79e328ab883542c164899fffa16eb3296f310e302512352a864fd809beaab4169113027c6ccca99a92c6ce35c30f9449a3add70f10db1ed08078e8e6cbaafef630aab7e9fc8adb09c18e33fe1af3620d1e4d069ac11325e23cc18e5519a1ed249caf8ddba871c701f1287cc160019766988f63e089bd9bf1af7e6f5b9002e3b6c264d69a8bac16914ab55c418d3a8e974677cdcbea36c912e90386a839a3777b878e680c07c7cc99f42a7dd71924babf7fb0627d1f2cc60d9d390d1e150d47386be6eefec9ddbb83b28fa7e2fd28cc3867cbe42d13b00545af8a048cc07016ec79808b180e0b258c564739185da754f2e'
	},
	TestCase{
		key: '0d41cb4ac25217feb20e86fc2490e8d2ea2e8225c051252a9395cc4f56e1ae5a'
		tag: '42df9f9a59d6dc05c98fd9e9577f7176'
		msg: '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'
	},
	TestCase{
		key: 'ddbd5d6c5ebd61fa72b453dd849dc302c98a0f3e300f4768bf1dc698a3827dd2'
		tag: 'af608b71a353e63c64911558baa122f3'
		msg: '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'
	},
	TestCase{
		key: 'b15578da1020f662ada0ad4f33a180d9f8ad4991b3720bc42a22b52625c7414a'
		tag: 'b0e4ad4a010afd6dd41ed82868cda555'
		msg: '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'
	},
	TestCase{
		key: '392468efccff36dade31fc1c62eb38bb61394fe448def9d9d9beec2413ddb418'
		tag: 'e1122e7c8e6965b90addbd46d8a548d6'
		msg: '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'
	},
	// Override initial state to ensure large h (subject to h < 2(2¹³⁰ - 5)) is
	// deserialized from the state correctly.
	// THis test disabled, because of its different internal state representation.
	TestCase{
		key:   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'
		state: '0000000000000007fffffffffffffffffffffffffffffff5' // 2(2¹³⁰ - 5) - 1
		msg:   ''
		tag:   'f9ffffffffffffffffffffffffffffff'
	},
	TestCase{
		key:   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'
		state: '000000000000000700000000000000000000000000000000' // 2¹³⁰
		msg:   ''
		tag:   '04000000000000000000000000000000'
	},
	TestCase{
		key:   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'
		state: '0000000000000007fffffffffffffffffffffffffffffff5' // 2(2¹³⁰ - 5) - 1
		msg:   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'
		tag:   '1b000e5e5dfe8f5c4da11dd17b7654e7'
	},
	TestCase{
		key:   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'
		state: '000000000000000700000000000000000000000000000001' // 2¹³⁰
		msg:   'ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff'
		tag:   '380859a4a5860b0e0967edfd711d37de'
	},
]
