// These tests are derived from the Secure Hash Algorithm Validation System
// test vectors contained in:
// https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/shs/shabytetestvectors.zip
//
// For SHA512_224, the test vectors come from:
//     SHA512_224ShortMsg.rsp
import crypto.sha512
import encoding.hex

// This structure deals with short message tests
struct SHA512_224TestCase {
	name    string
	message string
	digest  string
}

// short message test cases
const short_cases = [
	SHA512_224TestCase{
		name:    'test case 0'
		message: ''
		digest:  '6ed0dd02806fa89e25de060c19d3ac86cabb87d6a0ddd05c333b84f4'
	},
	SHA512_224TestCase{
		name:    'test case 1'
		message: 'cf'
		digest:  '4199239e87d47b6feda016802bf367fb6e8b5655eff6225cb2668f4a'
	},
	SHA512_224TestCase{
		name:    'test case 2'
		message: 'ca2d'
		digest:  '392b99b593b85e147f031986c2a9edfdb4ffd9f24c77c452d339c9fc'
	},
	SHA512_224TestCase{
		name:    'test case 3'
		message: '497604'
		digest:  'a9c345d58a959af20a42c84e28523ba47e3bf8fad8e8c3f32b7a72ae'
	},
	SHA512_224TestCase{
		name:    'test case 4'
		message: 'c2eb0c88'
		digest:  'c5b97f1e46d91c3940be8cfef1e49baa77b475db943e49f00cddbe72'
	},
	SHA512_224TestCase{
		name:    'test case 5'
		message: 'd28bc2aa3f'
		digest:  'e79de552120a87585e7313e24ecd74139a3be9ba2812936c4b33ca23'
	},
	SHA512_224TestCase{
		name:    'test case 6'
		message: 'b4142ca79f6d'
		digest:  '2a0ed8eb0e18bc189f5c836dc6a5825baa6f7286fa5b9689b641bb3b'
	},
	SHA512_224TestCase{
		name:    'test case 7'
		message: '8daedf3c420c8c'
		digest:  '4edac91a8eba35a1431ef4d49705db12da382a786d15636774cbd708'
	},
	SHA512_224TestCase{
		name:    'test case 8'
		message: '6963446913771410'
		digest:  '21f6c373637e6a5e89d6e88811110c5c3fa12e497144912914c546e1'
	},
	SHA512_224TestCase{
		name:    'test case 9'
		message: '6f316ab2b40f096819'
		digest:  '5899542e9a78d1403929c4346993e213b83c32f91274bddbc63e4192'
	},
	SHA512_224TestCase{
		name:    'test case 10'
		message: '5819a77fa1a8f9c35d17'
		digest:  '2e944a5c534b80922f6c21e97277b4a01a4a0fa652d6d40b37fb0fc9'
	},
	SHA512_224TestCase{
		name:    'test case 11'
		message: '3c7e038401fa74c6c06e41'
		digest:  'a74af68819afe81bcdaceba64201c0d41f843e4b08e4002a375be761'
	},
	SHA512_224TestCase{
		name:    'test case 12'
		message: 'e243b9f901855447fa2af302'
		digest:  '330c77ed20dc93cabf0df3442c620c9df5c7a02d7c79fc1d2ff7ff94'
	},
	SHA512_224TestCase{
		name:    'test case 13'
		message: 'c5f942edc46407c6019cd926a0'
		digest:  'e180a0d83683cb7e79d812ab5d4288399c78895c6a918e99a2839155'
	},
	SHA512_224TestCase{
		name:    'test case 14'
		message: 'e57d4c478ad6a64b8bf3352e75b6'
		digest:  '18b3e8f5a1fb603c67060c12a736e57c5ccc0b89fc62c90b02dd926b'
	},
	SHA512_224TestCase{
		name:    'test case 15'
		message: '32d049928a583b5b954785805d8b26'
		digest:  '18f87e8e6b54ec4336442f76eda0b32dcb64d463c0be69e348f86c34'
	},
	SHA512_224TestCase{
		name:    'test case 16'
		message: '3b2542cb3fa560dd0d796d59ee151fee'
		digest:  'dc4ab34db241b3fc2055c54f70a8fd304523d8c4202852f5f077644f'
	},
	SHA512_224TestCase{
		name:    'test case 17'
		message: 'ecbe367959a91b15eb08048b33752ef76d'
		digest:  'b76eaa7dde45ed50c07b5bafafbfdcc0959c1a64eca7adc391aba245'
	},
	SHA512_224TestCase{
		name:    'test case 18'
		message: '40bd7d47b636c2a749a247fdda75807c238b'
		digest:  '087ed68f1db90ffb2fb4ff7dc4b17fe08100b64383850378ef543339'
	},
	SHA512_224TestCase{
		name:    'test case 19'
		message: '331c8939ef685ae0efcd4030378a8b0ab720c7'
		digest:  '97874d56296a28f8df280448f76a9e806605da257a6083598f232e4d'
	},
	SHA512_224TestCase{
		name:    'test case 20'
		message: '26634d1eb3cc2793830c97f49674bda74743eb8f'
		digest:  '8294b112588dfb9c553ac88fbd8441722cec6a1d435311fe72e0110c'
	},
	SHA512_224TestCase{
		name:    'test case 21'
		message: '62ed23ab88c30fe1350f702f3b6d45253a0fc6aec1'
		digest:  '309db70638771e2eec309abfda219e5f13bac5fa9d5e84e727764bca'
	},
	SHA512_224TestCase{
		name:    'test case 22'
		message: '827dbbe36fb2663c95eaf81c7ce441dca66405050ad8'
		digest:  '39e786a93dc90e96471e466950a5a0f7f142ca69c1251b932e8bbf45'
	},
	SHA512_224TestCase{
		name:    'test case 23'
		message: 'a2ada61a95cf8fb6f93cbd0aa67b70e286dc853afa51f8'
		digest:  'e821247c7bc7ac8ef07fb31ac25fb48a3fbc50fde399a3cb5889545f'
	},
	SHA512_224TestCase{
		name:    'test case 24'
		message: '5633ee7b8870002c01a3c6f68f7812f13adfe0bf6fa794bf'
		digest:  'c34b467858b65202fd8c73ffac67cd8ee55118d12beb2b110b63ae43'
	},
	SHA512_224TestCase{
		name:    'test case 25'
		message: 'da1dd1ebea3cc9474cf255dc23e35695026e86d6093abab251'
		digest:  'e650802f29ad20b900148cf3e9361399e6207d455257fbae3efee50c'
	},
	SHA512_224TestCase{
		name:    'test case 26'
		message: '8f71c9344ce6a535d104803d22119f57b5f9477e253817a52afa'
		digest:  '6b3a288bd84137cd7600ccf2d2ab9b797d0e87a18ccb468183396235'
	},
	SHA512_224TestCase{
		name:    'test case 27'
		message: 'd668a21780594024db025dcb373ef3639bcd5743145527e24da67d'
		digest:  '2b93446d3aa2f1f9ea858e47806cc472f40e33085b7c7a6522cf6548'
	},
	SHA512_224TestCase{
		name:    'test case 28'
		message: '03f8cf98035eac158ff18fd93b44cdb621597255d7c3d97a9e087989'
		digest:  'e1fc5f508deed8bf74883b6fdef8c3e0c86306e7f1d6300cce4d36dc'
	},
	SHA512_224TestCase{
		name:    'test case 29'
		message: '2a1f57998360e9216f5040e8f75f5bffba4368eeedede54aa0bb058a43'
		digest:  'bdc91e679fc2f4e40af13f03eca3f61a7ba33ac2c613d948ed7fc0bc'
	},
	SHA512_224TestCase{
		name:    'test case 30'
		message: 'cd5fee5fde5e9aa2884b4f4882cfa7d5571f8fd572c5f9bf77a3d21fda35'
		digest:  '72b43417b071f4811833027731b0ca28549c0357530fe258ca00533e'
	},
	SHA512_224TestCase{
		name:    'test case 31'
		message: 'a9a73038a28e40feb374819df3a26db846fda284de7e88db01e2348c0ca38c'
		digest:  'a5baddceed5b58614c95b8a621d58cd4b12559bd34795ef5be9cd603'
	},
	SHA512_224TestCase{
		name:    'test case 32'
		message: 'be4850f5be6cb68cdba5948e3b94dbbce82989aa75b97073e55139aac849a894'
		digest:  'c11eedaadc5249cefafc5a9fadf39e812db700bd2767b8de1cdd6099'
	},
	SHA512_224TestCase{
		name:    'test case 33'
		message: 'ced78bf0734a527f5c07ef6c8adf80ae252b8cff100d393ec1f8c01150721fcecd'
		digest:  'd4237aa1956d37a6dcab7b07269f18228f7e3156c3555cf4d3d68c1a'
	},
	SHA512_224TestCase{
		name:    'test case 34'
		message: 'e400bc8994cff23eae189df3b1e7e440a6b5b1786197128105b5dfe2c6fe9d1ef31a'
		digest:  '45351adffefcfb5e39085461f8bff5f5652a5aa5a0401f04d4f56452'
	},
	SHA512_224TestCase{
		name:    'test case 35'
		message: 'f15284a11c61e129ea0606bd6531f2f1213776e01e253d1def530bed1c3c42b3c68caa'
		digest:  'fa425bc732d6033566c073560b2c5fe322aa4fa22aaa3ec51154ffd8'
	},
	SHA512_224TestCase{
		name:    'test case 36'
		message: 'd858eed942fe926b52bada623fb8d0f18458e4163332e1c18c581f38cedafbba4d908462'
		digest:  'b2b308db948d7169c8eebb4d7a19be96d5c4b5849593ec587d3865e6'
	},
	SHA512_224TestCase{
		name:    'test case 37'
		message: '78b8ad2372632398cf56717df3b141a4f38d5d83029bc1f02d07f52949a39917294141d414'
		digest:  '71be05862b44e9f57ad9be17f1de185471eabd111c1725e835e09625'
	},
	SHA512_224TestCase{
		name:    'test case 38'
		message: 'faff894c4a2ed70c4c9d6703b6bc3a3d2b2cba8f04769de350fc0cd01930056eea03fe688da1'
		digest:  '59f5a8c843b982989a08e6852e34f0c03397c99a87864dfb1bbafe1c'
	},
	SHA512_224TestCase{
		name:    'test case 39'
		message: 'e6065e1eb7cf901c3b08dba4aeca7d4673c3fdbe60391c1b5d4d09202d7dd4ea8b6d25d379e72e'
		digest:  'ea8564c988cb67e302bd40df21a5f796a0a116846493c86c3a1d080e'
	},
	SHA512_224TestCase{
		name:    'test case 40'
		message: '029d14c5a4a57a649a86c5f2e022e6ee098f29c9d4606548ff2775296038ec0931af0585acb4d9a0'
		digest:  '716668cba7b1c0ddc1c4b3e6fe3f8917eada62506b6855e5af2b54e5'
	},
	SHA512_224TestCase{
		name:    'test case 41'
		message: 'e3089c05ce1549c47a97785d82474c73cb096a27c5205de8ed9e3a8c971f7fa0eab741fd2c29879c40'
		digest:  '75a0fa978c45d268124d8cd9ef0a08ecabbbed53412cfc7cb1c00398'
	},
	SHA512_224TestCase{
		name:    'test case 42'
		message: '901db0f70f2d10626a8af408c5d15196693d49382178ad0de3b1c1d8b3ef3ff8c5a59e9ae286932d0c9a'
		digest:  '1470a4199fe4c7bc330760c1197e79457b1644764e28a62d386a78b4'
	},
	SHA512_224TestCase{
		name:    'test case 43'
		message: '0431a7bfbbec1bb8116a62e1db7e1346862d31ad5110ff1bb9fa169a35dbb43a24e4575604ec8b18e41300'
		digest:  '1e7590e408c038b794e9820b25d011c262062b96d111dccc46dc6783'
	},
	SHA512_224TestCase{
		name:    'test case 44'
		message: 'd141753cef2e1b442eb01708f46719cd8653b22e1d9709073d7ccf9242b875d03cf19c2bfc7dcd596f6c6a83'
		digest:  '0f357b8ca676a86ca63620b2239327accbd4252814a6b8c56895ea58'
	},
	SHA512_224TestCase{
		name:    'test case 45'
		message: 'ce94e0126b020d68115a5ea41083b4717a13d9c57a5b6d0ae7a3237c19b62dd2fca2f42062b90dfc9edb68e04e'
		digest:  '935d391d626a135772319ec93fd652bd9fd5680107c987872ae67212'
	},
	SHA512_224TestCase{
		name:    'test case 46'
		message: 'efc2d81b30043bb26c8d2c7a7d5bfe8ee579e463122d56a0e172b63060a83495d88bef7f0b625c9dba78ecb36ad0'
		digest:  'd9214789b7b736a7b28cebff8829a3999ff0c7496095b130c1e2e829'
	},
	SHA512_224TestCase{
		name:    'test case 47'
		message: '5b27d6fe9981d136b52157b839bb5b78d7844df5a397efbe80986435d89889939139892c3c13acf58f56704315b0db'
		digest:  '4c792bb6b1e33a3a042dd4a69ada802e3ec0bd839daa88635b225efb'
	},
	SHA512_224TestCase{
		name:    'test case 48'
		message: '4dbe1290524bd73d9db5f21f9d035e183dc285b85ba755057c769777be227c470e3679ea9a7355d889bb8191ea2ea7e2'
		digest:  '33a5a8d6119bb6dd7b2e72ece8e4d5d02aa99048c0459169ee9e6d04'
	},
	SHA512_224TestCase{
		name:    'test case 49'
		message: 'bdfa173995a5b99fba74d8e255b711771731a8e1851d7dd693733b9d7567d3bab23229cd7a160cb8399c6c006c21c776b4'
		digest:  '33c934b5102063213e7b53675c47b19c397fc3a51fb37ab46e179b10'
	},
	SHA512_224TestCase{
		name:    'test case 50'
		message: '9463c563aa1fda894757398e11b04a4b4cee0dce9dab01ad134f449a673cfb388349f42e42728a7977cec90630c1d3803c72'
		digest:  '4cae18a39a5ca4512b959da06bc9d63d07168084e35aaf03c8ba5d8d'
	},
	SHA512_224TestCase{
		name:    'test case 51'
		message: '2ee6e813ec84c2aded5800fc1cae94c09054e25d6bf0c4d10296d1e83e9adfb342a3c6364d016761e241ed4227fe53788cfa27'
		digest:  '89c6929b3c19a1472a4524e1427823768e46451e48faa7a7db7c8604'
	},
	SHA512_224TestCase{
		name:    'test case 52'
		message: '864833cb2912fa44929287bd0db7984968e5920d49f9725dc48707222d655f7f2b6e7fb40e80564263c7f450c53ef84df67247d7'
		digest:  '2c102a8349c4e362abd6a258217763229898e27c2e721c73bc880752'
	},
	SHA512_224TestCase{
		name:    'test case 53'
		message: 'e4dfe69723d430834bdcda7fa138f1fd26f22674be71169d914acfe2b2599014ff11d15c48108374b3f699165e78c7b547b08a6c69'
		digest:  '52310bacf78ddd40d7e93eb321da4de541a7296f98f0ad9745915887'
	},
	SHA512_224TestCase{
		name:    'test case 54'
		message: '6a3000176d0b029992d921e102eeee94dfb9b029d4019e4af06f517db44d78036a8dce00b90d49dd771994ebbe32c049420d3707e1df'
		digest:  '07de1ed3f8ea774a8e5e080669206c8572edbc3f165870913ab1e4f6'
	},
	SHA512_224TestCase{
		name:    'test case 55'
		message: '55cab876556079eff98bf7b90e318f8ff583c2be55de882c0defe6996d1bc225a51ef7127df2a5cc47f2ca26123f17e72163fc859c3406'
		digest:  '811fd907f3f74b9f214193048d70eda44f3a7465cdb882173bf79a0a'
	},
	SHA512_224TestCase{
		name:    'test case 56'
		message: '5dab016d4feb42ea606123c0ac4c3259cc7177af9cc955a64177b29f6bfc584ad8ae9abb1a673ac1997a8a73dbebd20b03dff2f89851482e'
		digest:  '340b5a71ed65407a686706b068edb92d0ad2a079c1c88bf78ad8a3a0'
	},
	SHA512_224TestCase{
		name:    'test case 57'
		message: '0921afb403afd31e8b0839dcc15ba2c0fc32bcc90fa78ff8ccd51960c78e357d5d4cc2c6f469008583923de75679b24f0bcaa8f0e161df9b32'
		digest:  'dd30882cda2144b31ab179e8f905460168b8758d8be8d8078dd610ab'
	},
	SHA512_224TestCase{
		name:    'test case 58'
		message: 'c0facb87b9283d87fdd02760e4fa437b896b13de28786ba0cb192f9f263e5e75bf664fa7526d3b2073a5fcde752914ec8b6ef1902d1b70c5bd1b'
		digest:  '1e7adf0737572d1aaa5d4c917c94d05dc092d6e59059df659490f1c7'
	},
	SHA512_224TestCase{
		name:    'test case 59'
		message: 'e0eafe1aed719fd7122a3edc9df7c4bedfbc1b0c3b1be6317c826f9c6b68e01ea2d3b7939b1e74c40cf9948c77b1c33afcb24e37734ad7e9b7af4c'
		digest:  'c260130197dc7774313c75843dd820d589a68df59e1aef7cf347f565'
	},
	SHA512_224TestCase{
		name:    'test case 60'
		message: 'c32ad1f821fbd8b5b7153ea8aa11055ea8fbde548660ecae28a8cb2ba2a6d1599c6e9305fb67abf6bea1b9db2568a88119193349c834a7da90b3ea13'
		digest:  'c91ee4c740a5663b3b19ed778a0516c01a3c0cbd3dfd6c09ef1afdb9'
	},
	SHA512_224TestCase{
		name:    'test case 61'
		message: '2657c625f5ea51f66b1375f479123d863ba0514d208430574c61ba963ad2f191a6b8005aaad37dc02790ebc566228353ffd4342afa9ee81d35f6097e05'
		digest:  '92f1334f253260e513d5e6f0d1a6af6ed2fbbcdb7fac85c39d6d4040'
	},
	SHA512_224TestCase{
		name:    'test case 62'
		message: '5655c3671fbd8288ea14944af925eaec653408f56233650fc4391018c22f9977facd2facb6c5062b6e3dfc6b8e837b3a78fca153cd5e9e32e91f5e27ac2e'
		digest:  '58d6e83ba02833ba0e23fe3472d28e8f986398620f7464ff18e9f6ea'
	},
	SHA512_224TestCase{
		name:    'test case 63'
		message: '350bd6018cbbed86e050b3e7e60683e1f619b73b4c2e764b74eecdbb00f4018d7ce9245da4ede7e6d439afc8bf03eb8964a2a4d93ef3551abbfbfe1ff0ce74'
		digest:  'c8ec854e632b2c132d505affe1bc00ddbe99452e5b59ba76a02dd838'
	},
	SHA512_224TestCase{
		name:    'test case 64'
		message: '4cd27324c28364873c6ddbc3e3a7e2cda9e8a72ad2f72201b262f874b8739f30ab60c34334c2e92f9d48533cd8ad2312c3e7c386aaa283b50dec844fa432d636'
		digest:  'ced6081761ff5259f132aa831b7a1b432d093fc857da0eeeb82be71f'
	},
	SHA512_224TestCase{
		name:    'test case 65'
		message: 'a4f2063fe6ec17550d7aa831091952dc4dc041a283c488187ef9b75e701ab0a25d6ab6e5cf9cc702ccf02cec05a04e37507acbda58cec933938a8b4b75a4425ce4'
		digest:  'af6b7fafcbe9cb25003c5702995fd656365a20e113c8e5b10312cdf0'
	},
	SHA512_224TestCase{
		name:    'test case 66'
		message: 'c1d30f33ab4b410d0f8a809eb5cb681ec9153bde26693b20d440572cce350aa009fe8108b3a0fcaeb8e376b48b04885f23ab8bb6441cfd4cffb1255a7c88fa10f779'
		digest:  '408f4fae70d0fd745ae44ed66740bdf1fec0dd4e23db06920c6f8ec5'
	},
	SHA512_224TestCase{
		name:    'test case 67'
		message: '61b34b8aaae2e19e2d61924642351ade2baa6925f7a3a735a72e76b9905b775a004f0f7a24946b2529b3c144a205d9142ac6cdd74a6b0dca9d35a5563107c38a317573'
		digest:  '19658b285db6cb9dbb4ceabfe671434cf1b85455a8b4da385ead830c'
	},
	SHA512_224TestCase{
		name:    'test case 68'
		message: '3413b09d3d12a333680f798c0e488776579d3f6a65f20412ced53c085060a6e1c061ed6a44b97e31c20297479875a1458e0ac08d08e0a8d281e0b2db349e2f831c7ffc05'
		digest:  '008a3c6ca57853adb1af5d44595e62016b7af47226492b9f3c1a9713'
	},
	SHA512_224TestCase{
		name:    'test case 69'
		message: 'b6e690d17d9d7f07185502bf6d275c84e3ac4f5f77c3d4b30d8e106603be84410c11849a3c18ea305f3e0b568f06ea540694ac0a306f4cd6048590b895703cb4e4f78578f9'
		digest:  'f51dd594c7dfeccef95e3d4024bb00f931a9e9516958c32d2259f5ca'
	},
	SHA512_224TestCase{
		name:    'test case 70'
		message: '13e6b1b4f021d610c81c97f0f952daba2766034d815b5dda4603bcf788ba60ee31541d5b4353b9f6645d96ad99ee90f6524b2963a7b7e476e1e8eeb83cbc0305eb29902a5d72'
		digest:  '09900c5ae3074fe73e6c4eef51f785e57947bafbe1d8dea38868e3d1'
	},
	SHA512_224TestCase{
		name:    'test case 71'
		message: '3a300f9c2dc3531917222034791824b49436be4dda760f67a21e54211fe77394935fd35cc08c71559fd231a2551672f46324e4081d4e9e725d566b9a788f6504713ec6d46d46fa'
		digest:  'd71c18afd448113a45d06ba616a1945a567e26f336cb94a45c2519d7'
	},
	SHA512_224TestCase{
		name:    'test case 72'
		message: '81c64d2313dce3126ba16d3216c0f334910f063b00720be322fdd68285587baef66860ff3c2b971506e46bdd21793911ee1982fe14213d8a7241727558db04dff6abcf0b14adf262'
		digest:  'c85acd75ec4ddef1f179f98fdcf8e006a1d80afc14749310aa97599d'
	},
	SHA512_224TestCase{
		name:    'test case 73'
		message: '09645b920bf81735d6b5caa43bb76dfda7497003a6595c2ca62ab7f5fe4928792ec2db426893275f50616eb6016a459c50bbd5a354a381b32667ca5de5376f6bfd61d8bca9c5b44fb7'
		digest:  '650e31a81dcc7f3498900d205d611b101e89d7243e1dd77551b6c422'
	},
	SHA512_224TestCase{
		name:    'test case 74'
		message: '3fd1d8fb9e7faf18a49a1bdc77fba093cbe09af00d461999e19aeee7cf9106e63955f7a491e1d9a8c9ae57c00a8f0500a8d9f50756c88584961e7af9740d53bf7457d74bc9b4f1afc774'
		digest:  'fb2589e942890fed8f99ecfe02d3a51683a5da582be9dadd25305d33'
	},
	SHA512_224TestCase{
		name:    'test case 75'
		message: '5af0eb316e793fa9ea9d5141fa26cdc3cfcf9852976163a8ac1d81a10b5af5fd096fd06141d3baa0d183ce39c6a4ab21d46ac7c24b38074346098d0a250ed27f3e98f413d0395182382654'
		digest:  '8f4c6a61321c6e500cf0dc2b7d7d97da9301f71420fac604feae19f2'
	},
	SHA512_224TestCase{
		name:    'test case 76'
		message: 'b26d3233b551029077beea92859889691ac3853a9f972b2a6e8e55b8db40914f75927f1b892a7b1de8267c7aea0d3c19b3059a6cb5473c8e76d1788f945a750c068b8d02a46cf6e3ea1da2ea'
		digest:  '4d8360f3692defa11eda027f94099b3054b841f809d2aeefc885d2b8'
	},
	SHA512_224TestCase{
		name:    'test case 77'
		message: '6fb0f3856dcc9addbd155916f0227630959f01c764acab99bc80dc435052ff4b1d7921639fd2baec4462caba213c2817e3a5170a09771bd951c30bcc70ba663406ceeaacd114639f21edc20a8c'
		digest:  '71f8e09f911dc8ac67999c10c059c203af4af5e436eed8f8a97a6f2d'
	},
	SHA512_224TestCase{
		name:    'test case 78'
		message: '8471e939d4531486c14c259aeec6f3c00dfdd6c050a8baa820db71cc122c4e0c1715ef55f3995a6bf02a4ce5d3cbcd790f72087939b05a9ebc93a905ceccf777a20d36eddfb55e5352afbe8ddf97'
		digest:  '065cb93bb2406003776440d9a02af1e4170e777c7d2c13a113262ce0'
	},
	SHA512_224TestCase{
		name:    'test case 79'
		message: 'c87a7448b7189a11e532a4320874186407fb32470d18904cdd512fd265a9968f95225132717fa146654e725ad9268d5f062e0f5108de1a1a340acab3ab1c6b8c2fa1e92e3607871f3da4d4055ffbdc'
		digest:  'a39c7c5eacdb22669587a1846c2975b41f9b74833d9979de2d7f11c2'
	},
	SHA512_224TestCase{
		name:    'test case 80'
		message: '26465ea06dbb61b4193a982c984a0358085fd6f14ca027ff60ede9fae9211d301194476691e51fc20ff68ae9ab37c864d528ef4861199816dd34d8e022be72ad299a4a1f200f1d53bdcb7009d327c605'
		digest:  '21afd5f923bc7bae234f14ffd662f174b4d179c7241ddb57576fe502'
	},
	SHA512_224TestCase{
		name:    'test case 81'
		message: '5dbf5c2711e26fba3423ecbff0d56eb8c934cb6f78ee8e96f6e2a8aad727621dd95096b3bdee1f2d55578e25dfa61e399103eb6b75255ec87714ec48bd3400f6367f35507612c9f1c1eb4f6d65f1a37d4d'
		digest:  'ec5a139936f2c42a1f4110e2c52bf7c3cd3bae49955c608fadfa163d'
	},
	SHA512_224TestCase{
		name:    'test case 82'
		message: '25be46e3df2595aaaa9be9d21017aa2d64306720488c79b4719ce3eaaf867d4f276352d1e3ab46d18eedfc7978c4f650e106fe0b733ec5a7105dcbb4ceac7ef44005a624f843001a49f72910525dbe40378e'
		digest:  '276a0dbb542fb69ed409a91994625f8469b051be25b5968b3a6fb0e3'
	},
	SHA512_224TestCase{
		name:    'test case 83'
		message: '2bf69b4b50312f07fcd68d466e329cea515af2bc9b78df5eb664ae82de74275aca94594f8e9996f6c0d2be40d9e0c1f1dfc85f534147322b0366efec1a4de92b1a5cd6530e78d69aa7a3b1339ad060c543f5d0'
		digest:  '45a66110f93ad7873923b97336c23e87642e79c7897d9011ddecf03a'
	},
	SHA512_224TestCase{
		name:    'test case 84'
		message: 'a81a32139d69fbd64000f8d4a6ee10290ccc83e8c9e77b6081a2e59e9c157e33f6fa3137d8caa08d28230169dc6d2b10b4bdf0df29f9ae8fa13e21b7e47e7b92d36eb703589f2ad699ea5b49711214cb20a5b903'
		digest:  '4143fa448de5ebc14aa204d934ee3971051c4e3074d4caaf0cc62af1'
	},
	SHA512_224TestCase{
		name:    'test case 85'
		message: '471e434b7ef39a0551b2ed9250006353ecd915c82b77649ea1768fe99b3585ae9d7ff7a40f6bb23b33802cde5a1d5c755818cf2f35a6f97e257bc963ba3280eb59eec2faeb42ed47d44eaba8292bc3901d8a35d5bd'
		digest:  'a12a51a59f116f87f757f2fde30e11980d7d96ebc959774a734ede74'
	},
	SHA512_224TestCase{
		name:    'test case 86'
		message: '3e346c6aaf8f977620c3ac68fb608aef432d6cab296d1a93da6261949c10c07a19436414fad35793b827f434aff89e1f7557b18e69f2b9318f93e2d29be9b0354b8bf042fa3aef657366e93603f4c455d23ffe98725d'
		digest:  '307467396d59103a897e6a5b16dbaa2a0dc365e22395465a035fd293'
	},
	SHA512_224TestCase{
		name:    'test case 87'
		message: '1ce1f01b46b4c9e04ce4b5e22fbbed4b025427c55848b03df98bbbe7d990f9834ab469de70d9729bf5512f33ffff0a3b5fb88ce60cef73cacf9205e73f2fa5e5f05ba49de85459db52e57462778a8f31e9d50605011009'
		digest:  '5ba94e26394485eb0a9e3f3d3d9a6083b35daed5f0c889bc78924c35'
	},
	SHA512_224TestCase{
		name:    'test case 88'
		message: 'd624f63f53be3d6c0a20c12263e4436e462407d72e3d854280d361131c169d9531430e9c556d0c69d57d6ecd4978b6926c0088162b1e6139fc3e8579717b395c0d1d330f56604f4c2b78838058d1152e689b0b8ebb86e47b'
		digest:  '37cdd523d922362ad463216f340f254e76aaaa0b45f6dffca30d55d0'
	},
	SHA512_224TestCase{
		name:    'test case 89'
		message: '7f601132a3833a3e2a33a0f1a058d6a2db06f11e39afec8829974b64ed89ffee9ec98ab070496353371f9cb62a37c23de745056cb8fe98b415885b8c6a2fb8c41a59ca16c3bcc5ffc4ce92cdfc7db9f8d52184b581af62c984'
		digest:  '10871c0e562e11aa59406eb4bcc832efe4942260ef08cfcf99e7db58'
	},
	SHA512_224TestCase{
		name:    'test case 90'
		message: '12bf3a3337d3b6d553bb8cea92c369b930048ed2902c81f088820324ff244330bb9351317472d5822d8b01dcc413c4c2c72f411c89b77f227539627ba341daa6a9d75a9ab44a8ca9d72c43aaa29eb157f7cedaf419bd48320646'
		digest:  '6f763886f273aa5f943ececf7b765d83ed1a78b43abef25907226b52'
	},
	SHA512_224TestCase{
		name:    'test case 91'
		message: '424f1d07b708e60cc4d00396cb0e90597f3977aea7904be61f0b74de1f918dd3bf6b0a78bbf91e4674a3dc1438139ce7c3a1a00b060ff7632777a2fdc881f6e98cd9414cd7e1b0b0d6a3709ec009146f5b50375fc1ba62e3648330'
		digest:  '7742c1511fc5a0297634fb3d928a058626518b7fb88db428e2a282ae'
	},
	SHA512_224TestCase{
		name:    'test case 92'
		message: '87e1b9edc7c6576998ca27a41ecf90410327adb96db5e8a12d4eca2c3a6511f05de803ed2d6503cb3e716263964a778160bbf2e8e5406d97b76b444b2f86a7451a7aed4db0f02bf7c6b506254463c5cdd83eabfdec5c831cd6097c72'
		digest:  '4331fbe48d8a7fd2e69ab0279509131d44eb132427311724e2dc19cc'
	},
	SHA512_224TestCase{
		name:    'test case 93'
		message: '8fd0705684ca323c8ace2e7791d407697d05d522357dd23bf64f5501eec905d706fa46d5ff72f8b5c04961f0ee636724bef103b9073bf2c49c9eae2bd8d09c0b7a6c8756b3aa4dc17e6dd8dd3084508350298f76a8ced6449c1abe6dbb'
		digest:  '43ee3b234ea91228c05358fa56bb5072031fa413be371413a257a0da'
	},
	SHA512_224TestCase{
		name:    'test case 94'
		message: 'bdccba55407761a6b90ae88ecea8b95ce13ae66a34066d505184bb58a96c6e327b145fe9f7f9e016a49b4254ff8f9bf365cfc5f13254aa252be7bfbe1934cfecb720faddce5cfdeb9f3ab41481c0ddaacbfc1e4bf4964a14f9ace012ffd9'
		digest:  '21fa866d8695008099e400210d04a2013701887d7afa98e8324e96de'
	},
	SHA512_224TestCase{
		name:    'test case 95'
		message: 'a8aa236cde557d8d365e6ed356636ee9117f7ae9e6bfb772c9da59fb271cc36724a4efd927cf11c998f47758e852b4e0d246fdd0e1c2d3b733fae39e7c602a9b4dd35ca3971387a06b79f3c303e54e1c06f705680882f52a503d43fddfbc88'
		digest:  '102fb3632c675224b30ffaa5a322aef3354e1ff0183cdbfd06e5a17f'
	},
	SHA512_224TestCase{
		name:    'test case 96'
		message: '44c6c75e377f21fc9cd7c164ca5c4cb82c5538a58dfb323992e6bcf588c61b246053706bf88725a09d0a8adfcdeec0db419cd7732b0e3386bc3f3407e9e016546f4d15c314bfd57e30c302926deb3342cbc315a1e706c5607c127de42a9a739b'
		digest:  'b9b62986eebdb35c88b12e0257537a05394ef5a16fad01c2fec57d6f'
	},
	SHA512_224TestCase{
		name:    'test case 97'
		message: 'd3c4a0b17f46cc5653bbd8300dfb0df6d0af3fb7c7639a830bdc9f68c7ab69a09a0235083cc721d465e73481eaed9f0c29d34262a8177217f912fa5ae1865f29b97669c1133ade59a1d2910f537b9b23c080e963c753f7e676dcdafd2a118467f8'
		digest:  '49e4553a6c26e7a82cf70b15528037cdd658ae33653bf4ce8e86799b'
	},
	SHA512_224TestCase{
		name:    'test case 98'
		message: 'f6e3df95a4526aaf671b1ab9342db3ba0a6043da82b355c4a8d7664e09f29d5c0d9db601872e40e88b3d08f38061d521753b55b5b9b65895a58c3cf0bd0da823f81c7ede5b7a0a82cc4780ce26c5a5bb8960cca4f4eb04aa152d90575f9c97825e68'
		digest:  '4854cb2a3a9c9e3ec91b4e7d79a753f6e2240f4b6e2050590d6f1647'
	},
	SHA512_224TestCase{
		name:    'test case 99'
		message: 'c472244bd3325a84d37220e7ca6ecd40e3f20e52677a811ce8987f5bc0963950ba5c06c30e95fb48930ebdc1305e932493ccbf7f266d0f393f8daf9a9850418dfe992fa54d2b405f9b0f6aa7b435e89dc75e8c985e38556b93cb44e881156ca3faf149'
		digest:  'd459400905f4c1eeff90decd9cc994b394b21430379d6328c03af436'
	},
	SHA512_224TestCase{
		name:    'test case 100'
		message: '8ff0520805a336aa768a1a75f2bec3ea876ac2246f2d496087bf3ebc03d88e3f604dc41a9b7cda2da1eb4ecd062c79c5f9041d9a78cc3d14c01631f90894f91c6e36d29632930754228a0ad25529c3251ccd283f368b0e6f43cf3e277b43cc02a0f3a405'
		digest:  '616fa8495f55e77a308c223afe470b42c0e1a014574aa36d01f484bb'
	},
	SHA512_224TestCase{
		name:    'test case 101'
		message: '6a4cef9b8dca0ad7767515c8a61ac32c730b2894b60a106bc5699534a170f11370ff908f2d746a969ecf8fc1861503da1df2cd9cbda14bb9de11c8224440508277ebc30b746a2976a2d64545076c6d4cbc36996f9ee8b9ad47084c5e3d512eb8685ab198ca'
		digest:  'd20490eb9633a321a69eb27d42dd619aceee533e850075fd16589bef'
	},
	SHA512_224TestCase{
		name:    'test case 102'
		message: '9a4fd44a08d9f5e7731603ac92b18e3d880ff2d5b9d7cee6dbb2b7ff504f2df3b3abcd38b18fc98d1a5a96df3700e46e5f53d32cc581646594de2da53ee452d71078b0aa0b6f5b39ef514bea985d52968dcd51ebd75bac0f0f03fb86bb0b0356db41e469be84'
		digest:  'bf3a90e1e0e8b1a20b08331da3c1996c8395a1e3b835799f879cc7df'
	},
	SHA512_224TestCase{
		name:    'test case 103'
		message: 'afce173b057b48406e163584c8d8c1a93b15804612c2278dca34412b692a2470120e24b0d57806e9cbf4b8a563053f22f76af0b1a39fe35c56a7a5cb29092d69f01911c20f78fa2f90de56b672174c2c434cf59cf184c9eddd6a57be31513a95747548395b840c'
		digest:  '373d08139139af96f22810c8e8443a781c88344b73b9b717a97f4a04'
	},
	SHA512_224TestCase{
		name:    'test case 104'
		message: '5421cbce307d2c89a27a7b0a1d515e6927002bb65d3f0990b8b452f3106a3c52de80b9e6461e9b4555adfc368748ddc48c6d0d914bf94d6d2853b12782e749a380d6012d0eebc60d22d2ea5c4484ca164d06ae706f53f06b759761a94c9de87fc38b6c4d8fcdf40a'
		digest:  '5bc1dcf59342499195a61a0661af7a8e96097660a9f7e576572f3d0b'
	},
	SHA512_224TestCase{
		name:    'test case 105'
		message: '5add9071d1d5d1799c0f9223109b761b7b9ee1c30aa20025ea300a2baea1c512f1314b1c3c550e852a4f1ab9c9ff9ac444fc9e0abfe31b3883ef337a5bcab7ef3da3eb5a2d81891a5fd7085c9c581d8fe60c83185fa4f847670b888a053d0de1885cf7926e01491e84'
		digest:  '0c370410dc8e22ee61c49d7a3bd3d7e80cc4f3f51acb26ae4ec5a3a1'
	},
	SHA512_224TestCase{
		name:    'test case 106'
		message: 'e965ba8d1236a10b362cd75cacd0175cad0af65e13d9c57a6047b729029d470fd6d3d417ea4a8ec587be96c86f7661cf63bd5e9ee896d8614bac1d40e388e0339606dc189e0aba038d6f35d58f407c10f02bd7e9de328e9f6fa4070ce64c2e7a32e819b7b7dded6c5128'
		digest:  '175c4a7fa491e7dbe3a2c8604b07bab0e2a2e34b270e16b4392e9716'
	},
	SHA512_224TestCase{
		name:    'test case 107'
		message: '314158336fbf7845251e1a620613ad8c1a5996e512ae25cdd91ec84b6b837faeea4d9074ef7f7c6a1efa3115fdfe815106eed338fa531945792402b9d045703e124d2ccb6a8cd9c7af4d26713958487ff3bba805b2cad899c565ccb5ce79475dd0c9930188ef75a4019d60'
		digest:  'd6cd1901bbeaf4b69a7c211c50546fce3538a8388460ec29ff8166df'
	},
	SHA512_224TestCase{
		name:    'test case 108'
		message: '2c2d86197b39c194c65a93bdadc72a9e590fa75fa8ae6b758adf9116abb2bbf0525e121a89d8884b7ea5614a29ecf1a52fcfa7a606fb4a835a7e83d5b0da9c4b4a5787a04a9d22ca1a81b9750a20ccaf3b2a13e5bd81c00ef403042640a3d4dedcbfed9207ae167b6298a526'
		digest:  '49dae21e7733ce09e392dd980ec3a0c2c54269376e231c19dcf5a774'
	},
	SHA512_224TestCase{
		name:    'test case 109'
		message: 'bc106b071b53c4d855e1a3dda41db1a7a4460586b448a3fd9fd842747cbf532dd2763e7900f4ef2cdcc4914e9b5a9913f1a7070d78ebe8a156cb20b81ba366b6e81d90b0d60311da9551c03e548fdbb4e15d4b0e1250299363fc82afe05257e8462de814652733c9c8b1b71698'
		digest:  '3baff8bc1f4351fc62ca40904a002198b5ef6924afba1d1fc61d8ae1'
	},
	SHA512_224TestCase{
		name:    'test case 110'
		message: 'f627848c4afecd8df33da19c606aff119a3762c61c0539c3e6a8961bcd6a2a111552717f41a62a792aae0a2d4fbbb10d6c68dd314e20def6e5e04aec9b3ef1efe4ab139316f5d1c67c10e2b0a51587de028bc1a0b1e6c1af934da013a4a050448cbad39071f3aabc10e991ee0c81'
		digest:  '2e0b5455cce69262066fdbf9390ad0dc56138b221b75d51298bb5b2e'
	},
	SHA512_224TestCase{
		name:    'test case 111'
		message: '15cb777ef3e451b928dbf288e46a3627044ff5de42add884a9af6b424d6e7399381581a6a743c7a577b02bb5da149ada4e449f48d09e34df4ca8d8f259f4e14c23471475a8f97331289f564ad6e8bd8fd4c5e51d5ecd19dd46dfcb4ea009e385bea857725fd1fc6423f09ccf42af48'
		digest:  '9fa12561f1df9a2d793292e0f4df5327af529336b5b2118952f5c24e'
	},
	SHA512_224TestCase{
		name:    'test case 112'
		message: 'd24df75a00cf92677bb41a620fae519723937ebfe1f7b430970056505d76db4ff91acf16ff391a7a3d8085b655127a18acd80bfa831837f4644a6850c0273fbed6029449d65bb98a47b2ff1ca6997c50500d0b21a206936a5e4d8d56508ec01832ae4fddce5ef6ff62f1917c486adea6'
		digest:  '5cae12ea9652269ea2aafc656cb83424746ea1d5d491f9a159594b2a'
	},
	SHA512_224TestCase{
		name:    'test case 113'
		message: '730963b1a462d234e8705db9278cc58dcb817bec6c9e62bcad0ad57c6158ec77f35bfb3d043c5c5355c96f2ae810de4e622d26e0b4605346fa630a21a3facd7fd3cb7b5305701622f5cbc9febd992ab83e4abc52c111b8b3de370d9e9eb4a5e5dc00eadaf1278907901c751fee4f303426'
		digest:  '46c7046aa633fb5bc4d6952f853e39c53e771009183b66961593a3cf'
	},
	SHA512_224TestCase{
		name:    'test case 114'
		message: 'c1460b2e1e385c0b5756e210bffaa8f1cb12ebc3c60e7105c5fd1ce0def2beae95ad2fba97779d46ab4e24d3e5429e9c1c1fbd649f4c577124f300d90162defc9fb2b9522ba652678ed5f623a00c524f98ee8955c8dd8447871e8a236e371056c79cd27e428b2274ea35808a581e6c1126c6'
		digest:  'd8402ee634e18c967d110bd7053a302da650ccb38f259cbee5a24715'
	},
	SHA512_224TestCase{
		name:    'test case 115'
		message: '48a84e3c4850a29e433c8a243bf4beb05b8a5bff38cc418158bfb968f1495cfff22e303e473e139b59cc5d22d561c1c8afb274fd66d09180899fbd777a8907bf72731181c5f0750668b562ccb04e86d95c0f314fd64cb741b6cd16c5887cd84acf613b2c4ad0908675e1e1485321d6896b87d9'
		digest:  '3748808c1a4d1dc1b82ebab4cffc9d8cf5c251d045dc8affd756dae3'
	},
	SHA512_224TestCase{
		name:    'test case 116'
		message: '41b2b071ee6a8e0bc62e9e03a8705f6da218e69b111c234c0a516520f981204f88bdef9bc3f4d58379c5c8bae59270d78dd62c4e6bb8569f4be0c8cd74899e128ad73ed1b36b96ccf936a597f08a3d4d987707818454c7773c9ce194e1fc1d140fa9a7fb07f75a4e0e283d8210814e3589a47f1c'
		digest:  '8776f08070e38582d4604170a9e658fc17034cbde9e03a002e728131'
	},
	SHA512_224TestCase{
		name:    'test case 117'
		message: '54bf375944c002ccac87f55ff3eb58c45844f62d917ddd03b26b5fdb515d5ca6e8f5c4277a3657ab7addaff4e8f58176bda620c6894e424f5bd30747e13737e1d06ccbc4866ab0bdef80c7b0a5e730d55b9dc8207b70f0010e59914edd0c5d7ed5cd5717b0859569d215447d71eaee49f9d2f825af'
		digest:  '0cd690736dd1f390694f65c3b89e0cc2269f3b8208c806929a8f8cb9'
	},
	SHA512_224TestCase{
		name:    'test case 118'
		message: 'd924b16dfe2f612d923b25669b45d9117d30a3531d5d30ad477c41734da2677ff19b0a268581b06b518d7c73352ef53990580a83601f5215aa8498241c8f960df19e014715fcc69689b594820aa99a08ede0928f4e1dfe1e69b6b1cb48a292779fd49db8f4c72a17025b727c60df4fe65e6f435f3080'
		digest:  '304e38689250380e322505b4e23fff35b341e2d94cc62cfb44b51db5'
	},
	SHA512_224TestCase{
		name:    'test case 119'
		message: '7d2d12b351a70306230e7dcc7664ca3883928093395ba8d5828f1f76c8813d82b8bab5ec45efbc24f0316eeb47646d67acf2cb7258c12a38a0622101768be24b32c71a1c936c199600cd33cc14b266d45a053d1e55cef303c462dca123d601e775ba0aec41046285db4b9516e436fda814339d3c376c4d'
		digest:  '40a7947aad6c1e73436d78672038fb99a4ed7cb145445e276643cf38'
	},
	SHA512_224TestCase{
		name:    'test case 120'
		message: '5fee687802a5aac1332917cc5d0a9439ad95c764219bce45e2743094d34413306a7db0c1bfa5059ce25ac9e87293a5ff6b28a1174095dd222afb81f3bf67f6a1c65757304a7a226eedb8341c7f72f44ecf0bce15c1a7e35aa132fe0db7a5ca801c84430dccab6bfffb4b3eb0ad30e7ee814534a3cd5948b5'
		digest:  'd6657f115db759444744a125e40ade361d8a40fd88c91187dc5a50b3'
	},
	SHA512_224TestCase{
		name:    'test case 121'
		message: '696730b7eca27066d541c4150a22e6a6b062c951f2177df6fd4254b14fff43b032ff8a418338972fb8db284f2287b1566ebcfded7292170827ab6b73f16629ad2755eafacff296b4a98d85e27089e40c5cc85b4deb64eca9e1a01fa18d1e5c054f52980a5736349caf3f21f57c90496f816f2a96e4f5c48235'
		digest:  '391d6c2b645d1eaa7a7dd68140fbb7c23dd4ac0347887a782e4c3442'
	},
	SHA512_224TestCase{
		name:    'test case 122'
		message: 'c8995ce09a8d80b790789915b77e53cb2ba0b3974327f0a509468555edfb843fba362d75c83dbf07744f8ecdcd8756609004cf6a574781200ddcda0ca53d7e100bf3dea85bfe0c6f5277cee50274f16fc92cdf9f835c74b6ffe90e67b0c6e3dea5189801bb980ff6123980e5bd0ccbffbe600cbf312d706fc7b3'
		digest:  '3a6ad16c5957b54ee6691d227b2d2ecf9348b68435a50647a2cd55c9'
	},
	SHA512_224TestCase{
		name:    'test case 123'
		message: '247f8754b008acea234d3b5cab5c613a695af8febdbcf3e4c5bbfdfb20015d7cf8a91d5b3f666b1d38023fe93a0b3ee1b1c9dfb36572cc6affa18b8bc9055d5db5566688a18e0fe1d955a985ddbd5325652f1da0a3a66dd7c81aa2bdc940f8ad6370ab015de660e00d2ec0571dda66ab8f76fbeeffbe9fd1280e8b'
		digest:  'fe66026c1c2410927fa442cc122d5aed90308273401d429fd1ac8f61'
	},
	SHA512_224TestCase{
		name:    'test case 124'
		message: '93dbada71b890e80d9b88a338a7df020a7bf9ffd9898536225fd3e8a2da25b589a18ff989438b188f5c53b96fd4d016c4850922afca57df2471866aa095858fdcd115da3fe554ce35840e3569cdc15ed8842a5350602f4daa4c5186b44312d5c140df051c1a68a9ae871a9f9cef1d5ac50db02a71d796122224c7c53'
		digest:  'f73154e76b5998e0792fb0ab1d721ee730b64a1b3a543061fc76cd17'
	},
	SHA512_224TestCase{
		name:    'test case 125'
		message: '6178736ea6aff4b99e7da97dee000adf72b37069df8ad5bba63fdb7e1be6109ad6c871a9596e2dabf4fcabfbe6c1cd82c50b772a20404b95622ab710f99332c2b812c2297a24c2c8097ee2193ea26fc969b1d524f28062cf41ed4cb5dc74f643cffd6fd851f83b77a4ac57c03dcc771e254c92419f72a0cf0b30a3560a'
		digest:  '33a7425316d2f6be1bdbb870d953132d55e29fdf8530463977940908'
	},
	SHA512_224TestCase{
		name:    'test case 126'
		message: '1c21b5842312111a9e2782fe97917e946b8d65987c43c91b27ed0cc8f63b316c16f6ecb60be5fb3ba2edb9bc277bf1acd927f0ed9b91dedbec8f4e8b198b1354b126558f3a21e59c227ece26f9e22da7dd3e422e35aec660a2a27fbc323b1324d7f81a99189176470723e7c1738eda8d5e1b82e3b1632ab22f6ccae4a814'
		digest:  'db43ca3dd7bca3d899648692e4ed8f03119628df313e1573723c58e9'
	},
	SHA512_224TestCase{
		name:    'test case 127'
		message: '4b9895235cb4956aefffe815415252e7d6b21921bd7f675315eff071d0bbd429b718c774aee96f6c3a330d5d40d1601e1069c7a2a19ea5ca1e87097da2608ffb4180816e478b42c3c4e9edb748773935eb7ca0df90dec0eb6b960130c1617880efb80b39ae03d617950ace4ce0aca4d36fd3ed0112a77f5d03021eb1b42458'
		digest:  '9a9176e97aec99ab07f468f6a226876710d6d877021d27061d4d0132'
	},
	SHA512_224TestCase{
		name:    'test case 128'
		message: '9e127870be2431bcb4f4eb4efd5c2a6c5870c55e7a5e3b7503994a4cb136be4ed396887801450f600b22cb772fc00f8b8f0d2690e231a29f69b9f13f24f531e4479e45b5e8bc2992fac782567e0d7a59f853ca3a20bf18dbdbf684ac69817e2de075daaed9532659692d3b73530a12df7b8cd9e49ed0463041962c1ce7a24c31'
		digest:  '7e2cf6226623535784c59cd6a7b27dac60ee23fdce8a804dbd6dfedd'
	},
]

fn test_short_messages() {
	for c in short_cases {
		message := hex.decode(c.message)!
		expected_result := hex.decode(c.digest)!

		actual_result := sha512.sum512_224(message)

		assert actual_result == expected_result, 'failed ${c.name}'
	}
}
