/*
 * Decompiled with CFR 0.152.
 */
package app.freerouting.api.security;

import app.freerouting.api.security.ApiKeyProvider;
import app.freerouting.logger.FRLogger;
import com.google.api.client.googleapis.javanet.GoogleNetHttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.gson.GsonFactory;
import com.google.api.services.sheets.v4.Sheets;
import com.google.api.services.sheets.v4.model.Spreadsheet;
import com.google.api.services.sheets.v4.model.ValueRange;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

public class GoogleSheetsApiKeyProvider
implements ApiKeyProvider {
    private static final String APPLICATION_NAME = "Freerouting API Key Validator";
    private static final JsonFactory JSON_FACTORY = GsonFactory.getDefaultInstance();
    private static final int CACHE_REFRESH_INTERVAL_MINUTES = 5;
    private static final Pattern GUID_PATTERN = Pattern.compile("^[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}$");
    private final String spreadsheetId;
    private final String googleApiKey;
    private final Sheets sheetsService;
    private final ScheduledExecutorService scheduler;
    private volatile ConcurrentHashMap<String, Boolean> apiKeyCache;
    private volatile String firstSheetName;
    private volatile boolean isHealthy;
    private volatile long lastSuccessfulRefresh;

    public GoogleSheetsApiKeyProvider(String spreadsheetUrl, String googleApiKey) {
        if (spreadsheetUrl == null || spreadsheetUrl.trim().isEmpty()) {
            throw new IllegalArgumentException("Google Sheets URL cannot be null or empty");
        }
        if (googleApiKey == null || googleApiKey.trim().isEmpty()) {
            throw new IllegalArgumentException("Google API key cannot be null or empty");
        }
        this.spreadsheetId = this.extractSpreadsheetId(spreadsheetUrl);
        this.googleApiKey = googleApiKey;
        this.apiKeyCache = new ConcurrentHashMap();
        this.isHealthy = false;
        try {
            NetHttpTransport httpTransport = GoogleNetHttpTransport.newTrustedTransport();
            this.sheetsService = new Sheets.Builder(httpTransport, JSON_FACTORY, null).setApplicationName(APPLICATION_NAME).build();
        }
        catch (IOException | GeneralSecurityException e) {
            FRLogger.error("Failed to initialize Google Sheets service", null, e);
            throw new RuntimeException("Failed to initialize Google Sheets API client", e);
        }
        this.refresh();
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r, "GoogleSheetsApiKeyProvider-Refresh");
            thread.setDaemon(true);
            return thread;
        });
        this.scheduler.scheduleAtFixedRate(this::refresh, 5L, 5L, TimeUnit.MINUTES);
        FRLogger.info("Google Sheets API key provider initialized with " + this.apiKeyCache.size() + " keys");
    }

    private String extractSpreadsheetId(String url) {
        String[] parts = url.split("/");
        for (int i = 0; i < parts.length - 1; ++i) {
            String id;
            if (!"d".equals(parts[i]) || (id = parts[i + 1]) == null || id.isEmpty()) continue;
            return id;
        }
        throw new IllegalArgumentException("Invalid Google Sheets URL format: " + url);
    }

    @Override
    public boolean validateApiKey(String apiKey) {
        if (apiKey == null || apiKey.trim().isEmpty()) {
            return false;
        }
        if (!GUID_PATTERN.matcher(apiKey.trim()).matches()) {
            FRLogger.debug("API key validation failed: invalid GUID format - " + apiKey);
            return false;
        }
        Boolean isValid = this.apiKeyCache.get(apiKey.trim());
        if (isValid != null && isValid.booleanValue()) {
            FRLogger.debug("API key validation successful: " + apiKey);
            return true;
        }
        FRLogger.debug("API key validation failed: key not found or access not granted - " + apiKey);
        return false;
    }

    @Override
    public void refresh() {
        try {
            ValueRange response;
            List<List<Object>> values;
            FRLogger.debug("Refreshing API keys from Google Sheets...");
            if (this.firstSheetName == null) {
                this.firstSheetName = this.getFirstSheetName();
                FRLogger.debug("Using sheet: " + this.firstSheetName);
            }
            if ((values = (response = (ValueRange)this.sheetsService.spreadsheets().values().get(this.spreadsheetId, this.firstSheetName).setKey(this.googleApiKey).execute()).getValues()) == null || values.isEmpty()) {
                FRLogger.warn("No data found in Google Sheet '" + this.firstSheetName + "'");
                return;
            }
            if (values.size() < 2) {
                FRLogger.warn("Google Sheet contains only headers, no API key data found");
                return;
            }
            List<Object> headers = values.get(0);
            int apiKeyColumnIndex = this.findColumnIndex(headers, "API Key");
            int accessGrantedColumnIndex = this.findColumnIndex(headers, "Access granted?");
            if (apiKeyColumnIndex == -1 || accessGrantedColumnIndex == -1) {
                FRLogger.error("Required columns not found in Google Sheet. Expected 'API Key' and 'Access granted?' columns. Found: " + String.valueOf(headers), null, null);
                return;
            }
            ConcurrentHashMap<String, Boolean> newCache = new ConcurrentHashMap<String, Boolean>();
            int validKeysCount = 0;
            int skippedRows = 0;
            for (int i = 1; i < values.size(); ++i) {
                List<Object> row = values.get(i);
                if (row.isEmpty()) continue;
                if (row.size() <= Math.max(apiKeyColumnIndex, accessGrantedColumnIndex)) {
                    ++skippedRows;
                    continue;
                }
                String apiKey = row.get(apiKeyColumnIndex).toString().trim();
                String accessGranted = row.get(accessGrantedColumnIndex).toString().trim();
                if (apiKey.isEmpty()) continue;
                if (!GUID_PATTERN.matcher(apiKey).matches()) {
                    FRLogger.debug("Skipping invalid GUID in row " + (i + 1) + ": " + apiKey);
                    ++skippedRows;
                    continue;
                }
                boolean isGranted = "Yes".equalsIgnoreCase(accessGranted);
                newCache.put(apiKey, isGranted);
                if (!isGranted) continue;
                ++validKeysCount;
            }
            this.apiKeyCache = newCache;
            this.isHealthy = true;
            this.lastSuccessfulRefresh = System.currentTimeMillis();
            String message = String.format("Successfully refreshed %d valid API keys from Google Sheets (total entries: %d, skipped: %d)", validKeysCount, newCache.size(), skippedRows);
            FRLogger.info(message);
        }
        catch (IOException e) {
            FRLogger.error("Failed to refresh API keys from Google Sheets. Using cached data.", null, e);
            this.isHealthy = false;
        }
    }

    private String getFirstSheetName() throws IOException {
        Spreadsheet spreadsheet = (Spreadsheet)this.sheetsService.spreadsheets().get(this.spreadsheetId).setKey(this.googleApiKey).setFields("sheets.properties.title").execute();
        if (spreadsheet.getSheets() == null || spreadsheet.getSheets().isEmpty()) {
            throw new IOException("No sheets found in spreadsheet");
        }
        String sheetName = spreadsheet.getSheets().get(0).getProperties().getTitle();
        if (sheetName == null || sheetName.isEmpty()) {
            throw new IOException("First sheet has no title");
        }
        return sheetName;
    }

    private int findColumnIndex(List<Object> headers, String headerName) {
        for (int i = 0; i < headers.size(); ++i) {
            if (!headers.get(i).toString().trim().equalsIgnoreCase(headerName)) continue;
            return i;
        }
        return -1;
    }

    @Override
    public String getProviderName() {
        return "Google Sheets";
    }

    @Override
    public boolean isHealthy() {
        return this.isHealthy;
    }

    public int getCacheSize() {
        return this.apiKeyCache.size();
    }

    public long getLastSuccessfulRefresh() {
        return this.lastSuccessfulRefresh;
    }

    public void shutdown() {
        if (this.scheduler != null && !this.scheduler.isShutdown()) {
            this.scheduler.shutdown();
            try {
                if (!this.scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.scheduler.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.scheduler.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        FRLogger.info("Google Sheets API key provider shut down");
    }
}

