# 如何优雅的记录日记


实际开发过程中，有很多需要记录日志的需求。比如，订单数据修改，需要记录修改前后的快照数据。

如何去优雅的记录日志呢？


- canal 监听binlog
- 如果代码逻辑比较简单，可以使用AOP注解的方式，去记录日志，可以参考 【https://tech.meituan.com/2021/09/16/operational-logbook.html】
- 如果逻辑比较复杂，可以添加mybatis拦截器
- 如果觉得mybatis拦截器有性能问题，可以直接在代码异步组装数据


## 下面是mybatis拦截器的思路代码

```java

@Intercepts({@Signature(type = Executor.class, method = "update", args = {MappedStatement.class, Object.class})})
public class OrderStatisticInterceptor implements Interceptor {


    private static final List<String> TARGET_TABLES = listOf(" o_order ", " o_order_cancel ");


    private final OrderStatisticBiz statisticBiz;

    public OrderStatisticInterceptor(OrderStatisticBiz statisticBiz) {
        this.statisticBiz = statisticBiz;
    }

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        Object[] args = invocation.getArgs();
        MappedStatement ms = (MappedStatement) args[0];
        Object parameter = args[1];

        Executor executor = (Executor) invocation.getTarget();
        Object result = executor.update(ms, parameter);

        afterExecute(ms, parameter, result);

        return result;
    }

    @Override
    public Object plugin(Object target) {
        return Plugin.wrap(target, this);
    }


    @Override
    public void setProperties(Properties properties) {
    }


    private void afterExecute(MappedStatement ms, Object parameter, Object result) {

        SqlCommandType type = ms.getSqlCommandType();
        String sql = ms.getBoundSql(parameter).getSql();

        // 只对更新或写入操作进行处理
        if (result instanceof Integer) {
            Integer r = (Integer) result;
            // 只对更新成功的记录进行操作
            if (0 == r) {
                return;
            }
        } else {
            return;
        }

        // 只对目标表进行处理
        if (!isTarget(sql)) {
            return;
        }


        System.out.println("==============>>>> CommandType: " + type);
        System.out.println("==============>>>> Parameters: " + JsonKit.toJson(parameter));
        System.out.println("==============>>>> Result: " + result);

        Long orderId = this.takeOrderId(parameter);

        this.statisticBiz.statistic(orderId);
    }


    private Long takeOrderId(Object parameter) {
        if (parameter instanceof Order) {
            return ((Order) parameter).getOrderCenterId();
        }
        if (parameter instanceof OrderCancel) {
            return ((OrderCancel) parameter).getOrderCenterId();
        }
        if (parameter instanceof UpdateWrapper) {
            UpdateWrapper uw = (UpdateWrapper) parameter;
            System.out.println("==============>>>> Parameters: " + toJson(uw.getParamNameValuePairs()));
        }
        return 0L;

    }


    /**
     * 判断是否是目标表
     *
     * @param sql SQL语句
     * @return 是否是目标表
     */
    private boolean isTarget(String sql) {
        for (String table : TARGET_TABLES) {
            if (sql.contains(table)) {
                return true;
            }
        }
        return false;
    }

}
```




