<p data-nodeid="673" class="">缓存（Cache）是指将程序或系统中常用的数据对象存储在像内存这样特定的介质中，以避免在每次程序调用时，重新创建或组织数据所带来的性能损耗，从而提高了系统的整体运行速度。</p>
<p data-nodeid="674">以目前的系统架构来说，用户的请求一般会先经过缓存系统，如果缓存中没有相关的数据，就会在其他系统中查询到相应的数据并保存在缓存中，最后返回给调用方。</p>
<p data-nodeid="675">缓存既然如此重要，那本课时我们就来重点看一下，应该如何实现本地缓存和分布式缓存？</p>
<h3 data-nodeid="676">典型回答</h3>
<p data-nodeid="677">本地缓存是指程序级别的缓存组件，它的特点是本地缓存和应用程序会运行在同一个进程中，所以本地缓存的操作会非常快，因为在同一个进程内也意味着不会有网络上的延迟和开销。</p>
<p data-nodeid="678">本地缓存适用于单节点非集群的应用场景，它的优点是快，缺点是多程序无法共享缓存，比如分布式用户 Session 会话信息保存，由于每次用户访问的服务器可能是不同的，如果不能共享缓存，那么就意味着每次的请求操作都有可能被系统阻止，因为会话信息只保存在某一个服务器上，当请求没有被转发到这台存储了用户信息的服务器时，就会被认为是非登录的违规操作。</p>
<p data-nodeid="679">除此之外，无法共享缓存可能会造成系统资源的浪费，这是因为每个系统都单独维护了一份属于自己的缓存，而同一份缓存有可能被多个系统单独进行存储，从而浪费了系统资源。</p>
<p data-nodeid="680"><strong data-nodeid="762">分布式缓存是指将应用系统和缓存组件进行分离的缓存机制</strong>，这样多个应用系统就可以共享一套缓存数据了，它的特点是共享缓存服务和可集群部署，为缓存系统提供了高可用的运行环境，以及缓存共享的程序运行机制。</p>
<p data-nodeid="681">本地缓存可以使用 EhCache 和 Google 的 Guava 来实现，而分布式缓存可以使用 Redis 或 Memcached 来实现。</p>
<p data-nodeid="682">由于 Redis 本身就是独立的缓存系统，因此可以作为第三方来提供共享的数据缓存，而 Redis 的分布式支持主从、哨兵和集群的模式，所以它就可以支持分布式的缓存，而 Memcached 的情况也是类似的。</p>
<h3 data-nodeid="683">考点分析</h3>
<p data-nodeid="684">本课时的面试题显然不只是为了问你如何实现本地缓存和分布式缓存这么简单，主要考察的是你对缓存系统的理解，以及对缓存本质原理的洞察，和缓存相关的面试题还有这些：</p>
<ul data-nodeid="685">
<li data-nodeid="686">
<p data-nodeid="687">更加深入的谈谈 EhCache 和 Guava。</p>
</li>
<li data-nodeid="688">
<p data-nodeid="689">如何自己手动实现一个缓存系统？</p>
</li>
</ul>
<h3 data-nodeid="690">知识扩展</h3>
<h4 data-nodeid="691">1. EhCache 和 Guava 的使用及特点分析</h4>
<p data-nodeid="692">EhCache 是目前比较流行的开源缓存框架，是用纯 Java 语言实现的简单、快速的 Cache 组件。EhCache 支持内存缓存和磁盘缓存，支持 LRU（Least Recently Used，最近很少使用）、LFU（Least Frequently Used，最近不常被使用）和 FIFO（First In First Out，先进先出）等多种淘汰算法，并且支持分布式的缓存系统。</p>
<p data-nodeid="693">EhCache 最初是独立的本地缓存框架组件，在后期的发展中（从 1.2 版）开始支持分布式缓存，分布式缓存主要支持 RMI、JGroups、EhCache Server 等方式。</p>
<p data-nodeid="694"><strong data-nodeid="778">LRU 和 LFU 的区别</strong></p>
<p data-nodeid="695">LRU 算法有一个缺点，比如说很久没有使用的一个键值，如果最近被访问了一次，那么即使它是使用次数最少的缓存，它也不会被淘汰；而 LFU 算法解决了偶尔被访问一次之后，数据就不会被淘汰的问题，它是根据总访问次数来淘汰数据的，其核心思想是“如果数据过去被访问多次，那么将来它被访问次数也会比较多”。因此 LFU 可以理解为比 LRU 更加合理的淘汰算法。</p>
<p data-nodeid="696"><strong data-nodeid="783">EhCache 基础使用</strong></p>
<p data-nodeid="697">首先，需要在项目中添加 EhCache 框架，如果为 Maven 项目，则需要在 pom.xml 中添加如下配置：</p>
<pre class="lang-html" data-nodeid="698"><code data-language="html"><span class="hljs-comment">&lt;!--&nbsp;https://mvnrepository.com/artifact/org.ehcache/ehcache&nbsp;--&gt;</span>
<span class="hljs-tag">&lt;<span class="hljs-name">dependency</span>&gt;</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">groupId</span>&gt;</span>org.ehcache<span class="hljs-tag">&lt;/<span class="hljs-name">groupId</span>&gt;</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">artifactId</span>&gt;</span>ehcache<span class="hljs-tag">&lt;/<span class="hljs-name">artifactId</span>&gt;</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">version</span>&gt;</span>3.8.1<span class="hljs-tag">&lt;/<span class="hljs-name">version</span>&gt;</span>
<span class="hljs-tag">&lt;/<span class="hljs-name">dependency</span>&gt;</span>
</code></pre>
<p data-nodeid="699">无配置参数的 EhCache 3.x 使用代码如下：</p>
<pre class="lang-java" data-nodeid="700"><code data-language="java"><span class="hljs-keyword">import</span>&nbsp;org.ehcache.Cache;
<span class="hljs-keyword">import</span>&nbsp;org.ehcache.CacheManager;
<span class="hljs-keyword">import</span>&nbsp;org.ehcache.config.builders.CacheConfigurationBuilder;
<span class="hljs-keyword">import</span>&nbsp;org.ehcache.config.builders.CacheManagerBuilder;
<span class="hljs-keyword">import</span>&nbsp;org.ehcache.config.builders.ResourcePoolsBuilder;

<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">EhCacheExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建缓存管理器</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;CacheManager&nbsp;cacheManager&nbsp;=&nbsp;CacheManagerBuilder.newCacheManagerBuilder().build();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;初始化&nbsp;EhCache</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;cacheManager.init();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建缓存（存储器）</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Cache&lt;String,&nbsp;String&gt;&nbsp;myCache&nbsp;=&nbsp;cacheManager.createCache(<span class="hljs-string">"MYCACHE"</span>,
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;CacheConfigurationBuilder.newCacheConfigurationBuilder(
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String.class,&nbsp;String.class,
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;ResourcePoolsBuilder.heap(<span class="hljs-number">10</span>)));&nbsp;<span class="hljs-comment">//&nbsp;设置缓存的最大容量</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;设置缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;myCache.put(<span class="hljs-string">"key"</span>,&nbsp;<span class="hljs-string">"Hello,Java."</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;读取缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String&nbsp;value&nbsp;=&nbsp;myCache.get(<span class="hljs-string">"key"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;输出缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(value);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;关闭缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;cacheManager.close();
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="701">其中：</p>
<ul data-nodeid="702">
<li data-nodeid="703">
<p data-nodeid="704">CacheManager：是缓存管理器，可以通过单例或者多例的方式创建，也是 Ehcache 的入口类；</p>
</li>
<li data-nodeid="705">
<p data-nodeid="706">Cache：每个 CacheManager 可以管理多个 Cache，每个 Cache 可以采用 hash 的方式存储多个元素。</p>
</li>
</ul>
<p data-nodeid="707">它们的关系如下图所示：<br>
<img src="https://s0.lgstatic.com/i/image3/M01/0C/17/Ciqah16NsS2AUntaAAA6FUv7b_I176.png" alt="" data-nodeid="792"></p>
<p data-nodeid="708">更多使用方法，<a href="http://www.ehcache.org/documentation/3.8/getting-started.html" data-nodeid="796">请参考官方文档</a>。</p>
<p data-nodeid="709">EhCache 的特点是，它使用起来比较简单，并且本身的 jar 包不是不大，简单的配置之后就可以正常使用了。EhCache 的使用比较灵活，它支持多种缓存策略的配置，它同时支持内存和磁盘缓存两种方式，在 EhCache 1.2 之后也开始支持分布式缓存了。</p>
<p data-nodeid="710">Guava Cache 是 Google 开源的 Guava 里的一个子功能，它是一个内存型的本地缓存实现方案，提供了线程安全的缓存操作机制。</p>
<p data-nodeid="711">Guava Cache 的架构设计灵感来源于 ConcurrentHashMap，它使用了多个 segments 方式的细粒度锁，在保证线程安全的同时，支持了高并发的使用场景。Guava Cache 类似于 Map 集合的方式对键值对进行操作，只不过多了过期淘汰等处理逻辑。</p>
<p data-nodeid="712">在使用 Guava Cache 之前，我们需要先在 pom.xml 中添加 Guava 框架，配置如下：</p>
<pre class="lang-html" data-nodeid="713"><code data-language="html"><span class="hljs-comment">&lt;!--&nbsp;https://mvnrepository.com/artifact/com.google.guava/guava&nbsp;--&gt;</span>
<span class="hljs-tag">&lt;<span class="hljs-name">dependency</span>&gt;</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">groupId</span>&gt;</span>com.google.guava<span class="hljs-tag">&lt;/<span class="hljs-name">groupId</span>&gt;</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">artifactId</span>&gt;</span>guava<span class="hljs-tag">&lt;/<span class="hljs-name">artifactId</span>&gt;</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">version</span>&gt;</span>28.2-jre<span class="hljs-tag">&lt;/<span class="hljs-name">version</span>&gt;</span>
<span class="hljs-tag">&lt;/<span class="hljs-name">dependency</span>&gt;</span>
</code></pre>
<p data-nodeid="714">Guava Cache 的创建有两种方式，一种是 LoadingCache，另一种是 Callable，代码示例如下：</p>
<pre class="lang-java" data-nodeid="715"><code data-language="java"><span class="hljs-keyword">import</span>&nbsp;com.google.common.cache.*;

<span class="hljs-keyword">import</span>&nbsp;java.util.concurrent.Callable;
<span class="hljs-keyword">import</span>&nbsp;java.util.concurrent.ExecutionException;
<span class="hljs-keyword">import</span>&nbsp;java.util.concurrent.TimeUnit;

<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">GuavaExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;ExecutionException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建方式一：LoadingCache</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;LoadingCache&lt;String,&nbsp;String&gt;&nbsp;loadCache&nbsp;=&nbsp;CacheBuilder.newBuilder()
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;并发级别设置为&nbsp;5，是指可以同时写缓存的线程数</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.concurrencyLevel(<span class="hljs-number">5</span>)
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;设置&nbsp;8&nbsp;秒钟过期</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.expireAfterWrite(<span class="hljs-number">8</span>,&nbsp;TimeUnit.SECONDS)
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//设置缓存容器的初始容量为&nbsp;10</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.initialCapacity(<span class="hljs-number">10</span>)
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;设置缓存最大容量为&nbsp;100，超过之后就会按照&nbsp;LRU&nbsp;算法移除缓存项</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.maximumSize(<span class="hljs-number">100</span>)
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;设置要统计缓存的命中率</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.recordStats()
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;设置缓存的移除通知</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.removalListener(<span class="hljs-keyword">new</span>&nbsp;RemovalListener&lt;Object,&nbsp;Object&gt;()&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">onRemoval</span><span class="hljs-params">(RemovalNotification&lt;Object,&nbsp;Object&gt;&nbsp;notification)</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(notification.getKey()&nbsp;+&nbsp;<span class="hljs-string">"&nbsp;was&nbsp;removed,&nbsp;cause&nbsp;is&nbsp;"</span>&nbsp;+&nbsp;notification.getCause());
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;})
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;指定&nbsp;CacheLoader，缓存不存在时，可自动加载缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.build(
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">new</span>&nbsp;CacheLoader&lt;String,&nbsp;String&gt;()&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-meta">@Override</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;String&nbsp;<span class="hljs-title">load</span><span class="hljs-params">(String&nbsp;key)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;Exception&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;自动加载缓存的业务</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;<span class="hljs-string">"cache-value:"</span>&nbsp;+&nbsp;key;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;设置缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;loadCache.put(<span class="hljs-string">"c1"</span>,&nbsp;<span class="hljs-string">"Hello,&nbsp;c1."</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;查询缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String&nbsp;val&nbsp;=&nbsp;loadCache.get(<span class="hljs-string">"c1"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(val);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;查询不存在的缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String&nbsp;noval&nbsp;=&nbsp;loadCache.get(<span class="hljs-string">"noval"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(noval);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建方式二：Callable</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Cache&lt;String,&nbsp;String&gt;&nbsp;cache&nbsp;=&nbsp;CacheBuilder.newBuilder()
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.maximumSize(<span class="hljs-number">2</span>)&nbsp;<span class="hljs-comment">//&nbsp;设置缓存最大长度</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;.build();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;设置缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;cache.put(<span class="hljs-string">"k1"</span>,&nbsp;<span class="hljs-string">"Hello,&nbsp;k1."</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;查询缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String&nbsp;value&nbsp;=&nbsp;cache.get(<span class="hljs-string">"k1"</span>,&nbsp;<span class="hljs-keyword">new</span>&nbsp;Callable&lt;String&gt;()&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-meta">@Override</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;String&nbsp;<span class="hljs-title">call</span><span class="hljs-params">()</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;缓存不存在时，执行</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;<span class="hljs-string">"nil"</span>;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;});
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;输出缓存值</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(value);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;查询缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String&nbsp;nokey&nbsp;=&nbsp;cache.get(<span class="hljs-string">"nokey"</span>,&nbsp;<span class="hljs-keyword">new</span>&nbsp;Callable&lt;String&gt;()&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-meta">@Override</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;String&nbsp;<span class="hljs-title">call</span><span class="hljs-params">()</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;缓存不存在时，执行</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;<span class="hljs-string">"nil"</span>;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;});
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;输出缓存值</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(nokey);
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="716">以上程序的执行结果为：</p>
<pre class="lang-java" data-nodeid="717"><code data-language="java">Hello,&nbsp;c1.
cache-value:noval
Hello,&nbsp;k1.
nil
</code></pre>
<p data-nodeid="718">可以看出 Guava Cache 使用了编程式的 build 生成器进行创建和管理，让使用者可以更加灵活地操纵代码，并且 Guava Cache 提供了灵活多样的个性化配置，以适应各种使用场景。</p>
<h4 data-nodeid="719">2. 手动实现一个缓存系统</h4>
<p data-nodeid="720">上面我们讲了通过 EhCache 和 Guava 实现缓存的方式，接下来我们来看看自己如何自定义一个缓存系统，当然这里说的是自己手动实现一个本地缓存。</p>
<p data-nodeid="721">要自定义一个缓存，首先要考虑的是数据类型，我们可以使用 Map 集合中的 HashMap、Hashtable 或 ConcurrentHashMap 来实现，非并发情况下我们可以使用 HashMap，并发情况下可以使用 Hashtable 或 ConcurrentHashMap，由于 ConcurrentHashMap 的性能比 Hashtable 的高，因此在高并发环境下我们可以倾向于选择 ConcurrentHashMap，不过它们对元素的操作都是类似的。</p>
<p data-nodeid="722">选定了数据类型之后，我们还需要考虑缓存过期和缓存淘汰等问题，在这里我们可以借鉴 Redis 对待过期键的处理策略。</p>
<p data-nodeid="723">目前比较常见的过期策略有以下三种：</p>
<ul data-nodeid="724">
<li data-nodeid="725">
<p data-nodeid="726">定时删除</p>
</li>
<li data-nodeid="727">
<p data-nodeid="728">惰性删除</p>
</li>
<li data-nodeid="729">
<p data-nodeid="730">定期删除</p>
</li>
</ul>
<p data-nodeid="731"><strong data-nodeid="819">定时删除</strong>是指在设置键值的过期时间时，创建一个定时事件，当到达过期时间后，事件处理器会执行删除过期键的操作。它的优点是可以及时的释放内存空间，缺点是需要开启多个延迟执行事件来处理清除任务，这样就会造成大量任务事件堆积，占用了很多系统资源。</p>
<p data-nodeid="732"><strong data-nodeid="824">惰性删除</strong>不会主动删除过期键，而是在每次请求时才会判断此值是否过期，如果过期则删除键值，否则就返回正常值。它的优点是只会占用少量的系统资源，缺点是清除不够及时，会造成一定的空间浪费。</p>
<p data-nodeid="733"><strong data-nodeid="829">定期删除</strong>是指每隔一段时间检查一次数据库，随机删除一些过期键值。</p>
<p data-nodeid="734">Redis 使用的是定期删除和惰性删除这两种策略，我们本课时也会参照这两种策略。</p>
<p data-nodeid="735">先来说一下自定义缓存的实现思路，首先需要定义一个存放缓存值的实体类，这个类里包含了缓存的相关信息，比如缓存的 key 和 value，缓存的存入时间、最后使用时间和命中次数（预留字段，用于支持 LFU 缓存淘汰），再使用 ConcurrentHashMap 保存缓存的 key 和 value 对象（缓存值的实体类），然后再新增一个缓存操作的工具类，用于添加和删除缓存，最后再缓存启动时，开启一个无限循环的线程用于检测并删除过期的缓存，实现代码如下。</p>
<p data-nodeid="736">首先，定义一个缓存值实体类，代码如下：</p>
<pre class="lang-java te-preview-highlight" data-nodeid="4885"><code data-language="java"><span class="hljs-keyword">import</span> lombok.Getter;
<span class="hljs-keyword">import</span> lombok.Setter;

<span class="hljs-comment">/**
 * 缓存实体类
 */</span>
<span class="hljs-meta">@Getter</span>
<span class="hljs-meta">@Setter</span>
<span class="hljs-keyword">public</span> <span class="hljs-class"><span class="hljs-keyword">class</span> <span class="hljs-title">MyCache</span> <span class="hljs-keyword">implements</span> <span class="hljs-title">Comparable</span>&lt;<span class="hljs-title">MyCache</span>&gt; </span>{
    <span class="hljs-comment">// 缓存键</span>
    <span class="hljs-keyword">private</span> Object key;
    <span class="hljs-comment">// 缓存值</span>
    <span class="hljs-keyword">private</span> Object value;
    <span class="hljs-comment">// 最后访问时间</span>
    <span class="hljs-keyword">private</span> <span class="hljs-keyword">long</span> lastTime;
    <span class="hljs-comment">// 创建时间</span>
    <span class="hljs-keyword">private</span> <span class="hljs-keyword">long</span> writeTime;
    <span class="hljs-comment">// 存活时间</span>
    <span class="hljs-keyword">private</span> <span class="hljs-keyword">long</span> expireTime;
    <span class="hljs-comment">// 命中次数</span>
    <span class="hljs-keyword">private</span> Integer hitCount;

    <span class="hljs-meta">@Override</span>
    <span class="hljs-function"><span class="hljs-keyword">public</span> <span class="hljs-keyword">int</span> <span class="hljs-title">compareTo</span><span class="hljs-params">(MyCache o)</span> </span>{
        <span class="hljs-keyword">return</span> hitCount.compareTo(o.hitCount);
    }
}
</code></pre>













<p data-nodeid="738">然后定义一个全局缓存对象，代码如下：</p>
<pre class="lang-java" data-nodeid="739"><code data-language="java"><span class="hljs-keyword">import</span>&nbsp;java.util.concurrent.ConcurrentHashMap;
<span class="hljs-keyword">import</span>&nbsp;java.util.concurrent.ConcurrentMap;

<span class="hljs-comment">/**
&nbsp;*&nbsp;Cache&nbsp;全局类
&nbsp;*/</span>
<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">CacheGlobal</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;全局缓存对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;ConcurrentMap&lt;String,&nbsp;MyCache&gt;&nbsp;concurrentMap&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;ConcurrentHashMap&lt;&gt;();
}
</code></pre>
<p data-nodeid="740">定义过期缓存检测类的代码如下：</p>
<pre class="lang-java" data-nodeid="741"><code data-language="java"><span class="hljs-keyword">import</span>&nbsp;java.util.concurrent.TimeUnit;

<span class="hljs-comment">/**
&nbsp;*&nbsp;过期缓存检测线程
&nbsp;*/</span>
<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">ExpireThread</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Runnable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-meta">@Override</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">run</span><span class="hljs-params">()</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">while</span>&nbsp;(<span class="hljs-keyword">true</span>)&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">try</span>&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;每十秒检测一次</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;TimeUnit.SECONDS.sleep(<span class="hljs-number">10</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;缓存检测和清除的方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;expireCache();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}&nbsp;<span class="hljs-keyword">catch</span>&nbsp;(Exception&nbsp;e)&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;e.printStackTrace();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;缓存检测和清除的方法
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">private</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">expireCache</span><span class="hljs-params">()</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(<span class="hljs-string">"检测缓存是否过期缓存"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">for</span>&nbsp;(String&nbsp;key&nbsp;:&nbsp;CacheGlobal.concurrentMap.keySet())&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;MyCache&nbsp;cache&nbsp;=&nbsp;CacheGlobal.concurrentMap.get(key);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;当前时间&nbsp;-&nbsp;写入时间</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">long</span>&nbsp;timoutTime&nbsp;=&nbsp;System.currentTimeMillis() - cache.getWriteTime();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">if</span>&nbsp;(cache.getExpireTime()&nbsp;&gt;&nbsp;timoutTime)&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;没过期</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">continue</span>;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;清除过期缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;CacheGlobal.concurrentMap.remove(key);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="742">接着，我们要新增一个缓存操作的工具类，用于查询和存入缓存，实现代码如下：</p>
<pre class="lang-java" data-nodeid="743"><code data-language="java"><span class="hljs-keyword">import</span> org.apache.commons.lang3.StringUtils;

<span class="hljs-keyword">import</span> java.util.concurrent.TimeUnit;

<span class="hljs-comment">/**
 * 缓存操作工具类
 */</span>
<span class="hljs-keyword">public</span> <span class="hljs-class"><span class="hljs-keyword">class</span> <span class="hljs-title">CacheUtils</span> </span>{

    <span class="hljs-comment">/**
     * 添加缓存
     * <span class="hljs-doctag">@param</span> key
     * <span class="hljs-doctag">@param</span> value
     * <span class="hljs-doctag">@param</span> expire
     */</span>
    <span class="hljs-function"><span class="hljs-keyword">public</span> <span class="hljs-keyword">void</span> <span class="hljs-title">put</span><span class="hljs-params">(String key, Object value, <span class="hljs-keyword">long</span> expire)</span> </span>{
        <span class="hljs-comment">// 非空判断，借助 commons-lang3</span>
        <span class="hljs-keyword">if</span> (StringUtils.isBlank(key)) <span class="hljs-keyword">return</span>;
        <span class="hljs-comment">// 当缓存存在时，更新缓存</span>
        <span class="hljs-keyword">if</span> (CacheGlobal.concurrentMap.containsKey(key)) {
            MyCache cache = CacheGlobal.concurrentMap.get(key);
            cache.setHitCount(cache.getHitCount() + <span class="hljs-number">1</span>);
            cache.setWriteTime(System.currentTimeMillis());
            cache.setLastTime(System.currentTimeMillis());
            cache.setExpireTime(expire);
            cache.setValue(value);
            <span class="hljs-keyword">return</span>;
        }
        <span class="hljs-comment">// 创建缓存</span>
        MyCache cache = <span class="hljs-keyword">new</span> MyCache();
        cache.setKey(key);
        cache.setValue(value);
        cache.setWriteTime(System.currentTimeMillis());
        cache.setLastTime(System.currentTimeMillis());
        cache.setHitCount(<span class="hljs-number">1</span>);
        cache.setExpireTime(expire);
        CacheGlobal.concurrentMap.put(key, cache);
    }

    <span class="hljs-comment">/**
     * 获取缓存
     * <span class="hljs-doctag">@param</span> key
     * <span class="hljs-doctag">@return</span>
     */</span>
    <span class="hljs-function"><span class="hljs-keyword">public</span> Object <span class="hljs-title">get</span><span class="hljs-params">(String key)</span> </span>{
        <span class="hljs-comment">// 非空判断</span>
        <span class="hljs-keyword">if</span> (StringUtils.isBlank(key)) <span class="hljs-keyword">return</span> <span class="hljs-keyword">null</span>;
        <span class="hljs-comment">// 字典中不存在</span>
        <span class="hljs-keyword">if</span> (CacheGlobal.concurrentMap.isEmpty()) <span class="hljs-keyword">return</span> <span class="hljs-keyword">null</span>;
        <span class="hljs-keyword">if</span> (!CacheGlobal.concurrentMap.containsKey(key)) <span class="hljs-keyword">return</span> <span class="hljs-keyword">null</span>;
        MyCache cache = CacheGlobal.concurrentMap.get(key);
        <span class="hljs-keyword">if</span> (cache == <span class="hljs-keyword">null</span>) <span class="hljs-keyword">return</span> <span class="hljs-keyword">null</span>;
        <span class="hljs-comment">// 惰性删除，判断缓存是否过期</span>
        <span class="hljs-keyword">long</span> timoutTime = System.currentTimeMillis() - cache.getWriteTime();
        <span class="hljs-comment">// 缓存过期</span>
        <span class="hljs-keyword">if</span> (cache.getExpireTime() &lt;= timoutTime) {
            <span class="hljs-comment">// 清除过期缓存</span>
            CacheGlobal.concurrentMap.remove(key);
            <span class="hljs-keyword">return</span> <span class="hljs-keyword">null</span>;
        }
        cache.setHitCount(cache.getHitCount() + <span class="hljs-number">1</span>);
        cache.setLastTime(System.currentTimeMillis());
        <span class="hljs-keyword">return</span> cache.getValue();
    }
}
</code></pre>
<p data-nodeid="744">最后是调用缓存的测试代码：</p>
<pre class="lang-java" data-nodeid="745"><code data-language="java"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">MyCacheTest</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;CacheUtils&nbsp;cache&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;CacheUtils();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;存入缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;cache.put(<span class="hljs-string">"key"</span>,&nbsp;<span class="hljs-string">"老王"</span>,&nbsp;<span class="hljs-number">10</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;查询缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String&nbsp;val&nbsp;=&nbsp;(String)&nbsp;cache.get(<span class="hljs-string">"key"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(val);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;查询不存在的缓存</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;String&nbsp;noval&nbsp;=&nbsp;(String)&nbsp;cache.get(<span class="hljs-string">"noval"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(noval);
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="746">以上程序的执行结果如下：</p>
<pre class="lang-java" data-nodeid="747"><code data-language="java">老王
<span class="hljs-keyword">null</span>
</code></pre>
<p data-nodeid="748">到目前为止，自定义缓存系统就已经实现完了。</p>
<h3 data-nodeid="749">小结</h3>
<p data-nodeid="750" class="">本课时讲解了本地缓存和分布式缓存这两个概念和实现的具体方式，其中本地缓存可以通过自己手动编码或借助 Guava Cache 来实现，而分布式缓存可以使用 Redis 或 EhCache 来实现。此外，本课时重点演示了手动实现缓存代码的方式和实现思路，并使用定期删除和惰性删除策略来实现缓存的清除，希望学完本课时后能对你有所帮助。</p>

---

### 精选评论

##### **远：
> 定期删除，遍历keyset再get是不是换成直接遍历entry要高效些？

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 可以看下我的这篇《HashMap 的 7 种遍历方式与性能分析！》有专门的性能分析哈 https://mp.weixin.qq.com/s?__biz=MzU1NTkwODE4Mw==&mid=2247485208&idx=1&sn=62be917b1431243b898354d17112b06f&scene=21#wechat_redirect

##### **威：
> 老师，我想问下文章中的本地缓存和 spring cache 有什么关联么？Spring cache 也可以缓存到本地的

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; spring 内置 cache 属于本地缓存的一种，它的本质是使用 map 进行数据存储的。

##### **伟：
> 1.CacheUtils里的put和get方法，在并发情况下，会出现问题。比如，在put方法中执行到MyCache cache = CacheGlobal.concurrentMap.get(key);之后，进行设置值，还没有设置完，另一个线程就调用get方法，这时候，get到的MyCache值中的一些属性是错乱的。<div>2.get方法只是返回的value值，1中的问题会导致value获取不到另一个线程中设置的值。所以<span style="font-size: 16.0125px;">CacheValue的value值是不是设置成valitale比较好。</span></div>

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 很棒，这是一个简易版的 Cache 类只是为了说明过期策略的。

##### *飞：
> public int compareTo(CacheValue o) {
        return hitCount.compareTo(o.hitCount);
}这里的 CacheValue 这个类是从哪来的？

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 这里不是 CacheValue 对象而是 MyCache 对象，内容以更改，课程的所有源码在 github：https://github.com/vipstone/interview-lagou/blob/master/src/main/java/com/lagou/interview/custom/MyCache.java

##### **帅：
> 惰性删除不会主动删除过期键，而是在每次请求时才会判断此值是否过期，如果过期则删除键值，否则就返回 null。 这里如果过期的话删除键值 返回null，否则不是应该返回正确的结果嘛？为什么否则就返回null 呢😁

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 感谢反馈，笔误，这块应该是“否则就返回正常值”，文中已更正。

##### **耀：
> 回看第二遍，非常有帮组，有兴趣的同学可以补充了解如下知识点：1.强大的 Apache Ignite（是一个以内存为中心的分布式数据库、缓存和处理平台，可以在PB级数据中，以内存级的速度进行事务性、分析性以及流式负载的处理）2.新一代的缓存Caffeine(性能比Google Guava好)。作者的github是(ben-manes - Overview)，曾经写了ConcurrentLinkedHashMap这个类，而这个类又是GuavaCache的基础。Ben Manes一拍脑袋，决定更上层楼。

##### **烨：
> 自己实现的缓存代码里面concurrentMap的类型是ConcurrentMap，该接口不存在keySet()方法，keySet()方法是AbstractMap抽象类中定义的

##### **明：
> &nbsp;&nbsp;<span style="font-size: 0.427rem;">// 当缓存存在时，更新缓存</span><div><div>&nbsp; &nbsp; &nbsp; &nbsp; if (CacheGlobal.concurrentMap.containsKey(key)) {</div><div>// 如果两个线程同时执行到这个位置,都已经得到了cache对象,这个时候cache的hitcount = 11</div><div>&nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; MyCache cache = CacheGlobal.concurrentMap.get(key);</div><div>// 这个时候两个线程交替或者并发对这个hitcount加1,hitcount就会=12.其实应该是13的</div><div>// 这个时候每个线程都保存有cache这个副本,相互独立,感觉这个地方有问题</div><div>&nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; cache.setHitCount(cache.getHitCount() + 1);</div><div>&nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; cache.setWriteTime(System.currentTimeMillis());</div><div>&nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; cache.setLastTime(System.currentTimeMillis());</div><div>&nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; cache.setExpireTime(expire);</div><div>&nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; cache.setValue(value);</div><div>&nbsp; &nbsp; &nbsp; &nbsp; &nbsp; &nbsp; return;</div><div>&nbsp; &nbsp; &nbsp; &nbsp; }</div></div><div><br></div>

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 这是一个简易版的 Cache 类，其实只是为了演示过期策略的哈。

##### *行：
> 惰性删除 清除过期缓存是不是写错分支了

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 嗯？么有写错啊

##### **生：
> 老师讲的定时或定期删除搞乱了。。

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 是比较绕，可以多看两遍就清楚了。

##### *鹏：
> 获取缓存值方法里，清除过期缓存的位置不正确

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 您指的是哪一行代码，麻烦复制一下。

##### **湘：
> 确实会有并发问题，ConcurrentHashMap 只保证了MyCache 不会有并发问题，但MyCache 里面的属性值却不能保证，并发情况下会导致MyCache 属性的值混乱

##### **乐：
> 没什么说的 干货满满🙋

##### **用户8700：
> 虽然基于并发map，但手写的缓存没有考虑并发问题。而且contain方法有性能问题

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 自定义缓存使用的是 ConcurrentHashMap 不会有并发问题，HashMap 会有并发问题。containsKey 在 JDK 8 时因为数据结构在必要时会转成红黑树，所以不会有性能问题。如果是链表的话，最差的情况可以查询效率是 O(n)，可能会出现问题，但 JDK 8 之后不会了。

