<p data-nodeid="1" class="">使用克隆可以为我们快速地构建出一个已有对象的副本，它属于 Java 基础的一部分，也是面试中常被问到的知识点之一。</p>
<p data-nodeid="2">我们本课时的面试题是，什么是浅克隆和深克隆？如何实现克隆？</p>
<h3 data-nodeid="3">典型回答</h3>
<p data-nodeid="4"><strong data-nodeid="114">浅克隆</strong>（Shadow Clone）是把原型对象中成员变量为值类型的属性都复制给克隆对象，把原型对象中成员变量为引用类型的引用地址也复制给克隆对象，也就是原型对象中如果有成员变量为引用对象，则此引用对象的地址是共享给原型对象和克隆对象的。</p>
<p data-nodeid="5">简单来说就是浅克隆只会复制原型对象，但不会复制它所引用的对象，如下图所示：</p>
<p data-nodeid="6"><img src="https://s0.lgstatic.com/i/image3/M01/07/2A/Ciqah16EEHqAS-yBAACmrlSP9OU378.png" alt="" data-nodeid="117"></p>
<p data-nodeid="7"><strong data-nodeid="122">深克隆</strong>（Deep Clone）是将原型对象中的所有类型，无论是值类型还是引用类型，都复制一份给克隆对象，也就是说深克隆会把原型对象和原型对象所引用的对象，都复制一份给克隆对象，如下图所示：</p>
<p data-nodeid="8"><img src="https://s0.lgstatic.com/i/image3/M01/07/2A/Ciqah16EEHqACOHEAAB2WjY2QEQ611.png" alt="" data-nodeid="124"></p>
<p data-nodeid="9">在 Java 语言中要实现克隆则需要实现 Cloneable 接口，并重写 Object 类中的 clone() 方法，实现代码如下：</p>
<pre class="lang-java" data-nodeid="10"><code data-language="java"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">CloneExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建被赋值对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p1&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;People();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;p1.setId(<span class="hljs-number">1</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;p1.setName(<span class="hljs-string">"Java"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;克隆&nbsp;p1&nbsp;对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p2&nbsp;=&nbsp;(People)&nbsp;p1.clone();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;打印名称</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(<span class="hljs-string">"p2:"</span>&nbsp;+&nbsp;p2.getName());
&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">People</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Cloneable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;属性</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;重写&nbsp;clone&nbsp;方法
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;<span class="hljs-doctag">@throws</span>&nbsp;CloneNotSupportedException
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-meta">@Override</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">protected</span>&nbsp;Object&nbsp;<span class="hljs-title">clone</span><span class="hljs-params">()</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;<span class="hljs-keyword">super</span>.clone();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;Integer&nbsp;<span class="hljs-title">getId</span><span class="hljs-params">()</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">setId</span><span class="hljs-params">(Integer&nbsp;id)</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">this</span>.id&nbsp;=&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;String&nbsp;<span class="hljs-title">getName</span><span class="hljs-params">()</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">setName</span><span class="hljs-params">(String&nbsp;name)</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">this</span>.name&nbsp;=&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="11">以上程序执行的结果为：</p>
<pre class="lang-java" data-nodeid="12"><code data-language="java">p2:Java
</code></pre>
<h3 data-nodeid="13">考点分析</h3>
<p data-nodeid="14">克隆相关的面试题不算太难，但因为使用频率不高，因此很容易被人忽略，面试官通常会在一面或者二面的时候问到此知识点，和它相关的面试题还有以下这些：</p>
<ul data-nodeid="15">
<li data-nodeid="16">
<p data-nodeid="17">在 java.lang.Object 中对 clone() 方法的约定有哪些？</p>
</li>
<li data-nodeid="18">
<p data-nodeid="19">Arrays.copyOf() 是深克隆还是浅克隆？</p>
</li>
<li data-nodeid="20">
<p data-nodeid="21">深克隆的实现方式有几种？</p>
</li>
<li data-nodeid="22">
<p data-nodeid="23">Java 中的克隆为什么要设计成，既要实现空接口 Cloneable，还要重写 Object 的 clone() 方法？</p>
</li>
</ul>
<h3 data-nodeid="24">知识扩展</h3>
<h4 data-nodeid="25">clone() 源码分析</h4>
<p data-nodeid="26">要想真正的了解克隆，首先要从它的源码入手，代码如下：</p>
<pre class="lang-html" data-nodeid="27"><code data-language="html">/**
&nbsp;*&nbsp;Creates&nbsp;and&nbsp;returns&nbsp;a&nbsp;copy&nbsp;of&nbsp;this&nbsp;object.&nbsp;&nbsp;The&nbsp;precise&nbsp;meaning
&nbsp;*&nbsp;of&nbsp;"copy"&nbsp;may&nbsp;depend&nbsp;on&nbsp;the&nbsp;class&nbsp;of&nbsp;the&nbsp;object.&nbsp;The&nbsp;general
&nbsp;*&nbsp;intent&nbsp;is&nbsp;that,&nbsp;for&nbsp;any&nbsp;object&nbsp;{@code&nbsp;x},&nbsp;the&nbsp;expression:
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">blockquote</span>&gt;</span>
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">pre</span>&gt;</span>
&nbsp;*&nbsp;x.clone()&nbsp;!=&nbsp;x<span class="hljs-tag">&lt;/<span class="hljs-name">pre</span>&gt;</span><span class="hljs-tag">&lt;/<span class="hljs-name">blockquote</span>&gt;</span>
&nbsp;*&nbsp;will&nbsp;be&nbsp;true,&nbsp;and&nbsp;that&nbsp;the&nbsp;expression:
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">blockquote</span>&gt;</span>
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">pre</span>&gt;</span>
&nbsp;*&nbsp;x.clone().getClass()&nbsp;==&nbsp;x.getClass()<span class="hljs-tag">&lt;/<span class="hljs-name">pre</span>&gt;</span><span class="hljs-tag">&lt;/<span class="hljs-name">blockquote</span>&gt;</span>
&nbsp;*&nbsp;will&nbsp;be&nbsp;{@code&nbsp;true},&nbsp;but&nbsp;these&nbsp;are&nbsp;not&nbsp;absolute&nbsp;requirements.
&nbsp;*&nbsp;While&nbsp;it&nbsp;is&nbsp;typically&nbsp;the&nbsp;case&nbsp;that:
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">blockquote</span>&gt;</span>
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">pre</span>&gt;</span>
&nbsp;*&nbsp;x.clone().equals(x)<span class="hljs-tag">&lt;/<span class="hljs-name">pre</span>&gt;</span><span class="hljs-tag">&lt;/<span class="hljs-name">blockquote</span>&gt;</span>
&nbsp;*&nbsp;will&nbsp;be&nbsp;{@code&nbsp;true},&nbsp;this&nbsp;is&nbsp;not&nbsp;an&nbsp;absolute&nbsp;requirement.
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">p</span>&gt;</span>
&nbsp;*&nbsp;By&nbsp;convention,&nbsp;the&nbsp;returned&nbsp;object&nbsp;should&nbsp;be&nbsp;obtained&nbsp;by&nbsp;calling
&nbsp;*&nbsp;{@code&nbsp;super.clone}.&nbsp;&nbsp;If&nbsp;a&nbsp;class&nbsp;and&nbsp;all&nbsp;of&nbsp;its&nbsp;superclasses&nbsp;(except
&nbsp;*&nbsp;{@code&nbsp;Object})&nbsp;obey&nbsp;this&nbsp;convention,&nbsp;it&nbsp;will&nbsp;be&nbsp;the&nbsp;case&nbsp;that
&nbsp;*&nbsp;{@code&nbsp;x.clone().getClass()&nbsp;==&nbsp;x.getClass()}.
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">p</span>&gt;</span>
&nbsp;*&nbsp;By&nbsp;convention,&nbsp;the&nbsp;object&nbsp;returned&nbsp;by&nbsp;this&nbsp;method&nbsp;should&nbsp;be&nbsp;independent
&nbsp;*&nbsp;of&nbsp;this&nbsp;object&nbsp;(which&nbsp;is&nbsp;being&nbsp;cloned).&nbsp;&nbsp;To&nbsp;achieve&nbsp;this&nbsp;independence,
&nbsp;*&nbsp;it&nbsp;may&nbsp;be&nbsp;necessary&nbsp;to&nbsp;modify&nbsp;one&nbsp;or&nbsp;more&nbsp;fields&nbsp;of&nbsp;the&nbsp;object&nbsp;returned
&nbsp;*&nbsp;by&nbsp;{@code&nbsp;super.clone}&nbsp;before&nbsp;returning&nbsp;it.&nbsp;&nbsp;Typically,&nbsp;this&nbsp;means
&nbsp;*&nbsp;copying&nbsp;any&nbsp;mutable&nbsp;objects&nbsp;that&nbsp;comprise&nbsp;the&nbsp;internal&nbsp;"deep&nbsp;structure"
&nbsp;*&nbsp;of&nbsp;the&nbsp;object&nbsp;being&nbsp;cloned&nbsp;and&nbsp;replacing&nbsp;the&nbsp;references&nbsp;to&nbsp;these
&nbsp;*&nbsp;objects&nbsp;with&nbsp;references&nbsp;to&nbsp;the&nbsp;copies.&nbsp;&nbsp;If&nbsp;a&nbsp;class&nbsp;contains&nbsp;only
&nbsp;*&nbsp;primitive&nbsp;fields&nbsp;or&nbsp;references&nbsp;to&nbsp;immutable&nbsp;objects,&nbsp;then&nbsp;it&nbsp;is&nbsp;usually
&nbsp;*&nbsp;the&nbsp;case&nbsp;that&nbsp;no&nbsp;fields&nbsp;in&nbsp;the&nbsp;object&nbsp;returned&nbsp;by&nbsp;{@code&nbsp;super.clone}
&nbsp;*&nbsp;need&nbsp;to&nbsp;be&nbsp;modified.
&nbsp;*&nbsp;<span class="hljs-tag">&lt;<span class="hljs-name">p</span>&gt;</span>
&nbsp;*&nbsp;......
&nbsp;*/
protected&nbsp;native&nbsp;Object&nbsp;clone()&nbsp;throws&nbsp;CloneNotSupportedException;
</code></pre>
<p data-nodeid="28">从以上源码的注释信息中我们可以看出，Object 对 clone() 方法的约定有三条：</p>
<ul data-nodeid="29">
<li data-nodeid="30">
<p data-nodeid="31">对于所有对象来说，x.clone() !=x 应当返回 true，因为克隆对象与原对象不是同一个对象；</p>
</li>
<li data-nodeid="32">
<p data-nodeid="33">对于所有对象来说，x.clone().getClass() == x.getClass() 应当返回 true，因为克隆对象与原对象的类型是一样的；</p>
</li>
<li data-nodeid="34">
<p data-nodeid="35">对于所有对象来说，x.clone().equals(x) 应当返回 true，因为使用 equals 比较时，它们的值都是相同的。</p>
</li>
</ul>
<p data-nodeid="36">除了注释信息外，我们看 clone() 的实现方法，发现 clone() 是使用 native 修饰的本地方法，因此执行的性能会很高，并且它返回的类型为 Object，因此在调用克隆之后要把对象强转为目标类型才行。</p>
<h4 data-nodeid="37">Arrays.copyOf()</h4>
<p data-nodeid="38">如果是数组类型，我们可以直接使用 Arrays.copyOf() 来实现克隆，实现代码如下：</p>
<pre class="lang-java" data-nodeid="39"><code data-language="java">People[]&nbsp;o1&nbsp;=&nbsp;{<span class="hljs-keyword">new</span>&nbsp;People(<span class="hljs-number">1</span>,&nbsp;<span class="hljs-string">"Java"</span>)};
People[]&nbsp;o2&nbsp;=&nbsp;Arrays.copyOf(o1,&nbsp;o1.length);
<span class="hljs-comment">//&nbsp;修改原型对象的第一个元素的值</span>
o1[<span class="hljs-number">0</span>].setName(<span class="hljs-string">"Jdk"</span>);
System.out.println(<span class="hljs-string">"o1:"</span>&nbsp;+&nbsp;o1[<span class="hljs-number">0</span>].getName());
System.out.println(<span class="hljs-string">"o2:"</span>&nbsp;+&nbsp;o2[<span class="hljs-number">0</span>].getName());
</code></pre>
<p data-nodeid="40">以上程序的执行结果为：</p>
<pre class="lang-java te-preview-highlight" data-nodeid="813"><code data-language="java">o1:Jdk
o2:Jdk
</code></pre>

<p data-nodeid="42">从结果可以看出，我们在修改克隆对象的第一个元素之后，原型对象的第一个元素也跟着被修改了，这说明 Arrays.copyOf() 其实是一个浅克隆。</p>
<p data-nodeid="43">因为数组比较特殊数组本身就是引用类型，因此在使用 Arrays.copyOf() 其实只是把引用地址复制了一份给克隆对象，如果修改了它的引用对象，那么指向它的（引用地址）所有对象都会发生改变，因此看到的结果是，修改了克隆对象的第一个元素，原型对象也跟着被修改了。</p>
<h4 data-nodeid="44">深克隆实现方式汇总</h4>
<p data-nodeid="45">深克隆的实现方式有很多种，大体可以分为以下几类：</p>
<ul data-nodeid="46">
<li data-nodeid="47">
<p data-nodeid="48">所有对象都实现克隆方法；</p>
</li>
<li data-nodeid="49">
<p data-nodeid="50">通过构造方法实现深克隆；</p>
</li>
<li data-nodeid="51">
<p data-nodeid="52">使用 JDK 自带的字节流实现深克隆；</p>
</li>
<li data-nodeid="53">
<p data-nodeid="54">使用第三方工具实现深克隆，比如 Apache Commons Lang；</p>
</li>
<li data-nodeid="55">
<p data-nodeid="56">使用 JSON 工具类实现深克隆，比如 Gson、FastJSON 等。</p>
</li>
</ul>
<p data-nodeid="57">接下来我们分别来实现以上这些方式，在开始之前先定义一个公共的用户类，代码如下：</p>
<pre class="lang-java" data-nodeid="58"><code data-language="java"><span class="hljs-comment">/**
&nbsp;*&nbsp;用户类
&nbsp;*/</span>
<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">People</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Address&nbsp;address;&nbsp;<span class="hljs-comment">//&nbsp;包含&nbsp;Address&nbsp;引用对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
}
<span class="hljs-comment">/**
&nbsp;*&nbsp;地址类
&nbsp;*/</span>
<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">Address</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;city;
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
}
</code></pre>
<p data-nodeid="59">可以看出在 People 对象中包含了一个引用对象 Address。</p>
<p data-nodeid="60"><strong data-nodeid="160">1.所有对象都实现克隆</strong></p>
<p data-nodeid="61">这种方式我们需要修改 People 和 Address 类，让它们都实现 Cloneable 的接口，让所有的引用对象都实现克隆，从而实现 People 类的深克隆，代码如下：</p>
<pre class="lang-java" data-nodeid="62"><code data-language="java"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">CloneExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建被赋值对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Address&nbsp;address&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;Address(<span class="hljs-number">110</span>,&nbsp;<span class="hljs-string">"北京"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p1&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;People(<span class="hljs-number">1</span>,&nbsp;<span class="hljs-string">"Java"</span>,&nbsp;address);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;克隆&nbsp;p1&nbsp;对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p2&nbsp;=&nbsp;p1.clone();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;修改原型对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;p1.getAddress().setCity(<span class="hljs-string">"西安"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;输出&nbsp;p1&nbsp;和&nbsp;p2&nbsp;地址信息</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(<span class="hljs-string">"p1:"</span>&nbsp;+&nbsp;p1.getAddress().getCity()&nbsp;+
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-string">"&nbsp;p2:"</span>&nbsp;+&nbsp;p2.getAddress().getCity());
&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;用户类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">People</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Cloneable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Address&nbsp;address;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;重写&nbsp;clone&nbsp;方法
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;<span class="hljs-doctag">@throws</span>&nbsp;CloneNotSupportedException
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-meta">@Override</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">protected</span>&nbsp;People&nbsp;<span class="hljs-title">clone</span><span class="hljs-params">()</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;people&nbsp;=&nbsp;(People)&nbsp;<span class="hljs-keyword">super</span>.clone();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;people.setAddress(<span class="hljs-keyword">this</span>.address.clone());&nbsp;<span class="hljs-comment">//&nbsp;引用类型克隆赋值</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;people;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;地址类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">Address</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Cloneable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;city;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;重写&nbsp;clone&nbsp;方法
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;<span class="hljs-doctag">@throws</span>&nbsp;CloneNotSupportedException
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-meta">@Override</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">protected</span>&nbsp;Address&nbsp;<span class="hljs-title">clone</span><span class="hljs-params">()</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;(Address)&nbsp;<span class="hljs-keyword">super</span>.clone();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="63">以上程序的执行结果为：</p>
<pre class="lang-java" data-nodeid="64"><code data-language="java">p1:西安&nbsp;p2:北京
</code></pre>
<p data-nodeid="65">从结果可以看出，当我们修改了原型对象的引用属性之后，并没有影响克隆对象，这说明此对象已经实现了深克隆。</p>
<p data-nodeid="66"><strong data-nodeid="167">2.通过构造方法实现深克隆</strong></p>
<p data-nodeid="67">《Effective Java》 中推荐使用构造器（Copy Constructor）来实现深克隆，如果构造器的参数为基本数据类型或字符串类型则直接赋值，如果是对象类型，则需要重新 new 一个对象，实现代码如下：</p>
<pre class="lang-java" data-nodeid="68"><code data-language="java"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">SecondExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Address&nbsp;address&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;Address(<span class="hljs-number">110</span>,&nbsp;<span class="hljs-string">"北京"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p1&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;People(<span class="hljs-number">1</span>,&nbsp;<span class="hljs-string">"Java"</span>,&nbsp;address);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;调用构造函数克隆对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p2&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;People(p1.getId(),&nbsp;p1.getName(),
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">new</span>&nbsp;Address(p1.getAddress().getId(),&nbsp;p1.getAddress().getCity()));

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;修改原型对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;p1.getAddress().setCity(<span class="hljs-string">"西安"</span>);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;输出&nbsp;p1&nbsp;和&nbsp;p2&nbsp;地址信息</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(<span class="hljs-string">"p1:"</span>&nbsp;+&nbsp;p1.getAddress().getCity()&nbsp;+
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-string">"&nbsp;p2:"</span>&nbsp;+&nbsp;p2.getAddress().getCity());
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;用户类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">People</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Address&nbsp;address;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;地址类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">Address</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;city;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="69">以上程序的执行结果为：</p>
<pre data-nodeid="70"><code>p1:西安&nbsp;p2:北京
</code></pre>
<p data-nodeid="71">从结果可以看出，当我们修改了原型对象的引用属性之后，并没有影响克隆对象，这说明此对象已经实现了深克隆。</p>
<p data-nodeid="72"><strong data-nodeid="174">3.通过字节流实现深克隆</strong></p>
<p data-nodeid="73">通过 JDK 自带的字节流实现深克隆的方式，是先将要原型对象写入到内存中的字节流，然后再从这个字节流中读出刚刚存储的信息，来作为一个新的对象返回，那么这个新对象和原型对象就不存在任何地址上的共享，这样就实现了深克隆，代码如下：</p>
<pre class="lang-java" data-nodeid="74"><code data-language="java"><span class="hljs-keyword">import</span>&nbsp;java.io.*;

<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">ThirdExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Address&nbsp;address&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;Address(<span class="hljs-number">110</span>,&nbsp;<span class="hljs-string">"北京"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p1&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;People(<span class="hljs-number">1</span>,&nbsp;<span class="hljs-string">"Java"</span>,&nbsp;address);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;通过字节流实现克隆</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p2&nbsp;=&nbsp;(People)&nbsp;StreamClone.clone(p1);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;修改原型对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;p1.getAddress().setCity(<span class="hljs-string">"西安"</span>);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;输出&nbsp;p1&nbsp;和&nbsp;p2&nbsp;地址信息</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(<span class="hljs-string">"p1:"</span>&nbsp;+&nbsp;p1.getAddress().getCity()&nbsp;+
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-string">"&nbsp;p2:"</span>&nbsp;+&nbsp;p2.getAddress().getCity());
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;通过字节流实现克隆
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">StreamClone</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;&lt;T&nbsp;extends&nbsp;Serializable&gt;&nbsp;<span class="hljs-function">T&nbsp;<span class="hljs-title">clone</span><span class="hljs-params">(People&nbsp;obj)</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;T&nbsp;cloneObj&nbsp;=&nbsp;<span class="hljs-keyword">null</span>;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">try</span>&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;写入字节流</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;ByteArrayOutputStream&nbsp;bo&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;ByteArrayOutputStream();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;ObjectOutputStream&nbsp;oos&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;ObjectOutputStream(bo);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;oos.writeObject(obj);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;oos.close();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;分配内存,写入原始对象,生成新对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;ByteArrayInputStream&nbsp;bi&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;ByteArrayInputStream(bo.toByteArray());<span class="hljs-comment">//获取上面的输出字节流</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;ObjectInputStream&nbsp;oi&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;ObjectInputStream(bi);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;返回生成的新对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;cloneObj&nbsp;=&nbsp;(T)&nbsp;oi.readObject();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;oi.close();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}&nbsp;<span class="hljs-keyword">catch</span>&nbsp;(Exception&nbsp;e)&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;e.printStackTrace();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">return</span>&nbsp;cloneObj;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;用户类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">People</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Serializable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Address&nbsp;address;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;地址类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">Address</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Serializable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;city;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="75">以上程序的执行结果为：</p>
<pre data-nodeid="76"><code>p1:西安&nbsp;p2:北京
</code></pre>
<p data-nodeid="77">此方式需要注意的是，由于是通过字节流序列化实现的深克隆，因此每个对象必须能被序列化，必须实现 Serializable 接口，标识自己可以被序列化，否则会抛出异常 (java.io.NotSerializableException)。</p>
<p data-nodeid="78"><strong data-nodeid="181">4.通过第三方工具实现深克隆</strong></p>
<p data-nodeid="79">本课时使用 Apache Commons Lang 来实现深克隆，实现代码如下：</p>
<pre class="lang-java" data-nodeid="80"><code data-language="java"><span class="hljs-keyword">import</span>&nbsp;org.apache.commons.lang3.SerializationUtils;

<span class="hljs-keyword">import</span>&nbsp;java.io.Serializable;

<span class="hljs-comment">/**
&nbsp;*&nbsp;深克隆实现方式四：通过&nbsp;apache.commons.lang&nbsp;实现
&nbsp;*/</span>
<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">FourthExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Address&nbsp;address&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;Address(<span class="hljs-number">110</span>,&nbsp;<span class="hljs-string">"北京"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p1&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;People(<span class="hljs-number">1</span>,&nbsp;<span class="hljs-string">"Java"</span>,&nbsp;address);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;调用&nbsp;apache.commons.lang&nbsp;克隆对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p2&nbsp;=&nbsp;(People)&nbsp;SerializationUtils.clone(p1);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;修改原型对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;p1.getAddress().setCity(<span class="hljs-string">"西安"</span>);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;输出&nbsp;p1&nbsp;和&nbsp;p2&nbsp;地址信息</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println(<span class="hljs-string">"p1:"</span>&nbsp;+&nbsp;p1.getAddress().getCity()&nbsp;+
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-string">"&nbsp;p2:"</span>&nbsp;+&nbsp;p2.getAddress().getCity());
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;用户类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">People</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Serializable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Address&nbsp;address;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;地址类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/</span>
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">Address</span>&nbsp;<span class="hljs-keyword">implements</span>&nbsp;<span class="hljs-title">Serializable</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;city;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;忽略构造方法、set、get&nbsp;方法</span>
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="81">以上程序的执行结果为：</p>
<pre data-nodeid="82"><code>p1:西安&nbsp;p2:北京
</code></pre>
<p data-nodeid="83">可以看出此方法和第三种实现方式类似，都需要实现 Serializable 接口，都是通过字节流的方式实现的，只不过这种实现方式是第三方提供了现成的方法，让我们可以直接调用。</p>
<p data-nodeid="84"><strong data-nodeid="188">5.通过 JSON 工具类实现深克隆</strong></p>
<p data-nodeid="85">本课时我们使用 Google 提供的 JSON 转化工具 Gson 来实现，其他 JSON 转化工具类也是类似的，实现代码如下：</p>
<pre class="lang-java" data-nodeid="86"><code data-language="java"><span class="hljs-keyword">import</span>&nbsp;com.google.gson.Gson;

<span class="hljs-comment">/**
&nbsp;*&nbsp;深克隆实现方式五：通过&nbsp;JSON&nbsp;工具实现
&nbsp;*/</span>
<span class="hljs-keyword">public</span>&nbsp;<span class="hljs-class"><span class="hljs-keyword">class</span>&nbsp;<span class="hljs-title">FifthExample</span>&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-function"><span class="hljs-keyword">public</span>&nbsp;<span class="hljs-keyword">static</span>&nbsp;<span class="hljs-keyword">void</span>&nbsp;<span class="hljs-title">main</span><span class="hljs-params">(String[]&nbsp;args)</span>&nbsp;<span class="hljs-keyword">throws</span>&nbsp;CloneNotSupportedException&nbsp;</span>{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;创建对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Address&nbsp;address&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;Address(<span class="hljs-number">110</span>,&nbsp;<span class="hljs-string">"北京"</span>);
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p1&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;People(<span class="hljs-number">1</span>,&nbsp;<span class="hljs-string">"Java"</span>,&nbsp;address);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-comment">//&nbsp;调用&nbsp;Gson&nbsp;克隆对象</span>
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Gson&nbsp;gson&nbsp;=&nbsp;<span class="hljs-keyword">new</span>&nbsp;Gson();
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;People&nbsp;p2&nbsp;=&nbsp;gson.fromJson(gson.toJson(p1),&nbsp;People.class);

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;//&nbsp;修改原型对象
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;p1.getAddress().setCity("西安");

&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;//&nbsp;输出&nbsp;p1&nbsp;和&nbsp;p2&nbsp;地址信息
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;System.out.println("p1:"&nbsp;+&nbsp;p1.getAddress().getCity()&nbsp;+
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"&nbsp;p2:"&nbsp;+&nbsp;p2.getAddress().getCity());
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;用户类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;class&nbsp;People&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;name;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Address&nbsp;address;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;//&nbsp;忽略构造方法、set、get&nbsp;方法
&nbsp;&nbsp;&nbsp;&nbsp;}

&nbsp;&nbsp;&nbsp;&nbsp;/**
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*&nbsp;地址类
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;*/
&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">static</span>&nbsp;class&nbsp;Address&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;Integer&nbsp;id;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<span class="hljs-keyword">private</span>&nbsp;String&nbsp;city;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;//&nbsp;忽略构造方法、set、get&nbsp;方法
&nbsp;&nbsp;&nbsp;&nbsp;}
}
</code></pre>
<p data-nodeid="87">以上程序的执行结果为：</p>
<pre data-nodeid="88"><code>p1:西安&nbsp;p2:北京
</code></pre>
<p data-nodeid="89">使用 JSON 工具类会先把对象转化成字符串，再从字符串转化成新的对象，因为新对象是从字符串转化而来的，因此不会和原型对象有任何的关联，这样就实现了深克隆，其他类似的 JSON 工具类实现方式也是一样的。</p>
<h4 data-nodeid="90">克隆设计理念猜想</h4>
<p data-nodeid="91">对于克隆为什么要这样设计，官方没有直接给出答案，我们只能凭借一些经验和源码文档来试着回答一下这个问题。Java 中实现克隆需要两个主要的步骤，一是 实现 Cloneable 空接口，二是重写 Object 的 clone() 方法再调用父类的克隆方法 (super.clone())，那为什么要这么做？</p>
<p data-nodeid="92">从源码中可以看出 Cloneable 接口诞生的比较早，JDK 1.0 就已经存在了，因此从那个时候就已经有克隆方法了，那我们怎么来标识一个类级别对象拥有克隆方法呢？克隆虽然重要，但我们不能给每个类都默认加上克隆，这显然是不合适的，那我们能使用的手段就只有这几个了：</p>
<ul data-nodeid="93">
<li data-nodeid="94">
<p data-nodeid="95">在类上新增标识，此标识用于声明某个类拥有克隆的功能，像 final 关键字一样；</p>
</li>
<li data-nodeid="96">
<p data-nodeid="97">使用 Java 中的注解；</p>
</li>
<li data-nodeid="98">
<p data-nodeid="99">实现某个接口；</p>
</li>
<li data-nodeid="100">
<p data-nodeid="101">继承某个类。</p>
</li>
</ul>
<p data-nodeid="102">先说第一个，为了一个重要但不常用的克隆功能， 单独新增一个类标识，这显然不合适；再说第二个，因为克隆功能出现的比较早，那时候还没有注解功能，因此也不能使用；第三点基本满足我们的需求，第四点和第一点比较类似，为了一个克隆功能需要牺牲一个基类，并且 Java 只能单继承，因此这个方案也不合适。采用排除法，无疑使用实现接口的方式是那时最合理的方案了，而且在 Java 语言中一个类可以实现多个接口。</p>
<p data-nodeid="103">那为什么要在 Object 中添加一个 clone() 方法呢？</p>
<p data-nodeid="104">因为 clone() 方法语义的特殊性，因此最好能有 JVM 的直接支持，既然要 JVM 直接支持，就要找一个 API 来把这个方法暴露出来才行，最直接的做法就是把它放入到一个所有类的基类 Object 中，这样所有类就可以很方便地调用到了。</p>
<h3 data-nodeid="105">小结</h3>
<p data-nodeid="106">本课时我们讲了浅克隆和深克隆的概念，以及 Object 对 clone() 方法的约定；还演示了数组的 copyOf() 方法其实为浅克隆，以及深克隆的 5 种实现方式；最后我们讲了 Java 语言中克隆的设计思路猜想，希望这些内容能切实的帮助到你。</p>

---

### 精选评论

##### **5260：
> java8中的Arrays.copyOf已经是深拷贝了

##### **8335：
> 看到楼上有同学说jdk8是深拷贝 有点怀疑 然后自己电脑上用jdk1.7 和jdk1.8测试了一下 Arrays.copyOf()都是浅拷贝啊&nbsp; 网上的资料也是浅拷贝 不知道是不是弄错了😂

 ###### &nbsp;&nbsp;&nbsp; 编辑回复：
> &nbsp;&nbsp;&nbsp; Arrays.copyOf() 是浅克隆

##### **昆：
> Arrays.copyOf1.8中测试结论还是浅拷贝

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; Arrays.copyOf() 是浅克隆

##### **升：
> Arrays.copyOf 对于数值类型的是深拷贝，只能拷贝值，对于对象类型的，是浅拷贝

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 结论是对的，但我们通常把这种情况理解和归纳为浅拷贝哈，就像 true && false 的结果是 false，只有 true && true 才是 true 一样。

##### **知道：
> x.clone().equals(x) 这个是 false 的啊，equals的默认实现是==，比较内容及比较地址的。大伙可以试试看

##### **6602：
> <div><span style="font-size: 16.0125px;">People&nbsp;p1 =&nbsp;</span><span style="font-size: 0.427rem;">new&nbsp;</span><span style="font-size: 0.427rem;">People (1,2,3);</span></div><div><span style="font-size: 16.0125px;">People p2;</span></div><div><span style="font-size: 16.0125px;">p2 = p1;</span></div><div><span style="font-size: 16.0125px;">这样算克隆吗</span></div><div><span style="font-size: 16.0125px;"><br></span></div><div><br></div>

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 这应该叫赋值吧

##### 20201230：
> cloneable closeable serializable 这类相当于 标记接口

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 对的，相当于告诉 JVM 我要干XXX了。

##### **泽：
> Arrays.copyOf 的 执行结果写错了吧

 ###### &nbsp;&nbsp;&nbsp; 讲师回复：
> &nbsp;&nbsp;&nbsp; 嗯，没有啊，Arrays.copyOf() 浅克隆修改一个另一个也变了。

