package org.my.code.lesson2;

/**
 * 锁膨胀
 * <p>
 * 基本概念
 * 锁膨胀是指锁的状态从偏向锁升级到轻量级锁，再升级到重量级锁的过程。这是 Java 虚拟机（JVM）为了应对不同程度的线程竞争，自动对锁进行优化升级的机制。在多线程环境下，根据线程竞争锁的情况，JVM 会动态地调整锁的状态，以平衡性能和并发安全之间的关系。
 * <p>
 * 偏向锁（Biased Locking）
 * 原理：偏向锁是 Java 6 引入的一种优化机制。当一个线程访问同步块并获取锁时，JVM 会将这个锁标记为偏向该线程，这个过程通过在对象头的部分位设置偏向标志和线程 ID 来实现。在没有其他线程竞争的情况下，后续该线程再次进入同步块时，只需要检查对象头中的偏向标志和线程 ID 是否匹配自己，就可以直接进入同步块，而不需要进行任何同步操作，这样可以提高性能。
 * 膨胀场景：当有另外一个线程尝试获取这个偏向锁时，偏向锁就会膨胀为轻量级锁。例如，线程 A 已经获取了一个偏向锁，当线程 B 也来访问这个同步块并尝试获取锁时，JVM 检测到存在竞争，就会将偏向锁升级为轻量级锁。
 * <p>
 * 轻量级锁（Lightweight Locking）
 * 原理：轻量级锁是通过在栈帧中建立一个锁记录（Lock Record）来实现的。当一个线程进入同步块时，JVM 会在当前线程的栈帧中创建一个锁记录，然后将对象头中的部分数据复制到锁记录中，同时使用 CAS（比较并交换）操作尝试将对象头指向这个锁记录，以此来获取锁。如果 CAS 操作成功，那么该线程就获取了轻量级锁，可以执行同步块中的代码。在没有多线程竞争或者竞争程度较低的情况下，轻量级锁可以通过 CAS 操作快速地获取和释放锁，避免了重量级锁的开销。
 * 膨胀场景：当有多个线程同时竞争轻量级锁，并且自旋（通过循环尝试获取锁）一定次数后仍然无法获取锁时，轻量级锁就会膨胀为重量级锁。例如，多个线程频繁地尝试获取同一个轻量级锁，导致 CAS 操作不断失败，经过 JVM 设定的自旋次数后，JVM 认为竞争过于激烈，就会将轻量级锁升级为重量级锁。
 * <p>
 * 重量级锁（Heavyweight Locking）
 * 原理：重量级锁依赖于操作系统的互斥量（Mutex）来实现。当一个线程获取了重量级锁后，其他线程如果想要获取这个锁，就会被阻塞并进入操作系统的等待队列。这种方式保证了在高并发竞争环境下的线程安全，但由于涉及到操作系统的上下文切换等操作，开销较大。
 * 性能影响：锁膨胀到重量级锁后，性能会受到较大影响。因为线程的阻塞和唤醒需要操作系统的介入，会产生比较大的系统开销。例如，在频繁获取和释放锁的高并发场景下，如果大量的锁都膨胀为重量级锁，可能会导致系统的吞吐量下降和响应时间变长。
 */
public class LockInflationExample {


    /**
     * 在这个示例中：
     * 首先，线程 1 访问同步块并获取锁，在没有竞争的情况下，这个锁可能会是偏向锁。
     * 然后，线程 2 在线程 1 持有锁的情况下尝试获取锁，这可能会导致偏向锁膨胀为轻量级锁。
     * 最后，当更多的线程（从线程 3 到线程 10）加入竞争时，经过一定的竞争过程，轻量级锁可能会膨胀为重量级锁。这展示了在多线程竞争情况下锁膨胀的一个可能的过程。不过，JVM 的锁膨胀机制是复杂的，实际的膨胀过程还会受到 JVM 参数和具体运行环境等因素的影响。
     *
     * @param args
     */
    public static void main(String[] args) {
        // 创建一个对象，用于作为锁对象
        Object lock = new Object();
        // 线程1获取偏向锁
        Thread thread1 = new Thread(() -> {
            synchronized (lock) {
                // 模拟一些操作
                try {
                    Thread.sleep(5000);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        });
        thread1.start();
        try {
            // 等待线程1获取并持有偏向锁
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        // 线程2尝试获取锁，可能导致偏向锁膨胀为轻量级锁
        Thread thread2 = new Thread(() -> {
            synchronized (lock) {
                // 模拟一些操作
                try {
                    Thread.sleep(5000);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
        });
        thread2.start();
        try {
            // 等待线程2与线程1竞争锁，可能导致轻量级锁膨胀为重量级锁
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        // 更多线程尝试获取锁，加剧竞争
        for (int i = 3; i <= 10; i++) {
            Thread thread = new Thread(() -> {
                synchronized (lock) {
                    // 模拟一些操作
                    try {
                        Thread.sleep(5000);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
            });
            thread.start();
        }
    }


    /**
     * 如何避免锁膨胀
     *
     * 减少锁的竞争
     * 缩小同步代码块范围：
     * 原理：只对真正需要同步保护的共享资源操作部分使用synchronized关键字或者其他锁机制。如果将大量非共享资源相关的代码也包含在同步块中，会增加锁的持有时间，从而增加其他线程等待锁的机会，加剧锁竞争，导致锁更容易膨胀。
     *
     * 降低锁的使用频率：
     * 原理：如果可以通过其他方式来避免频繁地使用锁，例如使用线程本地变量（Thread - Local）来存储每个线程独有的数据，就可以减少对共享资源的访问，从而减少锁的使用频率。
     *
     * 选择合适的锁策略
     * 偏向锁的合理使用：
     * 原理：在大多数情况下，一个锁总是被同一个线程获取和释放，偏向锁可以提供较好的性能。但如果应用场景中存在较多的锁竞争情况，偏向锁的撤销和膨胀机制可能会带来额外的开销。可以通过设置 JVM 参数-XX:-UseBiasedLocking来禁用偏向锁，在明确知道存在较多线程竞争锁的场景下，这样可以避免偏向锁频繁膨胀带来的性能损耗。
     * 示例：假设一个多线程的 Web 应用服务器，处理大量短生命周期的请求，每个请求可能会频繁地竞争锁。在这种情况下，禁用偏向锁可能会提高性能。可以在启动服务器的脚本中添加 JVM 参数-XX:-UseBiasedLocking来实现。
     * 使用读写锁（ReadWriteLock）代替普通锁：
     * 原理：在大量读操作和少量写操作的场景下，读写锁可以显著提高并发性能。读写锁允许多个线程同时读取共享资源（读锁），只有在进行写操作时才会互斥（写锁）。这样可以减少锁的竞争，避免锁过度膨胀。
     *
     * 优化数据结构和算法
     * 使用无锁数据结构：
     * 原理：无锁数据结构通过使用原子操作（如CAS - 比较并交换）来实现并发安全，避免了传统锁的开销和锁竞争。例如，java.util.concurrent.atomic包中的原子类（如AtomicInteger、AtomicReference等）就是基于无锁算法实现的。
     *
     * 优化算法减少共享资源依赖：
     * 原理：有些算法可能会频繁地访问和修改共享资源，通过优化算法，减少对共享资源的依赖，可以降低锁的需求。例如，在一些数据处理算法中，可以将数据进行分区处理，使得每个线程主要处理自己分区内的数据，减少对全局共享数据的访问。
     * 示例：假设一个数组求和的多线程算法，原始算法是所有线程同时访问和修改一个全局的求和变量。优化后，可以将数组分成多个子数组，每个线程负责一个子数组的求和，最后再将各个子数组的和相加，这样就减少了对共享资源的竞争。
     *
     */


    /**
     * 锁膨胀对程序性能有哪些具体影响？
     *
     * 偏向锁膨胀的性能影响
     * 偏向锁撤销开销：当偏向锁发生膨胀时，首先要进行偏向锁的撤销操作。在这个过程中，JVM 需要通过一个安全点（Safe Point）来暂停拥有偏向锁的线程，检查锁对象的状态，这涉及到一定的性能开销。例如，在高并发场景下，如果频繁地有线程竞争偏向锁，导致偏向锁不断地撤销和膨胀，会消耗大量的 CPU 资源用于这些额外的检查和状态变更操作。
     * 锁升级后的性能变化：偏向锁膨胀为轻量级锁后，后续线程获取锁的方式发生改变。从简单的检查对象头中的偏向标志和线程 ID，变为使用 CAS（比较并交换）操作来获取锁。如果 CAS 操作频繁失败（例如，多个线程竞争激烈），会导致线程不断地自旋尝试获取锁，这增加了 CPU 的占用率，降低了程序的性能。
     *
     * 轻量级锁膨胀的性能影响
     * 自旋开销：轻量级锁在竞争不激烈的情况下，通过自旋等待可以快速获取锁。但是，当轻量级锁膨胀为重量级锁时，意味着自旋等待已经无法有效获取锁，即存在多个线程长时间竞争锁的情况。在自旋过程中，线程会不断地循环检查锁是否可用，这消耗了 CPU 资源。而且，过多的自旋可能会导致 CPU 缓存一致性协议（如 MESI 协议）的频繁交互，进一步降低性能。
     * 重量级锁的阻塞和唤醒开销：轻量级锁膨胀为重量级锁后，线程获取不到锁时会被阻塞，进入操作系统的等待队列。线程的阻塞和唤醒涉及到操作系统的上下文切换，这是一个相对昂贵的操作。每次上下文切换都需要保存和恢复线程的执行上下文，包括寄存器状态、程序计数器等信息。在高并发场景下，如果大量线程频繁地被阻塞和唤醒，会导致系统的吞吐量下降，响应时间变长。
     *
     * 对整体系统吞吐量和响应时间的影响
     * 吞吐量降低：随着锁的膨胀，尤其是膨胀到重量级锁后，系统中线程的并发执行效率会降低。更多的线程处于等待状态或者在自旋，实际能够有效执行任务的线程数量减少，导致单位时间内系统能够完成的任务数量（即吞吐量）下降。例如，在一个处理大量并发请求的 Web 服务器中，如果频繁发生锁膨胀，服务器每秒能够处理的请求数量会明显减少。
     * 响应时间变长：当锁膨胀导致线程被阻塞或者自旋时间过长时，请求的响应时间会增加。对于对响应时间敏感的应用，如实时交易系统或者用户交互界面，较长的响应时间可能会导致用户体验变差，甚至可能影响系统的业务逻辑正确性。例如，一个用户在操作界面上点击按钮后，由于锁膨胀导致相关操作长时间等待锁，用户可能会认为系统出现故障或者卡顿。
     */


    /**
     * 锁粗化（Lock Coarsening）优化
     * 原理：与通常希望减小锁粒度的思路相反，JVM 会在某些情况下进行锁粗化。当 JVM 检测到一系列连续的、对同一个对象的加锁和解锁操作出现在一个线程中时，为了减少频繁加锁解锁的开销，JVM 会将这些操作合并为一个范围更大的加锁解锁操作。例如，如果一个循环体内频繁地对同一个对象进行同步操作，JVM 可能会把整个循环体作为一个同步块，只在循环开始时加锁，循环结束时解锁。
     *
     * 在这个例子中，JVM 可能会将整个循环体的同步操作合并，使得锁的粒度从每次循环加锁解锁变为整个循环体加锁解锁，减少了频繁加锁解锁的开销。
     * 适用场景：适用于在一个线程中对同一个对象有频繁的、短时间的同步操作的场景，如在一个复杂的计算方法中，对一个共享的临时数据结构进行多次修改操作。
     */
    private Object lock = new Object();
    public void doSomething() {
        for (int i = 0; i < 10; i++) {
            synchronized (lock) {
                // 对共享资源进行一些操作
            }
        }
    }


    /**
     * 锁消除（Lock Elimination）优化
     * 原理：JVM 在编译阶段可以通过逃逸分析（Escape Analysis）来判断一个对象是否只在一个线程内被访问。如果一个对象不会逃逸出当前线程（即不会被其他线程访问），那么对这个对象的同步操作是没有必要的，JVM 会将这些同步操作消除。例如，在一个方法内部创建的局部对象，并且这个对象只在这个方法内部被访问，即使这个对象有同步方法或者在同步块中被使用，JVM 也可能会消除这些同步操
     *
     *
     * JVM 通过逃逸分析发现localObj不会被其他线程访问，可能会消除这个同步操作，从而优化了性能，避免了不必要的锁开销。
     * 适用场景：适用于在方法内部使用的局部对象的同步操作，尤其是在编译器能够确定对象不会被其他线程访问的情况下，如在一些简单的工具方法中对局部数据结构的操作。
     */
    public class LockEliminationExample {
        public void localObjectMethod() {
            // 定义一个局部对象
            Object localObj = new Object();
            // 对局部对象进行同步操作，但这个对象不会被其他线程访问
            synchronized (localObj) {
                // 一些操作
            }
        }
    }






}
