# 7.@Bean的后置处理器

## 7.1首先看下配置类

@Bean是一种以工厂方法的方式来创建bean的，而@Configuration标注的类就相当于是一个bean工厂。

```java
@Configuration
@ComponentScan("com.example.testspring.a05.Component")
public class Config {

    @Bean
    public Bean1 bean1(){
        return new Bean1();
    }

    public Bean2 bean2(){
        return new Bean2();
    }

    @Bean
    public Bean3 bean3(){
        return new Bean3();
    }

}
```

7.2 解析@Bean的后处理器工作原理

- 首先还是使用GenericApplicationContext这个相对简约的beanFactory。
- CachingMetadataReaderFactory通过读取类路径的方式，直接获取config的类信息（这种方式不需要使用到类加载）
- 获取所有添加了@bean注解的方法
- 拿到方法之后，生成对应的beanDefinition，然后向beanFactory中注册beanDefinition。
- 在调用 context.refresh(); 的时候，生成对应的bean

```java
public class A05Application {
    @SneakyThrows
    public static void main(String[] args) {
        GenericApplicationContext context = new GenericApplicationContext();
        context.registerBean("config", Config.class);

        // 先读取class配置类的信息
        CachingMetadataReaderFactory factory = new CachingMetadataReaderFactory();
        MetadataReader reader = factory.getMetadataReader(new ClassPathResource("com/example/testspring/a05/Config.class"));

        // 获取标注@Bean的 【方法】的信息
        Set<MethodMetadata> annotatedMethods = reader.getAnnotationMetadata().getAnnotatedMethods(Bean.class.getName());

        for (MethodMetadata metadata : annotatedMethods) {
            System.out.println(metadata.getMethodName());

            // 设置工厂方法
            BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition();
            builder.setFactoryMethodOnBean(metadata.getMethodName(), "config");

            // 拿到bean定义
            AbstractBeanDefinition beanDefinition = builder.getBeanDefinition();
            context.getDefaultListableBeanFactory().registerBeanDefinition(metadata.getMethodName(), beanDefinition);
        }

        context.refresh();

        for (String name : context.getBeanDefinitionNames()) {
            System.out.println("name : " + name);
        }

        context.close();
    }
}
```

## 7.3 工厂方法带参数的场景

**config类做一下修改：**

```java
@Configuration
@ComponentScan("com.example.testspring.a05.Component")
public class Config {

    @Bean
    public Bean1 bean1(){
        return new Bean1();
    }

    public Bean2 bean2(){
        return new Bean2();
    }

    @Bean
    public Bean3 bean3(Bean1 bean1){
        return new Bean3();
    }

}
```

**设置自动装配**

> 对于构造方法的参数或者工厂方法（@Bean标注的方法），设置自动装配是用 `AbstractBeanDefinition.AUTOWIRE_CONSTRUCTOR`

```java
 // 设置工厂方法
 BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition();
 builder.setFactoryMethodOnBean(metadata.getMethodName(), "config");
 builder.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_CONSTRUCTOR);
```

## 7.4 @Bean注解的属性解析过程

```java
String initMethod = metadata.getAllAnnotationAttributes(Bean.class.getName()).getFirst("initMethod").toString();
if (initMethod != null && initMethod.length() > 0) {
	builder.setInitMethodName(initMethod);
}
```

运行结果：

```
bean1
bean3
16:06:19.997 [main] DEBUG org.springframework.context.support.GenericApplicationContext - Refreshing org.springframework.context.support.GenericApplicationContext@21588809
16:06:20.062 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'config'
16:06:20.077 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'bean1'
16:06:20.077 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'bean3'
16:06:20.082 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Autowiring by type from bean name 'bean3' via factory method to bean named 'bean1'
bean3#init ....
name : config
name : bean1
name : bean3
16:06:20.122 [main] DEBUG org.springframework.context.support.GenericApplicationContext - Closing org.springframework.context.support.GenericApplicationContext@21588809, started on Thu Jan 19 16:06:19 CST 2023
```

**完整代码：**

```java
public class A05Application {
    @SneakyThrows
    public static void main(String[] args) {
        GenericApplicationContext context = new GenericApplicationContext();
        context.registerBean("config", Config.class);

        // 先读取class配置类的信息
        CachingMetadataReaderFactory factory = new CachingMetadataReaderFactory();
        MetadataReader reader = factory.getMetadataReader(new ClassPathResource("com/example/testspring/a05/Config.class"));

        // 获取标注@Bean的 【方法】的信息
        Set<MethodMetadata> annotatedMethods = reader.getAnnotationMetadata().getAnnotatedMethods(Bean.class.getName());

        for (MethodMetadata metadata : annotatedMethods) {
            System.out.println(metadata.getMethodName());

            // 设置工厂方法
            BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition();
            builder.setFactoryMethodOnBean(metadata.getMethodName(), "config");
            builder.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_CONSTRUCTOR);

            String initMethod = metadata.getAllAnnotationAttributes(Bean.class.getName()).getFirst("initMethod").toString();
            if (initMethod != null && initMethod.length() > 0) {
                builder.setInitMethodName(initMethod);
            }

            // 拿到bean定义
            AbstractBeanDefinition beanDefinition = builder.getBeanDefinition();
            context.getDefaultListableBeanFactory().registerBeanDefinition(metadata.getMethodName(), beanDefinition);
        }

        context.refresh();

        for (String name : context.getBeanDefinitionNames()) {
            System.out.println("name : " + name);
        }

        context.close();
    }
}
```



## 7.5 封装成后置处理器

```java
public class AtBeanPostProcessor implements BeanFactoryPostProcessor {

    @SneakyThrows
    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory configurableListableBeanFactory) throws BeansException {
        // 先读取class配置类的信息
        CachingMetadataReaderFactory factory = new CachingMetadataReaderFactory();
        MetadataReader reader = factory.getMetadataReader(new ClassPathResource("com/example/testspring/a05/Config.class"));

        // 获取标注@Bean的 【方法】的信息
        Set<MethodMetadata> annotatedMethods = reader.getAnnotationMetadata().getAnnotatedMethods(Bean.class.getName());

        for (MethodMetadata metadata : annotatedMethods) {
            System.out.println(metadata.getMethodName());

            // 设置工厂方法
            BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition();
            builder.setFactoryMethodOnBean(metadata.getMethodName(), "config");
            builder.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_CONSTRUCTOR);

            String initMethod = metadata.getAllAnnotationAttributes(Bean.class.getName()).getFirst("initMethod").toString();
            if (initMethod != null && initMethod.length() > 0) {
                builder.setInitMethodName(initMethod);
            }

            if (configurableListableBeanFactory instanceof DefaultListableBeanFactory) {
                DefaultListableBeanFactory beanFactory = (DefaultListableBeanFactory) configurableListableBeanFactory;

                // 拿到bean定义
                AbstractBeanDefinition beanDefinition = builder.getBeanDefinition();
                beanFactory.registerBeanDefinition(metadata.getMethodName(), beanDefinition);
            }
        }
    }
}

```

**测试上面封装的处理器**

```java
public class A05Application {
    @SneakyThrows
    public static void main(String[] args) {
        GenericApplicationContext context = new GenericApplicationContext();
        context.registerBean("config", Config.class);

        context.registerBean(AtBeanPostProcessor.class);

        context.refresh();

        for (String name : context.getBeanDefinitionNames()) {
            System.out.println("name : " + name);
        }

        context.close();
    }
}

```

