

**项目源码** ： [🔗](https://github.com/geekibli/demo-spring)


## 1.BeanFactory和ApplicationContext


### 1.1 到底什么是BeanFactory

1. 它是ApplicationContext的父接口
2. 它才是Spring的核心容器, 主要的ApplicationContext 实现/组合 了它的功能



### 1.2 BeanFactory能做什么

1. 表面上只有getBean
2. 实际上控制反转、基本的依赖，注入、直至Bean的生命周期的各种功能，都由它的实现类提供
3. 它可以管理所有的bean
	

下面通过反射方式拿到beanFactory中的单例bean，当然是限定的一些类：

```java
public static void main(String[] args) throws NoSuchFieldException, IllegalAccessException {
        ConfigurableApplicationContext context = SpringApplication.run(TestSpringApplication.class, args);

        Field singletonObjects = DefaultSingletonBeanRegistry.class.getDeclaredField("singletonObjects");
        singletonObjects.setAccessible(true);
        ConfigurableListableBeanFactory beanFactory = context.getBeanFactory();
        Map<String, Object> map = (Map<String, Object>) singletonObjects.get(beanFactory);
        map.entrySet().stream().filter(e -> e.getKey().startsWith("component"))
                .forEach(e -> {
                    System.err.println(e.getKey() + "   " + e.getValue());
                });
    }
```

![](https://oscimg.oschina.net/oscnet/up-2817b1229a9ef68d7856a9db4294f3a4bc7.png)


### 1.3 ApplicationContext比BeanFactory扩展了哪些功能


![](https://oscimg.oschina.net/oscnet/up-8f860ac2720e02b1276316a46d70ac586c3.png)


- MessageSource: 国际化语言支持能力
- ResourcePatternResolver：一些通配符的支持能力
- ApplicationEventPublisher： 事件发布能力
- EnvironmentCapable： 环境信息读取能力，系统变量和配置文件


**下面看一下MessageSource能力的展示：**

![](https://oscimg.oschina.net/oscnet/up-ad1583066a2d339fab7994f113878c1e447.png)


**context获取资源**
```java
 Resource[] resources = context.getResources("classpath:application.properties");
        for (Resource resource : resources) {
            System.err.println(resource);
        }

Resource[] resources = context.getResources("classpath*:MATE-INF/spring.factories");
        for (Resource resource : resources) {
            System.err.println(resource);
        }
```
输出结果: 
```java
class path resource [application.properties]
```


**context获取环境配置**
```java
System.err.println(context.getEnvironment().getPropertySources().get("java_home"));
System.err.println(context.getEnvironment().getProperty("java_home"));
System.err.println(context.getEnvironment().getProperty("spring.profiles.active"));
```

**事件发布与监听**
![](https://oscimg.oschina.net/oscnet/up-56692d6e9e11a092596b81ca73312c39030.png)


> **事件的好处是可以进行解耦合。**


## 2.容器的实现

### 2.1 bean的声明和定义
```java
package com.example.testspring.beans;

import lombok.Data;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

public class TestBeanFactory {
    public static void main(String[] args) {
        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();
        AbstractBeanDefinition beanDefinition = BeanDefinitionBuilder.genericBeanDefinition(Config.class).setScope("singleton").getBeanDefinition();
        beanFactory.registerBeanDefinition("config", beanDefinition);

        String[] beanDefinitionNames = beanFactory.getBeanDefinitionNames();
        for (String beanDefinitionName : beanDefinitionNames) {
            System.out.println("beanDefinitionName : " + beanDefinitionName);
        }
    }

    @Configuration
    static class Config{
        @Bean
        public Bean1 getBean1(){
            return new Bean1();
        }

        @Bean
        public Bean2 getBean2(){
            return new Bean2();
        }
    }

    @Data
    static class Bean1 {

        public Bean1 (){
            System.out.println("init Bean1");
        }

        @Autowired
        private Bean2 bean2;

    }

    @Data
    static class Bean2 {
        public Bean2 (){
            System.out.println("init Bean2");
        }
    }

}

```
**输出结果：**

```java
beanDefinitionName : config
```

运行以上main方法，发现又有一个beanDefinition，**那为什么Bean1 , Bean2 没有打印出来呢？**


因为原始的DefaultListableBeanFactory并没有能力处理@Bean， @Configuration 这样的注解，需要借助一个工具类：

```java
public static void main(String[] args) {
        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();
        AbstractBeanDefinition beanDefinition = BeanDefinitionBuilder.genericBeanDefinition(Config.class).setScope("singleton").getBeanDefinition();
        beanFactory.registerBeanDefinition("config", beanDefinition);

        // 给 beanFactory 添加一些常用的后置处理器
        AnnotationConfigUtils.registerAnnotationConfigProcessors(beanFactory);
        
        String[] beanDefinitionNames = beanFactory.getBeanDefinitionNames();
        for (String beanDefinitionName : beanDefinitionNames) {
            System.out.println("beanDefinitionName : " + beanDefinitionName);
        }
    }
```
**运行结果：**
```java
beanDefinitionName : config
beanDefinitionName : org.springframework.context.annotation.internalConfigurationAnnotationProcessor
beanDefinitionName : org.springframework.context.annotation.internalAutowiredAnnotationProcessor
beanDefinitionName : org.springframework.context.annotation.internalCommonAnnotationProcessor
beanDefinitionName : org.springframework.context.event.internalEventListenerProcessor
beanDefinitionName : org.springframework.context.event.internalEventListenerFactory
```


这里可以看到增加了一些Spring默认的beanFactory后置处理器，比如，internalConfigurationAnnotationProcessor。
这是Spring内置的注解配置类的处理器，现在它是出现在了BeanFactory中了，但是由于它还没有工作，所以，打印的结果中，我们依旧看不到Bean1,Bean2。

那么，我们需要执行这些后置处理器的后置处理方法：

```java
public static void main(String[] args) {
        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();
        AbstractBeanDefinition beanDefinition = BeanDefinitionBuilder.genericBeanDefinition(Config.class).setScope("singleton").getBeanDefinition();
        beanFactory.registerBeanDefinition("config", beanDefinition);

        // 给 beanFactory 添加一些常用的后置处理器
        AnnotationConfigUtils.registerAnnotationConfigProcessors(beanFactory);

        beanFactory.getBeansOfType(BeanFactoryPostProcessor.class)
                .values().stream()
                .forEach(beanFactoryPostProcessor -> {
                    beanFactoryPostProcessor.postProcessBeanFactory(beanFactory);
                });

        String[] beanDefinitionNames = beanFactory.getBeanDefinitionNames();
        for (String beanDefinitionName : beanDefinitionNames) {
            System.out.println("beanDefinitionName : " + beanDefinitionName);
        }
    }
```

**配置类如下：**  

> 这里设置了一下bean的名字

```java
@Configuration
    static class Config{
        @Bean(name = "bean1")
        public Bean1 getBean1(){
            return new Bean1();
        }

        @Bean(name = "bean2")
        public Bean2 getBean2(){
            return new Bean2();
        }
    }
```
如果不设置 `@Bean(name = "bean1")`， 打印出来的beanDefinitionName就是方法的名字。 **getBean1**

**执行结果：**
```java
beanDefinitionName : config
beanDefinitionName : org.springframework.context.annotation.internalConfigurationAnnotationProcessor
beanDefinitionName : org.springframework.context.annotation.internalAutowiredAnnotationProcessor
beanDefinitionName : org.springframework.context.annotation.internalCommonAnnotationProcessor
beanDefinitionName : org.springframework.context.event.internalEventListenerProcessor
beanDefinitionName : org.springframework.context.event.internalEventListenerFactory
beanDefinitionName : bean1
beanDefinitionName : bean2
```

由此，我们可以看到，Bean1, Bean2是打印出来了。


### 2.2 如何使用bean

```java
System.err.println(beanFactory.getBean(Bean1.class).getBean2());
```
**运行结果：**

```java
init Bean1
null
```
可以看到，bean2的实例打印结果是空。为什么，**对象注入没有生效**。

需要bean的后置处理器 `org.springframework.context.annotation.internalAutowiredAnnotationProcessor` 来处理对应的方法来处理 @Autowired 注解。

针对Bean的声明周期的各个阶段提供扩展，比如 @Autowired, @Resource...

**把beanPostProcessor添加到beanFactory中**

```java
beanFactory.getBeansOfType(BeanPostProcessor.class).values().forEach(beanFactory::addBeanPostProcessor);

        System.err.println(beanFactory.getBean(Bean1.class).getBean2());
```
**再次查看运行结果：**

```java
21:56:59.269 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'bean1'
21:56:59.270 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'config'
init Bean1
21:56:59.309 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'bean2'
init Bean2
TestBeanFactory.Bean2()
```

**注意**：单例bean并不是beanFactory一开始就都创建好的，而是，在使用的时候，才回去初始化。

如果需要事先就初始化所有的单例对象，需要设置一个方法：

```java
beanFactory.preInstantiateSingletons();
```

### 2.3 beanFactory小结

**a. beanFactory不会主动去做的事**

1. 不会主动调用BeanFactory 后置处理器
2. 不会主动添加Bena后置处理器
3. 不会主动初始化单例bean
4. 不会解析beanFactory,还不会解析 ${} 和 #{}


**b. beanFactory后置处理器有排序的逻辑。**

> beanFactory很多基础的功能都没有主动去做。需要手动去把后置处理器设置好后才可以使用。



### 2.4 beanFactory扩展-后置处理器顺序

后置处理器的添加顺序可以看一下下面这个方法的源码 👇：

```java
org.springframework.context.annotation.AnnotationConfigUtils#registerAnnotationConfigProcessors(org.springframework.beans.factory.support.BeanDefinitionRegistry, java.lang.Object)
```

可以看到@Autowired的处理器加在@Resource的梳理器前面。存储的结构是 `Set<BeanDefinitionHolder> beanDefs = new LinkedHashSet(8);`

**如果一个属性注入是，即有 @Autowired 又有 @Resource注解，会怎样？** 下面尝试一下：

增加两个bean，同时加到容器中（@Configuration + @Bean）

```java
 interface Inner {
        
    }
    
@Data
static class Bean3 implements Inner {
    
    public Bean3(){
        System.out.println("init Bean3");
    }
}

@Data
static class Bean4 implements Inner {

    public Bean4(){
        System.out.println("init Bean4");
    }
}
```

如果注入的类型是 `Inner` 接口类型，会怎样？

![](https://oscimg.oschina.net/oscnet/up-de5c6eb457b97ded607e1f6abbc36af2b4c.png)

会有编译错误，提示有2个类型，Bean3 和 Bean4，没有明确要注入那个类型。

但是如果要是改一下注入的变量名称呢？

```java
 @Data
static class Bean1 {
    public Bean1() {
        System.out.println("init Bean1");
    }
    @Autowired
    private Inner bean3;
}
```
**可以看到是可以正确注入的。**

由此可以知道 @Autowired 优先按照类型注入，如果类型不能确定唯一，需要指定名称注入。

或者可以借助 注解 @Qualifier(value = "bean3") 来注入。

```java
@Data
static class Bean1 {
    public Bean1() {
        System.out.println("init Bean1");
    }
    @Autowired
    @Qualifier(value = "bean3")
    private Inner inner;
}
```

**如果对象同时添加了 @Autowired 和 @Resource(name = "bean4")，那么真实注入的对象是什么类型？**

```java
@Data
static class Bean1 {
    public Bean1() {
        System.out.println("init Bean1");
    }
    @Autowired
    @Resource(name = "bean4")
    private Inner bean3;
}
```

执行结果：

```java
TestBeanFactory.Bean3()
```

为什么呢？ 这就是beanFactory后置处理器的添加顺序了。

```java
beanFactory.getBeansOfType(BeanPostProcessor.class).values()
                .forEach(beanPostProcessor -> {
                            System.out.println("beanPostProcessor...  " + beanPostProcessor.getClass().getName());
                            beanFactory.addBeanPostProcessor(beanPostProcessor);
                        }
                );
```
查看结果：

```java
beanPostProcessor...  org.springframework.beans.factory.annotation.AutowiredAnnotationBeanPostProcessor
beanPostProcessor...  org.springframework.context.annotation.CommonAnnotationBeanPostProcessor
```

**如何控制后置处理器的添加顺序呢？**

```java
beanFactory.getBeansOfType(BeanPostProcessor.class).values()
                .stream().sorted(beanFactory.getDependencyComparator())
                .forEach(beanPostProcessor -> {
                            System.out.println("beanPostProcessor...  " + beanPostProcessor.getClass().getName());
                            beanFactory.addBeanPostProcessor(beanPostProcessor);
                        }
                );
```

执行结果：

```java
beanPostProcessor...  org.springframework.context.annotation.CommonAnnotationBeanPostProcessor
beanPostProcessor...  org.springframework.beans.factory.annotation.AutowiredAnnotationBeanPostProcessor
```

**排序是如何生效的？**

![](https://oscimg.oschina.net/oscnet/up-8e5e7d76b4ea357ea545d790b0f68f14bd0.png)

> order数值越小 优先加入。

下面是2个后置处理器的定义和构造函数：
```java

 public CommonAnnotationBeanPostProcessor() {
        this.setOrder(2147483644);
        this.setInitAnnotationType(PostConstruct.class);
        this.setDestroyAnnotationType(PreDestroy.class);
        this.ignoreResourceType("javax.xml.ws.WebServiceContext");
}


public class AutowiredAnnotationBeanPostProcessor extends InstantiationAwareBeanPostProcessorAdapter implements MergedBeanDefinitionPostProcessor, PriorityOrdered, BeanFactoryAware {
    protected final Log logger = LogFactory.getLog(this.getClass());
    private final Set<Class<? extends Annotation>> autowiredAnnotationTypes = new LinkedHashSet(4);
    private String requiredParameterName = "required";
    private boolean requiredParameterValue = true;
    private int order = 2147483645;
    ...
}    
```


## 3. ApplicationContext的实现


> ApplicationContext的四种实现；

### 3.1 展示ClassPathXmlApplicationContext加载和获取bean

下面看一下最为精简的spring容器：以大家熟知的ClassPathXmlApplicationContext为例：

```java
public class TestApplicationContext {
    public static void main(String[] args) {
        testXmlPath();
    }

    private static void testXmlPath(){
        ClassPathXmlApplicationContext applicationContext = new ClassPathXmlApplicationContext("b01.xml");
        String[] beanDefinitionNames = applicationContext.getBeanDefinitionNames();
        for (String name : beanDefinitionNames) {
            System.out.println("name : " + name);
        }

        System.out.println(applicationContext.getBean("bean1"));
    }
}
```
**bean的定义文件，b01.xml文件如下。**

```xml
<?xml version="1.0" encoding="UTF-8"?>
<beans xmlns="http://www.springframework.org/schema/beans"
       xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
       xsi:schemaLocation="http://www.springframework.org/schema/beans http://www.springframework.org/schema/beans/spring-beans.xsd">

    <beans>
        <bean id="bean1" class="com.example.testspring.beans.TestBeanFactory.Bean1"/>
        <bean id="bean2" class="com.example.testspring.beans.TestBeanFactory.Bean2"/>
    </beans>
</beans>
```
**查看执行效果：**

```java
19:29:47.662 [main] DEBUG org.springframework.beans.factory.xml.XmlBeanDefinitionReader - Loaded 2 bean definitions from class path resource [b01.xml]
19:29:47.749 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'bean1'
init Bean1
19:29:47.764 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'bean2'
init Bean2
name : bean1
name : bean2
TestBeanFactory.Bean1(bean3=null)
```

可以看到这种基于xml文件的配置方式，非常繁琐。


### 3.2 ClassPathXmlApplicationContext如何读取xml中的bean配置然后加载到beanFactory中的？



```java
public static void main(String[] args) {

        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();
        String[] beanDefinitionNames = beanFactory.getBeanDefinitionNames();
        for (String beanDefinitionName : beanDefinitionNames) {
            System.err.println("beanDefinitionNames : " + beanDefinitionName);
        }

        // 一开始beanFactory没有任何的bean定义
        System.out.println(".........");

        // 通过XmlBeanDefinitionReader读取xml的bean定义信息，并且存放到beanFactory中
        XmlBeanDefinitionReader reader = new XmlBeanDefinitionReader(beanFactory);
        reader.loadBeanDefinitions(new ClassPathResource("b01.xml"));
        String[] beanDefinitionNames1 = beanFactory.getBeanDefinitionNames();
        for (String beanDefinitionName : beanDefinitionNames1) {
            System.err.println("beanDefinitionNames : " + beanDefinitionName);
        }

}
```
**执行结果:**
```java
.........
19:24:33.512 [main] DEBUG org.springframework.beans.factory.xml.XmlBeanDefinitionReader - Loaded 2 bean definitions from class path resource [b01.xml]
beanDefinitionNames : bean1
beanDefinitionNames : bean2
```


### 3.3 AnnotationConfigApplicationContext加载和获取bean

> 基于注解配置的方式加载bean定义

```java
private static void testAnnotationConfiqApplicationContext(){
        AnnotationConfigApplicationContext applicationContext = new AnnotationConfigApplicationContext(Config.class);
        for (String name : applicationContext.getBeanDefinitionNames()) {
            System.out.println("name : " + name);
        }
    }

    @Configuration
    static class Config {

        @Bean("bean1")
        public Bean1 getBean1(){
            return new Bean1();
        }

        @Bean("bean2")
        public Bean2 getBean2(Bean1 bean1){
            Bean2 bean2 = new Bean2();
            bean2.setBean1(bean1);
            return bean2;
        }

    }

    @Data
    @NoArgsConstructor
    static class Bean1{

    }

    @Data
    @NoArgsConstructor
    static class Bean2 {
        private Bean1 bean1;
    }
```
运行结果：

```java
19:43:04.578 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Autowiring by type from bean name 'bean2' via factory method to bean named 'bean1'
name : org.springframework.context.annotation.internalConfigurationAnnotationProcessor
name : org.springframework.context.annotation.internalAutowiredAnnotationProcessor
name : org.springframework.context.annotation.internalCommonAnnotationProcessor
name : org.springframework.context.event.internalEventListenerProcessor
name : org.springframework.context.event.internalEventListenerFactory
name : testApplicationContext.Config
name : bean1
name : bean2
```
这里我们不仅可以看到bean1和bean2,还可以看到我们的配置类 `testApplicationContext.Config`。  

而且，看打印结果，AnnotationConfigApplicationContext把一些默认的后置处理器也都加上了。



### 3.4 AnnotationConfigServletWebServerApplicationContext web应用容器

```java
private static void testAnnotationConfigServletWebServerApplicationContext(){
        AnnotationConfigServletWebServerApplicationContext context = new AnnotationConfigServletWebServerApplicationContext(WebConfig.class);
        for (String name : context.getBeanDefinitionNames()) {
            System.out.println("name : " + name);
        }
    }

    @Configuration
    static class WebConfig{
    	// 配置server
        @Bean
        public ServletWebServerFactory servletWebServerFactory (){
            return new TomcatServletWebServerFactory();
        }
        // 配置请求分发器
        @Bean
        public DispatcherServlet dispatchServlet (){
            return new DispatcherServlet();
        }
        // 配置server和dispatchServlet的绑定关系
        @Bean
        public DispatcherServletRegistrationBean dispatcherServletRegistrationBean(DispatcherServlet dispatcherServlet){
            // path 是限制哪些请求经过dispatchServlet来进行分发
            return new DispatcherServletRegistrationBean(dispatcherServlet, "/");
        }
        // 声明一个controller
        @Bean("/hello")
        public Controller myController(){
            return ((httpServletRequest, httpServletResponse) -> {
                httpServletResponse.getWriter().println("hello....");
                return null;
            });
        }
    }
```


## 4. Spring Beans的生命周期

### 4.1 bean的生命周期

```java
@Component
public class LifeCycleBean {

    public LifeCycleBean(){
        System.out.println("init LifeCycleBean");
    }

    @Autowired
    public void autowire(@Value("java_home") String javaHome){
        System.out.println("javaHome = " + javaHome);
    }

    @PostConstruct
    public void init(){
        System.out.println("初始化");
    }

    @PreDestroy
    public void destroy(){
        System.out.println("销毁");
    }

}
```

```java
@SpringBootApplication
public class TestSpringApplication {
    public static void main(String[] args)  {
        ConfigurableApplicationContext context = SpringApplication.run(TestSpringApplication.class, args);
        context.close();;
    }
}
```


### 4.2 bean的后置处理器

下面看一下BeanPostProcessor，**可以对bean做一些功能的增强**。

```java
@Slf4j
@Component
public class MyBeanPostProcessor implements InstantiationAwareBeanPostProcessor, DestructionAwareBeanPostProcessor {
    @Override
    // 实例化前（即调用构造方法前）执行的方法
    public Object postProcessBeforeInstantiation(Class<?> beanClass, String beanName) throws BeansException {
        if (beanName.equals("lifeCycleBean"))
            System.out.println("<<<<<<<<<<< 实例化前执行，如@PreDestroy");
        // 返回null保持原有对象不变，返回不为null，会替换掉原有对象
        return null;
    }

    @Override
    // 实例化后执行的方法
    public boolean postProcessAfterInstantiation(Object bean, String beanName) throws BeansException {
        if (beanName.equals("lifeCycleBean")) {
            System.out.println("<<<<<<<<<<< 实例化后执行，这里如果返回 false 会跳过依赖注入阶段");
            // return false;
        }

        return true;
    }

    @Override
    // 依赖注入阶段执行的方法
    public PropertyValues postProcessProperties(PropertyValues pvs, Object bean, String beanName) throws BeansException {
        if (beanName.equals("lifeCycleBean"))
            System.out.println("<<<<<<<<<<< 依赖注入阶段执行，如@Autowired、@Value、@Resource");
        return pvs;
    }

    @Override
    // 销毁前执行的方法
    public void postProcessBeforeDestruction(Object bean, String beanName) throws BeansException {
        if (beanName.equals("lifeCycleBean"))
            System.out.println("<<<<<<<<<<<销毁之前执行");
    }

    @Override
    // 初始化之前执行的方法
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        if (beanName.equals("lifeCycleBean"))
            System.out.println("<<<<<<<<<<< 初始化之前执行，这里返回的对象会替换掉原本的bean，如 @PostConstruct、@ConfigurationProperties");
        return bean;
    }

    @Override
    // 初始化之后执行的方法
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (beanName.equals("lifeCycleBean"))
            System.out.println("<<<<<<<<<<< 初始化之后执行，这里返回的对象会替换掉原本的bean，如 代理增强");
        return bean;
    }
}
```
**执行结果：**

```java
<<<<<<<<<<< 实例化前执行，如@PreDestroy
init LifeCycleBean
<<<<<<<<<<< 实例化后执行，这里如果返回 false 会跳过依赖注入阶段
<<<<<<<<<<< 依赖注入阶段执行，如@Autowired、@Value、@Resource
javaHome = java_home
<<<<<<<<<<< 初始化之前执行，这里返回的对象会替换掉原本的bean，如 @PostConstruct、@ConfigurationProperties
初始化
<<<<<<<<<<< 初始化之后执行，这里返回的对象会替换掉原本的bean，如 代理增强
2022-11-28 20:34:27.519  INFO 2260 --- [           main] o.s.s.concurrent.ThreadPoolTaskExecutor  : Initializing ExecutorService 'applicationTaskExecutor'
2022-11-28 20:34:27.676  INFO 2260 --- [           main] o.s.b.w.embedded.tomcat.TomcatWebServer  : Tomcat started on port(s): 8080 (http) with context path ''
2022-11-28 20:34:27.687  INFO 2260 --- [           main] c.e.t.test3.TestSpringApplication        : Started TestSpringApplication in 1.938 seconds (JVM running for 2.513)
2022-11-28 20:34:28.956  INFO 2260 --- [           main] o.s.s.concurrent.ThreadPoolTaskExecutor  : Shutting down ExecutorService 'applicationTaskExecutor'
<<<<<<<<<<<销毁之前执行
销毁
```


### 4.3 模版方法-设计模式



```java
public class TestMethodTemplatePattern {
    public static void main(String[] args) {
        MyBeanFactory beanFactory = new MyBeanFactory();
        beanFactory.addBeanPostProcessor(bean -> System.out.println("解析 @Autowired"));
        beanFactory.addBeanPostProcessor(bean -> System.out.println("解析 @Resource"));
        beanFactory.getBean();
    }

    static class MyBeanFactory {
        public Object getBean() {
            Object bean = new Object();
            System.out.println("构造：" + bean);
            System.out.println("依赖注入：" + bean);
            for (BeanPostProcessor processor : processors) {
                processor.inject(bean);
            }
            System.out.println("初始化：" + bean);
            return bean;
        }

        private List<BeanPostProcessor> processors = new ArrayList<>();

        public void addBeanPostProcessor(BeanPostProcessor beanPostProcessor) {
            processors.add(beanPostProcessor);
        }
    }

    interface BeanPostProcessor {
        void inject(Object bean);
    }
}

```

**运行结果：**

```java
构造：java.lang.Object@5e9f23b4
依赖注入：java.lang.Object@5e9f23b4
解析 @Autowired
解析 @Resource
初始化：java.lang.Object@5e9f23b4
```

这种方式的好处就是在扩展功能的时候，不需要修改原来代码的逻辑，只需要按需添加新的功能就可以了。

模版方法中有静有动，静指的是哪些固定的步骤或者逻辑，动指的是可以后续扩展的功能，可以事先定义成接口，需要的时候，具体的实现。



## 5. Bean后处理器



### 5.1 常见的BeanPostProcessor

[**课程链接**](https://www.bilibili.com/video/BV1P44y1N7QG?p=16&vd_source=3ff1db20d26ee8426355e893ae553d51)

**Bean后处理器的主要作用**就是 **为Bean的生命周期的各个阶段进行扩展**

下面说一下下面这种写法：

```java
@Autowired
public void autowire(@Value("${java.home}") String javaHome){
    System.out.println("javaHome = " + javaHome);
}
```

通常我们可能使用的是直接在属性上面添加 @Value注解就可了。但是上面这种方式实现的效果是一样的，并且是可以打印出来具体的值的。

如果，autowire方法的参数不是String，而是一个对象，Spring会到容器中找到对应类型的bean，并且注入到autowire方法里面。

**GenericApplicationContext**
> 1. GenericApplicationContext 是一个【干净】的容器，默认不会添加任何后处理器，方便做测试  
2. 这里用DefaultListableBeanFactory也可以完成测试，只是会比使用GenericApplicationContext麻烦一些
3. DefaultListableBeanFactory中自己添加了一些bean的后置处理器


首先看一下一些用到的类是如何定义的：

**Bean1**
```java
package com.example.testspring.a04;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;

public class Bean1 {

    private Bean2 bean2;

    @Autowired
    public void setBean2(Bean2 bean2){
        System.out.println("@Autowired done");
        this.bean2 = bean2;
    }

    private Bean3 bean3;

    @Resource
    public void setBean3(Bean3 bean3){
        System.out.println("@Resource done");
        this.bean3 = bean3;
    }

    private String home;

    @Autowired
    public void autowire(@Value("${java.home}") String home){
        System.out.println("@Value done");
        this.home = home;
    }

    @PostConstruct
    public void init(){
        System.out.println("初始化 done");
    }

    @PreDestroy
    public void destroy(){
        System.out.println("销毁.. done");
    }

    @Override
    public String toString() {
        return "Bean1{" +
                "bean2=" + bean2 +
                ", bean3=" + bean3 +
                ", home='" + home + '\'' +
                '}';
    }
}

```

**@Autowired + @Value 两个可以配合使用，做一些值注入。**

**Bean2**

```java
package com.example.testspring.a04;

public class Bean2 {
}

```

**Bean3**
```java
package com.example.testspring.a04;

public class Bean3 {
}

```

**Bean4**
```java
package com.example.testspring.a04;

import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;

@Data
@Configuration
@ConfigurationProperties(prefix = "java")
public class Bean4 {

    private String version;
    private String home;
}

```

**测试代码**

```java
 public static void main(String[] args) {
        // ⬇️GenericApplicationContext 是一个【干净】的容器，默认不会添加任何后处理器，方便做测试
        // 这里用DefaultListableBeanFactory也可以完成测试，只是会比使用GenericApplicationContext麻烦一些
        GenericApplicationContext context = new GenericApplicationContext();

        // 用原始方法注册三个Bean
        context.registerBean("bean1", Bean1.class);
        context.registerBean("bean2", Bean2.class);
        context.registerBean("bean3", Bean3.class);
        // ⬇️初始化容器
        context.refresh();
        // ⬇️销毁容器
        context.close();
    }
```

**context.refresh();**    

1. 执行beanFactory后置处理器
2. 添加bean后置处理器
3. 初始化所有的bean单例


下图为实际效果：👇

![](https://oscimg.oschina.net/oscnet/up-d4ede0a13afe9ea31b3b7cb207c4347afd3.png)

**可以看到bean1中的方法中的参数都没有打印出来,,,**

然后我们手动添加对应的处理器之后在尝试打印：

**设置解析 @Value 注解的解析器**

```java
context.getDefaultListableBeanFactory().setAutowireCandidateResolver(new ContextAnnotationAutowireCandidateResolver());
```

**添加解析 @Autowired 和 @Value 注解的后处理器**

```java
context.registerBean(AutowiredAnnotationBeanPostProcessor.class);
```

**添加解析 @Resource、@PostConstruct、@PreDestroy 注解的后处理器**

```java
 context.registerBean(CommonAnnotationBeanPostProcessor.class);
```

**然后看一下执行效果：**

![](https://oscimg.oschina.net/oscnet/up-75b7149bce0a9ac942d98f39763eeb62fed.png)


然后看一下Bean4的定义，看一下配置bean是如何加载的：

```java
@Data
@Configuration
@ConfigurationProperties(prefix = "java")
public class Bean4 {

    private Integer version;
    private String home;
}
```

没有添加对应后置处理器之前，直接打印 `System.out.println(context.getBean(Bean4.class));`

```java
Bean4(version=null, home=null)
```

**添加对应的处理器：**


```java
// 添加解析 @ConfigurationProperties注解的后处理器
// ConfigurationPropertiesBindingPostProcessor后处理器不能像上面几种后处理器那样用context直接注册上去
//context.registerBean(ConfigurationPropertiesBindingPostProcessor.class);
// 需要反着来注册一下
ConfigurationPropertiesBindingPostProcessor.register(context.getDefaultListableBeanFactory());
```

**查看打印效果：**

```java
Bean4(version=1.8.0_333, home=/Library/Java/JavaVirtualMachines/jdk1.8.0_333.jdk/Contents/Home/jre)
```

最后完整代码：

```java
public class TestBeanPostProcessor {
    public static void main(String[] args) {
        // ⬇️GenericApplicationContext 是一个【干净】的容器，默认不会添加任何后处理器，方便做测试
        // 这里用DefaultListableBeanFactory也可以完成测试，只是会比使用GenericApplicationContext麻烦一些
        GenericApplicationContext context = new GenericApplicationContext();

        // 用原始方法注册三个Bean
        context.registerBean("bean1", Bean1.class);
        context.registerBean("bean2", Bean2.class);
        context.registerBean("bean3", Bean3.class);
        context.registerBean("bean4", Bean4.class);


        // 设置解析 @Value 注解的解析器
        context.getDefaultListableBeanFactory().setAutowireCandidateResolver(new ContextAnnotationAutowireCandidateResolver());
        // 添加解析 @Autowired 和 @Value 注解的后处理器
        context.registerBean(AutowiredAnnotationBeanPostProcessor.class);
        // 添加解析 @Resource、@PostConstruct、@PreDestroy 注解的后处理器
        context.registerBean(CommonAnnotationBeanPostProcessor.class);
        // 添加解析 @ConfigurationProperties注解的后处理器
        // ConfigurationPropertiesBindingPostProcessor后处理器不能像上面几种后处理器那样用context直接注册上去
//         context.registerBean(ConfigurationPropertiesBindingPostProcessor.class);
        // 需要反着来注册一下
        ConfigurationPropertiesBindingPostProcessor.register(context.getDefaultListableBeanFactory());
        // ⬇️初始化容器
        context.refresh();
        System.out.println(context.getBean(Bean4.class));
        // ⬇️销毁容器
        context.close();
    }
}
```



### 5.2 AutowiredAnnotationBeanPostProcessor

下面分析AutowiredAnnotationBeanPostProcessor处理器处理的流程👇：


#### 5.2.1 作用

- 可以解析@Autowired注解
- 可以解析@Value注解

#### 5.2.2 运行分析

​	**beanFactory.registerSingleton**方法注册的是一个实例化好的对象，不会在去用工厂走创建，依赖注入，实例化等过程。


```java
public class BigInAutowired {

    public static void main(String[] args) {
        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();

        beanFactory.registerSingleton("bean2", new Bean2());
        beanFactory.registerSingleton("bean3", new Bean3());
        // 解析@Value
        beanFactory.setAutowireCandidateResolver(new ContextAnnotationAutowireCandidateResolver());

        // 查找那些属性或者方法加了 @Autowired
        AutowiredAnnotationBeanPostProcessor postProcessor = new AutowiredAnnotationBeanPostProcessor();
        // 执行依赖注入的时候，可能需要去beanFactory中找一些bean, 比如bean1 依赖 bean2，在初始化bean1的时候去beanFactory中找bean2的实例，赋值给bean1的属性
        postProcessor.setBeanFactory(beanFactory);

        Bean1 bean1 = new Bean1();
        System.out.println("bean1 : " + bean1);
        // 调用后处理器的postProcessProperties方法，在依赖注入阶段会执行这个方法。
        // 执行依赖注入，解析@Autowired @Value
        // 第一个参数，指定每个bean1中那个属性是什么值，不走自动状态的逻辑，所以写成null
        postProcessor.postProcessProperties(null, bean1, "bean1");
        System.out.println("bean1 : " + bean1);
        
    }
}
```
**运行结果：**

```java
bean1 : Bean1{bean2=null, bean3=null, home='null'}
@Value done
@Autowired done
bean1 : Bean1{bean2=com.example.testspring.a04.Bean2@1a968a59, bean3=null, home='${java.home}'}
```

**postProcessor.postProcessProperties(null, bean1, "bean1");	方法会去解析@Autowired 和@Value注解。**

我们可以看一下 `postProcessProperties` 这个方法，是个私有方法。 是AutowiredAnnotationBeanPostProcessor后置处理器的核心方法；

```java
public PropertyValues postProcessProperties(PropertyValues pvs, Object bean, String beanName) {
  		// 先找那些变量或者方法，添加了@Autowired
        InjectionMetadata metadata = this.findAutowiringMetadata(beanName, bean.getClass(), pvs);

        try {
          // 通过反射进行属性的赋值
            metadata.inject(bean, beanName, pvs);
            return pvs;
        } catch (BeanCreationException var6) {
            throw var6;
        } catch (Throwable var7) {
            throw new BeanCreationException(beanName, "Injection of autowired dependencies failed", var7);
        }
    }
```

**下面来模拟一下后置处理器的处理过程：**

**findAutowiringMetadata**这个方法是一个私有的方法，可以通过反射的方式，来调用。



![](https://oscimg.oschina.net/oscnet/up-9e0c1678b6cc8316e458fdea29b9e1ec8aa.png)



代码如下：

```java
public class BigInAutowired {

    public static void main(String[] args) throws Throwable {
        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();

        beanFactory.registerSingleton("bean2", new Bean2());
        beanFactory.registerSingleton("bean3", new Bean3());
        // 解析@Value
        beanFactory.setAutowireCandidateResolver(new ContextAnnotationAutowireCandidateResolver());

        // 查找那些属性或者方法加了 @Autowired
        AutowiredAnnotationBeanPostProcessor postProcessor = new AutowiredAnnotationBeanPostProcessor();
        // 执行依赖注入的时候，可能需要去beanFactory中找一些bean, 比如bean1 依赖 bean2，在初始化bean1的时候去beanFactory中找bean2的实例，赋值给bean1的属性
        postProcessor.setBeanFactory(beanFactory);

        Bean1 bean1 = new Bean1();
        System.out.println("bean1 : " + bean1);

        // 下面两步模拟后置处理的过程：

        Method findAutowiringMetadata = AutowiredAnnotationBeanPostProcessor.class.getDeclaredMethod("findAutowiringMetadata", String.class, Class.class, PropertyValues.class);
        findAutowiringMetadata.setAccessible(true);
        // 获取bean1上加了@Value @Autowired 注解的属性和方法 等信息
        InjectionMetadata injectionMetadata = (InjectionMetadata) findAutowiringMetadata.invoke(postProcessor, "bean1", Bean1.class, null);

        // 拿到元数据之后，执行依赖注入，注入时按照类型查找
        injectionMetadata.inject(bean1, "bean1", null);
        System.out.println("bean1 : " + bean1);

    }
}
```

**执行结果**：

```
bean1 : Bean1{bean2=null, bean3=null, home='null'}
@Autowired done
@Value done
bean1 : Bean1{bean2=com.example.testspring.a04.Bean2@6b26e945, bean3=null, home='${java.home}'}
```

根据上面的执行结果可以看到，home='${java.home}' 没有解析出来，需要在beanFactory上面添加占位符的解析器才可以：

```java
// 解析 ${}
beanFactory.addEmbeddedValueResolver(new StandardEnvironment()::resolvePlaceholders);
```

再次运行，查看结果：

```
bean1 : Bean1{bean2=null, bean3=null, home='null'}
17:27:08.577 [main] DEBUG org.springframework.core.env.PropertySourcesPropertyResolver - Found key 'java.home' in PropertySource 'systemProperties' with value of type String
@Value done
@Autowired done
bean1 : Bean1{bean2=com.example.testspring.a04.Bean2@4c9f8c13, bean3=null, home='/Library/Java/JavaVirtualMachines/jdk1.8.0_333.jdk/Contents/Home/jre'}
```

完整代码如下：

```java
public class BigInAutowired {

    public static void main(String[] args) throws Throwable {
        DefaultListableBeanFactory beanFactory = new DefaultListableBeanFactory();

        beanFactory.registerSingleton("bean2", new Bean2());
        beanFactory.registerSingleton("bean3", new Bean3());
        // 解析@Value
        beanFactory.setAutowireCandidateResolver(new ContextAnnotationAutowireCandidateResolver());
        // 解析 ${}
        beanFactory.addEmbeddedValueResolver(new StandardEnvironment()::resolvePlaceholders);

        // 查找那些属性或者方法加了 @Autowired
        AutowiredAnnotationBeanPostProcessor postProcessor = new AutowiredAnnotationBeanPostProcessor();
        // 执行依赖注入的时候，可能需要去beanFactory中找一些bean, 比如bean1 依赖 bean2，在初始化bean1的时候去beanFactory中找bean2的实例，赋值给bean1的属性
        postProcessor.setBeanFactory(beanFactory);

        Bean1 bean1 = new Bean1();
        System.out.println("bean1 : " + bean1);

        // 下面两步模拟后置处理的过程：

        Method findAutowiringMetadata = AutowiredAnnotationBeanPostProcessor.class.getDeclaredMethod("findAutowiringMetadata", String.class, Class.class, PropertyValues.class);
        findAutowiringMetadata.setAccessible(true);
        // 获取bean1上加了@Value @Autowired 注解的属性和方法 等信息
        InjectionMetadata injectionMetadata = (InjectionMetadata) findAutowiringMetadata.invoke(postProcessor, "bean1", Bean1.class, null);

        // 拿到元数据之后，执行依赖注入，注入时按照类型查找
        injectionMetadata.inject(bean1, "bean1", null);
        System.out.println("bean1 : " + bean1);

    }
}
```

### 小结

1. AutowiredAnnotationBeanPostProcessor在执行的时候，核心方法就是findAutowiringMetadata
2. findAutowiringMetadata方法的核心，是先去找到那些方法或者属性添加的@Autowired或者@Value，然后在按照类型注入
3. 我们模拟了一下AutowiredAnnotationBeanPostProcessor的处理，先用反射调用findAutowiringMetadata，然后元数据，再injectionMetadata.inject(bean1, "bean1", null); 执行依赖注入。



### 5.3 InjectionMetadata分析

根据上面一小节我们可以知道，InjectionMetadata中存储了添加@Autowaired和@Value的属性或者方法的数据。

那 injectionMetadata.inject(bean1, "bean1", null); 方法做了那些事情呢？

#### **5.3.1如何按照类型查找值**

```java
// 按照类型查找属性值
Field bean3 = Bean1.class.getDeclaredField("bean3");
DependencyDescriptor dd1 = new DependencyDescriptor(bean3, false);
Object o = beanFactory.doResolveDependency(dd1, null, null, null);
System.out.println(o);
```

执行结果：

```
bean1 : Bean1{bean2=null, bean3=null, home='null'}
@Autowired done
17:39:48.117 [main] DEBUG org.springframework.core.env.PropertySourcesPropertyResolver - Found key 'java.home' in PropertySource 'systemProperties' with value of type String
@Value done
bean1 : Bean1{bean2=com.example.testspring.a04.Bean2@4b5d6a01, bean3=com.example.testspring.a04.Bean3@4a22f9e2, home='/Library/Java/JavaVirtualMachines/jdk1.8.0_333.jdk/Contents/Home/jre'}
com.example.testspring.a04.Bean3@4a22f9e2
```

分析：

doResolveDependency能够根据属性的类型，在beanFactory中找对对应的bean，然后设置到对应的属性上。

方法上添加@Autowired也是同理的：

```java
// 方法上面添加@Autowired
Method setBean2 = Bean1.class.getDeclaredMethod("setBean2", Bean2.class);
DependencyDescriptor dd2 = new DependencyDescriptor(new MethodParameter(setBean2, 0), false);
Object o1 = beanFactory.doResolveDependency(dd2, null, null, null);
System.out.println(o1);
```

如果我们吧beanFactory中去掉bean2，也就是从工厂中查找不到bean2，同时，依赖注入的时候设置成true。

```java
beanFactory.registerSingleton("bean2", new Bean2());	// 这行注释掉
DependencyDescriptor dd2 = new DependencyDescriptor(new MethodParameter(setBean2, 0), true);
```

**执行结果如下：**

```java
bean1 : Bean1{bean2=null, bean3=null, home='null'}
Exception in thread "main" org.springframework.beans.factory.UnsatisfiedDependencyException: Error creating bean with name 'bean1': Unsatisfied dependency expressed through method 'setBean2' parameter 0; nested exception is org.springframework.beans.factory.NoSuchBeanDefinitionException: No qualifying bean of type 'com.example.testspring.a04.Bean2' available: expected at least 1 bean which qualifies as autowire candidate. Dependency annotations: {}
	at org.springframework.beans.factory.annotation.AutowiredAnnotationBeanPostProcessor$AutowiredMethodElement.inject(AutowiredAnnotationBeanPostProcessor.java:723)
	at org.springframework.beans.factory.annotation.InjectionMetadata.inject(InjectionMetadata.java:130)
	at com.example.testspring.a04.BigInAutowired.main(BigInAutowired.java:49)
Caused by: org.springframework.beans.factory.NoSuchBeanDefinitionException: No qualifying bean of type 'com.example.testspring.a04.Bean2' available: expected at least 1 bean which qualifies as autowire candidate. Dependency annotations: {}
	at org.springframework.beans.factory.support.DefaultListableBeanFactory.raiseNoMatchingBeanFound(DefaultListableBeanFactory.java:1717)
	at org.springframework.beans.factory.support.DefaultListableBeanFactory.doResolveDependency(DefaultListableBeanFactory.java:1273)
	at org.springframework.beans.factory.support.DefaultListableBeanFactory.resolveDependency(DefaultListableBeanFactory.java:1227)
	at org.springframework.beans.factory.annotation.AutowiredAnnotationBeanPostProcessor$AutowiredMethodElement.inject(AutowiredAnnotationBeanPostProcessor.java:715)
	... 2 more
```















































