import difflib
import json
from zhconv import convert
import string
from typing import Any, Dict, List, Union, Optional
import pandas as pd

from maa.agent.agent_server import AgentServer
from maa.custom_recognition import CustomRecognition
from maa.context import Context
from maa.define import RectType
from utils.logger import logger


@AgentServer.custom_recognition("MonopolyStatsRecord")
class MonopolyStatsRecord(CustomRecognition):
    """
    读取玩家的六项数值
    """

    def analyze(
        self, context: Context, argv: CustomRecognition.AnalyzeArg
    ) -> Union[CustomRecognition.AnalyzeResult, Optional[RectType]]:
        STATS_ROIS = [
            [118, 156, 57, 28],
            [201, 159, 45, 23],
            [281, 159, 45, 24],
            [124, 207, 43, 21],
            [201, 206, 43, 21],
            [281, 203, 44, 25],
        ]
        stats = []
        for roi in STATS_ROIS:
            reco_detail = context.run_recognition(
                "大富翁-读取个人数值", argv.image, {"大富翁-读取个人数值": {"roi": roi}}
            )
            if (
                not reco_detail
                or not getattr(reco_detail, "hit", False)
                or not reco_detail.best_result
            ):
                logger.info(f"未在 ROI {roi} 识别到属性数值")
                return None
            stat = reco_detail.best_result.text
            # logger.info(f"已在roi:{roi}区域识别到属性数值{stat}")
            stats.append(int(stat))
        MonopolyStatsRecord.stats = stats
        logger.info(
            f"已读取当前属性：智慧{stats[0]}，武力{stats[1]}, 幸运{stats[2]}，领袖{stats[3]}，气质{stats[4]}，口才{stats[5]}"
        )
        return CustomRecognition.AnalyzeResult(box=[0, 0, 0, 0], detail=str(stats))


@AgentServer.custom_recognition("MonopolySinglePkStats")
class MonopolySinglePkStats(CustomRecognition):
    """
    在 PK 界面读取属性、数值要求及事件内容
    """

    def __init__(self):
        super().__init__()
        self.description_bank = self.read_excel("agent/monopoly.xlsx")
        self.similarity_threshold = 0.5  # 相似度阈值

    @staticmethod
    def split_name_value(s: str):
        mapping = {
            "智慧": "智慧",
            "武力": "武力",
            "幸運": "幸运",
            "幸运": "幸运",
            "領袖": "领袖",
            "领袖": "领袖",
            "氣質": "气质",
            "气质": "气质",
            "口才": "口才",
        }
        for word in mapping:
            if s.startswith(word):
                return mapping[word], s[len(word) :]
        return None, None

    def clean_text(self, text):
        # 强制转成字符串，避免 int/None 出错
        text = str(text) if text is not None else ""

        # 创建翻译表，将所有标点符号和空格映射为 None
        chinese_punctuation = "，。！？【】（）《》“”‘’；：、——·〈〉……—"
        translator = str.maketrans(
            "", "", string.punctuation + chinese_punctuation + " \t\n\r\u3000"
        )

        # 使用翻译表移除标点符号和空格
        cleaned_text = text.translate(translator)
        return cleaned_text.strip()

    def read_excel(self, file_path):
        # 读取第1个sheet并跳过第一行
        df = pd.read_excel(file_path, sheet_name=0).iloc[1:]

        results = []
        for _, row in df.iterrows():
            description = self.clean_text(row.iloc[0])
            label = row.iloc[1]
            results.append({"d": description, "label": label})
        return results

    def find_label(self, description: str):
        best_match = None
        max_sim = 0

        for item in self.description_bank:
            # 截断到前 25 个字
            item_d = item["d"][:25] if len(item["d"]) > 25 else item["d"]
            sim = difflib.SequenceMatcher(None, description, item_d).ratio()
            if sim > max_sim:
                max_sim = sim
                best_match = item

        label_map = {
            1: "炸工坊",
            2: "减税收",
        }
        if max_sim > self.similarity_threshold:
            label_text = label_map.get(best_match["label"])
            # 如果找到了，返回结果
            logger.info(
                f"已匹配到【{label_text}】事件: {best_match['d']}, 相似度：{max_sim}"
            )
        else:
            logger.info("未匹配到炸工坊事件")
            return None
        return best_match["d"] if max_sim > self.similarity_threshold else None

    def analyze(
        self, context: Context, argv: CustomRecognition.AnalyzeArg
    ) -> Union[CustomRecognition.AnalyzeResult, Optional[RectType]]:
        reco_detail = context.run_recognition("大富翁-读取PK要求", argv.image)
        if (
            not reco_detail
            or not getattr(reco_detail, "hit", False)
            or not reco_detail.best_result
        ):
            logger.info("未识别到 PK 要求")
            return None
        stat_name_n_value = reco_detail.best_result.text
        stat_name, value = self.split_name_value(stat_name_n_value)
        logger.info(f"识别到PK要求：{stat_name} {value}")

        description_detail = context.run_recognition(
            "大富翁-读取PK事件内容", argv.image
        )
        raw_description = ""
        if (
            description_detail
            and getattr(description_detail, "hit", False)
            and description_detail.filtered_results
        ):
            for r in description_detail.filtered_results:
                raw_description = raw_description + r.text
        else:
            logger.info("警告：未能识别到事件内容")
        cleaned_description = self.clean_text(raw_description)
        description = convert(cleaned_description, "zh-cn")
        # logger.info(f"{description}")

        label = self.find_label(description)
        if label:
            suggestion = True
        else:
            suggestion = False

        STATS_ROIS = [
            [118, 156, 57, 28],
            [201, 159, 45, 23],
            [281, 159, 45, 24],
            [124, 207, 43, 21],
            [201, 206, 43, 21],
            [281, 203, 44, 25],
        ]
        pc_stats = []
        for roi in STATS_ROIS:
            reco_detail = context.run_recognition(
                "大富翁-读取个人数值", argv.image, {"大富翁-读取个人数值": {"roi": roi}}
            )
            if (
                not reco_detail
                or not getattr(reco_detail, "hit", False)
                or not reco_detail.best_result
            ):
                logger.info(f"未在 ROI {roi} 读取到玩家属性数值")
                return None
            pc_stat = reco_detail.best_result.text
            # logger.info(f"已在roi:{roi}区域识别到属性数值{stat}")
            pc_stats.append(int(pc_stat))

        pkstats = [stat_name, int(value), description, label, suggestion, pc_stats]
        # logger.info(f"{pkstats}")
        MonopolySinglePkStats.pkstats = pkstats
        return CustomRecognition.AnalyzeResult(box=[0, 0, 0, 0], detail=str(pkstats))


@AgentServer.custom_recognition("MonopolyOfficeRecord")
class MonopolyOfficeRecord(CustomRecognition):
    """
    识别并记录公务事件名称
    """

    def analyze(
        self, context: Context, argv: CustomRecognition.AnalyzeArg
    ) -> Union[CustomRecognition.AnalyzeResult, Optional[RectType]]:
        event_name = ""
        reco_detail = context.run_recognition("大富翁-读取公务事件名称", argv.image)
        if (
            not reco_detail
            or not getattr(reco_detail, "hit", False)
            or not reco_detail.best_result
        ):
            logger.info("未识别到公务事件名称")
            return None
        raw_text = reco_detail.best_result.text
        event_name = convert(raw_text, "zh-cn")
        MonopolyOfficeRecord.event_name = event_name
        logger.info(f"识别到公务事件：(原文){raw_text},(简中){event_name}")
        return CustomRecognition.AnalyzeResult(box=[0, 0, 0, 0], detail=event_name)
