import { socialUserInfoGuard } from '@logto/connector-kit';
import { z } from 'zod';

export { type ExtendedSocialUserInfo, extendedSocialUserInfoGuard } from '@logto/schemas';

// Since the SAML SSO user info will extend the basic social user info (will contain extra info like `organization`, `role` etc.), but for now we haven't decide what should be included in extended user info, so we just use the basic social user info guard here to keep SSOT.
const samlAttributeMappingGuard = socialUserInfoGuard
  .pick({
    id: true,
    email: true,
    name: true,
  })
  .required();

export const defaultAttributeMapping: AttributeMap = {
  id: 'nameID',
  email: 'email',
  name: 'name',
};

const customizableAttributeMapGuard = samlAttributeMappingGuard.partial();

export type CustomizableAttributeMap = z.infer<typeof customizableAttributeMapGuard>;

export type AttributeMap = z.infer<typeof samlAttributeMappingGuard>;

/**
 * This is the metadata of SAML service provider, automatically generated by `tenantId` and `ssoConnectorId`.
 * See details in {@link @logto/core/src/sso/SamlConnector/index.ts}.
 */
const samlServiceProviderMetadataGuard = z.object({
  entityId: z.string().min(1),
  assertionConsumerServiceUrl: z.string().min(1),
});

export type SamlServiceProviderMetadata = z.infer<typeof samlServiceProviderMetadataGuard>;

/**
 * We only concern about the `entityId`, `signInEndpoint` and `x509Certificate` for now.
 */
export const samlIdentityProviderMetadataGuard = z.object({
  entityId: z.string(),
  signInEndpoint: z.string(),
  x509Certificate: z.string(),
  certificateExpiresAt: z.number(), // Timestamp in milliseconds.
  isCertificateValid: z.boolean(),
});
export type SamlIdentityProviderMetadata = z.infer<typeof samlIdentityProviderMetadataGuard>;

export const manualSamlConnectorConfigGuard = samlIdentityProviderMetadataGuard.pick({
  entityId: true,
  signInEndpoint: true,
  x509Certificate: true,
});

export const samlConnectorConfigGuard = z.union([
  // Config using Metadata URL
  z.object({
    metadataUrl: z.string(),
    attributeMapping: customizableAttributeMapGuard.optional(),
  }),
  // Config using Metadata XML
  z.object({
    metadata: z.string(),
    attributeMapping: customizableAttributeMapGuard.optional(),
  }),
  // Config using Metadata detail
  manualSamlConnectorConfigGuard.extend({
    attributeMapping: customizableAttributeMapGuard.optional(),
  }),
]);
export type SamlConnectorConfig = z.infer<typeof samlConnectorConfigGuard>;

const samlMetadataGuard = z.object({
  defaultAttributeMapping: samlAttributeMappingGuard,
  serviceProvider: samlServiceProviderMetadataGuard,
  identityProvider: samlIdentityProviderMetadataGuard.optional(),
});

export type SamlMetadata = z.infer<typeof samlMetadataGuard>;
