import { type TokenResponse } from '@logto/connector-kit';
import { type ExtendedSocialUserInfo } from '@logto/schemas';
import { generateStandardId } from '@logto/shared/universal';
import { conditional } from '@silverhand/essentials';
import camelcaseKeys from 'camelcase-keys';
import snakecaseKeys from 'snakecase-keys';

import { EnvSet } from '#src/env-set/index.js';
import assertThat from '#src/utils/assert-that.js';

import { SsoConnectorError, SsoConnectorErrorCodes } from '../types/error.js';
import {
  type OidcTokenResponse,
  scopePostProcessor,
  type BaseOidcConfig,
  type BasicOidcConnectorConfig,
} from '../types/oidc.js';
import {
  type CreateSingleSignOnSession,
  type SingleSignOnConnectorSession,
} from '../types/session.js';

import { mockGetTokenResponse, mockGetUserInfo } from './test-utils.js';
import {
  fetchOidcConfig,
  fetchToken,
  getIdTokenClaims,
  getTokenByRefreshToken,
  getUserInfo,
} from './utils.js';

/**
 * OIDC connector
 *
 * @remark General connector for OIDC provider.
 * This class provides the basic functionality to connect with a OIDC provider.
 * All the OIDC single sign-on connector should extend this class.
 *  @see @logto/connector-kit.
 *
 * @property config The OIDC connector config
 * @method getOidcConfig Fetch the full-list of OIDC config from the issuer. Throws error if config is invalid
 * @method getAuthorizationUrl Generate the authorization URL for the OIDC provider
 * @method getUserInfo Handle the sign-in callback from the OIDC provider and return the user info
 */
class OidcConnector {
  constructor(private readonly config: BasicOidcConnectorConfig) {}

  /* Fetch the full-list of OIDC config from the issuer. Throws error if config is invalid */
  async getOidcConfig(): Promise<BaseOidcConfig> {
    const { issuer, scope } = this.config;

    const oidcConfig = await fetchOidcConfig(issuer);

    return {
      ...this.config,
      ...oidcConfig,
      scope: scopePostProcessor(scope),
    };
  }

  /**
   * Generate the authorization URL for the OIDC provider
   *
   * @param oidcQueryParams The query params for the OIDC provider
   * @param oidcQueryParams.state The state generated by Logto experience client
   * @param oidcQueryParams.redirectUri The redirect uri for the OIDC provider
   * @param setSession Set the connector session data to the oidc provider session storage.
   */
  async getAuthorizationUrl(
    {
      state,
      redirectUri,
      connectorId,
    }: { state: string; redirectUri: string; connectorId: string },
    setSession: CreateSingleSignOnSession,
    prompt?: 'login' | 'consent' | 'none' | 'select_account'
  ) {
    const oidcConfig = await this.getOidcConfig();
    const nonce = generateStandardId();

    await setSession({ nonce, redirectUri, connectorId, state });

    const queryParameters = new URLSearchParams({
      state,
      nonce,
      ...snakecaseKeys({
        clientId: oidcConfig.clientId,
        responseType: 'code',
        redirectUri,
      }),
      ...conditional(prompt && { prompt }),
      scope: oidcConfig.scope,
    });

    return `${oidcConfig.authorizationEndpoint}?${queryParameters.toString()}`;
  }

  get issuer() {
    return this.config.issuer;
  }

  /**
   * Handle the sign-in callback from the OIDC provider and return the user info
   *
   * @param data unknown oidc authorization response
   * @param connectorSession The connector session data from the oidc provider session storage
   * @returns The user info from the OIDC provider
   */
  async getUserInfo(
    connectorSession: SingleSignOnConnectorSession,
    data: unknown
  ): Promise<{ userInfo: ExtendedSocialUserInfo; tokenResponse?: OidcTokenResponse }> {
    const { isIntegrationTest } = EnvSet.values;

    if (isIntegrationTest) {
      return {
        userInfo: mockGetUserInfo(connectorSession, data),
        tokenResponse: mockGetTokenResponse(data),
      };
    }

    const oidcConfig = await this.getOidcConfig();
    const { nonce, redirectUri } = connectorSession;

    // Fetch token from the OIDC provider using authorization code
    const tokenResponse = await fetchToken(oidcConfig, data, redirectUri);
    const { accessToken, idToken } = camelcaseKeys(tokenResponse);

    assertThat(
      accessToken,
      new SsoConnectorError(SsoConnectorErrorCodes.AuthorizationFailed, {
        message: 'The access token is missing from the response.',
      })
    );

    // Verify the id token and get the user claims
    const idTokenClaims = await getIdTokenClaims(idToken, oidcConfig, nonce);

    // If userinfo endpoint is not provided, use the id token claims as user info,
    // otherwise, fetch the user info from the userinfo endpoint
    const { sub, name, picture, email, email_verified, phone, phone_verified, ...rest } =
      oidcConfig.userinfoEndpoint
        ? await getUserInfo(accessToken, oidcConfig.userinfoEndpoint)
        : idTokenClaims;

    return {
      userInfo: {
        id: sub,
        ...conditional(name && { name }),
        ...conditional(picture && { avatar: picture }),
        ...conditional(email && email_verified && { email }),
        ...conditional(phone && phone_verified && { phone }),
        ...camelcaseKeys(rest),
      },
      tokenResponse,
    };
  }

  async getTokenByRefreshToken(refreshToken: string): Promise<TokenResponse> {
    const oidcConfig = await this.getOidcConfig();
    return getTokenByRefreshToken(oidcConfig, refreshToken);
  }
}

export default OidcConnector;
