import { builtInLanguageOptions as consoleBuiltInLanguageOptions } from '@logto/phrases';
import { Theme } from '@logto/schemas';
import classNames from 'classnames';
import { useRef, useState } from 'react';
import { useTranslation } from 'react-i18next';

import ExternalLinkIcon from '@/assets/icons/external-link.svg?react';
import Globe from '@/assets/icons/globe.svg?react';
import Palette from '@/assets/icons/palette.svg?react';
import Profile from '@/assets/icons/profile.svg?react';
import SignOut from '@/assets/icons/sign-out.svg?react';
import UserAvatar from '@/components/UserAvatar';
import UserInfoCard from '@/components/UserInfoCard';
import { isCloud } from '@/consts/env';
import Divider from '@/ds-components/Divider';
import Dropdown, { DropdownItem } from '@/ds-components/Dropdown';
import FlipOnRtl from '@/ds-components/FlipOnRtl';
import Spacer from '@/ds-components/Spacer';
import { Ring as Spinner } from '@/ds-components/Spinner';
import useCurrentUser from '@/hooks/use-current-user';
import useRedirectUri from '@/hooks/use-redirect-uri';
import useSignOut from '@/hooks/use-sign-out';
import useTenantPathname from '@/hooks/use-tenant-pathname';
import useUserPreferences from '@/hooks/use-user-preferences';
import { DynamicAppearanceMode } from '@/types/appearance-mode';
import { onKeyDownHandler } from '@/utils/a11y';

import SubMenu from './SubMenu';
import UserInfoSkeleton from './UserInfoSkeleton';
import styles from './index.module.scss';

function UserInfo() {
  const { signOut } = useSignOut();
  const { getUrl } = useTenantPathname();
  const { t } = useTranslation(undefined, { keyPrefix: 'admin_console' });
  const { user, isLoading: isLoadingUser } = useCurrentUser();
  const anchorRef = useRef<HTMLDivElement>(null);
  const [showDropdown, setShowDropdown] = useState(false);
  const postSignOutRedirectUri = useRedirectUri('signOut');

  const [isLoading, setIsLoading] = useState(false);
  const {
    data: { language, appearanceMode },
    update,
  } = useUserPreferences();

  if (isLoadingUser) {
    return <UserInfoSkeleton />;
  }

  return (
    <>
      <div
        ref={anchorRef}
        role="button"
        tabIndex={0}
        className={classNames(styles.container, showDropdown && styles.active)}
        onKeyDown={onKeyDownHandler(() => {
          setShowDropdown(true);
        })}
        onClick={() => {
          setShowDropdown(true);
        }}
      >
        <UserAvatar user={user} />
      </div>
      <Dropdown
        hasOverflowContent
        anchorRef={anchorRef}
        className={styles.dropdown}
        isOpen={showDropdown}
        horizontalAlign="end"
        onClose={() => {
          setShowDropdown(false);
        }}
      >
        <UserInfoCard className={styles.userInfo} user={user} avatarSize="large" />
        <Divider />
        <DropdownItem
          className={classNames(styles.dropdownItem, isLoading && styles.loading)}
          icon={<Profile className={styles.icon} />}
          onClick={() => {
            // In OSS version, there will be a `/console` context path in the URL.
            const profileRouteWithConsoleContext = getUrl('/profile');

            // Open the profile page in a new tab. In Logto Cloud, the profile page is not nested in the tenant independent,
            // whereas in OSS version, it is under the `/console` context path.
            window.open(isCloud ? '/profile' : profileRouteWithConsoleContext, '_blank');
          }}
        >
          {t('menu.profile')}
          <Spacer />
          <div className={styles.icon}>
            <FlipOnRtl>
              <ExternalLinkIcon />
            </FlipOnRtl>
          </div>
        </DropdownItem>
        <Divider />
        <SubMenu
          className={styles.dropdownItem}
          icon={<Globe className={styles.icon} />}
          title="menu.language"
          options={consoleBuiltInLanguageOptions}
          selectedOption={language}
          onItemClick={(language) => {
            void update({ language });
            setShowDropdown(false);
          }}
        />
        <SubMenu
          className={styles.dropdownItem}
          icon={<Palette className={styles.icon} />}
          title="menu.appearance.label"
          options={[
            {
              value: DynamicAppearanceMode.System,
              title: t('menu.appearance.system'),
            },
            {
              value: Theme.Light,
              title: t('menu.appearance.light'),
            },
            {
              value: Theme.Dark,
              title: t('menu.appearance.dark'),
            },
          ]}
          selectedOption={appearanceMode}
          onItemClick={(appearanceMode) => {
            void update({ appearanceMode });
            setShowDropdown(false);
          }}
        />
        <Divider />
        <DropdownItem
          className={classNames(styles.dropdownItem, isLoading && styles.loading)}
          icon={
            <FlipOnRtl>
              <SignOut className={styles.icon} />
            </FlipOnRtl>
          }
          onClick={(event) => {
            event.stopPropagation();

            if (isLoading) {
              return;
            }
            setIsLoading(true);
            void signOut(postSignOutRedirectUri.href);
          }}
        >
          {t('menu.sign_out')}
          <Spacer />
          {isLoading && <Spinner className={styles.spinner} />}
        </DropdownItem>
      </Dropdown>
    </>
  );
}

export default UserInfo;
