﻿using Newtonsoft.Json.Linq;
using System.Text.Json;
using System.Text.RegularExpressions;

namespace ConsoleApp1
{
    internal class Program
    {
        private static readonly HttpClient client = new HttpClient();
        private const string UserAgent = "Mozilla/5.0 (Linux; Android 12) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/115.0.0.0 Mobile Safari/537.36";
        
        public static async Task Main(string[] args)
        {
            Console.Write("输入抖音分享链接：");

            string url = CheckUrl(Console.ReadLine().Trim());
            if (string.IsNullOrEmpty(url))
            {
                Console.WriteLine("无效的链接");
                return;
            }

            string shareUrl = url.Trim();

            // 解析短链接
            string fullUrl = await GetRealUrl(shareUrl);
            if (string.IsNullOrEmpty(fullUrl))
            {
                Console.WriteLine("短链接解析失败");
                return;
            }

            // 2023新方案：从页面源码提取JSON数据
            string htmlContent = await GetPageHtml(fullUrl);
            string videoData = ExtractVideoData(htmlContent);
            if (videoData == null)
            {
                Console.WriteLine("页面数据解析失败");
                return;
            }

            // 解析JSON获取视频地址          
            if (!string.IsNullOrEmpty(videoData))
            {             
                await DownloadVideo(videoData, $"video_{DateTime.Now:yyyyMMddHHmmss}.mp4");
                Console.WriteLine("下载完成");
            }
            else
            {
                Console.WriteLine("无水印地址解析失败");
            }
        }
        static string CheckUrl(string inputText)
        {
            var matches = Regex.Matches(inputText, @"https:\/\/(?:[a-zA-Z]|\d|[-._~:/?#\[\]@!$&'()*+,;=%])+")
           .Cast<Match>()
           .Where(m => m.Value.Contains("douyin") || m.Value.Contains("aweme")) // 智能过滤
           .Select(m => m.Value.TrimEnd(')', ']', '。', '！')); // 去除尾部分隔符

            // 结果输出
            if (matches.Any())
            {
                Console.WriteLine("提取到以下有效地址：");
                foreach (var url in matches.Distinct())
                {
                   return url;
                }                
            }
            return null;
        } 

        static async Task<string> GetRealUrl(string url)
        {
            try
            {
                var request = new HttpRequestMessage(HttpMethod.Get, url)
                {
                    Headers = {
                    { "User-Agent", UserAgent },
                    { "Referer", "https://www.douyin.com/" }
                }
                };
                var response = await client.SendAsync(request);
                return response.RequestMessage.RequestUri.ToString();
            }
            catch { return null; }
        }

        static async Task<string> GetPageHtml(string url)
        {
            var request = new HttpRequestMessage(HttpMethod.Get, url)
            {
                Headers = {
                { "User-Agent", UserAgent },
                { "Cookie", "s_v_web_id=your_web_id" } // 实际需要动态获取
            }
            };
            return await client.SendAsync(request).Result.Content.ReadAsStringAsync();
        }

        static string ExtractVideoData(string html)
        {
            var routerDataMatch = Regex.Match(html, @"_ROUTER_DATA\s*=\s*({[\s\S]*?});");
            if (!routerDataMatch.Success)
            {
                Console.WriteLine("未找到视频数据");
                return null;
            }

            try
            {
                // 解析 JSON 数据
                JObject routerData = JObject.Parse(routerDataMatch.Groups[1].Value);

                // 访问视频地址路径
                JToken videoInfo = routerData.SelectToken(
                    "loaderData['video_(id)/page']" +
                    ".videoInfoRes.item_list[0]" +
                    ".video.play_addr.url_list[0]");

                if (videoInfo == null)
                {
                    Console.WriteLine("视频地址不存在");
                    return null;
                }

                // 原始地址处理
                string videoUrl = videoInfo.ToString();
                Console.WriteLine($"原始地址：\n{videoUrl}");

                // 生成无水印地址
                string noWatermarkUrl = videoUrl.Replace("/playwm/", "/play/");
                return noWatermarkUrl;               
            }
            catch (Exception ex)
            {
                return ex.Message;
            }

        }
    
       
        static async Task DownloadVideo(
        string url,
        string filename,
        int reportIntervalMs = 100,
        ProgressDisplayType displayType = ProgressDisplayType.Both,
        int progressBarWidth = 30)
        {
            using var response = await client.GetAsync(url, HttpCompletionOption.ResponseHeadersRead);
            response.EnsureSuccessStatusCode();

            var totalBytes = response.Content.Headers.ContentLength ?? -1;
            using var stream = await response.Content.ReadAsStreamAsync();
            using var fileStream = File.Create(filename);

            var buffer = new byte[8192];
            var totalReadBytes = 0L;
            var lastReportTime = DateTime.MinValue;
            var cts = new CancellationTokenSource();

            Console.CancelKeyPress += (s, e) => // 支持Ctrl+C取消
            {
                cts.Cancel();
                e.Cancel = true;
                CleanupFailedDownload(filename);
            };

            try
            {
                while (true)
                {
                    var bytesRead = await stream.ReadAsync(buffer, 0, buffer.Length, cts.Token);
                    if (bytesRead == 0) break;

                    await fileStream.WriteAsync(buffer, 0, bytesRead, cts.Token);
                    totalReadBytes += bytesRead;

                    if (DateTime.Now - lastReportTime > TimeSpan.FromMilliseconds(reportIntervalMs))
                    {
                        UpdateProgressDisplay(totalReadBytes, totalBytes, displayType, progressBarWidth);
                        lastReportTime = DateTime.Now;
                    }
                }

                Console.WriteLine("\n下载完成"); // 所有分块完成后换行
            }
            catch (OperationCanceledException)
            {
                Console.ForegroundColor = ConsoleColor.Red;
                Console.WriteLine("\n下载已取消");
                Console.ResetColor();
                throw;  // 向上传播取消异常
            }
            finally
            {
                fileStream.Close();
            }
        }

        private static void UpdateProgressDisplay(
            long current,
            long total,
            ProgressDisplayType displayType,
            int barWidth)
        {
            decimal progress = total > 0 ?
                Math.Min((decimal)current / total, 1m) :
                -1;

            Console.CursorLeft = 0; // 光标回退到行首

            if (displayType != ProgressDisplayType.ProgressBarOnly)
            {
                // 百分比显示模块
                Console.ForegroundColor = ConsoleColor.Blue;
                Console.Write(total > 0 ?
                    $"进度: {progress:P0} | " :
                    $"已下载: {FormatBytes(current)} | ");
            }

            if (displayType != ProgressDisplayType.PercentageOnly)
            {
                // ASCII进度条绘制模块
                Console.ForegroundColor = ConsoleColor.Green;
                if (progress >= 0)
                {
                    var filledWidth = (int)(barWidth * progress);
                    var bar = new string('#', filledWidth).PadRight(barWidth, '-');
                    Console.Write($"[{bar}]");
                }
                else
                {
                    var pos = (DateTime.Now.Second % barWidth); // 无总大小时的跳动标记
                    var bar = new string('-', barWidth).ToCharArray();
                    bar[pos] = '#';
                    Console.Write($"[{new string(bar)}]");
                }
            }

            Console.ResetColor();
        }

        private static string FormatBytes(long bytes)
        {
            string[] sizes = { "B", "KB", "MB", "GB" };
            int order = 0;
            decimal len = bytes;
            while (len >= 1024 && order < sizes.Length - 1)
            {
                len /= 1024;
                order++;
            }
            return $"{len:n1} {sizes[order]}";
        }

        private static void CleanupFailedDownload(string path)
        {
            try
            {
                if (File.Exists(path))
                {
                    File.Delete(path);
                    Console.WriteLine($"已清理未完成文件: {path}");
                }
            }
            catch
            {
                Console.WriteLine($"无法清理临时文件: {path}");
            }
        }

       
    }
    enum ProgressDisplayType
    {
        PercentageOnly,      // 仅数字百分比
        ProgressBarOnly,     // 仅图形进度条 
        Both                 // 百分比+进度条
    }
}
