import { execSync } from 'child_process';
import { test, expect } from '@playwright/test';

import translations from '../client/i18n/locales/english/translations.json';
import { alertToBeVisible } from './utils/alerts';

test.describe('Email sign-up page when user is not signed in', () => {
  test.use({ storageState: { cookies: [], origins: [] } });

  test.beforeEach(async ({ page }) => {
    await page.goto('/learn');
  });

  test('should not display newsletter options', async ({ page }) => {
    await expect(
      page.getByText(translations.misc['email-blast'])
    ).not.toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['yes-please'] })
    ).not.toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['no-thanks'] })
    ).not.toBeVisible();
  });
});

test.describe('Email sign-up page when user is signed in', () => {
  test.beforeEach(async ({ page }) => {
    // It's necessary to seed with a user that has not accepted the privacy
    // terms, otherwise the user will be redirected away from the email sign-up
    // page.
    execSync('node ./tools/scripts/seed/seed-demo-user --certified-user');
    await page.goto('/learn');
  });

  test('should display the newsletter options correctly', async ({ page }) => {
    await expect(
      page.getByText(translations.misc['email-signup'])
    ).toBeVisible();
    await expect(
      page.getByText(translations.misc['email-blast'])
    ).toBeVisible();
    await expect(page.getByText(translations.misc['quincy'])).toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['yes-please'] })
    ).toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['no-thanks'] })
    ).toBeVisible();
  });

  test('should disable weekly newsletter if the user clicks No', async ({
    page
  }) => {
    await expect(
      page.getByText(translations.misc['email-blast'])
    ).toBeVisible();

    const noThanksButton = page.getByRole('button', {
      name: translations.buttons['no-thanks']
    });
    await expect(noThanksButton).toBeVisible();
    await noThanksButton.click();
    await alertToBeVisible(
      page,
      translations.flash['subscribe-to-quincy-updated']
    );
    await expect(
      page.getByText(translations.misc['email-blast'])
    ).not.toBeVisible();
    await page.goto('/settings');
    await expect(
      page.getByRole('button', { name: translations.buttons['no-thanks'] })
    ).toHaveAttribute('aria-pressed', 'true');
    await expect(
      page.getByRole('button', { name: translations.buttons['yes-please'] })
    ).toHaveAttribute('aria-pressed', 'false');
  });

  test('should enable weekly newsletter if the user clicks Yes', async ({
    page
  }) => {
    await expect(
      page.getByText(translations.misc['email-blast'])
    ).toBeVisible();

    const yesPleaseButton = page.getByRole('button', {
      name: translations.buttons['yes-please']
    });

    await expect(yesPleaseButton).toBeVisible();
    await yesPleaseButton.click();
    await alertToBeVisible(
      page,
      translations.flash['subscribe-to-quincy-updated']
    );
    await page.goto('/settings');
    await expect(
      page.getByRole('group', { name: translations.settings.email.weekly })
    ).toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['yes-please'] })
    ).toHaveAttribute('aria-pressed', 'true');
    await expect(
      page.getByRole('button', { name: translations.buttons['no-thanks'] })
    ).toHaveAttribute('aria-pressed', 'false');
  });
});

test.describe('Email sign-up page when the user is new', () => {
  test.use({ storageState: 'playwright/.auth/development-user.json' });

  test.beforeEach(async ({ page }) => {
    execSync('node ./tools/scripts/seed/seed-demo-user');

    await page.goto('/learn');
  });

  test('should not display newsletter options', async ({ page }) => {
    await expect(
      page.getByText(translations.misc['email-blast'])
    ).not.toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['yes-please'] })
    ).not.toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['no-thanks'] })
    ).not.toBeVisible();
  });
});

test.describe('Email sign-up page when the user has made a selection', () => {
  test.use({ storageState: 'playwright/.auth/development-user.json' });

  test.beforeEach(async ({ page }) => {
    execSync(
      'node ./tools/scripts/seed/seed-demo-user --certified-user --set-false sendQuincyEmail'
    );
    await page.goto('/learn');
  });

  test('should not display newsletter options', async ({ page }) => {
    await expect(
      page.getByText(translations.misc['email-blast'])
    ).not.toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['yes-please'] })
    ).not.toBeVisible();
    await expect(
      page.getByRole('button', { name: translations.buttons['no-thanks'] })
    ).not.toBeVisible();
  });
});
