
/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

#pragma safety enable





#ifndef __OWNERSHIP_H__
#define __OWNERSHIP_H__

#ifdef __STDC_OWNERSHIP__


#ifdef _WIN64
    typedef struct _iobuf FILE;
    typedef unsigned __int64 size_t;    
#endif

#ifdef  _WIN32
    typedef struct _iobuf FILE;
    typedef unsigned int     size_t;
#endif


#ifdef __linux__

    typedef struct _IO_FILE FILE;
    typedef __SIZE_TYPE__ size_t; // valid since C23

#endif

/*
  ownership is suported
*/
void* _Owner _Opt calloc(size_t nmemb, size_t size);
void free(void* _Owner _Opt ptr);
void* _Owner _Opt malloc(size_t size);
void* _Owner _Opt realloc(void* _Opt ptr, size_t size);
char* _Owner _Opt strdup(const char* src);

inline char* _Opt strrchr(char const *  _String, int _Ch);

int snprintf(
        _Ctor char*       const _Buffer,
        size_t      const _BufferCount,
        char const* const _Format,
        ...);

long strtol(
    char const* _String,
    char**     _Opt _EndPtr,
    int         _Radix
    );


FILE* _Owner _Opt fopen(char const* _FileName, char const* _Mode);
int fclose(FILE* _Owner _Stream);

size_t fread(
        _Ctor void*  _Buffer,
        size_t _ElementSize,
        size_t _ElementCount,
        FILE*  _Stream
        );

long long strtoll(
    char const* _String,
    char** _Opt _EndPtr,
    int         _Radix
    );

double strtod(
    char const* _String,
    char**      _Opt _EndPtr
    );

long double strtold(char const* _String,char** _Opt _EndPtr);


unsigned long long strtoull(
    char const* _String,
    char**      _Opt _EndPtr,
    int         _Radix
    );

float strtof(char const* _String, char** _Opt _EndPtr);

//typedef unsigned long long time_t;
//static time_t time(time_t* const _Opt _Time);

#else
/*
  ownership not suported
*/

#define _Ctor
#define _Opt
#define _Owner
#define _Dtor
#define _View
#define static_debug(x)
#define static_set(x, s)
#endif

#endif




#include <assert.h>


#include <stdio.h>


#include <string.h>


#include <stdlib.h>



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

//#pragma once



#include <stdbool.h>

bool enable_vt_mode(void);

/*change foreground color*/

#define BLACK     "\x1b[30m" 
#define BLUE     "\x1b[34m"
#define GREEN     "\x1b[32m"
#define CYAN     "\x1b[36m"
#define RED "\x1b[31;1m"
#define MAGENTA     "\x1b[35m"
#define BROWN     "\x1b[31m"
#define LIGHTGRAY "\x1b[37m"
#define DARKGRAY "\x1b[90m"
#define LIGHTBLUE    "\x1b[34;1m"
#define  LIGHTGREEN "\x1b[92m"
#define LIGHTCYAN "\x1b[36;1m"
#define LIGHTRED "\x1b[91m"
#define LIGHTMAGENTA "\x1b[95m"
#define YELLOW "\x1b[93m"
#define WHITE "\x1b[97m"

//https//en.wikipedia.org/wiki/ANSI_escape_code


#define BK_BLACK "\x1b[40m"
#define BK_BLUE "\x1b[44m"
#define BK_GREEN  "\x1b[42m"
#define BK_CYAN "\x1b[46m"
#define BK_RED "\x1b[41;1m"
#define BK_MAGENTA "\x1b[45m"
#define BK_BROWN "\x1b[41m"
#define BK_LIGHTGRAY "\x1b[40;1m"
#define BK_DARKGRAY "\x1b[40m"
#define BK_LIGHTBLUE "\x1b[44;1m"
#define BK_LIGHTGREEN "\x1b[42,1m"
#define BK_LIGHTCYAN "\x1b[46;1m"
#define BK_LIGHTRED "\x1b[41;1m"
#define BK_LIGHTMAGENTA "\x1b[45;1m"
#define BK_YELLOW             "\x1b[43;1m"
#define BK_WHITE             "\x1b[47;1m"
#define BK_BLINK "\x1b[40m"

#define RESET "\x1b[0m"


int c_kbhit(void);
int c_getch(void);
void c_clrscr();
void c_gotoxy(int x, int y);


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once



#include <stdarg.h>

struct osstream
{
    char* _Owner _Opt c_str;
    int size;
    int capacity;
};


void ss_close( _Dtor struct osstream * stream);

int ss_vafprintf(struct osstream* stream, const char* fmt, va_list args);
int ss_fprintf(struct osstream* stream, const char* fmt, ...);
int ss_putc(char ch, struct osstream* stream);
void ss_clear(struct osstream* stream);
void ss_swap(struct osstream* a, struct osstream* b);



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once

struct declarator;
struct enumerator;
struct enum_specifier;
struct init_declarator;
struct struct_or_union_specifier;
struct macro;
struct struct_entry;

void declarator_delete(struct declarator* _Owner _Opt p);
void init_declarator_delete(struct init_declarator* _Owner _Opt p);

void enumerator_delete(struct enumerator* _Owner _Opt p);
void enum_specifier_delete(struct enum_specifier* _Owner _Opt p);
void struct_or_union_specifier_delete(struct struct_or_union_specifier* _Owner _Opt p);
void struct_entry_delete(struct struct_entry* _Opt _Owner p);

void macro_delete(struct macro* _Owner _Opt p);


/*
* tag allow more than one type of object be inserted at the same map
*/
enum tag
{
    TAG_TYPE_NUMBER,

    TAG_TYPE_ENUN_SPECIFIER,
    TAG_TYPE_STRUCT_OR_UNION_SPECIFIER,

    TAG_TYPE_ENUMERATOR,
    TAG_TYPE_DECLARATOR,
    TAG_TYPE_INIT_DECLARATOR,
    TAG_TYPE_MACRO,
    TAG_TYPE_STRUCT_ENTRY
};


struct map_entry 
{
    struct map_entry* _Owner _Opt next;
    unsigned int hash;
    char* _Owner key;

    enum tag type; /*type of the object pointed by p*/

    union 
    {
        size_t number;
        struct enum_specifier* _Opt _Owner p_enum_specifier;
        struct enumerator* _Opt _Owner p_enumerator;
        struct struct_or_union_specifier* _Opt _Owner p_struct_or_union_specifier;
        struct declarator* _Opt _Owner p_declarator;
        struct init_declarator* _Opt _Owner p_init_declarator;
        struct macro* _Opt _Owner p_macro;
        struct struct_entry* _Opt _Owner p_struct_entry;
    } data;
    
};

struct hash_map 
{
    struct map_entry* _Owner _Opt * _Owner _Opt table;
    int capacity;
    int  size;
};

void hashmap_remove_all(struct hash_map* map);
void hashmap_destroy(_Dtor struct hash_map* map);
struct map_entry* _Opt hashmap_find(struct hash_map* map, const char* key);
void* _Opt hashmap_remove(struct hash_map* map, const char* key, enum tag* _Opt p_type_opt);

/*
  hash_item_set is used to insert pointer with it type into a hashmap and also
  used to undo the map insertion using map and key info.
*/
struct hash_item_set
{
    size_t  number;
    struct enum_specifier* _Owner _Opt p_enum_specifier;
    struct enumerator* _Owner _Opt p_enumerator;
    struct struct_or_union_specifier* _Owner _Opt p_struct_or_union_specifier;
    struct declarator* _Owner _Opt p_declarator;
    struct init_declarator* _Owner _Opt p_init_declarator;
    struct macro* _Owner _Opt p_macro;
    struct struct_entry* _Owner _Opt p_struct_entry;
};
void hash_item_set_destroy(_Dtor struct hash_item_set* p);

int hashmap_set(struct hash_map* map, const char* key, struct hash_item_set * item);



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once

enum token_type
{
    /*When changing here we need also change in tokenizer.c::get_token_name*/
    TK_NONE = 0,
    TK_NEWLINE = '\n',
    TK_WHITE_SPACE = ' ',
    TK_EXCLAMATION_MARK = '!',
    TK_QUOTATION_MARK = '"',
    TK_NUMBER_SIGN = '#',
    TK_DOLLAR_SIGN = '$',
    TK_PERCENT_SIGN = '%',
    TK_AMPERSAND = '&',
    TK_APOSTROPHE = '\'',
    TK_LEFT_PARENTHESIS = '(',
    TK_RIGHT_PARENTHESIS = ')',
    TK_ASTERISK = '*',
    TK_PLUS_SIGN = '+',
    TK_COMMA = ',',
    TK_HYPHEN_MINUS = '-',
    TK_FULL_STOP = '.',
    TK_SOLIDUS = '/',
    TK_COLON = ':',
    TK_SEMICOLON = ';',
    TK_LESS_THAN_SIGN = '<',
    TK_EQUALS_SIGN = '=',
    TK_GREATER_THAN_SIGN = '>',
    TK_QUESTION_MARK = '?',
    TK_COMMERCIAL_AT = '@',
    TK_LEFT_SQUARE_BRACKET = '[',
    TK_REVERSE_SOLIDUS = '//',
    TK_RIGHT_SQUARE_BRACKET = ']',
    TK_CIRCUMFLEX_ACCENT = '^',
    TK_FLOW_LINE = '_',
    TK_GRAVE_ACCENT = '`',
    TK_LEFT_CURLY_BRACKET = '{',
    TK_VERTICAL_LINE = '|',
    TK_RIGHT_CURLY_BRACKET = '}',
    TK_TILDE = '~',
    TK_PREPROCESSOR_LINE,
    TK_PRAGMA,
    TK_PRAGMA_END, /*marks the end of pragma internal usage*/
    TK_STRING_LITERAL,
    TK_CHAR_CONSTANT,    
    TK_LINE_COMMENT,
    TK_COMMENT,
    TK_PPNUMBER,

    ANY_OTHER_PP_TOKEN, //@ por ex

    /*PPNUMBER is converted to one of these at parser phase*/
    TK_COMPILER_DECIMAL_CONSTANT,
    TK_COMPILER_OCTAL_CONSTANT,
    TK_COMPILER_HEXADECIMAL_CONSTANT,
    TK_COMPILER_BINARY_CONSTANT,
    TK_COMPILER_DECIMAL_FLOATING_CONSTANT,
    TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT,


    TK_PLACEMARKER,

    TK_BLANKS,
    TK_PLUSPLUS = '++',
    TK_MINUSMINUS = '--',
    TK_ARROW = '->',
    TK_SHIFTLEFT = '<<',
    TK_SHIFTRIGHT = '>>',
    TK_LOGICAL_OPERATOR_OR = '||',
    TK_LOGICAL_OPERATOR_AND = '&&',
    
    TK_PLUS_ASSIGN = '+=',
    TK_MINUS_ASSIGN = '-=',
    TK_MULTI_ASSIGN = '*=',
    TK_DIV_ASSIGN = '/=',
    TK_MOD_ASSIGN = '%=',
    TK_SHIFT_LEFT_ASSIGN = '<<=',
    TK_SHIFT_RIGHT_ASSIGN = '>>=',
    TK_AND_ASSIGN = '&=',
    TK_OR_ASSIGN ='|=',
    TK_NOT_ASSIGN = '^=',

    TK_MACRO_CONCATENATE_OPERATOR = '##',

    TK_IDENTIFIER,
    TK_IDENTIFIER_RECURSIVE_MACRO, /*used to avoid macro recursion*/

    TK_BEGIN_OF_FILE,

    //C23 keywords
    TK_KEYWORD_AUTO,
    TK_KEYWORD_BREAK,
    TK_KEYWORD_CASE,
    TK_KEYWORD_CONSTEXPR,
    TK_KEYWORD_CHAR,
    TK_KEYWORD_CONST,
    TK_KEYWORD_CONTINUE,
    TK_KEYWORD_CAKE_CATCH, /*extension*/
    TK_KEYWORD_DEFAULT,
    TK_KEYWORD_DO,
    TK_KEYWORD_DEFER, /*extension*/
    TK_KEYWORD_DOUBLE,
    TK_KEYWORD_ELSE,
    TK_KEYWORD_ENUM,
    TK_KEYWORD_EXTERN,
    TK_KEYWORD_FLOAT,
    TK_KEYWORD_FOR,
    TK_KEYWORD_GOTO,
    TK_KEYWORD_IF,
    TK_KEYWORD_INLINE,
    TK_KEYWORD_INT,
    TK_KEYWORD_LONG,
    TK_KEYWORD_MSVC__INT8,
    TK_KEYWORD_MSVC__INT16,
    TK_KEYWORD_MSVC__INT32,
    TK_KEYWORD_MSVC__INT64,

    TK_KEYWORD_REGISTER,
    TK_KEYWORD_RESTRICT,
    TK_KEYWORD_RETURN,
    TK_KEYWORD_SHORT,
    TK_KEYWORD_SIGNED,
    
    TK_KEYWORD_SIZEOF,
    TK_KEYWORD__COUNTOF, //C2Y

    
    TK_KEYWORD_STATIC,
    TK_KEYWORD_STRUCT,
    TK_KEYWORD_SWITCH,
    TK_KEYWORD_TYPEDEF,
    TK_KEYWORD_CAKE_TRY, /*extension*/
    TK_KEYWORD_CAKE_THROW, /*extension*/
    TK_KEYWORD_UNION,
    TK_KEYWORD_UNSIGNED,
    TK_KEYWORD_VOID,
    TK_KEYWORD_VOLATILE,
    TK_KEYWORD_WHILE,
    
    TK_KEYWORD__ALIGNAS,
    TK_KEYWORD__ALIGNOF,
    TK_KEYWORD__ATOMIC,
     
//#ifdef _WIN32 
    
    //https://learn.microsoft.com/en-us/cpp/cpp/ptr32-ptr64?view=msvc-170&redirectedfrom=MSDN
    TK_KEYWORD_MSVC__PTR32,
    TK_KEYWORD_MSVC__PTR64,

    TK_KEYWORD_MSVC__FASTCALL,
    TK_KEYWORD_MSVC__STDCALL,
    TK_KEYWORD_MSVC__CDECL,    
    TK_KEYWORD_MSVC__DECLSPEC,
//#endif

    TK_KEYWORD__ASM, 
    TK_KEYWORD__BOOL,
    TK_KEYWORD__COMPLEX,
    TK_KEYWORD__DECIMAL128,
    TK_KEYWORD__DECIMAL32,
    TK_KEYWORD__DECIMAL64,
    TK_KEYWORD__GENERIC,
    TK_KEYWORD__IMAGINARY,
    TK_KEYWORD__NORETURN,
    TK_KEYWORD__STATIC_ASSERT,    
    TK_KEYWORD_ASSERT, /*extension*/
    TK_KEYWORD__THREAD_LOCAL,

    TK_KEYWORD_TYPEOF, /*C23*/
    
    TK_KEYWORD_TRUE,  /*C23*/
    TK_KEYWORD_FALSE,  /*C23*/
    TK_KEYWORD_NULLPTR,  /*C23*/
    TK_KEYWORD_TYPEOF_UNQUAL, /*C23*/
    TK_KEYWORD__BITINT /*C23*/,

    

    /*cake extension*/
    TK_KEYWORD_CAKE_OWNER,
    TK_KEYWORD__CTOR,
    TK_KEYWORD__DTOR, 
    TK_KEYWORD_CAKE_VIEW,    
    TK_KEYWORD_CAKE_OPT, 
    

    /*extension compile time functions*/
    TK_KEYWORD_CAKE_STATIC_DEBUG, /*extension*/
    TK_KEYWORD_CAKE_STATIC_DEBUG_EX, /*extension*/
    TK_KEYWORD_STATIC_STATE, /*extension*/
    TK_KEYWORD_STATIC_SET, /*extension*/
    
    /*https://en.cppreference.com/w/cpp/header/type_traits*/
    
    TK_KEYWORD_IS_POINTER,
    TK_KEYWORD_IS_LVALUE,
    TK_KEYWORD_IS_CONST,
    TK_KEYWORD_IS_OWNER,
    TK_KEYWORD_IS_ARRAY,
    TK_KEYWORD_IS_FUNCTION,
    TK_KEYWORD_IS_SCALAR,
    TK_KEYWORD_IS_ARITHMETIC,
    TK_KEYWORD_IS_FLOATING_POINT,
    TK_KEYWORD_IS_INTEGRAL,
    

};

enum token_flags
{
    TK_FLAG_NONE = 0,
    TK_FLAG_FINAL = 1 << 0,                    /*compiler will see this token*/
    TK_FLAG_MACRO_EXPANDED = 1 << 1,           /*this token was generated from macro expansion*/
    TK_FLAG_HAS_SPACE_BEFORE = 1 << 2,         /*this token has spaces before*/
    TK_FLAG_HAS_NEWLINE_BEFORE = 1 << 3,       /*this token has newline before*/
    TK_FLAG_IDENTIFIER_IS_TYPEDEF = 1 << 4,    /*saves time on typedef search*/
    TK_FLAG_IDENTIFIER_IS_NOT_TYPEDEF = 1 << 5,

    TK_C_BACKEND_FLAG_HIDE = 1 << 6,                 /*c backend hidden*/
    
    TK_FLAG_IDENTIFIER_IS_ENUMERATOR = 1 << 7,       /*saves time on search*/

    TK_FLAG_IDENTIFIER_IS_NOT_ENUMERATOR = 1 << 8,   /*we know it is not enumerator*/    

    TK_FLAG_SLICED = 1 << 9,                         /*line-slicing in the middle*/

    TK_FLAG_LINE_CONTINUATION = 1 << 10 ,            /*token has one or more line-slicing*/

    TK_C_BACKEND_FLAG_SHOW_AGAIN = 1 << 11,          /*was hidden but maybe reappears*/
};

struct token
{
    enum token_type type;
    char* _Owner lexeme; //TODO make const
    char* original;

    int line;
    int col;

    /*include level - 0 is the current file*/
    int level;

    enum token_flags flags;

    /*points to the token with file name or macro*/
    struct token* token_origin;

    struct token* _Owner _Opt next;
    struct token* _Opt prev;
};

void token_delete( struct token* _Owner _Opt p);

struct token_list
{
    struct token* _Owner _Opt head;
    struct token* _Opt tail;
};

void token_list_set_file(struct token_list* list, struct token* filetoken, int line, int col);
bool token_list_is_empty(struct token_list* p);
void token_list_swap(struct token_list* a, struct token_list* b);

struct token* _Owner _Opt clone_token(struct token* p);
struct token* token_list_add(struct token_list* list, struct token* _Owner pnew);
void token_list_remove(struct token_list* list, struct token* first, struct token* last);
struct token_list token_list_remove_get(struct token_list* list, struct token* first, struct token* last);
void token_list_append_list(struct token_list* dest, struct token_list* source);
void token_list_append_list_at_beginning(struct token_list* dest, struct token_list* source);
struct token* token_list_clone_and_add(struct token_list* list, struct token* pnew);
char* _Owner _Opt token_list_join_tokens(struct token_list* list, bool bliteral);
void token_list_clear(struct token_list* list);
bool token_is_blank(const struct token* _Opt p);
bool token_is_identifier_or_keyword(enum token_type t);
void token_range_add_flag(struct token* first, struct token* last, enum token_flags flag);
void token_range_remove_flag(struct token* first, struct token* last, enum token_flags flag);
void token_range_add_show(struct token* first, struct token* last);

void print_tokens_html(struct token* p_token);

struct marker
{    
    const char* _Opt file;
    int line;
    int start_col;
    int end_col;

    // Line
    //~~~~~~~~^~~~~~~~~
    //begin  caret   end

    const struct token* _Opt p_token_caret;
    const struct token* _Opt p_token_begin;
    const struct token* _Opt p_token_end;
};

void print_line_and_token(struct marker* p_marker, bool visual_studio_ouput_format);

void print_position(const char* path, int line, int col, bool msvc_format);

struct stream
{
    const char* const source;
    const char* current;
    int line;
    int col;
    int line_continuation_count;
    const char* path;
};

int is_digit(const struct stream* p);
int is_nondigit(const struct stream* p);
void stream_match(struct stream* stream);

bool style_has_space(const struct token*  token);
bool style_has_one_space(const struct token*  token);

enum token_type parse_number(const char* lexeme, char suffix[4], _Ctor char erromsg[100]);
const unsigned char* _Opt utf8_decode(const unsigned char* s, _Ctor unsigned int* c);
const unsigned char* _Opt escape_sequences_decode_opt(const unsigned char* p, unsigned int* out_value);


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

//#pragma once

typedef int errno_t;

#if __STDC_VERSION__  >= 202311L 
#define NODISCARD [[nodiscard]]
#else
#define NODISCARD
#endif


#ifndef __CAKE__

//emulate _Countof
#define _Countof(A) (sizeof(A)/sizeof((A)[0]))

#define try  
#define catch if (0) catch_label:
#define throw do { throw_break_point(); goto catch_label;}while (0)

#endif

const char* get_posix_error_message(int error);
int windows_error_to_posix(int i);

void throw_break_point();


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once

/*
* Compiler options shared with compiler and preprocessor
*/

enum target
{
    TARGET_NONE = -1,
    TARGET_C89_IL = 0 /*default*/
};

enum language_version
{
    LANGUAGE_C23,
    LANGUAGE_C2Y,
    LANGUAGE_CAK,
};

enum diagnostic_id {

    W_NONE = 0,  /*not a real warning, used in especial cases*/
    
    
    W_UNUSED_VARIABLE, //-Wunused-variable
    W_DEPRECATED,
    W_ENUN_CONVERSION,//-Wenum-conversion

    W_ADDRESS, //-Waddress (always true)
    W_UNUSED_PARAMETER, //-Wno-unused-parameter
    W_DECLARATOR_HIDE, // gcc no
    W_TYPEOF_ARRAY_PARAMETER,//
    W_ATTRIBUTES, //-Wattributes
    W_UNUSED_VALUE, //-Wunused-value
    W_STYLE, //-Wstyle
    W_COMMENT,
    W_LINE_SLICING,
    W_STRING_SLICED,
    W_DISCARDED_QUALIFIERS,
    W_DECLARATOR_STATE,
    W_UNINITIALZED,
    W_RETURN_LOCAL_ADDR,
    W_MUST_USE_ADDRESSOF,
    W_ARRAY_INDIRECTION,
    /*ownership type system errors*/
    W_OWNERSHIP_MISSING_OWNER_QUALIFIER,
    W_OWNERSHIP_NOT_OWNER,
    W_OWNERSHIP_USING_TEMPORARY_OWNER,
    W_OWNERSHIP_MOVE_ASSIGNMENT_OF_NON_OWNER,
    W_OWNERSHIP_NON_OWNER_TO_OWNER_ASSIGN,
    W_OWNERSHIP_DISCARDING_OWNER,
    W_OWNERSHIP_NON_OWNER_MOVE,    
    //////////////////////////////////////////////
    W_FLOW_NON_NULL, //-Wnonnull
    W_FLOW_MISSING_DTOR,
    W_FLOW_UNINITIALIZED,
    W_FLOW_LIFETIME_ENDED,
    W_FLOW_MOVED,
    W_FLOW_NULL_DEREFERENCE,
    W_FLOW_MAYBE_NULL_TO_NON_OPT_ARG,
    W_FLOW_NULLABLE_TO_NON_NULLABLE,
    W_FLOW_DIVIZION_BY_ZERO,    
    //////////////////////////////////////////////
    W_DIVIZION_BY_ZERO,
    W_CONSTANT_VALUE, /*sample 0 * a */
    W_PASSING_NULL_AS_ARRAY,
    W_INCOMPATIBLE_ENUN_TYPES,
    W_MULTICHAR_ERROR,
    W_OUT_OF_BOUNDS,
    W_ASSIGNMENT_OF_ARRAY_PARAMETER,
    W_CONDITIONAL_IS_CONSTANT,
    W_SWITCH,
    W_UNSUAL_NULL_POINTER_CONSTANT,
    W_SIZEOF_ARRAY_ARGUMENT,
    W_CONST_NOT_INITIALIZED,
    W_NULL_CONVERTION,
    W_IMPLICITLY_UNSIGNED_LITERAL,
    W_INTEGER_OVERFLOW,
    W_ARRAY_SIZE,
    
    
    W_EMPTY_STATEMENT,
    W_ERROR_INCOMPATIBLE_TYPES,
    W_UNUSED_LABEL,
    W_REDEFINING_BUITIN_MACRO,
    W_UNUSED_FUNCTION,
    W_NOT_DEFINED57,
    W_NOT_DEFINED58,
    W_NOT_DEFINED59,
    W_NOT_DEFINED60,
    W_NOT_DEFINED61,

    W_LOCATION, /*prints code location*/
    W_NOTE,

    //----------------------------------------------------------------
    
    W_TO_MANY_INITIALIZERS = 100,

    //---------------------------------------------------------------

    C_ERROR_INVALID_QUALIFIER_FOR_POINTER = 640,
    C_ERROR_UNEXPECTED = 650,
    C_ERROR_TOO_MANY_ARGUMENTS = 660,
    C_ERROR_TOO_FEW_ARGUMENTS = 670,
    C_ERROR_NOT_FOUND = 680,
    C_ERROR_NO_MATCH_FOR_GENERIC = 690,
    C_ERROR_SUBSCRIPTED_VALUE_IS_NEITHER_ARRAY_NOR_POINTER = 700,
    C_ERROR_CALLED_OBJECT_IS_NOT_FUNCTION_OR_FUNCTION_POINTER = 710,
    C_ERROR_STRUCT_MEMBER_NOT_FOUND = 720,
    C_ERROR_STRUCTURE_OR_UNION_REQUIRED = 730,
    C_ERROR_STRUCT_IS_INCOMPLETE = 740,
    C_ERROR_DECLARATOR_NOT_FOUND = 750,
    C_ERROR_EXPECTED_DECLARATOR_NAME = 760,
    C_ERROR_UNKNOWN_ATTRIBUTE_NAME = 770,
    C_ERROR_INDIRECTION_REQUIRES_POINTER_OPERAND = 780,
    C_ERROR_INVALID_TOKEN = 790,
    C_ERROR_EXPECTED_STRUCT_TYPE = 800,
    C_ERROR_EXPECTED_TYPE_NAME = 810,
    C_ERROR_LEFT_IS_NOT_ARITHMETIC = 820,
    C_ERROR_RIGHT_IS_NOT_ARITHMETIC = 830,
    C_ERROR_LEFT_IS_NOT_INTEGER = 840,
    C_ERROR_RIGHT_IS_NOT_INTEGER = 850,
    C_ERROR_INVALID_TYPE = 860,
    C_ERROR_LEFT_IS_NOT_SCALAR = 870,
    C_ERROR_RIGHT_IS_NOT_SCALAR = 880,
    C_ERROR_INCOMPATIBLE_POINTER_TYPES = 890, //warning?
    C_ERROR_ASSIGNMENT_OF_FUNCTION = 900,
    C_ERROR_ASSIGNMENT_TO_EXPRESSION_WITH_ARRAY_TYPE = 910,
    C_ERROR_ASSIGNMENT_OF_READ_ONLY_OBJECT = 920,
    C_ERROR_LVALUE_ASSIGNMENT = 930,
    C_ERROR_CONDITION_MUST_HAVE_SCALAR_TYPE = 940,
    C_ERROR_INCOMPATIBLE_TYPES = 950,
    C_ERROR_EXPECTED_CONSTANT_EXPRESSION = 960,
    C_ERROR_UNEXPECTED_TOKEN = 970,
    C_ERROR_CANNOT_COMBINE_WITH_PREVIOUS_LONG_LONG = 980,
    C_ERROR_EXPECTED_DECLARATION = 990,
    C_ERROR_STATIC_OR_TYPE_QUALIFIERS_NOT_ALLOWED_IN_NON_PARAMETER = 1000,
    C_ERROR_OBJ_OWNER_CAN_BE_USED_ONLY_IN_POINTER = 1010,
    C_ERROR_REDECLARATION = 1020,
    C_ERROR_TAG_TYPE_DOES_NOT_MATCH_PREVIOUS_DECLARATION = 1030,
    C_ERROR_MISSING_ENUM_TAG_NAME = 1040,
    C_ERROR_MULTIPLE_DEFINITION_ENUM = 1050,
    C_ERROR_STATIC_ASSERT_FAILED = 1060,
    C_ERROR_STATIC_SET = 1070,
    C_ANALIZER_ERROR_STATIC_STATE_FAILED = 1080,
    C_ERROR_ATTR_UNBALANCED = 1090,
    C_ERROR_UNEXPECTED_END_OF_FILE = 1100,
    C_ERROR_THROW_STATEMENT_NOT_WITHIN_TRY_BLOCK = 1110,
    C_ERROR_VOID_FUNCTION_SHOULD_NOT_RETURN_VALUE = 1120,
    C_ERROR_NON_VOID_FUNCTION_SHOULD_RETURN_VALUE = 1121,
    C_ERROR_ARGUMENT_SIZE_SMALLER_THAN_PARAMETER_SIZE = 1130,
    C_ERROR_TOKEN_NOT_VALID_IN_PREPROCESSOR_EXPRESSIONS = 1140,
    C_ERROR_FILE_NOT_FOUND = 1150,
    C_ERROR_MISSING_CLOSE_PARENTHESIS = 1160,
    C_ERROR_EXPRESSION_ERROR = 1170,
    C_ERROR_PREPROCESSOR_C_ERROR_DIRECTIVE = 1180,
    C_ERROR_TOO_FEW_ARGUMENTS_TO_FUNCTION_LIKE_MACRO = 1190,
    C_ERROR_TOO_MANY_ARGUMENTS_TO_FUNCTION_LIKE_MACRO = 1191,
    C_ERROR_PREPROCESSOR_MACRO_INVALID_ARG = 1200,
    C_ERROR_PREPROCESSOR_MISSING_MACRO_ARGUMENT = 1210,
    C_ERROR_ADDRESS_OF_REGISTER = 1220,
    C_ERROR_OPERATOR_NEEDS_LVALUE = 1230, //C2105
    C_ERROR_CHARACTER_TOO_LARGE = 1240,
    C_ERROR_PRAGMA_ERROR = 1250,
    C_ERROR_OUT_OF_MEM = 1260,
    C_ERROR_STORAGE_SIZE = 1270,
    C_ERROR_RETURN_LOCAL_OWNER_TO_NON_OWNER = 1280,
    C_ERROR_AUTO_NEEDS_SINGLE_DECLARATOR = 1290,
    C_ERROR_TWO_OR_MORE_SPECIFIERS = 1300,
    C_ERROR_OPERATOR_INCREMENT_CANNOT_BE_USED_IN_OWNER = 1310,
    C_ERROR_OPERATOR_DECREMENT_CANNOT_BE_USED_IN_OWNER = 1320,
    C_PRE_DIVISION_BY_ZERO = 1330,
    C_ERROR_INT_TO_POINTER = 1340,
    C_ERROR_LITERAL_OVERFLOW = 1350,
    C_CHARACTER_NOT_ENCODABLE_IN_A_SINGLE_CODE_UNIT = 1360,
    C_MULTICHAR_ERROR = 1370,
    C_INVALID_TOKEN = 1380,
    C_INVALID_ARGUMENT_NELEMENTSOF = 1390,
    C_ERROR_RETURN_CANNOT_BE_USED_INSIDE_DEFER = 1400,
    C_ERROR_FUNCTION_RETURNS_FUNCTION = 1410,
    C_ERROR_FUNCTION_RETURNS_ARRAY = 1420,    
    C_ERROR_LABEL_NOT_DEFINED = 1430,    
    C_ERROR_DUPLICATED_LABEL = 1440,
    C_ERROR_DUPLICATED_CASE = 1450,
    C_ERROR_SUBSCRIPT_IS_NOT_AN_INTEGER = 1560,    
    C_ERROR_DUPLICATE_DEFAULT_GENERIC_ASSOCIATION = 1570, 
    C_ERROR_MULTIPLE_DEFAULT_LABELS_IN_ONE_SWITCH = 1780,
    C_ERROR_POINTER_TO_FLOATING_TYPE = 1790,
    C_ERROR_FLOATING_TYPE_TO_POINTER = 1800,
    C_ERROR_NULLPTR_CAST_ERROR = 1810,
    C_ERROR_MACRO_REDEFINITION = 1820,
    C_ERROR_INVALID_PREPROCESSING_DIRECTIVE = 1830,
    C_ERROR_FUNCTION_CANNOT_BE_MEMBER  = 1840,
    C_ERROR_NON_INTEGRAL_ENUM_TYPE = 1850,
    C_ERROR_REQUIRES_COMPILE_TIME_VALUE = 1860,
    C_ERROR_OUTER_SCOPE = 1870,
};


bool is_diagnostic_configurable(enum diagnostic_id id);
bool is_diagnostic_warning(enum diagnostic_id id);
bool is_diagnostic_error(enum diagnostic_id id);
bool is_diagnostic_note(enum diagnostic_id id);


/*
* These warnings are removed when "nullable=disable"
*/
#define WFLAG(W) (1ULL << W)
#define NULLABLE_DISABLE_REMOVED_WARNINGS  (WFLAG(W_FLOW_NULL_DEREFERENCE) | WFLAG(W_FLOW_NULLABLE_TO_NON_NULLABLE))

#define OWNERSHIP_DISABLE_REMOVED_WARNINGS  (WFLAG(W_FLOW_UNINITIALIZED))


int get_diagnostic_phase(enum diagnostic_id w);

enum style
{
    STYLE_CAKE,

    // https://llvm.org/docs/CodingStandards.html
    STYLE_LLVM,// A style complying with the LLVM coding standards

    //https://google.github.io/styleguide/cppguide.html
    STYLE_GOOGLE,// A style complying with Google’s C++ style guide

    //https://chromium.googlesource.com/chromium/src/+/refs/heads/main/styleguide/styleguide.md
    //https://www.kernel.org/doc/html/latest/process/coding-style.html
    STYLE_CHROMIUM,// A style complying with Chromium’s style guide

    //https://firefox-source-docs.mozilla.org/code-quality/coding-style/index.html
    STYLE_MOZILLA,// A style complying with Mozilla’s style guide

    //https://www.webkit.org/code-style-guidelines/
    STYLE_WEBKIT,// A style complying with WebKit’s style guide

    STYLE_MICROSOFT,// A style complying with Microsoft’s style guide

    STYLE_GNU,// A style complying with the GNU coding standards

};
int get_warning_name(enum diagnostic_id w, int n, char buffer[/*n*/]);
unsigned long long  get_warning_bit_mask(const char* wname);

enum diagnostic_id  get_warning(const char* wname);

struct diagnostic
{
    /*
      each message has number (0-63) that corresponds to the bit index
      Messages bigger than W_NOTE are errors or bigger than 63
    */

    /*set of warnings reported as errors*/
    unsigned long long errors;
    /*set of warnings reported as warnings*/
    unsigned long long warnings;
    /*set of warnings reported as notes*/
    unsigned long long notes;
};

int get_diagnostic_type(struct diagnostic* d, enum diagnostic_id w);
extern struct diagnostic default_diagnostic;

void diagnostic_remove(struct diagnostic *d, enum diagnostic_id w);

struct diagnostic_stack
{
    int top_index;
    struct diagnostic stack[10];
};

int diagnostic_stack_push_empty(struct diagnostic_stack* diagnostic_stack);
void diagnostic_stack_pop(struct diagnostic_stack* diagnostic_stack);


struct options
{
    enum language_version input;
    enum target target;

    /*
      #pragma CAKE diagnostic push
      #pragma CAKE diagnostic pop
    */
    struct diagnostic_stack diagnostic_stack;

    enum style style;

    /*
       Causes the compiler to output a list of the include files.
       The option also displays nested include files, that is,
       the files included by the files that you include.
    */
    bool show_includes;

    /*
       -disable-assert
    */
    bool disable_assert;

    
    /*
       -flow-analysis
    */
    bool flow_analysis;

    /*
    * -testmode
    */
    bool test_mode;

    /*
    * -nullchecks
    */
    bool null_checks_enabled;

    bool ownership_enabled;

    /*
      -E
    */
    bool preprocess_only;

    /*
      -preprocess-def-macro
    */
    bool preprocess_def_macro;

    bool clear_error_at_end; //used by tests
    
    /*
      -sarif
    */
    bool sarif_output;

    /*
      -no-output
      if true cake does not generate output
    */
    bool no_output;

    /*
     -const-literal
     makes literal strings const
    */
    bool const_literal;

    /*
      -fdiagnostics-format=msvc
      -msvc-output
    */
    bool visual_studio_ouput_format;

    /*
      -dump-tokens
      print tokens before preprocessor
    */
    bool dump_tokens;

    /*
      -dump-pp-tokens
      (print tokens after preprocessor)
    */
    bool dump_pptokens;

    /*
      -autoconfig
    */
    bool auto_config;

    bool do_static_debug;
    int static_debug_lines;

    /*
      -o filename
      defines the ouputfile when 1 file is used
    */
    char output[200];
    char sarifpath[200];
};

int fill_options(struct options* options,
                 int argc,
                 const char** argv);

void print_help();



#define CAKE_CFG_FNAME "/cakeconfig.h"

struct include_dir
{
    const char* _Owner path;
    struct include_dir* _Owner _Opt next;
};

struct include_dir_list
{
    struct include_dir* _Owner _Opt head;
    struct include_dir* _Opt tail;
};

enum preprocessor_ctx_flags
{
    PREPROCESSOR_CTX_FLAGS_NONE = 0,
    PREPROCESSOR_CTX_FLAGS_ONLY_FINAL = 1 << 0
};

struct preprocessor_ctx
{
    struct options options;
    enum preprocessor_ctx_flags flags;
    struct hash_map macros;
    struct include_dir_list include_dir;

    /*map of pragma once already included files*/
    struct hash_map pragma_once_map;
    
    struct token* _Opt current;
    struct token_list input_list;
    unsigned int count_macro_value;
    bool conditional_inclusion;
    int n_warnings;
    int n_errors;    
};

void preprocessor_ctx_destroy( _Dtor struct preprocessor_ctx* p);

void pre_unexpected_end_of_file(struct token* _Opt p_token, struct preprocessor_ctx* ctx);
bool preprocessor_diagnostic(enum diagnostic_id w, struct preprocessor_ctx* ctx, const struct token* _Opt p_token, const char* fmt, ...);


struct tokenizer_ctx
{
    struct options options;
    int n_warnings;
    int n_errors;    
};

struct token_list tokenizer(struct tokenizer_ctx* ctx, const char* text, const char* _Opt filename_opt, int level, enum token_flags addflags);
void add_standard_macros(struct preprocessor_ctx* ctx);
struct include_dir* _Opt include_dir_add(struct include_dir_list* list, const char* path);

struct token_list preprocessor(struct preprocessor_ctx* ctx, struct token_list* input_list, int level);
struct token_list copy_replacement_list(struct preprocessor_ctx* ctx, const struct token_list* list);

void token_list_append_list(struct token_list* dest, _Dtor struct token_list* source);
void print_list(struct token_list* list);
void token_list_destroy(_Opt _Dtor struct token_list* list);
bool token_is_blank(const struct token* p);
void token_list_pop_back(struct token_list* list);
void token_list_pop_front(struct token_list* list);
struct token* _Owner _Opt token_list_pop_front_get(struct token_list* list);
void remove_line_continuation(char* s);
struct token* token_list_clone_and_add(struct token_list* list, struct token* pnew);
bool token_list_is_equal(const struct token_list* list_a, const struct token_list* list_b);
void token_list_insert_after(struct token_list* list, struct token* _Opt after, struct token_list* append);
void token_list_insert_before(struct token_list* token_list, struct token* after, struct token_list* append_list);
void token_list_paste_string_after(struct token_list* list,
    struct token* after,
    const char* s);
void token_list_paste_string_before(struct token_list* list,
    struct token* before,
    const char* s);
struct token_list tokenizer(struct tokenizer_ctx* p, const char* text, const char* _Opt filename_opt, int level, enum token_flags addflags);

void print_code_as_we_see(const struct token_list* list, bool remove_comments);
const char* _Owner _Opt get_code_as_compiler_see(const struct token_list* list);
const char* _Owner _Opt get_code_as_we_see_plus_macros(const struct token_list* list);
const char* _Owner _Opt get_code_as_we_see(const struct token_list* list, bool remove_comments);

void print_tokens(const struct token* _Opt p_token);
void print_preprocessed(const struct token* p_token);
const char* _Owner _Opt print_preprocessed_to_string(const struct token* p_token);
const char* _Owner _Opt print_preprocessed_to_string2(const struct token* _Opt p_token);
void check_unused_macros(const struct hash_map* map);

const char* get_token_name(enum token_type tk);
void print_all_macros(const struct preprocessor_ctx* prectx);

int string_literal_char_byte_size(const char* s);
int string_literal_byte_size_not_zero_included(const char* s);

int get_char_type(const char* s);
int include_config_header(struct preprocessor_ctx* ctx, const char* file_name);
int stringify(const char* input, int n, char output[]);


#ifdef _WIN32


#include <Windows.h>
#endif

#if defined _MSC_VER && !defined __POCC__


#include <crtdbg.h>
#endif

#if defined _MSC_VER && !defined __POCC__


#include <debugapi.h>
#endif

/*
  PROVISORY - unchecked was removed, now we control flow ownership error with pragma
  TODO review alternatives from Domingo's branch.
*/
#ifdef __CAKE__
#pragma cake diagnostic push
#pragma cake diagnostic ignored "-Wdiscard-owner"
#pragma cake diagnostic ignored "-Wmissing-destructor"
#pragma cake diagnostic ignored "-Wnon-owner-move"
#pragma cake diagnostic ignored "-Wnon-owner-to-_Owner-move"
#endif

//#pragma cake diagnostic pop

bool style_has_space(const struct token* token)
{
    return token_is_blank(token->prev);
}

bool style_has_one_space(const struct token* token)
{
    return token->prev &&
        token->prev->type == TK_BLANKS;
}

void print_literal2(const char* s);

void token_list_clear(struct token_list* list)
{
    struct token* _Owner _Opt p = list->head;
    while (p)
    {
        struct token* _Owner _Opt next = p->next;
        p->next = NULL;
        token_delete(p);
        p = next;
    }

    list->head = NULL;
    list->tail = NULL;
}


void token_range_add_show(struct token* first, struct token* last)
{
    for (struct token* current = first;
         current != last->next;
         current = current->next)
    {
        current->flags = current->flags & ~TK_C_BACKEND_FLAG_HIDE;
        if (current->next == NULL)
            break;
    }
}

void token_range_remove_flag(struct token* first, struct token* last, enum token_flags flag)
{
    for (struct token* _Opt current = first;
        current && current != last->next;
        current = current->next)
    {
        current->flags = current->flags & ~flag;
    }
}

void token_range_add_flag(struct token* first, struct token* last, enum token_flags flag)
{
    for (struct token* _Opt current = first;
        current && current != last->next;
        current = current->next)
    {
        current->flags |= flag;
    }
}

void token_list_pop_back(struct token_list* list)
{
    if (list->head == NULL)
        return;

    if (list->head == list->tail)
    {
        token_delete(list->head);
        list->head = NULL;
        list->tail = NULL;
    }
    else
    {
        assert(list->tail != NULL);
        assert(list->tail->prev != NULL);
        list->tail = list->tail->prev;
        token_delete(list->tail->next);
        list->tail->next = NULL;
        if (list->tail == list->head)
        {
            list->tail->prev = NULL;
        }
    }
    assert(list->head == NULL || list->head->prev == NULL);
}

void token_list_pop_front(struct token_list* list) /*unchecked*/
{
    if (list->head == NULL)
        return;

    struct token* _Owner _Opt p = list->head;
    assert(p->prev == NULL);

    if (list->head == list->tail)
    {
        list->head = NULL;
        list->tail = NULL;
    }
    else
    {
        list->head = p->next;
        if (list->head)
            list->head->prev = NULL;
    }
    p->next = NULL;
    p->prev = NULL;
    token_delete(p);

    assert(list->head == NULL || list->head->prev == NULL);
}

struct token* _Owner _Opt token_list_pop_front_get(struct token_list* list)
{
    if (list->head == NULL)
        return NULL;

    struct token* _Owner _Opt old_head = list->head;

    list->head = old_head->next; // move head forward

    if (list->head != NULL)
    {
        list->head->prev = NULL;
    }
    else
    {
        list->tail = NULL;
    }

    assert(list->head == NULL || list->head->prev == NULL);

    old_head->prev = NULL;
    old_head->next = NULL;
    return old_head;
}

void token_list_swap(struct token_list* a, struct token_list* b)
{
    struct token_list temp = *a;
    *a = *b;
    *b = temp;
}

void token_delete(struct token* _Owner _Opt p)
{
    if (p)
    {
        /*
         * ownership warning here is about the p->next
         * we need a way to remove only this specific warning
        */
        assert(p->next == NULL);
        free(p->lexeme);
        free(p);
    }
}

void token_list_set_file(struct token_list* list, struct token* filetoken, int line, int col)
{
    struct token* _Opt p = list->head;
    while (p)
    {
        p->token_origin = filetoken;
        p->line = line;
        p->col = col;
        p = p->next;
    }
}

void token_list_destroy(_Opt _Dtor struct token_list* list)
{
    struct token* _Owner _Opt p = list->head;
    while (p)
    {
        struct token* _Owner _Opt next = p->next;
        p->next = NULL;
        token_delete(p);
        p = next;
    }
}

char* _Owner _Opt token_list_join_tokens(struct token_list* list, bool bliteral)
{
    struct osstream ss = { 0 };
    if (bliteral)
        ss_fprintf(&ss, "\"");
    bool has_space = false;
    struct token* _Opt current = list->head;

    while (current)
    {
        if (token_is_blank(current))
        {
            has_space = true;
            current = current->next;
            continue;
        }

        if (has_space)
            ss_fprintf(&ss, " ");

        const char* p = current->lexeme;
        while (*p)
        {
            if (*p == '"')
                ss_fprintf(&ss, "\\\"");
            else
                ss_fprintf(&ss, "%c", *p);
            p++;
        }


        current = current->next;
        if (current)
            has_space = current->flags & TK_FLAG_HAS_SPACE_BEFORE;
    }

    if (bliteral)
        ss_fprintf(&ss, "\"");

    char* _Owner _Opt cstr = ss.c_str;
    ss.c_str = NULL; /*MOVED*/

    ss_close(&ss);

    return cstr;
}


void token_list_paste_string_after(struct token_list* list,
    struct token* after,
    const char* s)
{
    struct tokenizer_ctx tctx = { 0 };
    struct token_list l = tokenizer(&tctx, s, NULL, 0, TK_FLAG_FINAL);
    token_list_insert_after(list, after, &l);
    token_list_destroy(&l);
}

void token_list_paste_string_before(struct token_list* list,
    struct token* before,
    const char* s)
{
    struct tokenizer_ctx tctx = { 0 };
    struct token_list l = tokenizer(&tctx, s, NULL, 0, TK_FLAG_FINAL);
    token_list_insert_before(list, before, &l);
    token_list_destroy(&l);
}


void token_list_insert_after(struct token_list* token_list, struct token* _Opt after, struct token_list* append_list)
{
    if (append_list->head == NULL)
    {
        return;//nothing to append
    }

    if (token_list->head == NULL)
    {
        assert(after == NULL);
        token_list->head = append_list->head;
        token_list->tail = append_list->tail;
        append_list->head = NULL;
        append_list->tail = NULL;
        return;
    }

    if (after == NULL)
    {
        assert(append_list->tail != NULL);
        assert(append_list->tail->next == NULL);
        append_list->tail->next = token_list->head;
        token_list->head->prev = append_list->tail; //TODO empty case

        token_list->head = append_list->head;
        append_list->head->prev = NULL;
    }
    else
    {
        struct token* _Owner _Opt follow = after->next;
        if (token_list->tail == after)
        {
            token_list->tail = append_list->tail;
        }
        else if (token_list->head == after)
        {
        }
        assert(append_list->tail != NULL);
        assert(append_list->tail->next == NULL);
        append_list->tail->next = follow;
        follow->prev = append_list->tail;
        after->next = append_list->head;
        append_list->head->prev = after;

    }

    append_list->head = NULL;
    append_list->tail = NULL;
    assert(token_list->head == NULL || token_list->head->prev == NULL);
}

void token_list_insert_before(struct token_list* token_list, struct token* after, struct token_list* append_list)
{
    token_list_insert_after(token_list, after->prev, append_list);
}

bool token_list_is_equal(const struct token_list* list_a, const struct token_list* list_b)
{
    struct token* _Opt p_tka = list_a->head;
    struct token* _Opt p_tkb = list_b->head;

    while (p_tka && p_tkb)
    {
        if (p_tka->type != p_tkb->type)
            return false;

        if (strcmp(p_tka->lexeme, p_tkb->lexeme) != 0)
            return false;

        p_tka = p_tka->next;
        p_tkb = p_tkb->next;
    }

    return p_tka == NULL && p_tkb == NULL;
}

struct token* token_list_add(struct token_list* list, struct token* _Owner pnew) /*unchecked*/
{
    assert(pnew->next == NULL);
    assert(pnew->prev == NULL);

    if (list->head == NULL)
    {
        pnew->prev = NULL;
        pnew->next = NULL;
        list->head = pnew;
        list->tail = pnew;
    }
    else
    {
        assert(list->tail != NULL);
        assert(list->tail->next == NULL);

        pnew->prev = list->tail;
        list->tail->next = pnew;
        list->tail = pnew;
    }
    assert(list->tail != NULL);
    assert(list->tail->next == NULL);

    assert(list->head == NULL || list->head->prev == NULL);

    return list->tail;

}

int is_digit(const struct stream* p)
{
    /*
     digit : one of
     0 1 2 3 4 5 6 7 8 9
    */
    return (p->current[0] >= '0' && p->current[0] <= '9');
}

bool token_is_identifier_or_keyword(enum token_type t)
{
    switch (t)
    {
    case TK_IDENTIFIER: return true;

    case TK_KEYWORD_AUTO:
    case TK_KEYWORD_BREAK:
    case TK_KEYWORD_CASE:
    case TK_KEYWORD_CONSTEXPR:
    case TK_KEYWORD_CHAR:
    case TK_KEYWORD_CONST:
    case TK_KEYWORD_CONTINUE:
    case TK_KEYWORD_CAKE_CATCH: /*extension*/
    case TK_KEYWORD_DEFAULT:
    case TK_KEYWORD_DO:
    case TK_KEYWORD_DEFER: /*extension*/
    case TK_KEYWORD_DOUBLE:
    case TK_KEYWORD_ELSE:
    case TK_KEYWORD_ENUM:
    case TK_KEYWORD_EXTERN:
    case TK_KEYWORD_FLOAT:
    case TK_KEYWORD_FOR:
    case TK_KEYWORD_GOTO:
    case TK_KEYWORD_IF:
    case TK_KEYWORD_INLINE:
    case TK_KEYWORD_INT:
    case TK_KEYWORD_LONG:
    case TK_KEYWORD_MSVC__INT8:
    case TK_KEYWORD_MSVC__INT16:
    case TK_KEYWORD_MSVC__INT32:
    case TK_KEYWORD_MSVC__INT64:

    case TK_KEYWORD_REGISTER:
    case TK_KEYWORD_RESTRICT:
    case TK_KEYWORD_RETURN:
    case TK_KEYWORD_SHORT:
    case TK_KEYWORD_SIGNED:
    case TK_KEYWORD_SIZEOF:

    case TK_KEYWORD_STATIC:
    case TK_KEYWORD_STRUCT:
    case TK_KEYWORD_SWITCH:
    case TK_KEYWORD_TYPEDEF:
    case TK_KEYWORD_CAKE_TRY: /*extension*/
    case TK_KEYWORD_CAKE_THROW: /*extension*/
    case TK_KEYWORD_UNION:
    case TK_KEYWORD_UNSIGNED:
    case TK_KEYWORD_VOID:
    case TK_KEYWORD_VOLATILE:
    case TK_KEYWORD_WHILE:

    case TK_KEYWORD__ALIGNAS:
    case TK_KEYWORD__ALIGNOF:
    case TK_KEYWORD__ATOMIC:
        //microsoft
        //KEYWORD__FASTCALL:
        //KEYWORD__STDCALL
        // 
    case TK_KEYWORD__ASM:
        //end microsoft
    case TK_KEYWORD__BOOL:
    case TK_KEYWORD__COMPLEX:
    case TK_KEYWORD__DECIMAL128:
    case TK_KEYWORD__DECIMAL32:
    case TK_KEYWORD__DECIMAL64:
    case TK_KEYWORD__GENERIC:
    case TK_KEYWORD__IMAGINARY:
    case TK_KEYWORD__NORETURN:
    case TK_KEYWORD__STATIC_ASSERT:
    case TK_KEYWORD_ASSERT: /*extension*/
    case TK_KEYWORD__THREAD_LOCAL:

    case TK_KEYWORD_TYPEOF: /*C23*/

    case TK_KEYWORD_TRUE:  /*C23*/
    case TK_KEYWORD_FALSE:  /*C23*/
    case TK_KEYWORD_NULLPTR:  /*C23*/
    case TK_KEYWORD_TYPEOF_UNQUAL: /*C23*/
    case TK_KEYWORD__BITINT /*C23*/:



        /*cake extension*/
    case TK_KEYWORD_CAKE_OWNER:
    case TK_KEYWORD__CTOR:
    case TK_KEYWORD__DTOR:
    case TK_KEYWORD_CAKE_VIEW:
    case TK_KEYWORD_CAKE_OPT:


        /*extension compile time functions*/
    case TK_KEYWORD_CAKE_STATIC_DEBUG: /*extension*/
    case TK_KEYWORD_CAKE_STATIC_DEBUG_EX: /*extension*/
    case TK_KEYWORD_STATIC_STATE: /*extension*/
    case TK_KEYWORD_STATIC_SET: /*extension*/

        /*https://en.cppreference.com/w/cpp/header/type_traits*/

    case TK_KEYWORD_IS_POINTER:
    case TK_KEYWORD_IS_LVALUE:
    case TK_KEYWORD_IS_CONST:
    case TK_KEYWORD_IS_OWNER:
    case TK_KEYWORD_IS_ARRAY:
    case TK_KEYWORD_IS_FUNCTION:
    case TK_KEYWORD_IS_SCALAR:
    case TK_KEYWORD_IS_ARITHMETIC:
    case TK_KEYWORD_IS_FLOATING_POINT:
    case TK_KEYWORD_IS_INTEGRAL:
        return true;
    default:
        break;
    }

    return false;
}


bool token_is_blank(const struct token* p)
{
    return p->type == TK_BEGIN_OF_FILE ||
        p->type == TK_BLANKS ||
        p->type == TK_LINE_COMMENT ||
        p->type == TK_COMMENT;
}

struct token* _Opt token_list_clone_and_add(struct token_list* list, struct token* pnew)
{
    struct token* _Owner _Opt clone = clone_token(pnew);

    if (clone == NULL)
        return NULL;

    return token_list_add(list, clone);
}

void token_list_append_list_at_beginning(struct token_list* dest, struct token_list* source)
{
    if (source->head == NULL)
    {
        return;
    }

    if (dest->head == NULL)
    {
        dest->head = source->head;
        dest->tail = source->tail;
    }
    else
    {
        assert(source->tail != NULL);
        assert(source->tail->next == NULL);
        source->tail->next = dest->head;
        dest->head = source->head;
    }

    source->head = NULL;
    source->tail = NULL;
    assert(dest->head == NULL || dest->head->prev == NULL);
}

void token_list_append_list(struct token_list* dest, struct token_list* source)
{
    if (source->head == NULL)
    {
        return;
    }
    if (dest->head == NULL)
    {
        dest->head = source->head;
        dest->tail = source->tail;
    }
    else
    {
        assert(dest->tail != NULL);
        assert(dest->tail->next == NULL);
        dest->tail->next = source->head;
        source->head->prev = dest->tail;
        dest->tail = source->tail;
    }
    source->head = NULL;
    source->tail = NULL;
    assert(dest->head == NULL || dest->head->prev == NULL);
}


struct token* _Owner _Opt clone_token(struct token* p)
{
    _Opt struct token* _Owner _Opt token = calloc(1, sizeof * token);
    if (token == NULL)
        return NULL;

    char* _Owner _Opt lexeme = strdup(p->lexeme);
    if (lexeme == NULL)
    {
        free(token);
        return NULL;
    }

    *token = *p;
    token->lexeme = lexeme;
    token->next = NULL;
    token->prev = NULL;

    return token;
}


struct token_list token_list_remove_get(struct token_list* list, struct token* first, struct token* last)
{
    /*
       token_list_remove_get removes a range of tokens from a doubly - linked token list and 
       returns them as a new list.  It does not delete the tokens; it just detaches them from 
       the original list.
    */

    struct token_list r = { 0 };

    struct token* _Opt before_first = first->prev;
    struct token* _Owner _Opt after_last = last->next; /*MOVED*/

    if (before_first)
    {
        before_first->next = after_last;
    }
    else
    {
        list->head = last->next;
    }

    if (after_last)
    {
        after_last->prev = before_first;
    }
    else
    {
        list->tail = NULL;
    }

    last->next = NULL; /*MOVED*/

    r.head = (struct token* _Owner)first;
    first->prev = NULL;
    r.tail = last;


    return r;
}

void token_list_remove(struct token_list* list, struct token* first, struct token* last)
{
    struct token_list r = token_list_remove_get(list, first, last);
    token_list_destroy(&r);
}


bool token_list_is_empty(struct token_list* p)
{
    assert((p->head == NULL && p->tail == NULL) ||
        (p->head != NULL && p->tail != NULL));

    return p->head == NULL;
}

void print_list(struct token_list* list)
{
    struct token* _Opt current = list->head;
    while (current)
    {
        if (current != list->head)
        {
            printf(u8"˰");
            //printf("`");
        }
        print_literal2(current->lexeme);
        printf(RESET);
        if (current == list->tail)
        {
            //printf("`");
        }
        current = current->next;
    }
    printf(u8"\n");
}

void print_literal2(const char* s)
{
    while (*s)
    {
        switch (*s)
        {
        case '\n':
            printf("\\n");
            break;
        default:
            printf("%c", *s);
        }
        s++;
    }
    //printf("`");
}


void print_token(const struct token* p_token)
{
    for (int i = 0; i < p_token->level; i++)
    {
        printf("  ");
    }
    if (p_token->flags & TK_FLAG_FINAL)
        printf(LIGHTGREEN);
    else
        printf(LIGHTGRAY);
    char buffer0[50] = { 0 };
    snprintf(buffer0, sizeof buffer0, "%d:%d", p_token->line, p_token->col);
    printf("%-6s ", buffer0);
    printf("%-20s ", get_token_name(p_token->type));
    if (p_token->flags & TK_FLAG_MACRO_EXPANDED)
    {
        printf(LIGHTCYAN);
    }
    char buffer[50] = { 0 };
    strcat(buffer, "[");
    if (p_token->flags & TK_FLAG_FINAL)
    {
        strcat(buffer, "final ");
    }
    if (p_token->flags & TK_C_BACKEND_FLAG_HIDE)
    {
        strcat(buffer, "hide ");
    }
    if (p_token->flags & TK_FLAG_MACRO_EXPANDED)
    {
        strcat(buffer, "expanded ");
    }
    if (p_token->flags & TK_FLAG_HAS_SPACE_BEFORE)
    {
        strcat(buffer, "space ");
    }
    if (p_token->flags & TK_FLAG_HAS_NEWLINE_BEFORE)
    {
        strcat(buffer, "newline ");
    }
    strcat(buffer, "]");
    printf("%-20s ", buffer);
    print_literal2(p_token->lexeme);
    printf("\n");
    printf(RESET);
}

void print_tokens(const struct token* _Opt p_token)
{
    printf("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n" RESET);
    const struct token* _Opt current = p_token;
    while (current)
    {
        print_token(current);
        current = current->next;
    }
    printf("\n");
    printf("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n" RESET);
    printf(RESET);
}


void print_token_html(struct token* p_token)
{
    printf("<span class=\"");


    if (!(p_token->flags & TK_FLAG_FINAL))
    {
        printf("notfinal ");
    }

    if (p_token->flags & TK_FLAG_FINAL)
    {
        printf("final ");
    }
    if (p_token->flags & TK_C_BACKEND_FLAG_HIDE)
    {
        printf("hide ");
    }
    if (p_token->flags & TK_FLAG_MACRO_EXPANDED)
    {
        printf("expanded ");
    }
    if (p_token->flags & TK_FLAG_HAS_SPACE_BEFORE)
    {
        printf("space ");
    }
    if (p_token->flags & TK_FLAG_HAS_NEWLINE_BEFORE)
    {
        printf("newline ");
    }

    printf("\">");

    print_literal2(p_token->lexeme);

    printf("</span>");

    if (p_token->type == TK_NEWLINE || p_token->type == TK_BEGIN_OF_FILE)
    {
        printf("<br>\n");
    }
}

/*
 CSS for html ouput

 <style>
        .final {
          color:blue;
        }

        .notfinal {
          color:gray;
        }

        .hide {
          text-decoration: line-through;
          color:red;
        }

        .expanded {
           background-color:yellow;
        }

        span {
            border-style: solid;
            border-color: gray;
            border-width: 1px 1px;
            padding:1px;
            margin:2px;
        }

</style>

*/
void print_tokens_html(struct token* p_token)
{
    printf("<pre>\n");
    struct token* _Opt current = p_token;
    while (current)
    {
        print_token_html(current);
        current = current->next;
    }
    printf("\n</pre>");
}

void print_position(const char* path, int line, int col, bool visual_studio_ouput_format)
{

    if (visual_studio_ouput_format)
    {
        //MSVC format
        printf("%s(%d,%d): ", path ? path : "<>", line, col);
    }
    else
    {
        //GCC format
        printf(WHITE "%s:%d:%d: ", path ? path : "<>", line, col);
    }
}

void print_line_and_token(struct marker* p_marker, bool visual_studio_ouput_format)
{

    try
    {
        const struct token* _Opt p_token = p_marker->p_token_caret ? p_marker->p_token_caret : p_marker->p_token_begin;

        if (p_token == NULL)
            throw;

        const int line = p_marker->line;

        if (!visual_studio_ouput_format)
            printf(RESET);

        char nbuffer[20] = { 0 };
        int n = snprintf(nbuffer, sizeof nbuffer, "%d", line);
        printf(" %s |", nbuffer);


        //lets find the begin of line
        const struct token* p_line_begin = p_token;
        while (p_line_begin->prev && (p_line_begin->prev->type != TK_NEWLINE && p_line_begin->prev->type != TK_BEGIN_OF_FILE))
        {
            p_line_begin = p_line_begin->prev;
        }


        const struct token* _Opt p_token_begin = p_marker->p_token_begin ? p_marker->p_token_begin : p_marker->p_token_caret;
        const struct token* _Opt p_token_end = p_marker->p_token_end ? p_marker->p_token_end : p_marker->p_token_caret;

        if (p_token_begin == NULL)
            throw;


        //only expand macros if the error is inside
        const bool expand_macro = p_token_begin->flags & TK_FLAG_MACRO_EXPANDED;

        if (!visual_studio_ouput_format)
            printf(LIGHTBLUE);

        const struct token* _Opt p_item = p_line_begin;
        while (p_item)
        {
            if (!visual_studio_ouput_format)
            {
                if (p_item->flags & TK_FLAG_MACRO_EXPANDED)
                {
                    printf(DARKGRAY);
                }
                else if (p_item->type >= TK_KEYWORD_AUTO &&
                         p_item->type <= TK_KEYWORD_IS_INTEGRAL)
                {
                    printf(BLUE);
                }
                else if (p_item->type == TK_COMMENT ||
                         p_item->type == TK_LINE_COMMENT)
                {
                    printf(YELLOW);
                }
            }

            if (!(p_item->flags & TK_FLAG_MACRO_EXPANDED) || expand_macro)
            {
                const char* p = p_item->lexeme;
                while (*p)
                {
                    putc(*p, stdout);
                    p++;
                }
            }

            if (!visual_studio_ouput_format)
            {
                printf(RESET);
            }

            if (p_item->type == TK_NEWLINE)
                break;
            p_item = p_item->next;
        }

        if (!visual_studio_ouput_format)
            printf(RESET);

        if (p_item == NULL) printf("\n");

        printf(" %*s |", n, " ");
        bool complete = false;
        int start_col = 1;
        int end_col = 1;
        bool onoff = false;
        p_item = p_line_begin;
        while (p_item)
        {
            if (p_item == p_token_begin)
            {
                if (!visual_studio_ouput_format)
                    printf(LIGHTGREEN);
                onoff = true;
                end_col = start_col;
            }

            if (!(p_item->flags & TK_FLAG_MACRO_EXPANDED) || expand_macro)
            {
                const char* p = p_item->lexeme;
                while (*p)
                {

                    if (onoff)
                    {
                        putc('~', stdout);
                        end_col++;
                    }
                    else
                    {
                        putc(' ', stdout);
                        if (!complete) start_col++;
                    }
                    p++;
                }
            }

            if (p_item->type == TK_NEWLINE)
                break;

            if (p_item == p_token_end)
            {
                complete = true;
                onoff = false;
                if (!visual_studio_ouput_format)
                    printf(RESET);
            }

            p_item = p_item->next;
        }

        if (!visual_studio_ouput_format)
            printf(RESET);

        printf("\n");
        p_marker->start_col = start_col;
        p_marker->end_col = end_col;
    }
    catch
    {
    }
}

static void digit_sequence_opt(struct stream* stream)
{
    while (is_digit(stream))
    {
        stream_match(stream);
    }
}

static void binary_exponent_part(struct stream* stream)
{
    // p signopt digit - sequence
    // P   signopt digit - sequence

    stream_match(stream); // p or P
    if (stream->current[0] == '+' || stream->current[0] == '-')
    {
        stream_match(stream); // p or P
    }
    digit_sequence_opt(stream);
}

static bool is_hexadecimal_digit(struct stream* stream)
{
    return (stream->current[0] >= '0' && stream->current[0] <= '9') ||
        (stream->current[0] >= 'a' && stream->current[0] <= 'f') ||
        (stream->current[0] >= 'A' && stream->current[0] <= 'F');
}

static bool is_octal_digit(struct stream* stream)
{
    return stream->current[0] >= '0' && stream->current[0] <= '7';
}

static void hexadecimal_digit_sequence(struct stream* stream)
{
    /*
     hexadecimal-digit-sequence:
     hexadecimal-digit
     hexadecimal-digit ’_Opt hexadecimal-digit
    */

    stream_match(stream);
    while (stream->current[0] == '\'' ||
        is_hexadecimal_digit(stream))
    {
        if (stream->current[0] == '\'')
        {
            stream_match(stream);
            if (!is_hexadecimal_digit(stream))
            {
                // erro
            }
            stream_match(stream);
        }
        else
            stream_match(stream);
    }
}

static void integer_suffix_opt(struct stream* stream, char suffix[4])
{
    /*
        (6.4.4.2) integer-suffix:
          unsigned-suffix long-suffixopt
          unsigned-suffix long-long-suffix
          unsigned-suffix bit-precise-int-suffix
          long-suffix unsigned-suffixopt
          long-long-suffix unsigned-suffixopt
          bit-precise-int-suffix unsigned-suffixop
    */

    //test 3100
    if (/*unsigned-suffix*/
        stream->current[0] == 'U' || stream->current[0] == 'u')
    {
        suffix[0] = 'U';
        stream_match(stream);


        /*long-suffixopt*/
        if (stream->current[0] == 'l' || stream->current[0] == 'L')
        {
            suffix[1] = 'L';
            stream_match(stream);
        }

        /*long-long-suffix*/
        if (stream->current[0] == 'l' || stream->current[0] == 'L')
        {
            suffix[2] = 'L';
            stream_match(stream);
        }
    }
    else if ((stream->current[0] == 'l' || stream->current[0] == 'L'))
    {
        suffix[0] = 'L';

        /*long-suffix*/
        stream_match(stream);

        /*long-long-suffix*/
        if ((stream->current[0] == 'l' || stream->current[0] == 'L'))
        {
            suffix[1] = 'L';
            stream_match(stream);
        }

        if (/*unsigned-suffix*/
            stream->current[0] == 'U' || stream->current[0] == 'u')
        {

            //normalize the output from LLU to ul 
            suffix[3] = suffix[2];
            suffix[2] = suffix[1];
            suffix[1] = suffix[0];
            suffix[0] = 'U';
            stream_match(stream);
        }
    }
    ///////////////MICROSOFT ////////////////////////
        //TODO unit test
    else if (stream->current[0] == 'i' &&
             stream->current[1] == '8')
    {
        stream_match(stream);
        stream_match(stream);
        stream_match(stream);
        suffix[0] = 'i';
        suffix[1] = '8';
    }
    else if (stream->current[0] == 'i' &&
             stream->current[1] == '3' &&
             stream->current[2] == '2')
    {
        stream_match(stream);
        stream_match(stream);
        stream_match(stream);
        suffix[0] = 'i';
        suffix[1] = '3';
        suffix[2] = '2';
    }
    else if (stream->current[0] == 'i' &&
             stream->current[1] == '6' &&
             stream->current[2] == '4')
    {
        stream_match(stream);
        stream_match(stream);
        stream_match(stream);
        suffix[0] = 'i';
        suffix[1] = '6';
        suffix[2] = '4';
    }
    ///////////////MICROSOFT ////////////////////////
}

static void exponent_part_opt(struct stream* stream)
{
    /*
    exponent-part:
    e signopt digit-sequence
    E signopt digit-sequence
    */
    if (stream->current[0] == 'e' || stream->current[0] == 'E')
    {
        stream_match(stream);

        if (stream->current[0] == '-' || stream->current[0] == '+')
        {
            stream_match(stream);
        }
        digit_sequence_opt(stream);
    }
}

static void floating_suffix_opt(struct stream* stream, char suffix[4])
{

    if (stream->current[0] == 'l' || stream->current[0] == 'L')
    {
        suffix[0] = 'L';
        stream_match(stream);
    }
    else if (stream->current[0] == 'f' || stream->current[0] == 'F')
    {
        suffix[0] = 'F';
        stream_match(stream);
    }
}

static bool is_binary_digit(struct stream* stream)
{
    return stream->current[0] >= '0' && stream->current[0] <= '1';
}

static bool is_nonzero_digit(struct stream* stream)
{
    return stream->current[0] >= '1' && stream->current[0] <= '9';
}

enum token_type parse_number_core(struct stream* stream, char suffix[4], _Ctor char errmsg[100])
{
    errmsg[0] = '\0';

    enum token_type type = TK_NONE;
    if (stream->current[0] == '.')
    {
        type = TK_COMPILER_DECIMAL_FLOATING_CONSTANT;
        stream_match(stream);

        if (stream->current[0] == '.')
        {
            snprintf(errmsg, 100, "too many decimal points in number");
            return TK_NONE;
        }

        digit_sequence_opt(stream);
        exponent_part_opt(stream);
        floating_suffix_opt(stream, suffix);
    }
    else if (stream->current[0] == '0' && (stream->current[1] == 'x' || stream->current[1] == 'X'))
    {
        type = TK_COMPILER_HEXADECIMAL_CONSTANT;

        stream_match(stream);
        stream_match(stream);

        if (is_hexadecimal_digit(stream))
        {
            while (is_hexadecimal_digit(stream))
            {
                stream_match(stream);
            }
        }
        else
        {
            snprintf(errmsg, 100, "expected hexadecimal digit");
            return TK_NONE;
        }

        integer_suffix_opt(stream, suffix);

        if (stream->current[0] == '.')
        {
            type = TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT;
            hexadecimal_digit_sequence(stream);
        }

        if (stream->current[0] == 'p' ||
            stream->current[0] == 'P')
        {
            type = TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT;
            binary_exponent_part(stream);
        }

        if (type == TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT)
        {
            floating_suffix_opt(stream, suffix);
        }
    }
    else if (stream->current[0] == '0' && (stream->current[1] == 'b' || stream->current[1] == 'B'))
    {
        type = TK_COMPILER_BINARY_CONSTANT;
        stream_match(stream);
        stream_match(stream);
        if (is_binary_digit(stream))
        {
            while (is_binary_digit(stream))
            {
                stream_match(stream);
            }
        }
        else
        {
            snprintf(errmsg, 100, "expected binary digit");
            return TK_NONE;
        }
        integer_suffix_opt(stream, suffix);
    }
    else if (stream->current[0] == '0') // octal
    {
        type = TK_COMPILER_OCTAL_CONSTANT;

        stream_match(stream);

        if (stream->current[0] == 'O' || stream->current[0] == 'o')
        {
            //C2Y
            //https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3319.htm
            stream_match(stream);
        }

        while (is_octal_digit(stream))
        {
            stream_match(stream);
        }
        integer_suffix_opt(stream, suffix);

        if (stream->current[0] == '.')
        {
            type = TK_COMPILER_DECIMAL_FLOATING_CONSTANT;
            hexadecimal_digit_sequence(stream);
            floating_suffix_opt(stream, suffix);
        }
    }
    else if (is_nonzero_digit(stream)) // decimal
    {
        type = TK_COMPILER_DECIMAL_CONSTANT;

        stream_match(stream);
        while (is_digit(stream))
        {
            stream_match(stream);
        }
        integer_suffix_opt(stream, suffix);

        if (stream->current[0] == 'e' || stream->current[0] == 'E')
        {
            exponent_part_opt(stream);
            floating_suffix_opt(stream, suffix);

        }
        else if (stream->current[0] == '.')
        {
            stream_match(stream);
            type = TK_COMPILER_DECIMAL_FLOATING_CONSTANT;

            if (stream->current[0] == '.')
            {
                snprintf(errmsg, 100, "too many decimal points in number");
                return TK_NONE;
            }

            digit_sequence_opt(stream);

            exponent_part_opt(stream);
            floating_suffix_opt(stream, suffix);
        }
    }

    return type;
}

enum token_type parse_number(const char* lexeme, char suffix[4], _Ctor char errmsg[100])
{
    struct stream stream = {
        .source = lexeme,
        .current = lexeme,
        .line = 1,
        .col = 1,
        .path = "",
    };

    return parse_number_core(&stream, suffix, errmsg);
}

/*
    https://en.wikipedia.org/wiki/UTF-8
    Since the restriction of the Unicode code-space to 21-bit values in 2003,
    UTF-8 is defined to encode code points in one to four bytes, depending on the number
    of significant bits in the numerical value of the code point. The following table shows
    the structure of the encoding. The x characters are replaced by the bits of the code point.

    Code point <->UTF - 8 conversion
    First         | Last           | Byte 1   | Byte 2   | Byte 3   | Byte 4
    --------------| -------------- |----------|----------|----------| ----------
    U+0000      0 | U+007F     127 | 0xxxxxxx |          |          |
    U+0080    128 | U+07FF    2047 | 110xxxxx | 10xxxxxx |          |
    U+0800   2048 | U+FFFF   65535 | 1110xxxx | 10xxxxxx | 10xxxxxx |
    U+10000 65536 | U+10FFFF 69631 | 11110xxx | 10xxxxxx | 10xxxxxx | 10xxxxxx
*/

const unsigned char* _Opt utf8_decode(const unsigned char* s, _Ctor unsigned int* c)
{
    *c = 0; //out

    if (s[0] == '\0')
    {
        *c = 0;
        return NULL; /*end*/
    }

    const unsigned char* _Opt next = NULL;
    if (s[0] < 0x80)
    {
        *c = s[0];
        assert(*c >= 0x0000 && *c <= 0x007F);
        next = s + 1;
    }
    else if ((s[0] & 0xe0) == 0xc0)
    {
        *c = ((int)(s[0] & 0x1f) << 6) |
            ((int)(s[1] & 0x3f) << 0);
        assert(*c >= 0x0080 && *c <= 0x07FF);
        next = s + 2;
    }
    else if ((s[0] & 0xf0) == 0xe0)
    {
        *c = ((int)(s[0] & 0x0f) << 12) |
            ((int)(s[1] & 0x3f) << 6) |
            ((int)(s[2] & 0x3f) << 0);
        assert(*c >= 0x0800 && *c <= 0xFFFF);
        next = s + 3;
    }
    else if ((s[0] & 0xf8) == 0xf0 && (s[0] <= 0xf4))
    {
        *c = ((int)(s[0] & 0x07) << 18) |
            ((int)(s[1] & 0x3f) << 12) |
            ((int)(s[2] & 0x3f) << 6) |
            ((int)(s[3] & 0x3f) << 0);
        assert(*c >= 0x10000 && *c <= 0x10FFFF);
        next = s + 4;
    }
    else
    {
        *c = 0;      // invalid
        next = s + 1; // skip this byte
    }

    if (*c >= 0xd800 && *c <= 0xdfff)
    {
        *c = 0; // surrogate half
    }

    return next;
}

static bool is_hex_digit(unsigned char c)
{
    if (c >= '0' && c <= '9')
        return true;
    else if (c >= 'a' && c <= 'f')
        return true;
    else if (c >= 'A' && c <= 'F')
        return true;
    return false;
}

const unsigned char* _Opt escape_sequences_decode_opt(const unsigned char* p, unsigned int* out_value)
{
    // TODO OVERFLOW CHECK
    if (*p == 'x')
    {
        p++;
        int result = 0;
        while (is_hex_digit(*p))
        {
            int byte = 0;
            if (*p >= '0' && *p <= '9')
                byte = (*p - '0');
            else if (*p >= 'a' && *p <= 'f')
                byte = (*p - 'a') + 10;
            else if (*p >= 'A' && *p <= 'F')
                byte = (*p - 'A') + 10;

            result = (result << 4) | (byte & 0xF);
            p++;
        }

        *out_value = result;
    }
    else if (*p == 'u' || *p == 'U')
    {
        // TODO  assuming input is checked
        // missing tests
        const int num_of_hex_digits = *p == 'U' ? 8 : 4;

        p++;
        unsigned long long result = 0;
        for (int i = 0; i < num_of_hex_digits; i++)
        {
            int byte = 0;
            if (*p >= '0' && *p <= '9')
                byte = (*p - '0');
            else if (*p >= 'a' && *p <= 'f')
                byte = (*p - 'a') + 10;
            else if (*p >= 'A' && *p <= 'F')
                byte = (*p - 'A') + 10;

            result = (result << 4) | (byte & 0xF);
            p++;
        }

        *out_value = (int)result;
    }
    else if (*p == '0')
    {
        // octal digit
        p++;

        int result = 0;
        while ((*p >= '0' && *p <= '7'))
        {
            int byte;
            byte = (*p - '0');
            result = (result << 4) | (byte & 0xF);
            p++;
        }
        *out_value = result;
    }
    else
    {
        switch (*p)
        {
        case 'a':
            *out_value = '\a';
            break;
        case 'b':
            *out_value = '\b';
            break;
        case 'f':
            *out_value = '\f';
            break;
        case 'n':
            *out_value = '\n';
            break;
        case 'r':
            *out_value = '\r';
            break;
            ;
        case 't':
            *out_value = '\t';
            break;
        case '\'':
            *out_value = '\'';
            break;
        case '\\':
            *out_value = '\\';
            break;
        case '"':
            *out_value = '"';
            break;
        default:
            // this is handled at tokenizer
            assert(false);
            return NULL;
        }
        p++;
    }

    return p;
}

#ifdef TEST

void token_list_remove_get_test()
{
    struct token_list list = { 0 };
    struct token* pnew = calloc(1, sizeof * pnew);
    token_list_add(&list, pnew);
    struct token_list r = token_list_remove_get(&list, pnew, pnew);
    assert(list.head == NULL);
    assert(list.tail == NULL);
}

void token_list_remove_get_test2()
{
    struct token_list list = { 0 };
    struct token* pnew1 = calloc(1, sizeof * pnew1);
    token_list_add(&list, pnew1);
    struct token* pnew2 = calloc(1, sizeof * pnew2);
    token_list_add(&list, pnew2);

    struct token_list r = token_list_remove_get(&list, pnew1, pnew1);
    assert(list.head == pnew2);
    assert(list.tail == pnew2);
}


#endif


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

#pragma safety enable


#ifdef _WIN32
#endif

#if defined _MSC_VER
#endif

static unsigned int string_hash(const char* key)
{
    // hash key to unsigned int value by pseudorandomizing transform
    // (algorithm copied from STL char hash in xfunctional)
    unsigned int hash_val = 2166136261U;
    unsigned int first = 0;
    unsigned int last = (unsigned int)strlen(key);
    unsigned int stride = 1 + last / 10;

    for (; first < last; first += stride)
    {
        hash_val = 16777619U * hash_val ^ (unsigned int)key[first];
    }

    return (hash_val);
}


void map_entry_delete(struct map_entry* _Owner _Opt p)
{
    if (p == NULL)
        return;

    switch (p->type)
    {
    case TAG_TYPE_NUMBER:break;

    case TAG_TYPE_ENUN_SPECIFIER:
        enum_specifier_delete(p->data.p_enum_specifier);
        break;
    case TAG_TYPE_STRUCT_OR_UNION_SPECIFIER:
        struct_or_union_specifier_delete(p->data.p_struct_or_union_specifier);
        break;

    case TAG_TYPE_ENUMERATOR:
        enumerator_delete(p->data.p_enumerator);
        break;
    case TAG_TYPE_DECLARATOR:
        declarator_delete(p->data.p_declarator);
        break;
    case TAG_TYPE_INIT_DECLARATOR:
        init_declarator_delete(p->data.p_init_declarator);
        break;
    case TAG_TYPE_MACRO:
        macro_delete(p->data.p_macro);
        break;

    case TAG_TYPE_STRUCT_ENTRY:
        struct_entry_delete(p->data.p_struct_entry);
        break;
    }

    free(p->key);
    free(p);
}

void hashmap_remove_all(struct hash_map* map)
{

    if (map->table != NULL)
    {
        for (int i = 0; i < map->capacity; i++)
        {
            struct map_entry* _Owner _Opt pentry = map->table[i];

            while (pentry != NULL)
            {
                struct map_entry* _Owner _Opt next = pentry->next;
                map_entry_delete(pentry);
                pentry = next;
            }
        }

        free(map->table);
        map->table = NULL;
        map->size = 0;
    }
}

void hashmap_destroy(_Dtor struct hash_map* map)
{
    hashmap_remove_all(map);
    assert(map->table == NULL);
}

struct map_entry* _Opt hashmap_find(struct hash_map* map, const char* key)
{
    if (map->table == NULL)
        return NULL;

    const unsigned int hash = string_hash(key);
    const int index = hash % map->capacity;

    struct map_entry* _Opt pentry = map->table[index];

    for (; pentry != NULL; pentry = pentry->next)
    {
        if (pentry->hash == hash && strcmp(pentry->key, key) == 0)
        {
            return pentry;
        }
    }

    return NULL;
}


void* _Opt hashmap_remove(struct hash_map* map, const char* key, enum tag* _Opt p_type_opt)
{
    if (map->table != NULL)
    {
        const unsigned int hash = string_hash(key);
        struct map_entry** pp_entry = &map->table[hash % map->capacity];
        struct map_entry* _Opt p_entry = *pp_entry;

        for (; p_entry != NULL; p_entry = p_entry->next)
        {
            if ((p_entry->hash == hash) && (strcmp(p_entry->key, key) == 0))
            {
                *pp_entry = p_entry->next;

                if (p_type_opt)
                    *p_type_opt = p_entry->type;

                void* _Opt p = p_entry->data.p_declarator;
                free((void* _Owner)p_entry->key);
                free((void* _Owner)p_entry);

                return p;
            }
            pp_entry = &p_entry->next;
        }
    }

    return NULL;
}

void hash_item_set_destroy(_Dtor struct hash_item_set* p)
{
    declarator_delete(p->p_declarator);
    enumerator_delete(p->p_enumerator);
    enum_specifier_delete(p->p_enum_specifier);
    init_declarator_delete(p->p_init_declarator);
    struct_or_union_specifier_delete(p->p_struct_or_union_specifier);
    macro_delete(p->p_macro);

}

int hashmap_set(struct hash_map* map, const char* key, struct hash_item_set* item /*in out*/)
{
    int result = 0;

    void* _Opt p = NULL;
    enum tag type = TAG_TYPE_NUMBER;
    if (item->p_declarator)
    {
        type = TAG_TYPE_DECLARATOR;
        p = item->p_declarator;
        item->p_declarator = NULL;//

    }
    else if (item->p_enumerator)
    {
        type = TAG_TYPE_ENUMERATOR;
        p = item->p_enumerator;
        item->p_enumerator = NULL;

    }
    else if (item->p_enum_specifier)
    {
        type = TAG_TYPE_ENUN_SPECIFIER;
        p = item->p_enum_specifier;
        item->p_enum_specifier = NULL;

    }
    else if (item->p_init_declarator)
    {
        type = TAG_TYPE_INIT_DECLARATOR;
        p = item->p_init_declarator;
        item->p_init_declarator = NULL;

    }
    else if (item->p_struct_or_union_specifier)
    {
        type = TAG_TYPE_STRUCT_OR_UNION_SPECIFIER;
        p = item->p_struct_or_union_specifier;
        item->p_struct_or_union_specifier = NULL;

    }
    else if (item->p_macro)
    {
        type = TAG_TYPE_MACRO;
        p = item->p_macro;
        item->p_macro = NULL;
    }
    else if (item->p_struct_entry)
    {
        type = TAG_TYPE_STRUCT_ENTRY;
        p = item->p_struct_entry;
        item->p_struct_entry = NULL;
    }
    else //if (item->number)
    {
        type = TAG_TYPE_NUMBER;
        p = (void*)item->number;
    }
    // else
    // {
     //    assert(false);
     //}

    try
    {
        if (map->table == NULL)
        {
            if (map->capacity < 1)
            {
                map->capacity = 1000;
            }

            map->table = calloc(map->capacity, sizeof(map->table[0]));
            if (map->table == NULL) throw;
        }

        if (map->table != NULL)
        {
            unsigned int hash = string_hash(key);
            int index = hash % map->capacity;

            struct map_entry* _Opt pentry = map->table[index];

            for (; pentry != NULL; pentry = pentry->next)
            {
                if (pentry->hash == hash && strcmp(pentry->key, key) == 0)
                {
                    break;
                }
            }

            if (pentry == NULL)
            {
                struct map_entry* _Owner _Opt p_new_entry = calloc(1, sizeof(*pentry));
                if (p_new_entry == NULL) throw;

                p_new_entry->hash = hash;

                p_new_entry->data.p_declarator = (void*)p;

                p_new_entry->type = type;

                char* _Opt _Owner temp_key = strdup(key);
                if (temp_key == NULL)
                {
                    map_entry_delete(p_new_entry);
                    throw;
                }

                p_new_entry->key = temp_key;
                p_new_entry->next = map->table[index];
                map->table[index] = p_new_entry;
                map->size++;
                result = 0;
            }
            else
            {
                switch (pentry->type)
                {
                case TAG_TYPE_NUMBER:break;

                case TAG_TYPE_ENUN_SPECIFIER:
                    assert(pentry->data.p_enum_specifier != NULL);
                    item->p_enum_specifier = pentry->data.p_enum_specifier;
                    break;
                case TAG_TYPE_STRUCT_OR_UNION_SPECIFIER:
                    assert(pentry->data.p_struct_or_union_specifier != NULL);
                    item->p_struct_or_union_specifier = pentry->data.p_struct_or_union_specifier;
                    break;

                case TAG_TYPE_ENUMERATOR:
                    assert(pentry->data.p_enumerator != NULL);
                    item->p_enumerator = pentry->data.p_enumerator;
                    break;
                case TAG_TYPE_DECLARATOR:
                    assert(pentry->data.p_declarator != NULL);
                    item->p_declarator = pentry->data.p_declarator;
                    break;
                case TAG_TYPE_INIT_DECLARATOR:
                    assert(pentry->data.p_init_declarator != NULL);
                    item->p_init_declarator = pentry->data.p_init_declarator;
                    break;
                case TAG_TYPE_MACRO:
                    assert(pentry->data.p_macro != NULL);
                    item->p_macro = pentry->data.p_macro;
                    break;
                case TAG_TYPE_STRUCT_ENTRY:
                    assert(pentry->data.p_struct_entry != NULL);
                    item->p_struct_entry = pentry->data.p_struct_entry;
                    break;
                }

                result = 1;
                pentry->data.p_declarator = (void*)p;
                pentry->type = type;
            }
        }
    }
    catch
    {
    }
    return result;
}



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

#pragma safety enable


#ifdef _WIN32


#include <conio.h>
#else


#include <termios.h>


#include <unistd.h>


#include <fcntl.h>


#include <sys/ioctl.h>
#endif

#ifndef WIN32

bool enable_vt_mode(void)
{
    return true;
}

int c_kbhit(void)
{
    struct termios oldt = { 0 };
    struct termios newt = { 0 };
    int ch;
    int oldf;

    tcgetattr(STDIN_FILENO, &oldt);
    newt = oldt;
    newt.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &newt);
    oldf = fcntl(STDIN_FILENO, F_GETFL, 0);
    fcntl(STDIN_FILENO, F_SETFL, oldf | O_NONBLOCK);

    ch = getchar();

    tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
    fcntl(STDIN_FILENO, F_SETFL, oldf);

    if (ch != EOF)
    {
        ungetc(ch, stdin);
        return 1;
    }

    return 0;
}

/* Read 1 character without echo */
int c_getch(void)
{
    struct termios old = { 0 };
    struct termios new = { 0 };
    int ch;

    tcgetattr(0, &old);

    new = old;
    new.c_lflag &= ~ICANON;
    new.c_lflag &= ~ECHO;
    tcsetattr(0, TCSANOW, &new);

    ch = getchar();

    tcsetattr(0, TCSANOW, &old);

    return ch;
}

#else

bool enable_vt_mode(void)
{
    //missing in mingw (installed with codeblocs)
#ifndef ENABLE_VIRTUAL_TERMINAL_PROCESSING  
#define ENABLE_VIRTUAL_TERMINAL_PROCESSING  0x0004
#endif

    DWORD mode = 0;
    HANDLE h_out = GetStdHandle(STD_OUTPUT_HANDLE);
    if (h_out != INVALID_HANDLE_VALUE &&
            GetConsoleMode(h_out, &mode) != 0 &&
            SetConsoleMode(h_out, mode |= ENABLE_VIRTUAL_TERMINAL_PROCESSING) != 0 &&
            SetConsoleOutputCP(CP_UTF8) != 0)
    {
        return true;//ok
    }
    return false;//error
}

int c_kbhit(void)
{
    return _kbhit();
}
int c_getch(void)
{
    return _getch();
}

#endif

void c_clrscr()
{
    puts("\x1b[2J\x1b[1;1H");
    puts("\x1b[3J");
    fflush(stdout);
}

void c_gotoxy(int x, int y)
{
    printf("\x1b[%d;%dH", y, x);
    fflush(stdout);
}

/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

//#pragma safety enable

/*

   1 - The initial input is a string.
   2 - This string is transformed into a linked list of tokens

   ┌───┐   ┌───┐   ┌───┐   ┌───┐
   │   ├──►│   ├──►│   ├──►│   │──► NULL
   └───┘   └───┘   └───┘   └───┘

   The list is then expanded using includes and macros through a preprocessor.


   ┌───┐                  ┌───┐   ┌───┐   ┌───┐
   │   ├──────┐           │   ├──►│x  ├──►│x  │ ──► NULL   Level 0
   └───┘      │           └───┘   └───┘   └───┘
            ┌─▼─┐   ┌───┐   ▲
            │   ├───┤   ├───┘  (includes)                  Level 1
            └───┘   └───┘


    Each item in the list has additional properties:

    level         :  An integer indicating the level of inclusion.

    bmacroexpanded: A boolean indicating whether the token was generated
                    from macro expansion.

    bfinal        : A boolean indicating whether the token is the
                    final, the one seen by the parser.

*/



#include <ctype.h>


#include <sys/stat.h>


#include <errno.h>


#include <stddef.h>


#include <time.h>


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once
 

#ifdef _WIN32 


#include <direct.h>


#include <sys/types.h>

#ifdef __CAKE__
#pragma cake diagnostic push
#pragma cake diagnostic ignored "-Wstyle"
#endif


//https://docs.microsoft.com/pt-br/cpp/c-runtime-library/reference/mkdir-wmkdir?_View=msvc-160
#define mkdir(a, b) _mkdir(a)
#define rmdir _rmdir
#define chdir _chdir

#ifdef __CAKE__
#pragma cake diagnostic pop
#endif

/*
 opendir,  readdir closedir for windows.
 include dirent.h on linux
*/



enum
{
    DT_UNKNOWN = 0,
    DT_FIFO = 1,
    DT_CHR = 2,
    DT_DIR = 4,
    DT_BLK = 6,
    DT_REG = 8,
    DT_LNK = 10,
    DT_SOCK = 12,
    DT_WHT = 14
};

struct dirent
{
    ino_t d_ino;             /* Inode number */
    off_t d_off;             /* Not an offset; see below */
    unsigned short d_reclen; /* Length of this record */
    unsigned char d_type;    /* Type of file; not supported
                                     by all filesystem types*/
    char d_name[256];        /* Null-terminated filename */
};

#ifdef __CAKE__
#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wstyle"
#endif
struct TAGDIR;
typedef struct TAGDIR DIR;

#ifdef __CAKE__
#pragma CAKE diagnostic pop
#endif

DIR* _Owner _Opt opendir(const char* name);
int closedir(DIR* _Owner dirp);
struct dirent* _Opt readdir(DIR* dirp);


#else

//TODO fails on macos because it has a diferent declaration
//typedef struct __dirstream DIR;
//DIR * _Owner _Opt opendir (const char *__name);
//int closedir(DIR* _Owner dirp);


#define MAX_PATH 500

//https://man7.org/linux/man-pages/man2/mkdir.2.html


#include <dirent.h>
#endif



char* _Opt realpath(const char* restrict path, char* restrict resolved_path);

int get_self_path(char* buffer, int maxsize);

char* _Owner _Opt read_file(const char* path, bool append_newline);
char* dirname(char* path);
char* basename(const char* filename);

const char* get_posix_error_message(int error);


bool path_is_relative(const char* path);
bool path_is_absolute(const char* path);
void path_normalize(char* path);
bool path_is_normalized(const char* path);




/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once

struct preprocessor_ctx;

int pre_constant_expression(struct preprocessor_ctx* ctx, long long* pvalue);



#ifdef _WIN32
#endif

#if defined _MSC_VER && !defined __POCC__
#endif

#define STRINGIFY(x) #x
#define TOSTRING(x) STRINGIFY(x)


// Includes tokens that are not necessary for compilation at second level of includes
enum { INCLUDE_ALL = 1 };

///////////////////////////////////////////////////////////////////////////////
void naming_convention_macro(struct preprocessor_ctx* ctx, struct token* token);
///////////////////////////////////////////////////////////////////////////////

static bool is_builtin_macro(const char* name);


struct macro_parameter
{
    const char* _Owner name;
    struct macro_parameter* _Owner _Opt next;

    /*
      https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3457.htm#number-of-expansions
      For each such parameter this expansion is performed exactly once
      (this list and flag are clean and reused when performing argument expansion)
    */
    struct token_list expanded_list;
    bool already_expanded;
};



struct macro
{
    const struct token* p_name_token;
    const char* _Owner name;
    struct token_list replacement_list; /*copy*/
    struct macro_parameter* _Owner _Opt parameters;
    bool is_function;
    int usage;

    
    bool def_macro;
};


void macro_delete(struct macro* _Owner _Opt macro);
bool macro_is_same(const struct macro* macro_a, const struct macro* macro_b);


void include_dir_list_destroy(_Dtor struct include_dir_list* list)
{
    struct include_dir* _Owner _Opt p = list->head;
    while (p)
    {
        struct include_dir* _Owner _Opt next = p->next;
        free((void* _Owner)p->path);
        free(p);
        p = next;
    }
}

void preprocessor_ctx_destroy(_Dtor struct preprocessor_ctx* p)
{
    hashmap_destroy(&p->macros);
    include_dir_list_destroy(&p->include_dir);
    hashmap_destroy(&p->pragma_once_map);
    token_list_destroy(&p->input_list);
}

struct token_list preprocessor(struct preprocessor_ctx* ctx, struct token_list* input_list, int level);

static void tokenizer_set_error(struct tokenizer_ctx* ctx, struct stream* stream, const char* fmt, ...)
{
    ctx->n_errors++;

    char buffer[200] = { 0 };

#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wnullable-to-non-nullable"
#pragma CAKE diagnostic ignored "-Wanalyzer-null-dereference"



    va_list args = { 0 };
    va_start(args, fmt);
    /*int n =*/ vsnprintf(buffer, sizeof(buffer), fmt, args);
    va_end(args);

#pragma CAKE diagnostic pop

    print_position(stream->path, stream->line, stream->col, ctx->options.visual_studio_ouput_format);
    if (ctx->options.visual_studio_ouput_format)
    {
        printf("error: "  "%s\n", buffer);
    }
    else
    {
        printf(LIGHTRED "error: " WHITE "%s\n", buffer);
    }
}


static void tokenizer_set_warning(struct tokenizer_ctx* ctx, struct stream* stream, const char* fmt, ...)
{
    ctx->n_warnings++;


    char buffer[200] = { 0 };

#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wnullable-to-non-nullable"
#pragma CAKE diagnostic ignored "-Wanalyzer-null-dereference"


    va_list args = { 0 };
    va_start(args, fmt);
    /*int n =*/ vsnprintf(buffer, sizeof(buffer), fmt, args);
    va_end(args);

#pragma CAKE diagnostic pop

    print_position(stream->path, stream->line, stream->col, ctx->options.visual_studio_ouput_format);
    if (ctx->options.visual_studio_ouput_format)
    {
        printf("warning: " "%s\n", buffer);
    }
    else
    {
        printf(LIGHTMAGENTA "warning: " WHITE "%s\n", buffer);
    }

}


void pre_unexpected_end_of_file(struct token* _Opt p_token, struct preprocessor_ctx* ctx)
{
    preprocessor_diagnostic(C_ERROR_UNEXPECTED_TOKEN,
        ctx,
        p_token,
        "unexpected end of file");
}

bool preprocessor_diagnostic(enum diagnostic_id w, struct preprocessor_ctx* ctx, const struct token* _Opt p_token_opt, const char* fmt, ...)
{
    struct marker marker = { 0 };

    if (p_token_opt == NULL) return false;

    marker.file = p_token_opt->token_origin->lexeme;
    marker.line = p_token_opt->line;
    marker.start_col = p_token_opt->col;
    marker.end_col = p_token_opt->col;
    marker.p_token_caret = p_token_opt;

    /*warnings inside headers are ignored*/
    const bool included_file_location = p_token_opt->level > 0;

    bool is_error = false;
    bool is_warning = false;
    bool is_note = false;

    if (w > W_NOTE)
    {
        is_error = true;
    }
    else
    {
        is_error =
            (ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].errors & (1ULL << w)) != 0;

        is_warning =
            (ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings & (1ULL << w)) != 0;

        is_note =
            ((ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].notes & (1ULL << w)) != 0);
    }

    if (is_error)
    {
        ctx->n_errors++;
    }
    else if (is_warning)
    {
        ctx->n_warnings++;
    }
    else if (is_note)
    {

    }
    else
    {
        return false;
    }

    if (!is_error && included_file_location)
    {
        //notes are warning are not printed in included files
        return false;
    }

    print_position(marker.file, marker.line, marker.start_col, ctx->options.visual_studio_ouput_format);

    char buffer[200] = { 0 };

#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wnullable-to-non-nullable"
#pragma CAKE diagnostic ignored "-Wanalyzer-null-dereference"

    va_list args = { 0 };

    va_start(args, fmt);
    /*int n =*/ vsnprintf(buffer, sizeof(buffer), fmt, args);
    va_end(args);
#pragma CAKE diagnostic pop

    if (ctx->options.visual_studio_ouput_format)
    {
        if (is_warning)
            printf("warning: " "%s\n", buffer);
        else if (is_error)
            printf("warning: " "%s\n", buffer);
        else if (is_note)
            printf("note: " "%s\n", buffer);

        print_line_and_token(&marker, ctx->options.visual_studio_ouput_format);
    }
    else
    {
        if (is_error)
            printf(LIGHTRED "error: " WHITE "%s\n", buffer);
        else if (is_warning)
            printf(LIGHTMAGENTA "warning: " WHITE "%s\n", buffer);
        else if (is_note)
            printf(LIGHTCYAN "note: " WHITE "%s\n", buffer);

        print_line_and_token(&marker, ctx->options.visual_studio_ouput_format);

    }

    return true;
}

struct include_dir* _Opt include_dir_add(struct include_dir_list* list, const char* path)
{
    try
    {
        struct include_dir* _Owner _Opt p_new_include_dir = calloc(1, sizeof * p_new_include_dir);
        if (p_new_include_dir == NULL)
            throw;

        int len = strlen(path);
        if (path[len - 1] == '\\')
        {
            //windows path format ending with \ .
            const char* _Owner _Opt temp = strdup(path);
            if (temp == NULL)
            {
                free(p_new_include_dir);
                throw;
            }
            p_new_include_dir->path = temp;
        }
        else if (path[len - 1] != '/')
        {
            /*
              not ending with \, we add it
            */
            const char* _Owner _Opt temp = calloc(len + 2, sizeof(char));
            if (temp == NULL)
            {
                free(p_new_include_dir);
                throw;
            }

            p_new_include_dir->path = temp;
            snprintf((char*)p_new_include_dir->path, len + 2, "%s/", path);
        }
        else
        {
            const char* _Owner _Opt temp = strdup(path);
            if (temp == NULL)
            {
                free(p_new_include_dir);
                throw;
            }
            p_new_include_dir->path = temp;
        }

        if (list->head == NULL)
        {
            list->head = p_new_include_dir;
            list->tail = p_new_include_dir;
        }
        else
        {
            assert(list->tail != NULL);
            assert(list->tail->next == NULL);
            list->tail->next = p_new_include_dir;
            list->tail = p_new_include_dir;
        }
        return list->tail;
    }
    catch
    {

    }
    return NULL;
}

/*
  We must ensure we have always the same representation for path when searching and inserting
  at pragma once map
*/

static void pragma_once_add(struct preprocessor_ctx* ctx, const char* path)
{
    //FAILING ON EMSCRIPT
    //assert(path_is_absolute(path));
    //assert(path_is_normalized(path));
    struct hash_item_set item = { 0 };
    item.number = 1;
    hashmap_set(&ctx->pragma_once_map, path, &item /*in out*/);
    hash_item_set_destroy(&item);
}

static bool pragma_once_already_included(struct preprocessor_ctx* ctx, const char* path)
{
    //FAILING ON EMSCRIPT
    //assert(path_is_absolute(path));
    //assert(path_is_normalized(path));
    return hashmap_find(&ctx->pragma_once_map, path) != NULL;
}

const char* _Owner _Opt  find_and_read_include_file(struct preprocessor_ctx* ctx,
    const char* path, /*as in include*/
    const char* current_file_dir, /*this is the dir of the file that includes*/
    bool is_angle_bracket_form,
    bool* p_already_included, /*out file already included pragma once*/
    char full_path_out[], /*this is the final full path of the file*/
    int full_path_out_size,
   bool include_next)
{
    char newpath[200] = { 0 };
    full_path_out[0] = '\0';

    if (path_is_absolute(path))
    {
        snprintf(newpath, sizeof newpath, "%s", path);
        path_normalize(newpath);
        if (pragma_once_already_included(ctx, newpath))
        {
            *p_already_included = true;
            return NULL;
        }

        char* _Owner _Opt content = read_file(newpath, true);
        if (content != NULL)
        {
            snprintf(full_path_out, full_path_out_size, "%s", path);
            return content;
        }
        return NULL;
    }


    char* _Owner _Opt content = NULL;

    if (!is_angle_bracket_form)
    {
        /*
          For the angle-bracket form #include <file>, the preprocessor’s default
          behavior is to look only in the standard system directories.
        */

        //https://gcc.gnu.org/onlinedocs/cpp/Search-Path.html
        //https://learn.microsoft.com/en-us/cpp/preprocessor/hash-include-directive-c-cpp?view=msvc-170

        snprintf(newpath, sizeof newpath, "%s/%s", current_file_dir, path);

#ifdef __EMSCRIPTEN__
        /*realpath returns empty on emscriptem*/
        snprintf(full_path_out, full_path_out_size, "%s", newpath);
#else
        if (!realpath(newpath, full_path_out))
            full_path_out[0] = '\0';
#endif

        path_normalize(full_path_out);


        if (pragma_once_already_included(ctx, full_path_out))
        {
            *p_already_included = true;
            return NULL;
        }

        if (full_path_out[0] != '\0')
        {
            content = read_file(full_path_out, true);
        }
        if (content != NULL)
            return content;
    }

    /*
       Searching on include directories
    */
    struct include_dir* _Opt current = ctx->include_dir.head;
    while (current)
    {
        int len = strlen(current->path);
        if (current->path[len - 1] == '/')
        {
            snprintf(full_path_out, full_path_out_size, "%s%s", current->path, path);
        }
        else
        {
            snprintf(full_path_out, full_path_out_size, "%s/%s", current->path, path);
        }

        path_normalize(full_path_out);
        if (pragma_once_already_included(ctx, full_path_out))
        {
            *p_already_included = true;
            return NULL;
        }

        content = read_file(full_path_out, true);
        if (content != NULL)
        {
            if (include_next)
            {
                free(content);
                content = NULL;
                include_next = false;
            }
            else
                return content;
        }
        current = current->next;
    }
    full_path_out[0] = '\0';
    return NULL;
}

/*used to check recursion*/
struct macro_expanded
{
    const char* name;
    struct macro_expanded* _Opt p_previous;
};


void add_macro(struct preprocessor_ctx* ctx, const char* name)
{
    try
    {
        char* _Owner _Opt name_local = strdup(name);
        if (name_local == NULL)
        {
            throw;
        }

        struct macro* _Owner _Opt macro = calloc(1, sizeof * macro);
        if (macro == NULL)
        {
            free(name_local);
            throw;
        }

        macro->name = name_local;
        struct hash_item_set item = { .p_macro = macro };
        hashmap_set(&ctx->macros, name, &item);
        hash_item_set_destroy(&item);
    }
    catch
    {
    }
}

struct macro_argument
{
    /*the parameter this argument is associated with*/
    struct macro_parameter* _Opt macro_parameter;
    struct token_list tokens;
    struct macro_argument* _Owner _Opt next; /*linked list*/
};

void macro_argument_delete(struct macro_argument* _Owner _Opt p);



struct token_list copy_argument_list_tokens(struct token_list* list)
{
    // Makes a copy of the tokens, trimming the beginning and end
    // Any space, comments, etc., will become a single space
    struct token_list r = { 0 };
    struct token* _Opt current = list->head; /*null is fine*/

    // skip all leading white spaces
    while (current &&
        (token_is_blank(current) ||
            current->type == TK_NEWLINE))
    {
        current = current->next;
    }

    // Removes leading space flag if present
    bool is_first = true;

    for (; current;)
    {
        if (current && (token_is_blank(current) ||
            current->type == TK_NEWLINE))
        {
            if (current == list->tail)
                break;

            current = current->next;
            continue;
        }
        struct token* token = token_list_clone_and_add(&r, current);
        if (token->flags & TK_FLAG_HAS_NEWLINE_BEFORE)
        {
            token->flags = token->flags & ~TK_FLAG_HAS_NEWLINE_BEFORE;
            token->flags |= TK_FLAG_HAS_SPACE_BEFORE;
        }
        if (is_first)
        {
            token->flags = token->flags & ~TK_FLAG_HAS_SPACE_BEFORE;
            token->flags = token->flags & ~TK_FLAG_HAS_NEWLINE_BEFORE;
            is_first = false;
        }
        remove_line_continuation(token->lexeme);

        if (current == list->tail)
            break;

        current = current->next;
    }

    return r;
}

void macro_argument_delete(struct macro_argument* _Owner _Opt p)
{
    if (p)
    {
        assert(p->next == NULL);
        token_list_destroy(&p->tokens);
        free(p);
    }
}

struct token_list copy_argument_list(struct macro_argument* p_macro_argument)
{
    try
    {
        struct token_list list = copy_argument_list_tokens(&p_macro_argument->tokens);
        if (list.head == NULL)
        {
            /*it's never empty...if puts a TK PLACEMARKER*/
            struct token* _Owner _Opt p_new_token = calloc(1, sizeof * p_new_token);
            if (p_new_token == NULL) throw;

            char* _Owner _Opt temp = strdup("");
            if (temp == NULL)
            {
                token_delete(p_new_token);
                throw;
            }

            p_new_token->lexeme = temp;
            p_new_token->type = TK_PLACEMARKER;
            token_list_add(&list, p_new_token);
        }
        return list;
    }
    catch
    {
    }

    struct token_list empty = { 0 };
    return empty;
}


struct macro_argument_list
{
    struct token_list tokens;
    struct macro_argument* _Owner _Opt head;
    struct macro_argument* _Opt tail;
};

void macro_argument_list_destroy(_Dtor struct macro_argument_list* list)
{
    token_list_destroy(&list->tokens);
    struct macro_argument* _Owner _Opt p = list->head;
    while (p)
    {
        struct macro_argument* _Owner _Opt next = p->next;
        p->next = NULL;
        macro_argument_delete(p);
        p = next;
    }
}

void print_macro_arguments(struct macro_argument_list* arguments)
{
    struct macro_argument* _Opt p_argument = arguments->head;
    while (p_argument)
    {
        if (p_argument->macro_parameter)
            printf("%s:", p_argument->macro_parameter->name);

        print_list(&p_argument->tokens);
        p_argument = p_argument->next;
    }
}

struct macro_argument* _Opt find_macro_argument_by_name(struct macro_argument_list* parameters, const char* name)
{
    /*
     * The arguments are collected in the macro expansion and each one (except ...)
     * is associated with one of the macro parameters.
     */
    struct macro_argument* _Opt p_macro_argument = parameters->head;
    while (p_macro_argument)
    {
        if (strcmp(p_macro_argument->macro_parameter->name, name) == 0)
        {
            return p_macro_argument;
        }
        p_macro_argument = p_macro_argument->next;
    }
    return NULL;
}

void argument_list_add(struct macro_argument_list* list, struct macro_argument* _Owner pnew)
{
    assert(pnew->next == NULL);
    if (list->head == NULL)
    {
        list->head = pnew;
        assert(list->tail == NULL);
        list->tail = pnew;
    }
    else
    {
        assert(list->tail != NULL);
        assert(list->tail->next == NULL);
        list->tail->next = pnew;
        list->tail = pnew;
    }
}

void print_macro(struct macro* macro)
{
    printf("%s", macro->name);
    if (macro->is_function)
        printf("(");
    struct macro_parameter* _Opt parameter = macro->parameters;
    while (parameter)
    {
        if (macro->parameters != parameter)
            printf(",");
        printf("%s", parameter->name);
        parameter = parameter->next;
    }
    if (macro->is_function)
        printf(") ");
    print_list(&macro->replacement_list);
}

void macro_parameters_delete(struct macro_parameter* _Owner _Opt parameters)
{
    struct macro_parameter* _Owner _Opt p = parameters;
    while (p)
    {
        struct macro_parameter* _Owner _Opt p_next = p->next;
        free((void* _Owner)p->name);
        free(p);
        p = p_next;
    }
}

bool macro_is_same(const struct macro* macro_a, const struct macro* macro_b)
{
    if (macro_a->is_function != macro_b->is_function)
        return false;

    if (strcmp(macro_a->name, macro_b->name) != 0)
        return false;

    if (!token_list_is_equal(&macro_a->replacement_list, &macro_b->replacement_list) != 0)
        return false;

    const struct macro_parameter* p_a = macro_a->parameters;
    const struct macro_parameter* p_b = macro_b->parameters;
    while (p_a && p_b)
    {
        if (strcmp(p_a->name, p_b->name) != 0)
            return false;

        p_a = p_a->next;
        p_b = p_b->next;
    }

    return p_a == NULL && p_b == NULL;
}
void macro_delete(struct macro* _Owner _Opt macro)
{
    if (macro)
    {
        token_list_destroy(&macro->replacement_list);

        struct macro_parameter* _Owner _Opt p_macro_parameter = macro->parameters;
        while (p_macro_parameter)
        {
            struct macro_parameter* _Owner _Opt p_next = p_macro_parameter->next;
            free((void* _Owner)p_macro_parameter->name);
            token_list_destroy(&p_macro_parameter->expanded_list);
            free(p_macro_parameter);
            p_macro_parameter = p_next;
        }

        free((void* _Owner) macro->name);
        free(macro);
    }
}

struct macro* _Opt find_macro(struct preprocessor_ctx* ctx, const char* name)
{
    struct map_entry* _Opt p_entry = hashmap_find(&ctx->macros, name);
    if (p_entry == NULL)
        return NULL;

    return p_entry->data.p_macro;
}

void stream_print_line(struct stream* stream)
{
    const char* p = stream->current;
    while ((p - 1) >= stream->source &&
        *(p - 1) != '\n')
    {
        p--;
    }
    while (*p && *(p + 1) != '\n')
    {
        printf("%c", *p);
        p++;
    }
    printf("\n");
    for (int i = 0; i < stream->col - 1; i++)
        printf(" ");
    printf("^\n");
}

void stream_match(struct stream* stream)
{
    if (stream->current[0] == '\n')
    {
        stream->line++;
        stream->col = 1;
    }
    else
    {
        stream->col++;
    }

    if (stream->current[0] == '\0')
        return;
    stream->current++;

    while (stream->current[0] == '\\' &&
             (stream->current[1] == '\n' ||
                 (stream->current[1] == '\r' && stream->current[2] == '\n')
                 ))
    {
        /*
            2. Each instance of a backslash character (\) immediately followed by a new-line character is
            deleted, splicing physical source lines to form logical source lines. Only the last backslash on
            any physical source line shall be eligible for being part of such a splice. A source file that is
            not empty shall end in a new-line character, which shall not be immediately preceded by a
            backslash character before any such splicing takes place.
        */
        if (stream->current[1] == '\r' && stream->current[2] == '\n')
        {
            stream->current++; // 
            stream->current++; // r 
            stream->current++; // n
        }
        else
        {
            stream->current++;
            stream->current++;  // n
        }

        stream->line++;
        stream->col = 1;

        stream->line_continuation_count++;
    }

}

void print_line(struct token* p)
{
    printf("%s\n", p->token_origin->lexeme);
    struct token* _Opt prev = p;
    while (prev->prev && prev->prev->type != TK_NEWLINE)
    {
        prev = prev->prev;
    }
    struct token* _Opt next = prev;
    while (next && next->type != TK_NEWLINE)
    {
        printf("%s", next->lexeme);
        next = next->next;
    }
    printf("\n");
}

int is_nondigit(const struct stream* p)
{
    /*
    nondigit: one of
     _ a b c d e f g h i j k l m
     n o p q r s t u v w x y z
     A B C D E F G H I J K L M
     N O P Q R S T U V W X Y Z
    */
    return (p->current[0] >= 'a' && p->current[0] <= 'z') ||
        (p->current[0] >= 'A' && p->current[0] <= 'Z') ||
        (p->current[0] == '_');
}


enum token_type is_punctuator(struct stream* stream)
{
    enum token_type type = TK_NONE;
    /*
     punctuator: one of
      [ ] ( ) { } . ->
      ++ -- & * + - ~ !
      / % << >> < > <= >= == != ^ | && ||
      ? : :: ; ...
      = *= /= %= += -= <<= >>= &= ^= |=
      , # ##
      <: :> <% %> %: %:%:
    */
    switch (stream->current[0])
    {
    case '[':
        type = '[';
        stream_match(stream);
        break;
    case ']':
        type = ']';
        stream_match(stream);
        break;
    case '(':
        type = '(';
        stream_match(stream);
        break;
    case ')':
        type = ')';
        stream_match(stream);
        break;
    case '{':
        type = '{';
        stream_match(stream);
        break;
    case '}':
        type = '}';
        stream_match(stream);
        break;
    case ';':
        type = ';';
        stream_match(stream);
        break;
    case ',':
        type = ',';
        stream_match(stream);
        break;
    case '!':
        type = '!';
        stream_match(stream);
        if (stream->current[0] == '=')
        {
            type = '!=';
            stream_match(stream);
        }
        break;
    case ':':
        type = ':';
        stream_match(stream);
        if (stream->current[0] == ':')
        {
            type = '::';
            stream_match(stream);
        }
        break;
    case '~':
        type = '~';
        stream_match(stream);
        break;
    case '?':
        type = '?';
        stream_match(stream);
        break;
    case '/':
        type = '/';
        stream_match(stream);
        if (stream->current[0] == '=')
        {
            type = '/=';
            stream_match(stream);
        }
        break;
    case '*':
        type = '*';
        stream_match(stream);
        if (stream->current[0] == '=')
        {
            type = '*=';
            stream_match(stream);
        }
        break;
    case '%':
        type = '%';
        stream_match(stream);
        if (stream->current[0] == '=')
        {
            type = '%=';
            stream_match(stream);
        }
        break;
    case '-':
        type = '-';
        stream_match(stream);
        if (stream->current[0] == '>')
        {
            type = '->';
            stream_match(stream);
        }
        else if (stream->current[0] == '-')
        {
            type = '--';
            stream_match(stream);
        }
        else if (stream->current[0] == '=')
        {
            type = '-=';
            stream_match(stream);
        }
        break;
    case '|':
        type = '|';
        stream_match(stream);
        if (stream->current[0] == '|')
        {
            type = '||';
            stream_match(stream);
        }
        else if (stream->current[0] == '=')
        {
            type = '|=';
            stream_match(stream);
        }
        break;
    case '+':
        type = '+';
        stream_match(stream);
        if (stream->current[0] == '+')
        {
            type = '++';
            stream_match(stream);
        }
        else if (stream->current[0] == '=')
        {
            type = '+=';
            stream_match(stream);
        }
        break;
    case '=':
        type = '=';
        stream_match(stream);
        if (stream->current[0] == '=')
        {
            type = '==';
            stream_match(stream);
        }
        break;
    case '^':
        type = '^';
        stream_match(stream);
        if (stream->current[0] == '=')
        {
            type = '^=';
            stream_match(stream);
        }
        break;
    case '&':
        type = '&';
        stream_match(stream);
        if (stream->current[0] == '&')
        {
            type = '&&';
            stream_match(stream);
        }
        else if (stream->current[0] == '=')
        {
            type = '&=';
            stream_match(stream);
        }
        break;
    case '>':
        type = '>';
        stream_match(stream);
        if (stream->current[0] == '>')
        {
            type = '>>';
            stream_match(stream);
            if (stream->current[0] == '=')
            {
                type = '>>=';
                stream_match(stream);
            }
        }
        else if (stream->current[0] == '=')
        {
            type = '>=';
            stream_match(stream);
        }

        break;
    case '<':
        type = '<';
        stream_match(stream);
        if (stream->current[0] == '<')
        {
            type = '<<';
            stream_match(stream);
            if (stream->current[0] == '=')
            {
                type = '<<=';
                stream_match(stream);
            }
        }
        else if (stream->current[0] == '=')
        {
            type = '<=';
            stream_match(stream);
        }
        break;
    case '#':
        type = '#';
        stream_match(stream);
        if (stream->current[0] == '#')
        {
            type = '##';
            stream_match(stream);
        }
        break;
    case '.':
        type = '.';
        stream_match(stream);
        if (stream->current[0] == '.' && stream->current[1] == '.')
        {
            type = '...';
            stream_match(stream);
            stream_match(stream);
        }
        break;
    }
    return type;
}


struct token* _Owner _Opt new_token(const char* lexeme_head, const char* lexeme_tail, enum token_type type)
{
    struct token* _Owner _Opt p_new_token = NULL;

    try
    {
        p_new_token = calloc(1, sizeof * p_new_token);
        if (p_new_token == NULL) throw;

        size_t sz = lexeme_tail - lexeme_head;
        char* _Owner _Opt temp = calloc(sz + 1, sizeof(char));
        if (temp == NULL) throw;

        p_new_token->lexeme = temp;
        p_new_token->type = type;
        strncpy(p_new_token->lexeme, lexeme_head, sz);
    }
    catch
    {
        token_delete(p_new_token);
        p_new_token = NULL;
    }
    return p_new_token;
}

struct token* _Owner _Opt identifier(struct stream* stream)
{
    const char* start = stream->current;
    stream_match(stream);
    /*
    identifier:
      identifier-nondigit
      identifier identifier-nondigit
      identifier digit

    identifier-nondigit:
      nondigit
      universal-character-name
      other implementation-defined characters
    */
    while (is_nondigit(stream) || is_digit(stream))
    {
        stream_match(stream);
    }

    struct token* _Owner _Opt p_new_token = new_token(start, stream->current, TK_IDENTIFIER);


    return p_new_token;
}


static bool first_of_character_constant(struct stream* stream)
{
    return stream->current[0] == '\'' ||
        (stream->current[0] == 'u' && stream->current[1] == '8' && stream->current[2] == '\'') ||
        (stream->current[0] == 'u' && stream->current[1] == '\'') ||
        (stream->current[0] == 'U' && stream->current[1] == '\'') ||
        (stream->current[0] == 'L' && stream->current[1] == '\'');
}

struct token* _Owner _Opt character_constant(struct tokenizer_ctx* ctx, struct stream* stream)
{
    const char* start = stream->current;

    /*
      encoding-prefix: one of
       u8 u U L
    */
    if (stream->current[0] == 'u')
    {
        stream_match(stream);
        if (stream->current[0] == '8')
            stream_match(stream);
    }
    else if (stream->current[0] == 'U' ||
        stream->current[0] == 'L')
    {
        stream_match(stream);
    }


    stream_match(stream); //"


    while (stream->current[0] != '\'')
    {
        if (stream->current[0] == '\\')
        {
            stream_match(stream);
            stream_match(stream);
        }
        else
            stream_match(stream);

        if (stream->current[0] == '\0' ||
            stream->current[0] == '\n')
        {
            tokenizer_set_warning(ctx, stream, "missing terminating ' character");
            break;
        }
    }
    stream_match(stream);
    struct token* _Owner _Opt p_new_token = new_token(start, stream->current, TK_CHAR_CONSTANT);
    return p_new_token;
}

static bool first_of_string_literal(struct stream* stream)
{
    /*
    string-literal:
    encoding_prefix_opt " s-char-sequenceopt "

    encoding_prefix:
    u8
    u
    U
    L
    */

    return stream->current[0] == '"' ||
        (stream->current[0] == 'u' && stream->current[1] == '8' && stream->current[2] == '"') ||
        (stream->current[0] == 'u' && stream->current[1] == '"') ||
        (stream->current[0] == 'U' && stream->current[1] == '"') ||
        (stream->current[0] == 'L' && stream->current[1] == '"');
}

struct token* _Owner _Opt string_literal(struct tokenizer_ctx* ctx, struct stream* stream)
{
    struct token* _Owner _Opt p_new_token = NULL;
    const char* start = stream->current;

    try
    {
        /*encoding_prefix_opt*/
        if (stream->current[0] == 'u')
        {
            stream_match(stream);
            if (stream->current[0] == '8')
                stream_match(stream);
        }
        else if (stream->current[0] == 'U' ||
            stream->current[0] == 'L')
        {
            stream_match(stream);
        }


        stream_match(stream); //"


        while (stream->current[0] != '"')
        {
            if (stream->current[0] == '\0' ||
                stream->current[0] == '\n')
            {
                tokenizer_set_error(ctx, stream, "missing terminating \" character");
                throw;
            }

            if (stream->current[0] == '\\')
            {
                stream_match(stream);
                stream_match(stream);
            }
            else
                stream_match(stream);
        }
        stream_match(stream);
        p_new_token = new_token(start, stream->current, TK_STRING_LITERAL);
    }
    catch
    {
    }

    return p_new_token;
}

int get_char_type(const char* s)
{
    if (s[0] == 'L')
        return 2; /*wchar*/

    return 1;
}
/*
  Returns the char byte size according with the literal suffix
*/
int string_literal_char_byte_size(const char* s)
{
    if (s[0] == 'u')
    {
        //must be followed by u8 but not checked here
    }
    else if (s[0] == 'U' || s[0] == 'L')
    {
        return (int)sizeof(wchar_t);
    }

    return 1;
}

int string_literal_byte_size_not_zero_included(const char* s)
{

    _Opt struct stream stream = { .source = s };

    stream.current = s;
    stream.line = 1;
    stream.col = 1;
    stream.path = "";

    int size = 0;
    const int charsize = string_literal_char_byte_size(s);

    try
    {
        /*encoding_prefix_opt*/
        if (stream.current[0] == 'u')
        {
            stream_match(&stream);
            if (stream.current[0] == '8')
                stream_match(&stream);
        }
        else if (stream.current[0] == 'U' ||
            stream.current[0] == 'L')
        {
            stream_match(&stream);
        }


        stream_match(&stream); //"


        while (stream.current[0] != '"')
        {
            if (stream.current[0] == '\0' ||
                stream.current[0] == '\n')
            {
                throw;
            }

            if (stream.current[0] == '\\')
            {
                stream_match(&stream);
                stream_match(&stream);
                size++;
            }
            else
            {
                stream_match(&stream);
                size++;
            }
        }
        stream_match(&stream);
    }
    catch
    {
    }

    /*
       Last \0 is not included
    */

    return size * charsize;
}

static struct token* _Owner _Opt ppnumber(struct stream* stream)
{
    /*
     pp-number:
      digit
      . digit
      pp-number identifier-continue
      pp-number ’ digit
      pp-number ’ nondigit
      pp-number e sign
      pp-number E sign
      pp-number p sign
      pp-number P sign
      pp-number .
    */

    /*
     identifier-continue:
       digit
       nondigit
       XID_Continue character
       universal-character-name of class XID_Continue
    */

    const char* start = stream->current;
    if (is_digit(stream))
    {
        stream_match(stream);//digit
    }
    else if (stream->current[0] == '.')
    {
        stream_match(stream); //.
        stream_match(stream); //digit
    }
    else
    {
        assert(false);
    }

    for (;;)
    {
        if (stream->current[0] == '\'')
        {
            //digit separators c23
            stream_match(stream);
            if (is_digit(stream))
            {
                stream_match(stream);
            }
            else if (is_nondigit(stream))
            {
                stream_match(stream);
            }
            else
            {
                assert(false);
                break;
            }
        }
        else if ((stream->current[0] == 'e' ||
            stream->current[0] == 'E' ||
            stream->current[0] == 'p' ||
            stream->current[0] == 'P') &&
            (stream->current[1] == '+' || stream->current[1] == '-'))
        {
            stream_match(stream);//e E  p P
            stream_match(stream);//sign
        }
        else if (stream->current[0] == '.')
        {
            stream_match(stream);//.
        }
        else if (is_digit(stream) || is_nondigit(stream))
        {
            ////identifier-continue
            /*
            * OBS test for is_nondigit must be AFTER
            * test for e E p P
            */
            stream_match(stream);//nondigit
        }
        else
        {
            break;
        }
    }
    struct token* _Owner _Opt p_new_token = new_token(start, stream->current, TK_PPNUMBER);

    return p_new_token;
}

struct token_list embed_tokenizer(struct preprocessor_ctx* ctx,
    const struct token* position,
    const char* filename_opt,
    int level, enum token_flags addflags)
{
    struct token_list list = { 0 };

    FILE* _Owner _Opt file = NULL;

    bool b_first = true;
    int line = 1;
    int col = 1;
    int count = 0;
    try
    {
#ifndef MOCKFILES
        file = (FILE * _Owner _Opt)fopen(filename_opt, "rb");
        if (file == NULL)
        {
            preprocessor_diagnostic(C_ERROR_FILE_NOT_FOUND, ctx, position, "file '%s' not found", filename_opt);
            throw;
        }
#else
        /*web versions only text files that are included*/
        char* textfile = read_file(filename_opt, true);
        if (textfile == NULL)
        {
            preprocessor_diagnostic(C_ERROR_FILE_NOT_FOUND, ctx, ctx->current, "file '%s' not found", filename_opt);
            throw;
        }

        const char* pch = textfile;
#endif

        unsigned char ch = 0;
#ifndef MOCKFILES
        while (fread(&ch, 1, 1, file))
        {
#else
        while (*pch)
        {
            ch = *pch;
            pch++;
#endif
            if (b_first)
            {
                b_first = false;
            }
            else
            {
                char b[] = ",";
                struct token* _Owner _Opt p_new_token = new_token(b, &b[1], TK_COMMA);
                if (p_new_token == NULL)
                {
                    throw;
                }

                p_new_token->flags |= addflags;
                p_new_token->level = level;
                p_new_token->token_origin = NULL;
                p_new_token->line = line;
                p_new_token->col = col;
                token_list_add(&list, p_new_token);

                if (count > 0 && count % 25 == 0)
                {
                    /*new line*/
                    char newline[] = "\n";
                    struct token* _Owner _Opt p_new3 = new_token(newline, &newline[1], TK_NEWLINE);
                    if (p_new3 == NULL)
                    {
                        throw;
                    }

                    p_new3->level = level;
                    p_new3->token_origin = NULL;
                    p_new3->line = line;
                    p_new3->col = col;
                    token_list_add(&list, p_new3);
                }
            }

            char buffer[30] = { 0 };
            int c = snprintf(buffer, sizeof buffer, "%d", (int)ch);

            struct token* _Owner _Opt p_new_token = new_token(buffer, &buffer[c], TK_PPNUMBER);
            if (p_new_token == NULL)
            {
                throw;
            }

            p_new_token->flags |= addflags;
            p_new_token->level = level;
            p_new_token->token_origin = NULL;
            p_new_token->line = line;
            p_new_token->col = col;
            token_list_add(&list, p_new_token);


            count++;
        }
#ifdef MOCKFILES
        free(textfile);
#endif

        /*new line*/
        char newline[] = "\n";
        struct token* _Owner _Opt p_new_token = new_token(newline, &newline[1], TK_NEWLINE);
        if (p_new_token == NULL)
        {
            throw;
        }

        p_new_token->level = level;
        p_new_token->token_origin = NULL;
        p_new_token->line = line;
        p_new_token->col = col;
        token_list_add(&list, p_new_token);

        assert(list.head != NULL);
    }
    catch
    {
    }

    if (file)
        fclose(file);

    return list;
}

static bool set_sliced_flag(struct stream* stream, struct token* p_new_token)
{
    if (stream->line_continuation_count > 0)
    {
        p_new_token->flags |= TK_FLAG_LINE_CONTINUATION;
        if (stream->line_continuation_count == 1)
        {
            int l = strlen(p_new_token->lexeme);
            if (p_new_token->lexeme[l - 1] == '\n')
            {
                /*not sliced, line continuation is at end of token*/
            }
            else
            {
                p_new_token->flags |= TK_FLAG_SLICED;
            }
        }
        else
        {
            p_new_token->flags |= TK_FLAG_SLICED;
        }
    }

    return p_new_token->flags & TK_FLAG_SLICED;
}

struct token_list tokenizer(struct tokenizer_ctx* ctx, const char* text, const char* _Opt filename_opt, int level, enum token_flags addflags)
{
    struct token_list list = { 0 };

    struct stream stream =
    {
        .col = 1,
        .line = 1,
        .source = text,
        .current = text,
        .path = filename_opt ? filename_opt : ""
    };

    try
    {
        struct token* _Opt p_first = NULL;
        if (filename_opt != NULL)
        {
            const char* begin = filename_opt;
            const char* end = filename_opt + strlen(filename_opt);
            struct token* _Owner _Opt p_new = new_token(begin, end, TK_BEGIN_OF_FILE);
            if (p_new == NULL)
                throw;

            path_normalize(p_new->lexeme);
            p_new->level = level;
            p_first = token_list_add(&list, p_new);
        }


        //struct token* current = pFirst;
        bool new_line = true;
        bool has_space = false;
        while (1)
        {
            const int line = stream.line;
            const int col = stream.col;
            stream.line_continuation_count = 0;

            if (stream.current[0] == '\0')
            {
                stream_match(&stream);
                break;
            }
            if (is_digit(&stream) ||
                (stream.current[0] == '.' && isdigit(stream.current[0])))
            {
                struct token* _Owner _Opt p_new_token = ppnumber(&stream);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = false;
                has_space = false;
                continue;
            }

            /*
             Tem que vir antes identifier
            */
            if (first_of_string_literal(&stream))
            {
                struct token* _Owner _Opt p_new_token = string_literal(ctx, &stream);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = false;
                has_space = false;

                continue;
            }

            if (first_of_character_constant(&stream))
            {
                //TODO if we have ' in the middle then it is not character constant
                struct token* _Owner _Opt p_new_token = character_constant(ctx, &stream);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = false;
                has_space = false;

                continue;
            }

            if (is_nondigit(&stream))
            {
                struct token* _Owner _Opt p_new_token = identifier(&stream);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;

                new_line = false;
                has_space = false;
                if (set_sliced_flag(&stream, p_new_token))
                {
                    tokenizer_set_warning(ctx, &stream, "token sliced");
                }
                token_list_add(&list, p_new_token);
                continue;
            }

            if (stream.current[0] == ' ' ||
                stream.current[0] == '\t' ||
                stream.current[0] == '\f')
            {
                const char* start = stream.current;
                while (stream.current[0] == ' ' ||
                    stream.current[0] == '\t' ||
                    stream.current[0] == '\f'
                    )
                {
                    stream_match(&stream);
                }
                struct token* _Owner _Opt p_new_token = new_token(start, stream.current, TK_BLANKS);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                /*bNewLine = false;*/ //deixa assim
                has_space = true;

                continue;
            }
            if (stream.current[0] == '/' &&
                stream.current[1] == '/')
            {
                const char* start = stream.current;
                stream_match(&stream);
                stream_match(&stream);
                //line comment
                while (stream.current[0] != '\n')
                {
                    stream_match(&stream);

                    if (stream.current[0] == '\0')
                        break;
                }
                struct token* _Owner _Opt p_new_token = new_token(start, stream.current, TK_LINE_COMMENT);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = true;
                has_space = false;


                if (stream.current[0] == '\0')
                    break;

                continue;
            }
            if (stream.current[0] == '/' &&
                stream.current[1] == '*')
            {
                const char* start = stream.current;
                stream_match(&stream);
                stream_match(&stream);
                //line comment
                for (;;)
                {
                    if (stream.current[0] == '*' && stream.current[1] == '/')
                    {
                        stream_match(&stream);
                        stream_match(&stream);
                        break;
                    }
                    else if (stream.current[0] == '\0')
                    {
                        tokenizer_set_error(ctx, &stream, "missing end of comment");
                        break;
                    }
                    else
                    {
                        stream_match(&stream);
                    }
                }
                struct token* _Owner _Opt p_new_token = new_token(start, stream.current, TK_COMMENT);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                token_list_add(&list, p_new_token);
                new_line = false;
                has_space = false;

                /*
                * Ignore line splicing inside comments.
                * if you are curious to see when it happens just add
                * set_sliced_flag
                */

                continue;
            }
            if (new_line && stream.current[0] == '#')
            {
                const char* start = stream.current;
                stream_match(&stream);
                struct token* _Owner _Opt p_new_token = new_token(start, stream.current, '#');
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                p_new_token->type = TK_PREPROCESSOR_LINE;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = false;
                has_space = false;

                continue;
            }


            if (stream.current[0] == '\n' || stream.current[0] == '\r')
            {
                if (stream.current[0] == '\r' && stream.current[1] == '\n')
                {
                    stream_match(&stream);
                    stream_match(&stream);
                }
                else
                {
                    stream_match(&stream);
                }
                char  newline[] = "\n";
                struct token* _Owner _Opt p_new_token = new_token(newline, newline + 1, TK_NEWLINE);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = true;
                has_space = false;

                continue;
            }
            const char* start = stream.current;
            enum token_type t = is_punctuator(&stream);
            if (t != TK_NONE)
            {

                struct token* _Owner _Opt p_new_token = new_token(start, stream.current, t);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = false;
                has_space = false;

                continue;
            }
            else
            {
                stream_match(&stream);
                struct token* _Owner _Opt p_new_token = new_token(start, stream.current, ANY_OTHER_PP_TOKEN);
                if (p_new_token == NULL) throw;

                p_new_token->flags |= has_space ? TK_FLAG_HAS_SPACE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= new_line ? TK_FLAG_HAS_NEWLINE_BEFORE : TK_FLAG_NONE;
                p_new_token->flags |= addflags;

                p_new_token->level = level;
                p_new_token->token_origin = p_first;
                p_new_token->line = line;
                p_new_token->col = col;
                set_sliced_flag(&stream, p_new_token);
                token_list_add(&list, p_new_token);
                new_line = false;
                has_space = false;

                continue;
            }

            break;
        }
    }
    catch
    {
    }

    assert(list.head != NULL);
    return list;
}


bool fread2(void* buffer, size_t size, size_t count, FILE * stream, size_t * sz)
{
    *sz = 0;//out
    bool result = false;
    size_t n = fread(buffer, size, count, stream);
    if (n == count)
    {
        *sz = n;
        result = true;
    }
    else if (n < count)
    {
        if (feof(stream))
        {
            *sz = n;
            result = true;
        }
    }
    return result;
}


bool preprocessor_token_ahead_is_identifier(struct token* p, const char* lexeme);
struct token_list group_part(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level);
struct token_list group_opt(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    /*
      group:
       group-part
       group group-part
    */
    struct token_list r = { 0 };
    try
    {

        if (token_list_is_empty(input_list))
        {
            return r;
        }
        while (!token_list_is_empty(input_list))
        {
            assert(input_list->head != NULL);

            if (input_list->head->type == TK_PREPROCESSOR_LINE &&
                (preprocessor_token_ahead_is_identifier(input_list->head, "endif") ||
                    preprocessor_token_ahead_is_identifier(input_list->head, "else") ||
                    preprocessor_token_ahead_is_identifier(input_list->head, "elif") ||
                    preprocessor_token_ahead_is_identifier(input_list->head, "elifdef") ||
                    preprocessor_token_ahead_is_identifier(input_list->head, "elifndef")))
            {
                /*follow of group-part*/
                break;
            }
            else
            {
                struct token_list r2 = group_part(ctx, input_list, is_active, level);
                token_list_append_list(&r, &r2);
                token_list_destroy(&r2);
                if (ctx->n_errors > 0) throw;
            }
        }
    }
    catch
    {
    }

    return r;
}

bool is_parser_token(struct token* p)
{
    return p->type != TK_COMMENT &&
        p->type != TK_BLANKS &&
        p->type != TK_LINE_COMMENT &&
        p->type != TK_NEWLINE;
}

bool is_never_final(enum token_type type)
{
    return type == TK_BEGIN_OF_FILE ||
        type == TK_BLANKS ||
        type == TK_LINE_COMMENT ||
        type == TK_COMMENT ||
        type == TK_PLACEMARKER ||
        type == TK_NEWLINE;
}

enum token_type is_keyword(const char* text);

struct token* _Opt preprocessor_look_ahead_core(struct token* p)
{
    struct token* _Opt current = p->next;

    while (current &&
        (current->type == TK_BLANKS ||
            current->type == TK_PLACEMARKER ||
            current->type == TK_LINE_COMMENT ||
            current->type == TK_COMMENT))
    {
        current = current->next;
    }
    return current;
}

bool preprocessor_token_ahead_is(struct token* p, enum token_type t)
{
    struct token* _Opt p_token = preprocessor_look_ahead_core(p);
    if (p_token != NULL && p_token->type == t)
        return true;
    return false;
}

bool preprocessor_token_ahead_is_identifier(struct token* p, const char* lexeme)
{
    assert(p != NULL);
    struct token* _Opt p_token = preprocessor_look_ahead_core(p);
    if (p_token != NULL && p_token->type == TK_IDENTIFIER)
    {
        return strcmp(p_token->lexeme, lexeme) == 0;
    }
    return false;
}

static void skip_blanks_level(struct preprocessor_ctx* ctx, struct token_list* dest, struct token_list* input_list, int level)
{
    while (input_list->head)
    {
        if (!token_is_blank(input_list->head))
            break;

        if (INCLUDE_ALL || level == 0)
        {
            struct token* _Owner _Opt p =
                token_list_pop_front_get(input_list);
            assert(p != NULL); //because input_list is not empty
            token_list_add(dest, p);
        }
        else
            token_list_pop_front(input_list);
    }
}

static void skip_blanks(struct preprocessor_ctx* ctx, struct token_list* dest, struct token_list* input_list)
{
    while (input_list->head)
    {
        if (!token_is_blank(input_list->head))
            break;
        struct token* _Owner _Opt p =
            token_list_pop_front_get(input_list);
        assert(p != NULL); //because input_list is not empty

        token_list_add(dest, p);
    }
}

void prematch_level(struct token_list* dest, struct token_list* input_list, int level)
{
    if (INCLUDE_ALL || level == 0)
    {
        struct token* _Owner _Opt p = token_list_pop_front_get(input_list);
        if (p)
        {
            token_list_add(dest, p);
        }
    }
    else
        token_list_pop_front(input_list);
}

void prematch(struct token_list* dest, struct token_list* input_list)
{
    struct token* _Owner _Opt p = token_list_pop_front_get(input_list);
    if (p)
    {
        token_list_add(dest, p);
    }
}

struct token_list pp_tokens_opt(struct preprocessor_ctx* ctx, struct token_list* input_list, int level);

struct token_list process_defined(struct preprocessor_ctx* ctx, struct token_list* input_list)
{
    struct token_list r = { 0 };

    try
    {
        while (input_list->head != NULL)
        {
            if (input_list->head->type == TK_IDENTIFIER &&
                strcmp(input_list->head->lexeme, "defined") == 0)
            {
                token_list_pop_front(input_list);
                skip_blanks(ctx, &r, input_list);

                if (input_list->head == NULL)
                {
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                bool has_parentesis = false;
                if (input_list->head->type == '(')
                {
                    token_list_pop_front(input_list);
                    has_parentesis = true;
                }

                skip_blanks(ctx, &r, input_list);


                if (input_list->head == NULL)
                {
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                struct macro* _Opt macro = find_macro(ctx, input_list->head->lexeme);
                struct token* _Owner _Opt p_new_token = token_list_pop_front_get(input_list);
                if (p_new_token == NULL)
                {
                    throw;
                }

                p_new_token->type = TK_PPNUMBER;
                char* _Owner _Opt temp = NULL;

                if (macro)
                {
                    temp = strdup("1");
                }
                else
                {
                    temp = strdup("0");
                }

                if (temp == NULL)
                {
                    token_delete(p_new_token);
                    throw;
                }

                free(p_new_token->lexeme);
                p_new_token->lexeme = temp;

                token_list_add(&r, p_new_token);

                if (has_parentesis)
                {
                    if (input_list->head == NULL)
                    {
                        throw;
                    }

                    if (input_list->head->type != ')')
                    {
                        preprocessor_diagnostic(C_ERROR_MISSING_CLOSE_PARENTHESIS, ctx, input_list->head, "missing )");
                        throw;
                    }
                    token_list_pop_front(input_list);
                }


            }
            else if (input_list->head->type == TK_IDENTIFIER &&
                (strcmp(input_list->head->lexeme, "__has_include") == 0 ||
                    strcmp(input_list->head->lexeme, "__has_embed") == 0)
                )
            {
                token_list_pop_front(input_list); //pop __has_include
                skip_blanks(ctx, &r, input_list);
                token_list_pop_front(input_list); //pop (
                skip_blanks(ctx, &r, input_list);


                char path[100] = { 0 };
                bool is_angle_bracket_form = false;

                if (input_list->head == NULL)
                {
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                if (input_list->head->type == TK_STRING_LITERAL)
                {
                    strcat(path, input_list->head->lexeme);
                    token_list_pop_front(input_list); //pop "file"
                }
                else
                {
                    is_angle_bracket_form = true;
                    token_list_pop_front(input_list); //pop <

                    if (input_list->head == NULL)
                    {
                        pre_unexpected_end_of_file(r.tail, ctx);
                        throw;
                    }

                    while (input_list->head->type != '>')
                    {
                        strcat(path, input_list->head->lexeme);
                        token_list_pop_front(input_list); //pop (

                        if (input_list->head == NULL)
                        {
                            pre_unexpected_end_of_file(r.tail, ctx);
                            throw;
                        }
                    }
                    token_list_pop_front(input_list); //pop >
                }

                char fullpath[300] = { 0 };



                char full_path_result[200] = { 0 };
                bool already_included = false;
                const char* _Owner _Opt s = find_and_read_include_file(ctx,
                    path,
                    fullpath,
                    is_angle_bracket_form,
                    &already_included,
                    full_path_result,
                    sizeof full_path_result,
                  false);

                bool has_include = s != NULL;
                free((void* _Owner)s);

                struct token* _Owner _Opt p_new_token = calloc(1, sizeof * p_new_token);
                if (p_new_token == NULL)
                {
                    throw;
                }

                p_new_token->type = TK_PPNUMBER;

                char* _Owner _Opt temp = strdup(has_include ? "1" : "0");
                if (temp == NULL)
                {
                    token_delete(p_new_token);
                    throw;
                }
                p_new_token->lexeme = temp;
                p_new_token->flags |= TK_FLAG_FINAL;

                token_list_add(&r, p_new_token);
                token_list_pop_front(input_list); //pop )
            }
            else if (input_list->head->type == TK_IDENTIFIER &&
                strcmp(input_list->head->lexeme, "__has_c_attribute") == 0)
            {
                token_list_pop_front(input_list); //pop __has_include
                skip_blanks(ctx, &r, input_list);
                token_list_pop_front(input_list); //pop (
                skip_blanks(ctx, &r, input_list);

                if (input_list->head == NULL)
                {
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                char path[100] = { 0 };
                while (input_list->head->type != ')')
                {
                    strcat(path, input_list->head->lexeme);
                    token_list_pop_front(input_list); //pop (

                    if (input_list->head == NULL)
                    {
                        pre_unexpected_end_of_file(r.tail, ctx);
                        throw;
                    }
                }
                token_list_pop_front(input_list); //pop >

                /*nodiscard
                * The __has_c_attribute conditional inclusion expression (6.10.1) shall
                * return the value 202003L
                * when given nodiscard as the pp-tokens operand.
                */

                /*maybe_unused
                * The __has_c_attribute conditional inclusion expression (6.10.1) shall return
                * the value 202106L when given maybe_unused as the pp-tokens operand.
                */

                /*deprecated
                * The __has_c_attribute conditional inclusion expression (6.10.1) shall return the value 201904L
                * when given deprecated as the pp-tokens operand
                */

                /*noreturn
                * The __has_c_attribute conditional inclusion expression (6.10.1) shall return the value 202202L
                * when given noreturn as the pp-tokens operand.
                */

                /*reproducible
                 * The __has_c_attribute conditional inclusion expression (6.10.1) shall return the value 202207L
                 * when given reproducible as the pp-tokens operand.
                */

                /*
                * The __has_c_attribute conditional inclusion expression (6.10.1) shall return the value 202207L
                * when given unsequenced as the pp-tokens operand.
                */
                bool has_c_attribute = false;

                struct token* _Owner _Opt p_new_token = calloc(1, sizeof * p_new_token);
                if (p_new_token == NULL)
                {
                    throw;
                }

                p_new_token->type = TK_PPNUMBER;
                char* _Owner _Opt temp = strdup(has_c_attribute ? "1" : "0");
                if (temp == NULL)
                {
                    token_delete(p_new_token);
                    throw;
                }

                p_new_token->lexeme = temp;
                p_new_token->flags |= TK_FLAG_FINAL;

                token_list_add(&r, p_new_token);
                token_list_pop_front(input_list); //pop )
            }
            else
            {
                struct token* _Owner _Opt tk = token_list_pop_front_get(input_list);
                if (tk)
                    token_list_add(&r, tk);
            }
        }
    }
    catch
    {
        //TODO clear?
    }

    return r;
}

struct token_list process_identifiers(struct preprocessor_ctx* ctx, _Dtor struct token_list* list)
{
    assert(!token_list_is_empty(list));

    struct token_list list2 = { 0 };

    try
    {
        while (list->head != NULL)
        {
            if (list->head->type == TK_IDENTIFIER)
            {
                struct macro* _Opt macro = find_macro(ctx, list->head->lexeme);
                struct token* _Owner _Opt p_new_token = token_list_pop_front_get(list);
                assert(p_new_token != NULL); //because the list is not empty

                p_new_token->type = TK_PPNUMBER;

                if (macro)
                {
                    char* _Owner _Opt temp = strdup("1");
                    if (temp == NULL)
                    {
                        token_delete(p_new_token);
                        throw;
                    }

                    free(p_new_token->lexeme);
                    p_new_token->lexeme = temp;
                }
                else
                {
                    /*
                    * after all replacements due to macro expansion and
                      evaluations of defined macro expressions, has_include expressions, and has_c_attribute expressions
                      have been performed, all remaining identifiers other than true (including those lexically identical
                      to keywords such as false) are replaced with the pp-number 0, true is replaced with pp-number
                      1, and then each preprocessing token is converted into a token.
                    */
                    if (strcmp(p_new_token->lexeme, "true") == 0)
                    {
                        p_new_token->lexeme[0] = '1';
                        p_new_token->lexeme[1] = '\0';
                    }
                    else if (strcmp(p_new_token->lexeme, "false") == 0)
                    {
                        p_new_token->lexeme[0] = '0';
                        p_new_token->lexeme[1] = '\0';
                    }
                    else
                    {
                        char* _Owner _Opt temp = strdup("0");
                        if (temp == NULL)
                        {
                            token_delete(p_new_token);
                            throw;
                        }

                        free(p_new_token->lexeme);
                        p_new_token->lexeme = temp;
                    }
                }
                token_list_add(&list2, p_new_token);
            }
            else
            {
                struct token* _Owner _Opt ptk = token_list_pop_front_get(list);
                assert(ptk != NULL); //because the list is not empty
                token_list_add(&list2, ptk);
            }
        }
        assert(!token_list_is_empty(&list2));
    }
    catch
    {
    }
    return list2;
}

struct token_list ignore_preprocessor_line(struct token_list* input_list)
{
    struct token_list r = { 0 };
    while (input_list->head && input_list->head->type != TK_NEWLINE)
    {
        struct token* _Owner _Opt tk = token_list_pop_front_get(input_list);
        assert(tk != NULL); //because the list is not empty
        token_list_add(&r, tk);
    }
    return r;
}

//todo passar lista para reotnro
long long preprocessor_constant_expression(struct preprocessor_ctx* ctx,
    struct token_list* output_list,
    struct token_list* input_list,
    int level
)
{
    assert(input_list->head != NULL);

    struct token* first = input_list->head;

    ctx->conditional_inclusion = true;

    struct token_list r = { 0 };
    while (input_list->head && input_list->head->type != TK_NEWLINE)
    {
        struct token* _Owner _Opt tk = token_list_pop_front_get(input_list);
        assert(tk != NULL); //because the list is not empty
        token_list_add(&r, tk);

        /*
          We call preprocessor that emmit warnings if line continuation
          is used outside macro directives.
          Let's remove TK_FLAG_LINE_CONTINUATION from the original token
          to avoid warning inside constant expressions
        */
        assert(r.tail != NULL);
        r.tail->flags &= ~TK_FLAG_LINE_CONTINUATION;
    }

    struct token_list list1 = copy_replacement_list(ctx, &r);
    token_list_swap(output_list, &r);


    int flags = ctx->flags;
    ctx->flags |= PREPROCESSOR_CTX_FLAGS_ONLY_FINAL;

    /*defined X  por exemplo é mantido sem ser expandido*/

    struct token_list list2 = preprocessor(ctx, &list1, 1);
    ctx->flags = flags;

    long long value = 0;

    if (list2.head == NULL)
    {
        preprocessor_diagnostic(C_ERROR_EXPRESSION_ERROR, ctx, first, "empty expression");
    }
    else
    {
        /*aonde defined has_c_aatribute sao transformados em constantes*/
        struct token_list list3 = process_defined(ctx, &list2);

        struct token_list list4 = process_identifiers(ctx, &list3);

        assert(list4.head != NULL);

        struct preprocessor_ctx pre_ctx = { 0 };


        pre_ctx.input_list = list4;
        pre_ctx.current = pre_ctx.input_list.head;

        if (pre_constant_expression(&pre_ctx, &value) != 0)
        {
            preprocessor_diagnostic(C_ERROR_EXPRESSION_ERROR, ctx, first, "expression error");
        }

        ctx->conditional_inclusion = false;

        preprocessor_ctx_destroy(&pre_ctx);
    }

    token_list_destroy(&list1);
    token_list_destroy(&r);
    token_list_destroy(&list2);

    return value;
}

void match_level(struct token_list* dest, struct token_list* input_list, int level)
{
    if (INCLUDE_ALL || level == 0)
    {
        struct token* _Owner _Opt tk = token_list_pop_front_get(input_list);
        if (tk)
        {
            token_list_add(dest, tk);
        }
    }
    else
        token_list_pop_front(input_list);
}


int match_token_level(struct token_list* dest, struct token_list* input_list, enum token_type type, int level,
    struct preprocessor_ctx* ctx)
{
    try
    {
        if (input_list->head == NULL ||
            input_list->head->type != type)
        {
            if (type == TK_NEWLINE && input_list->head == NULL)
            {
                //vou aceitar final de arquivo como substituro do endline
                //exemplo #endif sem quebra de linha
            }
            else
            {
                if (input_list->head)
                    preprocessor_diagnostic(C_ERROR_UNEXPECTED_TOKEN, ctx, input_list->head, "expected token %s got %s\n", get_token_name(type), get_token_name(input_list->head->type));
                else
                    preprocessor_diagnostic(C_ERROR_UNEXPECTED_TOKEN, ctx, dest->tail, "expected EOF \n");

                throw;
            }
        }
        if (input_list->head != NULL)
        {
            if (INCLUDE_ALL || level == 0)
                token_list_add(dest, token_list_pop_front_get(input_list));
            else
                token_list_pop_front(input_list);
        }
    }
    catch
    {
    }
    return ctx->n_errors > 0;
}


struct token_list if_group(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level, bool* p_result)
{
    *p_result = 0; //out

    assert(input_list->head != NULL);

    struct token_list r = { 0 };
    try
    {
        /*
         if-group:
           # if constant-expression new-line group_opt
           # ifdef identifier new-line group_opt
           # ifndef identifier new-line group_opt
        */
        match_token_level(&r, input_list, TK_PREPROCESSOR_LINE, level, ctx);
        skip_blanks_level(ctx, &r, input_list, level);

        if (input_list->head == NULL)
            throw;

        assert(input_list->head->type == TK_IDENTIFIER);
        if (strcmp(input_list->head->lexeme, "ifdef") == 0)
        {
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //ifdef
            skip_blanks_level(ctx, &r, input_list, level);

            if (input_list->head == NULL)
            {
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            if (is_active)
            {
                struct macro* _Opt macro = find_macro(ctx, input_list->head->lexeme);
                *p_result = (macro != NULL) ? 1 : 0;
                //printf("#ifdef %s (%s)\n", input_list->head->lexeme, *p_result ? "true" : "false");
            }
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
            skip_blanks_level(ctx, &r, input_list, level);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
        }
        else if (strcmp(input_list->head->lexeme, "ifndef") == 0)
        {
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //ifndef
            skip_blanks_level(ctx, &r, input_list, level);

            if (input_list->head == NULL)
            {
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            if (is_active)
            {
                struct macro* _Opt macro = find_macro(ctx, input_list->head->lexeme);
                *p_result = (macro == NULL) ? 1 : 0;
            }
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
            skip_blanks_level(ctx, &r, input_list, level);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
        }
        else if (strcmp(input_list->head->lexeme, "if") == 0)
        {
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //if
            skip_blanks_level(ctx, &r, input_list, level);
            if (is_active)
            {
                struct token_list r0 = { 0 };
                *p_result = preprocessor_constant_expression(ctx, &r0, input_list, level);
                token_list_append_list(&r, &r0);
                token_list_destroy(&r0);
            }
            else
            {
                struct token_list r0 = ignore_preprocessor_line(input_list);
                token_list_append_list(&r, &r0);
                token_list_destroy(&r0);
            }
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
        }
        else
        {

            preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, input_list->head, "unexpected");
            throw;
        }
        struct token_list r2 = group_opt(ctx, input_list, is_active && *p_result, level);
        token_list_append_list(&r, &r2);
        assert(r2.head == NULL);
        assert(r2.tail == NULL);
    }
    catch
    {
    }

    return r;
}

struct token_list elif_group(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level, bool* p_elif_result)
{
    *p_elif_result = 0; //out
    assert(input_list->head != NULL);

    struct token_list r = { 0 };

    try
    {
        /*
         elif-group:
          # elif constant-expression new-line group_opt

          C23
          # elifdef identifier new-line group_opt
          # elifndef identifier new-line group_opt
        */
        match_token_level(&r, input_list, TK_PREPROCESSOR_LINE, level, ctx);
        skip_blanks(ctx, &r, input_list);

        if (input_list->head == NULL)
            throw;

        unsigned long long result = 0;
        if (strcmp(input_list->head->lexeme, "elif") == 0)
        {
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
            skip_blanks(ctx, &r, input_list);

            if (is_active)
            {
                struct token_list r0 = { 0 };
                result = preprocessor_constant_expression(ctx, &r0, input_list, level);
                token_list_append_list(&r, &r0);
                token_list_destroy(&r0);
            }
            else
            {
                struct token_list r0 = ignore_preprocessor_line(input_list);
                token_list_append_list(&r, &r0);
                token_list_destroy(&r0);
            }
        }
        else if (strcmp(input_list->head->lexeme, "elifdef") == 0)
        {
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
            skip_blanks(ctx, &r, input_list);

            if (input_list->head == NULL)
                throw;

            if (is_active)
            {
                result = (hashmap_find(&ctx->macros, input_list->head->lexeme) != NULL) ? 1 : 0;
            }
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
        }
        else if (strcmp(input_list->head->lexeme, "elifndef") == 0)
        {
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
            skip_blanks(ctx, &r, input_list);

            if (input_list->head == NULL)
                throw;

            if (is_active)
            {
                result = (hashmap_find(&ctx->macros, input_list->head->lexeme) == NULL) ? 1 : 0;
            }
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
        }
        *p_elif_result = (result != 0);
        skip_blanks(ctx, &r, input_list);
        match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
        struct token_list r2 = group_opt(ctx, input_list, is_active && *p_elif_result, level);
        token_list_append_list(&r, &r2);
        token_list_destroy(&r2);
    }
    catch
    {
    }

    return r;
}

struct token_list elif_groups(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level, bool* pelif_result)
{
    assert(input_list->head != NULL);

    struct token_list r = { 0 };
    /*
    elif-groups:
      elif-group
      elif-groups elif-group
    */
    try
    {
        bool already_found_elif_true = false;
        bool elif_result = false;
        struct token_list r2 = elif_group(ctx, input_list, is_active, level, &elif_result);

        if (input_list->head == NULL)
        {
            token_list_destroy(&r2);
            throw;
        }

        token_list_append_list(&r, &r2);

        if (elif_result)
            already_found_elif_true = true;

        if (input_list->head->type == TK_PREPROCESSOR_LINE &&
            (
                preprocessor_token_ahead_is_identifier(input_list->head, "elif") ||
                preprocessor_token_ahead_is_identifier(input_list->head, "elifdef") ||
                preprocessor_token_ahead_is_identifier(input_list->head, "elifndef")
                )
            )
        {
            /*
              Depois que acha 1 true already_found_elif_true os outros sao false.
            */
            struct token_list r3 = elif_groups(ctx, input_list, is_active && !already_found_elif_true, level, &elif_result);
            token_list_append_list(&r, &r3);
            if (elif_result)
                already_found_elif_true = true;
            token_list_destroy(&r3);
        }
        /*
           Se algum dos elifs foi true retorna true
        */
        *pelif_result = already_found_elif_true;
        token_list_destroy(&r2);
    }
    catch
    {
        //TODO clear
    }

    return r;
}

struct token_list else_group(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    /*
      else-group:
       # else new-line group_opt
    */

    struct token_list r = { 0 };
    try
    {
        match_token_level(&r, input_list, TK_PREPROCESSOR_LINE, level, ctx);
        skip_blanks_level(ctx, &r, input_list, level);
        if (ctx->n_errors > 0) throw;

        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //else
        skip_blanks_level(ctx, &r, input_list, level);
        match_token_level(&r, input_list, TK_NEWLINE, level, ctx);

        struct token_list r2 = group_opt(ctx, input_list, is_active, level);
        token_list_append_list(&r, &r2);

        token_list_destroy(&r2);
    }
    catch
    {
        //tODO
    }

    return r;
}

struct token_list endif_line(struct preprocessor_ctx* ctx, struct token_list* input_list, int level)
{
    /*
     endif-line:
       # endif new-line
    */

    struct token_list r = { 0 };

    match_token_level(&r, input_list, TK_PREPROCESSOR_LINE, level, ctx); //#
    skip_blanks_level(ctx, &r, input_list, level);
    match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //endif
    skip_blanks_level(ctx, &r, input_list, level);
    match_token_level(&r, input_list, TK_NEWLINE, level, ctx);

    return r;
}
struct token_list identifier_list(struct preprocessor_ctx* ctx, struct macro* macro, struct token_list* input_list, int level);
struct token_list replacement_list(struct preprocessor_ctx* ctx, struct macro* macro, struct token_list* input_list, int level);
static bool is_empty_assert(struct token_list* replacement_list);

struct token_list def_line(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level, struct macro** pp_macro)
{
    //https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3524.txt

    /*
    def-line:
       # def identifier new-line
       # def identifier lparen identifier-list(opt) ) new-line
       # def identifier lparen ... ) new-line
       # def identifier lparen identifier-list , ... ) new-line
    */
    struct token_list r = { 0 };

    try
    {
        /*
          This code is the same of define...TODO share
        */
        struct macro* _Owner _Opt macro = calloc(1, sizeof * macro);
        if (macro == NULL)
        {
            preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, "out of mem");
            throw;
        }

        macro->def_macro = true;

        match_token_level(&r, input_list, TK_PREPROCESSOR_LINE, level, ctx); //#

        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //def
        skip_blanks_level(ctx, &r, input_list, level);

        if (input_list->head == NULL)
        {
            macro_delete(macro);
            pre_unexpected_end_of_file(r.tail, ctx);
            throw;
        }

        struct token* macro_name_token = input_list->head;

        if (is_builtin_macro(macro_name_token->lexeme))
        {
            preprocessor_diagnostic(W_REDEFINING_BUITIN_MACRO,
                ctx,
                input_list->head,
                "redefining builtin macro");
        }

        if (hashmap_find(&ctx->macros, input_list->head->lexeme) != NULL)
        {
            //printf("warning: '%s' macro redefined at %s %d\n",
              //     input_list->head->lexeme,
                ///   input_list->head->token_origin->lexeme,
                  // input_list->head->line);
        }

        char* _Owner _Opt temp = strdup(input_list->head->lexeme);
        if (temp == NULL)
        {
            macro_delete(macro);
            throw;
        }
        assert(macro->name == NULL);
        macro->name = temp;


        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //nome da macro

        if (input_list->head == NULL)
        {
            macro_delete(macro);
            pre_unexpected_end_of_file(r.tail, ctx);
            throw;
        }

        /*sem skip*/
        //p = preprocessor_match_token(p, is_active, level, false, IDENTIFIER); /*name*/
        if (input_list->head->type == '(')
        {
            macro->is_function = true;

            match_token_level(&r, input_list, '(', level, ctx);
            skip_blanks_level(ctx, &r, input_list, level);

            if (input_list->head == NULL)
            {
                macro_delete(macro);
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            if (input_list->head->type == '...')
            {
                struct macro_parameter* _Owner _Opt p_macro_parameter = calloc(1, sizeof * p_macro_parameter);
                if (p_macro_parameter == NULL)
                {
                    macro_delete(macro);
                    throw;
                }

                char* _Owner _Opt temp2 = strdup("__VA_ARGS__");
                if (temp2 == NULL)
                {
                    macro_delete(macro);
                    macro_parameters_delete(p_macro_parameter);
                    throw;
                }

                p_macro_parameter->name = temp2;
                macro->parameters = p_macro_parameter;

                // assert(false);
                match_token_level(&r, input_list, '...', level, ctx); //nome da macro
                skip_blanks_level(ctx, &r, input_list, level);
                match_token_level(&r, input_list, ')', level, ctx); //nome da macro
            }
            else if (input_list->head->type == ')')
            {
                match_token_level(&r, input_list, ')', level, ctx);
                skip_blanks_level(ctx, &r, input_list, level);
            }
            else
            {
                struct token_list r3 = identifier_list(ctx, macro, input_list, level);
                token_list_append_list(&r, &r3);
                token_list_destroy(&r3);

                skip_blanks_level(ctx, &r, input_list, level);
                if (input_list->head == NULL)
                {
                    macro_delete(macro);
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                if (input_list->head->type == '...')
                {
                    struct macro_parameter* _Owner _Opt p_macro_parameter = calloc(1, sizeof * p_macro_parameter);
                    if (p_macro_parameter == NULL)
                    {
                        macro_delete(macro);
                        throw;
                    }

                    char* _Owner _Opt temp3 = strdup("__VA_ARGS__");
                    if (temp3 == NULL)
                    {
                        macro_delete(macro);
                        macro_parameters_delete(p_macro_parameter);
                        throw;
                    }

                    p_macro_parameter->name = temp3;
                    struct macro_parameter* _Opt p_last = macro->parameters;
                    assert(p_last != NULL);
                    while (p_last->next)
                    {
                        p_last = p_last->next;
                    }
                    p_last->next = p_macro_parameter;

                    match_token_level(&r, input_list, '...', level, ctx);
                }
                skip_blanks_level(ctx, &r, input_list, level);
                match_token_level(&r, input_list, ')', level, ctx);
            }
        }
        else
        {
            macro->is_function = false;
        }
        skip_blanks_level(ctx, &r, input_list, level);
        if (input_list->head == NULL)
        {
            macro_delete(macro);
            pre_unexpected_end_of_file(r.tail, ctx);
            throw;
        }

        naming_convention_macro(ctx, macro_name_token);

        struct hash_item_set item = { 0 };
        item.p_macro = macro;
        hashmap_set(&ctx->macros, macro->name, &item);
        hash_item_set_destroy(&item);
        *pp_macro = macro;
    }
    catch
    {
    }

    return r;
}
struct token_list replacement_group(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    //https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3524.txt

    /*replacement-group:
        pp-tokens(opt) new-line
        replacement-group pp-tokens(opt) new-line
    */
    struct token_list r = { 0 };
    try
    {
        for (;;)
        {
            if (input_list->head == NULL)
            {
                preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, r.tail, "missing #enddef");
                throw;
            }

            if (input_list->head->type == TK_PREPROCESSOR_LINE && (
                preprocessor_token_ahead_is_identifier(input_list->head, "enddef")))
            {
                break;
            }
            prematch_level(&r, input_list, level);
        }
    }
    catch
    {
    }
    return r;
}


struct token_list enddef_line(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    //https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3524.txt
    /*
      enddef-line:
        # enddef new-line
    */

    struct token_list r = { 0 };
    try
    {
        if (input_list->head == NULL)
        {
            pre_unexpected_end_of_file(r.tail, ctx);
            throw;
        }

        match_token_level(&r, input_list, TK_PREPROCESSOR_LINE, level, ctx); //#
        skip_blanks_level(ctx, &r, input_list, level);
        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //enddef
        skip_blanks_level(ctx, &r, input_list, level);
        match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
    }
    catch
    {
    }
    return r;
}

struct token_list def_section(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    /*
     def-section:
       def-line replacement-group(opt) enddef-line
    */
    struct token_list r = { 0 };
    try
    {
        struct macro* p_macro = NULL;
        struct token_list r2 = def_line(ctx, input_list, is_active, level, &p_macro);
        token_list_append_list(&r, &r2);

        if (ctx->n_errors > 0 || p_macro == NULL)
        {
            token_list_destroy(&r2);
            throw;
        }

        struct token_list r3 = replacement_group(ctx, input_list, is_active, level);

        if (ctx->n_errors > 0)
        {

            token_list_destroy(&r2);
            token_list_destroy(&r3);
            throw;
        }

        struct token_list copy = copy_replacement_list(ctx, &r3);
        token_list_append_list(&p_macro->replacement_list, &copy);

        token_list_append_list(&r, &r3);
        struct token_list r4 = enddef_line(ctx, input_list, is_active, level);
        token_list_append_list(&r, &r4);

        token_list_destroy(&r2);
        token_list_destroy(&r3);
        token_list_destroy(&r4);
        token_list_destroy(&copy);
    }
    catch
    {
    }


    return r;
}

struct token_list if_section(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    /*
     if-section:
       if-group elif-groups_opt else-group_opt endif-line
    */
    assert(input_list->head != NULL);

    struct token_list r = { 0 };

    try
    {
        bool if_result = false;
        struct token_list r2 = if_group(ctx, input_list, is_active, level, &if_result);
        if (ctx->n_errors > 0)
        {
            token_list_destroy(&r2);
            throw;
        }

        if (input_list->head == NULL)
        {
            token_list_destroy(&r2);
            throw;
        }

        token_list_append_list(&r, &r2);
        bool elif_result = false;
        if (input_list->head->type == TK_PREPROCESSOR_LINE && (
            preprocessor_token_ahead_is_identifier(input_list->head, "elif") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "elifdef") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "elifndef")))
        {
            struct token_list r3 = elif_groups(ctx, input_list, is_active && !if_result, level, &elif_result);
            token_list_append_list(&r, &r3);
            token_list_destroy(&r3);
        }

        if (input_list->head == NULL)
        {
            token_list_destroy(&r2);
            pre_unexpected_end_of_file(r.tail, ctx);
            throw;
        }


        if (input_list->head->type == TK_PREPROCESSOR_LINE &&
            preprocessor_token_ahead_is_identifier(input_list->head, "else"))
        {
            struct token_list r4 = else_group(ctx, input_list, is_active && !if_result && !elif_result, level);
            token_list_append_list(&r, &r4);
            token_list_destroy(&r4);
        }

        if (ctx->n_errors > 0)
        {
            token_list_destroy(&r2);
            throw;
        }

        struct token_list r5 = endif_line(ctx, input_list, level);
        token_list_append_list(&r, &r5);
        token_list_destroy(&r5);
        token_list_destroy(&r2);
    }
    catch
    {
    }

    return r;
}

struct token_list identifier_list(struct preprocessor_ctx* ctx, struct macro* macro, struct token_list* input_list, int level)
{
    assert(input_list->head != NULL);
    struct token_list r = { 0 };

    try
    {
        /*
          identifier-list:
          identifier
          identifier-list , identifier
        */
        skip_blanks(ctx, &r, input_list);

        if (input_list->head == NULL)
            throw;

        struct macro_parameter* _Owner _Opt p_macro_parameter = calloc(1, sizeof * p_macro_parameter);
        if (p_macro_parameter == NULL) throw;

        char* _Owner _Opt temp = strdup(input_list->head->lexeme);
        if (temp == NULL)
        {
            macro_parameters_delete(p_macro_parameter);
            throw;
        }
        p_macro_parameter->name = temp;

        assert(macro->parameters == NULL);
        macro->parameters = p_macro_parameter;

        struct macro_parameter* p_last_parameter = macro->parameters;

        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
        skip_blanks(ctx, &r, input_list);

        if (input_list->head == NULL)
        {
            throw;
        }

        while (input_list->head->type == ',')
        {
            match_token_level(&r, input_list, ',', level, ctx);
            skip_blanks(ctx, &r, input_list);

            if (input_list->head == NULL)
            {
                throw;
            }

            if (input_list->head->type == '...')
            {
                break;
            }

            struct macro_parameter* _Owner _Opt p_new_macro_parameter = calloc(1, sizeof * p_new_macro_parameter);
            if (p_new_macro_parameter == NULL)
                throw;

            char* _Opt _Owner temp2 = strdup(input_list->head->lexeme);
            if (temp2 == NULL)
            {
                macro_parameters_delete(p_new_macro_parameter);
                throw;
            }

            p_new_macro_parameter->name = temp2;

            assert(p_last_parameter->next == NULL);
            p_last_parameter->next = p_new_macro_parameter;
            p_last_parameter = p_last_parameter->next;

            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
            skip_blanks(ctx, &r, input_list);

            if (input_list->head == NULL)
            {
                throw;
            }
        }
    }
    catch
    {
    }
    return r;
}


struct token_list replacement_list(struct preprocessor_ctx* ctx, struct macro* macro, struct token_list* input_list, int level)
{
    struct token_list r = { 0 };

    try
    {
        if (input_list->head == NULL)
        {
            pre_unexpected_end_of_file(NULL, ctx);
            throw;
        }

        while (input_list->head->type != TK_NEWLINE)
        {
            match_level(&r, input_list, level);
            if (input_list->head == NULL)
            {
                //terminou define sem quebra de linha
                break;
            }
        }

        assert(macro->replacement_list.head == NULL);
        struct token_list copy = copy_replacement_list(ctx, &r);
        token_list_append_list(&macro->replacement_list, &copy);
        token_list_destroy(&copy);
    }
    catch
    {
    }
    return r;
}

struct token_list pp_tokens_opt(struct preprocessor_ctx* ctx, struct token_list* input_list, int level)
{
    struct token_list r = { 0 };
    while (input_list->head && input_list->head->type != TK_NEWLINE)
    {
        prematch_level(&r, input_list, level);
    }
    return r;
}

static bool is_empty_assert(struct token_list* replacement_list)
{
    struct token* _Opt token = replacement_list->head;

    if (token == NULL)
        return false;

    if (strcmp(token->lexeme, "(")) return false;
    token = token->next;

    if (token == NULL) return false;
    if (strcmp(token->lexeme, "(")) return false;
    token = token->next;

    if (token == NULL) return false;
    if (strcmp(token->lexeme, "void")) return false;
    token = token->next;

    if (token == NULL) return false;
    if (strcmp(token->lexeme, ")")) return false;
    token = token->next;

    if (token == NULL) return false;
    if (strcmp(token->lexeme, "0")) return false;
    token = token->next;

    if (token == NULL) return false;
    if (strcmp(token->lexeme, ")")) return false;
    token = token->next;

    if (token != NULL) return false;

    return true;
}


struct token_list control_line(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{

    /*
        control-line:
            # "include" pp-tokens new-line
            # "embed" pp-tokens new-line
            # "define" identifier replacement-list new-line
            # "define" identifier ( identifier-list _Opt ) replacement-list new-line
            # "define" identifier ( ... ) replacement-list new-line
            # "define" identifier ( identifier-list , ... ) replacement-list new-line
            # "undef" identifier new-line
            # "line" pp-tokens new-line
            # "error" pp-tokens _Opt new-line
            # "warning" pp-tokens _Opt new-line
            # "pragma" pp-tokens _Opt new-line
            # new-line
    */

    struct token_list r = { 0 };

    try
    {
        if (!is_active)
        {
            //se nao esta ativo eh ignorado
            struct token_list r7 = pp_tokens_opt(ctx, input_list, level);
            token_list_append_list(&r, &r7);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
            token_list_destroy(&r7);
            return r;
        }

        if (input_list->head == NULL)
        {
            pre_unexpected_end_of_file(r.tail, ctx);
            throw;
        }

        //struct token* const ptoken = input_list->head;
        match_token_level(&r, input_list, TK_PREPROCESSOR_LINE, level, ctx);
        skip_blanks_level(ctx, &r, input_list, level);

        if (input_list->head == NULL)
        {
            pre_unexpected_end_of_file(r.tail, ctx);
            throw;
        }

        if (strcmp(input_list->head->lexeme, "include") == 0 ||
            strcmp(input_list->head->lexeme, "include_next") == 0)
        {
            bool include_next = strcmp(input_list->head->lexeme, "include_next") == 0;
            /*
              # include pp-tokens new-line
            */
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //include
            skip_blanks_level(ctx, &r, input_list, level);

            if (input_list->head == NULL)
            {
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            char path[100] = { 0 };
            bool is_angle_bracket_form = false;
            if (input_list->head->type == TK_STRING_LITERAL)
            {
                strcat(path, input_list->head->lexeme);
                prematch_level(&r, input_list, level);
            }
            else
            {
                is_angle_bracket_form = true;
                while (input_list->head->type != '>')
                {
                    strcat(path, input_list->head->lexeme);
                    prematch_level(&r, input_list, level);

                    if (input_list->head == NULL)
                    {
                        pre_unexpected_end_of_file(r.tail, ctx);
                        throw;
                    }
                }
                strcat(path, input_list->head->lexeme);
                prematch_level(&r, input_list, level);
            }


            while (input_list->head->type != TK_NEWLINE)
            {
                prematch_level(&r, input_list, level);
                if (input_list->head == NULL)
                {
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }
            }

            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);

            path[strlen(path) - 1] = '\0';

            /*this is the dir of the current file*/
            char current_file_dir[300] = { 0 };
            snprintf(current_file_dir, sizeof current_file_dir, "%s", r.tail->token_origin->lexeme);
            dirname(current_file_dir);


            char full_path_result[200] = { 0 };
            bool already_included = false;
            const char* _Owner _Opt content = find_and_read_include_file(ctx,
                path + 1,
                current_file_dir,
                is_angle_bracket_form,
                &already_included,
                full_path_result,
                sizeof full_path_result,
                include_next);

            if (content != NULL)
            {
                if (ctx->options.show_includes)
                {
                    for (int i = 0; i < (level + 1); i++)
                        printf(".");
                    printf("%s\n", full_path_result);
                }

                struct tokenizer_ctx tctx = { 0 };
                struct token_list list = tokenizer(&tctx, content, full_path_result, level + 1, TK_FLAG_NONE);
                free((void* _Owner)content);

                struct token_list list2 = preprocessor(ctx, &list, level + 1);
                token_list_append_list(&r, &list2);

                token_list_destroy(&list2);
                token_list_destroy(&list);
            }
            else
            {
                if (!already_included)
                {
                    preprocessor_diagnostic(C_ERROR_FILE_NOT_FOUND, ctx, r.tail, "file %s not found", path + 1);

                    for (struct include_dir* _Opt p = ctx->include_dir.head; p; p = p->next)
                    {
                        /*let's print the include path*/
                        preprocessor_diagnostic(W_NOTE, ctx, r.tail, "dir = '%s'", p->path);
                    }
                }
                else
                {
                    //pragma once..
                }
            }

        }
        else if (strcmp(input_list->head->lexeme, "embed") == 0)
        {
            struct token_list discard0 = { 0 };
            struct token_list* p_list = &r;

            /*
              C23
              # embed pp-tokens new-line
            */

            const struct token* const p_embed_token = input_list->head;

            match_token_level(p_list, input_list, TK_IDENTIFIER, level, ctx); //embed

            skip_blanks_level(ctx, p_list, input_list, level);


            if (input_list->head == NULL)
            {
                throw;
            }

            char path[100] = { 0 };
            if (input_list->head->type == TK_STRING_LITERAL)
            {
                strcat(path, input_list->head->lexeme);
                prematch_level(p_list, input_list, level);
            }
            else
            {
                while (input_list->head->type != '>')
                {
                    strcat(path, input_list->head->lexeme);
                    prematch_level(p_list, input_list, level);

                    if (input_list->head == NULL)
                    {
                        throw;
                    }
                }
                strcat(path, input_list->head->lexeme);
                prematch_level(p_list, input_list, level);
            }

            if (input_list->head)
            {
                while (input_list->head->type != TK_NEWLINE)
                {
                    prematch_level(p_list, input_list, level);
                    if (input_list->head == NULL)
                    {
                        pre_unexpected_end_of_file(p_list->tail, ctx);
                        throw;
                    }
                }
            }
            match_token_level(p_list, input_list, TK_NEWLINE, level, ctx);

            char fullpath[300] = { 0 };
            path[strlen(path) - 1] = '\0';

            snprintf(fullpath, sizeof(fullpath), "%s", path + 1);


            int nlevel = level;

            enum token_flags f = 0;

            f = TK_FLAG_FINAL;
            //we cannot see it just like include
            nlevel = nlevel + 1;

            struct token_list list = embed_tokenizer(ctx, p_embed_token, fullpath, nlevel, f);

            if (ctx->n_errors > 0)
            {
                token_list_destroy(&list);
                throw;
            }

            token_list_append_list(&r, &list);
            token_list_destroy(&list);
            token_list_destroy(&discard0);
        }
        else if (strcmp(input_list->head->lexeme, "define") == 0)
        {
            //TODO strcmp nao pode ser usado temos que criar uma funcao especial

            /*
             #de\
             fine A 1

            A
            */

            struct macro* _Owner _Opt macro = calloc(1, sizeof * macro);
            if (macro == NULL)
            {
                preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, "out of mem");
                throw;
            }

            /*
                # define identifier                           replacement-list new-line
                # define identifier ( identifier-list_opt )    replacement-list new-line
                # define identifier ( ... )                   replacement-list new-line
                # define identifier ( identifier-list , ... ) replacement-list new-line
            */
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //define
            skip_blanks_level(ctx, &r, input_list, level);

            if (input_list->head == NULL)
            {
                macro_delete(macro);
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            struct token* macro_name_token = input_list->head;

            if (is_builtin_macro(macro_name_token->lexeme))
            {
                preprocessor_diagnostic(W_REDEFINING_BUITIN_MACRO,
                    ctx,
                    input_list->head,
                    "redefining builtin macro");
            }

            macro->p_name_token = macro_name_token;

            char* _Owner _Opt temp = strdup(input_list->head->lexeme);
            if (temp == NULL)
            {
                macro_delete(macro);
                throw;
            }
            assert(macro->name == NULL);
            macro->name = temp;


            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx); //nome da macro

            if (input_list->head == NULL)
            {
                macro_delete(macro);
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            if (input_list->head->type == '(')
            {
                macro->is_function = true;

                match_token_level(&r, input_list, '(', level, ctx);
                skip_blanks_level(ctx, &r, input_list, level);

                if (input_list->head == NULL)
                {
                    macro_delete(macro);
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                if (input_list->head->type == '...')
                {
                    struct macro_parameter* _Owner _Opt p_macro_parameter = calloc(1, sizeof * p_macro_parameter);
                    if (p_macro_parameter == NULL)
                    {
                        macro_delete(macro);
                        throw;
                    }

                    char* _Owner _Opt temp2 = strdup("__VA_ARGS__");
                    if (temp2 == NULL)
                    {
                        macro_delete(macro);
                        macro_parameters_delete(p_macro_parameter);
                        throw;
                    }

                    p_macro_parameter->name = temp2;
                    macro->parameters = p_macro_parameter;

                    // assert(false);
                    match_token_level(&r, input_list, '...', level, ctx); //nome da macro
                    skip_blanks_level(ctx, &r, input_list, level);
                    match_token_level(&r, input_list, ')', level, ctx); //nome da macro
                }
                else if (input_list->head->type == ')')
                {
                    match_token_level(&r, input_list, ')', level, ctx);
                    skip_blanks_level(ctx, &r, input_list, level);
                }
                else
                {

                    struct token_list r3 = identifier_list(ctx, macro, input_list, level);
                    token_list_append_list(&r, &r3);
                    token_list_destroy(&r3);

                    skip_blanks_level(ctx, &r, input_list, level);
                    if (input_list->head == NULL)
                    {
                        macro_delete(macro);
                        pre_unexpected_end_of_file(r.tail, ctx);
                        throw;
                    }

                    if (input_list->head->type == '...')
                    {
                        struct macro_parameter* _Owner _Opt p_macro_parameter = calloc(1, sizeof * p_macro_parameter);
                        if (p_macro_parameter == NULL)
                        {
                            macro_delete(macro);
                            throw;
                        }

                        char* _Owner _Opt temp3 = strdup("__VA_ARGS__");
                        if (temp3 == NULL)
                        {
                            macro_delete(macro);
                            macro_parameters_delete(p_macro_parameter);
                            throw;
                        }

                        p_macro_parameter->name = temp3;
                        struct macro_parameter* _Opt p_last = macro->parameters;
                        assert(p_last != NULL);
                        while (p_last->next)
                        {
                            p_last = p_last->next;
                        }
                        p_last->next = p_macro_parameter;

                        match_token_level(&r, input_list, '...', level, ctx);
                    }
                    skip_blanks_level(ctx, &r, input_list, level);
                    match_token_level(&r, input_list, ')', level, ctx);
                }
            }
            else
            {
                macro->is_function = false;
            }
            skip_blanks_level(ctx, &r, input_list, level);
            if (input_list->head == NULL)
            {
                macro_delete(macro);
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }
            struct token_list r4 = replacement_list(ctx, macro, input_list, level);
            token_list_append_list(&r, &r4);
            token_list_destroy(&r4);

            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
            if (!ctx->options.disable_assert && strcmp(macro->name, "assert") == 0)
            {
                //cake overrides macro assert in debug and release to be defined as 
                //assert(__VA_ARGS__)
                if (!is_empty_assert(&macro->replacement_list))
                {
                    macro_parameters_delete(macro->parameters);

                    struct macro_parameter* _Owner _Opt p_macro_parameter = calloc(1, sizeof * p_macro_parameter);
                    if (p_macro_parameter == NULL)
                    {
                        macro_delete(macro);
                        throw;
                    }

                    char* _Owner _Opt temp2 = strdup("__VA_ARGS__");
                    if (temp2 == NULL)
                    {
                        macro_delete(macro);
                        macro_parameters_delete(p_macro_parameter);
                        throw;
                    }
                    p_macro_parameter->name = temp2;
                    macro->parameters = p_macro_parameter;

                    token_list_destroy(&macro->replacement_list);
                    struct tokenizer_ctx tctx = { 0 };
                    macro->replacement_list = tokenizer(&tctx, "assert(__VA_ARGS__)", NULL, level, TK_FLAG_NONE);
                }
            }

            naming_convention_macro(ctx, macro_name_token);

            struct macro* existing_macro = find_macro(ctx, macro->name);
            if (existing_macro &&
                !macro_is_same(macro, existing_macro))
            {
                preprocessor_diagnostic(C_ERROR_MACRO_REDEFINITION,
                ctx,
                macro->p_name_token,
                "macro redefinition");

                preprocessor_diagnostic(W_NOTE,
                ctx,
                existing_macro->p_name_token,
                "previous definition");

                macro_delete(macro);

                throw;
            }

            struct hash_item_set item = { 0 };
            item.p_macro = macro;
            hashmap_set(&ctx->macros, macro->name, &item);
            hash_item_set_destroy(&item);
        }
        else if (strcmp(input_list->head->lexeme, "undef") == 0)
        {
            /*
             # undef identifier new-line
            */
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//undef
            skip_blanks_level(ctx, &r, input_list, level);

            if (input_list->head == NULL)
            {
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            struct macro* _Owner _Opt macro = (struct macro* _Owner _Opt) hashmap_remove(&ctx->macros, input_list->head->lexeme, NULL);
            assert(find_macro(ctx, input_list->head->lexeme) == NULL);
            if (macro)
            {
                macro_delete(macro);
                match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//undef
            }
            else
            {
                match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//undef
                /*no warnings*/
            }
            skip_blanks_level(ctx, &r, input_list, level);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
        }
        else if (strcmp(input_list->head->lexeme, "line") == 0)
        {
            /*
               # line pp-tokens new-line
            */
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//line
            struct token_list r5 = pp_tokens_opt(ctx, input_list, level);
            token_list_append_list(&r, &r5);
            token_list_destroy(&r5);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
        }
        else if (strcmp(input_list->head->lexeme, "error") == 0)
        {
            /*
              # error pp-tokensopt new-line
            */
            ctx->n_warnings++;
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//error
            struct token_list r6 = pp_tokens_opt(ctx, input_list, level);
            preprocessor_diagnostic(C_ERROR_PREPROCESSOR_C_ERROR_DIRECTIVE, ctx, input_list->head, "#error");
            token_list_append_list(&r, &r6);
            token_list_destroy(&r6);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);


        }
        else if (strcmp(input_list->head->lexeme, "warning") == 0)
        {
            /*
              # warning pp-tokensopt new-line
            */
            ctx->n_warnings++;

            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//warning

            struct token_list r6 = pp_tokens_opt(ctx, input_list, level);
            preprocessor_diagnostic(W_NONE, ctx, input_list->head, "#warning");
            token_list_append_list(&r, &r6);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
            token_list_destroy(&r6);
        }
        else if (strcmp(input_list->head->lexeme, "pragma") == 0)
        {
            /*
              # pragma pp-tokensopt new-line
            */
            /*
               #pragma will survive and compiler will handle as
               pragma declaration
            */
            match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//pragma

            if (r.tail)
            {
                r.tail->type = TK_PRAGMA;
                r.tail->flags |= TK_FLAG_FINAL;
            }
            skip_blanks_level(ctx, &r, input_list, level);

            if (input_list->head == NULL)
            {
                pre_unexpected_end_of_file(r.tail, ctx);
                throw;
            }

            if (input_list->head->type == TK_IDENTIFIER)
            {
                if (strcmp(input_list->head->lexeme, "CAKE") == 0)
                {
                    match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);
                    if (r.tail)
                    {
                        r.tail->flags |= TK_FLAG_FINAL;
                    }
                    skip_blanks_level(ctx, &r, input_list, level);
                }

                if (input_list->head == NULL)
                {
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                if (strcmp(input_list->head->lexeme, "once") == 0)
                {
                    pragma_once_add(ctx, input_list->head->token_origin->lexeme);
                    match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//pragma
                    if (r.tail)
                    {
                        r.tail->flags |= TK_FLAG_FINAL;
                    }
                }
                else if (strcmp(input_list->head->lexeme, "dir") == 0)
                {
                    match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//pragma
                    skip_blanks_level(ctx, &r, input_list, level);

                    if (input_list->head == NULL)
                    {
                        pre_unexpected_end_of_file(r.tail, ctx);
                        throw;
                    }

                    if (input_list->head->type != TK_STRING_LITERAL)
                    {
                        preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, input_list->head, "expected string");
                        throw;
                    }

                    char path[200] = { 0 };
                    strncpy(path, input_list->head->lexeme + 1, strlen(input_list->head->lexeme) - 2);
                    include_dir_add(&ctx->include_dir, path);
                    match_token_level(&r, input_list, TK_STRING_LITERAL, level, ctx);//pragma
                    if (r.tail)
                    {
                        r.tail->flags |= TK_FLAG_FINAL;
                    }
                }
                else if (strcmp(input_list->head->lexeme, "nullchecks") == 0)
                {
                    assert(false);
                    match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//nullchecks
                    assert(r.tail != NULL);
                    r.tail->flags |= TK_FLAG_FINAL;

                    skip_blanks_level(ctx, &r, input_list, level);
                    ctx->options.null_checks_enabled = true;
                }

                if (input_list->head == NULL)
                {
                    pre_unexpected_end_of_file(r.tail, ctx);
                    throw;
                }

                if (strcmp(input_list->head->lexeme, "diagnostic") == 0)
                {
                    match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//diagnostic
                    assert(r.tail != NULL);
                    r.tail->flags |= TK_FLAG_FINAL;

                    skip_blanks_level(ctx, &r, input_list, level);

                    if (input_list->head == NULL)
                    {
                        pre_unexpected_end_of_file(r.tail, ctx);
                        throw;
                    }

                    if (strcmp(input_list->head->lexeme, "push") == 0)
                    {
                        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//diagnostic
                        assert(r.tail != NULL);
                        r.tail->flags |= TK_FLAG_FINAL;

                        //#pragma GCC diagnostic push
                        if (ctx->options.diagnostic_stack.top_index <
                            sizeof(ctx->options.diagnostic_stack) / sizeof(ctx->options.diagnostic_stack.stack[0]))
                        {
                            ctx->options.diagnostic_stack.top_index++;

                            ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index] =
                                ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index - 1];
                        }
                    }
                    else if (strcmp(input_list->head->lexeme, "pop") == 0)
                    {
                        //#pragma GCC diagnostic pop
                        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//pop
                        assert(r.tail != NULL);
                        r.tail->flags |= TK_FLAG_FINAL;
                        if (ctx->options.diagnostic_stack.top_index > 0)
                        {
                            ctx->options.diagnostic_stack.top_index--;
                        }
                    }
                    else if (strcmp(input_list->head->lexeme, "warning") == 0)
                    {
                        //#pragma CAKE diagnostic warning "-Wenum-compare"

                        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//warning
                        assert(r.tail != NULL);
                        r.tail->flags |= TK_FLAG_FINAL;
                        skip_blanks_level(ctx, &r, input_list, level);

                        if (input_list->head && input_list->head->type == TK_STRING_LITERAL)
                        {
                            match_token_level(&r, input_list, TK_STRING_LITERAL, level, ctx);//""
                            assert(r.tail != NULL);
                            r.tail->flags |= TK_FLAG_FINAL;

                            unsigned long long  w = get_warning_bit_mask(input_list->head->lexeme + 1 + 2);
                            ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings |= w;
                        }
                    }
                    else if (strcmp(input_list->head->lexeme, "ignore") == 0)
                    {
                        //#pragma CAKE diagnostic ignore "-Wenum-compare"

                        match_token_level(&r, input_list, TK_IDENTIFIER, level, ctx);//ignore
                        assert(r.tail != NULL);
                        r.tail->flags |= TK_FLAG_FINAL;

                        skip_blanks_level(ctx, &r, input_list, level);

                        if (input_list->head && input_list->head->type == TK_STRING_LITERAL)
                        {
                            unsigned long long w = get_warning_bit_mask(input_list->head->lexeme + 1 + 2);
                            ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings &= ~w;
                        }
                    }
                }
            }

            struct token_list r7 = pp_tokens_opt(ctx, input_list, level);
            token_list_append_list(&r, &r7);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
            assert(r.tail != NULL);
            r.tail->type = TK_PRAGMA_END;
            r.tail->flags |= TK_FLAG_FINAL;
            token_list_destroy(&r7);
        }
        else if (input_list->head->type == TK_NEWLINE)
        {
            skip_blanks_level(ctx, &r, input_list, level);
            match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
        }
        else
        {
            //handled by the caller
            preprocessor_diagnostic(C_ERROR_UNEXPECTED_TOKEN,
                ctx,
                input_list->head,
                "unexpected\n");
            throw;
        }
    }
    catch
    {
    }

    return r;
}


static struct token_list non_directive(struct preprocessor_ctx* ctx, struct token_list* input_list, int level)
{
    /*
      non-directive:
      pp-tokens new-line
     */
    struct token_list r = pp_tokens_opt(ctx, input_list, level);
    skip_blanks_level(ctx, &r, input_list, level);
    match_token_level(&r, input_list, TK_NEWLINE, level, ctx);
    return r;
}

static struct macro_argument_list collect_macro_arguments(struct preprocessor_ctx* ctx,
    struct macro* macro,
    struct token_list* input_list, int level)
{

    struct macro_argument_list macro_argument_list = { 0 };

    try
    {
        if (input_list->head == NULL)
        {
            throw;
        }

        assert(input_list->head->type == TK_IDENTIFIER);//macro name
        const struct token* const macro_name_token = input_list->head;

        match_token_level(&macro_argument_list.tokens, input_list, TK_IDENTIFIER, level, ctx); //MACRO NAME

        if (!macro->is_function)
        {
            //This function is also called for non function like macros.
            //In this case we return empty
            return macro_argument_list;
        }


        int count = 1;

        //skip spaces after macro name
        skip_blanks(ctx, &macro_argument_list.tokens, input_list);

        //macro is a function
        match_token_level(&macro_argument_list.tokens, input_list, '(', level, ctx);

        //skip spaces after (
        skip_blanks(ctx, &macro_argument_list.tokens, input_list);

        if (input_list->head == NULL)
        {
            pre_unexpected_end_of_file(macro_argument_list.tokens.tail, ctx);
            throw;
        }

        if (input_list->head->type == ')')
        {
            /*
               empty argument list
            */

            if (macro->parameters != NULL)
            {
                struct macro_argument* _Owner _Opt  p_argument = calloc(1, sizeof(struct macro_argument));
                if (p_argument == NULL)
                {
                    throw;
                }
                struct macro_parameter* p_current_parameter = macro->parameters;
                p_argument->macro_parameter = p_current_parameter;
                argument_list_add(&macro_argument_list, p_argument);
            }
            match_token_level(&macro_argument_list.tokens, input_list, ')', level, ctx);
            return macro_argument_list;
        }

        if (macro->parameters == NULL)
        {
            //we have a non empty argument list, calling a macro without parameters
            preprocessor_diagnostic(C_ERROR_TOO_MANY_ARGUMENTS_TO_FUNCTION_LIKE_MACRO, ctx, macro_name_token, "too many arguments provided to function-like macro invocation\n");
            throw;
        }

        struct macro_parameter* p_current_parameter = macro->parameters;
        struct macro_argument* _Owner _Opt p_argument = calloc(1, sizeof(struct macro_argument));
        if (p_argument == NULL)
        {
            throw;
        }

        p_argument->macro_parameter = p_current_parameter;
        //collect next arguments...
        while (input_list->head != NULL)
        {
            if (input_list->head->type == '(')
            {
                count++;
                token_list_clone_and_add(&p_argument->tokens, input_list->head);
                match_token_level(&macro_argument_list.tokens, input_list, '(', level, ctx);
            }
            else if (input_list->head->type == ')')
            {
                count--;
                if (count == 0)
                {
                    match_token_level(&macro_argument_list.tokens, input_list, ')', level, ctx);
                    argument_list_add(&macro_argument_list, p_argument);
                    p_argument = NULL; //MOVED

                    if (p_current_parameter->next != NULL)
                    {
                        p_current_parameter = p_current_parameter->next;
                        if (strcmp(p_current_parameter->name, "__VA_ARGS__") == 0)
                        {
                            //we add this argument as being empty
                            p_argument = calloc(1, sizeof(struct macro_argument));
                            if (p_argument == NULL)
                            {
                                throw;
                            }

                            p_argument->macro_parameter = p_current_parameter;
                            argument_list_add(&macro_argument_list, p_argument);
                            p_argument = NULL; //MOVED
                        }
                        else
                        {
                            preprocessor_diagnostic(C_ERROR_TOO_FEW_ARGUMENTS_TO_FUNCTION_LIKE_MACRO, ctx, macro_name_token, "too few arguments provided to function-like macro invocation\n");
                            throw;
                        }
                    }

                    break;
                }
                else
                {
                    token_list_clone_and_add(&p_argument->tokens, input_list->head);
                    match_token_level(&macro_argument_list.tokens, input_list, ')', level, ctx);
                }
            }
            else if (count == 1 && input_list->head->type == ',')
            {
                if (strcmp(p_current_parameter->name, "__VA_ARGS__") == 0)
                {
                    token_list_clone_and_add(&p_argument->tokens, input_list->head);
                    match_token_level(&macro_argument_list.tokens, input_list, ',', level, ctx);
                }
                else //if (count == 1)
                {
                    match_token_level(&macro_argument_list.tokens, input_list, ',', level, ctx);
                    argument_list_add(&macro_argument_list, p_argument);
                    p_argument = NULL; /*MOVED*/

                    p_argument = calloc(1, sizeof(struct macro_argument));
                    if (p_argument == NULL)
                    {
                        throw;
                    }

                    if (p_current_parameter->next == NULL)
                    {
                        preprocessor_diagnostic(C_ERROR_TOO_MANY_ARGUMENTS_TO_FUNCTION_LIKE_MACRO, ctx, macro_argument_list.tokens.tail, "too many arguments provided to function-like macro invocation\n");
                        macro_argument_delete(p_argument);
                        p_argument = NULL; //DELETED
                        throw;
                    }

                    p_current_parameter = p_current_parameter->next;

                    p_argument->macro_parameter = p_current_parameter;
                }
            }
            else
            {
                token_list_clone_and_add(&p_argument->tokens, input_list->head);
                prematch_level(&macro_argument_list.tokens, input_list, level);
                //token_list_add(&list, token_list_pop_front(input_list));
            }
        }

        assert(p_argument == NULL);
    }
    catch
    {
    }

    return macro_argument_list;
}

struct token_list expand_macro(struct preprocessor_ctx* ctx, struct macro_expanded* _Opt p_list, struct macro* macro, struct macro_argument_list* arguments, int level, const struct token* origin);
struct token_list replacement_list_reexamination(struct preprocessor_ctx* ctx, struct macro_expanded* p_list, struct token_list* oldlist, int level, const struct token* origin);


struct token_list macro_copy_replacement_list(struct preprocessor_ctx* ctx, struct macro* macro, const struct token* origin);

/*#define hash_hash # ## #
#define mkstr(a) # a
#define in_between(a) mkstr(a)
#define join(c, d) in_between(c hash_hash d)

hash_hash

join(x, y)
*/
static struct token_list concatenate(struct preprocessor_ctx* ctx, struct token_list* input_list)
{
    //printf("input="); print_list(input_list);

    struct token_list  r = { 0 };
    try
    {
        //todo juntar tokens mesmo objet macro
        //struct token* p_previousNonBlank = 0;
        while (input_list->head)
        {
            //printf("r="); print_list(&r);
            //printf("input="); print_list(input_list);

            //#def macro
            //assert(!(input_list->head->flags & TK_FLAG_HAS_NEWLINE_BEFORE));
            if (input_list->head->type == '##')
            {
                if (r.tail == NULL)
                {
                    preprocessor_diagnostic(C_ERROR_PREPROCESSOR_MISSING_MACRO_ARGUMENT, ctx, input_list->head, "missing macro argument (should be checked before)");
                    break;
                }
                /*
                * arranca ## do input (sem adicionar)
                */
                token_list_pop_front(input_list);

                struct osstream ss = { 0 };

                /*
                *  Faz uma string com o fim r + começo do input (## ja foi removido)
                */
                if (r.tail->lexeme[0] != '\0')
                    ss_fprintf(&ss, "%s", r.tail->lexeme);

                if (input_list->head && input_list->head->lexeme[0] != '\0')
                    ss_fprintf(&ss, "%s", input_list->head->lexeme);


                //copiar o level para gerar um novo igual
                int level = input_list->head ? input_list->head->level : 0;

                /*
                * Já paga do input o token usado na concatenacao
                */
                token_list_pop_front(input_list);

                /*
                * Faz um novo token com a string montada
                */
                struct tokenizer_ctx tctx = { 0 };
                struct token_list newlist = { 0 };

                if (ss.c_str != NULL)
                {
                    newlist = tokenizer(&tctx, ss.c_str, NULL, level, TK_FLAG_NONE);
                }

                if (newlist.head)
                {
                    newlist.head->flags = r.tail->flags;
                }
                else
                {
                    struct token* _Owner _Opt p_new_token = calloc(1, sizeof * p_new_token);
                    if (p_new_token == NULL)
                    {
                        ss_close(&ss);
                        throw;
                    }
                    char* _Owner _Opt temp = strdup("");
                    if (temp == NULL)
                    {
                        ss_close(&ss);
                        token_delete(p_new_token);
                        throw;
                    }
                    p_new_token->lexeme = temp;
                    p_new_token->type = TK_PLACEMARKER;
                    token_list_add(&newlist, p_new_token);
                    assert(newlist.head != NULL);
                    newlist.head->flags = r.tail->flags;
                }
                /*
                * Arranca o anterior do r que foi usado para formar string
                */
                token_list_pop_back(&r);

                /*adiciona novo token no fim do r*/
                token_list_append_list(&r, &newlist);

                ss_close(&ss);

                token_list_destroy(&newlist);

                if (input_list->head == NULL)
                    break;
            }
            else
            {
                prematch(&r, input_list);
            }
        }
    }
    catch
    {
    }

    return r;
}

/*
  check if the argument list that corresponds to a trailing ...
  of the parameter list is present and has a non-empty substitution.
*/
static bool has_argument_list_empty_substitution(struct preprocessor_ctx* ctx,
    struct macro_expanded* p_list,
    struct macro_argument_list* p_macro_argument_list,
    const struct token* origin)
{
    if (p_macro_argument_list->head == NULL)
        return true;

    struct macro_argument* _Opt p_va_args_argument =
        find_macro_argument_by_name(p_macro_argument_list, "__VA_ARGS__");

    if (p_va_args_argument)
    {
        if (p_va_args_argument->tokens.head == NULL)
            return true;

        struct token_list argumentlist = copy_argument_list(p_va_args_argument);

        struct token_list r4 = replacement_list_reexamination(ctx, p_list, &argumentlist, 0, origin);
        const bool results_in_empty_substituition = (r4.head == NULL || r4.head->type == TK_PLACEMARKER);
        token_list_destroy(&r4);

        token_list_destroy(&argumentlist);

        return results_in_empty_substituition;
    }

    return false;
}

static struct token_list replace_macro_arguments(struct preprocessor_ctx* ctx, struct macro_expanded* p_list, struct token_list* input_list, struct macro_argument_list* arguments, const struct token* origin)
{
    struct token_list r = { 0 };

    try
    {
        /*clear previous usage*/
        struct macro_argument* _Owner _Opt p = arguments->head;
        while (p)
        {
            struct macro_argument* _Owner _Opt next = p->next;
            if (p->macro_parameter)
            {
                p->macro_parameter->already_expanded = false;
                token_list_clear(&p->macro_parameter->expanded_list);
            }
            p = next;
        }

        while (input_list->head)
        {
            assert(!(input_list->head->flags & TK_FLAG_HAS_NEWLINE_BEFORE));
            assert(!token_is_blank(input_list->head));
            assert(r.tail == NULL || !token_is_blank(r.tail));
            struct macro_argument* _Opt p_argument = NULL;
            if (input_list->head->type == TK_IDENTIFIER)
            {
                if (strcmp(input_list->head->lexeme, "__VA_OPT__") == 0)
                {
                    token_list_pop_front(input_list); //pop __VA_OPT__
                    token_list_pop_front(input_list); //pop (
                    int parenteses_count = 1;         //we already have one

                    const bool discard_va_opt =
                        has_argument_list_empty_substitution(ctx, p_list, arguments, origin);

                    if (discard_va_opt)
                    {
                        //discard all tokens __VA_OPT__(...)
                        while (input_list->head)
                        {
                            if (input_list->head->type == '(') parenteses_count++;
                            else if (input_list->head->type == ')') parenteses_count--;
                            token_list_pop_front(input_list);
                            if (parenteses_count == 0)
                                break;
                        }
                    }
                    else
                    {
                        // Search and remove the last balanced ')'
                        struct token* _Opt p_token = input_list->head;
                        for (; p_token; p_token = p_token->next)
                        {
                            if (p_token->type == '(') parenteses_count++;
                            else if (p_token->type == ')') parenteses_count--;

                            if (parenteses_count == 0)
                                break;
                        }
                        token_list_remove(input_list, p_token, p_token);
                    }
                    continue;
                }

                p_argument = find_macro_argument_by_name(arguments, input_list->head->lexeme);
            }

            if (p_argument)
            {
                if (r.tail != NULL && r.tail->type == '#')
                {

                    /*
                      deleta nome parametro da lista
                      antes copia flags dele
                    */

                    const enum token_flags flags = r.tail->flags;
                    token_list_pop_front(input_list);

                    //deleta tambem # do fim
                    while (token_is_blank(r.tail))
                    {
                        token_list_pop_back(&r);
                    }
                    token_list_pop_back(&r);

                    ///----------------------------
                    //transforma tudo em string e coloca no resultado
                    struct token_list argumentlist = copy_argument_list(p_argument);
                    char* _Owner _Opt s = token_list_join_tokens(&argumentlist, true);
                    if (s == NULL)
                    {
                        token_list_destroy(&argumentlist);
                        preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, input_list->head, "unexpected");
                        throw;
                    }
                    struct token* _Owner _Opt p_new_token = calloc(1, sizeof * p_new_token);

                    if (p_new_token == NULL)
                    {
                        free(s);
                        token_list_destroy(&argumentlist);
                        throw;
                    }

                    p_new_token->lexeme = s;
                    p_new_token->type = TK_STRING_LITERAL;
                    p_new_token->flags = flags;
                    token_list_add(&r, p_new_token);
                    token_list_destroy(&argumentlist);
                    continue;
                }
                else if (r.tail != NULL && r.tail->type == '##')
                {
                    //estou parametro e anterior era ##
                    token_list_pop_front(input_list);
                    struct token_list argumentlist = copy_argument_list(p_argument);
                    token_list_append_list(&r, &argumentlist);
                    token_list_destroy(&argumentlist);
                }
                else if (input_list->head->next && input_list->head->next->type == '##')
                {
                    //estou no parametro e o da frente eh ##
                    int flags = input_list->head->flags;
                    //tira nome parametro a lista
                    token_list_pop_front(input_list);
                    //passa tudo p resultado
                    struct token_list argumentlist = copy_argument_list(p_argument);
                    if (argumentlist.head != NULL)
                    {
                        argumentlist.head->flags = flags;
                    }
                    token_list_append_list(&r, &argumentlist);
                    // ja passa o ## tambem
                    prematch(&r, input_list);
                    token_list_destroy(&argumentlist);
                }
                else
                {
                    const int flags = input_list->head->flags;

                    //remove nome parametro do input
                    token_list_pop_front(input_list);

                    if (p_argument->macro_parameter == NULL)
                    {
                        throw;
                    }

                    if (!p_argument->macro_parameter->already_expanded)
                    {
                        /*
                          https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3457.htm#number-of-expansions
                          For each such parameter this expansion is performed exactly once,
                          and then preprocessing tokens naming the parameter are each replaced
                          with the resulting token list.
                        */
                        struct token_list copy_list = copy_argument_list(p_argument);
                        struct token_list r4 = replacement_list_reexamination(ctx, p_list, &copy_list, 0, origin);
                        token_list_swap(&p_argument->macro_parameter->expanded_list, &r4);
                        token_list_destroy(&r4);
                        p_argument->macro_parameter->already_expanded = true;
                    }

                    //Use the previous expansion
                    struct token_list copy_list = copy_argument_list_tokens(&p_argument->macro_parameter->expanded_list);
                    if (copy_list.head)
                    {
                        //fix flags
                        copy_list.head->flags = flags;
                    }
                    token_list_append_list(&r, &copy_list);
                    token_list_destroy(&copy_list);
                    if (ctx->n_errors > 0)
                    {
                        throw;
                    }
                }
            }
            else
            {
                prematch(&r, input_list);
            }
        }
    }
    catch
    {
    }

    return r;
}

struct token_list concatenate(struct preprocessor_ctx* ctx, struct token_list* input_list);

static bool macro_already_expanded(struct macro_expanded* _Opt p_list, const char* name)
{
    struct macro_expanded* _Opt p_item = p_list;
    while (p_item)
    {
        if (strcmp(name, p_item->name) == 0)
        {
            return true;
        }
        p_item = p_item->p_previous;
    }
    return false;
}

struct token_list replacement_list_reexamination(struct preprocessor_ctx* ctx,
    struct macro_expanded* p_list,
    struct token_list* oldlist,
    int level,
    const struct token* origin)
{
    struct token_list r = { 0 };
    try
    {
        //replacement_list_reexamination
        /*
        For both object-like and function-like macro invocations, before the replacement list is reexamined
        for more macro names to replace, each instance of a ## preprocessing token in the replacement list
        (not from an argument) is deleted and the preceding preprocessing token is concatenated with the
        following preprocessing token.
        */
        struct token_list new_list = concatenate(ctx, oldlist);
        while (new_list.head != NULL)
        {
            //OBS: #def macro have newlinew
            //assert(!(new_list.head->flags & TK_FLAG_HAS_NEWLINE_BEFORE));
            // assert(!token_is_blank(new_list.head));

            struct macro* _Opt macro = NULL;
            if (new_list.head->type == TK_IDENTIFIER)
            {
                macro = find_macro(ctx, new_list.head->lexeme);
                if (macro &&
                    macro->is_function &&
                    !preprocessor_token_ahead_is(new_list.head, '('))
                {
                    macro = NULL;
                }

                if (macro && macro_already_expanded(p_list, new_list.head->lexeme))
                {
                    new_list.head->type = TK_IDENTIFIER_RECURSIVE_MACRO;
                    macro = NULL;
                }


                if (ctx->conditional_inclusion)
                {
                    /*
                        When we are expanding in conditional inclusion the defined macro or defined (macro)
                        is not expanded and is considered later
                    */
                    if (r.tail &&
                        r.tail->type == TK_IDENTIFIER &&
                        strcmp(r.tail->lexeme, "defined") == 0)
                    {
                        macro = NULL;
                    }
                    else if (r.tail &&
                        r.tail->type == '(')
                    {
                        struct token* _Opt previous = r.tail->prev;
                        if (previous != NULL &&
                            previous->type == TK_IDENTIFIER &&
                            strcmp(previous->lexeme, "defined") == 0)
                        {
                            macro = NULL;
                        }
                    }
                }

            }
            if (macro)
            {
                int flags = new_list.head->flags;
                struct macro_argument_list arguments = collect_macro_arguments(ctx, macro, &new_list, level);
                if (ctx->n_errors > 0)
                {
                    macro_argument_list_destroy(&arguments);
                    token_list_destroy(&new_list);
                    throw;
                }


                struct token_list r3 = expand_macro(ctx, p_list, macro, &arguments, level, origin);
                if (ctx->n_errors > 0)
                {
                    token_list_destroy(&new_list);
                    token_list_destroy(&r3);
                    macro_argument_list_destroy(&arguments);
                    throw;
                }

                if (r3.head)
                {
                    r3.head->flags = flags;
                }
                token_list_append_list_at_beginning(&new_list, &r3);
                macro_argument_list_destroy(&arguments);
                token_list_destroy(&r3);
            }
            else
            {
                /*
                 This is a good place to set the level and macro flags
                 because there is always a macro rescan at the end
                */
                new_list.head->level = level;
                new_list.head->flags |= TK_FLAG_MACRO_EXPANDED;

                //OBS: #def macro have newlinew
                //assert(!(new_list.head->flags & TK_FLAG_HAS_NEWLINE_BEFORE));
                prematch(&r, &new_list); //it wasn't macro
            }
        }
    }
    catch
    {
    }

    return r;
}

/*
Performs the comparison ignoring the continuation of the line
TODO do a general review where strcmp is used in lexeme
and replace it with this one.
*/
int lexeme_cmp(const char* s1, const char* s2)
{
    while (*s1 && *s2)
    {

        while ((s1[0] == '\\' && s1[1] == '\n'))
        {
            s1++;
            s1++;
        }


        while (s2[0] == '\\' && s2[1] == '\n')
        {
            s2++;
            s2++;
        }

        if (*s1 != *s2)
            break;

        s1++;
        s2++;
    }

    while ((s1[0] == '\\' && s1[1] == '\n'))
    {
        s1++;
        s1++;
    }


    while (s2[0] == '\\' && s2[1] == '\n')
    {
        s2++;
        s2++;
    }

    return *(const unsigned char*)s1 - *(const unsigned char*)s2;
}

void remove_line_continuation(char* s)
{
    char* pread = s;
    char* pwrite = s;
    while (*pread)
    {
        if (pread[0] == '\\' &&
              (pread[1] == '\n' ||
                  (pread[1] == '\r' && pread[2] == '\n'))
            )
        {
            if (pread[1] == '\r' && pread[2] == '\n')
            {
                pread++;
                pread++;
                pread++;
            }
            else
            {
                pread++;
                pread++;
            }

        }
        else
        {
            *pwrite = *pread;
            pread++;
            pwrite++;
        }
    }
    *pwrite = *pread;
}

struct token_list  copy_replacement_list_core(struct preprocessor_ctx* ctx,
    const struct token_list* list,
    bool new_line_is_space)
{
    //Makes a copy of the tokens by trimming the beginning and end 
    //any space in comments etc. becomes a single space

    struct token_list r = { 0 };
    struct token* _Opt current = list->head;

    //get off all initial whites
    if (!new_line_is_space)
    {
        while (current && token_is_blank(current))
        {
            current = current->next;
        }
    }
    else
    {
        while (current && (token_is_blank(current) || current->type == TK_NEWLINE))
        {
            current = current->next;
        }
    }

    //remove space flag before if present
    bool is_first = true;

    for (; current;)
    {
        if (!new_line_is_space)
        {
            if (current && token_is_blank(current))
            {
                if (current == list->tail)
                    break;

                current = current->next;
                continue;
            }
        }
        else
        {
            if (current && (token_is_blank(current) || current->type == TK_NEWLINE))
            {
                if (current == list->tail)
                    break;

                current = current->next;
                continue;
            }
        }
        struct token* token_added = token_list_clone_and_add(&r, current);


        if (!ctx->options.preprocess_def_macro && token_added->type == TK_PREPROCESSOR_LINE)
        {
            token_added->type = '#';
            free(token_added->lexeme);
            token_added->lexeme = strdup("#");
        }

        if (token_added->flags & TK_FLAG_HAS_NEWLINE_BEFORE)
        {
            token_added->flags = token_added->flags & ~TK_FLAG_HAS_NEWLINE_BEFORE;
            token_added->flags |= TK_FLAG_HAS_SPACE_BEFORE;
        }
        if (is_first)
        {
            token_added->flags = token_added->flags & ~TK_FLAG_HAS_SPACE_BEFORE;
            token_added->flags = token_added->flags & ~TK_FLAG_HAS_NEWLINE_BEFORE;
            is_first = false;
        }
        remove_line_continuation(token_added->lexeme);

        if (current == list->tail)
            break;
        current = current->next;

    }
    return r;
}

struct token_list  copy_replacement_list(struct preprocessor_ctx* ctx,
    const struct token_list* list)
{
    return copy_replacement_list_core(ctx, list, !ctx->options.preprocess_def_macro);
}

struct token_list macro_copy_replacement_list(struct preprocessor_ctx* ctx, struct macro* macro, const struct token* origin)
{
    /*dynamic content macros*/
    if (strcmp(macro->name, "__LINE__") == 0)
    {
        struct tokenizer_ctx tctx = { 0 };
        char line[50] = { 0 };

        assert(origin != NULL);
        snprintf(line, sizeof line, "%d", origin->line);

        struct token_list r = tokenizer(&tctx, line, "", 0, TK_FLAG_NONE);

        token_list_pop_front(&r);

        if (r.head != NULL)
        {
            r.head->flags = 0;
        }
        return r;
    }
    else if (strcmp(macro->name, "__FILE__") == 0)
    {
        char buffer[300] = { 0 };
        if (stringify(origin->token_origin->lexeme, sizeof buffer, buffer) < 0)
        {
            //ops TODO
        }

        struct tokenizer_ctx tctx = { 0 };
        struct token_list r = tokenizer(&tctx, buffer, "", 0, TK_FLAG_NONE);
        token_list_pop_front(&r);
        if (r.head)
        {
            r.head->flags = 0;
        }
        return r;
    }
    else if (strcmp(macro->name, "__COUNTER__") == 0)
    {
        char line[50] = { 0 };
        snprintf(line, sizeof line, "%d", ctx->count_macro_value);
        ctx->count_macro_value++;

        struct tokenizer_ctx tctx = { 0 };
        struct token_list r = tokenizer(&tctx, line, "", 0, TK_FLAG_NONE);
        token_list_pop_front(&r);

        if (r.head != NULL)
        {
            r.head->flags = 0;
        }
        return r;
    }

    return copy_replacement_list(ctx, &macro->replacement_list);
}

void print_literal2(const char* s);


struct token_list expand_macro(struct preprocessor_ctx* ctx,
    struct macro_expanded* _Opt p_list_of_macro_expanded_opt,
    struct macro* macro,
    struct macro_argument_list* arguments,
    int level,
    const struct token* origin)
{
    macro->usage++;

    struct token_list r = { 0 };
    try
    {
        assert(!macro_already_expanded(p_list_of_macro_expanded_opt, macro->name));
        _Opt struct macro_expanded macro_expanded = { 0 };
        macro_expanded.name = macro->name;
        macro_expanded.p_previous = p_list_of_macro_expanded_opt;
        if (macro->is_function)
        {
            struct token_list copy = macro_copy_replacement_list(ctx, macro, origin);
            struct token_list copy2 = replace_macro_arguments(ctx, &macro_expanded, &copy, arguments, origin);
            struct token_list r2 = replacement_list_reexamination(ctx, &macro_expanded, &copy2, level, origin);

            token_list_append_list(&r, &r2);

            token_list_destroy(&copy);
            token_list_destroy(&copy2);
            token_list_destroy(&r2);

            if (ctx->n_errors > 0) throw;
        }
        else
        {
            struct token_list copy = macro_copy_replacement_list(ctx, macro, origin);
            struct token_list r3 = replacement_list_reexamination(ctx, &macro_expanded, &copy, level, origin);
            if (ctx->n_errors > 0)
            {
                token_list_destroy(&copy);
                token_list_destroy(&r3);
                throw;
            }

            token_list_append_list(&r, &r3);
            token_list_destroy(&copy);
            token_list_destroy(&r3);
        }

        if (ctx->n_errors > 0) throw;

        if (ctx->options.preprocess_def_macro && macro->def_macro)
        {
            struct token_list r0 = { 0 };
            token_list_append_list(&r0, &r);

            struct token_list list2 = preprocessor(ctx, &r0, level + 1);
            struct tokenizer_ctx tctx = { 0 };
            const char* _Opt _Owner result = print_preprocessed_to_string2(list2.head);

            token_list_clear(&r);
            r = tokenizer(&tctx, result, "", 0, TK_FLAG_MACRO_EXPANDED);
            struct token_list list3 = copy_replacement_list_core(ctx, &r, true);
            token_list_swap(&list3, &r);
            free((void*)result);
            token_list_destroy(&list2);
            token_list_destroy(&list3);
        }

    }
    catch
    {
    }

    //printf("result=");
    //print_tokens(r.head);
    return r;
}
void print_token(const struct token* p_token);

static struct token_list text_line(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    /*
          text-line:
          pp-tokens_opt new-line
    */
    struct token_list r = { 0 };

    try
    {
        while (input_list->head &&
            input_list->head->type != TK_PREPROCESSOR_LINE)
        {
            struct macro* _Opt macro = NULL;
            struct token* _Opt start_token = input_list->head;
            const struct token* _Opt origin = NULL;

            if (is_active && input_list->head->type == TK_IDENTIFIER)
            {
                origin = input_list->head;
                macro = find_macro(ctx, input_list->head->lexeme);
                if (macro &&
                    macro->is_function &&
                    !preprocessor_token_ahead_is(input_list->head, '('))
                {
                    macro = NULL;
                }

                if (ctx->conditional_inclusion)
                {
                    /*
                     Quando estamos expandindo em condinonal inclusion o defined macro ou defined (macro)
                     não é expandido e é considerado depois
                    */

                    if (r.tail &&
                        r.tail->type == TK_IDENTIFIER &&
                        strcmp(r.tail->lexeme, "defined") == 0)
                    {
                        macro = NULL;
                    }
                    else if (r.tail &&
                        r.tail->type == '(')
                    {
                        struct token* _Opt previous = r.tail->prev;
                        if (previous != NULL &&
                            previous->type == TK_IDENTIFIER &&
                            strcmp(previous->lexeme, "defined") == 0)
                        {
                            macro = NULL;
                        }
                    }
                }
            }
            if (macro)
            {
                // "Tetris" effect
                // #define f(a) a
                // #define F g
                // F(1)
                //quero deixar F(g) na saida.
                //e toda parte de dentro escondida no caso  1
                //F(1)`a` acho que vou imprimir desta forma ou so fundo diferente
                //
                enum token_flags flags = input_list->head->flags;
                struct macro_argument_list arguments = collect_macro_arguments(ctx, macro, input_list, level);
                if (ctx->n_errors > 0)
                {
                    macro_argument_list_destroy(&arguments);
                    throw;
                }


                struct token_list start_macro = expand_macro(ctx, NULL, macro, &arguments, level, origin);
                if (start_macro.head)
                {
                    start_macro.head->flags |= flags;
                }


                //seta nos tokens expandidos da onde eles vieram
                token_list_set_file(&start_macro, start_token->token_origin, start_token->line, start_token->col);

                token_list_append_list_at_beginning(input_list, &start_macro);

                if (ctx->flags & PREPROCESSOR_CTX_FLAGS_ONLY_FINAL)
                {
                }
                else
                {
                    if (level == 0 || INCLUDE_ALL)
                        token_list_append_list(&r, &arguments.tokens);
                }

                //print_tokens(r.head);
                while (macro)
                {
                    macro = NULL;
                    if (input_list->head && input_list->head->type == TK_IDENTIFIER)
                    {
                        macro = find_macro(ctx, input_list->head->lexeme);
                        if (macro && macro->is_function &&
                            !preprocessor_token_ahead_is(input_list->head, '('))
                        {
                            macro = NULL;
                        }
                        if (macro)
                        {
                            // printf("tetris\n");
                            int flags2 = input_list->head->flags;
                            struct macro_argument_list arguments2 = collect_macro_arguments(ctx, macro, input_list, level);
                            if (ctx->n_errors > 0)
                            {
                                macro_argument_list_destroy(&arguments2);
                                macro_argument_list_destroy(&arguments);
                                token_list_destroy(&start_macro);
                                throw;
                            }

                            if (ctx->flags & PREPROCESSOR_CTX_FLAGS_ONLY_FINAL)
                            {
                            }
                            else
                            {
                                if (level == 0 || INCLUDE_ALL)
                                {
                                    token_list_append_list(&r, &arguments2.tokens);
                                }
                            }


                            struct token_list r3 = expand_macro(ctx, NULL, macro, &arguments2, level, origin);
                            if (ctx->n_errors > 0)
                            {
                                macro_argument_list_destroy(&arguments2);
                                token_list_destroy(&r3);
                                macro_argument_list_destroy(&arguments);
                                token_list_destroy(&start_macro);
                                throw;
                            }

                            //seta nos tokens expandidos da onde eles vieram
                            token_list_set_file(&r3, start_token->token_origin, start_token->line, start_token->col);

                            if (r3.head)
                            {
                                r3.head->flags = flags2;
                            }
                            token_list_append_list_at_beginning(input_list, &r3);
                            macro_argument_list_destroy(&arguments2);
                            token_list_destroy(&r3);
                        }
                    }
                }

                macro_argument_list_destroy(&arguments);
                token_list_destroy(&start_macro);

                continue;
                //exit tetris...
                //entao tudo foi expandido desde a primeiroa
            }
            else
            {
                if (input_list->head->flags & TK_FLAG_LINE_CONTINUATION &&
                    !(input_list->head->flags & TK_FLAG_MACRO_EXPANDED)
                    )
                {
                    /*
                       The only place were line-continuation are really necessary is
                       inside preprocessor directives.
                       Here we are inside text-line so we can send a info that
                       here is optional.
                    */
                    if (input_list->head->type == TK_STRING_LITERAL)
                    {
                        preprocessor_diagnostic(W_NOTE, ctx, input_list->head, "you can use \"adjacent\" \"strings\"");
                    }
                    else if (input_list->head->type == TK_LINE_COMMENT)
                        preprocessor_diagnostic(W_COMMENT, ctx, input_list->head, "multi-line //comment");
                    else
                        preprocessor_diagnostic(W_LINE_SLICING, ctx, input_list->head, "unnecessary line-slicing");
                }

                bool blanks = token_is_blank(input_list->head) || input_list->head->type == TK_NEWLINE;
                bool is_final = is_active && !is_never_final(input_list->head->type);

                if (ctx->flags & PREPROCESSOR_CTX_FLAGS_ONLY_FINAL)
                {
                    if (is_final)
                    {
                        prematch(&r, input_list);
                        assert(r.tail != NULL);
                        r.tail->flags |= TK_FLAG_FINAL;
                    }
                    else
                    {
                        token_list_pop_front(input_list);//todo deletar
                    }
                }
                else
                {
                    if (blanks)
                    {
                        if (level == 0 || INCLUDE_ALL)
                        {
                            prematch(&r, input_list);
                        }
                        else
                            token_list_pop_front(input_list);//todo deletar
                    }
                    else
                    {
                        if (level == 0 || INCLUDE_ALL)
                        {
                            prematch(&r, input_list);
                            if (is_final)
                            {
                                assert(r.tail != NULL);
                                r.tail->flags |= TK_FLAG_FINAL;
                            }
                        }
                        else
                        {
                            if (is_final)
                            {
                                prematch(&r, input_list);
                                assert(r.tail != NULL);
                                r.tail->flags |= TK_FLAG_FINAL;
                            }
                            else
                            {
                                token_list_pop_front(input_list);//todo deletar
                            }
                        }
                    }
                }


            }
        }
    }
    catch
    {
    }

    return r;
}

struct token_list group_part(struct preprocessor_ctx* ctx, struct token_list* input_list, bool is_active, int level)
{
    /*
    group-part:
     if-section
     control-line
     text-line
     # non-directive
    */

    assert(input_list->head != NULL);

    if (input_list->head->type == TK_PREPROCESSOR_LINE)
    {
        if (preprocessor_token_ahead_is_identifier(input_list->head, "if") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "ifdef") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "ifndef"))
        {
            return if_section(ctx, input_list, is_active, level);
        }
        else if (preprocessor_token_ahead_is_identifier(input_list->head, "def"))
        {
            //C2Y
            return def_section(ctx, input_list, is_active, level);
        }
        else if (preprocessor_token_ahead_is_identifier(input_list->head, "include") ||
        preprocessor_token_ahead_is_identifier(input_list->head, "include_next") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "embed") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "define") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "undef") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "warning") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "line") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "error") ||
            preprocessor_token_ahead_is_identifier(input_list->head, "pragma") ||
            preprocessor_token_ahead_is(input_list->head, TK_NEWLINE))
        {
            return control_line(ctx, input_list, is_active, level);
        }
        else
        {
            if (is_active)
            {
                struct token* _Opt p_token = preprocessor_look_ahead_core(input_list->head);
                const char* directive_name = p_token ? p_token->lexeme : "";
                /*
                   13 The execution of a non-directive preprocessing directive results in undefined behavior.
                */
                preprocessor_diagnostic(C_ERROR_INVALID_PREPROCESSING_DIRECTIVE,
                    ctx,
                    input_list->head,
                    "invalid preprocessor directive '#%s'\n", directive_name);
            }
            //here I will consume the # inside to make it symmetrical
            return non_directive(ctx, input_list, level);
        }
    }
    return text_line(ctx, input_list, is_active, level);
}


struct token_list preprocessor(struct preprocessor_ctx* ctx, struct token_list* input_list, int level)
{
    struct token_list r = { 0 };
    if (input_list->head == NULL)
    {
        return r;
    }

    if (input_list->head->type == TK_BEGIN_OF_FILE)
    {
        prematch_level(&r, input_list, 1); //sempre coloca
    }

    struct token_list g = group_opt(ctx, input_list, true /*active*/, level);
    token_list_append_list(&r, &g);
    token_list_destroy(&g);
    return r;
}


static void mark_macros_as_used(struct hash_map* map)
{
    /*
     *  Objetivo era alertar macros nao usadas...
     */

    if (map->table != NULL)
    {
        for (int i = 0; i < map->capacity; i++)
        {
            struct map_entry* _Opt pentry = map->table[i];

            while (pentry != NULL)
            {
                assert(pentry->data.p_macro != NULL);
                struct macro* macro = pentry->data.p_macro;
                macro->usage = 1;
                pentry = pentry->next;
            }
        }
    }
}

void check_unused_macros(const struct hash_map* map)
{
    /*
     *  Objetivo era alertar macros nao usadas...
     */

    if (map->table != NULL)
    {
        for (int i = 0; i < map->capacity; i++)
        {
            struct map_entry* _Opt pentry = map->table[i];

            while (pentry != NULL)
            {
                assert(pentry->data.p_macro != NULL);

                struct macro* macro = pentry->data.p_macro;
                if (macro->usage == 0)
                {
                    //TODO adicionar conceito meu codigo , codigo de outros nao vou colocar erro
                    printf("%s not used\n", macro->name);
                }
                pentry = pentry->next;
            }
        }
    }
}

int include_config_header(struct preprocessor_ctx* ctx, const char* file_name)
{
    char local_cakeconfig_path[MAX_PATH] = { 0 };
    snprintf(local_cakeconfig_path, sizeof local_cakeconfig_path, "%s", file_name);
    dirname(local_cakeconfig_path);

    snprintf(local_cakeconfig_path, sizeof local_cakeconfig_path, "%s" CAKE_CFG_FNAME, local_cakeconfig_path);

    char* _Owner _Opt str = read_file(local_cakeconfig_path, true);
    while (str == NULL)
    {
        dirname(local_cakeconfig_path);
        dirname(local_cakeconfig_path);
        if (local_cakeconfig_path[0] == '\0')
            break;
        str = read_file(local_cakeconfig_path, true);
    }


    if (str == NULL)
    {
        //Search cakeconfig at cake executable dir

        char executable_path[MAX_PATH - sizeof(CAKE_CFG_FNAME)] = { 0 };
        get_self_path(executable_path, sizeof(executable_path));
        dirname(executable_path);
        char root_cakeconfig_path[MAX_PATH] = { 0 };
        snprintf(root_cakeconfig_path, sizeof root_cakeconfig_path, "%s" CAKE_CFG_FNAME, executable_path);
        str = read_file(root_cakeconfig_path, true);
    }

    if (str == NULL)
    {
        //"No such file or directory";
        return  ENOENT;
    }

    const enum diagnostic_id w =
        ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings;

    struct tokenizer_ctx tctx = { 0 };
    struct token_list l = tokenizer(&tctx, str, "standard macros inclusion", 0, TK_FLAG_NONE);
    struct token_list l10 = preprocessor(ctx, &l, 0);
    mark_macros_as_used(&ctx->macros);
    token_list_destroy(&l);
    free(str);
    token_list_destroy(&l10);

    /*restore*/
    ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings = w;

    return  0;
}

static bool is_builtin_macro(const char* name)
{
    if (strcmp(name, "__FILE__") == 0)
        return true;

    if (strcmp(name, "__CAKE__") == 0)
        return true;

    return false;
}

void add_standard_macros(struct preprocessor_ctx* ctx)
{
    /*
      This command prints all macros used by gcc
      echo | gcc -dM -E -
    */
    const struct diagnostic w =
        ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index];

    /*we dont want warnings here*/
    ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index] =
        (struct diagnostic){ 0 };

    static char mon[][4] = {
        "Jan", "Feb", "Mar", "Apr", "May", "Jun",
        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
    };

    time_t now = time(NULL);
    struct tm* tm = localtime(&now);

    struct tokenizer_ctx tctx = { 0 };


    char datastr[100] = { 0 };
    snprintf(datastr, sizeof datastr, "#define __DATE__ \"%s %2d %d\"\n", mon[tm->tm_mon], tm->tm_mday, tm->tm_year + 1900);
    struct token_list l1 = tokenizer(&tctx, datastr, "__DATE__ macro inclusion", 0, TK_FLAG_NONE);
    struct token_list tl1 = preprocessor(ctx, &l1, 0);

    token_list_destroy(&tl1);
    token_list_destroy(&l1);

    char timestr[100] = { 0 };
    snprintf(timestr, sizeof timestr, "#define __TIME__ \"%02d:%02d:%02d\"\n", tm->tm_hour, tm->tm_min, tm->tm_sec);
    struct token_list l2 = tokenizer(&tctx, timestr, "__TIME__ macro inclusion", 0, TK_FLAG_NONE);
    struct token_list tl2 = preprocessor(ctx, &l2, 0);

    token_list_destroy(&tl2);
    token_list_destroy(&l2);


    /*
      Some macros are dynamic like __LINE__ they are replaced  at
      macro_copy_replacement_list but they need to be registered here.
    */

    const char* pre_defined_macros_text =
        "#define __CAKE__ 202311L\n"
        "#define __STDC_VERSION__ 202311L\n"
        "#define __FILE__ \"__FILE__\"\n"
        "#define __LINE__ 0\n"
        "#define __COUNTER__ 0\n"
        "#define _CONSOLE\n"
        "#define __STDC_OWNERSHIP__ 1\n" /*cake extension*/
        //"#define __STDC_HOSTED__ " TOSTRING(__STDC_HOSTED__) "\n" /*breaks linux*/
        "#define __STDC_NO_ATOMICS__ " TOSTRING(__STDC_NO_ATOMICS__) "\n"
        "#define __STDC_NO_COMPLEX__  " TOSTRING(__STDC_NO_COMPLEX__) "\n"
        "#define __STDC_NO_THREADS__   " TOSTRING(__STDC_NO_THREADS__) "\n"
        "#define __STDC_NO_VLA__    " TOSTRING(__STDC_NO_VLA__) "\n"
        //"#define __STDC__    " TOSTRING(__STDC__) "\n"



#ifdef __EMSCRIPTEN__
        //include dir on emscripten
        "#pragma dir \"c:/\"\n"
#endif

#ifdef _WIN32

        //see
        //https://learn.microsoft.com/en-us/cpp/preprocessor/predefined-macros?_View=msvc-170
        "#define _WIN32 " TOSTRING(_WIN32) "\n"


#ifdef _WIN64
        "#define _WIN64 " TOSTRING(_WIN64) "\n"
#endif

        "#define _INTEGRAL_MAX_BITS " TOSTRING(_INTEGRAL_MAX_BITS) "\n" /*Use of __int64 should be conditional on the predefined macro _INTEGRAL_MAX_BITS*/

        "#define _MSC_VER " TOSTRING(_MSC_VER) "\n"
        "#define _M_IX86 "  TOSTRING(_M_IX86) "\n"
        "#define __pragma(a)\n"
        ;

#endif

#ifdef __linux__

    //https://gcc.gnu.org/onlinedocs/cpp/Common-Predefined-Macros.html
        /*some gcc stuff need to parse linux headers*/
    "#define __linux__\n"
        "#define __builtin_va_list void* \n"
        "#define __builtin_va_start(a, b)\n"
        "#define __builtin_va_end(a)\n"
        "#define __builtin_va_arg(a, b) ((b)0)\n"
        "#define __builtin_va_copy(a, b)\n"
        "#define __builtin_offsetof(type, member) 0\n"
        //see
        //https://gcc.gnu.org/onlinedocs/gcc/Other-Builtins.html
        //We parse and ignore GCC __attribute__
        //"#define __attribute__(x)\n"
        "#define __x86_64__ " TOSTRING(__x86_64__) "\n"

        "#define __CHAR_BIT__ " TOSTRING(__CHAR_BIT__) "\n"
        "#define __SIZE_TYPE__ " TOSTRING(__SIZE_TYPE__) "\n"
        "#define __PTRDIFF_TYPE__ " TOSTRING(__PTRDIFF_TYPE__) "\n"
        "#define __WCHAR_TYPE__ " TOSTRING(__WCHAR_TYPE__) "\n"
        "#define __WINT_TYPE__ " TOSTRING(__WINT_TYPE__) "\n"
        "#define __INTMAX_TYPE__ " TOSTRING(__INTMAX_TYPE__) "\n"
        "#define __UINTMAX_TYPE__ " TOSTRING(__UINTMAX_TYPE__) "\n"
        "#define __SIG_ATOMIC_TYPE__ " TOSTRING(__SIG_ATOMIC_TYPE__) "\n"
        "#define __INT8_TYPE__ " TOSTRING(__INT8_TYPE__) "\n"
        "#define __INT16_TYPE__ " TOSTRING(__INT16_TYPE__) "\n"
        "#define __INT32_TYPE__ " TOSTRING(__INT32_TYPE__) "\n"
        "#define __INT64_TYPE__ " TOSTRING(__INT64_TYPE__) "\n"
        "#define __UINT8_TYPE__ " TOSTRING(__UINT8_TYPE__) "\n"
        "#define __UINT16_TYPE__ " TOSTRING(__UINT16_TYPE__) "\n"
        "#define __UINT32_TYPE__ " TOSTRING(__UINT32_TYPE__) "\n"
        "#define __UINT64_TYPE__ " TOSTRING(__UINT64_TYPE__) "\n"
        "#define __INT_LEAST8_TYPE__ " TOSTRING(__INT_LEAST8_TYPE__) "\n"
        "#define __INT_LEAST16_TYPE__ " TOSTRING(__INT_LEAST16_TYPE__) "\n"
        "#define __INT_LEAST32_TYPE__ " TOSTRING(__INT_LEAST32_TYPE__) "\n"
        "#define __INT_LEAST64_TYPE__ " TOSTRING(__INT_LEAST64_TYPE__) "\n"
        "#define __UINT_LEAST8_TYPE__ " TOSTRING(__UINT_LEAST8_TYPE__) "\n"
        "#define __UINT_LEAST16_TYPE__ " TOSTRING(__UINT_LEAST16_TYPE__) "\n"
        "#define __UINT_LEAST32_TYPE__ " TOSTRING(__UINT_LEAST32_TYPE__) "\n"
        "#define __UINT_LEAST64_TYPE__ " TOSTRING(__UINT_LEAST64_TYPE__) "\n"
        "#define __INT_FAST8_TYPE__ " TOSTRING(__INT_FAST8_TYPE__) "\n"
        "#define __INT_FAST16_TYPE__ " TOSTRING(__INT_FAST16_TYPE__) "\n"
        "#define __INT_FAST32_TYPE__ " TOSTRING(__INT_FAST32_TYPE__) "\n"
        "#define __INT_FAST64_TYPE__ " TOSTRING(__INT_FAST64_TYPE__) "\n"
        "#define __UINT_FAST8_TYPE__ " TOSTRING(__UINT_FAST8_TYPE__) "\n"
        "#define __UINT_FAST16_TYPE__ " TOSTRING(__UINT_FAST16_TYPE__) "\n"
        "#define __UINT_FAST32_TYPE__ " TOSTRING(__UINT_FAST32_TYPE__) "\n"
        "#define __UINT_FAST64_TYPE__ " TOSTRING(__UINT_FAST64_TYPE__) "\n"
        "#define __INTPTR_TYPE__ " TOSTRING(__INTPTR_TYPE__) "\n"
        "#define __UINTPTR_TYPE__ " TOSTRING(__UINTPTR_TYPE__) "\n"

        "#define __DBL_MAX__ " TOSTRING(__DBL_MAX__) "\n"
        "#define __DBL_MIN__ " TOSTRING(__DBL_MIN__) "\n"
        "#define __FLT_RADIX__ " TOSTRING(__FLT_RADIX__) "\n"
        "#define __FLT_EPSILON__ " TOSTRING(__FLT_EPSILON__) "\n"
        "#define __DBL_EPSILON__ " TOSTRING(__DBL_EPSILON__) "\n"
        "#define __LDBL_EPSILON__ " TOSTRING(__LDBL_EPSILON__) "\n"
        "#define __DBL_DECIMAL_DIG__ " TOSTRING(__DBL_DECIMAL_DIG__) "\n"
        "#define __FLT_EVAL_METHOD__ " TOSTRING(__FLT_EVAL_METHOD__) "\n"
        "#define __FLT_RADIX__ " TOSTRING(__FLT_RADIX__) "\n"

        // gcc -dM -E

        "#define __DBL_MAX_EXP__ " TOSTRING(__DBL_MAX_EXP__) "\n"
        "#define __DECIMAL_DIG__ " TOSTRING(__DECIMAL_DIG__) "\n"
        "#define __FLT_DECIMAL_DIG__ " TOSTRING(__FLT_DECIMAL_DIG__) "\n"


        "#define __FLT_MIN_10_EXP__ " TOSTRING(__FLT_MIN_10_EXP__) "\n"
        "#define __FLT_MIN__ " TOSTRING(__FLT_MIN__) "\n"
        "#define __FLT_MAX__ " TOSTRING(__FLT_MAX__) "\n"
        "#define __FLT_EPSILON__ " TOSTRING(__FLT_EPSILON__) "\n"
        "#define __FLT_DIG__ " TOSTRING(__FLT_DIG__) "\n"
        "#define __FLT_MANT_DIG__ " TOSTRING(__FLT_MANT_DIG__) "\n"
        "#define __FLT_MIN_EXP__ " TOSTRING(__FLT_MIN_EXP__) "\n"
        "#define __FLT_MAX_10_EXP__ " TOSTRING(__FLT_MAX_10_EXP__) "\n"
        "#define __FLT_ROUNDS__ " TOSTRING(__FLT_ROUNDS__) "\n"
        "#define __FLT_EVAL_METHOD__ " TOSTRING(__FLT_EVAL_METHOD__) "\n"
        "#define __FLT_HAS_SUBNORM__ " TOSTRING(__FLT_HAS_SUBNORM__) "\n"

        "#define __FLT_MAX_EXP__ " TOSTRING(__FLT_MAX_EXP__) "\n"
        "#define __FLT_HAS_DENORM__ " TOSTRING(__FLT_HAS_DENORM__) "\n"


        "#define __SCHAR_MAX__ " TOSTRING(__SCHAR_MAX__) "\n"
        "#define __WCHAR_MAX__ " TOSTRING(__WCHAR_MAX__) "\n"
        "#define __SHRT_MAX__ " TOSTRING(__SHRT_MAX__) "\n"
        "#define __INT_MAX__ " TOSTRING(__INT_MAX__) "\n"
        "#define __LONG_MAX__ " TOSTRING(__LONG_MAX__) "\n"
        "#define __LONG_LONG_MAX__ " TOSTRING(__LONG_LONG_MAX__) "\n"
        "#define __WINT_MAX__ " TOSTRING(__WINT_MAX__) "\n"
        "#define __SIZE_MAX__ " TOSTRING(__SIZE_MAX__) "\n"
        "#define __PTRDIFF_MAX__ " TOSTRING(__PTRDIFF_MAX__) "\n"
        "#define __INTMAX_MAX__ " TOSTRING(__INTMAX_MAX__) "\n"
        "#define __UINTMAX_MAX__ " TOSTRING(__UINTMAX_MAX__) "\n"
        "#define __SIG_ATOMIC_MAX__ " TOSTRING(__SIG_ATOMIC_MAX__) "\n"
        "#define __INT8_MAX__ " TOSTRING(__INT8_MAX__) "\n"
        "#define __INT16_MAX__ " TOSTRING(__INT16_MAX__) "\n"
        "#define __INT32_MAX__ " TOSTRING(__INT32_MAX__) "\n"
        "#define __INT64_MAX__ " TOSTRING(__INT64_MAX__) "\n"
        "#define __UINT8_MAX__ " TOSTRING(__UINT8_MAX__) "\n"
        "#define __UINT16_MAX__ " TOSTRING(__UINT16_MAX__) "\n"
        "#define __UINT32_MAX__ " TOSTRING(__UINT32_MAX__) "\n"
        "#define __UINT64_MAX__ " TOSTRING(__UINT64_MAX__) "\n"
        "#define __INT_LEAST8_MAX__ " TOSTRING(__INT_LEAST8_MAX__) "\n"
        "#define __INT_LEAST16_MAX__ " TOSTRING(__INT_LEAST16_MAX__) "\n"
        "#define __INT_LEAST32_MAX__ " TOSTRING(__INT_LEAST32_MAX__) "\n"
        "#define __INT_LEAST64_MAX__ " TOSTRING(__INT_LEAST64_MAX__) "\n"
        "#define __UINT_LEAST8_MAX__ " TOSTRING(__UINT_LEAST8_MAX__) "\n"
        "#define __UINT_LEAST16_MAX__ " TOSTRING(__UINT_LEAST16_MAX__) "\n"
        "#define __UINT_LEAST32_MAX__ " TOSTRING(__UINT_LEAST32_MAX__) "\n"
        "#define __UINT_LEAST64_MAX__ " TOSTRING(__UINT_LEAST64_MAX__) "\n"
        "#define __INT_FAST8_MAX__ " TOSTRING(__INT_FAST8_MAX__) "\n"
        "#define __INT_FAST16_MAX__ " TOSTRING(__INT_FAST16_MAX__) "\n"
        "#define __INT_FAST32_MAX__ " TOSTRING(__INT_FAST32_MAX__) "\n"
        "#define __INT_FAST64_MAX__ " TOSTRING(__INT_FAST64_MAX__) "\n"
        "#define __UINT_FAST8_MAX__ " TOSTRING(__UINT_FAST8_MAX__) "\n"
        "#define __UINT_FAST16_MAX__ " TOSTRING(__UINT_FAST16_MAX__) "\n"
        "#define __UINT_FAST32_MAX__ " TOSTRING(__UINT_FAST32_MAX__) "\n"
        "#define __UINT_FAST64_MAX__ " TOSTRING(__UINT_FAST64_MAX__) "\n"
        "#define __INTPTR_MAX__ " TOSTRING(__INTPTR_MAX__) "\n"
        "#define __UINTPTR_MAX__ " TOSTRING(__UINTPTR_MAX__) "\n"
        "#define __WCHAR_MIN__ " TOSTRING(__WCHAR_MIN__) "\n"
        "#define __WINT_MIN__ " TOSTRING(__WINT_MIN__) "\n"
        "#define __SIG_ATOMIC_MIN__ " TOSTRING(__SIG_ATOMIC_MIN__) "\n"

        "#define __INT8_C " TOSTRING(__SIG_ATOMIC_MIN__) "\n"
        "#define __INT16_C " TOSTRING(__INT16_C) "\n"
        "#define __INT32_C " TOSTRING(__INT32_C) "\n"
        "#define __INT64_C " TOSTRING(__INT64_C) "\n"
        "#define __UINT8_C " TOSTRING(__UINT8_C) "\n"
        "#define __UINT16_C " TOSTRING(__UINT16_C) "\n"
        "#define __UINT32_C " TOSTRING(__UINT32_C) "\n"
        "#define __UINT64_C " TOSTRING(__UINT64_C) "\n"
        "#define __INTMAX_C " TOSTRING(__INTMAX_C) "\n"
        "#define __UINTMAX_C " TOSTRING(__UINTMAX_C) "\n"

        "#define __SCHAR_WIDTH__ " TOSTRING(__SCHAR_WIDTH__) "\n"
        "#define __SHRT_WIDTH__ " TOSTRING(__SHRT_WIDTH__) "\n"
        "#define __INT_WIDTH__ " TOSTRING(__INT_WIDTH__) "\n"
        "#define __LONG_WIDTH__ " TOSTRING(__LONG_WIDTH__) "\n"
        "#define __LONG_LONG_WIDTH__ " TOSTRING(__LONG_LONG_WIDTH__) "\n"
        "#define __PTRDIFF_WIDTH__ " TOSTRING(__PTRDIFF_WIDTH__) "\n"
        "#define __SIG_ATOMIC_WIDTH__ " TOSTRING(__SIG_ATOMIC_WIDTH__) "\n"
        "#define __SIZE_WIDTH__ " TOSTRING(__SIZE_WIDTH__) "\n"
        "#define __WCHAR_WIDTH__ " TOSTRING(__WCHAR_WIDTH__) "\n"
        "#define __WINT_WIDTH__ " TOSTRING(__WINT_WIDTH__) "\n"
        "#define __INT_LEAST8_WIDTH__ " TOSTRING(__INT_LEAST8_WIDTH__) "\n"
        "#define __INT_LEAST16_WIDTH__ " TOSTRING(__INT_LEAST16_WIDTH__) "\n"
        "#define __INT_LEAST32_WIDTH__ " TOSTRING(__INT_LEAST32_WIDTH__) "\n"
        "#define __INT_LEAST64_WIDTH__ " TOSTRING(__INT_LEAST64_WIDTH__) "\n"
        "#define __INT_FAST8_WIDTH__ " TOSTRING(__INT_FAST8_WIDTH__) "\n"
        "#define __INT_FAST16_WIDTH__ " TOSTRING(__INT_FAST16_WIDTH__) "\n"
        "#define __INT_FAST32_WIDTH__ " TOSTRING(__INT_FAST32_WIDTH__) "\n"
        "#define __INT_FAST64_WIDTH__ " TOSTRING(__INT_FAST64_WIDTH__) "\n"
        "#define __INTPTR_WIDTH__ " TOSTRING(__INTPTR_WIDTH__) "\n"
        "#define __INTMAX_WIDTH__ " TOSTRING(__INTMAX_WIDTH__) "\n"


        "#define __SIZEOF_INT__ " TOSTRING(__SIZEOF_INT__) "\n"
        "#define __SIZEOF_LONG__ " TOSTRING(__SIZEOF_LONG__) "\n"
        "#define __SIZEOF_LONG_LONG__ " TOSTRING(__SIZEOF_LONG_LONG__) "\n"
        "#define __SIZEOF_SHORT__ " TOSTRING(__SIZEOF_SHORT__) "\n"
        "#define __SIZEOF_POINTER__ " TOSTRING(__SIZEOF_POINTER__) "\n"
        "#define __SIZEOF_FLOAT__ " TOSTRING(__SIZEOF_FLOAT__) "\n"
        "#define __SIZEOF_DOUBLE__ " TOSTRING(__SIZEOF_DOUBLE__) "\n"
        "#define __SIZEOF_LONG_DOUBLE__ " TOSTRING(__SIZEOF_LONG_DOUBLE__) "\n"
        "#define __SIZEOF_SIZE_T__ " TOSTRING(__SIZEOF_SIZE_T__) "\n"
        "#define __SIZEOF_WCHAR_T__ " TOSTRING(__SIZEOF_WCHAR_T__) "\n"
        "#define __SIZEOF_WINT_T__ " TOSTRING(__SIZEOF_WINT_T__) "\n"
        "#define __SIZEOF_PTRDIFF_T__ " TOSTRING(__SIZEOF_PTRDIFF_T__) "\n"
#endif
        "\n";

    struct token_list l = tokenizer(&tctx, pre_defined_macros_text, "standard macros inclusion", 0, TK_FLAG_NONE);
    struct token_list l10 = preprocessor(ctx, &l, 0);

    //nao quer ver warning de nao usado nestas macros padrao
    mark_macros_as_used(&ctx->macros);
    token_list_destroy(&l);
    token_list_destroy(&l10);

    /*restore*/
    ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index] = w;
}




const char* get_token_name(enum token_type tk)
{
    switch (tk)
    {
    case TK_NONE: return "TK_NONE";
    case TK_NEWLINE: return "TK_NEWLINE";
    case TK_WHITE_SPACE: return "TK_WHITE_SPACE";
    case TK_EXCLAMATION_MARK: return "TK_EXCLAMATION_MARK";
    case TK_QUOTATION_MARK: return "TK_QUOTATION_MARK";
    case TK_NUMBER_SIGN: return "TK_NUMBER_SIGN";
    case TK_DOLLAR_SIGN: return "TK_DOLLAR_SIGN";
    case TK_PERCENT_SIGN: return "TK_PERCENT_SIGN";
    case TK_AMPERSAND: return "TK_AMPERSAND";
    case TK_APOSTROPHE: return "TK_APOSTROPHE";
    case TK_LEFT_PARENTHESIS: return "TK_LEFT_PARENTHESIS";
    case TK_RIGHT_PARENTHESIS: return "TK_RIGHT_PARENTHESIS";
    case TK_ASTERISK: return "TK_ASTERISK";
    case TK_PLUS_SIGN: return "TK_PLUS_SIGN";
    case TK_COMMA: return "TK_COMMA";
    case TK_HYPHEN_MINUS: return "TK_HYPHEN_MINUS";
    case TK_FULL_STOP: return "TK_FULL_STOP";
    case TK_SOLIDUS: return "TK_SOLIDUS";
    case TK_COLON: return "TK_COLON";
    case TK_SEMICOLON: return "TK_SEMICOLON";
    case TK_LESS_THAN_SIGN: return "TK_LESS_THAN_SIGN";
    case TK_EQUALS_SIGN: return "TK_EQUALS_SIGN";
    case TK_GREATER_THAN_SIGN: return "TK_GREATER_THAN_SIGN";
    case TK_QUESTION_MARK: return "TK_QUESTION_MARK";
    case TK_COMMERCIAL_AT: return "TK_COMMERCIAL_AT";
    case TK_LEFT_SQUARE_BRACKET: return "TK_LEFT_SQUARE_BRACKET";
    case TK_REVERSE_SOLIDUS: return "TK_REVERSE_SOLIDUS";
    case TK_RIGHT_SQUARE_BRACKET: return "TK_RIGHT_SQUARE_BRACKET";
    case TK_CIRCUMFLEX_ACCENT: return "TK_CIRCUMFLEX_ACCENT";
    case TK_FLOW_LINE: return "TK_FLOW_LINE";
    case TK_GRAVE_ACCENT: return "TK_GRAVE_ACCENT";
    case TK_LEFT_CURLY_BRACKET: return "TK_LEFT_CURLY_BRACKET";
    case TK_VERTICAL_LINE: return "TK_VERTICAL_LINE";
    case TK_RIGHT_CURLY_BRACKET: return "TK_RIGHT_CURLY_BRACKET";
    case TK_TILDE: return "TK_TILDE";
    case TK_PREPROCESSOR_LINE: return "TK_PREPROCESSOR_LINE";
    case TK_PRAGMA: return "TK_PRAGMA";
    case TK_STRING_LITERAL: return "TK_STRING_LITERAL";
    case TK_CHAR_CONSTANT: return "TK_CHAR_CONSTANT";
    case TK_LINE_COMMENT: return "TK_LINE_COMMENT";
    case TK_COMMENT: return "TK_COMMENT";
    case TK_PPNUMBER: return "TK_PPNUMBER";

    case TK_KEYWORD_MSVC__PTR32:return "TK_KEYWORD_MSVC__PTR32";
    case TK_KEYWORD_MSVC__PTR64:return "TK_KEYWORD_MSVC__PTR64";

    case ANY_OTHER_PP_TOKEN: return "ANY_OTHER_PP_TOKEN"; //@ por ex

        /*PPNUMBER sao convertidos para constantes antes do parse*/
    case TK_COMPILER_DECIMAL_CONSTANT: return "TK_COMPILER_DECIMAL_CONSTANT";
    case TK_COMPILER_OCTAL_CONSTANT: return "TK_COMPILER_OCTAL_CONSTANT";
    case TK_COMPILER_HEXADECIMAL_CONSTANT: return "TK_COMPILER_HEXADECIMAL_CONSTANT";
    case TK_COMPILER_BINARY_CONSTANT: return "TK_COMPILER_BINARY_CONSTANT";
    case TK_COMPILER_DECIMAL_FLOATING_CONSTANT: return "TK_COMPILER_DECIMAL_FLOATING_CONSTANT";
    case TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT: return "TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT";


    case TK_PLACEMARKER: return "TK_PLACEMARKER";

    case TK_BLANKS: return "TK_BLANKS";
    case TK_PLUSPLUS: return "TK_PLUSPLUS";
    case TK_MINUSMINUS: return "TK_MINUSMINUS";
    case TK_ARROW: return "TK_ARROW";
    case TK_SHIFTLEFT: return "TK_SHIFTLEFT";
    case TK_SHIFTRIGHT: return "TK_SHIFTRIGHT";
    case TK_LOGICAL_OPERATOR_OR: return "TK_LOGICAL_OPERATOR_OR";
    case TK_LOGICAL_OPERATOR_AND: return "TK_LOGICAL_OPERATOR_AND";

    case TK_MACRO_CONCATENATE_OPERATOR: return "TK_MACRO_CONCATENATE_OPERATOR";

    case TK_IDENTIFIER: return "TK_IDENTIFIER";
    case TK_IDENTIFIER_RECURSIVE_MACRO: return "TK_IDENTIFIER_RECURSIVE_MACRO"; /*usado para evitar recursao expansao macro*/

    case TK_BEGIN_OF_FILE: return "TK_BEGIN_OF_FILE";

        //C23 keywords
    case TK_KEYWORD_AUTO: return "TK_KEYWORD_AUTO";
    case TK_KEYWORD_BREAK: return "TK_KEYWORD_BREAK";
    case TK_KEYWORD_CASE: return "TK_KEYWORD_CASE";
    case TK_KEYWORD_CONSTEXPR: return "TK_KEYWORD_CONSTEXPR";
    case TK_KEYWORD_CHAR: return "TK_KEYWORD_CHAR";
    case TK_KEYWORD_CONST: return "TK_KEYWORD_CONST";
    case TK_KEYWORD_CONTINUE: return "TK_KEYWORD_CONTINUE";
    case TK_KEYWORD_CAKE_CATCH: return "TK_KEYWORD_CAKE_CATCH"; /*extension*/
    case TK_KEYWORD_DEFAULT: return "TK_KEYWORD_DEFAULT";
    case TK_KEYWORD_DO: return "TK_KEYWORD_DO";
    case TK_KEYWORD_DEFER: return "TK_KEYWORD_DEFER"; /*extension*/
    case TK_KEYWORD_DOUBLE: return "TK_KEYWORD_DOUBLE";
    case TK_KEYWORD_ELSE: return "TK_KEYWORD_ELSE";
    case TK_KEYWORD_ENUM: return "TK_KEYWORD_ENUM";
    case TK_KEYWORD_EXTERN: return "TK_KEYWORD_EXTERN";
    case TK_KEYWORD_FLOAT: return "TK_KEYWORD_FLOAT";
    case TK_KEYWORD_FOR: return "TK_KEYWORD_FOR";
    case TK_KEYWORD_GOTO: return "TK_KEYWORD_GOTO";
    case TK_KEYWORD_IF: return "TK_KEYWORD_IF";
    case TK_KEYWORD_INLINE: return "TK_KEYWORD_INLINE";
    case TK_KEYWORD_INT: return "TK_KEYWORD_INT";
    case TK_KEYWORD_LONG: return "TK_KEYWORD_LONG";
    case TK_KEYWORD_MSVC__INT8: return "TK_KEYWORD_MSVC__INT8";
    case TK_KEYWORD_MSVC__INT16: return "TK_KEYWORD_MSVC__INT16";
    case TK_KEYWORD_MSVC__INT32: return "TK_KEYWORD_MSVC__INT32";
    case TK_KEYWORD_MSVC__INT64: return "TK_KEYWORD_MSVC__INT64";


    case TK_KEYWORD_REGISTER: return "TK_KEYWORD_REGISTER";
    case TK_KEYWORD_RESTRICT: return "TK_KEYWORD_RESTRICT";
    case TK_KEYWORD_RETURN: return "TK_KEYWORD_RETURN";
    case TK_KEYWORD_SHORT: return "TK_KEYWORD_SHORT";
    case TK_KEYWORD_SIGNED: return "TK_KEYWORD_SIGNED";
    case TK_KEYWORD_SIZEOF: return "TK_KEYWORD_SIZEOF";

    case TK_KEYWORD_STATIC: return "TK_KEYWORD_STATIC";
    case TK_KEYWORD_STRUCT: return "TK_KEYWORD_STRUCT";
    case TK_KEYWORD_SWITCH: return "TK_KEYWORD_SWITCH";
    case TK_KEYWORD_TYPEDEF: return "TK_KEYWORD_TYPEDEF";
    case TK_KEYWORD_CAKE_TRY: return "TK_KEYWORD_CAKE_TRY"; /*extension*/
    case TK_KEYWORD_CAKE_THROW: return "TK_KEYWORD_CAKE_THROW"; /*extension*/
    case TK_KEYWORD_UNION: return "TK_KEYWORD_UNION";
    case TK_KEYWORD_UNSIGNED: return "TK_KEYWORD_UNSIGNED";
    case TK_KEYWORD_VOID: return "TK_KEYWORD_VOID";
    case TK_KEYWORD_VOLATILE: return "TK_KEYWORD_VOLATILE";
    case TK_KEYWORD_WHILE: return "TK_KEYWORD_WHILE";

    case TK_KEYWORD__ALIGNAS: return "TK_KEYWORD__ALIGNAS";
    case TK_KEYWORD__ALIGNOF: return "TK_KEYWORD__ALIGNOF";
    case TK_KEYWORD__ATOMIC: return "TK_KEYWORD__ATOMIC";

        //#ifdef _WIN32
    case TK_KEYWORD_MSVC__FASTCALL: return "TK_KEYWORD_MSVC__FASTCALL";
    case TK_KEYWORD_MSVC__STDCALL:return "TK_KEYWORD_MSVC__STDCALL";
    case TK_KEYWORD_MSVC__CDECL:return "TK_KEYWORD_MSVC__CDECL";
    case TK_KEYWORD_MSVC__DECLSPEC:return "TK_KEYWORD_MSVC__DECLSPEC";
        //#endif
    case TK_KEYWORD__ASM: return "TK_KEYWORD__ASM";
        //end microsoft
    case TK_KEYWORD__BOOL: return "TK_KEYWORD__BOOL";
    case TK_KEYWORD__COMPLEX: return "TK_KEYWORD__COMPLEX";
    case TK_KEYWORD__DECIMAL128: return "TK_KEYWORD__DECIMAL128";
    case TK_KEYWORD__DECIMAL32: return "TK_KEYWORD__DECIMAL32";
    case TK_KEYWORD__DECIMAL64: return "TK_KEYWORD__DECIMAL64";
    case TK_KEYWORD__GENERIC: return "TK_KEYWORD__GENERIC";
    case TK_KEYWORD__IMAGINARY: return "TK_KEYWORD__IMAGINARY";
    case TK_KEYWORD__NORETURN: return "TK_KEYWORD__NORETURN";
    case TK_KEYWORD__STATIC_ASSERT: return "TK_KEYWORD__STATIC_ASSERT";
    case TK_KEYWORD_ASSERT: return "TK_KEYWORD_ASSERT"; /*extension*/
    case TK_KEYWORD__THREAD_LOCAL: return "TK_KEYWORD__THREAD_LOCAL";

    case TK_KEYWORD_TYPEOF: return "TK_KEYWORD_TYPEOF"; /*C23*/

    case TK_KEYWORD_TRUE: return "TK_KEYWORD_TRUE";  /*C23*/
    case TK_KEYWORD_FALSE: return "TK_KEYWORD_FALSE";  /*C23*/
    case TK_KEYWORD_NULLPTR: return "TK_KEYWORD_NULLPTR";  /*C23*/
    case TK_KEYWORD_TYPEOF_UNQUAL: return "TK_KEYWORD_TYPEOF_UNQUAL"; /*C23*/
    case TK_KEYWORD__BITINT: return "TK_KEYWORD__BITINT";  /*C23*/



        /*cake extension*/
    case TK_KEYWORD_CAKE_OWNER: return "TK_KEYWORD_CAKE_OWNER";
    case TK_KEYWORD__CTOR: return "TK_KEYWORD__OUT";
    case TK_KEYWORD__DTOR: return "TK_KEYWORD__OBJ_OWNER";
    case TK_KEYWORD_CAKE_VIEW: return "TK_KEYWORD_CAKE_VIEW";
    case TK_KEYWORD_CAKE_OPT: return "TK_KEYWORD_CAKE_OPT";


        /*extension compile time functions*/
    case TK_KEYWORD_CAKE_STATIC_DEBUG: return "TK_KEYWORD_CAKE_STATIC_DEBUG"; /*extension*/
    case TK_KEYWORD_CAKE_STATIC_DEBUG_EX: return "TK_KEYWORD_CAKE_STATIC_DEBUG_EX"; /*extension*/
    case TK_KEYWORD_STATIC_STATE: return "TK_KEYWORD_STATIC_STATE"; /*extension*/
    case TK_KEYWORD_STATIC_SET: return "TK_KEYWORD_STATIC_SET"; /*extension*/

        /*https://en.cppreference.com/w/cpp/header/type_traits*/

    case TK_KEYWORD_IS_POINTER: return "TK_KEYWORD_IS_POINTER";
    case TK_KEYWORD_IS_LVALUE: return "TK_KEYWORD_IS_LVALUE";
    case TK_KEYWORD_IS_CONST: return "TK_KEYWORD_IS_CONST";
    case TK_KEYWORD_IS_OWNER: return "TK_KEYWORD_IS_OWNER";
    case TK_KEYWORD_IS_ARRAY: return "TK_KEYWORD_IS_ARRAY";
    case TK_KEYWORD_IS_FUNCTION: return "TK_KEYWORD_IS_FUNCTION";
    case TK_KEYWORD_IS_SCALAR: return "TK_KEYWORD_IS_SCALAR";
    case TK_KEYWORD_IS_ARITHMETIC: return "TK_KEYWORD_IS_ARITHMETIC";
    case TK_KEYWORD_IS_FLOATING_POINT: return "TK_KEYWORD_IS_FLOATING_POINT";
    case TK_KEYWORD_IS_INTEGRAL: return "TK_KEYWORD_IS_INTEGRAL";
    case TK_PRAGMA_END: return "TK_PRAGMA_END";
    case TK_KEYWORD__COUNTOF: return "TK_KEYWORD__COUNTOF";
    case TK_PLUS_ASSIGN: return "TK_PLUS_ASSIGN";
    case TK_MINUS_ASSIGN: return "TK_MINUS_ASSIGN";
    case TK_MULTI_ASSIGN: return "TK_MULTI_ASSIGN";
    case TK_DIV_ASSIGN: return "TK_DIV_ASSIGN";
    case TK_MOD_ASSIGN: return "TK_MOD_ASSIGN";
    case TK_SHIFT_LEFT_ASSIGN: return "TK_SHIFT_LEFT_ASSIGN";
    case TK_SHIFT_RIGHT_ASSIGN: return "TK_SHIFT_RIGHT_ASSIGN";
    case TK_AND_ASSIGN: return "TK_AND_ASSIGN";
    case TK_OR_ASSIGN: return "TK_OR_ASSIGN";
    case TK_NOT_ASSIGN: return "TK_NOT_ASSIGN";

    }
    return "TK_X_MISSING_NAME";
};


int stringify(const char* input, int n, char output[])
{
    int count = 0;
    if (count < n)
        output[count++] = '"';

    const char* p = input;
    while (*p)
    {
        if (*p == '\"' ||
            *p == '\\')
        {
            if (count < n)
                output[count++] = '\\';

            if (count < n)
                output[count++] = *p;
            p++;
        }
        else
        {
            if (count < n)
                output[count++] = *p;
            p++;
        }
    }

    if (count < n)
        output[count++] = '"';
    if (count < n)
        output[count++] = 0;

    if (count >= n)
        return -count;

    return count;
}


void print_literal(const char* _Opt s)
{
    if (s == NULL)
    {
        printf("\"");
        printf("\"");
        return;
    }
    printf("\"");
    while (*s)
    {
        switch (*s)
        {
        case '\n':
            printf("\\n");
            break;
        default:
            printf("%c", *s);
        }
        s++;
    }
    printf("\"");
}


const char* _Owner _Opt get_code_as_we_see_plus_macros(const struct token_list* list)
{
    struct osstream ss = { 0 };
    struct token* _Opt current = list->head;
    while (current)
    {
        if (current->level == 0 &&
            current->type != TK_BEGIN_OF_FILE)
        {
            if (current->flags & TK_FLAG_MACRO_EXPANDED)
                ss_fprintf(&ss, LIGHTCYAN);
            else
                ss_fprintf(&ss, WHITE);
            ss_fprintf(&ss, "%s", current->lexeme);
            ss_fprintf(&ss, RESET);
        }
        current = current->next;
    }

    const char* _Owner _Opt cstr = ss.c_str;
    ss.c_str = NULL; /*MOVED*/

    ss_close(&ss);

    return cstr;
}

/*useful to debug visit.c*/
void print_code_as_we_see(const struct token_list* list, bool remove_comments)
{
    if (list->head == NULL || list->tail == NULL)
    {
        return;
    }

    struct token* _Opt current = list->head;
    while (current && current != list->tail->next)
    {
        if (current->level == 0 &&
            !(current->flags & TK_FLAG_MACRO_EXPANDED) &&
            !(current->flags & TK_C_BACKEND_FLAG_HIDE) &&
            current->type != TK_BEGIN_OF_FILE)
        {
            if ((current->flags & TK_FLAG_HAS_SPACE_BEFORE) &&
                (current->prev != NULL && current->prev->type != TK_BLANKS))
            {
                //if an expanded macro is shown it does not have spaces so we insert
                printf(" ");
            }

            if (remove_comments)
            {
                if (current->type == TK_LINE_COMMENT)
                    printf("\n");
                else if (current->type == TK_COMMENT)
                    printf(" ");
                else
                    printf("%s", current->lexeme);
            }
            else
            {
                printf("%s", current->lexeme);
            }
        }
        current = current->next;
    }
}
const char* _Owner _Opt get_code_as_we_see(const struct token_list* list, bool remove_comments)
{
    if (list->head == NULL || list->tail == NULL)
        return NULL;

    struct osstream ss = { 0 };
    struct token* _Opt current = list->head;
    while (current && current != list->tail->next)
    {
        if (current->level == 0 &&
            !(current->flags & TK_FLAG_MACRO_EXPANDED) &&
            !(current->flags & TK_C_BACKEND_FLAG_HIDE) &&
            current->type != TK_BEGIN_OF_FILE)
        {
            if ((current->flags & TK_FLAG_HAS_SPACE_BEFORE) &&
                (current->prev != NULL && current->prev->type != TK_BLANKS))
            {
                //if an expanded macro is shown it has no spaces so we insert
                ss_fprintf(&ss, " ");
            }

            if (remove_comments)
            {
                if (current->type == TK_LINE_COMMENT)
                    ss_fprintf(&ss, "\n");
                else if (current->type == TK_COMMENT)
                    ss_fprintf(&ss, " ");
                else
                    ss_fprintf(&ss, "%s", current->lexeme);
            }
            else
            {
                ss_fprintf(&ss, "%s", current->lexeme);
            }
        }
        current = current->next;
    }

    const char* _Owner _Opt cstr = ss.c_str;
    ss.c_str = NULL; /*MOVED*/

    ss_close(&ss);

    return cstr;
}


const char* _Owner _Opt get_code_as_compiler_see(const struct token_list* list)
{
    if (list->head == NULL || list->tail == NULL)
    {
        return NULL;
    }

    struct osstream ss = { 0 };


    struct token* _Opt current = list->head;
    while (current && current != list->tail->next)
    {
        if (!(current->flags & TK_C_BACKEND_FLAG_HIDE) &&
            current->type != TK_BEGIN_OF_FILE &&
            (current->flags & TK_FLAG_FINAL))
        {
            if (current->flags & TK_FLAG_HAS_SPACE_BEFORE)
                ss_fprintf(&ss, " ");

            if (current->flags & TK_FLAG_HAS_NEWLINE_BEFORE)
                ss_fprintf(&ss, "\n");

            if (current->type == TK_LINE_COMMENT)
                ss_fprintf(&ss, "\n");
            else if (current->type == TK_COMMENT)
                ss_fprintf(&ss, " ");
            else
                ss_fprintf(&ss, "%s", current->lexeme);
        }
        current = current->next;
    }

    return ss.c_str;
}

const char* _Owner _Opt print_preprocessed_to_string2(const struct token* _Opt p_token)
{
    /*
      * At level > 0 (i.e. inside the includes)
      * This function prints the tokens as the compiler sees them
      * and inserts a space or line break to represent
      * the separation between tokens.

      * At level 0 (main file) it prints spaces, comments
      * etc. and inserts spaces in the macro expansion.
  */

    if (p_token == NULL)
        return strdup("(null)");

    struct osstream ss = { 0 };
    const struct token* _Opt current = p_token;
    while (current)
    {

        //We ignored the line continuation and it can appear anywhere in the lexemes. 
        //instead of removing it, you could just skip it when printing
        remove_line_continuation(current->lexeme);

        if (current->flags & TK_FLAG_FINAL)
        {
            if (current->level > 0)
            {
                //at the include levels we may be ignoring all
                //the spaces. in this case it is necessary to include them so as not to add the tokens
                if ((current->flags & TK_FLAG_HAS_NEWLINE_BEFORE))
                    ss_fprintf(&ss, "\n");
                else if ((current->flags & TK_FLAG_HAS_SPACE_BEFORE))
                    ss_fprintf(&ss, " ");
            }
            else
            {
                /*
                 at level 0 we print the spaces.. however in the case of macros
                 it is necessary to put a space because it does not exist.
                */
                if (current->flags & TK_FLAG_MACRO_EXPANDED)
                {
                    if ((current->flags & TK_FLAG_HAS_SPACE_BEFORE))
                        ss_fprintf(&ss, " ");
                }
            }

            if (current->lexeme[0] != '\0')
            {
                ss_fprintf(&ss, "%s", current->lexeme);
            }

            current = current->next;
        }
        else
        {
            if (current->level == 0)
            {
                if (current->type == TK_BLANKS ||
                    current->type == TK_NEWLINE)
                {
                    ss_fprintf(&ss, "%s", current->lexeme);
                }
            }

            current = current->next;
        }
    }

    return ss.c_str;
}

const char* _Owner _Opt print_preprocessed_to_string(const struct token* p_token)
{
    /*
    * Esta funcao imprime os tokens como o compilador ve
    * e insere um espaco ou quebra de linha para poder representar
    * a separacao entre os tokens.
    */

    struct osstream ss = { 0 };
    const struct token* _Opt current = p_token;

    /*
    * Ignora tudo o que é espaço no início
    */
    while (!(current->flags & TK_FLAG_FINAL) ||
        current->type == TK_BLANKS ||
        current->type == TK_COMMENT ||
        current->type == TK_LINE_COMMENT ||
        current->type == TK_NEWLINE ||
        current->type == TK_PREPROCESSOR_LINE)
    {
        current = current->next;
        if (current == NULL)
            return ss.c_str; /*MOVED*/
    }

    bool first = true;
    while (current)
    {
        assert(current->token_origin != NULL);
        if (current->flags & TK_FLAG_FINAL)
        {
            if (!first && current->flags & TK_FLAG_HAS_NEWLINE_BEFORE)
                ss_fprintf(&ss, "\n");
            else if (!first && current->flags & TK_FLAG_HAS_SPACE_BEFORE)
                ss_fprintf(&ss, " ");
            if (current->lexeme[0] != '\0')
                ss_fprintf(&ss, "%s", current->lexeme);
            first = false;
            current = current->next;
        }
        else
        {
            current = current->next;
        }
    }

    return ss.c_str; /*MOVED*/
}

void print_preprocessed(const struct token* p_token)
{
    const char* _Owner _Opt s = print_preprocessed_to_string(p_token);
    if (s)
    {
        printf("%s", s);
        free((void* _Owner)s);
    }
}

static bool is_screaming_case(const char* text)
{

    bool screaming_case = false;

    while (*text)
    {
        if ((*text >= 'A' && *text <= 'Z') ||
            (*text >= '0' && *text <= '9') ||
            (*text == '_'))
        {
            //ok
            screaming_case = true;
        }
        else
            return false;
        text++;
    }

    return screaming_case;
}

void print_all_macros(const struct preprocessor_ctx* prectx)
{
    for (int i = 0; i < prectx->macros.capacity; i++)
    {
        struct map_entry* _Opt entry = prectx->macros.table[i];
        if (entry == NULL) continue;
        assert(entry->data.p_macro != NULL);

        struct macro* macro = entry->data.p_macro;
        printf("#define %s", macro->name);
        if (macro->is_function)
        {
            printf("(");

            struct macro_parameter* _Opt parameter = macro->parameters;
            while (parameter)
            {
                printf("%s", parameter->name);
                if (parameter->next)
                    printf(",");
                parameter = parameter->next;
            }
            printf(")");
        }
        printf(" ");

        struct token* _Opt token = macro->replacement_list.head;
        while (token)
        {
            printf("%s", token->lexeme);

            if (token == macro->replacement_list.tail)
                break;

            token = token->next;
        }
        printf("\n");
    }
}
void naming_convention_macro(struct preprocessor_ctx* ctx, struct token* token)
{
    if (!is_screaming_case(token->lexeme))
    {
        preprocessor_diagnostic(W_NOTE, ctx, token, "use SCREAMING_CASE for macros");
    }

}


#ifdef TEST


//#pragma once

extern int g_unit_test_error_count;
extern int g_unit_test_success_count;
static void assert_func(int condition, const char* func, const char* file, int line, const char* message)
{
    if (!condition)
    {
        const char* pos = file;
        const char* p = file;
        while (*p)
        {
            if (*p == '/' || *p == '\\')
                pos = p;
            p++;
        }
        
        if (*pos == '/' || *pos == '\\')
            pos++;

        g_unit_test_error_count++;
        printf("\x1b[97m" "%s:%d:0:" "\x1b[91m" " test failed:" "\x1b[0m" " function '%s'\n", pos, line, func);
        
        char buffer[20] = { 0 };
        int n = snprintf(buffer, sizeof buffer, "%d", line);        
        printf(" %s |", buffer);
        printf("    assert(%s);\n", message);
        printf(" %*s |\n", n, " ");
    }
    else
    {
        g_unit_test_success_count++;
        //printf("\x1b[97m" "%s:%d:0" "\x1b[92m" " OK" "\x1b[0m" " at '%s'\n", file, line, func);        
    }
}

#undef assert
#define assert(expression) assert_func(expression, __func__, __FILE__, __LINE__, #expression)




void print_asserts(struct token* p_token)
{
    struct token* current = p_token;
    printf("struct { const char* lexeme; enum token_type token; int is_active; int is_final; } result[] = { \n");
    while (current)
    {
        printf("{ %-20s, %d, ", get_token_name(current->type), (current->flags & TK_FLAG_FINAL));
        print_literal(current->lexeme);
        printf("},\n");
        current = current->next;
    }
    printf("}\n");
}

void show_all(struct token* p_token)
{
    struct token* current = p_token;
    while (current)
    {
        if (current->flags & TK_FLAG_FINAL)
        {
            if (current->level == 0)
                printf(WHITE);
            else
                printf(BROWN);
        }
        else
        {
            if (current->level == 0)
                printf(LIGHTGRAY);
            else
                printf(BLACK);
        }
        printf("%s", current->lexeme);
        printf(RESET);
        current = current->next;
    }
}

void print_preprocessed_to_file(struct token* p_token, const char* filename)
{
    FILE* f = fopen(filename, "r");
    if (f)
    {
        const char* s = print_preprocessed_to_string(p_token);
        if (s)
        {
            fprintf(f, "%s", s);
            free((void* _Owner)s);
        }
        fclose(f);
    }
}

void show_visible(struct token* p_token)
{
    printf(WHITE "visible used   / " LIGHTGRAY "visible ignored\n" RESET);
    struct token* current = p_token;
    while (current)
    {
        if (current->level == 0)
        {
            if (current->flags & TK_FLAG_FINAL)
                printf(WHITE);
            else
                printf(LIGHTGRAY);
        }
        else
        {
            if (current->level == 0)
                printf(BLACK);
            else
                printf(BLACK);
        }
        printf("%s", current->lexeme);
        printf(RESET);
        current = current->next;
    }
}

void show_visible_and_invisible(struct token* p_token)
{
    printf(LIGHTGREEN "visible used   / " LIGHTGRAY "visible ignored\n" RESET);
    printf(LIGHTBLUE  "invisible used / " BROWN     "invisible ignored\n" RESET);
    struct token* current = p_token;
    while (current)
    {
        if (current->level == 0)
        {
            if (current->flags & TK_FLAG_FINAL)
                printf(LIGHTGREEN);
            else
                printf(LIGHTGRAY);
        }
        else
        {
            if (current->flags & TK_FLAG_FINAL)
                printf(LIGHTBLUE);
            else
                printf(BROWN);
        }
        printf("%s", current->lexeme);
        printf(RESET);
        current = current->next;
    }
}

int test_preprossessor_input_output(const char* input, const char* output)
{
    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "source", 0, TK_FLAG_NONE);

    struct preprocessor_ctx ctx = { 0 };

    struct token_list r = preprocessor(&ctx, &list, 0);
    const char* s = print_preprocessed_to_string(r.head);
    if (strcmp(s, output) != 0)
    {
        printf("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");
        printf("expected\n%s", output);
        printf("HAS\n%s", s);
        printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");
        print_tokens(r.head);
        printf("TEST 0 FAILED\n");
        return 1;
    }
    free((void* _Owner)s);
    return 0;
}

char* normalize_line_end(char* input)
{
    if (input == NULL)
        return NULL;
    char* pWrite = input;
    const char* p = input;
    while (*p)
    {
        if (p[0] == '\r' && p[1] == '\n')
        {
            *pWrite = '\n';
            p++;
            p++;
            pWrite++;
        }
        else
        {
            *pWrite = *p;
            p++;
            pWrite++;
        }
    }
    *pWrite = 0;
    return input;
}


int test_preprocessor_in_out(const char* input, const char* output)
{
    int res = 0;

    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "source", 0, TK_FLAG_NONE);

    struct preprocessor_ctx ctx = { 0 };

    struct token_list r = preprocessor(&ctx, &list, 0);
    const char* result = print_preprocessed_to_string(r.head);
    if (result == NULL)
    {
        result = strdup("");
    }

    if (strcmp(result, output) != 0)
    {
        /*
        printf("FAILED\n");
        printf("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");
        printf("assert\n");
        printf("%s`", output);
        printf("\nGOT\n");
        printf("%s`", result);
        printf("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");
        print_tokens(r.head);

        */
        res = 1;
    }

    free(result);

    return res;
}

int test_preprocessor_in_out_using_file(const char* fileName)
{
    int res = 0;
    const char* input = normalize_line_end(read_file(fileName, true));
    char* output = 0;
    if (input)
    {
        char* pos = strstr(input, "\n---");
        if (pos)
        {
            *pos = 0;
            //anda ate sair ---
            pos++;
            while (*pos != '\n')
            {
                pos++;
            }
            pos++; //skip \n
            output = pos;
            /*optional*/
            pos = strstr(output, "\n---");
            if (pos)
                *pos = 0;
        }
        res = test_preprocessor_in_out(input, output);
        free((void* _Owner)input);
    }
    return res;
}

void test_lexeme_cmp()
{
    assert(lexeme_cmp("a", "\\\na") == 0);
    assert(lexeme_cmp("a", "a\\\n") == 0);
    assert(lexeme_cmp("\\\na", "a") == 0);
    assert(lexeme_cmp("a\\\n", "a") == 0);
    assert(lexeme_cmp("a\\\nb", "ab") == 0);
    assert(lexeme_cmp("define", "define") == 0);
    assert(lexeme_cmp("de\\\nfine", "define") == 0);
}

void token_list_pop_front_test()
{

    struct token_list list = { 0 };
    token_list_pop_front(&list);
    struct tokenizer_ctx tctx = { 0 };
    list = tokenizer(&tctx, "a", NULL, 0, TK_FLAG_NONE);
    token_list_pop_front(&list);

    list = tokenizer(&tctx, "a,", NULL, 0, TK_FLAG_NONE);
    token_list_pop_front(&list);

    list = tokenizer(&tctx, "a,b", NULL, 0, TK_FLAG_NONE);
    token_list_pop_front(&list);
}

void token_list_pop_back_test()
{

    struct token_list list = { 0 };
    token_list_pop_back(&list);

    /*pop back quando so tem 1*/
    token_list_clear(&list);
    struct tokenizer_ctx tctx = { 0 };
    list = tokenizer(&tctx, "a", NULL, 0, TK_FLAG_NONE);
    token_list_pop_back(&list);
    assert(list.head == NULL && list.tail == NULL);


    /*
    * pop bacl com 2
    */

    token_list_clear(&list);
    list = tokenizer(&tctx, "a,", NULL, 0, TK_FLAG_NONE);
    token_list_pop_back(&list);

    assert(strcmp(list.head->lexeme, "a") == 0);

    assert(list.head != NULL &&
        list.head->prev == NULL &&
        list.head->next == NULL &&
        list.tail->prev == NULL &&
        list.tail->next == NULL &&
        list.tail == list.head);

    /*
    * pop back com 3
    */

    list = tokenizer(&tctx, "a,b", NULL, 0, TK_FLAG_NONE);
    token_list_pop_back(&list);
    assert(strcmp(list.head->lexeme, "a") == 0);
    assert(strcmp(list.head->next->lexeme, ",") == 0);
    assert(strcmp(list.tail->lexeme, ",") == 0);
    assert(strcmp(list.tail->prev->lexeme, "a") == 0);
    assert(list.head->prev == NULL);
    assert(list.tail->next == NULL);
}

int token_list_append_list_test()
{

    struct tokenizer_ctx tctx = { 0 };
    struct token_list source = { 0 };
    struct token_list dest = tokenizer(&tctx, "a", NULL, 0, TK_FLAG_NONE);
    token_list_append_list(&dest, &source);
    assert(strcmp(dest.head->lexeme, "a") == 0);


    token_list_clear(&source);
    token_list_clear(&dest);


    dest = tokenizer(&tctx, "a", NULL, 0, TK_FLAG_NONE);
    token_list_append_list(&dest, &source);

    assert(strcmp(dest.head->lexeme, "a") == 0);

    token_list_clear(&source);
    token_list_clear(&dest);
    source = tokenizer(&tctx, "a,", NULL, 0, TK_FLAG_NONE);
    dest = tokenizer(&tctx, "1", NULL, 0, TK_FLAG_NONE);
    token_list_append_list(&dest, &source);
    assert(strcmp(dest.head->lexeme, "1") == 0);
    assert(strcmp(dest.tail->lexeme, ",") == 0);
    assert(dest.tail->next == NULL);
    assert(dest.head->next->next == dest.tail);
    assert(dest.tail->prev->prev == dest.head);

    return 0;
}

void test_collect()
{
    const char* input =
        "#define F(A, B) A ## B\n"
        "F(a \n, b)";

    const char* output =
        "ab"
        ;


    assert(test_preprocessor_in_out(input, output) == 0);

}


void test_va_opt_0()
{
    const char* input =
        "#define F(...)  f(0 __VA_OPT__(,) __VA_ARGS__)\n"
        "F(a, b, c)";
    const char* output =
        "f(0, a, b, c)";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_va_opt_1()
{
    const char* input =
        "#define F(...)  f(0 __VA_OPT__(,) __VA_ARGS__)\n"
        "F()";
    const char* output =
        "f(0)";
    assert(test_preprocessor_in_out(input, output) == 0);
}


void test_va_opt_2()
{
    const char* input =
        "#define empty(...) (__VA_OPT__(!)1)\n"
        "empty()";
    const char* output =
        "(1)";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_va_opt_3()
{
    const char* input =
        "#define empty(...) (__VA_OPT__(!)1)\n"
        "empty(1)";
    const char* output =
        "(!1)";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_va_opt_4()
{
    const char* input =
        "#define LPAREN() (\n"
        "#define G(Q) 42\n"
        "#define F(R, X, ...) __VA_OPT__(G R X) )\n"
        "int x = F(LPAREN(), 0, <:-);\n"
        ;
    const char* output =
        "int x = 42;";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_va_opt_5()
{
    const char* input =
        "#define F(...) f(0 __VA_OPT__(,) __VA_ARGS__)\n"
        "#define EMPTY\n"
        "F(EMPTY)"
        ;
    const char* output =
        "f(0)";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_va_opt_6()
{
    const char* input =
        "#define G(X, ...) f(0, X __VA_OPT__(,) __VA_ARGS__)\n"
        "G(a)"
        ;

    const char* output =
        "f(0, a)";

    assert(test_preprocessor_in_out(input, output) == 0);
}
void test_va_opt_7()
{
    const char* input =
        "#define H4(X, ...) __VA_OPT__(a X ## X) ## b\n"
        "H4(, 1)"
        ;

    const char* output =
        "a b";

    assert(test_preprocessor_in_out(input, output) == 0);
}

void concatenation_problem()
{
    const char* input =
        "#define H4(X, ...) a X ## X ## b\n"
        "H4()"
        ;

    const char* output =
        "a b";

    assert(test_preprocessor_in_out(input, output) == 0);
}


void test_va_opt_G2()
{
    const char* input =
        "#define G(X, ...) f(0, X __VA_OPT__(,) __VA_ARGS__)\n"
        "G(a, )"
        ;

    const char* output =
        "f(0, a)";

    assert(test_preprocessor_in_out(input, output) == 0);
}


void test_va_opt()
{
    const char* input =
        "#define F(...)  f(0 __VA_OPT__(,) __VA_ARGS__)\n"
        "#define EMPTY\n"
        "F(EMPTY)";
    const char* output =
        "f(0)";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_empty_va_args()
{
    const char* input = "#define M(a, ...) a, __VA_ARGS__\n"
        "M(1)\n";
    const char* output =
        "1,";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_va_args_single()
{
    const char* input =
        "#define F(...) __VA_ARGS__\n"
        "F(1, 2)";
    const char* output =
        "1, 2";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_va_args_extra_args()
{
    const char* input =
        "#define F(a, ...) a __VA_ARGS__\n"
        "F(0, 1, 2)";
    const char* output =
        "0 1, 2";
    assert(test_preprocessor_in_out(input, output) == 0);
}


void test_empty_va_args_empty()
{
    const char* input =
        "#define F(...) a __VA_ARGS__\n"
        "F()";
    const char* output =
        "a";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void test_defined()
{
    const char* input =
        "#if defined X || defined (X)\n"
        "A\n"
        "#else\n"
        "B\n"
        "#endif\n";
    const char* output =
        "B";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void testline()
{
    const char* input =
        "#define M \\\n"
        "        a\\\n"
        "        b\n"
        "M";
    const char* output =
        "a b";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void ifelse()
{
    const char* input =
        "#if 1\n"
        "A\n"
        "#else\n"
        "B\n"
        "#endif\n";
    const char* output =
        "A";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void T1()
{
    const char* input =
        "#define f(a) f(x * (a))\n"
        "f(2 * (0, 1))";
    const char* output =
        "f(x * (2 * (0, 1)))";
    //se f tivesse 2 parametros
    //error: too few arguments provided to function-like macro invocation
    //se f nao tivesse nenhum ou menus
    //too many arguments provided to function-like macro invocation
    assert(test_preprocessor_in_out(input, output) == 0);
}

int EXAMPLE5()
{
    /*
    //EXAMPLE 5 To illustrate the rules for placemarker preprocessing tokens, the sequence

    //const char* input =
    //"#define t(x,y,z) x ## y ## z\n"
    //"int j[] = {t(+1,2,3), t(,4,5), t(6,,7), t(8,9,),t(10,,), t(,11,), t(,,12), t(,,) };";

    //const char* output =
      //  "int j[] = {+123, 45, 67, 89,10, 11, 12, };";

    const char* input =
        "#define t(x,y,z) x ## y ## z\n"
        "t(+1,2,3)";

    const char* output =
        "int j[] = {+123, 45, 67, 89,10, 11, 12, };";

    //se f tivesse 2 parametros
    //error: too few arguments provided to function-like macro invocation

    //se f nao tivesse nenhum ou menus
    //too many arguments provided to function-like macro invocation
    //test_preprocessor_in_out(input, output);
    */
    return 0;
}

void recursivetest1()
{
    //acho que este vai sero caso que precisa do hidden set.
    const char* input =
        "#define x 2\n"
        "#define f(a) f(x * (a))\n"
        "#define z z[0]\n"
        "f(f(z))";
    //resultado gcc da
    //const char* output =
    //  "f(2 * (f(2 * (z[0]))))";
    const char* output =
        "f(2 * (f(z[0])))";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void rectest()
{
    const char* input =
        "#define x 2\n"
        "#define f(a) f(x * (a))\n"
        "#define g f\n"
        "#define z z[0]\n"
        "f(y + 1) + f(f(z)) % t(t(g)(0) + t)(1);";
    //GCC
    //const char* output =
    //  "f(2 * (y + 1)) + f(2 * (f(2 * (z[0])))) % t(t(f)(0) + t)(1);";
    const char* output =
        "f(2 * (y + 1)) + f(2 * (f(z[0]))) % t(t(f)(0) + t)(1);";
    assert(test_preprocessor_in_out(input, output) == 0);
}

void emptycall()
{
    const char* input =
        "#define F(x) x\n"
        "F()"
        ;
    const char* output =
        ""
        ;
    assert(test_preprocessor_in_out(input, output) == 0);
}

void semiempty()
{
    const char* input =
        "#define F(x,y) x ## y\n"
        "F(1,)"
        ;
    const char* output =
        "1"
        ;
    assert(test_preprocessor_in_out(input, output) == 0);
}

void calling_one_arg_with_empty_arg()
{
    const char* input =
        "#define F(a) # a\n"
        "F()"
        ;
    const char* output =
        "\"\""
        ;
    assert(test_preprocessor_in_out(input, output) == 0);
}


void test_argument_with_parentesis()
{
    const char* input =
        "#define F(a, b) a ## b\n"
        "F((1, 2, 3),4)"
        ;
    const char* output =
        "(1, 2, 3)4"
        ;
    assert(test_preprocessor_in_out(input, output) == 0);
}

void two_empty_arguments()
{
    const char* input =
        "#define F(a, b) a ## b\n"
        "F(,)\n"
        ;
    const char* output =
        ""
        ;
    assert(test_preprocessor_in_out(input, output) == 0);
}

void simple_object_macro()
{
    const char* input = "#define B b\n"
        "#define M a B\n"
        "M\n"
        "c\n";
    const char* output =
        "a b\n"
        "c";
    assert(test_preprocessor_in_out(input, output) == 0);
}


void test_one_file()
{
    assert(test_preprocessor_in_out_using_file("tests/pre_debug.c") == 0);
}

void test2()
{
    const char* input =
        "#define F(a, b) 1 a ## b 4\n"
        "F(  2  ,  3 )"
        ;
    const char* output =
        "1 23 4"
        ;

    assert(test_preprocessor_in_out(input, output) == 0);
}


void test3()
{
#if 0
    const char* input =
        "#define F(a, b) 1 a ## 3 4\n"
        "F(  2   )"
        ;
    const char* output =
        "1 23 4"
        ;
#endif
    //este erro falta parametro b
    //too few arguments provided to function - like macro invocation
    //test_preprocessor_in_out(input, output);
}


void tetris()
{
    const char* input =
        "#define D(a) a\n"
        "#define C(a) a\n"
        "#define F(a) a\n"
        "#define M F\n"
        "M(F)(C)(D)e"
        ;
    const char* output =
        "De"
        ;
    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "source", 0, TK_FLAG_NONE);

    struct preprocessor_ctx ctx = { 0 };

    struct token_list r = preprocessor(&ctx, &list, 0);

    assert(test_preprocessor_in_out(input, output) == 0);
}

void recursive_macro_expansion()
{
    const char* input =
        "#define A 3 4 B\n"
        "#define B 1 2 A\n"
        "B";
    const char* output =
        "1 2 3 4 B"
        ;
    assert(test_preprocessor_in_out(input, output) == 0);
}

void empty_and_no_args()
{
    const char* input =
        "#define F() 1\n"
        "F()";
    const char* output =
        "1"
        ;
    assert(test_preprocessor_in_out(input, output) == 0);
}

void empty_and_args()
{
    const char* input =
        "#define F() 1\n"
        "F(1)";
    const char* output =
        "1"
        ;
    int code = test_preprocessor_in_out(input, output);
    assert(code != 0);
}

void test4()
{
    const char* input =
        "#define F(a, b) 1 2 ## a 4\n"
        "F(  3   )"
        ;
    const char* output =
        "1 23 4"
        ;


    int code = test_preprocessor_in_out(input, output);

    //esperado um erro (falta mensagem)
    //too few arguments provided to function-like macro invocation F (3)
    //engracado msc eh warning  warning C4003: not enough actual parameters for macro 'F'
    assert(code != 0);
}

void test_string()
{
    const char* input =
        "#define M(a, b) a # b\n"
        "M(A, \"B\")"
        ;
    const char* output =
        "A \"\\\"B\\\"\""
        ;


    test_preprocessor_in_out(input, output);
}

void test6()
{
    /*

    #define Y
    #define X defined (Y)

    #if X
    #warning !
    #endif
    */
}

void testerror()
{
    /*
    const char* input =
        "#define F(a) #b\n"
        "F(1)\n"
        ;
    const char* output =
        ""
        ;
    //tem que dar error
    test_preprocessor_in_out(input, output);
    */
}

int test_preprocessor_expression(const char* expr, long long expected)
{

    struct preprocessor_ctx ctx = { 0 };

    struct token_list r = { 0 };
    struct tokenizer_ctx tctx = { 0 };
    struct token_list input = tokenizer(&tctx, expr, "", 0, TK_FLAG_NONE);

    long long result = preprocessor_constant_expression(&ctx, &r, &input, 0);
    return result == expected ? 0 : 1;
}

int test_expression()
{

    //TODO preprocessador eh sempre long long.. signed passadno maior
    //deve dar erro

    if (test_preprocessor_expression("true", true) != 0)
        return __LINE__;

    if (test_preprocessor_expression("false", false) != 0)
        return __LINE__;


    if (test_preprocessor_expression("'A'", 'A') != 0)
        return __LINE__;

    if (test_preprocessor_expression("'ab'", 'ab') != 0)
        return __LINE__;

    if (test_preprocessor_expression("1+2", 1 + 2) != 0)
        return __LINE__;

    if (test_preprocessor_expression("1 + 2 * 3 / 2 ^ 2 & 4 | 3 % 6 >> 2 << 5 - 4 + !7",
        1 + 2 * 3 / 2 ^ 2 & 4 | 3 % 6 >> 2 << 5 - 4 + !7) != 0)
        return __LINE__;

    if (test_preprocessor_expression("1ull + 2l * 3ll",
        1ull + 2l * 3ll) != 0)
        return __LINE__;


    return 0;
}

int test_concatenation_o()
{
    const char* input =
        "# define F(t1, t2, t3) *i_##t1##_j k\n"
        "F(A, B, C)\n";

    const char* output =
        "*i_A_j k"
        ;


    return test_preprocessor_in_out(input, output);
}

int test_concatenation()
{
    const char* input =
        "#define F(t1, t2, t3) i##j##k\n"
        "F(A, B, C)\n";

    const char* output =
        "ijk"
        ;


    return test_preprocessor_in_out(input, output);


}

void bad_test()
{
    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, "0xfe-BAD(3)", "source", 0, TK_FLAG_NONE);

    const char* input = "#define BAD(x) ((x) & 0xff)\n"
        "0xfe-BAD(3);";
    const char* output =
        "0xfe-BAD(3);"
        ;

    test_preprocessor_in_out(input, output);
}
/*
#define A0
#define B0
#define A1(x) x B##x(
#define B1(x) x A##x(
A1(1)1)1)1)1)0))
*/
int test_spaces()
{
    const char* input =
        "#define throw A B\n"
        "throw\n"
        ;
    const char* output =
        "A B"
        ;


    return test_preprocessor_in_out(input, output);
}

int test_stringfy()
{
    const char* input =
        "#define M(T) #T\n"
        "M(unsigned   int)\n"
        ;
    const char* output =
        "\"unsigned int\""
        ;


    return test_preprocessor_in_out(input, output);

}


int test_tokens()
{
    const char* input =
        "L\"s1\" u8\"s2\""
        ;

    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "", 0, TK_FLAG_NONE);

    if (list.head->next->type != TK_STRING_LITERAL)
    {
        return __LINE__;
    }

    if (list.head->next->next->next->type != TK_STRING_LITERAL)
    {
        return __LINE__;
    }

    return tctx.n_errors;
}

int test_predefined_macros()
{
    const char* input =
        "__LINE__ __FILE__"
        ;
    const char* output =
        "1 \"source\""
        ;

    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "source", 0, TK_FLAG_NONE);

    struct preprocessor_ctx prectx = { 0 };
    prectx.macros.capacity = 5000;
    add_standard_macros(&prectx);
    struct token_list list2 = preprocessor(&prectx, &list, 0);


    const char* result = print_preprocessed_to_string(list2.head);
    if (result == NULL)
    {
        result = strdup("");
    }
    if (strcmp(result, output) != 0)
    {

    }


    return 0;
}

int test_utf8()
{

    const char* input =
        "u8\"maçã\"";

    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "source", 0, TK_FLAG_NONE);
    if (strcmp(list.head->next->lexeme, u8"u8\"maçã\"") != 0)
        return __LINE__;
    token_list_destroy(&list);
    return 0;
}

int test_counter()
{
    //https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3457.htm#number-of-expansions

    const char* input =
        "#define X(Z) Z Z\n"
        "X(__COUNTER__)\n";

    const char* output =
        "0 0"
        ;

    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "source", 0, TK_FLAG_NONE);

    struct preprocessor_ctx prectx = { 0 };
    prectx.macros.capacity = 5000;
    add_standard_macros(&prectx);
    struct token_list list2 = preprocessor(&prectx, &list, 0);

    const char* result = print_preprocessed_to_string(list2.head);
    if (result == NULL)
    {
        result = strdup("");
    }

    assert(test_preprocessor_in_out(result, output) == 0);

    return 0;
}

int bug_test()
{
    const char* input =
        "#define M(b) a #b \n"
        "M(1)\n";

    const char* output =
        "a \"1\""
        ;


    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "source", 0, TK_FLAG_NONE);



    assert(test_preprocessor_in_out(input, output) == 0);

    return 0;
}
int test_line_continuation()
{


    const char* input =
        "#define A B \\\n"
        "C\n"
        "A";

    const char* output =
        "1 \"source\""
        ;


    struct tokenizer_ctx tctx = { 0 };
    struct token_list list = tokenizer(&tctx, input, "", 0, TK_FLAG_NONE);

    struct preprocessor_ctx prectx = { 0 };
    prectx.macros.capacity = 5000;

    struct token_list list2 = preprocessor(&prectx, &list, 0);

    const char* result = print_preprocessed_to_string(list2.head);
    if (result == NULL)
    {
        result = strdup("");
    }
    if (strcmp(result, output) != 0)
    {
    }


    return 0;
}

int stringify_test()
{
    char buffer[200];
    int n = stringify("\"ab\\c\"", sizeof buffer, buffer);
    assert(n == sizeof(STRINGIFY("\"ab\\c\"")));
    const char* r = STRINGIFY("\"ab\\c\"");

    assert(strcmp(buffer, r) == 0);
    return 0;

}

#endif


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

#pragma safety enable


void ss_swap(_View struct osstream* a, _View struct osstream* b)
{
    _View struct osstream r = *a;
    *a = *b;
    *b = r;
}

void ss_clear(struct osstream* stream)
{
    if (stream->c_str)
        stream->c_str[0] = '\0';
    stream->size = 0;
}


void ss_close(_Dtor struct osstream* stream)
{
    free(stream->c_str);
}

static int reserve(struct osstream* stream, int size)
{
    int errorcode = 0;
    if (size > stream->capacity)
    {
        void* _Owner _Opt pnew = realloc(stream->c_str, (size + 1) * sizeof(char));
        if (pnew)
        {
            static_set(stream->c_str, "moved");
            stream->c_str = pnew;
            stream->capacity = size;
            stream->c_str[size] = 0;
        }
        else
        {
            errno = ENOMEM;
            errorcode = 1;
        }
    }
    return errorcode;
}

int ss_vafprintf(struct osstream* stream, const char* fmt, va_list args)
{
    assert(fmt != 0);
    int size = 0;

#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wnullable-to-non-nullable"
#pragma CAKE diagnostic ignored "-Wanalyzer-null-dereference"



    va_list tmpa = { 0 };

    va_copy(tmpa, args);

    size = vsnprintf(stream->c_str + stream->size, stream->capacity - stream->size, fmt, tmpa);

    va_end(tmpa);

#pragma CAKE diagnostic pop

    if (size < 0)
    {
        return -1;
    }

    if (reserve(stream, stream->size + size) != 0)
    {
        return -1;
    }

    size = vsprintf(stream->c_str + stream->size, fmt, args);
    if (size > 0)
    {
        stream->size += size;
    }
    return size;
}

/*
* Returns the character written. For fputc, a return value of EOF indicates an error
*/
int ss_putc(char ch, struct osstream* stream)
{
    if (reserve(stream, stream->size + 1) != 0)
    {
        return EOF;
    }
    stream->c_str[stream->size] = ch;
    stream->size++;

    return ch;
}

int ss_fprintf(struct osstream* stream, const char* fmt, ...)
{
#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wnullable-to-non-nullable"
#pragma CAKE diagnostic ignored "-Wanalyzer-null-dereference"

    va_list args = { 0 };
    va_start(args, fmt);
    int size = ss_vafprintf(stream, fmt, args);
    va_end(args);

#pragma CAKE diagnostic pop

    return size;
}



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

#pragma safety enable



#include <wchar.h>


#ifdef _WIN32
#endif


#if defined _MSC_VER && !defined __POCC__
#endif


#ifdef __POCC__
#define stat _stat
#endif

#ifdef _WIN32
#pragma comment (lib, "Rpcrt4.lib")

#else


#include <uuid/uuid.h>
/*
caso nao tenha este arquivos apt-get install uuid-dev
*/
#endif



bool path_is_normalized(const char* path)
{
#ifdef _WINDOWS_
    for (const char* p = path; *p; p++)
    {
        int before = *p;
        int after = tolower(*p);

        if (before != after)
            return false;

        if (after == '\\')
        {
            return false;
        }
    }
    return true;
#else
    return true;
#endif
}

void path_normalize(char* path)
{
#ifdef _WINDOWS_
    for (char* p = path; *p; p++)
    {
        *p = (char)tolower(*p);
        if (*p == '\\')
        {
            *p = '/';
        }
    }
#else

#endif
}

bool path_is_absolute(const char* path)
{
#ifdef _WINDOWS_
    const char ch = (char)tolower(path[0]);
    if (ch >= 'a' && ch <= 'z')
    {
        /*  c:/ or c:\ */
        if (path[1] == ':' && (path[2] == '\\' || path[2] == '/'))
            return true;
    }

    if (path[0] == '\\' && path[1] == '\\')
    {
        // //server
        return true;
    }
#else
    return path[0] == '/';
#endif

    return false;
}

bool path_is_relative(const char* path)
{
    return !path_is_absolute(path);
}


#ifdef _WIN32

#ifdef __CAKE__
#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wstyle"
#endif

struct TAGDIR
{
    void* handle;
    struct dirent dirent;
};

#ifdef __CAKE__
#pragma CAKE diagnostic pop
#endif

DIR* _Owner _Opt opendir(const char* name)
{
    assert(name != 0);
    WIN32_FIND_DATAA fdfile = { 0 };

    char path[MAX_PATH] = { 0 };
    strcat(path, name);
    strcat(path, "\\*.*");

    HANDLE handle = FindFirstFileA(path, &fdfile);

    if (handle != INVALID_HANDLE_VALUE)
    {
        _Opt DIR* _Owner _Opt p = calloc(1, sizeof * p);
        if (p)
        {
            p->handle = handle;
            return p;
        }
        else
        {
            /*calloc sets errno to ENOMEM if a memory allocation fails */
            FindClose(handle);
        }
    }
    else
    {
        errno = windows_error_to_posix(GetLastError());
    }

    return NULL;
}

int closedir(DIR* _Owner dirp)
{
    FindClose(dirp->handle);
    free(dirp);
    return 0;
}

struct dirent* _Opt readdir(DIR* dirp)
{
    WIN32_FIND_DATAA fdfile = { 0 };
    BOOL b = FindNextFileA(dirp->handle, &fdfile);
    if (b)
    {
        /*clear*/
        memset(&dirp->dirent, 0, sizeof(dirp->dirent));

        if (fdfile.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
        {
            dirp->dirent.d_type |= DT_DIR;
        }

        /*worst case trunks the string*/
        strncpy(dirp->dirent.d_name,
            fdfile.cFileName,
            sizeof(dirp->dirent.d_name) - 1);

        return &dirp->dirent;
    }
    else
    {
        errno = windows_error_to_posix(GetLastError());
    }
    return NULL;
}

/*
* realpath
   If there is no error, realpath() returns a pointer to the
       resolved_path.

       Otherwise, it returns NULL, the contents of the array
       resolved_path are undefined, and errno is set to indicate the
       error.
*/
char* _Opt realpath(const char* restrict path, char* restrict resolved_path)
{
    /*
    * _fullpath
    * Each of these functions returns a pointer to a buffer
      containing the absolute path name (absPath). If there's an
      error (for example, if the value passed in relPath includes a drive
      letter that isn't valid or can't be found, or if the length of the
      created absolute path name (absPath) is greater than maxLength), the function returns NULL.
    */
#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wflow-not-null"
    char* _Opt p = _fullpath(resolved_path, path, MAX_PATH);
    if (p)
    {
        char* p2 = resolved_path;
        while (*p2)
        {
            if (*p2 == '\\')
                *p2 = '/';
            p2++;
        }
    }
#pragma CAKE diagnostic pop

    return p;
}

#endif //_WINDOWS_

int copy_file(const char* pathfrom, const char* pathto)
{

    FILE* _Owner _Opt fd_from = fopen(pathfrom, "rb");
    if (fd_from == NULL)
        return -1;

    FILE* _Owner _Opt fd_to = fopen(pathto, "wb");
    if (fd_to == NULL)
    {
        fclose(fd_from);
        return -1;
    }

    char buf[4096] = { 0 };
    size_t nread;
    while (nread = fread(buf, sizeof(char), sizeof buf, fd_from), nread > 0) //lint !e668  (warning -- possibly passing null pointer to function 'fread(void *, size_t, size_t, FILE *)', arg. no. 4)
    {
        char* out_ptr = buf;
        size_t nwritten;

        do
        {
            nwritten = fwrite(out_ptr, sizeof(char), nread, fd_to);//lint !e668

            nread -= nwritten;
            out_ptr += nwritten;
        } while (nread > 0);
    }

    if (nread == 0)
    {
        fclose(fd_to);
        fclose(fd_from);

        /* Success! */
        return 0;
    }

    fclose(fd_to);
    fclose(fd_from);

    return -1;
}

int copy_folder(const char* from, const char* to)
{
#if !defined __EMSCRIPTEN__
    int errcode = mkdir(to, 0700);
    if (errcode != 0)
    {
        return errcode;
    }

    DIR* _Owner _Opt dir = opendir(from);

    if (dir == NULL)
    {
        return errno;
    }

    struct dirent* _Opt dp;
    while ((dp = readdir(dir)) != NULL)
    {
        if (strcmp(dp->d_name, ".") == 0 || strcmp(dp->d_name, "..") == 0)
        {
            /* skip self and parent */
            continue;
        }

        char fromlocal[MAX_PATH] = { 0 };
        snprintf(fromlocal, MAX_PATH, "%s/%s", from, dp->d_name);

        char tolocal[MAX_PATH] = { 0 };
        snprintf(tolocal, MAX_PATH, "%s/%s", to, dp->d_name);

        if (dp->d_type & DT_DIR)
        {
            errcode = copy_folder(fromlocal, tolocal);
        }
        else
        {
            errcode = copy_file(fromlocal, tolocal);
        }

        if (errcode != 0)
            break;
    }

    closedir(dir);
    return errcode;
#else
    return -1;
#endif
}

#ifdef _WIN32
int get_self_path(char* buffer, int maxsize)
{

#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wnullable-to-non-nullable"
#pragma CAKE diagnostic ignored "-Wanalyzer-null-dereference"

    DWORD r = GetModuleFileNameA(NULL, buffer, maxsize);

#pragma CAKE diagnostic pop

    return r;
}

#else

int get_self_path(char* buffer, int maxsize)
{

    memset(buffer, 0, maxsize); // readlink does not null terminate!
    if (readlink("/proc/self/exe", buffer, maxsize) == -1)
    {
        //perror("readlink");
        return 1;
    }
    else
    {
        //printf("%s\n", dest);
    }
    return 0;

}

#endif




#if !defined __EMSCRIPTEN__

/* Find the last occurrence of c1 or c2 in s. */
char* _Opt strrchr_ex(const char* s, int c1, int c2)
{
    const char* _Opt last = NULL;
    const char* p = s;
    while (*p)
    {
        if (*p == c1 || *p == c2)
        {
            last = p;
        }
        p++;
    }
    return (char*)last;
}

char* basename(const char* filename)
{
    char* _Opt p = strrchr_ex(filename, '/', '\\'); //added \ to windows path
    return p ? p + 1 : (char*)filename;
}

char* dirname(char* path)
{
    int last = -1;
    for (int i = 0; path[i]; i++)
    {
        if (path[i] == '\\' || path[i] == '/')
            last = i;
    }

    if (last != -1)
    {
        path[last] = 0;
    }
    else
    {
        path[0] = 0;
    }
    return path;
}
#endif

#ifndef MOCKFILES

char* _Owner _Opt read_file(const char* const path, bool append_newline)
{
    char* _Owner _Opt data = NULL;
    FILE* _Owner _Opt file = NULL;
    struct stat info = { 0 };

    if (stat(path, &info) != 0)
        return NULL;

    int mem_size_bytes = sizeof(char) * info.st_size + 1 /* \0 */ + 1 /*newline*/;
    
    if (mem_size_bytes < 4)
    {
        //we always read 3 chars even if file is small
        mem_size_bytes = 4; //BOM + /0
    }

    data = malloc(mem_size_bytes);
    if (data == NULL)
        return NULL;

    file = fopen(path, "r");
    if (file == NULL)
    {
        free(data);
        return NULL;
    }

    /* first we read 3 bytes */
    size_t bytes_read = fread(data, 1, 3, file);

    if (bytes_read < 3)
    {
        /* we have less than 3 bytes - no BOM */

        data[bytes_read] = '\0';
        if (feof(file))
        {
            fclose(file);
            return data;
        }

        free(data);
        fclose(file);

        return NULL;
    }

    size_t bytes_read_part2 = 0;

    /* check byte order mark (BOM) */
    if ((unsigned char)data[0] == (unsigned char)0xEF &&
        (unsigned char)data[1] == (unsigned char)0xBB &&
        (unsigned char)data[2] == (unsigned char)0xBF)
    {
        /* in this case we skip this BOM, reading again*/
        bytes_read_part2 = fread(&data[0], 1, info.st_size - 3, file);
    }
    else
    {
        bytes_read_part2 = fread(&data[3], 1, info.st_size - 3, file);
        bytes_read_part2 = bytes_read_part2 + 3;
    }

    data[bytes_read_part2] = 0;
    if (append_newline && data[bytes_read_part2 - 1] != '\n')
    {
        /*
        A source file that is not empty shall end in a new-line character, which shall not 
        be immediately preceded by a backslash character before any such splicing takes place.
        */
        data[bytes_read_part2] = '\n';

        //we already allocated an extra char for this
        assert(bytes_read_part2+1 < mem_size_bytes);
        data[bytes_read_part2+1] = '\0'; 
    }

    fclose(file);
    return data;
}

#else

/*
   used in web build
   embeded standard headers from .\include\
   the tool embed creates the .include version of each file
   in .\include\
*/

static const char file_assert_h[] = {



47,42,13,10,32,42,32,32,84,104,105,115,32,102,105,108,101,32,105,115,32,112,97,114,116
,32,111,102,32,99,97,107,101,32,99,111,109,112,105,108,101,114,13,10,32,42,32,32,104,116
,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,116,104,114,97,100,97,109,115
,47,99,97,107,101,13,10,42,47,32,13,10,13,10,35,105,102,100,101,102,32,78,68,69,66
,85,71,13,10,35,100,101,102,105,110,101,32,97,115,115,101,114,116,40,46,46,46,41,32,40
,40,118,111,105,100,41,48,41,13,10,35,101,108,115,101,13,10,35,100,101,102,105,110,101,32
,97,115,115,101,114,116,40,46,46,46,41,32,97,115,115,101,114,116,40,95,95,86,65,95,65
,82,71,83,95,95,41,13,10,35,101,110,100,105,102,13,10
};

static const char file_stdio_h[] = {



47,42,13,10,32,42,32,32,84,104,105,115,32,102,105,108,101,32,105,115,32,112,97,114,116
,32,111,102,32,99,97,107,101,32,99,111,109,112,105,108,101,114,13,10,32,42,32,32,104,116
,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,116,104,114,97,100,97,109,115
,47,99,97,107,101,13,10,42,47,13,10,13,10,35,112,114,97,103,109,97,32,111,110,99,101
,13,10,35,100,101,102,105,110,101,32,95,73,79,70,66,70,32,48,120,48,48,48,48,13,10
,35,100,101,102,105,110,101,32,95,73,79,76,66,70,32,48,120,48,48,52,48,13,10,35,100
,101,102,105,110,101,32,95,73,79,78,66,70,32,48,120,48,48,48,52,13,10,13,10,35,100
,101,102,105,110,101,32,66,85,70,83,73,90,32,32,53,49,50,13,10,13,10,35,100,101,102
,105,110,101,32,69,79,70,32,32,32,32,40,45,49,41,13,10,13,10,35,100,101,102,105,110
,101,32,70,73,76,69,78,65,77,69,95,77,65,88,32,32,32,32,50,54,48,13,10,35,100
,101,102,105,110,101,32,70,79,80,69,78,95,77,65,88,32,32,32,32,32,32,32,50,48,13
,10,13,10,35,100,101,102,105,110,101,32,76,95,116,109,112,110,97,109,32,32,32,50,54,48
,32,47,47,32,95,77,65,88,95,80,65,84,72,13,10,13,10,47,42,32,83,101,101,107,32
,109,101,116,104,111,100,32,99,111,110,115,116,97,110,116,115,32,42,47,13,10,13,10,35,100
,101,102,105,110,101,32,83,69,69,75,95,67,85,82,32,32,32,32,49,13,10,35,100,101,102
,105,110,101,32,83,69,69,75,95,69,78,68,32,32,32,32,50,13,10,35,100,101,102,105,110
,101,32,83,69,69,75,95,83,69,84,32,32,32,32,48,13,10,13,10,13,10,35,100,101,102
,105,110,101,32,84,77,80,95,77,65,88,32,32,32,32,32,32,32,32,32,50,49,52,55,52
,56,51,54,52,55,13,10,13,10,13,10,13,10,116,121,112,101,100,101,102,32,108,111,110,103
,32,108,111,110,103,32,102,112,111,115,95,116,59,13,10,116,121,112,101,100,101,102,32,105,110
,116,32,70,73,76,69,59,13,10,13,10,101,120,116,101,114,110,32,70,73,76,69,42,32,115
,116,100,105,110,59,13,10,101,120,116,101,114,110,32,70,73,76,69,42,32,115,116,100,111,117
,116,59,13,10,101,120,116,101,114,110,32,70,73,76,69,42,32,115,116,100,101,114,114,59,13
,10,13,10,116,121,112,101,100,101,102,32,117,110,115,105,103,110,101,100,32,108,111,110,103,32
,115,105,122,101,95,116,59,13,10,116,121,112,101,100,101,102,32,118,111,105,100,42,32,118,97
,95,108,105,115,116,59,13,10,105,110,116,32,114,101,109,111,118,101,40,99,111,110,115,116,32
,99,104,97,114,42,32,102,105,108,101,110,97,109,101,41,59,13,10,105,110,116,32,114,101,110
,97,109,101,40,99,111,110,115,116,32,99,104,97,114,42,32,111,108,100,44,32,99,111,110,115
,116,32,99,104,97,114,42,32,110,101,119,115,41,59,13,10,70,73,76,69,42,32,95,79,112
,116,32,116,109,112,102,105,108,101,40,118,111,105,100,41,59,13,10,99,104,97,114,42,32,116
,109,112,110,97,109,40,99,104,97,114,42,32,115,41,59,13,10,35,105,102,32,100,101,102,105
,110,101,100,40,95,95,83,84,68,67,95,79,87,78,69,82,83,72,73,80,95,95,41,32,13
,10,105,110,116,32,102,99,108,111,115,101,40,70,73,76,69,42,32,95,79,119,110,101,114,32
,115,116,114,101,97,109,41,59,13,10,35,101,108,115,101,13,10,105,110,116,32,102,99,108,111
,115,101,40,70,73,76,69,42,32,115,116,114,101,97,109,41,59,13,10,35,101,110,100,105,102
,13,10,105,110,116,32,102,102,108,117,115,104,40,70,73,76,69,42,32,115,116,114,101,97,109
,41,59,13,10,35,105,102,32,100,101,102,105,110,101,100,40,95,95,83,84,68,67,95,79,87
,78,69,82,83,72,73,80,95,95,41,32,13,10,70,73,76,69,42,32,95,79,119,110,101,114
,32,95,79,112,116,32,102,111,112,101,110,40,99,111,110,115,116,32,99,104,97,114,42,32,114
,101,115,116,114,105,99,116,32,102,105,108,101,110,97,109,101,44,32,99,111,110,115,116,32,99
,104,97,114,42,32,114,101,115,116,114,105,99,116,32,109,111,100,101,41,59,13,10,70,73,76
,69,42,32,95,79,119,110,101,114,32,95,79,112,116,32,102,114,101,111,112,101,110,40,99,111
,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,105,108,101,110,97
,109,101,44,32,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32
,109,111,100,101,44,32,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115,116,114,101
,97,109,41,59,13,10,35,101,108,115,101,13,10,70,73,76,69,42,32,102,111,112,101,110,40
,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,105,108,101
,110,97,109,101,44,32,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99
,116,32,109,111,100,101,41,59,13,10,70,73,76,69,42,32,102,114,101,111,112,101,110,40,99
,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,105,108,101,110
,97,109,101,44,32,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116
,32,109,111,100,101,44,32,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115,116,114
,101,97,109,41,59,13,10,35,101,110,100,105,102,13,10,118,111,105,100,32,115,101,116,98,117
,102,40,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115,116,114,101,97,109,44,32
,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,98,117,102,41,59,13,10,105,110,116
,32,115,101,116,118,98,117,102,40,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115
,116,114,101,97,109,44,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,98,117,102
,44,32,105,110,116,32,109,111,100,101,44,32,115,105,122,101,95,116,32,115,105,122,101,41,59
,13,10,105,110,116,32,102,112,114,105,110,116,102,40,70,73,76,69,42,32,114,101,115,116,114
,105,99,116,32,115,116,114,101,97,109,44,32,99,111,110,115,116,32,99,104,97,114,42,32,114
,101,115,116,114,105,99,116,32,102,111,114,109,97,116,44,32,46,46,46,41,59,13,10,105,110
,116,32,102,115,99,97,110,102,40,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115
,116,114,101,97,109,44,32,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105
,99,116,32,102,111,114,109,97,116,44,32,46,46,46,41,59,13,10,105,110,116,32,112,114,105
,110,116,102,40,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32
,102,111,114,109,97,116,44,32,46,46,46,41,59,13,10,105,110,116,32,115,99,97,110,102,40
,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,111,114,109
,97,116,44,32,46,46,46,41,59,13,10,105,110,116,32,115,110,112,114,105,110,116,102,40,99
,104,97,114,42,32,114,101,115,116,114,105,99,116,32,115,44,32,115,105,122,101,95,116,32,110
,44,32,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,111
,114,109,97,116,44,32,46,46,46,41,59,13,10,105,110,116,32,115,112,114,105,110,116,102,40
,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,115,44,32,99,111,110,115,116,32,99
,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,111,114,109,97,116,44,32,46,46,46
,41,59,13,10,105,110,116,32,115,115,99,97,110,102,40,99,111,110,115,116,32,99,104,97,114
,42,32,114,101,115,116,114,105,99,116,32,115,44,32,99,111,110,115,116,32,99,104,97,114,42
,32,114,101,115,116,114,105,99,116,32,102,111,114,109,97,116,44,32,46,46,46,41,59,13,10
,105,110,116,32,118,102,112,114,105,110,116,102,40,70,73,76,69,42,32,114,101,115,116,114,105
,99,116,32,115,116,114,101,97,109,44,32,99,111,110,115,116,32,99,104,97,114,42,32,114,101
,115,116,114,105,99,116,32,102,111,114,109,97,116,44,32,118,97,95,108,105,115,116,32,97,114
,103,41,59,13,10,105,110,116,32,118,102,115,99,97,110,102,40,70,73,76,69,42,32,114,101
,115,116,114,105,99,116,32,115,116,114,101,97,109,44,32,99,111,110,115,116,32,99,104,97,114
,42,32,114,101,115,116,114,105,99,116,32,102,111,114,109,97,116,44,32,118,97,95,108,105,115
,116,32,97,114,103,41,59,13,10,105,110,116,32,118,112,114,105,110,116,102,40,99,111,110,115
,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,111,114,109,97,116,44,32
,118,97,95,108,105,115,116,32,97,114,103,41,59,13,10,105,110,116,32,118,115,99,97,110,102
,40,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,102,111,114
,109,97,116,44,32,118,97,95,108,105,115,116,32,97,114,103,41,59,13,10,105,110,116,32,112
,117,116,115,40,99,111,110,115,116,32,99,104,97,114,42,32,115,116,114,41,59,13,10,105,110
,116,32,102,112,117,116,115,40,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114
,105,99,116,32,115,44,32,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115,116,114
,101,97,109,41,59,13,10,105,110,116,32,103,101,116,99,40,70,73,76,69,42,32,115,116,114
,101,97,109,41,59,13,10,105,110,116,32,103,101,116,99,104,97,114,40,118,111,105,100,41,59
,13,10,105,110,116,32,112,117,116,99,40,105,110,116,32,99,44,32,70,73,76,69,42,32,115
,116,114,101,97,109,41,59,13,10,105,110,116,32,112,117,116,99,104,97,114,40,105,110,116,32
,99,41,59,13,10,105,110,116,32,112,117,116,115,40,99,111,110,115,116,32,99,104,97,114,42
,32,115,41,59,13,10,105,110,116,32,117,110,103,101,116,99,40,105,110,116,32,99,44,32,70
,73,76,69,42,32,115,116,114,101,97,109,41,59,13,10,105,110,116,32,102,103,101,116,99,40
,70,73,76,69,42,32,115,116,114,101,97,109,41,59,13,10,115,105,122,101,95,116,32,102,114
,101,97,100,40,118,111,105,100,42,32,114,101,115,116,114,105,99,116,32,112,116,114,44,32,115
,105,122,101,95,116,32,115,105,122,101,44,32,115,105,122,101,95,116,32,110,109,101,109,98,44
,32,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115,116,114,101,97,109,41,59,13
,10,115,105,122,101,95,116,32,102,119,114,105,116,101,40,99,111,110,115,116,32,118,111,105,100
,42,32,114,101,115,116,114,105,99,116,32,112,116,114,44,32,115,105,122,101,95,116,32,115,105
,122,101,44,32,115,105,122,101,95,116,32,110,109,101,109,98,44,32,70,73,76,69,42,32,114
,101,115,116,114,105,99,116,32,115,116,114,101,97,109,41,59,13,10,105,110,116,32,102,103,101
,116,112,111,115,40,70,73,76,69,42,32,114,101,115,116,114,105,99,116,32,115,116,114,101,97
,109,44,32,102,112,111,115,95,116,42,32,114,101,115,116,114,105,99,116,32,112,111,115,41,59
,13,10,105,110,116,32,102,115,101,101,107,40,70,73,76,69,42,32,115,116,114,101,97,109,44
,32,108,111,110,103,32,105,110,116,32,111,102,102,115,101,116,44,32,105,110,116,32,119,104,101
,110,99,101,41,59,13,10,105,110,116,32,102,115,101,116,112,111,115,40,70,73,76,69,42,32
,115,116,114,101,97,109,44,32,99,111,110,115,116,32,102,112,111,115,95,116,42,32,112,111,115
,41,59,13,10,108,111,110,103,32,105,110,116,32,102,116,101,108,108,40,70,73,76,69,42,32
,115,116,114,101,97,109,41,59,13,10,118,111,105,100,32,114,101,119,105,110,100,40,70,73,76
,69,42,32,115,116,114,101,97,109,41,59,13,10,118,111,105,100,32,99,108,101,97,114,101,114
,114,40,70,73,76,69,42,32,115,116,114,101,97,109,41,59,13,10,105,110,116,32,102,101,111
,102,40,70,73,76,69,42,32,115,116,114,101,97,109,41,59,13,10,105,110,116,32,102,101,114
,114,111,114,40,70,73,76,69,42,32,115,116,114,101,97,109,41,59,13,10,118,111,105,100,32
,112,101,114,114,111,114,40,99,111,110,115,116,32,99,104,97,114,42,32,115,41,59,13,10,13
,10,13,10,13,10,35,105,102,110,100,101,102,32,78,85,76,76,13,10,35,100,101,102,105,110
,101,32,78,85,76,76,32,40,40,118,111,105,100,42,41,48,41,13,10,35,101,110,100,105,102
,13,10
};

static const char file_errno_h[] = {



47,42,13,10,32,42,32,32,84,104,105,115,32,102,105,108,101,32,105,115,32,112,97,114,116
,32,111,102,32,99,97,107,101,32,99,111,109,112,105,108,101,114,13,10,32,42,32,32,104,116
,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,116,104,114,97,100,97,109,115
,47,99,97,107,101,13,10,42,47,13,10,13,10,35,112,114,97,103,109,97,32,111,110,99,101
,13,10,13,10,105,110,116,42,32,95,101,114,114,110,111,40,118,111,105,100,41,59,13,10,35
,100,101,102,105,110,101,32,101,114,114,110,111,32,40,42,95,101,114,114,110,111,40,41,41,13
,10,32,13,10,13,10,35,100,101,102,105,110,101,32,69,80,69,82,77,32,32,32,32,32,32
,32,32,32,32,32,49,13,10,35,100,101,102,105,110,101,32,69,78,79,69,78,84,32,32,32
,32,32,32,32,32,32,32,50,13,10,35,100,101,102,105,110,101,32,69,83,82,67,72,32,32
,32,32,32,32,32,32,32,32,32,51,13,10,35,100,101,102,105,110,101,32,69,73,78,84,82
,32,32,32,32,32,32,32,32,32,32,32,52,13,10,35,100,101,102,105,110,101,32,69,73,79
,32,32,32,32,32,32,32,32,32,32,32,32,32,53,13,10,35,100,101,102,105,110,101,32,69
,78,88,73,79,32,32,32,32,32,32,32,32,32,32,32,54,13,10,35,100,101,102,105,110,101
,32,69,50,66,73,71,32,32,32,32,32,32,32,32,32,32,32,55,13,10,35,100,101,102,105
,110,101,32,69,78,79,69,88,69,67,32,32,32,32,32,32,32,32,32,56,13,10,35,100,101
,102,105,110,101,32,69,66,65,68,70,32,32,32,32,32,32,32,32,32,32,32,57,13,10,35
,100,101,102,105,110,101,32,69,67,72,73,76,68,32,32,32,32,32,32,32,32,32,32,49,48
,13,10,35,100,101,102,105,110,101,32,69,65,71,65,73,78,32,32,32,32,32,32,32,32,32
,32,49,49,13,10,35,100,101,102,105,110,101,32,69,78,79,77,69,77,32,32,32,32,32,32
,32,32,32,32,49,50,13,10,35,100,101,102,105,110,101,32,69,65,67,67,69,83,32,32,32
,32,32,32,32,32,32,32,49,51,13,10,35,100,101,102,105,110,101,32,69,70,65,85,76,84
,32,32,32,32,32,32,32,32,32,32,49,52,13,10,35,100,101,102,105,110,101,32,69,66,85
,83,89,32,32,32,32,32,32,32,32,32,32,32,49,54,13,10,35,100,101,102,105,110,101,32
,69,69,88,73,83,84,32,32,32,32,32,32,32,32,32,32,49,55,13,10,35,100,101,102,105
,110,101,32,69,88,68,69,86,32,32,32,32,32,32,32,32,32,32,32,49,56,13,10,35,100
,101,102,105,110,101,32,69,78,79,68,69,86,32,32,32,32,32,32,32,32,32,32,49,57,13
,10,35,100,101,102,105,110,101,32,69,78,79,84,68,73,82,32,32,32,32,32,32,32,32,32
,50,48,13,10,35,100,101,102,105,110,101,32,69,73,83,68,73,82,32,32,32,32,32,32,32
,32,32,32,50,49,13,10,35,100,101,102,105,110,101,32,69,78,70,73,76,69,32,32,32,32
,32,32,32,32,32,32,50,51,13,10,35,100,101,102,105,110,101,32,69,77,70,73,76,69,32
,32,32,32,32,32,32,32,32,32,50,52,13,10,35,100,101,102,105,110,101,32,69,78,79,84
,84,89,32,32,32,32,32,32,32,32,32,32,50,53,13,10,35,100,101,102,105,110,101,32,69
,70,66,73,71,32,32,32,32,32,32,32,32,32,32,32,50,55,13,10,35,100,101,102,105,110
,101,32,69,78,79,83,80,67,32,32,32,32,32,32,32,32,32,32,50,56,13,10,35,100,101
,102,105,110,101,32,69,83,80,73,80,69,32,32,32,32,32,32,32,32,32,32,50,57,13,10
,35,100,101,102,105,110,101,32,69,82,79,70,83,32,32,32,32,32,32,32,32,32,32,32,51
,48,13,10,35,100,101,102,105,110,101,32,69,77,76,73,78,75,32,32,32,32,32,32,32,32
,32,32,51,49,13,10,35,100,101,102,105,110,101,32,69,80,73,80,69,32,32,32,32,32,32
,32,32,32,32,32,51,50,13,10,35,100,101,102,105,110,101,32,69,68,79,77,32,32,32,32
,32,32,32,32,32,32,32,32,51,51,13,10,35,100,101,102,105,110,101,32,69,68,69,65,68
,76,75,32,32,32,32,32,32,32,32,32,51,54,13,10,35,100,101,102,105,110,101,32,69,78
,65,77,69,84,79,79,76,79,78,71,32,32,32,32,51,56,13,10,35,100,101,102,105,110,101
,32,69,78,79,76,67,75,32,32,32,32,32,32,32,32,32,32,51,57,13,10,35,100,101,102
,105,110,101,32,69,78,79,83,89,83,32,32,32,32,32,32,32,32,32,32,52,48,13,10,35
,100,101,102,105,110,101,32,69,78,79,84,69,77,80,84,89,32,32,32,32,32,32,32,52,49
,13,10,13,10,13,10,47,47,32,83,117,112,112,111,114,116,32,69,68,69,65,68,76,79,67
,75,32,102,111,114,32,99,111,109,112,97,116,105,98,105,108,105,116,121,32,119,105,116,104,32
,111,108,100,101,114,32,77,105,99,114,111,115,111,102,116,32,67,32,118,101,114,115,105,111,110
,115,13,10,35,100,101,102,105,110,101,32,69,68,69,65,68,76,79,67,75,32,32,32,32,32
,32,32,69,68,69,65,68,76,75,13,10,13,10,35,100,101,102,105,110,101,32,69,65,68,68
,82,73,78,85,83,69,32,32,32,32,32,32,49,48,48,13,10,35,100,101,102,105,110,101,32
,69,65,68,68,82,78,79,84,65,86,65,73,76,32,32,32,49,48,49,13,10,35,100,101,102
,105,110,101,32,69,65,70,78,79,83,85,80,80,79,82,84,32,32,32,32,49,48,50,13,10
,35,100,101,102,105,110,101,32,69,65,76,82,69,65,68,89,32,32,32,32,32,32,32,32,49
,48,51,13,10,35,100,101,102,105,110,101,32,69,66,65,68,77,83,71,32,32,32,32,32,32
,32,32,32,49,48,52,13,10,35,100,101,102,105,110,101,32,69,67,65,78,67,69,76,69,68
,32,32,32,32,32,32,32,49,48,53,13,10,35,100,101,102,105,110,101,32,69,67,79,78,78
,65,66,79,82,84,69,68,32,32,32,32,49,48,54,13,10,35,100,101,102,105,110,101,32,69
,67,79,78,78,82,69,70,85,83,69,68,32,32,32,32,49,48,55,13,10,35,100,101,102,105
,110,101,32,69,67,79,78,78,82,69,83,69,84,32,32,32,32,32,32,49,48,56,13,10,35
,100,101,102,105,110,101,32,69,68,69,83,84,65,68,68,82,82,69,81,32,32,32,32,49,48
,57,13,10,35,100,101,102,105,110,101,32,69,72,79,83,84,85,78,82,69,65,67,72,32,32
,32,32,49,49,48,13,10,35,100,101,102,105,110,101,32,69,73,68,82,77,32,32,32,32,32
,32,32,32,32,32,32,49,49,49,13,10,35,100,101,102,105,110,101,32,69,73,78,80,82,79
,71,82,69,83,83,32,32,32,32,32,49,49,50,13,10,35,100,101,102,105,110,101,32,69,73
,83,67,79,78,78,32,32,32,32,32,32,32,32,32,49,49,51,13,10,35,100,101,102,105,110
,101,32,69,76,79,79,80,32,32,32,32,32,32,32,32,32,32,32,49,49,52,13,10,35,100
,101,102,105,110,101,32,69,77,83,71,83,73,90,69,32,32,32,32,32,32,32,32,49,49,53
,13,10,35,100,101,102,105,110,101,32,69,78,69,84,68,79,87,78,32,32,32,32,32,32,32
,32,49,49,54,13,10,35,100,101,102,105,110,101,32,69,78,69,84,82,69,83,69,84,32,32
,32,32,32,32,32,49,49,55,13,10,35,100,101,102,105,110,101,32,69,78,69,84,85,78,82
,69,65,67,72,32,32,32,32,32,49,49,56,13,10,35,100,101,102,105,110,101,32,69,78,79
,66,85,70,83,32,32,32,32,32,32,32,32,32,49,49,57,13,10,35,100,101,102,105,110,101
,32,69,78,79,68,65,84,65,32,32,32,32,32,32,32,32,32,49,50,48,13,10,35,100,101
,102,105,110,101,32,69,78,79,76,73,78,75,32,32,32,32,32,32,32,32,32,49,50,49,13
,10,35,100,101,102,105,110,101,32,69,78,79,77,83,71,32,32,32,32,32,32,32,32,32,32
,49,50,50,13,10,35,100,101,102,105,110,101,32,69,78,79,80,82,79,84,79,79,80,84,32
,32,32,32,32,49,50,51,13,10,35,100,101,102,105,110,101,32,69,78,79,83,82,32,32,32
,32,32,32,32,32,32,32,32,49,50,52,13,10,35,100,101,102,105,110,101,32,69,78,79,83
,84,82,32,32,32,32,32,32,32,32,32,32,49,50,53,13,10,35,100,101,102,105,110,101,32
,69,78,79,84,67,79,78,78,32,32,32,32,32,32,32,32,49,50,54,13,10,35,100,101,102
,105,110,101,32,69,78,79,84,82,69,67,79,86,69,82,65,66,76,69,32,49,50,55,13,10
,35,100,101,102,105,110,101,32,69,78,79,84,83,79,67,75,32,32,32,32,32,32,32,32,49
,50,56,13,10,35,100,101,102,105,110,101,32,69,78,79,84,83,85,80,32,32,32,32,32,32
,32,32,32,49,50,57,13,10,35,100,101,102,105,110,101,32,69,79,80,78,79,84,83,85,80
,80,32,32,32,32,32,32,49,51,48,13,10,35,100,101,102,105,110,101,32,69,79,84,72,69
,82,32,32,32,32,32,32,32,32,32,32,49,51,49,13,10,35,100,101,102,105,110,101,32,69
,79,86,69,82,70,76,79,87,32,32,32,32,32,32,32,49,51,50,13,10,35,100,101,102,105
,110,101,32,69,79,87,78,69,82,68,69,65,68,32,32,32,32,32,32,49,51,51,13,10,35
,100,101,102,105,110,101,32,69,80,82,79,84,79,32,32,32,32,32,32,32,32,32,32,49,51
,52,13,10,35,100,101,102,105,110,101,32,69,80,82,79,84,79,78,79,83,85,80,80,79,82
,84,32,49,51,53,13,10,35,100,101,102,105,110,101,32,69,80,82,79,84,79,84,89,80,69
,32,32,32,32,32,32,49,51,54,13,10,35,100,101,102,105,110,101,32,69,84,73,77,69,32
,32,32,32,32,32,32,32,32,32,32,49,51,55,13,10,35,100,101,102,105,110,101,32,69,84
,73,77,69,68,79,85,84,32,32,32,32,32,32,32,49,51,56,13,10,35,100,101,102,105,110
,101,32,69,84,88,84,66,83,89,32,32,32,32,32,32,32,32,32,49,51,57,13,10,35,100
,101,102,105,110,101,32,69,87,79,85,76,68,66,76,79,67,75,32,32,32,32,32,49,52,48
,13,10,13,10
};

static const char file_string_h[] = {



32,13,10,116,121,112,101,100,101,102,32,105,110,116,32,101,114,114,110,111,95,116,59,13,10
,116,121,112,101,100,101,102,32,117,110,115,105,103,110,101,100,32,108,111,110,103,32,115,105,122
,101,95,116,59,13,10,116,121,112,101,100,101,102,32,117,110,115,105,103,110,101,100,32,108,111
,110,103,32,114,115,105,122,101,95,116,59,13,10,116,121,112,101,100,101,102,32,105,110,116,32
,119,99,104,97,114,95,116,59,13,10,118,111,105,100,42,32,109,101,109,99,104,114,40,118,111
,105,100,32,99,111,110,115,116,42,32,95,66,117,102,44,32,105,110,116,32,95,86,97,108,44
,32,115,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,105,110,116,32
,109,101,109,99,109,112,40,118,111,105,100,32,99,111,110,115,116,42,32,95,66,117,102,49,44
,32,118,111,105,100,32,99,111,110,115,116,42,32,95,66,117,102,50,44,32,115,105,122,101,95
,116,32,95,83,105,122,101,41,59,13,10,118,111,105,100,42,32,109,101,109,99,112,121,40,118
,111,105,100,42,32,95,68,115,116,44,32,118,111,105,100,32,99,111,110,115,116,42,32,95,83
,114,99,44,32,115,105,122,101,95,116,32,95,83,105,122,101,41,59,13,10,118,111,105,100,42
,32,109,101,109,109,111,118,101,40,118,111,105,100,42,32,95,68,115,116,44,32,118,111,105,100
,32,99,111,110,115,116,42,32,95,83,114,99,44,32,115,105,122,101,95,116,32,95,83,105,122
,101,41,59,13,10,118,111,105,100,42,32,109,101,109,115,101,116,40,118,111,105,100,42,32,95
,68,115,116,44,32,105,110,116,32,95,86,97,108,44,32,115,105,122,101,95,116,32,95,83,105
,122,101,41,59,13,10,99,104,97,114,42,32,115,116,114,99,104,114,40,99,104,97,114,32,99
,111,110,115,116,42,32,95,83,116,114,44,32,105,110,116,32,95,86,97,108,41,59,13,10,99
,104,97,114,32,42,115,116,114,99,112,121,40,95,67,116,111,114,32,99,104,97,114,32,42,114
,101,115,116,114,105,99,116,32,100,101,115,116,44,32,99,111,110,115,116,32,99,104,97,114,32
,42,114,101,115,116,114,105,99,116,32,115,114,99,32,41,59,13,10,99,104,97,114,42,32,115
,116,114,114,99,104,114,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,44,32
,105,110,116,32,95,67,104,41,59,13,10,99,104,97,114,42,32,115,116,114,115,116,114,40,99
,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,44,32,99,104,97,114,32,99,111,110
,115,116,42,32,95,83,117,98,83,116,114,41,59,13,10,119,99,104,97,114,95,116,42,32,119
,99,115,99,104,114,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114
,44,32,119,99,104,97,114,95,116,32,95,67,104,41,59,13,10,119,99,104,97,114,95,116,42
,32,119,99,115,114,99,104,114,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95
,83,116,114,44,32,119,99,104,97,114,95,116,32,95,67,104,41,59,13,10,119,99,104,97,114
,95,116,42,32,119,99,115,115,116,114,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42
,32,95,83,116,114,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,117
,98,83,116,114,41,59,13,10,115,116,97,116,105,99,32,105,110,108,105,110,101,32,101,114,114
,110,111,95,116,32,109,101,109,99,112,121,95,115,40,118,111,105,100,42,32,99,111,110,115,116
,32,95,68,101,115,116,105,110,97,116,105,111,110,44,32,114,115,105,122,101,95,116,32,99,111
,110,115,116,32,95,68,101,115,116,105,110,97,116,105,111,110,83,105,122,101,44,32,118,111,105
,100,32,99,111,110,115,116,42,32,99,111,110,115,116,32,95,83,111,117,114,99,101,44,32,114
,115,105,122,101,95,116,32,99,111,110,115,116,32,95,83,111,117,114,99,101,83,105,122,101,41
,59,13,10,115,116,97,116,105,99,32,105,110,108,105,110,101,32,101,114,114,110,111,95,116,32
,109,101,109,109,111,118,101,95,115,40,118,111,105,100,42,32,99,111,110,115,116,32,95,68,101
,115,116,105,110,97,116,105,111,110,44,32,114,115,105,122,101,95,116,32,99,111,110,115,116,32
,95,68,101,115,116,105,110,97,116,105,111,110,83,105,122,101,44,32,118,111,105,100,32,99,111
,110,115,116,42,32,99,111,110,115,116,32,95,83,111,117,114,99,101,44,32,114,115,105,122,101
,95,116,32,99,111,110,115,116,32,95,83,111,117,114,99,101,83,105,122,101,41,59,13,10,105
,110,116,32,95,109,101,109,105,99,109,112,40,118,111,105,100,32,99,111,110,115,116,42,32,95
,66,117,102,49,44,32,118,111,105,100,32,99,111,110,115,116,42,32,95,66,117,102,50,44,32
,115,105,122,101,95,116,32,95,83,105,122,101,41,59,13,10,118,111,105,100,42,32,109,101,109
,99,99,112,121,40,118,111,105,100,42,32,95,68,115,116,44,32,118,111,105,100,32,99,111,110
,115,116,42,32,95,83,114,99,44,32,105,110,116,32,95,86,97,108,44,32,115,105,122,101,95
,116,32,95,83,105,122,101,41,59,13,10,105,110,116,32,109,101,109,105,99,109,112,40,118,111
,105,100,32,99,111,110,115,116,42,32,95,66,117,102,49,44,32,118,111,105,100,32,99,111,110
,115,116,42,32,95,66,117,102,50,44,32,115,105,122,101,95,116,32,95,83,105,122,101,41,59
,13,10,101,114,114,110,111,95,116,32,119,99,115,99,97,116,95,115,40,119,99,104,97,114,95
,116,42,32,95,68,101,115,116,105,110,97,116,105,111,110,44,32,114,115,105,122,101,95,116,32
,95,83,105,122,101,73,110,87,111,114,100,115,44,32,119,99,104,97,114,95,116,32,99,111,110
,115,116,42,32,95,83,111,117,114,99,101,41,59,13,10,101,114,114,110,111,95,116,32,119,99
,115,99,112,121,95,115,40,119,99,104,97,114,95,116,42,32,95,68,101,115,116,105,110,97,116
,105,111,110,44,32,114,115,105,122,101,95,116,32,95,83,105,122,101,73,110,87,111,114,100,115
,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,111,117,114,99,101,41
,59,13,10,101,114,114,110,111,95,116,32,119,99,115,110,99,97,116,95,115,40,119,99,104,97
,114,95,116,42,32,95,68,101,115,116,105,110,97,116,105,111,110,44,32,114,115,105,122,101,95
,116,32,95,83,105,122,101,73,110,87,111,114,100,115,44,32,119,99,104,97,114,95,116,32,99
,111,110,115,116,42,32,95,83,111,117,114,99,101,44,32,114,115,105,122,101,95,116,32,95,77
,97,120,67,111,117,110,116,41,59,13,10,101,114,114,110,111,95,116,32,119,99,115,110,99,112
,121,95,115,40,119,99,104,97,114,95,116,42,32,95,68,101,115,116,105,110,97,116,105,111,110
,44,32,114,115,105,122,101,95,116,32,95,83,105,122,101,73,110,87,111,114,100,115,44,32,119
,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,111,117,114,99,101,44,32,114,115
,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,119,99,104,97,114,95
,116,42,32,119,99,115,116,111,107,95,115,40,119,99,104,97,114,95,116,42,32,95,83,116,114
,105,110,103,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,68,101,108,105
,109,105,116,101,114,44,32,119,99,104,97,114,95,116,42,42,32,95,67,111,110,116,101,120,116
,41,59,13,10,119,99,104,97,114,95,116,42,32,95,119,99,115,100,117,112,40,119,99,104,97
,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,41,59,13,10,119,99,104
,97,114,95,116,42,32,119,99,115,99,97,116,40,119,99,104,97,114,95,116,42,32,95,68,101
,115,116,105,110,97,116,105,111,110,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42
,32,95,83,111,117,114,99,101,41,59,32,105,110,116,32,119,99,115,99,109,112,40,119,99,104
,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,49,44,32,119,99,104
,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,50,41,59,13,10,119
,99,104,97,114,95,116,42,32,119,99,115,99,112,121,40,119,99,104,97,114,95,116,42,32,95
,68,101,115,116,105,110,97,116,105,111,110,44,32,119,99,104,97,114,95,116,32,99,111,110,115
,116,42,32,95,83,111,117,114,99,101,41,59,32,115,105,122,101,95,116,32,119,99,115,99,115
,112,110,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103
,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,67,111,110,116,114,111,108
,41,59,13,10,115,105,122,101,95,116,32,119,99,115,108,101,110,40,119,99,104,97,114,95,116
,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,41,59,13,10,115,105,122,101,95,116
,32,119,99,115,110,108,101,110,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95
,83,111,117,114,99,101,44,32,115,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41
,59,13,10,115,116,97,116,105,99,32,105,110,108,105,110,101,32,115,105,122,101,95,116,32,119
,99,115,110,108,101,110,95,115,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95
,83,111,117,114,99,101,44,32,115,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41
,59,13,10,119,99,104,97,114,95,116,42,32,119,99,115,110,99,97,116,40,119,99,104,97,114
,95,116,42,32,95,68,101,115,116,105,110,97,116,105,111,110,44,32,119,99,104,97,114,95,116
,32,99,111,110,115,116,42,32,95,83,111,117,114,99,101,44,32,115,105,122,101,95,116,32,95
,67,111,117,110,116,41,59,13,10,105,110,116,32,119,99,115,110,99,109,112,40,119,99,104,97
,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,49,44,32,119,99,104,97
,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,50,44,32,115,105,122,101
,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,119,99,104,97,114,95,116,42,32
,119,99,115,110,99,112,121,40,119,99,104,97,114,95,116,42,32,95,68,101,115,116,105,110,97
,116,105,111,110,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,111,117
,114,99,101,44,32,115,105,122,101,95,116,32,95,67,111,117,110,116,41,59,13,10,119,99,104
,97,114,95,116,42,32,119,99,115,112,98,114,107,40,119,99,104,97,114,95,116,32,99,111,110
,115,116,42,32,95,83,116,114,105,110,103,44,32,119,99,104,97,114,95,116,32,99,111,110,115
,116,42,32,95,67,111,110,116,114,111,108,41,59,13,10,115,105,122,101,95,116,32,119,99,115
,115,112,110,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110
,103,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,67,111,110,116,114,111
,108,41,59,13,10,119,99,104,97,114,95,116,42,32,119,99,115,116,111,107,40,119,99,104,97
,114,95,116,42,32,95,83,116,114,105,110,103,44,32,119,99,104,97,114,95,116,32,99,111,110
,115,116,42,32,95,68,101,108,105,109,105,116,101,114,44,32,119,99,104,97,114,95,116,42,42
,32,95,67,111,110,116,101,120,116,41,59,13,10,115,105,122,101,95,116,32,119,99,115,120,102
,114,109,40,119,99,104,97,114,95,116,42,32,95,68,101,115,116,105,110,97,116,105,111,110,44
,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,111,117,114,99,101,44,32
,115,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,105,110,116,32,119
,99,115,99,111,108,108,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116
,114,105,110,103,49,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116
,114,105,110,103,50,41,59,13,10,119,99,104,97,114,95,116,42,32,119,99,115,100,117,112,40
,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,41,59,13
,10,105,110,116,32,119,99,115,105,99,109,112,40,119,99,104,97,114,95,116,32,99,111,110,115
,116,42,32,95,83,116,114,105,110,103,49,44,32,119,99,104,97,114,95,116,32,99,111,110,115
,116,42,32,95,83,116,114,105,110,103,50,41,59,13,10,105,110,116,32,119,99,115,110,105,99
,109,112,40,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103
,49,44,32,119,99,104,97,114,95,116,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103
,50,44,32,115,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,119,99
,104,97,114,95,116,42,32,119,99,115,110,115,101,116,40,119,99,104,97,114,95,116,42,32,95
,83,116,114,105,110,103,44,32,119,99,104,97,114,95,116,32,95,86,97,108,117,101,44,32,115
,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,119,99,104,97,114,95
,116,42,32,119,99,115,114,101,118,40,119,99,104,97,114,95,116,42,32,95,83,116,114,105,110
,103,41,59,13,10,119,99,104,97,114,95,116,42,32,119,99,115,115,101,116,40,119,99,104,97
,114,95,116,42,32,95,83,116,114,105,110,103,44,32,119,99,104,97,114,95,116,32,95,86,97
,108,117,101,41,59,13,10,119,99,104,97,114,95,116,42,32,119,99,115,108,119,114,40,119,99
,104,97,114,95,116,42,32,95,83,116,114,105,110,103,41,59,32,119,99,104,97,114,95,116,42
,32,119,99,115,117,112,114,40,119,99,104,97,114,95,116,42,32,95,83,116,114,105,110,103,41
,59,13,10,105,110,116,32,119,99,115,105,99,111,108,108,40,119,99,104,97,114,95,116,32,99
,111,110,115,116,42,32,95,83,116,114,105,110,103,49,44,32,119,99,104,97,114,95,116,32,99
,111,110,115,116,42,32,95,83,116,114,105,110,103,50,41,59,13,10,99,104,97,114,42,32,115
,116,114,116,111,107,95,115,40,99,104,97,114,42,32,95,83,116,114,105,110,103,44,32,99,104
,97,114,32,99,111,110,115,116,42,32,95,68,101,108,105,109,105,116,101,114,44,32,99,104,97
,114,42,42,32,95,67,111,110,116,101,120,116,41,59,13,10,118,111,105,100,42,32,95,109,101
,109,99,99,112,121,40,118,111,105,100,42,32,95,68,115,116,44,32,118,111,105,100,32,99,111
,110,115,116,42,32,95,83,114,99,44,32,105,110,116,32,95,86,97,108,44,32,115,105,122,101
,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,99,104,97,114,42,32,115,116,114
,99,97,116,40,99,104,97,114,42,32,95,68,101,115,116,105,110,97,116,105,111,110,44,32,99
,104,97,114,32,99,111,110,115,116,42,32,95,83,111,117,114,99,101,41,59,13,10,105,110,116
,32,115,116,114,99,109,112,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,49
,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,50,41,59,13,10,105,110
,116,32,115,116,114,99,111,108,108,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116
,114,105,110,103,49,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,105,110
,103,50,41,59,13,10,99,104,97,114,42,32,115,116,114,101,114,114,111,114,40,105,110,116,32
,95,69,114,114,111,114,77,101,115,115,97,103,101,41,59,13,10,115,105,122,101,95,116,32,115
,116,114,108,101,110,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,41,59,13
,10,99,104,97,114,42,32,115,116,114,110,99,97,116,40,99,104,97,114,42,32,95,68,101,115
,116,105,110,97,116,105,111,110,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,83,111
,117,114,99,101,44,32,115,105,122,101,95,116,32,95,67,111,117,110,116,41,59,13,10,105,110
,116,32,115,116,114,110,99,109,112,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116
,114,49,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,50,44,32,115,105
,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,99,104,97,114,42,32,115
,116,114,110,99,112,121,40,99,104,97,114,42,32,95,68,101,115,116,105,110,97,116,105,111,110
,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,83,111,117,114,99,101,44,32,115,105
,122,101,95,116,32,95,67,111,117,110,116,41,59,13,10,115,105,122,101,95,116,32,115,116,114
,110,108,101,110,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,44
,32,115,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,115,116,97,116
,105,99,32,105,110,108,105,110,101,32,115,105,122,101,95,116,32,115,116,114,110,108,101,110,95
,115,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,44,32,115,105
,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,99,104,97,114,42,32,115
,116,114,112,98,114,107,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,44,32
,99,104,97,114,32,99,111,110,115,116,42,32,95,67,111,110,116,114,111,108,41,59,13,10,115
,105,122,101,95,116,32,115,116,114,115,112,110,40,99,104,97,114,32,99,111,110,115,116,42,32
,95,83,116,114,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,67,111,110,116,114,111
,108,41,59,13,10,99,104,97,114,42,32,115,116,114,116,111,107,40,99,104,97,114,42,32,95
,83,116,114,105,110,103,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,68,101,108,105
,109,105,116,101,114,41,59,13,10,13,10,35,105,102,32,100,101,102,105,110,101,100,40,95,95
,83,84,68,67,95,79,87,78,69,82,83,72,73,80,95,95,41,32,13,10,99,104,97,114,42
,32,95,79,119,110,101,114,32,95,79,112,116,32,115,116,114,100,117,112,40,99,104,97,114,32
,99,111,110,115,116,42,32,95,83,116,114,105,110,103,41,59,13,10,35,101,108,115,101,13,10
,99,104,97,114,42,32,115,116,114,100,117,112,40,99,104,97,114,32,99,111,110,115,116,42,32
,95,83,116,114,105,110,103,41,59,13,10,35,101,110,100,105,102,13,10,13,10,105,110,116,32
,115,116,114,99,109,112,105,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,105
,110,103,49,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,50
,41,59,13,10,105,110,116,32,115,116,114,105,99,109,112,40,99,104,97,114,32,99,111,110,115
,116,42,32,95,83,116,114,105,110,103,49,44,32,99,104,97,114,32,99,111,110,115,116,42,32
,95,83,116,114,105,110,103,50,41,59,13,10,99,104,97,114,42,32,115,116,114,108,119,114,40
,99,104,97,114,42,32,95,83,116,114,105,110,103,41,59,13,10,105,110,116,32,115,116,114,110
,105,99,109,112,40,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,49
,44,32,99,104,97,114,32,99,111,110,115,116,42,32,95,83,116,114,105,110,103,50,44,32,115
,105,122,101,95,116,32,95,77,97,120,67,111,117,110,116,41,59,13,10,99,104,97,114,42,32
,115,116,114,110,115,101,116,40,99,104,97,114,42,32,95,83,116,114,105,110,103,44,32,105,110
,116,32,95,86,97,108,117,101,44,32,115,105,122,101,95,116,32,95,77,97,120,67,111,117,110
,116,41,59,13,10,99,104,97,114,42,32,115,116,114,114,101,118,40,99,104,97,114,42,32,95
,83,116,114,105,110,103,41,59,13,10,99,104,97,114,42,32,115,116,114,115,101,116,40,99,104
,97,114,42,32,95,83,116,114,105,110,103,44,32,105,110,116,32,95,86,97,108,117,101,41,59
,32,99,104,97,114,42,32,115,116,114,117,112,114,40,99,104,97,114,42,32,95,83,116,114,105
,110,103,41,59
};

static const char file_math_h[] = {



47,42,13,10,32,42,32,32,84,104,105,115,32,102,105,108,101,32,105,115,32,112,97,114,116
,32,111,102,32,99,97,107,101,32,99,111,109,112,105,108,101,114,13,10,32,42,32,32,104,116
,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,116,104,114,97,100,97,109,115
,47,99,97,107,101,13,10,42,47,13,10,13,10,35,112,114,97,103,109,97,32,111,110,99,101
,13,10,13,10,100,111,117,98,108,101,32,97,99,111,115,40,100,111,117,98,108,101,32,95,95
,120,41,59,13,10,100,111,117,98,108,101,32,97,115,105,110,40,100,111,117,98,108,101,32,95
,95,120,41,59,13,10,100,111,117,98,108,101,32,97,116,97,110,40,100,111,117,98,108,101,32
,95,95,120,41,59,13,10,100,111,117,98,108,101,32,97,116,97,110,50,40,100,111,117,98,108
,101,32,95,95,121,44,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98
,108,101,32,99,111,115,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98
,108,101,32,115,105,110,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98
,108,101,32,116,97,110,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98
,108,101,32,99,111,115,104,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117
,98,108,101,32,115,105,110,104,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111
,117,98,108,101,32,116,97,110,104,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100
,111,117,98,108,101,32,97,99,111,115,104,40,100,111,117,98,108,101,32,95,95,120,41,59,13
,10,100,111,117,98,108,101,32,97,115,105,110,104,40,100,111,117,98,108,101,32,95,95,120,41
,59,13,10,100,111,117,98,108,101,32,97,116,97,110,104,40,100,111,117,98,108,101,32,95,95
,120,41,59,13,10,100,111,117,98,108,101,32,101,120,112,40,100,111,117,98,108,101,32,95,95
,120,41,59,13,10,100,111,117,98,108,101,32,102,114,101,120,112,40,100,111,117,98,108,101,32
,95,95,120,44,32,105,110,116,42,32,95,95,101,120,112,111,110,101,110,116,41,59,13,10,100
,111,117,98,108,101,32,108,100,101,120,112,40,100,111,117,98,108,101,32,95,95,120,44,32,105
,110,116,32,95,95,101,120,112,111,110,101,110,116,41,59,13,10,100,111,117,98,108,101,32,108
,111,103,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101,32,108
,111,103,49,48,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101
,32,109,111,100,102,40,100,111,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101,42
,32,95,95,105,112,116,114,41,59,13,10,100,111,117,98,108,101,32,101,120,112,109,49,40,100
,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101,32,108,111,103,49,112
,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101,32,108,111,103
,98,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101,32,101,120
,112,50,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101,32,108
,111,103,50,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101,32
,112,111,119,40,100,111,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101,32,95,95
,121,41,59,13,10,100,111,117,98,108,101,32,115,113,114,116,40,100,111,117,98,108,101,32,95
,95,120,41,59,13,10,100,111,117,98,108,101,32,104,121,112,111,116,40,100,111,117,98,108,101
,32,95,95,120,44,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10,100,111,117,98,108
,101,32,99,98,114,116,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98
,108,101,32,99,101,105,108,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117
,98,108,101,32,102,97,98,115,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111
,117,98,108,101,32,102,108,111,111,114,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10
,100,111,117,98,108,101,32,102,109,111,100,40,100,111,117,98,108,101,32,95,95,120,44,32,100
,111,117,98,108,101,32,95,95,121,41,59,13,10,105,110,116,32,105,115,105,110,102,40,100,111
,117,98,108,101,32,95,95,118,97,108,117,101,41,59,13,10,105,110,116,32,102,105,110,105,116
,101,40,100,111,117,98,108,101,32,95,95,118,97,108,117,101,41,59,13,10,100,111,117,98,108
,101,32,100,114,101,109,40,100,111,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101
,32,95,95,121,41,59,13,10,100,111,117,98,108,101,32,115,105,103,110,105,102,105,99,97,110
,100,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111,117,98,108,101,32,99,111
,112,121,115,105,103,110,40,100,111,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101
,32,95,95,121,41,59,13,10,100,111,117,98,108,101,32,110,97,110,40,99,111,110,115,116,32
,99,104,97,114,42,32,95,95,116,97,103,98,41,59,13,10,105,110,116,32,105,115,110,97,110
,40,100,111,117,98,108,101,32,95,95,118,97,108,117,101,41,59,13,10,100,111,117,98,108,101
,32,106,48,40,100,111,117,98,108,101,41,59,13,10,100,111,117,98,108,101,32,106,49,40,100
,111,117,98,108,101,41,59,13,10,100,111,117,98,108,101,32,106,110,40,105,110,116,44,32,100
,111,117,98,108,101,41,59,13,10,100,111,117,98,108,101,32,121,48,40,100,111,117,98,108,101
,41,59,13,10,100,111,117,98,108,101,32,121,49,40,100,111,117,98,108,101,41,59,13,10,100
,111,117,98,108,101,32,121,110,40,105,110,116,44,32,100,111,117,98,108,101,41,59,13,10,100
,111,117,98,108,101,32,101,114,102,40,100,111,117,98,108,101,41,59,13,10,100,111,117,98,108
,101,32,101,114,102,99,40,100,111,117,98,108,101,41,59,13,10,100,111,117,98,108,101,32,108
,103,97,109,109,97,40,100,111,117,98,108,101,41,59,13,10,100,111,117,98,108,101,32,116,103
,97,109,109,97,40,100,111,117,98,108,101,41,59,13,10,100,111,117,98,108,101,32,103,97,109
,109,97,40,100,111,117,98,108,101,41,59,13,10,100,111,117,98,108,101,32,108,103,97,109,109
,97,95,114,40,100,111,117,98,108,101,44,32,105,110,116,42,32,95,95,115,105,103,110,103,97
,109,112,41,59,13,10,100,111,117,98,108,101,32,114,105,110,116,40,100,111,117,98,108,101,32
,95,95,120,41,59,13,10,100,111,117,98,108,101,32,110,101,120,116,97,102,116,101,114,40,100
,111,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10
,100,111,117,98,108,101,32,110,101,120,116,116,111,119,97,114,100,40,100,111,117,98,108,101,32
,95,95,120,44,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10,100
,111,117,98,108,101,32,114,101,109,97,105,110,100,101,114,40,100,111,117,98,108,101,32,95,95
,120,44,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10,100,111,117,98,108,101,32,115
,99,97,108,98,110,40,100,111,117,98,108,101,32,95,95,120,44,32,105,110,116,32,95,95,110
,41,59,13,10,105,110,116,32,105,108,111,103,98,40,100,111,117,98,108,101,32,95,95,120,41
,59,13,10,100,111,117,98,108,101,32,115,99,97,108,98,108,110,40,100,111,117,98,108,101,32
,95,95,120,44,32,108,111,110,103,32,105,110,116,32,95,95,110,41,59,13,10,100,111,117,98
,108,101,32,110,101,97,114,98,121,105,110,116,40,100,111,117,98,108,101,32,95,95,120,41,59
,13,10,100,111,117,98,108,101,32,114,111,117,110,100,40,100,111,117,98,108,101,32,95,95,120
,41,59,13,10,100,111,117,98,108,101,32,116,114,117,110,99,40,100,111,117,98,108,101,32,95
,95,120,41,59,13,10,100,111,117,98,108,101,32,114,101,109,113,117,111,40,100,111,117,98,108
,101,32,95,95,120,44,32,100,111,117,98,108,101,32,95,95,121,44,32,105,110,116,42,32,95
,95,113,117,111,41,59,13,10,108,111,110,103,32,105,110,116,32,108,114,105,110,116,40,100,111
,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,108,111,110,103,32,105,110,116
,32,108,108,114,111,117,110,100,40,100,111,117,98,108,101,32,95,95,120,41,59,13,10,100,111
,117,98,108,101,32,102,100,105,109,40,100,111,117,98,108,101,32,95,95,120,44,32,100,111,117
,98,108,101,32,95,95,121,41,59,13,10,100,111,117,98,108,101,32,102,109,97,120,40,100,111
,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10,100
,111,117,98,108,101,32,102,109,105,110,40,100,111,117,98,108,101,32,95,95,120,44,32,100,111
,117,98,108,101,32,95,95,121,41,59,13,10,100,111,117,98,108,101,32,102,109,97,40,100,111
,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101,32,95,95,121,44,32,100,111,117
,98,108,101,32,95,95,122,41,59,13,10,100,111,117,98,108,101,32,115,99,97,108,98,40,100
,111,117,98,108,101,32,95,95,120,44,32,100,111,117,98,108,101,32,95,95,110,41,59,13,10
,102,108,111,97,116,32,97,99,111,115,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10
,102,108,111,97,116,32,97,115,105,110,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10
,102,108,111,97,116,32,97,116,97,110,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10
,102,108,111,97,116,32,97,116,97,110,50,102,40,102,108,111,97,116,32,95,95,121,44,32,102
,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,99,111,115,102,40,102,108
,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,115,105,110,102,40,102,108,111
,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,116,97,110,102,40,102,108,111,97
,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,99,111,115,104,102,40,102,108,111,97
,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,115,105,110,104,102,40,102,108,111,97
,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,116,97,110,104,102,40,102,108,111,97
,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,97,99,111,115,104,102,40,102,108,111
,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,97,115,105,110,104,102,40,102,108
,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,97,116,97,110,104,102,40,102
,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,101,120,112,102,40,102,108
,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,102,114,101,120,112,102,40,102
,108,111,97,116,32,95,95,120,44,32,105,110,116,42,32,95,95,101,120,112,111,110,101,110,116
,41,59,13,10,102,108,111,97,116,32,108,100,101,120,112,102,40,102,108,111,97,116,32,95,95
,120,44,32,105,110,116,32,95,95,101,120,112,111,110,101,110,116,41,59,13,10,102,108,111,97
,116,32,108,111,103,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116
,32,108,111,103,49,48,102,40,102,108,111,97,116,32,95,95,120,41,59,32,102,108,111,97,116
,32,95,95,108,111,103,49,48,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108
,111,97,116,32,109,111,100,102,102,40,102,108,111,97,116,32,95,95,120,44,32,102,108,111,97
,116,42,32,95,95,105,112,116,114,41,59,13,10,102,108,111,97,116,32,101,120,112,109,49,102
,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,108,111,103,49,112
,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,108,111,103,98
,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,101,120,112,50
,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,108,111,103,50
,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,112,111,119,102
,40,102,108,111,97,116,32,95,95,120,44,32,102,108,111,97,116,32,95,95,121,41,59,13,10
,102,108,111,97,116,32,115,113,114,116,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10
,102,108,111,97,116,32,104,121,112,111,116,102,40,102,108,111,97,116,32,95,95,120,44,32,102
,108,111,97,116,32,95,95,121,41,59,13,10,102,108,111,97,116,32,99,98,114,116,102,40,102
,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,99,101,105,108,102,40,102
,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,102,97,98,115,102,40,102
,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,102,108,111,111,114,102,40
,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,102,109,111,100,102,40
,102,108,111,97,116,32,95,95,120,44,32,102,108,111,97,116,32,95,95,121,41,59,13,10,105
,110,116,32,105,115,105,110,102,102,40,102,108,111,97,116,32,95,95,118,97,108,117,101,41,59
,13,10,105,110,116,32,102,105,110,105,116,101,102,40,102,108,111,97,116,32,95,95,118,97,108
,117,101,41,59,13,10,102,108,111,97,116,32,100,114,101,109,102,40,102,108,111,97,116,32,95
,95,120,44,32,102,108,111,97,116,32,95,95,121,41,59,13,10,102,108,111,97,116,32,115,105
,103,110,105,102,105,99,97,110,100,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102
,108,111,97,116,32,99,111,112,121,115,105,103,110,102,40,102,108,111,97,116,32,95,95,120,44
,32,102,108,111,97,116,32,95,95,121,41,59,13,10,102,108,111,97,116,32,110,97,110,102,40
,99,111,110,115,116,32,99,104,97,114,42,32,95,95,116,97,103,98,41,59,13,10,105,110,116
,32,105,115,110,97,110,102,40,102,108,111,97,116,32,95,95,118,97,108,117,101,41,59,13,10
,102,108,111,97,116,32,106,48,102,40,102,108,111,97,116,41,59,13,10,102,108,111,97,116,32
,106,49,102,40,102,108,111,97,116,41,59,13,10,102,108,111,97,116,32,106,110,102,40,105,110
,116,44,32,102,108,111,97,116,41,59,13,10,102,108,111,97,116,32,121,48,102,40,102,108,111
,97,116,41,59,13,10,102,108,111,97,116,32,121,49,102,40,102,108,111,97,116,41,59,13,10
,102,108,111,97,116,32,121,110,102,40,105,110,116,44,32,102,108,111,97,116,41,59,13,10,102
,108,111,97,116,32,101,114,102,102,40,102,108,111,97,116,41,59,13,10,102,108,111,97,116,32
,101,114,102,99,102,40,102,108,111,97,116,41,59,13,10,102,108,111,97,116,32,108,103,97,109
,109,97,102,40,102,108,111,97,116,41,59,13,10,102,108,111,97,116,32,116,103,97,109,109,97
,102,40,102,108,111,97,116,41,59,13,10,102,108,111,97,116,32,103,97,109,109,97,102,40,102
,108,111,97,116,41,59,13,10,102,108,111,97,116,32,108,103,97,109,109,97,102,95,114,40,102
,108,111,97,116,44,32,105,110,116,42,32,95,95,115,105,103,110,103,97,109,112,41,59,13,10
,102,108,111,97,116,32,114,105,110,116,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10
,102,108,111,97,116,32,110,101,120,116,97,102,116,101,114,102,40,102,108,111,97,116,32,95,95
,120,44,32,102,108,111,97,116,32,95,95,121,41,59,13,10,102,108,111,97,116,32,110,101,120
,116,116,111,119,97,114,100,102,40,102,108,111,97,116,32,95,95,120,44,32,108,111,110,103,32
,100,111,117,98,108,101,32,95,95,121,41,59,13,10,102,108,111,97,116,32,114,101,109,97,105
,110,100,101,114,102,40,102,108,111,97,116,32,95,95,120,44,32,102,108,111,97,116,32,95,95
,121,41,59,13,10,102,108,111,97,116,32,115,99,97,108,98,110,102,40,102,108,111,97,116,32
,95,95,120,44,32,105,110,116,32,95,95,110,41,59,13,10,105,110,116,32,105,108,111,103,98
,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,115,99,97,108
,98,108,110,102,40,102,108,111,97,116,32,95,95,120,44,32,108,111,110,103,32,105,110,116,32
,95,95,110,41,59,13,10,102,108,111,97,116,32,110,101,97,114,98,121,105,110,116,102,40,102
,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,114,111,117,110,100,102,40
,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,116,114,117,110,99,102
,40,102,108,111,97,116,32,95,95,120,41,59,13,10,102,108,111,97,116,32,114,101,109,113,117
,111,102,40,102,108,111,97,116,32,95,95,120,44,32,102,108,111,97,116,32,95,95,121,44,32
,105,110,116,42,32,95,95,113,117,111,41,59,13,10,108,111,110,103,32,105,110,116,32,108,114
,105,110,116,102,40,102,108,111,97,116,32,95,95,120,41,59,13,10,108,111,110,103,32,108,111
,110,103,32,105,110,116,32,108,108,114,111,117,110,100,102,40,102,108,111,97,116,32,95,95,120
,41,59,13,10,102,108,111,97,116,32,102,100,105,109,102,40,102,108,111,97,116,32,95,95,120
,44,32,102,108,111,97,116,32,95,95,121,41,59,13,10,102,108,111,97,116,32,102,109,97,120
,102,40,102,108,111,97,116,32,95,95,120,44,32,102,108,111,97,116,32,95,95,121,41,59,13
,10,102,108,111,97,116,32,102,109,105,110,102,40,102,108,111,97,116,32,95,95,120,44,32,102
,108,111,97,116,32,95,95,121,41,59,13,10,102,108,111,97,116,32,102,109,97,102,40,102,108
,111,97,116,32,95,95,120,44,32,102,108,111,97,116,32,95,95,121,44,32,102,108,111,97,116
,32,95,95,122,41,59,13,10,102,108,111,97,116,32,115,99,97,108,98,102,40,102,108,111,97
,116,32,95,95,120,44,32,102,108,111,97,116,32,95,95,110,41,59,13,10,108,111,110,103,32
,100,111,117,98,108,101,32,97,99,111,115,108,40,108,111,110,103,32,100,111,117,98,108,101,32
,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,97,115,105,110,108,40
,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100
,111,117,98,108,101,32,97,116,97,110,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95
,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,97,116,97,110,50,108,40
,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121,44,32,108,111,110,103,32,100,111,117
,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,99,111
,115,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110
,103,32,100,111,117,98,108,101,32,115,105,110,108,40,108,111,110,103,32,100,111,117,98,108,101
,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,116,97,110,108,40
,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100
,111,117,98,108,101,32,99,111,115,104,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95
,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,115,105,110,104,108,40,108
,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111
,117,98,108,101,32,116,97,110,104,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95
,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,97,99,111,115,104,108,40,108
,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111
,117,98,108,101,32,97,115,105,110,104,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95
,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,97,116,97,110,104,108,40
,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100
,111,117,98,108,101,32,101,120,112,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95
,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,102,114,101,120,112,108,40,108
,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32,105,110,116,42,32,95,95,101,120
,112,111,110,101,110,116,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,108,100,101
,120,112,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32,105,110,116,32
,95,95,101,120,112,111,110,101,110,116,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101
,32,108,111,103,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10
,108,111,110,103,32,100,111,117,98,108,101,32,108,111,103,49,48,108,40,108,111,110,103,32,100
,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32
,109,111,100,102,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32,108,111
,110,103,32,100,111,117,98,108,101,42,32,95,95,105,112,116,114,41,59,13,10,108,111,110,103
,32,100,111,117,98,108,101,32,101,120,112,109,49,108,40,108,111,110,103,32,100,111,117,98,108
,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,108,111,103,49
,112,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110
,103,32,100,111,117,98,108,101,32,108,111,103,98,108,40,108,111,110,103,32,100,111,117,98,108
,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,101,120,112,50
,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103
,32,100,111,117,98,108,101,32,108,111,103,50,108,40,108,111,110,103,32,100,111,117,98,108,101
,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,112,111,119,108,40
,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100,111,117
,98,108,101,32,95,95,121,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,115,113
,114,116,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111
,110,103,32,100,111,117,98,108,101,32,104,121,112,111,116,108,40,108,111,110,103,32,100,111,117
,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121,41
,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,99,98,114,116,108,40,108,111,110,103
,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108
,101,32,99,101,105,108,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59
,13,10,108,111,110,103,32,100,111,117,98,108,101,32,102,97,98,115,108,40,108,111,110,103,32
,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101
,32,102,108,111,111,114,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59
,13,10,108,111,110,103,32,100,111,117,98,108,101,32,102,109,111,100,108,40,108,111,110,103,32
,100,111,117,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100,111,117,98,108,101,32,95
,95,121,41,59,13,10,105,110,116,32,105,115,105,110,102,108,40,108,111,110,103,32,100,111,117
,98,108,101,32,95,95,118,97,108,117,101,41,59,13,10,105,110,116,32,102,105,110,105,116,101
,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,118,97,108,117,101,41,59,13,10
,108,111,110,103,32,100,111,117,98,108,101,32,100,114,101,109,108,40,108,111,110,103,32,100,111
,117,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121
,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,115,105,103,110,105,102,105,99,97
,110,100,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111
,110,103,32,100,111,117,98,108,101,32,99,111,112,121,115,105,103,110,108,40,108,111,110,103,32
,100,111,117,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100,111,117,98,108,101,32,95
,95,121,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,110,97,110,108,40,99,111
,110,115,116,32,99,104,97,114,42,32,95,95,116,97,103,98,41,59,13,10,105,110,116,32,105
,115,110,97,110,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,118,97,108,117,101
,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,106,48,108,40,108,111,110,103,32
,100,111,117,98,108,101,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,106,49,108
,40,108,111,110,103,32,100,111,117,98,108,101,41,59,13,10,108,111,110,103,32,100,111,117,98
,108,101,32,106,110,108,40,105,110,116,44,32,108,111,110,103,32,100,111,117,98,108,101,41,59
,13,10,108,111,110,103,32,100,111,117,98,108,101,32,121,48,108,40,108,111,110,103,32,100,111
,117,98,108,101,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,121,49,108,40,108
,111,110,103,32,100,111,117,98,108,101,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101
,32,121,110,108,40,105,110,116,44,32,108,111,110,103,32,100,111,117,98,108,101,41,59,13,10
,108,111,110,103,32,100,111,117,98,108,101,32,101,114,102,108,40,108,111,110,103,32,100,111,117
,98,108,101,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,101,114,102,99,108,40
,108,111,110,103,32,100,111,117,98,108,101,41,59,13,10,108,111,110,103,32,100,111,117,98,108
,101,32,108,103,97,109,109,97,108,40,108,111,110,103,32,100,111,117,98,108,101,41,59,13,10
,108,111,110,103,32,100,111,117,98,108,101,32,116,103,97,109,109,97,108,40,108,111,110,103,32
,100,111,117,98,108,101,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,103,97,109
,109,97,108,40,108,111,110,103,32,100,111,117,98,108,101,41,59,13,10,108,111,110,103,32,100
,111,117,98,108,101,32,108,103,97,109,109,97,108,95,114,40,108,111,110,103,32,100,111,117,98
,108,101,44,32,105,110,116,42,32,95,95,115,105,103,110,103,97,109,112,41,59,13,10,108,111
,110,103,32,100,111,117,98,108,101,32,114,105,110,116,108,40,108,111,110,103,32,100,111,117,98
,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,110,101,120
,116,97,102,116,101,114,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32
,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10,108,111,110,103,32,100
,111,117,98,108,101,32,110,101,120,116,116,111,119,97,114,100,108,40,108,111,110,103,32,100,111
,117,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121
,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,114,101,109,97,105,110,100,101,114
,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100
,111,117,98,108,101,32,95,95,121,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32
,115,99,97,108,98,110,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32
,105,110,116,32,95,95,110,41,59,13,10,105,110,116,32,105,108,111,103,98,108,40,108,111,110
,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98
,108,101,32,115,99,97,108,98,108,110,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95
,95,120,44,32,108,111,110,103,32,105,110,116,32,95,95,110,41,59,13,10,108,111,110,103,32
,100,111,117,98,108,101,32,110,101,97,114,98,121,105,110,116,108,40,108,111,110,103,32,100,111
,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,114
,111,117,110,100,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10
,108,111,110,103,32,100,111,117,98,108,101,32,116,114,117,110,99,108,40,108,111,110,103,32,100
,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32
,114,101,109,113,117,111,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32
,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121,44,32,105,110,116,42,32,95,95,113
,117,111,41,59,13,10,108,111,110,103,32,105,110,116,32,108,114,105,110,116,108,40,108,111,110
,103,32,100,111,117,98,108,101,32,95,95,120,41,59,13,10,108,111,110,103,32,108,111,110,103
,32,105,110,116,32,108,108,114,111,117,110,100,108,40,108,111,110,103,32,100,111,117,98,108,101
,32,95,95,120,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,102,100,105,109,108
,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32,108,111,110,103,32,100,111
,117,98,108,101,32,95,95,121,41,59,13,10,108,111,110,103,32,100,111,117,98,108,101,32,102
,109,97,120,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44,32,108,111,110
,103,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10,108,111,110,103,32,100,111,117,98
,108,101,32,102,109,105,110,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44
,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121,41,59,13,10,108,111,110,103,32
,100,111,117,98,108,101,32,102,109,97,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95
,95,120,44,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,121,44,32,108,111,110,103
,32,100,111,117,98,108,101,32,95,95,122,41,59,13,10,108,111,110,103,32,100,111,117,98,108
,101,32,115,99,97,108,98,108,40,108,111,110,103,32,100,111,117,98,108,101,32,95,95,120,44
,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,110,41,59,13,10
};

static const char file_stdlib_h[] = {



47,42,13,10,32,42,32,32,84,104,105,115,32,102,105,108,101,32,105,115,32,112,97,114,116
,32,111,102,32,99,97,107,101,32,99,111,109,112,105,108,101,114,13,10,32,42,32,32,104,116
,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,116,104,114,97,100,97,109,115
,47,99,97,107,101,13,10,42,47,13,10,13,10,116,121,112,101,100,101,102,32,108,111,110,103
,32,108,111,110,103,32,102,112,111,115,95,116,59,13,10,116,121,112,101,100,101,102,32,117,110
,115,105,103,110,101,100,32,108,111,110,103,32,115,105,122,101,95,116,59,13,10,13,10,35,100
,101,102,105,110,101,32,69,88,73,84,95,83,85,67,67,69,83,83,32,48,13,10,35,100,101
,102,105,110,101,32,69,88,73,84,95,70,65,73,76,85,82,69,32,49,13,10,35,100,101,102
,105,110,101,32,78,85,76,76,32,40,40,118,111,105,100,42,41,48,41,13,10,13,10,116,121
,112,101,100,101,102,32,105,110,116,32,119,99,104,97,114,95,116,59,13,10,91,91,110,111,100
,105,115,99,97,114,100,93,93,32,100,111,117,98,108,101,32,97,116,111,102,40,99,111,110,115
,116,32,99,104,97,114,42,32,110,112,116,114,41,59,13,10,91,91,110,111,100,105,115,99,97
,114,100,93,93,32,105,110,116,32,97,116,111,105,40,99,111,110,115,116,32,99,104,97,114,42
,32,110,112,116,114,41,59,13,10,91,91,110,111,100,105,115,99,97,114,100,93,93,32,108,111
,110,103,32,105,110,116,32,97,116,111,108,40,99,111,110,115,116,32,99,104,97,114,42,32,110
,112,116,114,41,59,13,10,91,91,110,111,100,105,115,99,97,114,100,93,93,32,108,111,110,103
,32,108,111,110,103,32,105,110,116,32,97,116,111,108,108,40,99,111,110,115,116,32,99,104,97
,114,42,32,110,112,116,114,41,59,13,10,100,111,117,98,108,101,32,115,116,114,116,111,100,40
,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,110,112,116,114
,44,32,99,104,97,114,42,42,32,114,101,115,116,114,105,99,116,32,101,110,100,112,116,114,41
,59,13,10,102,108,111,97,116,32,115,116,114,116,111,102,40,99,111,110,115,116,32,99,104,97
,114,42,32,114,101,115,116,114,105,99,116,32,110,112,116,114,44,32,99,104,97,114,42,42,32
,114,101,115,116,114,105,99,116,32,101,110,100,112,116,114,41,59,13,10,108,111,110,103,32,100
,111,117,98,108,101,32,115,116,114,116,111,108,100,40,99,111,110,115,116,32,99,104,97,114,42
,32,114,101,115,116,114,105,99,116,32,110,112,116,114,44,32,99,104,97,114,42,42,32,114,101
,115,116,114,105,99,116,32,101,110,100,112,116,114,41,59,13,10,108,111,110,103,32,105,110,116
,32,115,116,114,116,111,108,40,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114
,105,99,116,32,110,112,116,114,44,32,99,104,97,114,42,42,32,114,101,115,116,114,105,99,116
,32,101,110,100,112,116,114,44,32,105,110,116,32,98,97,115,101,41,59,13,10,108,111,110,103
,32,108,111,110,103,32,105,110,116,32,115,116,114,116,111,108,108,40,99,111,110,115,116,32,99
,104,97,114,42,32,114,101,115,116,114,105,99,116,32,110,112,116,114,44,32,99,104,97,114,42
,42,32,114,101,115,116,114,105,99,116,32,101,110,100,112,116,114,44,32,105,110,116,32,98,97
,115,101,41,59,13,10,117,110,115,105,103,110,101,100,32,108,111,110,103,32,105,110,116,32,115
,116,114,116,111,117,108,40,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105
,99,116,32,110,112,116,114,44,32,99,104,97,114,42,42,32,114,101,115,116,114,105,99,116,32
,101,110,100,112,116,114,44,32,105,110,116,32,98,97,115,101,41,59,13,10,117,110,115,105,103
,110,101,100,32,108,111,110,103,32,108,111,110,103,32,105,110,116,32,115,116,114,116,111,117,108
,108,40,99,111,110,115,116,32,99,104,97,114,42,32,114,101,115,116,114,105,99,116,32,110,112
,116,114,44,32,99,104,97,114,42,42,32,114,101,115,116,114,105,99,116,32,101,110,100,112,116
,114,44,32,105,110,116,32,98,97,115,101,41,59,13,10,105,110,116,32,114,97,110,100,40,118
,111,105,100,41,59,13,10,118,111,105,100,32,115,114,97,110,100,40,117,110,115,105,103,110,101
,100,32,105,110,116,32,115,101,101,100,41,59,13,10,118,111,105,100,42,32,97,108,105,103,110
,101,100,95,97,108,108,111,99,40,115,105,122,101,95,116,32,97,108,105,103,110,109,101,110,116
,44,32,115,105,122,101,95,116,32,115,105,122,101,41,59,13,10,13,10,35,105,102,32,100,101
,102,105,110,101,100,40,95,95,83,84,68,67,95,79,87,78,69,82,83,72,73,80,95,95,41
,32,13,10,91,91,110,111,100,105,115,99,97,114,100,93,93,32,118,111,105,100,42,32,95,79
,119,110,101,114,32,95,79,112,116,32,99,97,108,108,111,99,40,115,105,122,101,95,116,32,110
,109,101,109,98,44,32,115,105,122,101,95,116,32,115,105,122,101,41,59,13,10,118,111,105,100
,32,102,114,101,101,40,118,111,105,100,42,32,95,79,119,110,101,114,32,95,79,112,116,32,112
,116,114,41,59,13,10,91,91,110,111,100,105,115,99,97,114,100,93,93,32,118,111,105,100,42
,32,95,79,119,110,101,114,32,95,79,112,116,32,109,97,108,108,111,99,40,115,105,122,101,95
,116,32,115,105,122,101,41,59,13,10,91,91,110,111,100,105,115,99,97,114,100,93,93,32,118
,111,105,100,42,32,95,79,119,110,101,114,32,95,79,112,116,32,114,101,97,108,108,111,99,40
,118,111,105,100,42,32,95,79,112,116,32,112,116,114,44,32,115,105,122,101,95,116,32,115,105
,122,101,41,59,13,10,35,101,108,115,101,13,10,91,91,110,111,100,105,115,99,97,114,100,93
,93,32,118,111,105,100,42,32,99,97,108,108,111,99,40,115,105,122,101,95,116,32,110,109,101
,109,98,44,32,115,105,122,101,95,116,32,115,105,122,101,41,59,13,10,118,111,105,100,32,102
,114,101,101,40,118,111,105,100,42,32,112,116,114,41,59,13,10,91,91,110,111,100,105,115,99
,97,114,100,93,93,32,118,111,105,100,42,32,109,97,108,108,111,99,40,115,105,122,101,95,116
,32,115,105,122,101,41,59,13,10,91,91,110,111,100,105,115,99,97,114,100,93,93,32,118,111
,105,100,42,32,114,101,97,108,108,111,99,40,118,111,105,100,42,32,112,116,114,44,32,115,105
,122,101,95,116,32,115,105,122,101,41,59,13,10,35,101,110,100,105,102,13,10,13,10,91,91
,110,111,114,101,116,117,114,110,93,93,32,118,111,105,100,32,97,98,111,114,116,40,118,111,105
,100,41,59,13,10,105,110,116,32,97,116,101,120,105,116,40,118,111,105,100,32,40,42,102,117
,110,99,41,40,118,111,105,100,41,41,59,13,10,105,110,116,32,97,116,95,113,117,105,99,107
,95,101,120,105,116,40,118,111,105,100,32,40,42,102,117,110,99,41,40,118,111,105,100,41,41
,59,13,10,91,91,110,111,114,101,116,117,114,110,93,93,32,118,111,105,100,32,101,120,105,116
,40,105,110,116,32,115,116,97,116,117,115,41,59,13,10,91,91,110,111,114,101,116,117,114,110
,93,93,32,118,111,105,100,32,95,69,120,105,116,40,105,110,116,32,115,116,97,116,117,115,41
,59,13,10,99,104,97,114,42,32,103,101,116,101,110,118,40,99,111,110,115,116,32,99,104,97
,114,42,32,110,97,109,101,41,59,13,10,91,91,110,111,114,101,116,117,114,110,93,93,32,118
,111,105,100,32,113,117,105,99,107,95,101,120,105,116,40,105,110,116,32,115,116,97,116,117,115
,41,59,13,10,105,110,116,32,115,121,115,116,101,109,40,99,111,110,115,116,32,99,104,97,114
,42,32,115,116,114,105,110,103,41,59,13,10
};

static const char file_stddef_h[] = {



47,42,13,10,32,42,32,32,84,104,105,115,32,102,105,108,101,32,105,115,32,112,97,114,116
,32,111,102,32,99,97,107,101,32,99,111,109,112,105,108,101,114,13,10,32,42,32,32,104,116
,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,116,104,114,97,100,97,109,115
,47,99,97,107,101,13,10,42,47,13,10,13,10,35,100,101,102,105,110,101,32,117,110,114,101
,97,99,104,97,98,108,101,40,41,32,100,111,32,123,125,32,119,104,105,108,101,40,48,41,32
,13,10,116,121,112,101,100,101,102,32,108,111,110,103,32,105,110,116,32,112,116,114,100,105,102
,102,95,116,59,13,10,116,121,112,101,100,101,102,32,117,110,115,105,103,110,101,100,32,108,111
,110,103,32,115,105,122,101,95,116,59,13,10,116,121,112,101,100,101,102,32,105,110,116,32,119
,99,104,97,114,95,116,59,13,10,116,121,112,101,100,101,102,32,115,116,114,117,99,116,32,123
,13,10,32,32,108,111,110,103,32,108,111,110,103,32,95,95,109,97,120,95,97,108,105,103,110
,95,108,108,59,13,10,32,32,108,111,110,103,32,100,111,117,98,108,101,32,95,95,109,97,120
,95,97,108,105,103,110,95,108,100,59,13,10,125,32,109,97,120,95,97,108,105,103,110,95,116
,59,13,10,13,10,116,121,112,101,100,101,102,32,116,121,112,101,111,102,40,110,117,108,108,112
,116,114,41,32,110,117,108,108,112,116,114,95,116,59,13,10,13,10
};

static const char file_limits_h[] = {
    #include "include\limits.h.include"
};

static const char file_locale_h[] = {



47,42,13,10,32,42,32,32,84,104,105,115,32,102,105,108,101,32,105,115,32,112,97,114,116
,32,111,102,32,99,97,107,101,32,99,111,109,112,105,108,101,114,13,10,32,42,32,32,104,116
,116,112,115,58,47,47,103,105,116,104,117,98,46,99,111,109,47,116,104,114,97,100,97,109,115
,47,99,97,107,101,13,10,42,47,13,10,13,10,35,112,114,97,103,109,97,32,111,110,99,101
,13,10,116,121,112,101,100,101,102,32,105,110,116,32,119,99,104,97,114,95,116,59,13,10,47
,47,32,76,111,99,97,108,101,32,99,97,116,101,103,111,114,105,101,115,13,10,35,100,101,102
,105,110,101,32,76,67,95,65,76,76,32,32,32,32,32,32,32,32,32,32,48,13,10,35,100
,101,102,105,110,101,32,76,67,95,67,79,76,76,65,84,69,32,32,32,32,32,32,49,13,10
,35,100,101,102,105,110,101,32,76,67,95,67,84,89,80,69,32,32,32,32,32,32,32,32,50
,13,10,35,100,101,102,105,110,101,32,76,67,95,77,79,78,69,84,65,82,89,32,32,32,32
,32,51,13,10,35,100,101,102,105,110,101,32,76,67,95,78,85,77,69,82,73,67,32,32,32
,32,32,32,52,13,10,35,100,101,102,105,110,101,32,76,67,95,84,73,77,69,32,32,32,32
,32,32,32,32,32,53,13,10,13,10,35,100,101,102,105,110,101,32,76,67,95,77,73,78,32
,32,32,32,32,32,32,32,32,32,76,67,95,65,76,76,13,10,35,100,101,102,105,110,101,32
,76,67,95,77,65,88,32,32,32,32,32,32,32,32,32,32,76,67,95,84,73,77,69,13,10
,13,10,47,47,32,76,111,99,97,108,101,32,99,111,110,118,101,110,116,105,111,110,32,115,116
,114,117,99,116,117,114,101,13,10,115,116,114,117,99,116,32,108,99,111,110,118,13,10,123,13
,10,32,32,32,32,99,104,97,114,42,32,100,101,99,105,109,97,108,95,112,111,105,110,116,59
,13,10,32,32,32,32,99,104,97,114,42,32,116,104,111,117,115,97,110,100,115,95,115,101,112
,59,13,10,32,32,32,32,99,104,97,114,42,32,103,114,111,117,112,105,110,103,59,13,10,32
,32,32,32,99,104,97,114,42,32,105,110,116,95,99,117,114,114,95,115,121,109,98,111,108,59
,13,10,32,32,32,32,99,104,97,114,42,32,99,117,114,114,101,110,99,121,95,115,121,109,98
,111,108,59,13,10,32,32,32,32,99,104,97,114,42,32,109,111,110,95,100,101,99,105,109,97
,108,95,112,111,105,110,116,59,13,10,32,32,32,32,99,104,97,114,42,32,109,111,110,95,116
,104,111,117,115,97,110,100,115,95,115,101,112,59,13,10,32,32,32,32,99,104,97,114,42,32
,109,111,110,95,103,114,111,117,112,105,110,103,59,13,10,32,32,32,32,99,104,97,114,42,32
,112,111,115,105,116,105,118,101,95,115,105,103,110,59,13,10,32,32,32,32,99,104,97,114,42
,32,110,101,103,97,116,105,118,101,95,115,105,103,110,59,13,10,32,32,32,32,99,104,97,114
,32,32,32,32,32,105,110,116,95,102,114,97,99,95,100,105,103,105,116,115,59,13,10,32,32
,32,32,99,104,97,114,32,32,32,32,32,102,114,97,99,95,100,105,103,105,116,115,59,13,10
,32,32,32,32,99,104,97,114,32,32,32,32,32,112,95,99,115,95,112,114,101,99,101,100,101
,115,59,13,10,32,32,32,32,99,104,97,114,32,32,32,32,32,112,95,115,101,112,95,98,121
,95,115,112,97,99,101,59,13,10,32,32,32,32,99,104,97,114,32,32,32,32,32,110,95,99
,115,95,112,114,101,99,101,100,101,115,59,13,10,32,32,32,32,99,104,97,114,32,32,32,32
,32,110,95,115,101,112,95,98,121,95,115,112,97,99,101,59,13,10,32,32,32,32,99,104,97
,114,32,32,32,32,32,112,95,115,105,103,110,95,112,111,115,110,59,13,10,32,32,32,32,99
,104,97,114,32,32,32,32,32,110,95,115,105,103,110,95,112,111,115,110,59,13,10,32,32,32
,32,119,99,104,97,114,95,116,42,32,95,87,95,100,101,99,105,109,97,108,95,112,111,105,110
,116,59,13,10,32,32,32,32,119,99,104,97,114,95,116,42,32,95,87,95,116,104,111,117,115
,97,110,100,115,95,115,101,112,59,13,10,32,32,32,32,119,99,104,97,114,95,116,42,32,95
,87,95,105,110,116,95,99,117,114,114,95,115,121,109,98,111,108,59,13,10,32,32,32,32,119
,99,104,97,114,95,116,42,32,95,87,95,99,117,114,114,101,110,99,121,95,115,121,109,98,111
,108,59,13,10,32,32,32,32,119,99,104,97,114,95,116,42,32,95,87,95,109,111,110,95,100
,101,99,105,109,97,108,95,112,111,105,110,116,59,13,10,32,32,32,32,119,99,104,97,114,95
,116,42,32,95,87,95,109,111,110,95,116,104,111,117,115,97,110,100,115,95,115,101,112,59,13
,10,32,32,32,32,119,99,104,97,114,95,116,42,32,95,87,95,112,111,115,105,116,105,118,101
,95,115,105,103,110,59,13,10,32,32,32,32,119,99,104,97,114,95,116,42,32,95,87,95,110
,101,103,97,116,105,118,101,95,115,105,103,110,59,13,10,125,59,13,10,13,10,115,116,114,117
,99,116,32,116,109,59,13,10,13,10,99,104,97,114,42,32,115,101,116,108,111,99,97,108,101
,40,13,10,32,32,32,32,105,110,116,32,32,32,32,32,32,32,32,32,95,67,97,116,101,103
,111,114,121,44,13,10,32,32,32,32,99,104,97,114,32,99,111,110,115,116,42,32,95,76,111
,99,97,108,101,13,10,41,59,13,10,13,10,115,116,114,117,99,116,32,108,99,111,110,118,42
,32,108,111,99,97,108,101,99,111,110,118,40,118,111,105,100,41,59,13,10
};

const char file_wchar_h[] = {
  #include "include\wchar.h.include"
};

char* _Owner read_file(const char* path, bool append_newline)
{
    if (strcmp(path, "c:/stdio.h") == 0)
        return strdup(file_stdio_h);
    else if (strcmp(path, "c:/stdlib.h") == 0)
        return strdup(file_stdlib_h);
    else if (strcmp(path, "c:/stddef.h") == 0)
        return strdup(file_stddef_h);
    else if (strcmp(path, "c:/math.h") == 0)
        return strdup(file_math_h);
    else if (strcmp(path, "c:/errno.h") == 0)
        return strdup(file_errno_h);
    else if (strcmp(path, "c:/string.h") == 0)
        return strdup(file_string_h);
    else if (strcmp(path, "c:/assert.h") == 0)
        return strdup(file_assert_h);
    else if (strcmp(path, "c:/limits.h") == 0)
        return strdup(file_limits_h);
    else if (strcmp(path, "c:/locale.h") == 0)
        return strdup(file_locale_h);
    else if (strcmp(path, "c:/wchar.h") == 0)
        return strdup(file_wchar_h);

    return NULL;
}
#endif




/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

#pragma safety enable


bool is_diagnostic_note(enum diagnostic_id id)
{
    if (id == W_NOTE ||
        id == W_LOCATION)
    {
        return true;
    }

    return false;
}

bool is_diagnostic_warning(enum diagnostic_id id)
{
    return id > W_NOTE && id <= C_ERROR_INVALID_QUALIFIER_FOR_POINTER;
}

bool is_diagnostic_error(enum diagnostic_id id)
{
    return id >= C_ERROR_INVALID_QUALIFIER_FOR_POINTER;
}

bool is_diagnostic_configurable(enum diagnostic_id id)
{
    //We have 0-63 configurable (bit set)
    //configurable diagnostic also have names. Other have numbers only    
    return id >= 0 && id < W_LOCATION;
}

int diagnostic_stack_push_empty(struct diagnostic_stack* diagnostic_stack)
{
    int index = diagnostic_stack->top_index;
    diagnostic_stack->top_index++;
    diagnostic_stack->stack[diagnostic_stack->top_index].warnings = 0;
    diagnostic_stack->stack[diagnostic_stack->top_index].errors = 0;
    diagnostic_stack->stack[diagnostic_stack->top_index].notes = 0;
    return index;
}

void diagnostic_stack_pop(struct diagnostic_stack* diagnostic_stack)
{
    if (diagnostic_stack->top_index > 0)
    {
        diagnostic_stack->top_index--;
    }
    else
    {
        assert(false);
    }
}


struct diagnostic default_diagnostic = {
      .warnings = (~0ULL) & ~(
        NULLABLE_DISABLE_REMOVED_WARNINGS |
        (1ULL << W_NOTE) |
        (1ULL << W_STYLE) |
        (1ULL << W_UNUSED_PARAMETER) |
        (1ULL << W_UNUSED_VARIABLE))
};

static struct w {
    enum diagnostic_id w;
    const char* name;
}
s_warnings[] = {
    {W_UNUSED_VARIABLE, "unused-variable"},
    {W_UNUSED_FUNCTION, "unused-function"},
    {W_DEPRECATED, "deprecated"},
    {W_ENUN_CONVERSION,"enum-conversion"},

    {W_ADDRESS, "address"},
    {W_UNUSED_PARAMETER, "unused-parameter"},
    {W_DECLARATOR_HIDE, "hide-declarator"},
    {W_TYPEOF_ARRAY_PARAMETER, "typeof-parameter"},
    {W_ATTRIBUTES, "attributes"},
    {W_UNUSED_VALUE, "unused-value"},
    {W_STYLE, "style"},
    {W_COMMENT,"comment"},
    {W_LINE_SLICING,"line-slicing"},
    {W_SWITCH, "switch"},
    {W_UNSUAL_NULL_POINTER_CONSTANT, "unusual-null"},

    {W_DISCARDED_QUALIFIERS, "discarded-qualifiers"},
    {W_UNINITIALZED, "uninitialized"},
    {W_RETURN_LOCAL_ADDR, "return-local-addr"},
    {W_DIVIZION_BY_ZERO,"div-by-zero"},
    {W_CONSTANT_VALUE, "constant-value"},
    {W_SIZEOF_ARRAY_ARGUMENT, "sizeof-array-argument"},

    {W_STRING_SLICED,"string-slicing"},
    {W_DECLARATOR_STATE,"declarator-state"},
    {W_OWNERSHIP_MISSING_OWNER_QUALIFIER, "missing-owner-qualifier"},
    {W_OWNERSHIP_NOT_OWNER,"not-owner"},
    {W_OWNERSHIP_USING_TEMPORARY_OWNER,"temp-owner"},
    {W_OWNERSHIP_MOVE_ASSIGNMENT_OF_NON_OWNER, "non-owner-move"},
    {W_OWNERSHIP_NON_OWNER_TO_OWNER_ASSIGN, "non-owner-to-owner-move"},
    {W_OWNERSHIP_DISCARDING_OWNER, "discard-owner"},

    {W_OWNERSHIP_NON_OWNER_MOVE, "non-owner-move"},
    {W_FLOW_DIVIZION_BY_ZERO, "flow-div-by-zero"},

    /////////////////////////////////////////////////////////////////////////
    {W_FLOW_NON_NULL, "flow-not-null"},
    {W_FLOW_MISSING_DTOR, "missing-destructor"},
    {W_FLOW_MOVED, "using-moved-object"},
    {W_FLOW_UNINITIALIZED, "analyzer-maybe-uninitialized"},
    {W_FLOW_NULL_DEREFERENCE, "analyzer-null-dereference"}, // -fanalyzer
    {W_FLOW_MAYBE_NULL_TO_NON_OPT_ARG, "analyzer-non-opt-arg"},
    {W_FLOW_LIFETIME_ENDED, "lifetime-ended"},
    {W_FLOW_NULLABLE_TO_NON_NULLABLE, "nullable-to-non-nullable"},

    /////////////////////////////////////////////////////////////////////
    {W_MUST_USE_ADDRESSOF, "must-use-address-of"},
    {W_PASSING_NULL_AS_ARRAY, "null-as-array"},
    {W_INCOMPATIBLE_ENUN_TYPES, "incompatible-enum"},
    {W_MULTICHAR_ERROR, "multi-char"},
    {W_ARRAY_INDIRECTION,"array-indirection"},
    {W_OUT_OF_BOUNDS, "out-of-bounds"},
    {W_ASSIGNMENT_OF_ARRAY_PARAMETER, "array-parameter-assignment"},
    {W_CONDITIONAL_IS_CONSTANT,"conditional-constant"},

    {W_CONST_NOT_INITIALIZED, "const-init"},
    {W_NULL_CONVERTION, "null-conversion"},
    {W_IMPLICITLY_UNSIGNED_LITERAL, "implicitly-unsigned-literal"},
    {W_INTEGER_OVERFLOW, "overflow"},
    {W_ARRAY_SIZE, "array-size"},
    {W_EMPTY_STATEMENT, "empty-statement"},
    {W_ERROR_INCOMPATIBLE_TYPES, "incompatible-types"},
    {W_UNUSED_LABEL, "unused-label"}
};

void diagnostic_remove(struct diagnostic* d, enum diagnostic_id w)
{
    if (!is_diagnostic_configurable(w))
        return; //ops

    if ((d->errors & (1ULL << w)) != 0)
        d->errors &= ~(1ULL << w);

    if ((d->warnings & (1ULL << w)) != 0)
        d->warnings &= ~(1ULL << w);

    if ((d->notes & (1ULL << w)) != 0)
        d->notes &= ~(1ULL << w);
}

int get_diagnostic_type(struct diagnostic* d, enum diagnostic_id w)
{
    if (is_diagnostic_configurable(w))
    {
        if ((d->errors & (1ULL << w)) != 0)
            return 3;

        if ((d->warnings & (1ULL << w)) != 0)
            return 2;

        if ((d->notes & (1ULL << w)) != 0)
            return 1;
    }


    if (is_diagnostic_note(w))
        return 1;

    if (is_diagnostic_warning(w))
        return 2;

    if (is_diagnostic_error(w))
        return 3;

    return 3; //errors
}

int get_diagnostic_phase(enum diagnostic_id w)
{
    switch (w)
    {
        //TODO should be everything that starts with FLOW
    case W_FLOW_MISSING_DTOR:
    case W_FLOW_UNINITIALIZED:
    case W_FLOW_MOVED:
    case W_FLOW_NULL_DEREFERENCE:
    case W_FLOW_MAYBE_NULL_TO_NON_OPT_ARG:
    case W_FLOW_NON_NULL:
    case W_FLOW_LIFETIME_ENDED:
    case W_FLOW_DIVIZION_BY_ZERO:

        return 2; /*returns 2 if it flow analysis*/
    default:
        break;
    }
    return 0;
}


enum diagnostic_id  get_warning(const char* wname)
{
    if (!(wname[0] == '-' || wname[0] == 'E'))
    {
        return 0;
    }

    if (wname[0] == '-' && wname[1] == 'W')
    {
        for (int j = 0; j < sizeof(s_warnings) / sizeof(s_warnings[0]); j++)
        {
            if (strncmp(s_warnings[j].name, wname + 2, strlen(s_warnings[j].name)) == 0)
            {
                return s_warnings[j].w;
            }
        }
    }
    else if (wname[1] == 'E')
    {
        int ec = atoi(wname + 2);
        return ec;

    }
    return 0;
}

unsigned long long  get_warning_bit_mask(const char* wname)
{
    const bool disable_warning = wname[2] == 'n' && wname[3] == 'o';
    const char* final_name = disable_warning ? wname + 5 : wname + 2;
    assert(wname[0] == '-');
    for (int j = 0; j < sizeof(s_warnings) / sizeof(s_warnings[0]); j++)
    {

        if (strncmp(s_warnings[j].name, final_name, strlen(s_warnings[j].name)) == 0)
        {
            return (1ULL << ((unsigned long long)s_warnings[j].w));
        }
    }
    return 0;
}

int get_warning_name(enum diagnostic_id w, int n, char buffer[/*n*/])
{
    if (is_diagnostic_configurable(w))
    {
        //TODO because s_warnings is _Ctor of order ....
        //this is a linear seatch instead of just index! TODOD
        for (int j = 0; j < sizeof(s_warnings) / sizeof(s_warnings[0]); j++)
        {
            if (s_warnings[j].w == w)
            {
                snprintf(buffer, n, "-W%s", s_warnings[j].name);
                return 0;
            }
        }
    }
    else
    {
        snprintf(buffer, n, "E%d", w);
    }

    return 0;//"";
}

int fill_options(struct options* options,
    int argc,
    const char** argv)
{

    /*
       default at this moment is same as -Wall
    */
    options->diagnostic_stack.stack[0] = default_diagnostic;

    options->diagnostic_stack.stack[0].warnings &= ~(1ULL << W_STYLE);
    //&~items;


    /*first loop used to collect options*/
    for (int i = 1; i < argc; i++)
    {
        if (argv[i][0] != '-')
            continue;

        if (argv[i][1] == 'I' ||
            argv[i][1] == 'D')
        {
            /*
              Valid, but handled with preprocessor
            */
            continue;
        }

        if (strcmp(argv[i], "-no-output") == 0)
        {
            options->no_output = true;
            continue;
        }

        if (strcmp(argv[i], "-const-literal") == 0)
        {
            options->const_literal = true;
            continue;
        }

        if (strcmp(argv[i], "-o") == 0)
        {
            if (i + 1 < argc)
            {
                strcpy(options->output, argv[i + 1]);
                i++;
            }
            else
            {
                //ops
            }
            continue;
        }

        if (strcmp(argv[i], "-sarif-path") == 0)
        {
            if (i + 1 < argc)
            {
                strcpy(options->sarifpath, argv[i + 1]);
                i++;
            }
            else
            {
                //ops
            }
            continue;
        }

        if (strcmp(argv[i], "-H") == 0)
        {
            options->show_includes = true;
            continue;
        }

        if (strcmp(argv[i], "-E") == 0)
        {
            options->preprocess_only = true;
            continue;
        }

        if (strcmp(argv[i], "-preprocess-def-macro") == 0)
        {
            options->preprocess_def_macro = true;
            continue;
        }

        if (strcmp(argv[i], "-sarif") == 0)
        {
            options->sarif_output = true;
            continue;
        }

        if (strcmp(argv[i], "-fanalyzer") == 0)
        {
            options->flow_analysis = true;
            continue;
        }

        if (strcmp(argv[i], "-nullchecks") == 0)
        {
            options->null_checks_enabled = true;
            continue;
        }

        if (strcmp(argv[i], "-debug") == 0)
        {
            options->do_static_debug = true;
            if (i + 1 < argc)
            {
                i++;
                options->static_debug_lines = atoi(argv[i]);
            }
            else
            {
                //ops
            }
            continue;
        }

        if (strcmp(argv[i], "-ownership=enable") == 0)
        {
            options->ownership_enabled = true;
            continue;
        }

        if (strcmp(argv[i], "-ownership=disable") == 0)
        {
            options->ownership_enabled = false;
            continue;
        }

        if (strcmp(argv[i], "-test-mode") == 0)
        {
            options->test_mode = true;
            continue;
        }

        if (strcmp(argv[i], "-msvc-output") == 0 ||
            strcmp(argv[i], "-fdiagnostics-format=msvc") == 0) //same as clang
        {
            options->visual_studio_ouput_format = true;
            continue;
        }


        //
        if (strcmp(argv[i], "-style=cake") == 0)
        {
            options->style = STYLE_CAKE;
            continue;
        }

        if (strcmp(argv[i], "-style=gnu") == 0)
        {
            options->style = STYLE_GNU;
            continue;
        }

        if (strcmp(argv[i], "-style=microsoft") == 0)
        {
            options->style = STYLE_GNU;
            continue;
        }

        if (strcmp(argv[i], "-nullable=disable") == 0)
        {
            options->null_checks_enabled = false;
            unsigned long long w = NULLABLE_DISABLE_REMOVED_WARNINGS;
            options->diagnostic_stack.stack[0].warnings &= ~w;
            continue;
        }

        if (strcmp(argv[i], "-nullable=enabled") == 0)
        {
            options->null_checks_enabled = true;
            continue;
        }

        if (strcmp(argv[i], "-autoconfig") == 0 ||
            strcmp(argv[i], "-auto-config") == 0)
        {
            options->auto_config = true;
            continue;
        }


        if (strcmp(argv[i], "-std=c2x") == 0 ||
            strcmp(argv[i], "-std=c23") == 0)
        {
            options->input = LANGUAGE_C23;
            continue;
        }
        if (strcmp(argv[i], "-std=cxx") == 0)
        {
            options->input = LANGUAGE_CAK;
            continue;
        }

        //warnings
        if (argv[i][1] == 'W')
        {
            if (strcmp(argv[i], "-Wall") == 0)
            {
                options->diagnostic_stack.stack[0].warnings = ~0ULL;
                continue;
            }
            const bool disable_warning = (argv[i][2] == 'n' && argv[i][3] == 'o');

            unsigned long long w = get_warning_bit_mask(argv[i]);

            if (w == 0)
            {
                printf("unknown warning '%s'", argv[i]);
                return 1;
            }


            if (disable_warning)
            {
                options->diagnostic_stack.stack[0].warnings &= ~w;
            }
            else
            {
                if (w == W_STYLE)
                    options->diagnostic_stack.stack[0].warnings |= w;
                else
                    options->diagnostic_stack.stack[0].notes |= w;
            }
            continue;
        }

        if (strcmp(argv[i], "-dump-tokens") == 0)
        {
            options->dump_tokens = true;
            continue;
        }

        if (strcmp(argv[i], "-dump-pp-tokens") == 0)
        {
            options->dump_pptokens = true;
            continue;
        }

        if (strcmp(argv[i], "-disable-assert") == 0)
        {
            options->disable_assert = true;
            continue;
        }

        printf("unknown option '%s'", argv[i]);
        return 1;
    }
    return 0;
}


void print_help()
{
#define CAKE LIGHTCYAN "cake " RESET 

    const char* options =
        LIGHTGREEN "Usage :" RESET CAKE LIGHTBLUE "[OPTIONS] source1.c source2.c ...\n" RESET
        "\n"
        LIGHTGREEN "Samples:\n" RESET
        "\n"
        WHITE "    " CAKE " source.c\n" RESET
        "    Compiles source.c and outputs /out/source.c\n"
        "\n"        
        WHITE "    " CAKE " file.c -o file.cc && cl file.cc\n" RESET
        "    Compiles file.c and outputs file.cc then use cl to compile file.cc\n"
        "\n"
        WHITE "    " CAKE " file.c -direct-compilation -o file.cc && cl file.cc\n" RESET
        "    Compiles file.c and outputs file.cc for direct compilation then use cl to compile file.cc\n"
        "\n"
        LIGHTGREEN "Options:\n" RESET
        "\n"
        LIGHTCYAN "  -I                   " RESET " Adds a directory to the list of directories searched for include files \n"
        "                        (On windows, if you run cake at the visual studio command prompt cake \n"
        "                        uses the same include files used by msvc )\n"
        "\n"
        LIGHTCYAN "  -auto-config           " RESET "Generates cakeconfig.h with include directories\n"
        "\n"
        LIGHTCYAN "  -no-output            " RESET "Cake will not generate output\n"
        "\n"
        LIGHTCYAN "  -D                    " RESET "Defines a preprocessing symbol for a source file \n"
        "\n"
        LIGHTCYAN "  -E                    " RESET "Copies preprocessor output to standard output \n"
        "\n"
        LIGHTCYAN "  -o name.c             " RESET "Defines the output name when compiling one file\n"
        "\n"
        LIGHTCYAN "  -std=standard         " RESET "Assume that the input sources are for standard (c89, c99, c11, c2x, cxx) \n"
        "                        (not implemented yet, input is considered C23)                    \n"
        "\n"
        LIGHTCYAN "  -no-discard           " RESET "Makes [[nodiscard]] default implicitly \n"
        "\n"
        LIGHTCYAN "  -Wname -Wno-name      " RESET "Enables or disable warning\n"
        "\n"
        LIGHTCYAN "  -fanalyzer            " RESET "Runs flow analysis -  required for ownership\n"
        "\n"
        LIGHTCYAN "  -sarif                " RESET "Generates sarif files\n"
        "\n"
        LIGHTCYAN "  -H                    " RESET "Print the name of each header file used\n"
        "\n"
        LIGHTCYAN "  -sarif-path           " RESET "Set sarif output dir\n"
        "\n"
        LIGHTCYAN "  -msvc-output          " RESET "Output is compatible with visual studio\n"
        "\n"
        LIGHTCYAN "  -dump-tokens          " RESET "Output tokens before preprocessor\n"
        "\n"
        LIGHTCYAN "  -dump-pp-tokens       " RESET "Output tokens after preprocessor\n"
        "\n"
        LIGHTCYAN "  -disable-assert       " RESET "disables built-in assert\n"
        "\n"
        LIGHTCYAN "  -const-literal        " RESET "literal string becomes const\n"
        "\n"
        LIGHTCYAN "  -preprocess-def-macro " RESET "preprocess def macros after expansion\n"

        "More details at http://thradams.com/cake/manual.html\n"
        ;

    printf("%s", options);
}

#ifdef TEST

void test_get_warning_name()
{
    char dbg_name[100];
    get_warning_name(W_FLOW_MISSING_DTOR, sizeof dbg_name, dbg_name);
    assert(strcmp(dbg_name, "-Wmissing-destructor") == 0);

    unsigned long long  flags = get_warning_bit_mask(dbg_name);
    assert(flags == (1ULL << W_FLOW_MISSING_DTOR));


    get_warning_name(W_STYLE, sizeof dbg_name, dbg_name);
    assert(strcmp(dbg_name, "-Wstyle") == 0);

    unsigned long long  flags2 = get_warning_bit_mask(dbg_name);
    assert(flags2 == (1ULL << W_STYLE));
}

#endif


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

#pragma safety enable




/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
 *
 *  struct object is used to compute the compile time expressions (including constexpr)
 *
*/

//#pragma once



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once


struct parser_ctx;


enum type_category
{
    TYPE_CATEGORY_ITSELF,
    TYPE_CATEGORY_FUNCTION,
    TYPE_CATEGORY_ARRAY,
    TYPE_CATEGORY_POINTER,
};

enum attribute_flags
{
    STD_ATTRIBUTE_NONE = 0,
    STD_ATTRIBUTE_DEPRECATED = 1 << 0,
    STD_ATTRIBUTE_FALLTHROUGH = 1 << 1,
    STD_ATTRIBUTE_MAYBE_UNUSED = 1 << 2,
    STD_ATTRIBUTE_NODISCARD = 1 << 3,
    STD_ATTRIBUTE_NORETURN = 1 << 4,
    STD_ATTRIBUTE_UNSEQUENCED = 1 << 5,
    STD_ATTRIBUTE_REPRODUCIBLE = 1 << 6,

    //TODO decide attribute or not
    CAKE_ATTRIBUTE_CTOR = 1 << 7,
    CAKE_ATTRIBUTE_DTOR = 1 << 8,

    /*
     1 == 2 results in int in C
     lets add extra flag here
     not sure what is the best place to put in
     type specifier my generate some error
    */
    CAKE_HIDDEN_ATTRIBUTE_LIKE_BOOL = 1 << 25,
    // 'a'
    CAKE_HIDDEN_ATTRIBUTE_INT_LIKE_CHAR = 1 << 26,

    /*
       Storing calling convention on attributes to consuming less memory
    */
    CAKE_ATTRIBUTE_FASTCALL = 1 << 27,
    CAKE_ATTRIBUTE_STDCALL = 1 << 28,
    CAKE_ATTRIBUTE_CDECL = 1 << 29

};

enum type_specifier_flags
{
    TYPE_SPECIFIER_NONE = 0,
    TYPE_SPECIFIER_VOID = 1 << 0,
    TYPE_SPECIFIER_CHAR = 1 << 1,
    TYPE_SPECIFIER_SHORT = 1 << 2,
    TYPE_SPECIFIER_INT = 1 << 3,
    TYPE_SPECIFIER_LONG = 1 << 4,

    TYPE_SPECIFIER_FLOAT = 1 << 5,
    TYPE_SPECIFIER_DOUBLE = 1 << 6,
    TYPE_SPECIFIER_SIGNED = 1 << 7,
    TYPE_SPECIFIER_UNSIGNED = 1 << 8,
    TYPE_SPECIFIER_BOOL = 1 << 9,
    TYPE_SPECIFIER_COMPLEX = 1 << 10,
    TYPE_SPECIFIER_DECIMAL32 = 1 << 11,
    TYPE_SPECIFIER_DECIMAL64 = 1 << 12,
    TYPE_SPECIFIER_DECIMAL128 = 1 << 13,
    TYPE_SPECIFIER_ATOMIC = 1 << 14,
    TYPE_SPECIFIER_STRUCT_OR_UNION = 1 << 15,
    TYPE_SPECIFIER_ENUM = 1 << 16,
    TYPE_SPECIFIER_TYPEDEF = 1 << 17,

    //MICROSOFT
    TYPE_SPECIFIER_INT8 = 1 << 18,
    TYPE_SPECIFIER_INT16 = 1 << 19,
    TYPE_SPECIFIER_INT32 = 1 << 20,
    TYPE_SPECIFIER_INT64 = 1 << 21,

    TYPE_SPECIFIER_LONG_LONG = 1 << 22,

    TYPE_SPECIFIER_TYPEOF = 1 << 23,

    TYPE_SPECIFIER_NULLPTR_T = 1 << 24,
};

#ifdef _WIN32

#define CAKE_WCHAR_T_TYPE_SPECIFIER (TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_SHORT)

#ifdef _WIN64
#define  CAKE_SIZE_T_TYPE_SPECIFIER (TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT64)    
#else
#define  CAKE_SIZE_T_TYPE_SPECIFIER (TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT)    
#endif

#else 

#define CAKE_WCHAR_T_TYPE_SPECIFIER (TYPE_SPECIFIER_INT)

#ifdef __x86_64__
/* 64-bit */
#define  CAKE_SIZE_T_TYPE_SPECIFIER (TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_LONG)    
#else
#define  CAKE_SIZE_T_TYPE_SPECIFIER (TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT)    
#endif


#endif


enum type_qualifier_flags
{
    TYPE_QUALIFIER_NONE,
    TYPE_QUALIFIER_CONST = 1 << 0,
    TYPE_QUALIFIER_RESTRICT = 1 << 1,
    TYPE_QUALIFIER_VOLATILE = 1 << 2,
    TYPE_QUALIFIER__ATOMIC = 1 << 3,

    

    /*ownership extensions*/
    TYPE_QUALIFIER_OWNER = 1 << 4,
    TYPE_QUALIFIER_VIEW = 1 << 5,
    TYPE_QUALIFIER_OPT = 1 << 6,

    /*function contract*/
    TYPE_QUALIFIER_DTOR = 1 << 7,
    TYPE_QUALIFIER_CTOR = 1 << 8,

    TYPE_QUALIFIER_MSVC_PTR32 = 1 << 9,
    TYPE_QUALIFIER_MSVC_PTR64 = 1 << 10,

};

enum storage_class_specifier_flags
{
    STORAGE_SPECIFIER_NONE = 0,
    STORAGE_SPECIFIER_TYPEDEF = 1 << 0,
    STORAGE_SPECIFIER_EXTERN = 1 << 1,
    STORAGE_SPECIFIER_STATIC = 1 << 2,
    STORAGE_SPECIFIER_THREAD_LOCAL = 1 << 3,
    STORAGE_SPECIFIER_AUTO = 1 << 4,
    STORAGE_SPECIFIER_REGISTER = 1 << 5,
    STORAGE_SPECIFIER_CONSTEXPR = 1 << 6,

    /*extra flag just to annotate this*/
    STORAGE_SPECIFIER_CONSTEXPR_STATIC = 1 << 7,

    /*it is a function parameter*/
    STORAGE_SPECIFIER_PARAMETER = 1 << 11,

    STORAGE_SPECIFIER_AUTOMATIC_STORAGE = 1 << 12,
    STORAGE_SPECIFIER_FUNCTION_RETURN = 1 << 13,
    STORAGE_SPECIFIER_FUNCTION_RETURN_NODISCARD = 1 << 14,
};

enum function_specifier_flags
{
    FUNCTION_SPECIFIER_NONE = 0,
    FUNCTION_SPECIFIER_INLINE = 1 << 0,
    FUNCTION_SPECIFIER_NORETURN = 1 << 1,
};

enum alignment_specifier_flags
{
    ALIGNMENT_SPECIFIER_NONE =  0,
    ALIGNMENT_SPECIFIER_8_FLAGS = 1 << 0,
    ALIGNMENT_SPECIFIER_16_FLAGS = 1 << 1,
    ALIGNMENT_SPECIFIER_32_FLAGS = 1 << 2,
    ALIGNMENT_SPECIFIER_64_FLAGS = 1 << 3,
    ALIGNMENT_SPECIFIER_128_FLAGS = 1 << 4,
};

enum msvc_declspec_flags
{
    //https://learn.microsoft.com/en-us/cpp/cpp/declspec?view=msvc-170
    MSVC_DECLSPEC_NONE = 0,
    MSVC_DECLSPEC_ALIGN_8_FLAG = 1 << 1, //( NUMBER )
    MSVC_DECLSPEC_ALIGN_16_FLAG = 1 << 2, //( NUMBER )
    MSVC_DECLSPEC_ALIGN_32_FLAG = 1 << 3, //( NUMBER )
    MSVC_DECLSPEC_ALIGN_64_FLAG = 1 << 4, //( NUMBER )

    MSVC_DECLSPEC_ALLOCATE_FLAG = 1 << 5, //(" SEGNAME ")
    MSVC_DECLSPEC_ALLOCATOR_FLAG = 1 << 6, //
    MSVC_DECLSPEC_APPDOMAIN_FLAG = 1 << 7, //
    MSVC_DECLSPEC_CODE_SEG_FLAG = 1 << 8, //(" SEGNAME ")
    MSVC_DECLSPEC_DEPRECATED_FLAG = 1 << 9, //
    MSVC_DECLSPEC_DLLIMPORT_FLAG = 1 << 10, //
    MSVC_DECLSPEC_DLLEXPORT_FLAG = 1 << 11, //
    MSVC_DECLSPEC_EMPTY_BASES_FLAG = 1 << 12, //
    MSVC_DECLSPEC_HYBRID_PATCHABLE_FLAG = 1 << 13, //
    MSVC_DECLSPEC_JITINTRINSIC_FLAG = 1 << 14, //
    MSVC_DECLSPEC_NAKED_FLAG = 1 << 15, //
    MSVC_DECLSPEC_NOALIAS_FLAG = 1 << 16, //
    MSVC_DECLSPEC_NOINLINE_FLAG = 1 << 17, //
    MSVC_DECLSPEC_NORETURN_FLAG = 1 << 18, //
    MSVC_DECLSPEC_NOTHROW_FLAG = 1 << 19, //
    MSVC_DECLSPEC_NOVTABLE_FLAG = 1 << 20, //
    MSVC_DECLSPEC_NO_SANITIZE_ADDRESS_FLAG = 1 << 21, //
    MSVC_DECLSPEC_PROCESS_FLAG = 1 << 22, //
    MSVC_DECLSPEC_PROPERTY_FLAG = 1 << 23, //( { GET=GET-FUNC-NAME | ,PUT=PUT-FUNC-NAME } )
    MSVC_DECLSPEC_RESTRICT_FLAG = 1 << 24, //
    MSVC_DECLSPEC_SAFEBUFFERS_FLAG = 1 << 25, //
    MSVC_DECLSPEC_SELECTANY_FLAG = 1 << 26, //
    MSVC_DECLSPEC_SPECTRE_FLAG = 1 << 27, //(NOMITIGATION)
    MSVC_DECLSPEC_THREAD_FLAG = 1 << 28, //
    MSVC_DECLSPEC_UUID_FLAG = 1 << 29, //(" COMOBJECTGUID ")
};

struct declarator;
struct type;

enum assigment_type
{
    ASSIGMENT_TYPE_RETURN,    // T f() { return b; }
    ASSIGMENT_TYPE_PARAMETER, // void f(T a); f(b);
    ASSIGMENT_TYPE_OBJECTS,   // a = b
    ASSIGMENT_TYPE_INIT,   // T a = b
};


struct type_list
{
    struct type* _Owner _Opt head;
    struct type* _Opt tail;
};

void type_list_destroy(_Dtor struct type_list* p_type_list);
void type_list_push_back(struct type_list* books, struct type* _Owner new_book);
void type_list_push_front(struct type_list* books, struct type* _Owner new_book);


struct param;

struct param_list
{
    bool is_var_args;
    bool is_void;
    struct param* _Owner _Opt head;
    struct param* _Opt tail;
};

void param_list_destroy(_Dtor struct param_list* p);
void param_list_add(struct param_list* p, struct param* _Owner p_item);

struct type
{
    enum type_category category;

    enum attribute_flags  attributes_flags;
    enum msvc_declspec_flags msvc_declspec_flags;
    enum alignment_specifier_flags alignment_specifier_flags;
    enum type_specifier_flags type_specifier_flags;
    enum type_qualifier_flags type_qualifier_flags;
    enum storage_class_specifier_flags storage_class_specifier_flags;
    
    const char* _Owner _Opt name_opt;

    struct struct_or_union_specifier* _Opt struct_or_union_specifier;
    const struct enum_specifier* _Opt enum_specifier;

    //Expression used as array size. Can be constant or not constant (VLA)
    const struct expression* _Opt array_num_elements_expression;

    size_t num_of_elements;
    bool has_static_array_size;

    /*
      address_of is true when the type is created by address of operator.
      This is used to create _Dtor pointer.
    */
    bool address_of;

    struct param_list params;
    struct type* _Owner _Opt next;
};

const struct param_list* _Opt type_get_func_or_func_ptr_params(const struct type* p_type);

struct param
{
    struct type type;
    struct param* _Owner _Opt next;
};

void print_type(struct osstream* ss, const  struct type* type);
void print_type_no_names(struct osstream* ss, const struct type* p_type);

void print_item(struct osstream* ss, bool* first, const char* item);
struct type type_dup(const struct type* p_type);
void type_set(struct type* a, const struct type* b);
void type_destroy(_Opt _Dtor struct type* p_type);

struct type type_common(const struct type* p_type1, const struct type* p_type2);
struct type get_array_item_type(const struct type* p_type);
struct type type_remove_pointer(const struct type* p_type);

bool type_is_essential_bool(const struct type* p_type);
bool type_is_essential_char(const struct type* p_type);

bool type_is_enum(const struct type* p_type);
bool type_is_array(const struct type* p_type);

bool type_is_ctor(const struct type* p_type);
bool type_is_const(const struct type* p_type);
bool type_is_constexpr(const struct type* p_type);
bool type_is_const_or_constexpr(const struct type* p_type);
bool type_is_opt(const struct type* p_type, bool nullable_enabled);
bool type_is_view(const struct type* p_type);

bool type_is_owner(const struct type* p_type);
bool type_is_pointed_dtor(const struct type* p_type);
bool type_is_owner_or_pointer_to_dtor(const struct type* p_type);

bool type_is_pointer_to_const(const struct type* p_type);
bool type_is_pointer(const struct type* p_type);
bool type_is_pointer_to_out(const struct type* p_type);

bool type_is_nullptr_t(const struct type* p_type);
bool type_is_void_ptr(const struct type* p_type);
bool type_is_integer(const struct type* p_type);
bool type_is_char(const struct type* p_type);
bool type_is_array_of_char(const struct type* p_type);
bool type_is_unsigned_integer(const struct type* p_type);
bool type_is_signed_integer(const struct type* p_type);
bool type_is_floating_point(const struct type* p_type);
int type_get_integer_rank(const struct type* p_type1);

bool type_is_arithmetic(const struct type* p_type);

bool type_is_struct_or_union(const struct type* p_type);
bool type_is_union(const struct type* p_type);

bool type_is_void(const struct type* p_type);
bool type_is_function_or_function_pointer(const struct type* p_type);
bool type_is_function(const struct type* p_type);
bool type_is_nodiscard(const struct type* p_type);

bool type_is_deprecated(const struct type* p_type);
bool type_is_maybe_unused(const struct type* p_type);
bool type_is_pointer_or_array(const struct type* p_type);
bool type_is_same(const struct type* a, const struct type* b, bool compare_qualifiers);
bool type_is_compatible(const struct type* a, const struct type* b);
bool type_is_scalar(const struct type* p_type);
bool type_has_attribute(const struct type* p_type, enum attribute_flags attributes);
bool type_is_bool(const struct type* p_type);
bool type_is_decimal128(const struct type* p_type);
bool type_is_decimal64(const struct type* p_type);
bool type_is_decimal32(const struct type* p_type);
bool type_is_long_double(const struct type* p_type);
bool type_is_double(const struct type* p_type);
bool type_is_float(const struct type* p_type);
bool type_is_int(const struct type* p_type);
bool type_is_unsigned_int(const struct type* p_type);
bool type_is_empty(const struct type* p_type);

bool type_is_vla(const struct type* p_type);

struct type type_get_enum_type(const struct type* p_type);

struct argument_expression;
void check_argument_and_parameter(struct parser_ctx* ctx,
    struct argument_expression* current_argument,
    struct type* paramer_type,
    int param_num);

struct type type_convert_to(const struct type* p_type, enum language_version target);
struct type type_lvalue_conversion(const struct type* p_type, bool nullchecks_enabled);
void type_remove_qualifiers(struct type* p_type);
void type_add_const(struct type* p_type);
void type_swap(struct type* a, struct type* b);
void type_clear(struct type* a);
void type_integer_promotion(struct type* a);


struct type type_remove_pointer(const struct type* p_type);
struct type get_array_item_type(const struct type* p_type);

struct type type_param_array_to_pointer(const struct type* p_type, bool null_checks_enabled);

struct type type_make_literal_string(int size, enum type_specifier_flags chartype, enum type_qualifier_flags qualifiers);
struct type type_make_int();
struct type type_make_int_bool_like();
struct type type_make_size_t();
struct type type_make_long_double();
struct type type_make_double();
struct type type_make_float();


struct type type_make_enumerator(const struct enum_specifier* enum_specifier);
struct type make_void_type();
struct type make_void_ptr_type();
struct type make_size_t_type();
struct type make_with_type_specifier_flags(enum type_specifier_flags f);


struct type get_function_return_type(const struct type* p_type);
bool function_returns_void(const struct type* p_type);


enum sizeof_error
{
    ESIZEOF_NONE = 0,
    ESIZEOF_OVERLOW,
    ESIZEOF_VLA,
    ESIZEOF_INCOMPLETE,
    ESIZEOF_FUNCTION
};

enum sizeof_error type_get_sizeof(const struct type* p_type, size_t* size);

size_t type_get_alignof(const struct type* p_type);

struct type type_add_pointer(const struct type* p_type, bool null_checks_enabled);
void type_print(const struct type* a);
void type_println(const struct type* a);

enum type_category type_get_category(const struct type* p_type);
void print_type_qualifier_specifiers(struct osstream* ss, const struct type* type);

void type_visit_to_mark_anonymous(struct type* p_type);

void type_set_qualifiers_using_declarator(struct type* p_type, struct declarator* pdeclarator);
void type_merge_qualifiers_using_declarator(struct type* p_type, struct declarator* pdeclarator);

void print_type_declarator(struct osstream* ss, const struct type* p_type);
void type_remove_names(struct type* p_type);
const struct type* type_get_specifer_part(const struct type* p_type);
void print_msvc_declspec(struct osstream* ss, bool* first, enum msvc_declspec_flags  msvc_declspec_flags);

struct parser_ctx;

enum object_value_type 
{

    TYPE_SIGNED_INT,
    TYPE_UNSIGNED_INT,

    TYPE_BOOL,

    TYPE_SIGNED_CHAR,
    TYPE_UNSIGNED_CHAR,

    TYPE_SIGNED_SHORT,
    TYPE_UNSIGNED_SHORT,

    TYPE_SIGNED_LONG,
    TYPE_UNSIGNED_LONG,

    TYPE_SIGNED_LONG_LONG,
    TYPE_UNSIGNED_LONG_LONG,

    TYPE_FLOAT,
    TYPE_DOUBLE,
    TYPE_LONG_DOUBLE,
    
    TYPE_VOID_PTR,
    TYPE_VOID_PTR_REF,
};

enum object_value_state
{    
    CONSTANT_VALUE_STATE_UNINITIALIZED,        
    CONSTANT_VALUE_STATE_ANY,
    CONSTANT_VALUE_STATE_CONSTANT,
    //flow analysis
    CONSTANT_VALUE_NOT_EQUAL,
    CONSTANT_VALUE_EQUAL,
};

struct object
{    
    enum object_value_state state;
    enum object_value_type value_type;
    struct type type; //TODO to be removed

    const char* _Opt _Owner debug_name; //TODO we can remove this passing tthe type to print function

    union {
        _Bool bool_value;

        signed char signed_char_value;
        unsigned char unsigned_char_value;

        signed short signed_short_value;
        unsigned short unsigned_short_value;

        signed int signed_int_value;
        unsigned int unsigned_int_value;

        signed long signed_long_value;
        unsigned long unsigned_long_value;

        signed long long signed_long_long_value;
        unsigned long long unsigned_long_long_value;

        float float_value;
        double double_value;
        long double long_double_value;            

        void * void_pointer;
    } value;
    struct object* _Opt parent; //to be removed
    struct expression * _Opt p_init_expression;
    
    struct object* _Opt _Owner members;
    struct object* _Opt _Owner next;
};

void object_swap(struct object* a, struct object* b);
void object_print_value_debug(const struct object* a);
void object_destroy(_Opt _Dtor struct object* p);
void object_delete(struct object* _Opt _Owner p);
bool object_has_constant_value(const struct object* a);
void object_to_string(const struct object* a, char buffer[], int sz);


//Make constant value
struct object            object_make_wchar_t(wchar_t value);
struct object             object_make_size_t(size_t value);
struct object               object_make_bool(bool value);
struct object            object_make_nullptr();
struct object        object_make_signed_char(signed char value);
struct object      object_make_unsigned_char(unsigned char value);
struct object       object_make_signed_short(signed short value);
struct object     object_make_unsigned_short(unsigned short value);
struct object         object_make_signed_int(signed int value);
struct object       object_make_unsigned_int(unsigned int value);

struct object        object_make_signed_long(signed long value);
struct object      object_make_unsigned_long(unsigned long value);
struct object   object_make_signed_long_long(signed long long value);
struct object object_make_unsigned_long_long(unsigned long long value);
struct object              object_make_float(float value);
struct object             object_make_double(double value);
struct object        object_make_long_double(long double value);
struct object        object_make_reference(struct object* object);
struct object        object_make_pointer(struct object* object);
struct object        object_make_null_pointer();


//dynamic cast
void object_set_signed_int(struct object* a, long long value);
void object_set_unsigned_int(struct object* a, unsigned long long value);

struct object object_cast(enum object_value_type e, const struct object* a);
enum object_value_type  type_specifier_to_object_type(const enum type_specifier_flags type_specifier_flags);

errno_t object_increment_value(struct object* a);

//static cast
signed char object_to_signed_char(const struct object* a);
unsigned char object_to_unsigned_char(const struct object* a);
signed short object_to_signed_short(const struct object* a);
unsigned short object_to_unsigned_short(const struct object* a);
signed int object_to_signed_int(const struct object* a);
unsigned int object_to_unsigned_int(const struct object* a);
signed long object_to_signed_long(const struct object* a);
unsigned long object_to_unsigned_long(const struct object* a);
signed long long object_to_signed_long_long(const struct object* a);
unsigned long long object_to_unsigned_long_long(const struct object* a);
float object_to_float(const struct object* a);
double object_to_double(const struct object* a);
long double object_to_long_double(const struct object* a);
bool object_to_bool(const struct object* a);

int object_to_str(const struct object* a, int n, char str[/*n*/]);

int object_greater_than_or_equal(const struct object* a, const struct object* b);
int object_smaller_than_or_equal(const struct object* a, const struct object* b);
int object_equal(const struct object* a, const struct object* b);
int object_not_equal(const struct object* a, const struct object* b);
struct object object_add(const struct object* a, const struct object* b);
struct object object_sub(const struct object* a, const struct object* b);


//Overflow checks
bool unsigned_long_long_sub(_Ctor unsigned long long* result, unsigned long long a, unsigned long long b);
bool unsigned_long_long_mul(_Ctor unsigned long long* result, unsigned long long a, unsigned long long b);
bool unsigned_long_long_add(_Ctor unsigned long long* result, unsigned long long a, unsigned long long b);
bool signed_long_long_sub(_Ctor signed long long* result, signed long long a, signed long long b);
bool signed_long_long_add(_Ctor signed long long* result, signed long long a, signed long long b);
bool signed_long_long_mul(_Ctor signed long long* result, signed long long a, signed long long b);

void object_default_initialization(struct object* p_object, bool is_constant);

struct object* _Opt object_get_member(struct object* p_object, int index);

int make_object_with_name(const struct type* p_type, struct object* obj, const char* name);
int make_object(const struct type* p_type, struct object* obj);
struct object object_dup(const struct object* src);

bool object_is_reference(const struct object* p_object);
bool object_is_derived(const struct object* p_object);
bool object_is_signed(const struct object* p_object);
void object_set_any(struct object* p_object);

const struct object* object_get_referenced(const struct object* p_object);

NODISCARD
int object_set(
    struct parser_ctx* ctx, 
    struct object* to,
    struct expression* _Opt init_expression, 
    const struct object* from, 
    bool is_constant,
    bool requires_constant_initialization);

struct type;

enum object_value_type type_to_object_type(const struct type* type);

void object_print_to_debug(const struct object* object);

struct object* object_extend_array_to_index(const struct type* p_type, struct object* a, size_t n, bool is_constant);
struct object* object_get_non_const_referenced(struct object* p_object);



struct objects 
{
    struct object** items;
    int size;
    int capacity;
};

void objects_destroy(struct objects* arr);
int objects_push(struct objects* arr, struct object* obj); // returns 0 on success, ENOMEM on alloc fail




#include <limits.h>


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

//#pragma once



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

//#pragma once


struct parser_ctx;

enum expression_type
{
    EXPRESSION_TYPE_INVALID, 

    PRIMARY_EXPRESSION_ENUMERATOR,
    PRIMARY_EXPRESSION_DECLARATOR,    
    PRIMARY_EXPRESSION_STRING_LITERAL,
    PRIMARY_EXPRESSION__FUNC__, /*predefined identifier __func__ */
    PRIMARY_EXPRESSION_CHAR_LITERAL,
    PRIMARY_EXPRESSION_PREDEFINED_CONSTANT, /*true false*/
    PRIMARY_EXPRESSION_GENERIC,
    PRIMARY_EXPRESSION_NUMBER,
    PRIMARY_EXPRESSION_PARENTESIS,

    POSTFIX_EXPRESSION_FUNCTION_LITERAL,
    POSTFIX_EXPRESSION_COMPOUND_LITERAL,

    POSTFIX_FUNCTION_CALL, // ( ) 
    POSTFIX_ARRAY, // [ ]
    POSTFIX_DOT, // .
    POSTFIX_ARROW, // .
    POSTFIX_INCREMENT,
    POSTFIX_DECREMENT,


    UNARY_EXPRESSION_SIZEOF_EXPRESSION,
    UNARY_EXPRESSION_SIZEOF_TYPE,
    UNARY_EXPRESSION_NELEMENTSOF_TYPE,

    UNARY_EXPRESSION_TRAITS,
    UNARY_EXPRESSION_IS_SAME,
    UNARY_DECLARATOR_ATTRIBUTE_EXPR,
    UNARY_EXPRESSION_ALIGNOF,
    UNARY_EXPRESSION_ASSERT,

    UNARY_EXPRESSION_INCREMENT,
    UNARY_EXPRESSION_DECREMENT,

    UNARY_EXPRESSION_NOT,
    UNARY_EXPRESSION_BITNOT,
    UNARY_EXPRESSION_NEG,
    UNARY_EXPRESSION_PLUS,
    UNARY_EXPRESSION_CONTENT,
    UNARY_EXPRESSION_ADDRESSOF,

    CAST_EXPRESSION,

    MULTIPLICATIVE_EXPRESSION_MULT,
    MULTIPLICATIVE_EXPRESSION_DIV,
    MULTIPLICATIVE_EXPRESSION_MOD,

    ADDITIVE_EXPRESSION_PLUS,
    ADDITIVE_EXPRESSION_MINUS,

    SHIFT_EXPRESSION_RIGHT,
    SHIFT_EXPRESSION_LEFT,

    RELATIONAL_EXPRESSION_BIGGER_THAN,
    RELATIONAL_EXPRESSION_LESS_THAN,
    RELATIONAL_EXPRESSION_BIGGER_OR_EQUAL_THAN,
    RELATIONAL_EXPRESSION_LESS_OR_EQUAL_THAN,

    EQUALITY_EXPRESSION_EQUAL,
    EQUALITY_EXPRESSION_NOT_EQUAL,

    AND_EXPRESSION,
    EXCLUSIVE_OR_EXPRESSION,
    INCLUSIVE_OR_EXPRESSION,

    LOGICAL_OR_EXPRESSION,  //||
    LOGICAL_AND_EXPRESSION, //&&
    
    ASSIGNMENT_EXPRESSION_ASSIGN,
    ASSIGNMENT_EXPRESSION_PLUS_ASSIGN,
    ASSIGNMENT_EXPRESSION_MINUS_ASSIGN,
    ASSIGNMENT_EXPRESSION_MULTI_ASSIGN,
    ASSIGNMENT_EXPRESSION_DIV_ASSIGN,
    ASSIGNMENT_EXPRESSION_MOD_ASSIGN,
    ASSIGNMENT_EXPRESSION_SHIFT_LEFT_ASSIGN,
    ASSIGNMENT_EXPRESSION_SHIFT_RIGHT_ASSIGN,
    ASSIGNMENT_EXPRESSION_AND_ASSIGN,
    ASSIGNMENT_EXPRESSION_OR_ASSIGN,
    ASSIGNMENT_EXPRESSION_NOT_ASSIGN,
                       

    EXPRESSION_EXPRESSION,

    CONDITIONAL_EXPRESSION,
};

struct argument_expression_list
{
    /*
     argument-expression-list:
        assignment-expression
        argument-expression-list , assignment-expression
    */
    struct argument_expression* _Owner _Opt head;
    struct argument_expression* _Opt tail;
};

void argument_expression_list_destroy(_Dtor struct argument_expression_list* p);
void argument_expression_list_push(struct argument_expression_list* list, struct argument_expression* _Owner p);

struct generic_association
{
    /*
     generic-association:
       type-name : assignment-expression
       "default" : assignment-expression
    */

    struct type type;
    struct type_name* _Owner _Opt p_type_name;
    struct expression* _Owner expression;

    struct token* first_token;
    struct token* last_token;

    struct generic_association* _Owner _Opt next;
};

void generic_association_delete(struct generic_association* _Owner _Opt p);

struct generic_assoc_list
{
    struct generic_association* _Owner _Opt head;
    struct generic_association* _Opt tail;
};

void generic_assoc_list_add(struct generic_assoc_list* p, struct generic_association* _Owner item);
void generic_assoc_list_destroy(_Dtor struct generic_assoc_list* p);

struct generic_selection
{
    /*
      generic-selection:
        "_Generic" ( assignment-expression , generic-assoc-list )
    */


    /*
      Extension
      generic-selection:
        "_Generic" ( generic-argument, generic-assoc-list )

        generic-argument:
          assignment-expression
          type-name
    */


    struct expression* _Owner _Opt expression;
    struct type_name* _Owner _Opt type_name;
    /*
    * Points to the matching expression
    */
    struct expression* _Opt p_view_selected_expression;

    struct generic_assoc_list generic_assoc_list;
    struct token* first_token;
    struct token* last_token;
};

void generic_selection_delete(struct generic_selection* _Owner _Opt p);

struct expression
{
    enum expression_type expression_type;
    struct type type;

    struct object object;

    struct type_name* _Owner _Opt type_name;

    struct braced_initializer* _Owner _Opt braced_initializer;
    struct compound_statement* _Owner _Opt compound_statement; //function literal (lambda)
    struct generic_selection* _Owner _Opt generic_selection; //_Generic

    struct token* first_token;
    struct token* last_token;


    /*if expression is an identifier it points to its declaration*/
    struct declarator* _Opt declarator;
    struct init_declarator* _Opt p_init_declarator;

    int member_index; //used in post_fix .

    /*se for POSTFIX_FUNCTION_CALL post*/
    struct argument_expression_list argument_expression_list; //este node eh uma  chamada de funcao

    struct expression* _Owner _Opt condition_expr;
    struct expression* _Owner _Opt left;
    struct expression* _Owner _Opt right;

    bool is_assignment_expression;
};

//built-in semantics
bool expression_is_malloc(const struct expression* p);
bool expression_is_calloc(const struct expression* p);

void expression_delete(struct expression* _Owner _Opt p);

struct expression* _Owner _Opt assignment_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt constant_expression(struct parser_ctx* ctx, bool show_error_if_not_constant);
bool expression_is_subjected_to_lvalue_conversion(const struct expression*);

bool expression_get_variables(const struct expression* expr, int n, struct object* variables[/*n*/]);

bool expression_is_lvalue(const struct expression* expr);

bool expression_is_one(const struct expression* expression);
bool expression_is_zero(const struct expression* expression);
bool expression_is_null_pointer_constant(const struct expression* expression);
void expression_evaluate_equal_not_equal(const struct expression* left,
    const struct expression* right,
    struct expression* result,
    int op,
    bool disabled);

void check_diferent_enuns(struct parser_ctx* ctx,
                          const struct token* operator_token,
                          const struct expression* left,
                          const struct expression* right,
                          const char* message);

void check_assigment(struct parser_ctx* ctx,
    const struct type* left_type,
    const struct expression* right,
    enum assigment_type assigment_type);

void check_comparison(struct parser_ctx* ctx,
    struct expression* p_a_expression,
    struct expression* p_b_expression,
    const struct token* op_token);

struct object expression_eval(struct expression* p_expression);



//EXPERIMENTAL CONTRACTS
#define EXPERIMENTAL_CONTRACTS 1


struct scope
{
    int scope_level;
    struct hash_map tags;
    struct hash_map variables;

    struct scope* _Opt next;
    struct scope* _Opt previous;
};

void scope_destroy(_Dtor struct scope* p);

struct scope_list
{
    struct scope* _Opt head;
    struct scope* _Opt tail;
};
void scope_list_push(struct scope_list* list, struct scope* s);
void scope_list_pop(struct scope_list* list);

struct report
{
    int no_files;
    double cpu_time_used_sec;
    int error_count;
    int warnings_count;
    int info_count;

    bool test_mode;
    int test_failed;
    int test_succeeded;

    /*
     Warnings are removed (-test-mode) on demand at next line,
     but we can have more than one warning at same line then we need this array
    */
    enum diagnostic_id last_diagnostics_ids[2];

    //this error is expected in test-mode
    //it is used when the error aborts parsing
    enum diagnostic_id fatal_error_expected;

    /*
      direct commands like -autoconfig doesnt use report
    */
    bool ignore_this_report;
};

struct label_list_item
{
    struct token* p_last_usage;
    struct token* p_defined;
    struct label_list_item* next;
};

struct label_list
{
    struct label_list_item* _Opt _Owner head;
    struct label_list_item* _Opt tail;
};

struct label_list_item* _Opt label_list_find(struct label_list* list, const char* label_name);
void label_list_push(struct label_list* list, struct label_list_item* _Owner pitem);
void label_list_clear(struct label_list* list);


struct parser_ctx
{
    struct options options;

    /*
      file scope -> function params -> function -> inner scope
    */
    struct scope_list scopes;

    /*
    * Points to the function we're in. Or null in file scope.
    */
    struct declarator* _Opt p_current_function_opt;

    /*
    * Points to the scope where the current function is. (used in local functions)
    */
    struct scope* _Opt p_current_function_scope_opt;


    /*
    *  Used to track non-used labels or used and not defined labels
    */
    struct label_list label_list;

    /*
    * Points to the try-block we're in. Or null.
    */
    const struct try_statement* _Opt p_current_try_statement_opt;

    const struct defer_statement* _Opt p_current_defer_statement_opt;

    /*
    * Points to the selection_statement we're in. Or null.
    */
    const struct selection_statement* _Opt p_current_selection_statement;


    FILE* _Owner _Opt sarif_file;
    unsigned int sarif_entries;

    _View struct token_list input_list;
    struct token* _Opt current;
    struct token* _Opt previous;
    /*
       Expression inside sizeof etc.. are not evaluated
    */
    bool evaluation_is_disabled;

    bool inside_generic_association;

    int label_id; /*generates unique ids for labels*/

    struct report* p_report;

};

///////////////////////////////////////////////////////

void parser_ctx_destroy(_Opt _Dtor struct parser_ctx* ctx);


struct token* _Opt parser_look_ahead(const struct parser_ctx* ctx);
void unexpected_end_of_file(struct parser_ctx* ctx);
void parser_match(struct parser_ctx* ctx);
NODISCARD
int parser_match_tk(struct parser_ctx* ctx, enum token_type type);

struct token* _Opt previous_parser_token(const struct token* token);
struct declarator* _Opt find_declarator(const struct parser_ctx* ctx, const char* lexeme, struct scope** _Opt ppscope_opt);
struct enumerator* _Opt find_enumerator(const struct parser_ctx* ctx, const char* lexeme, struct scope** _Opt ppscope_opt);
struct map_entry* _Opt find_variables(const struct parser_ctx* ctx, const char* lexeme, struct scope* _Opt* _Opt ppscope_opt);

struct struct_or_union_specifier* _Opt find_struct_or_union_specifier(const struct parser_ctx* ctx, const char* lexeme);

void print_scope(struct scope_list* e);

/*
  This is the function called by the web version.
*/
char* _Opt _Owner CompileText(const char* options, const char* content);

_Bool compiler_diagnostic(enum diagnostic_id w,
    const struct parser_ctx* ctx,
    const struct token* _Opt p_token,
    const struct marker* _Opt p_marker,
    const char* fmt, ...);

int compile(int argc, const char** argv, struct report* error);

void print_type_qualifier_flags(struct osstream* ss, bool* first, enum type_qualifier_flags e_type_qualifier_flags);
bool print_type_alignment_flags(struct osstream* ss, bool* first, enum alignment_specifier_flags flags);
bool print_type_specifier_flags(struct osstream* ss, bool* first, enum type_specifier_flags e_type_specifier_flags);


struct expression_ctx;

struct declaration_specifier
{
    /*
       declaration-specifier:
         storage-class-specifier
         type-specifier-qualifier
         function-specifier
    */
    struct storage_class_specifier* _Owner _Opt storage_class_specifier;

    struct type_specifier_qualifier* _Owner _Opt type_specifier_qualifier;

    struct function_specifier* _Owner _Opt function_specifier;
    struct alignment_specifier* _Owner _Opt alignment_specifier;

    struct declaration_specifier* _Owner _Opt next;
};

struct declaration_specifier* _Owner _Opt declaration_specifier(struct parser_ctx* ctx);
void declaration_specifier_delete(struct declaration_specifier* _Owner _Opt p);

struct declaration_specifiers
{
    /*
     declaration-specifiers:
       declaration-specifier attribute-specifier-sequence _Opt
       declaration-specifier declaration-specifiers
    */

    /*cumulative flags*/
    enum attribute_flags  attributes_flags;
    enum type_specifier_flags type_specifier_flags;
    enum type_qualifier_flags type_qualifier_flags;
    enum storage_class_specifier_flags storage_class_specifier_flags;
    enum function_specifier_flags function_specifier_flags;
    enum msvc_declspec_flags msvc_declspec_flags;
    enum alignment_specifier_flags alignment_specifier_flags;
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt;

    /*shortcuts*/
    struct struct_or_union_specifier* _Opt struct_or_union_specifier;
    struct enum_specifier* _Opt enum_specifier;
    struct declarator* _Opt typedef_declarator;
    struct typeof_specifier* _Opt typeof_specifier;

    struct token* first_token; /*not _Owner*/
    struct token* last_token; /*not _Owner*/

    struct declaration_specifier* _Owner _Opt head;
    struct declaration_specifier* _Opt tail;
};

void print_declaration_specifiers(struct osstream* ss, struct declaration_specifiers* p);
struct declaration_specifiers* _Owner _Opt declaration_specifiers(struct parser_ctx* ctx, enum storage_class_specifier_flags default_storage_flag);
void declaration_specifiers_delete(struct declaration_specifiers* _Owner _Opt p);
void declaration_specifiers_add(struct declaration_specifiers* p, struct declaration_specifier* _Owner item);

struct static_assert_declaration
{
    /*
     static_assert-declaration:
       "static_assert" ( constant-expression , string-literal ) ;
       "static_assert" ( constant-expression ) ;
    */

    /*
      I am keeping the name static_assert_declaration but better is

      static_declaration:
       static_assert_declaration
       static_debug_declaration

      extension:
      "static_debug" ( constant-expression ) ;
      "static_set" ( constant-expression , string-literal) ;
    */

    struct token* first_token;
    struct token* last_token;
    struct expression* _Owner constant_expression;
    struct token* _Opt string_literal_opt;
};
struct static_assert_declaration* _Owner static_assert_declaration(struct parser_ctx* ctx);
void static_assert_declaration_delete(struct static_assert_declaration* _Owner _Opt p);

/*
  extension, pragma survives the preprocessor and become
  a pragma_declaration that can be used on the AST visit
*/

struct pragma_declaration {
    struct token* first_token;
    struct token* last_token;
};

struct pragma_declaration* _Owner pragma_declaration(struct parser_ctx* ctx);
void pragma_declaration_delete(struct pragma_declaration* _Owner _Opt p);

void execute_pragma(struct parser_ctx* ctx, struct pragma_declaration* p_pragma, bool on_flow_analysis);

struct attribute_specifier_sequence
{
    /*
     attribute-specifier-sequence:
       attribute-specifier-sequence _Opt attribute-specifier
    */

    struct token* first_token;
    struct token* last_token;
    enum attribute_flags  attributes_flags;
    struct attribute_specifier* _Owner _Opt head;
    struct attribute_specifier* _Opt tail;
};
struct attribute_specifier_sequence* _Owner _Opt attribute_specifier_sequence_opt(struct parser_ctx* ctx);
void attribute_specifier_sequence_delete(struct attribute_specifier_sequence* _Owner _Opt p);
void attribute_specifier_sequence_add(struct attribute_specifier_sequence* list, struct attribute_specifier* _Owner p_item);

struct attribute_specifier
{
    /*
     attribute-specifier:
        [ [ attribute-list ] ]
    */
    struct token* first_token;
    struct token* last_token;
    struct attribute_list* _Owner attribute_list;
    struct attribute_specifier* _Owner _Opt  next;
};

struct attribute_specifier* _Owner _Opt attribute_specifier(struct parser_ctx* ctx);
void attribute_specifier_delete(struct attribute_specifier* _Owner _Opt p);

struct attribute* _Owner _Opt attribute(struct parser_ctx* ctx);


struct storage_class_specifier
{
    /*
     storage-class-specifier:
      "auto"
      "constexpr"
      "extern"
      "register"
      "static"
      "thread_local"
      "typedef"
    */
    enum storage_class_specifier_flags flags;
    struct token* token;
};

struct storage_class_specifier* _Owner _Opt storage_class_specifier(struct parser_ctx* ctx);
void storage_class_specifier_delete(struct storage_class_specifier* _Owner _Opt p);

struct function_specifier
{
    /*
     function-specifier:
       inline
       _Noreturn
    */
    enum function_specifier_flags flags;
    struct token* token;
};
struct function_specifier* _Owner _Opt  function_specifier(struct parser_ctx* ctx);
void function_specifier_delete(struct function_specifier* _Owner _Opt  p);

struct typeof_specifier_argument
{
    /*
     typeof-specifier-argument:
       expression
       type-name
    */
    struct expression* _Owner _Opt expression;
    struct type_name* _Owner _Opt type_name;
};

void typeof_specifier_argument_delete(struct typeof_specifier_argument* _Owner _Opt p);

struct typeof_specifier
{
    /*
     typeof-specifier:
       "typeof" ( typeof-specifier-argument )
    */
    struct token* first_token;
    struct token* last_token;
    struct typeof_specifier_argument* _Owner typeof_specifier_argument;
    struct type type;
};

void typeof_specifier_delete(struct typeof_specifier* _Owner _Opt p);


struct msvc_declspec
{
    /*
        decl-specifier:
         __declspec ( extended-decl-modifier-seq )

        extended-decl-modifier-seq:
         extended-decl-modifieropt
         extended-decl-modifier extended-decl-modifier-seq

        extended-decl-modifier:
         align( number )
         allocate(" segname ")
         allocator
         appdomain
         code_seg(" segname ")
         deprecated
         dllimport
         dllexport
         empty_bases
         hybrid_patchable
         jitintrinsic
         naked
         noalias
         noinline
         noreturn
         nothrow
         novtable
         no_sanitize_address
         process
         property( { get=get-func-name | ,put=put-func-name } )
         restrict
         safebuffers
         selectany
         spectre(nomitigation)
         thread
         uuid(" ComObjectGUID ")
    */
    enum msvc_declspec_flags flags;
    struct token* token;
};

void msvc_declspec_delete(struct msvc_declspec* _Owner _Opt  p);
void msvc_declspec_sequence_opt(struct parser_ctx* ctx);

struct type_specifier
{
    /*
     type-specifier:
        "void"
        "char"
        "short"
        "int"
        "long"
        "float"
        "double"
        "signed"
        "unsigned"
        "_BitInt" ( constant-expression )
        "bool"
        "_Complex"
        "_Decimal32"
        "_Decimal64"
        "_Decimal128"
        atomic-type-specifier
        struct-or-union-specifier
        enum-specifier
        typedef-name
        typeof-specifier
    */
    enum type_specifier_flags flags;
    struct token* token;
    struct struct_or_union_specifier* _Owner _Opt struct_or_union_specifier;
    struct typeof_specifier* _Owner _Opt  typeof_specifier;
    struct enum_specifier* _Owner _Opt enum_specifier;
    struct msvc_declspec* _Owner _Opt msvc_declspec; //msvc
    struct declarator* _Opt typedef_declarator;
    struct atomic_type_specifier* _Owner _Opt  atomic_type_specifier;
};

struct type_specifier* _Owner _Opt type_specifier(struct parser_ctx* ctx);
void type_specifier_delete(struct type_specifier* _Owner _Opt p);

struct init_declarator_list
{
    /*
     init-declarator-list:
       init-declarator
       init-declarator-list , init-declarator
    */
    struct init_declarator* _Owner _Opt head;
    struct init_declarator* _Opt tail;
};

struct init_declarator_list init_declarator_list(struct parser_ctx* ctx,
    struct declaration_specifiers* p_declaration_specifiers);

void init_declarator_list_destroy(_Dtor struct init_declarator_list* p);
void init_declarator_list_add(struct init_declarator_list* list, struct init_declarator* _Owner p_item);

struct defer_list_item;

struct defer_list
{
    struct defer_list_item* _Opt _Owner head;
    struct defer_list_item* _Opt tail;
};


struct declaration
{
    /*
      declaration:
        declaration-specifiers init-declarator-list _Opt ;
        attribute-specifier-sequence declaration-specifiers init-declarator-list ;
        static_assert-declaration
        attribute-declaration
    */
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt;

    struct static_assert_declaration* _Owner _Opt static_assert_declaration;
    struct pragma_declaration* _Owner _Opt pragma_declaration;


    struct declaration_specifiers* _Owner _Opt  declaration_specifiers;
    struct init_declarator_list init_declarator_list;

    struct compound_statement* _Owner _Opt  function_body;
    struct defer_list defer_list; //arguments

    struct declarator* _Opt  contract_declarator;

    struct token* first_token;
    struct token* last_token;

    struct declaration* _Owner _Opt next;
};

void declaration_delete(struct declaration* _Owner _Opt p);
struct declaration* _Owner _Opt external_declaration(struct parser_ctx* ctx);

struct simple_declaration
{
    /*
    This is an extension to support C++ 17 if with initialization

    simple-declaration:
      declaration-specifiers init-declarator-list _Opt ;
      attribute-specifier-sequence declaration-specifiers init-declarator-list ;
    */
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt;
    struct declaration_specifiers* _Owner p_declaration_specifiers;
    struct init_declarator_list init_declarator_list;
    struct token* first_token;
    struct token* last_token;
};

void simple_declaration_delete(struct simple_declaration* _Owner _Opt p);

struct simple_declaration* _Owner _Opt simple_declaration(struct parser_ctx* ctx,
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt,
    bool ignore_semicolon);

struct condition
{
    /*
      This is an extension to support C++ 17 if with initialization

      condition :
       expression
       attribute-specifier-seq opt decl-specifier-seq declarator initializer
    */
    struct expression* _Owner _Opt expression;
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt;
    struct declaration_specifiers* _Owner _Opt p_declaration_specifiers;

    /*
      OBS:
      We must use p_init_declarator because it is kept on the scope
      as init_declarator when we are trying to parse init-statement or condition that
      are very similar
    */
    struct init_declarator* _Owner _Opt p_init_declarator;

    struct token* first_token;
    struct token* last_token;
};

void condition_delete(struct condition* _Owner _Opt p);
struct condition* _Owner _Opt condition(struct parser_ctx* ctx);

struct init_statement
{
    /*
        This is an extension to support C++ 17 if with initialization

        init-statement :
          expression-statement
          simple-declaration
    */
    struct expression_statement* _Owner _Opt p_expression_statement;
    struct simple_declaration* _Owner _Opt p_simple_declaration;
};

void init_statement_delete(struct init_statement* _Owner _Opt p);
struct init_statement* _Owner _Opt init_statement(struct parser_ctx* ctx, bool ignore_semicolon);

struct atomic_type_specifier
{
    /*
      atomic-type-specifier:
        "_Atomic" ( type-name )
    */
    struct token* token;
    struct type_name* _Owner type_name;
};

struct atomic_type_specifier* _Owner _Opt atomic_type_specifier(struct parser_ctx* ctx);
void atomic_type_specifier_delete(struct atomic_type_specifier* _Owner _Opt  p);

struct enumerator_list
{
    /*
     enumerator-list:
       enumerator
       enumerator-list , enumerator
    */
    struct enumerator* _Owner _Opt head;
    struct enumerator* _Opt tail;
};

struct enumerator_list enumerator_list(struct parser_ctx* ctx,
    const struct enum_specifier* p_enum_specifier
);

void enumerator_list_destroy(_Dtor struct enumerator_list* p_enum_specifier);
void enumerator_list_add(struct enumerator_list* list, struct enumerator* _Owner p_item);

struct enum_specifier
{
    /*
     enum-type-specifier:
       : specifier-qualifier-lis

     enum-specifier:
       "enum" attribute-specifier-sequence opt identifier opt enum-type-specifier opt  { enumerator-list }
       "enum" attribute-specifier-sequence opt identifier opt enum-type-specifier opt  { enumerator-list , }
       "enum" identifier enum-type-specifier opt
    */

    /*
       If has_shared_ownership is
        - true,  both AST and some map are sharing the ownership
        - false, only AST OR and some map have the ownership
    */
    bool has_shared_ownership;

    struct attribute_specifier_sequence* _Owner _Opt attribute_specifier_sequence_opt;
    struct specifier_qualifier_list* _Owner _Opt specifier_qualifier_list;

    char tag_name[200];

    struct enumerator_list enumerator_list;

    struct token* _Opt tag_token;
    struct token* first_token;
    /*points to the complete enum (can be self pointed)*/
    struct enum_specifier* _Opt p_complete_enum_specifier;
};

bool enum_specifier_has_fixed_underlying_type(const struct enum_specifier*);
struct enum_specifier* _Owner _Opt enum_specifier(struct parser_ctx*);

struct enum_specifier* _Owner enum_specifier_add_ref(struct enum_specifier* p);
void enum_specifier_delete(struct enum_specifier* _Owner _Opt p);
const struct enum_specifier* _Opt get_complete_enum_specifier(const struct enum_specifier* p_enum_specifier);
enum type_specifier_flags get_enum_type_specifier_flags(const struct enum_specifier* p_enum_specifier);

const struct enumerator* _Opt find_enumerator_by_value(const struct enum_specifier* p_enum_specifier, const struct object* object);

struct member_declaration_list
{
    /*
     member-declaration-list:
       member-declaration
       member-declaration-list member-declaration
    */

    struct token* first_token; /*TODO ? necessary*/
    struct token* last_token;
    struct member_declaration* _Owner _Opt head;
    struct member_declaration* _Opt tail;
};

struct member_declaration_list member_declaration_list(struct parser_ctx* ctx, struct struct_or_union_specifier*);
void member_declaration_list_destroy(_Opt _Dtor struct member_declaration_list* p);
void member_declaration_list_add(struct member_declaration_list* list, struct member_declaration* _Owner p_item);

struct member_declarator* _Opt find_member_declarator(struct member_declaration_list* list, const char* name, int* p_member_index);

struct struct_or_union_specifier
{
    /*
       If has_shared_ownership is
        - true,  both AST and some map are sharing the ownership
        - false, only AST OR and some map have the ownership
    */
    bool has_shared_ownership;

    struct attribute_specifier_sequence* _Owner _Opt attribute_specifier_sequence_opt;
    struct member_declaration_list member_declaration_list;

    struct token* first_token;
    struct token* last_token;

    bool is_owner;

    /*
    *Token that has a strut tag
    */
    struct token* _Opt tagtoken;

    char tag_name[200];
    /*geramos um tag name para anomimas, mas colocamos banonymousTag para true*/
    bool has_anonymous_tag;
    /*it was asked to show struct tag created for anonymous*/
    bool show_anonymous_tag;

    int scope_level; /*nivel escopo 0 global*/
    int visit_moved; /*nivel escopo 0 global*/

    /*
    * This points to the first struct_or_union_specifier that will have it´s
    * complete_struct_or_union_specifier_indirection pointing to the complete
    * struct_or_union_specifier.
    */
    struct struct_or_union_specifier* _Opt complete_struct_or_union_specifier_indirection;
};

struct struct_or_union_specifier* _Owner _Opt struct_or_union_specifier(struct parser_ctx* ctx);
struct struct_or_union_specifier* _Owner struct_or_union_specifier_add_ref(struct struct_or_union_specifier* p);
bool struct_or_union_specifier_is_union(const struct struct_or_union_specifier* p);
void struct_or_union_specifier_delete(struct struct_or_union_specifier* _Owner _Opt  p);

bool struct_or_union_specifier_is_complete(struct struct_or_union_specifier* p_struct_or_union_specifier);
struct struct_or_union_specifier* _Opt get_complete_struct_or_union_specifier(const struct struct_or_union_specifier* p_struct_or_union_specifier);

struct init_declarator
{
    /*
     init-declarator:
        declarator
        declarator = initializer
    */

    /*
       If has_shared_ownership is
        - true,  both AST and some map are sharing the ownership
        - false, only AST OR and some map have the ownership
    */
    bool has_shared_ownership;

    struct declarator* _Owner p_declarator;
    struct initializer* _Owner _Opt initializer;
    struct init_declarator* _Owner _Opt next;
};


struct init_declarator* _Owner init_declarator_add_ref(struct init_declarator* p);
void init_declarator_delete(struct init_declarator* _Owner _Opt p);
struct init_declarator* _Owner _Opt init_declarator(struct parser_ctx* ctx,
    struct declaration_specifiers* p_declaration_specifiers
);

struct initializer
{
    /*
     initializer:
       assignment-expression
       braced-initializer
    */
    struct token* first_token;
    struct braced_initializer* _Owner _Opt braced_initializer;
    struct expression* _Owner _Opt assignment_expression;
};

struct initializer* _Owner _Opt initializer(struct parser_ctx* ctx);
void initializer_delete(struct initializer* _Owner _Opt  p);

struct declarator
{
    /*
      declarator:
        pointer _Opt direct-declarator
    */

    /*
       If has_shared_ownership is
        - true,  both AST and Map are sharing the ownership
        - false, only AST OR Map have the ownership
    */
    bool has_shared_ownership;
    struct token* _Opt first_token_opt;
    struct token* _Opt last_token_opt;

    struct pointer* _Owner _Opt pointer;
    struct direct_declarator* _Owner _Opt direct_declarator;

    struct declaration_specifiers* _Opt declaration_specifiers;
    const struct specifier_qualifier_list* _Opt specifier_qualifier_list;

    struct token* _Opt name_opt; //shortcut , null for abstract declarator

    struct compound_statement* _Opt function_body;

    int num_uses; /*used to show not used warnings*/

    struct object object;

    /*user by flow analysis*/
    struct flow_object* _Opt p_flow_object;

    struct expression* _Opt _Owner p_expression_true;
    struct expression* _Opt _Owner p_expression_false;

    /*Used to check contracts*/
    struct expression* _Opt p_alias_of_expression;

    /*
       TODO it is duplicated with object
       final declarator type (after auto, typeof etc)
    */
    struct type type;
};

enum type_specifier_flags declarator_get_type_specifier_flags(const struct declarator* p);

struct declarator;
void print_declarator(struct osstream* ss, struct declarator* declarator, bool is_abstract);

struct declarator* _Owner _Opt declarator(struct parser_ctx* ctx,
    const struct specifier_qualifier_list* _Opt specifier_qualifier_list,
    struct declaration_specifiers* _Opt declaration_specifiers,
    bool abstract_acceptable,
    struct token** _Opt pptokenname);

struct declarator* _Owner declarator_add_ref(struct declarator* p);
void declarator_delete(struct declarator* _Owner _Opt  p);

struct array_declarator
{
    /*
     array-declarator:
        direct-declarator [ type-qualifier-list opt assignment-expression opt ]
        direct-declarator [ "static" type-qualifier-list opt assignment-expression ]
        direct-declarator [ type-qualifier-list "static" assignment-expression ]
        direct-declarator [ type-qualifier-list opt * ]
    */
    struct direct_declarator* _Owner _Opt direct_declarator;
    struct expression* _Owner _Opt  assignment_expression;
    struct expression* _Owner _Opt expression;
    struct type_qualifier_list* _Owner _Opt type_qualifier_list_opt;

    struct token* token;
    struct token* _Opt static_token_opt;
};

void array_declarator_delete(struct array_declarator* _Owner _Opt p);

/*
  Return a value > 0 if it has constant size
*/
size_t  array_declarator_get_size(const struct array_declarator* p_array_declarator);

struct function_declarator
{
    /*
     function-declarator:
       direct-declarator ( parameter-type-list _Opt )
    */
    struct direct_declarator* _Owner _Opt direct_declarator;
    struct scope parameters_scope;// used for scope parameters
    struct parameter_type_list* _Owner _Opt parameter_type_list_opt;
};

void function_declarator_delete(struct function_declarator* _Owner _Opt p);

struct direct_declarator
{
    /*
     direct-declarator:
        identifier attribute-specifier-sequence opt
        ( declarator )
        array-declarator attribute-specifier-sequence opt
        function-declarator attribute-specifier-sequence opt
    */
    struct token* _Opt name_opt;
    struct token* _Opt p_calling_convention;
    struct declarator* _Owner _Opt declarator;
    struct array_declarator* _Owner _Opt array_declarator;
    struct function_declarator* _Owner _Opt function_declarator;
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt;
};

void direct_declarator_delete(struct direct_declarator* _Owner _Opt p);

struct direct_declarator* _Owner _Opt direct_declarator(struct parser_ctx* ctx,
    const struct specifier_qualifier_list* _Opt specifier_qualifier_list,
    struct declaration_specifiers* _Opt declaration_specifiers,
    bool abstract_acceptable,
    struct token** _Opt pptoken_name
);

struct parameter_type_list
{
    /*
    parameter-type-list:
        parameter-list
        parameter-list , ...
        ...
    */
    bool is_var_args; /*(...)*/
    bool is_void;/*(void)*/
    struct parameter_list* _Owner _Opt parameter_list;
};

struct parameter_type_list* _Owner _Opt parameter_type_list(struct parser_ctx* ctx);
void parameter_type_list_delete(struct parameter_type_list* _Owner _Opt  p);

struct pointer
{
    /*
     pointer:
        * attribute-specifier-sequence opt type-qualifier-list opt
        * attribute-specifier-sequence opt type-qualifier-list opt pointer
    */
    struct attribute_specifier_sequence* _Owner _Opt  attribute_specifier_sequence_opt;
    struct type_qualifier_list* _Owner _Opt type_qualifier_list_opt;

    /*
      typedef int (__fastcall *pf)();
    */
    struct token* _Opt calling_convention;

    struct pointer* _Owner _Opt pointer;
};

struct pointer* _Owner _Opt  pointer_opt(struct parser_ctx* ctx);
void pointer_delete(struct pointer* _Owner _Opt p);

struct parameter_list
{
    /*
     parameter-list:
        parameter-declaration
        parameter-list , parameter-declaration
    */
    struct parameter_declaration* _Owner _Opt head;
    struct parameter_declaration* _Opt tail;
};

struct parameter_list* _Owner _Opt  parameter_list(struct parser_ctx* ctx);
void parameter_list_delete(struct parameter_list* _Owner _Opt  p);
void parameter_list_add(struct parameter_list* list, struct parameter_declaration* _Owner p_item);

struct parameter_declaration
{
    /*
     parameter-declaration:
      attribute-specifier-sequence opt declaration-specifiers declarator
      attribute-specifier-sequence opt declaration-specifiers abstract-declarator opt
    */
    struct attribute_specifier_sequence* _Owner _Opt  attribute_specifier_sequence_opt;

    struct declaration_specifiers* _Owner declaration_specifiers;
    struct declarator* _Owner _Opt  declarator;
    struct parameter_declaration* _Owner _Opt next;
};

struct parameter_declaration* _Owner _Opt parameter_declaration(struct parser_ctx* ctx);
void parameter_declaration_delete(struct parameter_declaration* _Owner _Opt  p);


struct type_name
{
    /*
     type-name:
       specifier-qualifier-list abstract-declarator opt
    */

    struct token* first_token;
    struct token* last_token;
    struct specifier_qualifier_list* _Owner specifier_qualifier_list;
    struct declarator* _Owner abstract_declarator; //TODO OPT
    struct type type;
};

struct type_name* _Owner _Opt type_name(struct parser_ctx* ctx);
void type_name_delete(struct type_name* _Owner _Opt p);
void print_type_name(struct osstream* ss, struct type_name* p);

struct argument_expression
{
    struct expression* _Owner expression;
    struct argument_expression* _Owner _Opt next;
    bool set_unkown; //used in flow analysis need to be removed..
};

void argument_expression_delete(struct argument_expression* _Owner _Opt  p);

struct braced_initializer
{
    /*
      { }
      { initializer-list }
      { initializer-list , }
    */

    struct token* first_token;
    struct token* last_token;
    struct initializer_list* _Owner _Opt initializer_list;
};

struct braced_initializer* _Owner _Opt braced_initializer(struct parser_ctx* ctx);
void braced_initializer_delete(struct braced_initializer* _Owner _Opt p);

struct type_specifier_qualifier
{
    /*
     type-specifier-qualifier:
       type-specifier
       type-qualifier
       alignment-specifier
    */

    struct type_specifier* _Owner _Opt type_specifier;
    struct type_qualifier* _Owner _Opt type_qualifier;
    struct alignment_specifier* _Owner _Opt alignment_specifier;

    struct type_specifier_qualifier* _Owner _Opt next;
};
struct type_specifier_qualifier* _Owner _Opt type_specifier_qualifier(struct parser_ctx* ctx);
void type_specifier_qualifier_delete(struct type_specifier_qualifier* _Owner _Opt  p);

struct specifier_qualifier_list
{
    /*
      specifier-qualifier-list:
        type-specifier-qualifier attribute-specifier-sequence _Opt
        type-specifier-qualifier specifier-qualifier-list
    */

    /*cumulative flags*/
    enum type_specifier_flags type_specifier_flags;
    enum type_qualifier_flags type_qualifier_flags;
    enum alignment_specifier_flags alignment_specifier_flags;


    /*shortcuts*/
    struct struct_or_union_specifier* _Opt struct_or_union_specifier;
    struct enum_specifier* _Opt enum_specifier;
    struct typeof_specifier* _Opt typeof_specifier;
    struct declarator* _Opt typedef_declarator;

    struct type_specifier_qualifier* _Owner _Opt head;
    struct type_specifier_qualifier* _Opt tail;
    struct token* first_token;
    struct token* last_token;

    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence;

};

struct specifier_qualifier_list* _Owner _Opt specifier_qualifier_list(struct parser_ctx* ctx);
void specifier_qualifier_list_delete(struct specifier_qualifier_list* _Owner _Opt p);
void specifier_qualifier_list_add(struct specifier_qualifier_list* list, struct type_specifier_qualifier* _Owner p_item);

void print_specifier_qualifier_list(struct osstream* ss, bool* first, struct specifier_qualifier_list* p_specifier_qualifier_list);

struct alignment_specifier
{
    /*
     alignment-specifier:
       "alignas" ( type-name )
       "alignas" ( constant-expression )
    */
    enum alignment_specifier_flags flags;
    struct type_name* _Owner _Opt type_name;
    struct expression* _Owner _Opt constant_expression;
    struct token* token;
};

struct alignment_specifier* _Owner _Opt alignment_specifier(struct parser_ctx* ctx);
void alignment_specifier_delete(struct alignment_specifier* _Owner _Opt p);

struct type_qualifier
{
    /*
     type-qualifier:
       "const"
       "restrict"
       "volatile"
       "_Atomic"
    */
    enum type_qualifier_flags flags;
    struct token* token;

    /*
      Next is used when inside struct type_qualifier_list
      Not used when inside struct type_specifier_qualifier
    */
    struct type_qualifier* _Owner _Opt next;
};

struct type_qualifier* _Owner _Opt type_qualifier(struct parser_ctx* ctx);
void type_qualifier_delete(struct type_qualifier* _Owner _Opt p);

struct member_declaration
{
    /*
     member-declaration:
       attribute-specifier-sequence opt specifier-qualifier-list member-declarator-list opt ;
       static_assert-declaration
    */
    struct specifier_qualifier_list* _Owner _Opt specifier_qualifier_list;
    struct member_declarator_list* _Owner _Opt member_declarator_list_opt;

    struct static_assert_declaration* _Owner _Opt static_assert_declaration;
    struct pragma_declaration* _Owner _Opt pragma_declaration;

    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt;
    struct member_declaration* _Owner _Opt next;

};

struct member_declaration* _Owner _Opt member_declaration(struct parser_ctx* ctx,
    struct struct_or_union_specifier*);
void member_declaration_delete(struct member_declaration* _Owner _Opt p);

struct member_declarator
{
    /*
     member-declarator:
       declarator
       declarator opt : constant-expression
    */

    struct declarator* _Owner _Opt declarator;
    struct expression* _Owner _Opt constant_expression;
    struct member_declarator* _Owner _Opt next;
};
void member_declarator_delete(struct member_declarator* _Owner _Opt p);

struct member_declarator_list
{
    /*
     member-declarator-list:
        member-declarator
        member-declarator-list , member-declarator
    */

    struct member_declarator* _Owner _Opt head;
    struct member_declarator* _Opt tail;
};

struct member_declarator* _Opt find_member_declarator_by_index(struct member_declaration_list* list, int member_index);

struct member_declarator_list* _Owner _Opt member_declarator_list(struct parser_ctx* ctx,
    struct struct_or_union_specifier*, /*not const*/
    const struct specifier_qualifier_list* specifier_qualifier_list
);
void member_declarator_list_delete(struct member_declarator_list* _Owner _Opt p);
void member_declarator_list_add(struct member_declarator_list* list, struct member_declarator* _Owner p_item);

struct block_item_list
{
    /*
     block-item-list:
       block-item
       block-item-list block-item
    */
    struct block_item* _Owner _Opt head;
    struct block_item* _Opt tail;
};

struct block_item_list block_item_list(struct parser_ctx* ctx, bool* error);
void block_item_list_destroy(_Dtor struct block_item_list* p);
void block_item_list_add(struct block_item_list* list, struct block_item* _Owner p_item);


struct compound_statement
{
    /*
     compound-statement:
       { block-item-list _Opt }
    */
    struct token* first_token; /*{*/
    struct token* last_token; /*}*/

    struct block_item_list block_item_list;

    //flow analysis flags
    struct diagnostic diagnostic_flags;

    struct defer_list defer_list;
};
struct compound_statement* _Owner _Opt compound_statement(struct parser_ctx* ctx);
void compound_statement_delete(struct compound_statement* _Owner _Opt p);

struct defer_statement
{
    /*
     defer-statement: (extension)
       "defer" secondary-block
    */
    struct token* first_token;
    struct token* last_token;
    struct secondary_block* _Owner secondary_block;
};

void defer_statement_delete(struct defer_statement* _Owner _Opt p);

struct defer_list_item
{
    struct declarator* _Opt declarator;
    struct defer_statement* _Opt defer_statement;
    struct defer_list_item* _Opt _Owner next;
};



void defer_list_add(struct defer_list* list, struct defer_list_item* _Owner p_item);
void defer_list_destroy(_Dtor struct defer_list* p);

struct try_statement
{
    /*
      try-statement: (extension)
       "try" secondary-block
       "try" secondary-block "catch" secondary-block
    */
    struct secondary_block* _Owner secondary_block;
    struct secondary_block* _Owner _Opt catch_secondary_block_opt;
    struct token* first_token; /*try*/
    struct token* last_token;
    struct token* _Opt catch_token_opt; /*catch*/

    int catch_label_id;
};

struct try_statement* _Owner _Opt try_statement(struct parser_ctx* ctx);
void try_statement_delete(struct try_statement* _Owner _Opt p);

struct case_label_list
{
    /*
       it is not the owner. The owner is the label statement
    */
    struct label* _Opt head;
    struct label* _Opt tail;
};

void case_label_list_push(struct case_label_list* list, struct label* pnew);
struct label* _Opt case_label_list_find(const struct case_label_list* list, const struct object* object);
struct label* _Opt case_label_list_find_default(const struct case_label_list* list);
struct label* _Opt case_label_list_find_range(const struct case_label_list* list, const struct object* begin, const struct object* end);

struct selection_statement
{
    /*
      selection-statement:
        "if" ( expression ) secondary-block
        "if" ( expression ) secondary-block "else" secondary-block
        "switch" ( expression ) secondary-block
    */

    //TODO rename
    /*C2Y
      selection-statement:
         if ( selection-header ) secondary-block
         if ( selection-header ) secondary-block else secondary-block
         switch ( selection-header ) secondary-block

     selection-header:
       expression
       declaration expression
       simple-declaration


     simple-declaration:
        attribute-specifier-sequenceopt declaration-specifiers declarator = initializer
    */

    struct init_statement* _Owner _Opt p_init_statement;
    struct condition* _Owner _Opt _Opt condition;

    struct secondary_block* _Owner secondary_block;
    struct secondary_block* _Owner _Opt else_secondary_block_opt;

    struct token* open_parentesis_token;
    struct token* close_parentesis_token;

    //case labels inside the switch
    struct case_label_list label_list;

    struct token* first_token;
    struct token* last_token;
    struct token* _Opt else_token_opt;
    struct defer_list defer_list;

    int label_id;
};

struct selection_statement* _Owner _Opt selection_statement(struct parser_ctx* ctx);
void selection_statement_delete(struct selection_statement* _Owner _Opt  p);

struct iteration_statement
{
    /*
      iteration-statement:
        "while" ( expression ) secondary-block
        "do" secondary-block "while" ( expression ) ;
        "for" ( expression _Opt ; expression _Opt ; expression _Opt ) secondary-block
        "for" ( declaration expression _Opt ; expression _Opt ) secondary-block
    */

    struct token* first_token;
    struct token* second_token; /*do {} while*/

    struct secondary_block* _Owner secondary_block;

    struct expression* _Owner _Opt expression1;
    struct expression* _Owner _Opt expression2;
    struct expression* _Owner _Opt expression0;
    struct declaration* _Owner _Opt declaration;
    struct defer_list defer_list;
};

struct iteration_statement* _Owner _Opt iteration_statement(struct parser_ctx* ctx);
void iteration_statement_delete(struct iteration_statement* _Owner _Opt p);

struct jump_statement
{
    /*
     jump-statement:
       "goto" identifier ;
       "continue" ;
       "break" ;
       "return" expression _Opt ;
    */

    struct token* _Opt label;
    struct token* first_token;
    struct token* last_token;
    struct expression* _Owner _Opt expression_opt;

    int label_id;
    struct defer_list defer_list;
};

struct jump_statement* _Owner _Opt jump_statement(struct parser_ctx* ctx);
void jump_statement_delete(struct jump_statement* _Owner _Opt p);

struct expression_statement
{
    /*
     expression-statement:
       expression _Opt;
       attribute-specifier-sequence expression ;
    */

    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt;
    struct expression* _Owner _Opt expression_opt;
};

struct expression_statement* _Owner _Opt expression_statement(struct parser_ctx* ctx, bool ignore_semicolon);
void expression_statement_delete(struct expression_statement* _Owner _Opt  p);

struct block_item
{
    /*
     block-item:
      declaration
      unlabeled-statement
      label
    */
    struct token* first_token; //?necessary
    struct declaration* _Owner _Opt declaration;
    struct unlabeled_statement* _Owner _Opt unlabeled_statement;
    struct label* _Owner _Opt label;

    struct block_item* _Owner _Opt next;
};

struct block_item* _Owner _Opt block_item(struct parser_ctx* ctx);
void block_item_delete(struct block_item* _Owner _Opt p);

struct compound_statement* _Owner _Opt function_body(struct parser_ctx* ctx);

struct designator
{
    /*
     designator:
       [ constant-expression ]
       . identifier
    */
    struct expression* _Owner _Opt constant_expression_opt;
    struct token* token;
    struct designator* _Owner _Opt next;
};

struct designator* _Owner _Opt designator(struct parser_ctx* ctx);
void designator_delete(struct designator* _Owner _Opt p);

struct initializer_list_item
{
    //A small grammar rewrite    
    /*
      initializer-list:
        designation opt initializer
        initializer-list , designation opt initializer
    */

    /*
      initializer-list-item:
        designation opt initializer

      initializer-list:
        initializer-list-item
        initializer-list , initializer-list-item
     */

    struct designation* _Opt _Owner designation;
    struct initializer* _Owner initializer;
    struct initializer_list_item* _Opt _Owner next;
};

void initializer_list_item_delete(struct initializer_list_item* _Owner _Opt p);

struct initializer_list
{
    /*
      initializer-list:
        designation opt initializer
        initializer-list , designation opt initializer
    */
    struct token* first_token;
    struct initializer_list_item* _Owner _Opt head;
    struct initializer_list_item* _Opt tail;
    int size;
};

struct initializer_list* _Owner _Opt initializer_list(struct parser_ctx* ctx);
void initializer_list_delete(struct initializer_list* _Owner _Opt p);
void initializer_list_add(struct initializer_list* list, struct initializer_list_item* _Owner p_item);

struct primary_block
{
    /*
       primary-block:
         compound-statement
         selection-statement
         iteration-statement
         defer-statement (extension)
         try-statement (extension)
    */

    struct compound_statement* _Owner _Opt compound_statement;
    struct selection_statement* _Owner _Opt selection_statement;
    struct iteration_statement* _Owner _Opt iteration_statement;
    struct defer_statement* _Owner _Opt defer_statement;
    struct try_statement* _Owner _Opt try_statement;
};

void primary_block_delete(struct primary_block* _Owner _Opt p);

struct secondary_block
{
    /*
     secondary-block:
       statement
    */
    struct token* first_token;
    struct token* last_token;
    struct statement* _Owner statement;
};

void secondary_block_delete(struct secondary_block* _Owner _Opt p);
bool secondary_block_ends_with_jump(struct secondary_block* _Opt p_secondary_block);

struct unlabeled_statement
{
    /*
      unlabeled-statement:
        expression-statement
        attribute-specifier-sequence opt primary-block
        attribute-specifier-sequence opt jump-statement
     */

    struct expression_statement* _Owner _Opt expression_statement;
    struct primary_block* _Owner _Opt primary_block;
    struct jump_statement* _Owner _Opt jump_statement;
};

struct unlabeled_statement* _Owner _Opt unlabeled_statement(struct parser_ctx* ctx);
void unlabeled_statement_delete(struct unlabeled_statement* _Owner _Opt p);

struct labeled_statement
{
    /*
     label statement:
       label statement
    */
    struct label* _Owner label;
    struct statement* _Owner statement;
};

struct labeled_statement* _Owner _Opt labeled_statement(struct parser_ctx* ctx);
void labeled_statement_delete(struct labeled_statement* _Owner _Opt p);

struct statement
{
    /*
     statement:
       labeled-statement
       unlabeled-statement
    */
    struct labeled_statement* _Owner _Opt labeled_statement;
    struct unlabeled_statement* _Owner _Opt unlabeled_statement;
};

struct statement* _Owner _Opt statement(struct parser_ctx* ctx);
void statement_delete(struct statement* _Owner _Opt  p);

struct designator_list
{
    /*
     designator-list:
       designator
       designator-list designator
    */
    struct designator* _Owner _Opt head;
    struct designator* _Opt tail;
};

struct designator_list* _Owner _Opt designator_list(struct parser_ctx* ctx);
void designator_list_delete(struct designator_list* _Owner _Opt p);
void designator_list_add(struct designator_list* list, struct designator* _Owner p_item);

struct designation
{
    /*
     designation:
       designator-list =
    */
    struct designator_list* _Owner designator_list;
    struct token* token;
};

struct designation* _Owner _Opt designation(struct parser_ctx* ctx);
void designation_delete(struct designation* _Owner _Opt p);

struct type_qualifier_list
{
    /*
     type-qualifier-list:
       type-qualifier
       type-qualifier-list type-qualifier
    */
    enum type_qualifier_flags flags;
    struct type_qualifier* _Owner _Opt head;
    struct type_qualifier* _Opt tail;
};

struct type_qualifier_list* _Owner _Opt type_qualifier_list(struct parser_ctx* ctx);
void type_qualifier_list_delete(struct type_qualifier_list* _Owner _Opt p);
void type_qualifier_list_add(struct type_qualifier_list* list, struct type_qualifier* _Owner p_item);

struct attribute_token
{
    enum attribute_flags attributes_flags;
    struct token* token;
};

struct attribute_token* _Owner _Opt attribute_token(struct parser_ctx* ctx);
void attribute_token_delete(struct attribute_token* _Owner _Opt p);
struct attribute
{
    enum attribute_flags  attributes_flags;
    struct attribute_token* _Owner attribute_token;
    struct attribute_argument_clause* _Owner attribute_argument_clause;
    struct attribute* _Owner _Opt next;
};

void attribute_delete(struct attribute* _Owner _Opt p);

struct attribute_list
{
    enum attribute_flags  attributes_flags;
    struct attribute* _Owner _Opt head;
    struct attribute* _Opt tail;
};

struct attribute_list* _Owner _Opt attribute_list(struct parser_ctx* ctx);
void attribute_list_destroy(_Dtor struct attribute_list* p);
void attribute_list_delete(struct attribute_list* _Owner _Opt p);

void attribute_list_add(struct attribute_list* list, struct attribute* _Owner p_item);

struct enumerator
{
    /*
      enumeration-constant:
        identifier

      enumerator:
        enumeration-constant attribute-specifier-sequence _Opt
        enumeration-constant attribute-specifier-sequence _Opt = constant-expression
    */

    /*
       If has_shared_ownership is
        - true,  both AST and some map are sharing the ownership
        - false, only AST OR and some map have the ownership
    */
    bool has_shared_ownership;

    struct token* token;
    struct attribute_specifier_sequence* _Owner _Opt attribute_specifier_sequence_opt;

    struct expression* _Owner _Opt constant_expression_opt;

    /*
      having the enum specifier we have better information about the type
    */
    const struct enum_specifier* enum_specifier;

    struct enumerator* _Owner _Opt next;
    struct object value;
};

struct enumerator* _Owner _Opt enumerator(struct parser_ctx* ctx, const struct enum_specifier* p_enum_specifier, struct object* p_enumerator_value);
struct enumerator* _Owner enumerator_add_ref(struct enumerator* p);
void enumerator_delete(struct enumerator* _Owner _Opt  p);

struct attribute_argument_clause
{
    /*
     attribute-argument-clause:
       ( balanced-token-sequence opt )
    */
    struct balanced_token_sequence* _Owner _Opt p_balanced_token_sequence;
    struct token* token;
};

struct attribute_argument_clause* _Owner _Opt attribute_argument_clause(struct parser_ctx* ctx);
void attribute_argument_clause_delete(struct attribute_argument_clause* _Owner _Opt p);

bool first_of_attribute(const struct parser_ctx* ctx);

struct balanced_token
{
    struct token* token;
    struct balanced_token* _Owner _Opt next;
};

struct balanced_token_sequence
{
    struct balanced_token* _Owner _Opt head;
    struct balanced_token* _Opt tail;
};

struct balanced_token_sequence* _Owner _Opt balanced_token_sequence_opt(struct parser_ctx* ctx);
void balanced_token_sequence_delete(struct balanced_token_sequence* _Owner _Opt  p);

bool is_first_of_conditional_expression(struct parser_ctx* ctx);
bool first_of_type_name(const struct parser_ctx* ctx);
bool first_of_type_name_ahead(const struct parser_ctx* ctx);

struct argument_expression_list argument_expression_list(struct parser_ctx* ctx);

struct declaration_list
{
    struct declaration* _Owner _Opt head;
    struct declaration* _Opt tail;
};

struct declaration_list translation_unit(struct parser_ctx* ctx, bool* berror);
void declaration_list_destroy(_Dtor struct declaration_list* list);

struct label
{
    /*
     label:
       attribute-specifier-sequence opt identifier :
       attribute-specifier-sequence opt "case" constant-expression :
       attribute-specifier-sequence opt "default" :
    */
    struct expression* _Owner _Opt constant_expression;
    struct expression* _Owner _Opt constant_expression_end;

    struct token* _Opt p_identifier_opt;
    struct token* p_first_token;
    struct label* _Opt next;

    int label_id; //unique id inside the function scope
};

struct label* _Owner _Opt label(struct parser_ctx* ctx);
void label_delete(struct label* _Owner _Opt p);

struct ast
{
    struct token_list token_list;
    struct declaration_list declaration_list;
};


struct ast get_ast(struct options* options, const char* filename, const char* source, struct report* report);
void ast_destroy(_Dtor struct ast* ast);
struct type make_type_using_declarator(struct parser_ctx* ctx, struct declarator* pdeclarator);


struct declaration_list parse(struct parser_ctx* ctx, struct token_list* list, bool* berror);
const char* _Owner _Opt compile_source(const char* pszoptions, const char* content, struct report* report);

int initializer_init_new(struct parser_ctx* ctx,
                         struct type* p_current_object_type,
                         struct object* p_current_object,
                         struct initializer* braced_initializer,
                         bool is_constant,
                         bool requires_constant_initialization);

struct object* _Opt find_object_declarator_by_index(struct object* p_object, struct member_declaration_list* list, int member_index);


NODISCARD
bool unsigned_long_long_sub(_Ctor unsigned long long* result, unsigned long long a, unsigned long long b)
{
    *result = 0;

    if (a < b)
        return false;

    *result = a - b;
    return true;
}

NODISCARD
bool unsigned_long_long_mul(_Ctor unsigned long long* result, unsigned long long a, unsigned long long b)
{
    *result = 0;

    if (b == 0)
    {
        /*
          b cannot be zero in the next test
          so we solve this case here
        */
        *result = 0;
        return true;
    }

    if (a > ULLONG_MAX / b)
        return false;

    *result = a * b;
    return true;
}

NODISCARD
bool unsigned_long_long_add(_Ctor unsigned long long* result, unsigned long long a, unsigned long long b)
{
    *result = 0;

    if (a > ULLONG_MAX - b)
    {
        //a=2
        //b=254
        return false;
    }
    *result = a + b;
    return true;
}

NODISCARD
bool signed_long_long_sub(_Ctor signed long long* result, signed long long a, signed long long b)
{
    *result = 0;

    if (a >= 0 && b >= 0)
    {
    }
    else if (a < 0 && b < 0)
    {
    }
    else
    {
        if (a < 0)
        {
            if (a < LLONG_MIN + b)
            {
                // (-128) - (-1)
                return false;
            }
        }
        else
        {
            if (b == LLONG_MIN)
            {
                // 0 - (-128)                
                return false;
            }

            if (a > LLONG_MAX - (-b))
            {
                /*
                *  1 - (-127)
                *  2 - (-126)
                */
                return false;
            }
        }
    }

    *result = a - b;
    return true;
}

NODISCARD
bool signed_long_long_add(_Ctor signed long long* result, signed long long a, signed long long b)
{
    *result = 0;

    if (a >= 0 && b >= 0)
    {
        /*both positive*/
        if (a > LLONG_MAX - b)
        {
            //2+126
            return false;
        }
    }
    else if (a < 0 && b < 0)
    {

        if (a == LLONG_MIN || b == LLONG_MIN)
        {
            //(-128) + (-128)
            return false;
        }

        if (a < LLONG_MIN - b)
        {
            // (-127) + (-127)
            return false;
        }
    }
    else
    {
        /*one positive another negative*/

    }

    *result = a + b;
    return true;
}

NODISCARD
bool signed_long_long_mul(_Ctor signed long long* result, signed long long a, signed long long b)
{
    *result = 0;

    if (a > 0 && b > 0)
    {
        if (a > LLONG_MAX / b)
        {
            //2*64
            return false;
        }
    }
    else if (a < 0 && b < 0)
    {

        if (a == LLONG_MIN || b == LLONG_MIN)
        {
            //(-128)*(-128)
            return false;
        }

        if (-a > LLONG_MAX / -b)
        {
            //-127 * -127
            return false;
        }
    }
    else
    {
        if (a == 0 || b == 0)
        {
            *result = 0;
            return true;
        }
        if (b > 0)
        {
            if (a < LLONG_MIN / b)
                //(-127) * (2)
                return false;
        }
        else
        {
            if (b < LLONG_MIN / a)
            {
                //2*(-128)
                return false;
            }
        }
    }

    *result = a * b;
    return true;
}

void object_swap(struct object* a, struct object* b)
{
    struct object temp = *a;
    *a = *b;
    *b = temp;
}

void object_destroy(_Opt _Dtor struct object* p)
{
    assert(p->next == NULL);

    type_destroy(&p->type);
    free((void* _Owner)p->debug_name);

    struct object* _Owner _Opt item = p->members;
    while (item)
    {
        struct object* _Owner _Opt next = item->next;
        item->next = NULL;
        object_delete(item);
        item = next;
    }
}

void object_delete(struct object* _Opt _Owner p)
{
    if (p)
    {
        object_destroy(p);
        free(p);
    }
}

bool object_has_constant_value(const struct object* a)
{
    a = object_get_referenced(a);
    return a->state == CONSTANT_VALUE_STATE_CONSTANT;
}

void object_to_string(const struct object* a, char buffer[], int sz)
{
    a = object_get_referenced(a);

    buffer[0] = 0;
    switch (a->value_type)
    {
    case TYPE_BOOL:
        snprintf(buffer, sz, "%s", a->value.signed_char_value ? "true" : "false");
        break;

    case TYPE_SIGNED_CHAR:
        snprintf(buffer, sz, "%c", a->value.signed_char_value);
        break;

    case TYPE_UNSIGNED_CHAR:
        snprintf(buffer, sz, "%c", a->value.unsigned_char_value);
        break;

    case TYPE_SIGNED_SHORT:
        snprintf(buffer, sz, "%c", a->value.signed_short_value);
        break;
    case TYPE_UNSIGNED_SHORT:
        snprintf(buffer, sz, "%c", a->value.signed_short_value);
        break;

    case TYPE_SIGNED_INT:
        snprintf(buffer, sz, "%d", a->value.signed_int_value);
        break;

    case TYPE_UNSIGNED_INT:
        snprintf(buffer, sz, "%u", a->value.signed_int_value);
        break;

    case TYPE_SIGNED_LONG:
        snprintf(buffer, sz, "%ld", a->value.signed_long_value);
        break;

    case TYPE_UNSIGNED_LONG:
        break;

    case TYPE_SIGNED_LONG_LONG:
        snprintf(buffer, sz, "%lud", a->value.signed_long_value);
        break;

    case TYPE_UNSIGNED_LONG_LONG:
        snprintf(buffer, sz, "%llu", a->value.signed_long_long_value);
        break;

    case TYPE_FLOAT:
        snprintf(buffer, sz, "%f", a->value.float_value);
        break;

    case TYPE_DOUBLE:
        snprintf(buffer, sz, "%f", a->value.double_value);
        break;

    case TYPE_LONG_DOUBLE:
        snprintf(buffer, sz, "%Lf", a->value.long_double_value);
        break;

    case TYPE_VOID_PTR:
        if (a->value.void_pointer == NULL)
            snprintf(buffer, sz, "null");
        else
            snprintf(buffer, sz, "%p", a->value.void_pointer);
        break;
    }
}

struct object object_make_size_t(size_t value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;

#if defined(_WIN64) || defined(__x86_64__) 
    r.value_type = TYPE_UNSIGNED_LONG_LONG;
    r.value.unsigned_long_long_value = value;
#else
    r.value_type = TYPE_UNSIGNED_INT;
    r.value.unsigned_int_value = value;
#endif

    return r;
}

struct object object_make_nullptr()
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;

    r.value_type = TYPE_SIGNED_INT;
    r.value.signed_short_value = 0;
    return r;
}

struct object object_make_wchar_t(wchar_t value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;

#ifdef _WIN32
    static_assert(_Generic(L' ', unsigned short : 1), "");
    r.value_type = TYPE_SIGNED_SHORT;
    r.value.signed_short_value = value;
#else
    static_assert(_Generic(L' ', int : 1), "");
    r.value_type = TYPE_SIGNED_INT;
    r.value.signed_int_value = value;
#endif

    return r;
}

struct object object_make_bool(bool value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;

    r.value_type = TYPE_BOOL;
    r.value.bool_value = value;
    return r;
}

#pragma warning( push )
#pragma warning( disable : 4244 )

int object_to_str(const struct object* a, int n, char str[/*n*/])
{
    str[0] = '\0';

    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL:
    case TYPE_SIGNED_CHAR:
    case TYPE_SIGNED_SHORT:
    case TYPE_SIGNED_INT:
    case TYPE_SIGNED_LONG:
    case TYPE_SIGNED_LONG_LONG:
    {
        long long v = object_to_signed_long_long(a);
        snprintf(str, n, "%lld", v);
    }
    break;

    case TYPE_UNSIGNED_CHAR:
    case TYPE_UNSIGNED_SHORT:
    case TYPE_UNSIGNED_INT:
    case TYPE_UNSIGNED_LONG:
    case TYPE_UNSIGNED_LONG_LONG:
    {
        unsigned long long v = object_to_unsigned_long_long(a);
        snprintf(str, n, "%llu", v);
    }
    break;

    case TYPE_FLOAT:
    case TYPE_DOUBLE:
    case TYPE_LONG_DOUBLE:
    {
        long double v = object_to_long_double(a);
        snprintf(str, n, "%Lf", v);
    }
    break;
    }

    return 0;
}

void object_set_signed_int(struct object* a, long long value)
{
    a = object_get_non_const_referenced(a);
    a->state = CONSTANT_VALUE_EQUAL;

    switch (a->value_type)
    {

    case TYPE_BOOL: a->value.bool_value = value; break;
    case TYPE_SIGNED_CHAR:  a->value.signed_char_value = value; break;
    case TYPE_UNSIGNED_CHAR:  a->value.unsigned_char_value = value; break;
    case TYPE_SIGNED_SHORT:  a->value.signed_short_value = value; break;
    case TYPE_UNSIGNED_SHORT:  a->value.unsigned_short_value = value; break;
    case TYPE_SIGNED_INT:  a->value.signed_int_value = value; break;
    case TYPE_UNSIGNED_INT:  a->value.unsigned_int_value = value; break;
    case TYPE_SIGNED_LONG:  a->value.signed_long_value = value; break;
    case TYPE_UNSIGNED_LONG:  a->value.unsigned_long_value = value; break;
    case TYPE_SIGNED_LONG_LONG:  a->value.signed_long_long_value = value; break;
    case TYPE_UNSIGNED_LONG_LONG:  a->value.unsigned_long_long_value = value; break;
    case TYPE_FLOAT:  a->value.float_value = value; break;
    case TYPE_DOUBLE:  a->value.double_value = value; break;
    case TYPE_LONG_DOUBLE:  a->value.long_double_value = value; break;

    case TYPE_VOID_PTR:  assert(0);  break;

    }

}

void object_set_unsigned_int(struct object* a, unsigned long long value)
{
    a = object_get_non_const_referenced(a);
    a->state = CONSTANT_VALUE_EQUAL;

    switch (a->value_type)
    {

    case TYPE_BOOL: a->value.bool_value = value; break;
    case TYPE_SIGNED_CHAR:  a->value.signed_char_value = value; break;
    case TYPE_UNSIGNED_CHAR:  a->value.unsigned_char_value = value; break;
    case TYPE_SIGNED_SHORT:  a->value.signed_short_value = value; break;
    case TYPE_UNSIGNED_SHORT:  a->value.unsigned_short_value = value; break;
    case TYPE_SIGNED_INT:  a->value.signed_int_value = value; break;
    case TYPE_UNSIGNED_INT:  a->value.unsigned_int_value = value; break;
    case TYPE_SIGNED_LONG:  a->value.signed_long_value = value; break;
    case TYPE_UNSIGNED_LONG:  a->value.unsigned_long_value = value; break;
    case TYPE_SIGNED_LONG_LONG:  a->value.signed_long_long_value = value; break;
    case TYPE_UNSIGNED_LONG_LONG:  a->value.unsigned_long_long_value = value; break;
    case TYPE_FLOAT:  a->value.float_value = value; break;
    case TYPE_DOUBLE:  a->value.double_value = value; break;
    case TYPE_LONG_DOUBLE:  a->value.long_double_value = value; break;
    case TYPE_VOID_PTR:  assert(0);  break;

    }

}

bool object_to_bool(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: return a->value.void_pointer != 0;
    }
    assert(0);
    return 0;
}
struct object object_make_signed_char(signed char value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_SIGNED_CHAR;
    r.value.signed_char_value = value;
    return r;
}

errno_t object_increment_value(struct object* a)
{
    a = object_get_non_const_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL:
        a->value.bool_value++;
        break;
    case TYPE_SIGNED_CHAR:
        a->value.signed_char_value++;
        break;
    case TYPE_UNSIGNED_CHAR:
        a->value.unsigned_char_value++;
        break;
    case TYPE_SIGNED_SHORT:
        a->value.signed_short_value++;
        break;
    case TYPE_UNSIGNED_SHORT:
        a->value.unsigned_short_value++;
        break;
    case TYPE_SIGNED_INT:
        a->value.signed_int_value++;
        break;
    case TYPE_UNSIGNED_INT:
        a->value.unsigned_int_value++;
        break;
    case TYPE_SIGNED_LONG:
        a->value.signed_long_value++;
        break;
    case TYPE_UNSIGNED_LONG:
        a->value.unsigned_long_value++;
        break;
    case TYPE_SIGNED_LONG_LONG:
        a->value.signed_long_long_value++;
        break;
    case TYPE_UNSIGNED_LONG_LONG:
        a->value.unsigned_long_long_value++;
        break;
    case TYPE_FLOAT:
        a->value.float_value++;
        break;
    case TYPE_DOUBLE:
        a->value.double_value++;
        break;
    case TYPE_LONG_DOUBLE:
        a->value.long_double_value++;
        break;

    case TYPE_VOID_PTR:  assert(0);  break;

    default:
        return 1;
    }

    return 0;
}

signed char object_to_signed_char(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR:  return 0;// a->value.void_pointer;  break;
    }
    assert(0);
    return 0;
}

struct object object_make_unsigned_char(unsigned char value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_UNSIGNED_CHAR;
    r.value.unsigned_char_value = value;
    return r;
}

unsigned char object_to_unsigned_char(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;

    }
    assert(0);
    return 0;
}
struct object object_make_signed_short(signed short value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_SIGNED_SHORT;
    r.value.signed_short_value = value;
    return r;
}

signed short object_to_signed_short(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;

    }
    assert(0);
    return 0;
}
struct object object_make_unsigned_short(unsigned short value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_UNSIGNED_SHORT;
    r.value.unsigned_short_value = value;
    return r;
}

unsigned short object_to_unsigned_short(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}
struct object object_make_signed_int(signed int value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_SIGNED_INT;
    r.value.signed_int_value = value;
    return r;
}

signed int object_to_signed_int(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: return (int)a->value.void_pointer; break;
    }
    assert(0);
    return 0;
}
struct object object_make_unsigned_int(unsigned int value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_UNSIGNED_INT;
    r.value.unsigned_int_value = value;
    return r;
}

unsigned int object_to_unsigned_int(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: return (int)a->value.void_pointer; break;
    }
    assert(0);
    return 0;
}
struct object object_make_signed_long(signed long value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_SIGNED_LONG;
    r.value.signed_long_value = value;
    return r;
}

signed long object_to_signed_long(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}
struct object object_make_unsigned_long(unsigned long value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_UNSIGNED_LONG;
    r.value.unsigned_long_value = value;
    return r;
}

unsigned long object_to_unsigned_long(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}
struct object object_make_signed_long_long(signed long long value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_SIGNED_LONG_LONG;
    r.value.signed_long_long_value = value;
    return r;
}

signed long long object_to_signed_long_long(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}
struct object object_make_unsigned_long_long(unsigned long long value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_UNSIGNED_LONG_LONG;
    r.value.unsigned_long_long_value = value;
    return r;
}

unsigned long long object_to_unsigned_long_long(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}
struct object object_make_float(float value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_FLOAT;
    r.value.float_value = value;
    return r;
}

float object_to_float(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}
struct object object_make_double(double value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_DOUBLE;
    r.value.double_value = value;
    return r;
}

double object_to_double(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}


struct object  object_make_null_pointer()
{
    struct object null_object = {
       .value_type = TYPE_VOID_PTR,
            .value.void_pointer = NULL,
            .state = CONSTANT_VALUE_EQUAL,
    };

    return null_object;
}

struct object object_make_pointer(struct object* object)
{
    object = object_get_non_const_referenced(object);

    struct object r = { 0 };
    r.state = CONSTANT_VALUE_EQUAL;
    r.value_type = TYPE_VOID_PTR;
    r.value.void_pointer = object;

    return r;
}

struct object object_make_reference(struct object* object)
{
    object = object_get_non_const_referenced(object);

    struct object r = { 0 };
    r.state = CONSTANT_VALUE_EQUAL;
    r.value_type = TYPE_VOID_PTR_REF;
    r.value.void_pointer = object;

    return r;
}

struct object object_make_long_double(long double value)
{
    struct object r = { 0 };
    r.state = CONSTANT_VALUE_STATE_CONSTANT;
    r.value_type = TYPE_LONG_DOUBLE;
    r.value.long_double_value = value;
    return r;
}

long double object_to_long_double(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL: return a->value.bool_value;
    case TYPE_SIGNED_CHAR: return a->value.signed_char_value;
    case TYPE_UNSIGNED_CHAR: return a->value.unsigned_char_value;
    case TYPE_SIGNED_SHORT: return a->value.signed_short_value;
    case TYPE_UNSIGNED_SHORT: return a->value.unsigned_short_value;
    case TYPE_SIGNED_INT: return a->value.signed_int_value;
    case TYPE_UNSIGNED_INT: return a->value.unsigned_int_value;
    case TYPE_SIGNED_LONG: return a->value.signed_long_value;
    case TYPE_UNSIGNED_LONG: return a->value.unsigned_long_value;
    case TYPE_SIGNED_LONG_LONG: return a->value.signed_long_long_value;
    case TYPE_UNSIGNED_LONG_LONG: return a->value.unsigned_long_long_value;
    case TYPE_FLOAT: return a->value.float_value;
    case TYPE_DOUBLE: return a->value.double_value;
    case TYPE_LONG_DOUBLE: return a->value.long_double_value;
    case TYPE_VOID_PTR: assert(0); break;
    }
    assert(0);
    return 0;
}



#pragma warning( pop )



struct object object_cast(enum object_value_type t, const struct object* v)
{
    v = object_get_referenced(v);

    //No changes
    if (v->value_type == t)
        return *v;


    //This function is generated by this program
    /*
            struct type
            {
                const char * type;
                const char * name;
                const char * value_type;
            };
            struct type types[] =
            {
                {"bool", "bool", "TYPE_BOOL"},
                {"signed char", "signed_char", "TYPE_SIGNED_CHAR"},
                {"unsigned char", "unsigned_char", "TYPE_UNSIGNED_CHAR"},
                {"signed short", "signed_short", "TYPE_SIGNED_SHORT"},
                {"unsigned short", "unsigned_short", "TYPE_UNSIGNED_SHORT"},
                {"signed int", "signed_int", "TYPE_SIGNED_INT"},
                {"unsigned int", "unsigned_int", "TYPE_UNSIGNED_INT"},
                {"signed long", "signed_long", "TYPE_SIGNED_LONG"},
                {"unsigned long", "unsigned_long", "TYPE_UNSIGNED_LONG"},
                {"signed long long", "signed_long_long", "TYPE_SIGNED_LONG_LONG"},
                {"unsigned long long", "unsigned_long_long", "TYPE_UNSIGNED_LONG_LONG"},
                {"float", "float", "TYPE_FLOAT"},
                {"double", "double", "TYPE_DOUBLE"},
                {"long double", "long_double", "TYPE_LONG_DOUBLE"}
            };


            int main()
            {
                FILE * f = fopen("imp.c", "w");
                if (f == NULL)
                    return;

                fprintf(f, "struct object cast(enum object_value_type t, struct object * v)\n");
                fprintf(f, "{\n");
                for (int i = 0; i < sizeof(types) / sizeof(types[0]); i++)
                {
                    fprintf(f, "if (t == %s)\n", types[i].value_type);
                    fprintf(f, "{\n");
                    for (int j = 0; j < sizeof(types) / sizeof(types[0]); j++)
                    {
                        if (i == j)
                            continue;
                        fprintf(f, "if (v->type == %s)\n", types[j].value_type);
                        fprintf(f, " return object_make_%s((%s)v->value.%s_value);\n", types[i].name, types[i].type, types[j].name);
                    }
                    fprintf(f, "}\n");
                }
                fprintf(f, "}\n");

                fclose(f);
            }
    */
    if (t == TYPE_BOOL)
    {
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_bool((bool)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_bool((bool)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_bool((bool)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_bool((bool)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_bool((bool)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_bool((bool)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_bool((bool)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_bool((bool)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_bool((bool)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_bool((bool)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_bool((bool)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_bool((bool)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_bool((bool)v->value.long_double_value);
    }
    if (t == TYPE_SIGNED_CHAR)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_signed_char((signed char)v->value.bool_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_signed_char((signed char)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_signed_char((signed char)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_signed_char((signed char)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_signed_char((signed char)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_signed_char((signed char)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_signed_char((signed char)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_signed_char((signed char)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_signed_char((signed char)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_signed_char((signed char)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_signed_char((signed char)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_signed_char((signed char)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_signed_char((signed char)v->value.long_double_value);
    }
    if (t == TYPE_UNSIGNED_CHAR)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_unsigned_char((unsigned char)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_unsigned_char((unsigned char)v->value.signed_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_unsigned_char((unsigned char)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_unsigned_char((unsigned char)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_unsigned_char((unsigned char)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_unsigned_char((unsigned char)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_unsigned_char((unsigned char)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_unsigned_char((unsigned char)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_unsigned_char((unsigned char)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_unsigned_char((unsigned char)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_unsigned_char((unsigned char)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_unsigned_char((unsigned char)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_unsigned_char((unsigned char)v->value.long_double_value);
    }
    if (t == TYPE_SIGNED_SHORT)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_signed_short((signed short)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_signed_short((signed short)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_signed_short((signed short)v->value.unsigned_char_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_signed_short((signed short)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_signed_short((signed short)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_signed_short((signed short)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_signed_short((signed short)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_signed_short((signed short)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_signed_short((signed short)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_signed_short((signed short)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_signed_short((signed short)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_signed_short((signed short)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_signed_short((signed short)v->value.long_double_value);
    }
    if (t == TYPE_UNSIGNED_SHORT)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_unsigned_short((unsigned short)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_unsigned_short((unsigned short)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_unsigned_short((unsigned short)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_unsigned_short((unsigned short)v->value.signed_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_unsigned_short((unsigned short)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_unsigned_short((unsigned short)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_unsigned_short((unsigned short)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_unsigned_short((unsigned short)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_unsigned_short((unsigned short)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_unsigned_short((unsigned short)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_unsigned_short((unsigned short)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_unsigned_short((unsigned short)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_unsigned_short((unsigned short)v->value.long_double_value);
    }
    if (t == TYPE_SIGNED_INT)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_signed_int((signed int)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_signed_int((signed int)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_signed_int((signed int)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_signed_int((signed int)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_signed_int((signed int)v->value.unsigned_short_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_signed_int((signed int)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_signed_int((signed int)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_signed_int((signed int)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_signed_int((signed int)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_signed_int((signed int)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_signed_int((signed int)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_signed_int((signed int)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_signed_int((signed int)v->value.long_double_value);
    }
    if (t == TYPE_UNSIGNED_INT)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_unsigned_int((unsigned int)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_unsigned_int((unsigned int)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_unsigned_int((unsigned int)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_unsigned_int((unsigned int)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_unsigned_int((unsigned int)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_unsigned_int((unsigned int)v->value.signed_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_unsigned_int((unsigned int)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_unsigned_int((unsigned int)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_unsigned_int((unsigned int)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_unsigned_int((unsigned int)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_unsigned_int((unsigned int)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_unsigned_int((unsigned int)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_unsigned_int((unsigned int)v->value.long_double_value);
    }
    if (t == TYPE_SIGNED_LONG)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_signed_long((signed long)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_signed_long((signed long)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_signed_long((signed long)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_signed_long((signed long)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_signed_long((signed long)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_signed_long((signed long)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_signed_long((signed long)v->value.unsigned_int_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_signed_long((signed long)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_signed_long((signed long)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_signed_long((signed long)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_signed_long((signed long)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_signed_long((signed long)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_signed_long((signed long)v->value.long_double_value);
    }
    if (t == TYPE_UNSIGNED_LONG)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_unsigned_long((unsigned long)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_unsigned_long((unsigned long)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_unsigned_long((unsigned long)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_unsigned_long((unsigned long)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_unsigned_long((unsigned long)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_unsigned_long((unsigned long)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_unsigned_long((unsigned long)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_unsigned_long((unsigned long)v->value.signed_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_unsigned_long((unsigned long)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_unsigned_long((unsigned long)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_unsigned_long((unsigned long)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_unsigned_long((unsigned long)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_unsigned_long((unsigned long)v->value.long_double_value);
    }
    if (t == TYPE_SIGNED_LONG_LONG)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_signed_long_long((signed long long)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_signed_long_long((signed long long)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_signed_long_long((signed long long)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_signed_long_long((signed long long)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_signed_long_long((signed long long)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_signed_long_long((signed long long)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_signed_long_long((signed long long)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_signed_long_long((signed long long)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_signed_long_long((signed long long)v->value.unsigned_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_signed_long_long((signed long long)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_signed_long_long((signed long long)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_signed_long_long((signed long long)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_signed_long_long((signed long long)v->value.long_double_value);
    }
    if (t == TYPE_UNSIGNED_LONG_LONG)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_unsigned_long_long((unsigned long long)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_unsigned_long_long((unsigned long long)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_unsigned_long_long((unsigned long long)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_unsigned_long_long((unsigned long long)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_unsigned_long_long((unsigned long long)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_unsigned_long_long((unsigned long long)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_unsigned_long_long((unsigned long long)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_unsigned_long_long((unsigned long long)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_unsigned_long_long((unsigned long long)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_unsigned_long_long((unsigned long long)v->value.signed_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_unsigned_long_long((unsigned long long)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_unsigned_long_long((unsigned long long)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_unsigned_long_long((unsigned long long)v->value.long_double_value);
    }
    if (t == TYPE_FLOAT)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_float((float)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_float((float)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_float((float)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_float((float)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_float((float)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_float((float)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_float((float)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_float((float)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_float((float)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_float((float)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_float((float)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_float((float)v->value.double_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_float((float)v->value.long_double_value);
    }
    if (t == TYPE_DOUBLE)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_double((double)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_double((double)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_double((double)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_double((double)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_double((double)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_double((double)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_double((double)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_double((double)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_double((double)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_double((double)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_double((double)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_double((double)v->value.float_value);
        if (v->value_type == TYPE_LONG_DOUBLE)
            return object_make_double((double)v->value.long_double_value);
    }
    if (t == TYPE_LONG_DOUBLE)
    {
        if (v->value_type == TYPE_BOOL)
            return object_make_long_double((long double)v->value.bool_value);
        if (v->value_type == TYPE_SIGNED_CHAR)
            return object_make_long_double((long double)v->value.signed_char_value);
        if (v->value_type == TYPE_UNSIGNED_CHAR)
            return object_make_long_double((long double)v->value.unsigned_char_value);
        if (v->value_type == TYPE_SIGNED_SHORT)
            return object_make_long_double((long double)v->value.signed_short_value);
        if (v->value_type == TYPE_UNSIGNED_SHORT)
            return object_make_long_double((long double)v->value.unsigned_short_value);
        if (v->value_type == TYPE_SIGNED_INT)
            return object_make_long_double((long double)v->value.signed_int_value);
        if (v->value_type == TYPE_UNSIGNED_INT)
            return object_make_long_double((long double)v->value.unsigned_int_value);
        if (v->value_type == TYPE_SIGNED_LONG)
            return object_make_long_double((long double)v->value.signed_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG)
            return object_make_long_double((long double)v->value.unsigned_long_value);
        if (v->value_type == TYPE_SIGNED_LONG_LONG)
            return object_make_long_double((long double)v->value.signed_long_long_value);
        if (v->value_type == TYPE_UNSIGNED_LONG_LONG)
            return object_make_long_double((long double)v->value.unsigned_long_long_value);
        if (v->value_type == TYPE_FLOAT)
            return object_make_long_double((long double)v->value.float_value);
        if (v->value_type == TYPE_DOUBLE)
            return object_make_long_double((long double)v->value.double_value);
    }
    struct object empty = { 0 };
    return empty;
}

void object_default_initialization(struct object* p_object, bool is_constant)
{
    if (p_object->members == NULL)
    {
        if (is_constant)
            p_object->state = CONSTANT_VALUE_STATE_CONSTANT;
        else
            p_object->state = CONSTANT_VALUE_EQUAL;
        p_object->value.unsigned_long_long_value = 0;
    }

    if (type_is_union(&p_object->type))
    {
        struct object* _Opt p = p_object->members;
        if (p)
        {
            object_default_initialization(p, is_constant);
        }
    }
    else
    {
        struct object* _Opt p = p_object->members;
        while (p)
        {
            object_default_initialization(p, is_constant);
            p = p->next;
        }
    }
}

struct object* object_get_non_const_referenced(struct object* p_object)
{
    if (p_object->value_type == TYPE_VOID_PTR_REF)
    {
        p_object = p_object->value.void_pointer;
    }

    assert(p_object->value_type != TYPE_VOID_PTR_REF);

    return p_object;
}


const struct object* object_get_referenced(const struct object* p_object)
{
    if (p_object->value_type == TYPE_VOID_PTR_REF)
    {
        p_object = p_object->value.void_pointer;
    }

    assert(p_object->value_type != TYPE_VOID_PTR_REF);

    return p_object;
}


int get_rank(enum object_value_type t)
{
    //https://cigix.me/c23#6.3.1.1
    if (t == TYPE_SIGNED_LONG_LONG ||
        t == TYPE_UNSIGNED_LONG_LONG)
    {
        return 80;
    }
    else if (t == TYPE_SIGNED_LONG ||
             t == TYPE_UNSIGNED_LONG)
    {
        return 50;
    }
    else if (t == TYPE_SIGNED_INT ||
             t == TYPE_UNSIGNED_INT)
    {
        return 40;
    }
    else if (t == TYPE_SIGNED_SHORT ||
             t == TYPE_UNSIGNED_SHORT)
    {
        return 30;
    }
    else if (t == TYPE_SIGNED_CHAR ||
             t == TYPE_UNSIGNED_CHAR)
    {
        return 20;
    }
    return 0;
}


int get_size(enum object_value_type t)
{
    if (t == TYPE_SIGNED_LONG_LONG ||
        t == TYPE_UNSIGNED_LONG_LONG)
    {
        return sizeof(long long);
    }
    else if (t == TYPE_SIGNED_LONG ||
             t == TYPE_UNSIGNED_LONG)
    {
        return sizeof(long);
    }
    else if (t == TYPE_SIGNED_INT ||
             t == TYPE_UNSIGNED_INT)
    {
        return sizeof(int);
    }
    else if (t == TYPE_SIGNED_SHORT ||
             t == TYPE_UNSIGNED_SHORT)
    {
        return sizeof(short);
    }
    else if (t == TYPE_SIGNED_CHAR ||
             t == TYPE_UNSIGNED_CHAR)
    {
        return sizeof(char);
    }
    else if (t == TYPE_VOID_PTR)
    {
        return sizeof(void*);
    }

    return 1;
}

bool is_signed(enum object_value_type t)
{
    switch (t)
    {
    case TYPE_BOOL:
    case TYPE_SIGNED_CHAR:
    case TYPE_SIGNED_SHORT:
    case TYPE_SIGNED_INT:
    case TYPE_SIGNED_LONG:
    case TYPE_SIGNED_LONG_LONG:
    case TYPE_DOUBLE:
    case TYPE_LONG_DOUBLE:
        return true;
    case TYPE_VOID_PTR: break;
    default:
        break;
    }
    return false;
}

enum object_value_type to_unsigned(enum object_value_type t)
{
    switch (t)
    {
    case TYPE_SIGNED_CHAR: return TYPE_UNSIGNED_CHAR;
    case TYPE_SIGNED_SHORT:return TYPE_UNSIGNED_SHORT;
    case TYPE_SIGNED_INT: return TYPE_UNSIGNED_INT;
    case TYPE_SIGNED_LONG:return  TYPE_UNSIGNED_LONG;
    case TYPE_SIGNED_LONG_LONG: return TYPE_UNSIGNED_LONG_LONG;
    default:
        break;
    }
    return t;
}

bool is_unsigned(enum object_value_type t)
{
    switch (t)
    {
    case TYPE_BOOL:
    case TYPE_UNSIGNED_CHAR:
    case TYPE_UNSIGNED_SHORT:
    case TYPE_UNSIGNED_INT:
    case TYPE_UNSIGNED_LONG:
    case TYPE_UNSIGNED_LONG_LONG:
        return true;
    default:
        break;
    }
    return false;
}

void object_set_any(struct object* p_object)
{
    p_object = object_get_non_const_referenced(p_object);
    p_object->state = CONSTANT_VALUE_STATE_ANY;
    struct object* p = p_object->members;
    while (p)
    {
        object_set_any(p);
        p = p->next;
    }
}

bool object_is_signed(const struct object* p_object)
{
    p_object = (struct object* _Opt) object_get_referenced(p_object);
    return is_signed(p_object->value_type);
}

bool object_is_derived(const struct object* p_object)
{
    if (p_object->value_type == TYPE_VOID_PTR_REF)
        return false;

    return p_object->members != NULL;
}

bool object_is_reference(const struct object* p_object)
{
    return p_object->value_type == TYPE_VOID_PTR_REF;
}

static void object_fix_parent(struct object* p_object, struct object* parent)
{
    struct object* _Opt it = p_object->members;
    while (it)
    {
        it->parent = parent;
        it = it->next;
    }
}

struct object* _Opt object_get_member(struct object* p_object, int index)
{
    p_object = (struct object* _Opt) object_get_referenced(p_object);

    if (p_object->members == NULL)
        return NULL; //tODO

    struct object* _Opt it = p_object->members;
    int count = 0;
    while (it)
    {
        if (index == count)
            return it;
        count++;
        it = it->next;
    }

    return NULL;
}

int object_set(
    struct parser_ctx* ctx,
    struct object* to,
    struct expression* _Opt p_init_expression,
    const struct object* from,
    bool is_constant,
    bool requires_constant_initialization)
{
    try
    {
        from = object_get_referenced(from);

        to->p_init_expression = p_init_expression;

        if (object_is_derived(to))
        {
            struct object* _Opt it_to = to->members;
            struct object* _Opt it_from = from->members;

            while (it_from && it_to)
            {
                object_set(ctx, it_to, NULL, it_from, is_constant, requires_constant_initialization);
                it_to = it_to->next;
                it_from = it_from->next;
            }
            if (it_from != NULL || it_to != NULL)
            {
                //TODO  in dev
              // assert(false);//TODO
            }
            //assert(it_from == NULL);
            //assert(it_to == NULL);
        }
        else
        {
            assert(to->members == NULL);

            to->state = from->state;
            to->value = object_cast(to->value_type, from).value;

            if (requires_constant_initialization &&
                !object_has_constant_value(from))
            {
                if (!type_is_pointer_or_array(&p_init_expression->type) &&
                    !type_is_function(&p_init_expression->type))
                {
                    struct token* _Opt tk = p_init_expression ?
                        p_init_expression->first_token : ctx->current;

                    compiler_diagnostic(C_ERROR_REQUIRES_COMPILE_TIME_VALUE,
                        ctx,
                        tk,
                        NULL,
                        "requires a compile time object");

                    throw;
                }
            }

            if (is_constant && from->state == CONSTANT_VALUE_STATE_CONSTANT)
            {
                /*
                  struct point { int x; } p = { .x = 1 };
                  const int i = p.x; //not a exactly constant
                  int j = i;}
                */

                //TODO
                /*
                      struct X {int x;};
                      int main() { constexpr struct X x = (struct X){ .x = 50 };}
                */
                to->state = CONSTANT_VALUE_STATE_CONSTANT;
            }
            else
            {
                if (to->state == CONSTANT_VALUE_STATE_CONSTANT)
                {
                    //Sample int i = 1; 1 is constant but i will not be
                    to->state = CONSTANT_VALUE_EQUAL;
                }
            }
        }
    }
    catch
    {
        return 1;
    }

    return 0;
}

struct object* _Owner _Opt make_object_ptr_core(const struct type* p_type, const char* name)
{
    struct object* _Owner _Opt p_object = NULL;

    try
    {
        if (p_type->category == TYPE_CATEGORY_FUNCTION)
        {
            p_object = calloc(1, sizeof * p_object);
            if (p_object == NULL)
                throw;
            p_object->debug_name = strdup(name);
            p_object->type = type_dup(p_type);
            return p_object;
        }

        if (p_type->category == TYPE_CATEGORY_POINTER)
        {
            p_object = calloc(1, sizeof * p_object);
            if (p_object == NULL)
                throw;

            *p_object = object_make_nullptr();
            p_object->state = CONSTANT_VALUE_STATE_UNINITIALIZED;
            assert(p_object->debug_name == NULL);
            p_object->debug_name = strdup(name);

            type_destroy(&p_object->type);
            p_object->type = type_dup(p_type);

            return p_object;
        }

        if (p_type->category == TYPE_CATEGORY_ARRAY)
        {
            p_object = calloc(1, sizeof * p_object);
            if (p_object == NULL)
                throw;
            p_object->type = type_dup(p_type);
            p_object->debug_name = strdup(name);

            if (p_type->num_of_elements > 0)
            {
                struct type array_item_type = get_array_item_type(p_type);

                //too big..
                const unsigned long long max_elements = p_type->num_of_elements > 1000 ? 1000 : p_type->num_of_elements;

                struct object* _Opt p_tail_object = NULL;
                for (unsigned long long i = 0; i < max_elements; i++)
                {
                    char buffer[200] = { 0 };
                    snprintf(buffer, sizeof buffer, "%s[%llu]", name, i);
                    struct object* _Owner _Opt p_member_obj = make_object_ptr_core(&array_item_type, buffer);
                    if (p_member_obj == NULL)
                    {
                        type_destroy(&array_item_type);
                        throw;
                    }
                    p_member_obj->parent = p_object;

                    free(p_member_obj->debug_name);
                    p_member_obj->debug_name = strdup(buffer);
                    if (p_tail_object == NULL)
                    {
                        assert(p_object->members == NULL);
                        p_object->members = p_member_obj;
                    }
                    else
                    {
                        assert(p_object->next == NULL);
                        p_tail_object->next = p_member_obj;
                    }

                    p_tail_object = p_member_obj;
                }
                type_destroy(&array_item_type);
            }

            return p_object;
        }


        if (p_type->struct_or_union_specifier == NULL)
        {
            p_object = calloc(1, sizeof * p_object);
            if (p_object == NULL)
                throw;


            p_object->state = CONSTANT_VALUE_STATE_UNINITIALIZED;
            p_object->value_type = type_to_object_type(p_type);
            p_object->value.signed_long_long_value = -1;
            p_object->debug_name = strdup(name);
            p_object->type = type_dup(p_type);

            return p_object;
        }


        struct struct_or_union_specifier* _Opt p_struct_or_union_specifier =
            get_complete_struct_or_union_specifier(p_type->struct_or_union_specifier);

        if (p_struct_or_union_specifier == NULL)
        {
            //incomplete
            throw;
        }

        p_object = calloc(1, sizeof * p_object);
        if (p_object == NULL)
            throw;

        p_object->debug_name = strdup(name);
        p_object->type = type_dup(p_type);

        struct object* _Opt p_last_member_obj = NULL;

        struct member_declaration* _Opt p_member_declaration =
            p_struct_or_union_specifier->member_declaration_list.head;

        while (p_member_declaration)
        {
            if (p_member_declaration->member_declarator_list_opt)
            {
                struct member_declarator* _Opt p_member_declarator =
                    p_member_declaration->member_declarator_list_opt->head;

                while (p_member_declarator)
                {
                    if (p_member_declarator->declarator)
                    {
                        char buffer[200] = { 0 };
                        snprintf(buffer, sizeof buffer, "%s.%s", name, p_member_declarator->declarator->name_opt->lexeme);


                        struct object* _Owner _Opt p_member_obj = make_object_ptr_core(&p_member_declarator->declarator->type, buffer);
                        if (p_member_obj == NULL)
                            throw;

                        p_member_obj->parent = p_object;

                        free(p_member_obj->debug_name);
                        p_member_obj->debug_name = strdup(buffer);

                        if (p_object->members == NULL)
                        {
                            p_object->members = p_member_obj;
                        }
                        else
                        {
                            //assert(p_last_member_obj->next != NULL);
                            p_last_member_obj->next = p_member_obj;
                        }
                        p_last_member_obj = p_member_obj;
                    }
                    p_member_declarator = p_member_declarator->next;
                }
            }
            else if (p_member_declaration->specifier_qualifier_list != NULL)
            {
                if (p_member_declaration->specifier_qualifier_list->struct_or_union_specifier)
                {
                    struct type t = { 0 };
                    t.category = TYPE_CATEGORY_ITSELF;
                    t.struct_or_union_specifier = p_member_declaration->specifier_qualifier_list->struct_or_union_specifier;
                    t.type_specifier_flags = TYPE_SPECIFIER_STRUCT_OR_UNION;

                    char buffer[200] = { 0 };
                    snprintf(buffer, sizeof buffer, ".%s", name);


                    struct object* _Owner _Opt p_member_obj = make_object_ptr_core(&t, buffer);
                    if (p_member_obj == NULL)
                        throw;

                    free(p_member_obj->debug_name);
                    p_member_obj->debug_name = strdup(buffer);

                    p_member_obj->parent = p_object;
                    if (p_last_member_obj == NULL)
                    {
                        assert(p_object->members == NULL);
                        p_object->members = p_member_obj;
                    }
                    else
                    {
                        p_last_member_obj->next = p_member_obj;
                    }
                    p_last_member_obj = p_member_obj;

                    type_destroy(&t);
                }
            }
            p_member_declaration = p_member_declaration->next;
        }
        return p_object;
    }
    catch
    {
        object_delete(p_object);
        p_object = NULL;
    }

    return NULL;

}

struct object* _Owner _Opt make_object_ptr(const struct type* p_type)
{
    return make_object_ptr_core(p_type, "");
}

int make_object_with_name(const struct type* p_type, struct object* obj, const char* name)
{
    struct object* _Owner _Opt p = make_object_ptr_core(p_type, name);
    if (p)
    {
        *obj = *p;
        object_fix_parent(obj, obj);
        free(p);
        return 0;
    }
    return 1;
}

struct object object_dup(const struct object* src)
{
    assert(src->members == NULL);
    //assert(src->next == NULL); ??

    struct object result = *src;
    result.type = type_dup(&src->type);

    if (src->debug_name)
        result.debug_name = strdup(src->debug_name);

    result.next = NULL;

    return result;
}

int make_object(const struct type* p_type, struct object* obj)
{
    return make_object_with_name(p_type, obj, "");
}


enum object_value_type  type_specifier_to_object_type(const enum type_specifier_flags type_specifier_flags)
{

    if (type_specifier_flags & TYPE_SPECIFIER_BOOL)
        return TYPE_BOOL;

    if (type_specifier_flags & TYPE_SPECIFIER_FLOAT)
        return TYPE_FLOAT;

    if (type_specifier_flags & TYPE_SPECIFIER_DOUBLE)
    {
        if (type_specifier_flags & TYPE_SPECIFIER_LONG)
            return TYPE_LONG_DOUBLE;
        return TYPE_DOUBLE;
    }


    if (type_specifier_flags & TYPE_SPECIFIER_UNSIGNED)
    {
        if (type_specifier_flags & TYPE_SPECIFIER_CHAR)
            return TYPE_UNSIGNED_CHAR;
        if (type_specifier_flags & TYPE_SPECIFIER_SHORT)
            return TYPE_UNSIGNED_SHORT;
        if (type_specifier_flags & TYPE_SPECIFIER_LONG)
            return TYPE_UNSIGNED_LONG; /*check before int*/
        if (type_specifier_flags & TYPE_SPECIFIER_INT)
            return TYPE_UNSIGNED_INT;
        if (type_specifier_flags & TYPE_SPECIFIER_LONG_LONG)
            return TYPE_UNSIGNED_LONG_LONG;
    }
    else
    {
        if (type_specifier_flags & TYPE_SPECIFIER_CHAR)
            return TYPE_SIGNED_CHAR;
        if (type_specifier_flags & TYPE_SPECIFIER_SHORT)
            return TYPE_SIGNED_SHORT;
        if (type_specifier_flags & TYPE_SPECIFIER_LONG)
            return TYPE_SIGNED_LONG; /*check before int*/
        if (type_specifier_flags & TYPE_SPECIFIER_INT)
            return TYPE_SIGNED_INT;
        if (type_specifier_flags & TYPE_SPECIFIER_LONG_LONG)
            return TYPE_SIGNED_LONG_LONG;
    }
    return TYPE_SIGNED_INT;
}

enum object_value_type type_to_object_type(const struct type* type)
{
    if (type_is_pointer(type))
    {
#if defined(_WIN64) || defined(__x86_64__) 
        return TYPE_UNSIGNED_LONG_LONG;
#else
        return TYPE_UNSIGNED_INT;
#endif
    }

    return type_specifier_to_object_type(type->type_specifier_flags);
}



void object_print_value_debug(const struct object* a)
{
    a = object_get_referenced(a);

    switch (a->value_type)
    {

    case TYPE_BOOL:
        printf("%s (bool)", a->value.bool_value ? "true" : "false");
        break;

    case TYPE_SIGNED_CHAR:

        printf("%d (signed char)", (int)a->value.signed_char_value);
        break;


    case TYPE_UNSIGNED_CHAR:
        printf("%d (unsigned char)", (int)a->value.unsigned_char_value);
        break;


    case TYPE_SIGNED_SHORT:
        printf("%d (short)", a->value.signed_short_value);
        break;

    case TYPE_UNSIGNED_SHORT:
        printf("%d (unsigned short)", a->value.unsigned_short_value);
        break;

    case TYPE_SIGNED_INT:
        printf("%d (int)", a->value.signed_int_value);
        break;
    case TYPE_UNSIGNED_INT:
        printf("%du (unsigned int)", a->value.unsigned_int_value);
        break;
    case TYPE_SIGNED_LONG:
        printf("%ld (long)", a->value.signed_long_value);
        break;
    case TYPE_UNSIGNED_LONG:
        printf("%lu (unsigned long)", a->value.unsigned_long_value);
        break;
    case TYPE_SIGNED_LONG_LONG:
        printf("%lld (long long)", a->value.signed_long_long_value);
        break;
    case TYPE_UNSIGNED_LONG_LONG:
        printf("%llu (unsigned long long)", a->value.unsigned_long_long_value);
        break;
    case TYPE_FLOAT:
        printf("%f (float)", a->value.float_value);
        break;
    case TYPE_DOUBLE:
        printf("%lf (double)", a->value.double_value);
        break;
    case TYPE_LONG_DOUBLE:
        printf("%Lf (long double)", a->value.long_double_value);
        break;
    case TYPE_VOID_PTR:
        printf("%p (void*)", a->value.void_pointer);
        break;
    }

}

void object_print_to_debug_core(const struct object* object, int n)
{

    if (object_is_reference(object))
    {
        object = object_get_referenced(object);
    }


    for (int i = 0; i < n; i++) printf("  ");
    if (object->debug_name)
        printf("%s ", object->debug_name);

    if (object->members != NULL)
    {

        type_print(&object->type);

        printf(" {\n");

        struct object* _Opt member = object->members;
        while (member)
        {
            object_print_to_debug_core(member, n + 1);
            member = member->next;
        }

        for (int i = 0; i < n; i++) printf("  ");
        printf("}\n");
    }
    else
    {



        type_print(&object->type);


        printf(" = ");

        object_print_value_debug(object);

        switch (object->state)
        {
        case CONSTANT_VALUE_STATE_UNINITIALIZED: printf(" uninitialized "); break;
        case CONSTANT_VALUE_STATE_ANY:printf(" unknown "); break;
        case CONSTANT_VALUE_EQUAL:printf(" exact "); break;
        case CONSTANT_VALUE_STATE_CONSTANT:printf(" constant_exact "); break;

        }

        printf("\n");
    }

}

void object_print_to_debug(const struct object* object)
{
    int n = 0;
    object_print_to_debug_core(object, n);
}


/*
   extends the array to the max_index returning the added item.
*/
struct object* object_extend_array_to_index(const struct type* p_type, struct object* a, size_t max_index, bool is_constant)
{
    struct object* _Opt it = a->members;

    try
    {
        int count = 0;
        while (it)
        {
            count++;
            if (it->next == NULL)
                break;
            it = it->next;
        }

        while (count < (max_index + 1))
        {
            if (it == NULL)
            {
                assert(a->members == NULL);
                a->members = make_object_ptr(p_type);
                if (a->members == NULL)
                    throw;

                char name[100] = { 0 };
                snprintf(name, sizeof name, "[%d]", count);

                free((void* _Owner)a->members->debug_name);
                a->members->debug_name = strdup(name);

                object_default_initialization(a->members, is_constant);

                it = a->members;
                it->parent = a;
                count++;
            }
            else
            {
                struct object* _Owner _Opt p = make_object_ptr(p_type);
                if (p == NULL)
                    throw;
                char name[100] = { 0 };
                snprintf(name, sizeof name, "[%d]", count);

                free((void* _Owner)p->debug_name);
                p->debug_name = strdup(name);


                p->parent = a;
                object_default_initialization(p, is_constant);

                assert(it->next == NULL);
                it->next = p;

                it = p;
                count++;
            }
        }
    }
    catch
    {
    }
    return it;
}


bool object_is_promoted(const struct object* a)
{
    /*
      types smaller than int are promoted to int
    */
    if ((a->value_type == TYPE_BOOL) ||
        (a->value_type == TYPE_SIGNED_CHAR) ||
        (a->value_type == TYPE_UNSIGNED_CHAR) ||
        (a->value_type == TYPE_SIGNED_SHORT) ||
        a->value_type == TYPE_UNSIGNED_SHORT)
    {
        return true;
    }
    return false;
}

enum object_value_type object_common(const struct object* a, const struct object* b)
{

    enum object_value_type a_type = a->value_type;
    enum object_value_type b_type = b->value_type;

    //See 6.3.1.8 Usual arithmetic conversions


    /*
       First, if the type of either operand is _Decimal128,
       the other operand is converted to _Decimal128.
    */

    /*
      Otherwise, if the type of either operand is _Decimal64,
      the other operand is converted to _Decimal64
    */

    /*
      Otherwise, if the type of either operand is _Decimal32,
      the other operand is converted to _Decimal32.
    */

    /*
      Otherwise, if the corresponding real type of either operand is long double,
      the other operand is converted, without change of type domain, to a type whose
      corresponding real type is long double
    */
    if (a_type == TYPE_LONG_DOUBLE || b_type == TYPE_LONG_DOUBLE)
    {
        return TYPE_LONG_DOUBLE;
    }

    /*
      Otherwise, if the corresponding real type of either operand is double,
      the other operand is converted, without change of type domain, to a type
      whose corresponding real type is double.
    */

    if (a_type == TYPE_DOUBLE || b_type == TYPE_DOUBLE)
    {
        return TYPE_LONG_DOUBLE;
    }

    /*
      Otherwise, if the corresponding real type of either operand is float,
      the other operand is converted, without change of type domain,
      to a type whose corresponding real type is float
    */
    if (a_type == TYPE_FLOAT || b_type == TYPE_FLOAT)
    {
        return TYPE_FLOAT;
    }


    /*
     Otherwise, if any of the two types is an enumeration, it is converted to its underlying type.
    */

    /*
      Then, the integer promotions are performed on both operands.
    */


    if (object_is_promoted(a))
    {
        a_type = TYPE_SIGNED_INT;
    }

    if (object_is_promoted(b))
    {
        b_type = TYPE_SIGNED_INT;
    }


    /*
      Next, the following rules are applied to the promoted operands
      if both operands have the same type, then no further conversion is needed
    */
    if (a_type == b_type)
    {
        return a_type;
    }

    /*
     Otherwise, if both operands have signed integer types or both have unsigned integer
     types, the operand with the type of lesser integer conversion rank is converted to the type
     of the operand with greater rank.
    */

    if (is_signed(a_type) == is_signed(b_type))
    {
        if (get_rank(a_type) > get_rank(b_type))
        {
            return a_type;
        }

        return b_type;
    }


    /*
     Otherwise, if the operand that has unsigned integer type has rank greater or equal to
     the rank of the type of the other operand, then the operand with signed integer type is
     converted to the type of the operand with unsigned integer type.
    */


    enum object_value_type  signed_promoted = is_signed(a_type) ? a_type : b_type;
    enum object_value_type  unsigned_promoted = is_unsigned(a_type) ? a_type : b_type;


    if (get_rank(unsigned_promoted) >= get_rank(signed_promoted))
    {
        return unsigned_promoted;
    }

    /*
      Otherwise, if the type of the operand with signed integer type can represent all the values
      of the type of the operand with unsigned integer type, then the operand with unsigned
      integer type is converted to the type of the operand with signed integer type
    */

    if (get_size(signed_promoted) > get_size(unsigned_promoted))
    {
        return signed_promoted;
    }

    /*
      Otherwise, both operands are converted to the unsigned integer type corresponding to
      the type of the operand with signed integer type
    */

    return to_unsigned(signed_promoted);

}

int object_greater_than_or_equal(const struct object* a, const struct object* b)
{
    a = object_get_referenced(a);
    b = object_get_referenced(b);

    enum object_value_type common_type = object_common(a, b);

    switch (common_type)
    {
    case TYPE_SIGNED_INT:
        return object_to_signed_int(a) >= object_to_signed_int(b);

    case TYPE_UNSIGNED_INT:
        return object_to_unsigned_int(a) >= object_to_unsigned_int(b);

    case TYPE_BOOL:
        return object_to_bool(a) >= object_to_bool(b);

    case TYPE_SIGNED_CHAR:
        return object_to_signed_char(a) >= object_to_signed_char(b);

        break;
    case TYPE_UNSIGNED_CHAR:
        return object_to_unsigned_char(a) >= object_to_unsigned_char(b);

    case TYPE_SIGNED_SHORT:
        return object_to_signed_short(a) >= object_to_signed_short(b);

    case TYPE_UNSIGNED_SHORT:
        return object_to_unsigned_short(a) >= object_to_unsigned_short(b);

    case TYPE_SIGNED_LONG:
        return object_to_signed_long(a) >= object_to_signed_long(b);

    case TYPE_UNSIGNED_LONG:
        return object_to_unsigned_long(a) >= object_to_unsigned_long(b);

    case TYPE_SIGNED_LONG_LONG:
        return object_to_signed_long_long(a) >= object_to_signed_long_long(b);

    case TYPE_UNSIGNED_LONG_LONG:
        return object_to_unsigned_long_long(a) >= object_to_unsigned_long_long(b);

    case TYPE_FLOAT:
        return object_to_float(a) >= object_to_float(b);

    case TYPE_DOUBLE:
        return object_to_double(a) >= object_to_double(b);

    case TYPE_LONG_DOUBLE:
        return object_to_long_double(a) >= object_to_long_double(b);

    }

    assert(false);
    return object_to_unsigned_long_long(a) >= object_to_unsigned_long_long(b);

}

int object_smaller_than_or_equal(const struct object* a, const struct object* b)
{
    a = object_get_referenced(a);
    b = object_get_referenced(b);

    enum object_value_type common_type = object_common(a, b);

    switch (common_type)
    {
    case TYPE_SIGNED_INT:
        return object_to_signed_int(a) <= object_to_signed_int(b);

    case TYPE_UNSIGNED_INT:
        return object_to_unsigned_int(a) <= object_to_unsigned_int(b);

    case TYPE_BOOL:
        return object_to_bool(a) <= object_to_bool(b);

    case TYPE_SIGNED_CHAR:
        return object_to_signed_char(a) <= object_to_signed_char(b);

        break;
    case TYPE_UNSIGNED_CHAR:
        return object_to_unsigned_char(a) <= object_to_unsigned_char(b);

    case TYPE_SIGNED_SHORT:
        return object_to_signed_short(a) <= object_to_signed_short(b);

    case TYPE_UNSIGNED_SHORT:
        return object_to_unsigned_short(a) <= object_to_unsigned_short(b);

    case TYPE_SIGNED_LONG:
        return object_to_signed_long(a) <= object_to_signed_long(b);

    case TYPE_UNSIGNED_LONG:
        return object_to_unsigned_long(a) <= object_to_unsigned_long(b);

    case TYPE_SIGNED_LONG_LONG:
        return object_to_signed_long_long(a) <= object_to_signed_long_long(b);

    case TYPE_UNSIGNED_LONG_LONG:
        return object_to_unsigned_long_long(a) <= object_to_unsigned_long_long(b);

    case TYPE_FLOAT:
        return object_to_float(a) <= object_to_float(b);

    case TYPE_DOUBLE:
        return object_to_double(a) <= object_to_double(b);

    case TYPE_LONG_DOUBLE:
        return object_to_long_double(a) <= object_to_long_double(b);

    }

    assert(false);
    return object_to_unsigned_long_long(a) <= object_to_unsigned_long_long(b);
}

struct object object_add(const struct object* a, const struct object* b)
{
    a = object_get_referenced(a);
    b = object_get_referenced(b);

    enum object_value_type common_type = object_common(a, b);

    switch (common_type)
    {
    case TYPE_SIGNED_INT:
        return object_make_signed_int(object_to_signed_int(a) + object_to_signed_int(b));

    case TYPE_UNSIGNED_INT:
        return object_make_unsigned_int(object_to_unsigned_int(a) + object_to_unsigned_int(b));

    case TYPE_BOOL:
        return object_make_bool(object_to_bool(a) + object_to_bool(b));

        //case TYPE_SIGNED_CHAR:
          //  return object_make_signed_char(object_to_signed_char(a) == object_to_signed_char(b);

          //  break;
        //case TYPE_UNSIGNED_CHAR:
    //        return object_to_unsigned_char(a) == object_to_unsigned_char(b);

        //case TYPE_SIGNED_SHORT:
          //  return object_to_signed_short(a) == object_to_signed_short(b);

        //case TYPE_UNSIGNED_SHORT:
          //  return object_to_unsigned_short(a) == object_to_unsigned_short(b);

    case TYPE_SIGNED_LONG:
        return object_make_signed_long(object_to_signed_long(a) + object_to_signed_long(b));

    case TYPE_UNSIGNED_LONG:
        return object_make_unsigned_long(object_to_unsigned_long(a) + object_to_unsigned_long(b));

    case TYPE_SIGNED_LONG_LONG:
        return object_make_signed_long_long(object_to_signed_long_long(a) + object_to_signed_long_long(b));

    case TYPE_UNSIGNED_LONG_LONG:
        return object_make_unsigned_long_long(object_to_unsigned_long_long(a) + object_to_unsigned_long_long(b));

    case TYPE_FLOAT:
        return object_make_float(object_to_float(a) + object_to_float(b));

    case TYPE_DOUBLE:
        return object_make_double(object_to_double(a) + object_to_double(b));

    case TYPE_LONG_DOUBLE:
        return object_make_long_double(object_to_long_double(a) + object_to_long_double(b));

    }

    assert(false);
    struct object o = { 0 };
    return o;
}


struct object object_sub(const struct object* a, const struct object* b)
{
    a = object_get_referenced(a);
    b = object_get_referenced(b);

    enum object_value_type common_type = object_common(a, b);

    switch (common_type)
    {
    case TYPE_SIGNED_INT:
        return object_make_signed_int(object_to_signed_int(a) - object_to_signed_int(b));

    case TYPE_UNSIGNED_INT:
        return object_make_unsigned_int(object_to_unsigned_int(a) - object_to_unsigned_int(b));

    case TYPE_BOOL:
        return object_make_bool(object_to_bool(a) - object_to_bool(b));

        //case TYPE_SIGNED_CHAR:
          //  return object_make_signed_char(object_to_signed_char(a) == object_to_signed_char(b);

          //  break;
        //case TYPE_UNSIGNED_CHAR:
    //        return object_to_unsigned_char(a) == object_to_unsigned_char(b);

        //case TYPE_SIGNED_SHORT:
          //  return object_to_signed_short(a) == object_to_signed_short(b);

        //case TYPE_UNSIGNED_SHORT:
          //  return object_to_unsigned_short(a) == object_to_unsigned_short(b);

    case TYPE_SIGNED_LONG:
        return object_make_signed_long(object_to_signed_long(a) - object_to_signed_long(b));

    case TYPE_UNSIGNED_LONG:
        return object_make_unsigned_long(object_to_unsigned_long(a) - object_to_unsigned_long(b));

    case TYPE_SIGNED_LONG_LONG:
        return object_make_signed_long_long(object_to_signed_long_long(a) - object_to_signed_long_long(b));

    case TYPE_UNSIGNED_LONG_LONG:
        return object_make_unsigned_long_long(object_to_unsigned_long_long(a) - object_to_unsigned_long_long(b));

    case TYPE_FLOAT:
        return object_make_float(object_to_float(a) - object_to_float(b));

    case TYPE_DOUBLE:
        return object_make_double(object_to_double(a) - object_to_double(b));

    case TYPE_LONG_DOUBLE:
        return object_make_long_double(object_to_long_double(a) - object_to_long_double(b));

    }

    assert(false);
    struct object o = { 0 };
    return o;
}


int object_equal(const struct object* a, const struct object* b)
{
    a = object_get_referenced(a);
    b = object_get_referenced(b);

    enum object_value_type common_type = object_common(a, b);

    switch (common_type)
    {
    case TYPE_SIGNED_INT:
        return object_to_signed_int(a) == object_to_signed_int(b);

    case TYPE_UNSIGNED_INT:
        return object_to_unsigned_int(a) == object_to_unsigned_int(b);

    case TYPE_BOOL:
        return object_to_bool(a) == object_to_bool(b);

    case TYPE_SIGNED_CHAR:
        return object_to_signed_char(a) == object_to_signed_char(b);

        break;
    case TYPE_UNSIGNED_CHAR:
        return object_to_unsigned_char(a) == object_to_unsigned_char(b);

    case TYPE_SIGNED_SHORT:
        return object_to_signed_short(a) == object_to_signed_short(b);

    case TYPE_UNSIGNED_SHORT:
        return object_to_unsigned_short(a) == object_to_unsigned_short(b);

    case TYPE_SIGNED_LONG:
        return object_to_signed_long(a) == object_to_signed_long(b);

    case TYPE_UNSIGNED_LONG:
        return object_to_unsigned_long(a) == object_to_unsigned_long(b);

    case TYPE_SIGNED_LONG_LONG:
        return object_to_signed_long_long(a) == object_to_signed_long_long(b);

    case TYPE_UNSIGNED_LONG_LONG:
        return object_to_unsigned_long_long(a) == object_to_unsigned_long_long(b);

    case TYPE_FLOAT:
        return object_to_float(a) == object_to_float(b);

    case TYPE_DOUBLE:
        return object_to_double(a) == object_to_double(b);

    case TYPE_LONG_DOUBLE:
        return object_to_long_double(a) == object_to_long_double(b);

    }

    assert(false);
    return object_to_unsigned_long_long(a) == object_to_unsigned_long_long(b);
}


int object_not_equal(const struct object* a, const struct object* b)
{
    a = object_get_referenced(a);
    b = object_get_referenced(b);

    enum object_value_type common_type = object_common(a, b);

    switch (common_type)
    {
    case TYPE_SIGNED_INT:
        return object_to_signed_int(a) != object_to_signed_int(b);

    case TYPE_UNSIGNED_INT:
        return object_to_unsigned_int(a) != object_to_unsigned_int(b);

    case TYPE_BOOL:
        return object_to_bool(a) != object_to_bool(b);

    case TYPE_SIGNED_CHAR:
        return object_to_signed_char(a) != object_to_signed_char(b);

        break;
    case TYPE_UNSIGNED_CHAR:
        return object_to_unsigned_char(a) != object_to_unsigned_char(b);

    case TYPE_SIGNED_SHORT:
        return object_to_signed_short(a) != object_to_signed_short(b);

    case TYPE_UNSIGNED_SHORT:
        return object_to_unsigned_short(a) != object_to_unsigned_short(b);

    case TYPE_SIGNED_LONG:
        return object_to_signed_long(a) != object_to_signed_long(b);

    case TYPE_UNSIGNED_LONG:
        return object_to_unsigned_long(a) != object_to_unsigned_long(b);

    case TYPE_SIGNED_LONG_LONG:
        return object_to_signed_long_long(a) != object_to_signed_long_long(b);

    case TYPE_UNSIGNED_LONG_LONG:
        return object_to_unsigned_long_long(a) != object_to_unsigned_long_long(b);

    case TYPE_FLOAT:
        return object_to_float(a) != object_to_float(b);

    case TYPE_DOUBLE:
        return object_to_double(a) != object_to_double(b);

    case TYPE_LONG_DOUBLE:
        return object_to_long_double(a) != object_to_long_double(b);

    }

    assert(false);
    return object_to_unsigned_long_long(a) != object_to_unsigned_long_long(b);
}


#define OBJECTS_INITIAL_CAPACITY 8

void objects_destroy(struct objects* arr)
{
    free(arr->items);
}

int objects_push(struct objects* arr, struct object* obj)
{
    if (arr->items == NULL)
    {
        arr->items = malloc(OBJECTS_INITIAL_CAPACITY * sizeof(struct object*));
        if (!arr->items)
        {
            arr->size = 0;
            arr->capacity = 0;
            return ENOMEM;
        }
        arr->size = 0;
        arr->capacity = OBJECTS_INITIAL_CAPACITY;
    }
    if (arr->size == arr->capacity)
    {
        size_t new_capacity = arr->capacity ? arr->capacity * 2 : OBJECTS_INITIAL_CAPACITY;
        struct object** new_items = realloc(arr->items, new_capacity * sizeof(struct object*));
        if (!new_items) return ENOMEM;
        arr->items = new_items;
        arr->capacity = new_capacity;
    }
    arr->items[arr->size++] = obj;
    return 0;
}




/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/


#pragma safety enable



#include <math.h>

#ifdef _WIN32
#endif

#if defined _MSC_VER && !defined __POCC__
#endif

struct expression* _Owner _Opt postfix_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt cast_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt multiplicative_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt unary_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt additive_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt shift_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt relational_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt equality_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt and_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt exclusive_or_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt inclusive_or_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt logical_and_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt logical_or_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt conditional_expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt expression(struct parser_ctx* ctx);
struct expression* _Owner _Opt conditional_expression(struct parser_ctx* ctx);

NODISCARD
static errno_t execute_bitwise_operator(struct parser_ctx* ctx, struct expression* new_expression, int op);

static int compare_function_arguments(struct parser_ctx* ctx,
                                      struct type* p_type,
                                      struct argument_expression_list* p_argument_expression_list)
{
    try
    {
        struct param* _Opt p_current_parameter_type = NULL;

        const struct param_list* _Opt p_param_list = type_get_func_or_func_ptr_params(p_type);
        if (p_param_list == NULL) throw;

        p_current_parameter_type = p_param_list->head;

        struct argument_expression* _Opt p_current_argument = p_argument_expression_list->head;

        if (p_current_parameter_type && type_is_void(&p_current_parameter_type->type))
        {
            //(void) function
            p_current_parameter_type = NULL;
        }

        while (p_current_argument && p_current_parameter_type)
        {
            check_assigment(ctx, &p_current_parameter_type->type, p_current_argument->expression, ASSIGMENT_TYPE_PARAMETER);
            p_current_argument = p_current_argument->next;
            p_current_parameter_type = p_current_parameter_type->next;
        }

        if (p_current_argument != NULL && !p_param_list->is_var_args)
        {
            compiler_diagnostic(C_ERROR_TOO_MANY_ARGUMENTS, ctx,
                                        p_current_argument->expression->first_token, NULL,
                                        "too many arguments");
            throw;
        }

        if (p_current_parameter_type != NULL && !p_param_list->is_void)
        {
            if (p_argument_expression_list->tail)
            {
                compiler_diagnostic(C_ERROR_TOO_FEW_ARGUMENTS, ctx,
                                            p_argument_expression_list->tail->expression->first_token,
                    NULL,
                                            "too few arguments");
            }
            else
            {
                compiler_diagnostic(C_ERROR_TOO_FEW_ARGUMENTS, ctx, ctx->current, NULL, "too few arguments");
            }
            throw;
        }
    }
    catch
    {
        return 1; /*error*/
    }

    return 0;
}

bool is_enumeration_constant(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    if (ctx->current->type != TK_IDENTIFIER)
    {
        return false;
    }

    if (ctx->current->flags & TK_FLAG_IDENTIFIER_IS_ENUMERATOR)
        return true;

    if (ctx->current->flags & TK_FLAG_IDENTIFIER_IS_NOT_ENUMERATOR)
        return false;

    const bool is_enumerator = find_enumerator(ctx, ctx->current->lexeme, NULL) != NULL;

    if (is_enumerator)
    {
        ctx->current->flags |= TK_FLAG_IDENTIFIER_IS_ENUMERATOR;
    }
    else
    {
        ctx->current->flags |= TK_FLAG_IDENTIFIER_IS_NOT_ENUMERATOR;
    }

    return is_enumerator;
}

bool is_first_of_floating_constant(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    /*
     floating-constant:
      decimal-floating-constant
      hexadecimal-floating-constan
    */
    return ctx->current->type == TK_COMPILER_DECIMAL_FLOATING_CONSTANT ||
        ctx->current->type == TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT;
}

bool is_first_of_integer_constant(const struct parser_ctx* ctx)
{
    /*
     integer-constant:
      decimal-constant integer-suffixopt
      octal-constant integer-suffixopt
      hexadecimal-constant integer-suffixopt
      binary-constant integer-suffixop
    */

    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_COMPILER_DECIMAL_CONSTANT ||
        ctx->current->type == TK_COMPILER_OCTAL_CONSTANT ||
        ctx->current->type == TK_COMPILER_HEXADECIMAL_CONSTANT ||
        ctx->current->type == TK_COMPILER_BINARY_CONSTANT;
}

bool is_predefined_constant(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_KEYWORD_TRUE ||
        ctx->current->type == TK_KEYWORD_FALSE ||
        ctx->current->type == TK_KEYWORD_NULLPTR;
}

bool is_first_of_constant(const struct parser_ctx* ctx)
{
    /*
     constant:
      integer-constant
      floating-constant
      enumeration-constant
      character-constant
      predefined-constant
    */
    if (ctx->current == NULL)
        return false;

    return is_first_of_integer_constant(ctx) ||
        is_first_of_floating_constant(ctx) ||
        is_enumeration_constant(ctx) ||
        (ctx->current->type == TK_CHAR_CONSTANT) ||
        is_predefined_constant(ctx);
}

bool is_first_of_primary_expression(const struct parser_ctx* ctx)
{
    /*
     primary-expression:
      identifier
      constant
      string-literal
      ( expression )
      generic-selection
      typeid (expression )
    */
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_IDENTIFIER ||
        is_first_of_constant(ctx) ||
        ctx->current->type == TK_STRING_LITERAL ||
        ctx->current->type == '(' ||
        ctx->current->type == TK_KEYWORD__GENERIC;
}

struct generic_association* _Owner _Opt generic_association(struct parser_ctx* ctx)
{
    struct generic_association* _Owner _Opt p_generic_association = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        p_generic_association = calloc(1, sizeof * p_generic_association);
        if (p_generic_association == NULL)
            throw;

        p_generic_association->first_token = ctx->current;
        /*generic - association:
            type-name : assignment-expression
            default : assignment-expression
            */
        if (ctx->current->type == TK_KEYWORD_DEFAULT)
        {
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }
        }
        else if (first_of_type_name(ctx))
        {
            bool old = ctx->inside_generic_association;
            ctx->inside_generic_association = true;


            p_generic_association->p_type_name = type_name(ctx);
            if (p_generic_association->p_type_name == NULL) throw;

            assert(p_generic_association->p_type_name->abstract_declarator != NULL);

            ctx->inside_generic_association = old;
            p_generic_association->type = make_type_using_declarator(ctx, p_generic_association->p_type_name->abstract_declarator);
        }
        else
        {
            compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "unexpected");
        }
        if (parser_match_tk(ctx, ':') != 0)
            throw;

        struct expression* _Owner _Opt p_expression_temp = assignment_expression(ctx);
        if (p_expression_temp == NULL)
        {
            throw;
        }

        p_generic_association->expression = p_expression_temp;

        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }
        p_generic_association->last_token = ctx->current;
    }
    catch
    {
        generic_association_delete(p_generic_association);
        p_generic_association = NULL;
    }

    return p_generic_association;
}

struct generic_assoc_list generic_association_list(struct parser_ctx* ctx)
{
    struct generic_assoc_list list = { 0 };
    try
    {
        struct generic_association* _Opt p_default_generic_association = NULL;

        struct generic_association* _Owner _Opt p_generic_association =
            generic_association(ctx);

        if (p_generic_association == NULL)
            throw;

        if (p_generic_association->first_token->type == TK_KEYWORD_DEFAULT)
        {
            p_default_generic_association = p_generic_association;
        }

        generic_assoc_list_add(&list, p_generic_association);

        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        while (ctx->current->type == ',')
        {
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            struct generic_association* _Owner _Opt p_generic_association2 = generic_association(ctx);
            if (p_generic_association2 == NULL)
                throw;

            if (p_generic_association2->first_token->type == TK_KEYWORD_DEFAULT)
            {
                if (p_default_generic_association != NULL)
                {
                    compiler_diagnostic(C_ERROR_DUPLICATE_DEFAULT_GENERIC_ASSOCIATION,
                        ctx,
                        p_generic_association2->first_token,
                        NULL,
                        "duplicate default generic association.");

                    compiler_diagnostic(W_NOTE,
                        ctx,
                        p_default_generic_association->first_token,
                        NULL,
                        "previous default generic association");
                }
                else
                {
                    p_default_generic_association = p_generic_association2;
                }
            }

            generic_assoc_list_add(&list, p_generic_association2);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }
        }
    }
    catch
    {
    }
    return list;
}
void generic_association_delete(struct generic_association* _Owner _Opt p)
{
    if (p)
    {
        assert(p->next == NULL);
        type_name_delete(p->p_type_name);
        expression_delete(p->expression);
        type_destroy(&p->type);
        free(p);
    }
}

void generic_assoc_list_add(struct generic_assoc_list* list, struct generic_association* _Owner pitem)
{
    if (list->head == NULL)
    {
        list->head = pitem;
    }
    else
    {
        assert(list->tail != NULL);
        assert(list->tail->next == NULL);
        list->tail->next = pitem;
    }
    list->tail = pitem;
}

void generic_assoc_list_destroy(_Dtor struct generic_assoc_list* p)
{
    struct generic_association* _Owner _Opt item = p->head;
    while (item)
    {
        struct generic_association* _Owner _Opt next = item->next;
        item->next = NULL;
        generic_association_delete(item);
        item = next;
    }
}
void generic_selection_delete(struct generic_selection* _Owner _Opt p)
{
    if (p)
    {
        expression_delete(p->expression);
        type_name_delete(p->type_name);
        generic_assoc_list_destroy(&p->generic_assoc_list);
        free(p);
    }
}
struct generic_selection* _Owner _Opt generic_selection(struct parser_ctx* ctx)
{
    /*C23
      generic-selection:
        _Generic ( assignment-expression , generic-assoc-ctx )
    */

    /*
      Extension
      generic-selection:
        "_Generic" ( generic-argument, generic-assoc-list )

        generic-argument:
          assignment-expression
          type-name
    */

    struct generic_selection* _Owner _Opt p_generic_selection = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        p_generic_selection = calloc(1, sizeof * p_generic_selection);
        if (p_generic_selection == NULL)
            throw;

        p_generic_selection->first_token = ctx->current;

        if (parser_match_tk(ctx, TK_KEYWORD__GENERIC) != 0)
            throw;
        if (parser_match_tk(ctx, '(') != 0)
            throw;

        if (first_of_type_name(ctx))
        {
            /*extension*/
            p_generic_selection->type_name = type_name(ctx);
            if (p_generic_selection->type_name == NULL)
                throw;
        }
        else
        {
            p_generic_selection->expression = assignment_expression(ctx);
            if (p_generic_selection->expression == NULL)
                throw;
        }

        if (parser_match_tk(ctx, ',') != 0)
            throw;

        p_generic_selection->generic_assoc_list = generic_association_list(ctx);
        if (p_generic_selection->generic_assoc_list.head == NULL) throw;

        struct type lvalue_type = { 0 };

        struct type* _Opt p_type = NULL;

        if (p_generic_selection->expression)
        {
            p_type = &p_generic_selection->expression->type;

            if (expression_is_subjected_to_lvalue_conversion(p_generic_selection->expression))
            {
                lvalue_type = type_lvalue_conversion(&p_generic_selection->expression->type, ctx->options.null_checks_enabled);
                p_type = &lvalue_type;
            }
        }
        else if (p_generic_selection->type_name)
        {
            p_type = &p_generic_selection->type_name->abstract_declarator->type;
        }
        else
        {
            throw;
        }

        struct generic_association* _Opt current = p_generic_selection->generic_assoc_list.head;
        while (current)
        {
            if (current->p_type_name)
            {
                if (type_is_same(p_type, &current->type, true))
                {
                    p_generic_selection->p_view_selected_expression = current->expression;
                    break;
                }
            }
            else
            {
                /*default*/
                p_generic_selection->p_view_selected_expression = current->expression;
            }
            current = current->next;
        }

        type_destroy(&lvalue_type);

        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }
        p_generic_selection->last_token = ctx->current;

        if (parser_match_tk(ctx, ')') != 0)
        {
            throw;
        }
    }
    catch
    {
        generic_selection_delete(p_generic_selection);
        p_generic_selection = NULL;
    }
    return p_generic_selection;
}



struct expression* _Owner _Opt character_constant_expression(struct parser_ctx* ctx)
{
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        p_expression_node = calloc(1, sizeof * p_expression_node);
        if (p_expression_node == NULL)
            throw;

        p_expression_node->expression_type = PRIMARY_EXPRESSION_CHAR_LITERAL;
        p_expression_node->first_token = ctx->current;
        p_expression_node->last_token = p_expression_node->first_token;
        p_expression_node->type.attributes_flags |= CAKE_HIDDEN_ATTRIBUTE_INT_LIKE_CHAR;
        p_expression_node->type.category = TYPE_CATEGORY_ITSELF;

        const unsigned char* _Opt p = (const unsigned char*)ctx->current->lexeme;

        if (p[0] == 'u' && p[1] == '8')
        {
            p++;
            p++;
            p++;

            // A UTF-8 character constant has type char8_t.
            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_CHAR;

            unsigned int c = 0;
            p = utf8_decode(p, &c);
            if (p == NULL)
            {
                throw;
            }

            if (c == '\\')
            {
                p = escape_sequences_decode_opt(p, &c);
                if (p == NULL) throw;
            }

            if (*p != '\'')
            {
                compiler_diagnostic(C_MULTICHAR_ERROR, ctx, ctx->current, NULL, "Unicode character literals may not contain multiple characters.");
            }

            if (c > 0x80)
            {
                compiler_diagnostic(C_CHARACTER_NOT_ENCODABLE_IN_A_SINGLE_CODE_UNIT, ctx, ctx->current, NULL, "character not encodable in a single code unit.");
            }

            p_expression_node->object = object_make_unsigned_char((unsigned char)c);//, ctx->evaluation_is_disabled);
        }
        else if (p[0] == 'u')
        {
            p++;
            p++;

            // A UTF-16 character constant has type char16_t which is an unsigned integer types defined in the <uchar.h> header
            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_SHORT;

            unsigned int c = 0;
            p = utf8_decode(p, &c);
            if (p == NULL)
            {
                throw;
            }

            if (c == '\\')
            {
                p = escape_sequences_decode_opt(p, &c);
                if (p == NULL) throw;
            }

            if (*p != '\'')
            {
                compiler_diagnostic(W_MULTICHAR_ERROR, ctx, ctx->current, NULL, "Unicode character literals may not contain multiple characters.");
            }

            if (c > USHRT_MAX)
            {
                compiler_diagnostic(W_MULTICHAR_ERROR, ctx, ctx->current, NULL, "Character too large for enclosing character literal type.");
            }

            p_expression_node->object = object_make_wchar_t((wchar_t)c);
        }
        else if (p[0] == 'U')
        {
            p++;
            p++;

            // A UTF-16 character constant has type char16_t which is an unsigned integer types defined in the <uchar.h> header
            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT;

            unsigned int c = 0;
            p = utf8_decode(p, &c);
            if (p == NULL)
            {
                throw;
            }

            if (c == '\\')
            {
                p = escape_sequences_decode_opt(p, &c);
                if (p == NULL) throw;
            }

            if (*p != '\'')
            {
                compiler_diagnostic(W_MULTICHAR_ERROR, ctx, ctx->current, NULL, "Unicode character literals may not contain multiple characters.");
            }

            if (c > UINT_MAX)
            {
                compiler_diagnostic(W_MULTICHAR_ERROR, ctx, ctx->current, NULL, "Character too large for enclosing character literal type.");
            }

            p_expression_node->object = object_make_wchar_t((wchar_t)c);
        }
        else if (p[0] == 'L')
        {
            // A wchar_t character constant is prefixed by the letter L
            p++;
            p++;

            p_expression_node->type.type_specifier_flags = CAKE_WCHAR_T_TYPE_SPECIFIER;

            /*
             wchar_t character constant prefixed by the letter L has type wchar_t, an integer type defined in
             the <stddef.h> header. The value of a wchar_t character constant containing a single multibyte
             character that maps to a single member of the extended execution character set is the wide character
             corresponding to that multibyte character in the implementation-defined wide literal encoding
             (6.2.9). The value of a wchar_t character constant containing more than one multibyte character or a
             single multibyte character that maps to multiple members of the extended execution character set,
             or containing a multibyte character or escape sequence not represented in the extended execution
             character set, is implementation-defined.
            */
            long long value = 0;
            while (*p != '\'')
            {
                unsigned int c = 0;
                p = utf8_decode(p, &c);
                if (p == NULL)
                {
                    throw;
                }

                if (c == '\\')
                {
                    p = escape_sequences_decode_opt(p, &c);
                    if (p == NULL) throw;
                }

                if (c < 0x80)
                {
                    value = value * 256 + c;
                }
                else
                {
                    //decoded
                    value = c;
                }
#ifdef _WIN32
                if (value > USHRT_MAX)
                {
                    compiler_diagnostic(W_OUT_OF_BOUNDS, ctx, ctx->current, NULL, "character constant too long for its type", ctx->current->lexeme);
                    break;
                }
#else
                if (value > UINT_MAX)
                {
                    compiler_diagnostic(W_OUT_OF_BOUNDS, ctx, ctx->current, NULL, "character constant too long for its type", ctx->current->lexeme);
                    break;
                }
#endif
            }

            p_expression_node->object = object_make_wchar_t((wchar_t)value);
        }
        else
        {
            p++;
            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_INT;

            /*
              An integer character constant has type int. The value of an integer character constant containing
              a single character that maps to a single value in the literal encoding (6.2.9) is the numerical value
              of the representation of the mapped character in the literal encoding interpreted as an integer.
              The value of an integer character constant containing more than one character (e.g., ’ab’), or
              containing a character or escape sequence that does not map to a single value in the literal encoding,
              is implementation-defined. If an integer character constant contains a single character or escape
              sequence, its value is the one that results when an object with type char whose value is that of the
              single character or escape sequence is converted to type int.
            */
            long long value = 0;
            while (*p != '\'')
            {
                unsigned int c = 0;
                p = utf8_decode(p, &c);
                if (p == NULL)
                {
                    throw;
                }

                if (c == '\\')
                {
                    p = escape_sequences_decode_opt(p, &c);
                    if (p == NULL) throw;
                }

                value = value * 256 + c;
                if (value > INT_MAX)
                {
                    compiler_diagnostic(W_OUT_OF_BOUNDS, ctx, ctx->current, NULL, "character constant too long for its type", ctx->current->lexeme);
                    break;
                }
            }
            p_expression_node->object = object_make_signed_int((int)value);
        }

        parser_match(ctx);
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        // warning: character constant too long for its type
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }
    return p_expression_node;
}

int convert_to_number(struct parser_ctx* ctx, struct expression* p_expression_node, bool disabled)
{
    if (ctx->current == NULL)
    {
        unexpected_end_of_file(ctx);
        return 1;
    }

    struct token* token = ctx->current;

    /*copy removing separators*/
    // one of the largest buffers needed would be 128 bits binary... 
    // 0xb1'1'1....
    int c = 0;
    char buffer[128 * 2 + 4] = { 0 };
    const char* s = token->lexeme;
    while (*s)
    {
        if (*s != '\'')
        {
            buffer[c] = *s;
            c++;
        }
        s++;
    }

    char errormsg[100] = { 0 };
    char suffix[4] = { 0 };
    enum token_type r = parse_number(buffer, suffix, errormsg);
    if (r == TK_NONE)
    {
        compiler_diagnostic(
            C_INVALID_TOKEN,
            ctx,
            token,
            NULL,
            errormsg);
        return 0;
    }

    switch (token->type)
    {
    case TK_COMPILER_DECIMAL_CONSTANT:
    case TK_COMPILER_OCTAL_CONSTANT:
    case TK_COMPILER_HEXADECIMAL_CONSTANT:
    case TK_COMPILER_BINARY_CONSTANT:
    {
        unsigned long long value = 0;
        switch (token->type)
        {
        case TK_COMPILER_DECIMAL_CONSTANT:
            value = strtoull(buffer, NULL, 10);
            break;
        case TK_COMPILER_OCTAL_CONSTANT:
            if (buffer[1] == 'o' || buffer[1] == 'O')
            {
                //C2Y
                //https://www.open-std.org/jtc1/sc22/wg14/www/docs/n3319.htm
                value = strtoull(buffer + 2, NULL, 8);
            }
            else
            {
                value = strtoull(buffer + 1, NULL, 8);
            }
            break;
        case TK_COMPILER_HEXADECIMAL_CONSTANT:
            value = strtoull(buffer + 2, NULL, 16);
            break;
        case TK_COMPILER_BINARY_CONSTANT:
            value = strtoull(buffer + 2, NULL, 2);
            break;
        default:
            break;
        }

        if (value == ULLONG_MAX && errno == ERANGE)
        {
            compiler_diagnostic(
            C_ERROR_LITERAL_OVERFLOW,
            ctx,
            token,
            NULL,
            "integer literal is too large to be represented in any integer type");
        }

        ///////////////MICROSOFT ////////////////////////
        //TODO i64 etc
        ////////////////////////////////////////////////

        if (suffix[0] == 'U')
        {
            /*fixing the type that fits the size*/
            if (value <= UINT_MAX && suffix[1] != 'L')
            {
                p_expression_node->object = object_make_unsigned_int((unsigned int)value);
                p_expression_node->type.type_specifier_flags = (TYPE_SPECIFIER_INT | TYPE_SPECIFIER_UNSIGNED);
            }
            else if (value <= ULONG_MAX && suffix[2] != 'L')
            {
                p_expression_node->object = object_make_unsigned_long((unsigned long)value);
                p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_LONG | TYPE_SPECIFIER_UNSIGNED;
            }
            else //if (value <= ULLONG_MAX)
            {
                p_expression_node->object = object_make_unsigned_long_long((unsigned long long)value);
                p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_LONG_LONG | TYPE_SPECIFIER_UNSIGNED;
            }
        }
        else
        {
            /*fixing the type that fits the size*/
            if (value <= INT_MAX && suffix[0] != 'L')
            {
                p_expression_node->object = object_make_signed_int((int)value);
                p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_INT;
            }
            else if (value <= LONG_MAX && suffix[1] != 'L' /*!= LL*/)
            {
                p_expression_node->object = object_make_signed_long((long)value);
                p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_LONG;
            }
            else if (value <= LLONG_MAX)
            {
                p_expression_node->object = object_make_signed_long_long((long long)value);
                p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_LONG_LONG;
            }
            else
            {
                compiler_diagnostic(
                    W_IMPLICITLY_UNSIGNED_LITERAL,
                    ctx,
                    token,
                    NULL,
                    "integer literal is too large to be represented in a signed integer type, interpreting as unsigned");
                p_expression_node->object = object_make_signed_long_long(value);
                p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_LONG_LONG | TYPE_SPECIFIER_UNSIGNED;
            }
        }

    }
    break;

    case TK_COMPILER_DECIMAL_FLOATING_CONSTANT:
    case TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT:
    {
        if (suffix[0] == 'F')
        {
            float value = strtof(buffer, NULL);
            if (value == HUGE_VALF && errno == ERANGE)
            {
            }
            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_FLOAT;
            p_expression_node->object = object_make_float(value);
        }
        else if (suffix[0] == 'L')
        {
            long double value = strtold(buffer, NULL);
            if (value == HUGE_VALL && errno == ERANGE)
            {
            }

            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_DOUBLE | TYPE_SPECIFIER_LONG;
            p_expression_node->object = object_make_long_double(value);
        }
        else
        {
            double value = strtod(buffer, NULL);
            if (value == HUGE_VAL && errno == ERANGE)
            {
            }
            p_expression_node->object = object_make_double(value);
            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_DOUBLE;
        }
    }
    break;

    default:
        assert(false);
    }

    return 0;
}

static bool is_integer_or_floating_constant(enum token_type type)
{
    return type == TK_COMPILER_DECIMAL_CONSTANT ||
        type == TK_COMPILER_OCTAL_CONSTANT ||
        type == TK_COMPILER_HEXADECIMAL_CONSTANT ||
        type == TK_COMPILER_BINARY_CONSTANT ||
        type == TK_COMPILER_DECIMAL_FLOATING_CONSTANT ||
        type == TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT;
}

struct expression* _Owner _Opt primary_expression(struct parser_ctx* ctx)
{
    /*
     primary-expression:
      identifier
      constant
      string-literal
      ( expression )
      generic-selection
    */

    if (ctx->current == NULL)
    {
        unexpected_end_of_file(ctx);
        return NULL;
    }

    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        if (ctx->current->type == TK_IDENTIFIER)
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);

            if (p_expression_node == NULL)
                throw;

            p_expression_node->first_token = ctx->current;
            p_expression_node->last_token = ctx->current;

            struct scope* _Opt p_scope = NULL;
            struct map_entry* _Opt p_entry = find_variables(ctx, ctx->current->lexeme, &p_scope);

            if (p_entry && p_entry->type == TAG_TYPE_ENUMERATOR)
            {
                assert(p_entry->data.p_enumerator != NULL);
                struct enumerator* p_enumerator = p_entry->data.p_enumerator;
                p_expression_node->expression_type = PRIMARY_EXPRESSION_ENUMERATOR;
                p_expression_node->object = p_enumerator->value;

                p_expression_node->type = type_make_enumerator(p_enumerator->enum_specifier);
            }
            else if (p_entry &&
                     (p_entry->type == TAG_TYPE_DECLARATOR || p_entry->type == TAG_TYPE_INIT_DECLARATOR))
            {
                struct declarator* _Opt p_declarator = NULL;
                struct init_declarator* _Opt p_init_declarator = NULL;
                if (p_entry->type == TAG_TYPE_INIT_DECLARATOR)
                {
                    assert(p_entry->data.p_init_declarator != NULL);
                    p_init_declarator = p_entry->data.p_init_declarator;
                    p_declarator = p_init_declarator->p_declarator;
                }
                else
                {
                    p_declarator = p_entry->data.p_declarator;
                }

                assert(p_declarator != NULL);

                if (type_is_deprecated(&p_declarator->type))
                {
                    compiler_diagnostic(W_DEPRECATED, ctx, ctx->current, NULL, "'%s' is deprecated", ctx->current->lexeme);
                }


                if (p_scope->scope_level == 0)
                {
                    //file scope
                }
                else if ((p_declarator->type.storage_class_specifier_flags & STORAGE_SPECIFIER_STATIC) ||
                        (p_declarator->type.storage_class_specifier_flags & STORAGE_SPECIFIER_THREAD_LOCAL))
                {
                    //file scope or thread
                }
                else if (ctx->p_current_function_scope_opt)
                {
                    bool b_type_is_function = type_is_function(&p_declarator->type);
                    if (!ctx->evaluation_is_disabled && !b_type_is_function)
                    {
                        bool inside_current_function_scope = false;
                        while (p_scope)
                        {
                            if (ctx->p_current_function_scope_opt == p_scope)
                            {
                                inside_current_function_scope = true;
                                break;
                            }
                            p_scope = p_scope->previous;
                        }
                        if (!inside_current_function_scope)
                        {
                            compiler_diagnostic(C_ERROR_OUTER_SCOPE,
                                ctx,
                                ctx->current,
                                NULL,
                                "'%s' cannot be evaluated in this scope", ctx->current->lexeme);
                        }
                    }
                }

                p_declarator->num_uses++;
                p_expression_node->declarator = p_declarator;
                p_expression_node->p_init_declarator = p_init_declarator;

                p_expression_node->expression_type = PRIMARY_EXPRESSION_DECLARATOR;

                p_expression_node->type = type_dup(&p_declarator->type);
                p_expression_node->object = object_make_reference(&p_declarator->object);

            }
            else if (ctx->p_current_function_opt &&
                     strcmp(ctx->current->lexeme, "__func__") == 0)
            {

                const char* func_name = ctx->p_current_function_opt->name_opt ?
                    ctx->p_current_function_opt->name_opt->lexeme :
                    "unnamed";



                p_expression_node->expression_type = PRIMARY_EXPRESSION__FUNC__;
                p_expression_node->first_token = ctx->current;
                p_expression_node->last_token = ctx->current;

                p_expression_node->type = type_make_literal_string(strlen(func_name) + 1, TYPE_SPECIFIER_CHAR, TYPE_QUALIFIER_CONST);
            }
            else
            {
                compiler_diagnostic(C_ERROR_NOT_FOUND, ctx, ctx->current, NULL, "not found '%s'", ctx->current->lexeme);
                throw;
            }
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }
        }
        else if (ctx->current->type == TK_STRING_LITERAL)
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL)
                throw;

            p_expression_node->expression_type = PRIMARY_EXPRESSION_STRING_LITERAL;
            p_expression_node->first_token = ctx->current;
            p_expression_node->last_token = ctx->current;

            enum type_specifier_flags char_type = TYPE_SPECIFIER_CHAR;

            if (get_char_type(ctx->current->lexeme) == 2)
            {
                /*
                   automatically finding out the type of wchar_t to copy
                   GCC or MSVC.
                   windows it is short linux is
                */
                char_type = CAKE_WCHAR_T_TYPE_SPECIFIER;
            }
            /*
              string concatenation should have been done in a previous phase
              but since we keep the source format here it was an alternative
            */

            const int char_byte_size = string_literal_char_byte_size(ctx->current->lexeme);
            int number_of_bytes = 0;
            struct object* last = NULL;

            while (ctx->current->type == TK_STRING_LITERAL)
            {
                //"part1" "part2" TODO check different types

                const unsigned char* it = ctx->current->lexeme + 1;
                unsigned int value = 0;
                while (it && *it != '"')
                {
                    if (*it == '\\')
                        it = escape_sequences_decode_opt(it, &value);
                    else
                    {
                        value = *it;
                        it++;
                    }

                    struct object* p_new = calloc(1, sizeof * p_new);
                    if (p_new == NULL) throw;

                    p_new->state = CONSTANT_VALUE_STATE_CONSTANT;
                    p_new->value_type = TYPE_SIGNED_CHAR;
                    p_new->value.signed_char_value = value;

                    if (p_expression_node->object.members == NULL)
                    {
                        p_expression_node->object.members = p_new;
                    }
                    else
                    {
                        if (last)
                            last->next = p_new;
                    }
                    last = p_new;
                }

                struct object* p_new = calloc(1, sizeof * p_new);
                if (p_new == NULL) throw;

                p_new->state = CONSTANT_VALUE_STATE_CONSTANT;
                p_new->value_type = TYPE_SIGNED_CHAR;
                p_new->value.signed_char_value = 0;

                if (last == NULL)
                {
                    p_expression_node->object.members = p_new;
                }
                else
                {
                    last->next = p_new;
                }

                number_of_bytes += string_literal_byte_size_not_zero_included(ctx->current->lexeme);

                parser_match(ctx);
                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    throw;
                }
            }

            enum type_qualifier_flags lit_flags = ctx->options.const_literal ? TYPE_QUALIFIER_CONST : TYPE_QUALIFIER_NONE;
            p_expression_node->type = type_make_literal_string(number_of_bytes + (1 * char_byte_size), char_type, lit_flags);
        }
        else if (ctx->current->type == TK_CHAR_CONSTANT)
        {
            p_expression_node = character_constant_expression(ctx);
        }

        else if (ctx->current->type == TK_KEYWORD_TRUE ||
                 ctx->current->type == TK_KEYWORD_FALSE)
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL)
                throw;

            p_expression_node->expression_type = PRIMARY_EXPRESSION_PREDEFINED_CONSTANT;
            p_expression_node->first_token = ctx->current;
            p_expression_node->last_token = ctx->current;

            p_expression_node->object = object_make_bool(ctx->current->type == TK_KEYWORD_TRUE);

            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_BOOL;
            p_expression_node->type.type_qualifier_flags = 0;

            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }
        }
        else if (ctx->current->type == TK_KEYWORD_NULLPTR)
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL)
                throw;

            p_expression_node->expression_type = PRIMARY_EXPRESSION_PREDEFINED_CONSTANT;
            p_expression_node->first_token = ctx->current;
            p_expression_node->last_token = ctx->current;

            p_expression_node->object = object_make_nullptr();

            /*TODO nullptr type*/
            p_expression_node->type.type_specifier_flags = TYPE_SPECIFIER_NULLPTR_T;
            p_expression_node->type.type_qualifier_flags = 0;

            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }
        }
        else if (is_integer_or_floating_constant(ctx->current->type))
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL)
                throw;

            p_expression_node->first_token = ctx->current;
            p_expression_node->last_token = ctx->current;
            p_expression_node->expression_type = PRIMARY_EXPRESSION_NUMBER;

            convert_to_number(ctx, p_expression_node, false /*ctx->evaluation_is_disabled*/);

            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }
        }
        else if (ctx->current->type == TK_KEYWORD__GENERIC)
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL)
                throw;
            p_expression_node->first_token = ctx->current;

            p_expression_node->expression_type = PRIMARY_EXPRESSION_GENERIC;

            p_expression_node->generic_selection = generic_selection(ctx);
            if (p_expression_node->generic_selection == NULL)
                throw;

            p_expression_node->last_token = p_expression_node->generic_selection->last_token;

            if (p_expression_node->generic_selection->p_view_selected_expression)
            {
                p_expression_node->type = type_dup(&p_expression_node->generic_selection->p_view_selected_expression->type);

                p_expression_node->object = p_expression_node->generic_selection->p_view_selected_expression->object;
            }
            else
            {
                compiler_diagnostic(C_ERROR_NO_MATCH_FOR_GENERIC, ctx, ctx->current, NULL, "no match for generic");
            }
        }
        else if (ctx->current->type == '(')
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL) throw;

            p_expression_node->expression_type = PRIMARY_EXPRESSION_PARENTESIS;
            p_expression_node->first_token = ctx->current;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            p_expression_node->right = expression(ctx);
            if (p_expression_node->right == NULL)
                throw;

            p_expression_node->type = type_dup(&p_expression_node->right->type);
            p_expression_node->object = p_expression_node->right->object;

            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            p_expression_node->last_token = ctx->current;
            if (parser_match_tk(ctx, ')') != 0)
                throw;
        }
        else
        {
            compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "unexpected");
            throw;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    assert(p_expression_node == NULL || (p_expression_node->first_token && p_expression_node->last_token));

    return p_expression_node;
}

void argument_expression_delete(struct argument_expression* _Owner _Opt p)
{
    if (p)
    {
        expression_delete(p->expression);
        assert(p->next == NULL);
        free(p);
    }
}

struct argument_expression_list argument_expression_list(struct parser_ctx* ctx)
{
    /*
     argument-expression-list:
      assignment-expression
      argument-expression-ctx , assignment-expression
    */

    /*
     argument-expression-list: (extended)
      assignment-expression
      move assignment-expression
      argument-expression-ctx , assignment-expression
      argument-expression-ctx , assignment-expression
    */

    struct argument_expression_list list = { 0 };
    struct argument_expression* _Owner _Opt p_argument_expression = NULL;

    try
    {
        p_argument_expression = calloc(1, sizeof(struct argument_expression));
        if (p_argument_expression == NULL)
            throw;

        struct expression* _Owner _Opt p_assignment_expression = assignment_expression(ctx);
        if (p_assignment_expression == NULL)
        {
            argument_expression_delete(p_argument_expression);
            throw;
        }

        p_argument_expression->expression = p_assignment_expression;
        argument_expression_list_push(&list, p_argument_expression);

        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        while (ctx->current->type == ',')
        {
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            struct argument_expression* _Owner _Opt p_argument_expression_2 = calloc(1, sizeof * p_argument_expression_2);
            if (p_argument_expression_2 == NULL)
                throw;
            struct expression* _Owner _Opt p_assignment_expression_2 = assignment_expression(ctx);
            if (p_assignment_expression_2 == NULL)
            {
                argument_expression_delete(p_argument_expression_2);
                throw;
            }
            p_argument_expression_2->expression = p_assignment_expression_2;

            argument_expression_list_push(&list, p_argument_expression_2);

            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }
        }
    }
    catch
    {
    }
    return list;
}

bool first_of_postfix_expression(const struct parser_ctx* ctx)
{
    //( type-name )  postfix confunde com (expression) primary
    if (first_of_type_name_ahead(ctx))
        return true;// I don't think it's necessary because primary also works for postfix
    return is_first_of_primary_expression(ctx);
}

static void fix_member_type(struct type* p_type, const struct type* struct_type, const struct type* member_type)
{
    if (struct_type->type_qualifier_flags & TYPE_QUALIFIER_CONST)
    {
        /*
          struct X { int i; };
          const struct X x;
          x.i ;//x.i is const
        */
        p_type->type_qualifier_flags |= TYPE_QUALIFIER_CONST;
    }

    /*
          struct X { int i; };
          const struct X x;
          x.i ;//x.i is also local, or parameter etc.
    */
    p_type->storage_class_specifier_flags = struct_type->storage_class_specifier_flags;

    if (struct_type->type_qualifier_flags & TYPE_QUALIFIER_VIEW)
    {
        /*
          struct X { _Owner int i; };
          _View struct X x;
          x.i ;//is is not _Owner
        */
        p_type->type_qualifier_flags &= ~TYPE_QUALIFIER_OWNER;
    }

    if (struct_type->type_qualifier_flags & TYPE_QUALIFIER_OPT)
    {
        /*
          struct X { _Owner int i; };
          _View struct X x;
          x.i ;//is is not _Owner
        */
        p_type->type_qualifier_flags |= TYPE_QUALIFIER_OPT;
    }

}

static void fix_arrow_member_type(struct type* p_type, const struct type* left, const struct type* member_type)
{
    struct type t = type_remove_pointer(left);

    if (t.type_qualifier_flags & TYPE_QUALIFIER_CONST)
    {
        /*
           const struct X * p;
        */

        p_type->type_qualifier_flags |= TYPE_QUALIFIER_CONST;
    }

    if (t.type_qualifier_flags & TYPE_QUALIFIER_OPT)
    {
        /*
           const struct X * p;
        */

        p_type->type_qualifier_flags |= TYPE_QUALIFIER_OPT;
    }

    if (t.type_qualifier_flags & TYPE_QUALIFIER_VIEW)
    {
        /*
          _View struct X * p;
        */
        p_type->type_qualifier_flags &= ~TYPE_QUALIFIER_OWNER;
    }

    type_destroy(&t);
}

struct expression* _Owner _Opt postfix_expression_tail(struct parser_ctx* ctx, struct expression* _Owner p_expression_node_param)
{

    /*
       argument cannot be null, but here p_expression_node can be null
    */
    struct expression* _Owner _Opt p_expression_node = p_expression_node_param;

    try
    {
        while (ctx->current != NULL)
        {
            if (ctx->current->type == '[')
            {
                struct expression* _Owner _Opt p_expression_node_new = calloc(1, sizeof * p_expression_node_new);
                if (p_expression_node_new == NULL) throw;
                p_expression_node->last_token = ctx->current;
                p_expression_node_new->first_token = ctx->current;
                p_expression_node_new->expression_type = POSTFIX_ARRAY;
                // the result of the subscription operator ([])

                if (!type_is_pointer_or_array(&p_expression_node->type))
                {
                    compiler_diagnostic(C_ERROR_SUBSCRIPTED_VALUE_IS_NEITHER_ARRAY_NOR_POINTER,
                                                ctx,
                                                ctx->current, NULL,
                                                "subscripted value is neither array nor pointer");
                }

                if (type_is_pointer(&p_expression_node->type))
                {
                    p_expression_node_new->type = type_remove_pointer(&p_expression_node->type);
                }
                else if (type_is_array(&p_expression_node->type))
                {
                    p_expression_node_new->type = get_array_item_type(&p_expression_node->type);
                }

                parser_match(ctx);
                if (ctx->current == NULL)
                {

                    unexpected_end_of_file(ctx);

                    expression_delete(p_expression_node_new);
                    throw;
                }

                /*contem a expresao de dentro do  [ ] */
                p_expression_node_new->right = expression(ctx);
                if (p_expression_node_new->right == NULL)
                {
                    expression_delete(p_expression_node_new);
                    throw;
                }

                if (!type_is_integer(&p_expression_node_new->right->type))
                {
                    compiler_diagnostic(C_ERROR_SUBSCRIPT_IS_NOT_AN_INTEGER,
                                            ctx,
                                            p_expression_node_new->right->first_token,
                                            NULL,
                                            "array subscript is not an integer");

                }

                if (object_has_constant_value(&p_expression_node_new->right->object))
                {
                    unsigned long long index =
                        object_to_unsigned_long_long(&p_expression_node_new->right->object);
                    if (type_is_array(&p_expression_node->type))
                    {
                        if (p_expression_node->type.num_of_elements > 0)
                        {
                            if (index >= (unsigned long long)p_expression_node->type.num_of_elements)
                            {
                                compiler_diagnostic(W_OUT_OF_BOUNDS,
                                                            ctx,
                                                            ctx->current, NULL,
                                                            "index %d is past the end of the array", index);
                            }


                            struct object* _Opt it = object_get_member(&p_expression_node->object, (int)index);

                            if (it != NULL)
                                p_expression_node_new->object = object_make_reference(it);
                        }
                    }
                }
                if (parser_match_tk(ctx, ']') != 0)
                {
                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                p_expression_node_new->left = p_expression_node;
                p_expression_node = p_expression_node_new;
            }
            else if (ctx->current->type == '(')
            {
                struct expression* _Owner _Opt p_expression_node_new = calloc(1, sizeof * p_expression_node_new);
                if (p_expression_node_new == NULL) throw;
                p_expression_node->last_token = ctx->current;
                p_expression_node_new->first_token = p_expression_node->first_token;
                p_expression_node_new->expression_type = POSTFIX_FUNCTION_CALL;

                if (!type_is_function_or_function_pointer(&p_expression_node->type))
                {
                    compiler_diagnostic(C_ERROR_CALLED_OBJECT_IS_NOT_FUNCTION_OR_FUNCTION_POINTER,
                                                ctx,
                                                ctx->current,
                                                NULL,
                                                "called object is not attr function or function pointer");
                }

                p_expression_node_new->type = get_function_return_type(&p_expression_node->type);

                parser_match(ctx);
                if (ctx->current == NULL)
                {

                    unexpected_end_of_file(ctx);

                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                if (ctx->current->type != ')')
                {
                    p_expression_node_new->argument_expression_list = argument_expression_list(ctx);
                }
                if (parser_match_tk(ctx, ')') != 0)
                {
                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                compare_function_arguments(ctx, &p_expression_node->type, &p_expression_node_new->argument_expression_list);

                if (ctx->previous == NULL)
                {
                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                make_object(&p_expression_node_new->type, &p_expression_node_new->object);
                p_expression_node_new->last_token = ctx->previous;
                p_expression_node_new->left = p_expression_node;
                p_expression_node = p_expression_node_new;
            }
            else if (ctx->current->type == '.')
            {
                struct expression* _Owner _Opt p_expression_node_new = calloc(1, sizeof * p_expression_node_new);
                if (p_expression_node_new == NULL) throw;
                p_expression_node->last_token = ctx->current;
                p_expression_node_new->first_token = ctx->current;
                p_expression_node_new->expression_type = POSTFIX_DOT;
                p_expression_node_new->left = p_expression_node;
                p_expression_node = NULL; /*MOVED*/

                p_expression_node_new->declarator = p_expression_node_new->left->declarator;

                parser_match(ctx);
                if (ctx->current == NULL)
                {

                    unexpected_end_of_file(ctx);

                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                if (p_expression_node_new->left->type.type_specifier_flags & TYPE_SPECIFIER_STRUCT_OR_UNION)
                {
                    assert(p_expression_node_new->left->type.struct_or_union_specifier != NULL);

                    struct struct_or_union_specifier* _Opt p_complete =
                        find_struct_or_union_specifier(ctx, p_expression_node_new->left->type.struct_or_union_specifier->tag_name);

                    if (p_complete)
                        p_complete = get_complete_struct_or_union_specifier(p_complete);

                    if (p_complete)
                    {
                        assert(ctx->current != NULL);

                        int member_index = 0;
                        struct member_declarator* _Opt p_member_declarator =
                            find_member_declarator(&p_complete->member_declaration_list, ctx->current->lexeme, &member_index);

                        if (p_member_declarator)
                        {
                            p_expression_node_new->member_index = member_index;

                            if (p_member_declarator->declarator)
                            {
                                p_expression_node_new->type = make_type_using_declarator(ctx, p_member_declarator->declarator);
                            }
                            else
                            {
                                /*
                                struct X {
                                    int : 1;
                                };
                                */
                            }

                            if (p_member_declarator->declarator != NULL)
                            {
                                fix_member_type(&p_expression_node_new->type,
                                                &p_expression_node_new->left->type,
                                                &p_member_declarator->declarator->type);
                            }

                            struct object* object = find_object_declarator_by_index(&p_expression_node_new->left->object, &p_complete->member_declaration_list, member_index);

                            if (object)
                            {
                                p_expression_node_new->object = object_make_reference(object);
                            }
                            else
                            {
                                //not fixed yet
                                 //assert(false);                                    
                            }
                        }
                        else
                        {
                            compiler_diagnostic(C_ERROR_STRUCT_MEMBER_NOT_FOUND,
                                                    ctx,
                                                    ctx->current, NULL,
                                                    "member '%s' not found in 'struct %s'",
                                                    ctx->current->lexeme,
                                                    p_complete->tag_name);
                        }
                    }
                    else
                    {
                        compiler_diagnostic(C_ERROR_STRUCT_MEMBER_NOT_FOUND,
                                                 ctx,
                                                 ctx->current, NULL,
                                                 "incomplete struct type '%s'",
                                                 p_expression_node_new->left->type.struct_or_union_specifier->tag_name);
                        //print_scope(&ctx->scopes);
                    }
                    if (parser_match_tk(ctx, TK_IDENTIFIER) != 0)
                    {
                        expression_delete(p_expression_node_new);
                        p_expression_node_new = NULL;
                        throw;
                    }
                }
                else
                {
                    compiler_diagnostic(C_ERROR_STRUCTURE_OR_UNION_REQUIRED,
                                                ctx,
                                                ctx->current, NULL,
                                                "structure or union required");
                }
                // todo apontar pro nome?
                p_expression_node = p_expression_node_new;
            }
            else if (ctx->current->type == '->')
            {
                /*
                             p_expression_node_new
                           (left)   /  \  (right)
                                   /    \
                  p_expression_node      NULL
                */

                struct expression* _Owner _Opt p_expression_node_new = calloc(1, sizeof * p_expression_node_new);
                if (p_expression_node_new == NULL) throw;

                p_expression_node->last_token = ctx->current;
                p_expression_node_new->first_token = p_expression_node->first_token;
                p_expression_node_new->last_token = ctx->current;
                p_expression_node_new->expression_type = POSTFIX_ARROW;

                // the result of a member access through pointer -> operator is lvalue

                parser_match(ctx);
                if (ctx->current == NULL)
                {
                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                if (type_is_pointer_or_array(&p_expression_node->type))
                {
                    struct type item_type = { 0 };
                    if (type_is_array(&p_expression_node->type))
                    {
                        compiler_diagnostic(W_ARRAY_INDIRECTION, ctx, ctx->current, NULL, "using indirection '->' in array");
                        item_type = get_array_item_type(&p_expression_node->type);
                    }
                    else
                    {
                        item_type = type_remove_pointer(&p_expression_node->type);
                    }

                    if (type_is_struct_or_union(&item_type))
                    {
                        assert(p_expression_node->type.next);
                        assert(p_expression_node->type.next->struct_or_union_specifier);

                        struct struct_or_union_specifier* _Opt p_complete =
                            get_complete_struct_or_union_specifier(p_expression_node->type.next->struct_or_union_specifier);

                        if (p_complete)
                        {
                            int member_index = 0;
                            struct member_declarator* _Opt p_member_declarator =
                                find_member_declarator(&p_complete->member_declaration_list, ctx->current->lexeme, &member_index);

                            if (p_member_declarator)
                            {
                                if (p_member_declarator->declarator)
                                {
                                    p_expression_node_new->member_index = member_index;
                                    p_expression_node_new->type = make_type_using_declarator(ctx, p_member_declarator->declarator);
                                    fix_arrow_member_type(&p_expression_node_new->type, &p_expression_node->type, &p_expression_node_new->type);
                                }
                                else
                                {
                                    assert(false); //TODO
                                }
                            }
                            else
                            {
                                compiler_diagnostic(C_ERROR_STRUCT_MEMBER_NOT_FOUND,
                                                            ctx,
                                                            ctx->current, NULL,
                                                            "member '%s' not found in struct '%s'",
                                                            ctx->current->lexeme,
                                                            p_expression_node->type.next->struct_or_union_specifier->tag_name);
                            }
                        }
                        else
                        {
                            compiler_diagnostic(C_ERROR_STRUCT_IS_INCOMPLETE,
                                                        ctx,
                                                        ctx->current, NULL,
                                                        "struct '%s' is incomplete.",
                                                        ctx->current->lexeme);
                        }
                        if (parser_match_tk(ctx, TK_IDENTIFIER) != 0)
                        {
                            type_destroy(&item_type);
                            expression_delete(p_expression_node_new);
                            p_expression_node_new = NULL;
                            throw;
                        }
                    }
                    else
                    {
                        compiler_diagnostic(C_ERROR_STRUCTURE_OR_UNION_REQUIRED,
                                                    ctx,
                                                    ctx->current, NULL,
                                                    "structure or union required");
                    }
                    type_destroy(&item_type);
                }
                else
                {
                    compiler_diagnostic(C_ERROR_STRUCTURE_OR_UNION_REQUIRED,
                                                ctx,
                                                ctx->current, NULL,
                                                "structure or union required");
                }

                p_expression_node_new->left = p_expression_node;
                p_expression_node = p_expression_node_new;
            }
            else if (ctx->current->type == '++')
            {
                p_expression_node->last_token = ctx->current;

                if (type_is_owner(&p_expression_node->type))
                {
                    compiler_diagnostic(C_ERROR_OPERATOR_INCREMENT_CANNOT_BE_USED_IN_OWNER,
                                                ctx,
                                                p_expression_node->first_token, NULL,
                                                "operator ++ cannot be used in _Owner pointers");
                }

                if (!expression_is_lvalue(p_expression_node))
                {
                    compiler_diagnostic(C_ERROR_OPERATOR_NEEDS_LVALUE,
                                                ctx,
                                                p_expression_node->first_token, NULL,
                                                "lvalue required as increment operand");
                }


                struct expression* _Owner _Opt p_expression_node_new = calloc(1, sizeof * p_expression_node_new);
                if (p_expression_node_new == NULL) throw;

                p_expression_node->last_token = ctx->current;
                p_expression_node_new->first_token = ctx->current;
                p_expression_node_new->expression_type = POSTFIX_INCREMENT;

                p_expression_node_new->type = type_dup(&p_expression_node->type);
                parser_match(ctx);
                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                p_expression_node_new->left = p_expression_node;
                p_expression_node = p_expression_node_new;
            }
            else if (ctx->current->type == '--')
            {
                p_expression_node->last_token = ctx->current;

                if (type_is_owner(&p_expression_node->type))
                {
                    compiler_diagnostic(C_ERROR_OPERATOR_DECREMENT_CANNOT_BE_USED_IN_OWNER,
                                                ctx,
                                                p_expression_node->first_token, NULL,
                                                "operator -- cannot be used in owner pointers");
                }

                if (!expression_is_lvalue(p_expression_node))
                {
                    compiler_diagnostic(C_ERROR_OPERATOR_NEEDS_LVALUE,
                                                ctx,
                                                p_expression_node->first_token, NULL,
                                                "lvalue required as decrement operand");
                }

                struct expression* _Owner _Opt p_expression_node_new = calloc(1, sizeof * p_expression_node_new);
                if (p_expression_node_new == NULL) throw;


                p_expression_node_new->first_token = ctx->current;
                p_expression_node_new->expression_type = POSTFIX_DECREMENT;

                p_expression_node_new->type = type_dup(&p_expression_node->type);
                parser_match(ctx);
                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    expression_delete(p_expression_node_new);
                    p_expression_node_new = NULL;
                    throw;
                }

                p_expression_node_new->left = p_expression_node;
                p_expression_node = p_expression_node_new;
            }
            else
            {
                struct token* _Opt p_last = previous_parser_token(ctx->current);
                if (p_last == NULL)
                    throw; //unexpected

                p_expression_node->last_token = p_last;
                break;
            }
        }
    }
    catch
    {
    }

    return p_expression_node;
}

struct expression* _Owner _Opt postfix_expression_type_name(struct parser_ctx* ctx, struct type_name* _Owner p_type_name_par)
{
    /*
        ( type-name ) { initializer-ctx }
        ( type-name ) { initializer-ctx , }

        //My extension : if type-name is function then follow is compound-statement
        ( type-name ) compound-statement

    */
    struct type_name* _Owner _Opt p_type_name = p_type_name_par; //MOVED
    struct expression* _Owner _Opt p_expression_node = NULL;

    try
    {
        p_expression_node = calloc(1, sizeof * p_expression_node);
        if (p_expression_node == NULL)
            throw;

        assert(p_expression_node->type_name == NULL);

        struct token* _Opt p_previous = previous_parser_token(p_type_name->first_token);
        if (p_previous == NULL)
            throw;

        p_expression_node->first_token = p_previous;
        assert(p_expression_node->first_token->type == '(');

        p_expression_node->type_name = p_type_name; /*MOVED*/
        p_type_name = NULL; /*MOVED*/
        p_expression_node->type = make_type_using_declarator(ctx, p_expression_node->type_name->abstract_declarator);

        if (type_is_function(&p_expression_node->type_name->abstract_declarator->type))
        {
            p_expression_node->expression_type = POSTFIX_EXPRESSION_FUNCTION_LITERAL;



            struct scope* parameters_scope =
                &p_expression_node->type_name->abstract_declarator->direct_declarator->function_declarator->parameters_scope;

            scope_list_push(&ctx->scopes, parameters_scope);

            struct declarator* _Opt p_current_function_opt = ctx->p_current_function_opt;
            ctx->p_current_function_opt = p_expression_node->type_name->abstract_declarator;

            struct scope* p_current_function_scope_opt = ctx->p_current_function_scope_opt;
            ctx->p_current_function_scope_opt = ctx->scopes.tail;

            p_expression_node->compound_statement = function_body(ctx);

            scope_list_pop(&ctx->scopes);
            ctx->p_current_function_opt = p_current_function_opt; //restore
            ctx->p_current_function_scope_opt = p_current_function_scope_opt; //restore

        }
        else
        {
            p_expression_node->expression_type = POSTFIX_EXPRESSION_COMPOUND_LITERAL;
            p_expression_node->braced_initializer = braced_initializer(ctx);
            p_expression_node->type = type_dup(&p_expression_node->type_name->type);
            //TODO

            if (p_expression_node->type.storage_class_specifier_flags & STORAGE_SPECIFIER_TYPEDEF)
            {
            }
            else
            {
                int er = make_object(&p_expression_node->type, &p_expression_node->object);
                if (er != 0)
                {
                    compiler_diagnostic(C_ERROR_STRUCT_IS_INCOMPLETE, ctx, p_expression_node->first_token, NULL, "incomplete struct/union type");
                    throw;
                }
            }

            const bool is_constant = type_is_const_or_constexpr(&p_expression_node->type);

            object_default_initialization(&p_expression_node->object, is_constant);
            
            struct initializer initializer = { 0 };
            initializer.braced_initializer = p_expression_node->braced_initializer;
            initializer.first_token = p_expression_node->first_token;

            const bool requires_constant_initialization = false;

            initializer_init_new(ctx,
                         &p_expression_node->type,
                         &p_expression_node->object,
                         &initializer,
                         is_constant,
                         requires_constant_initialization);
        }

        if (ctx->previous == NULL)
            throw;

        p_expression_node->last_token = ctx->previous;

        p_expression_node = postfix_expression_tail(ctx, p_expression_node);
        if (p_expression_node == NULL)
            throw;
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }
    type_name_delete(p_type_name);
    return p_expression_node;
}

struct expression* _Owner _Opt postfix_expression(struct parser_ctx* ctx)
{
    /*
      postfix-expression:
        primary-expression
        postfix-expression [ expression ]
        postfix-expression ( argument-expression-list_opt)
        postfix-expression . identifier
        postfix-expression -> identifier
        postfix-expression ++
        postfix-expression --
        ( type-name ) { initializer-ctx }
        ( type-name ) { initializer-ctx , }

        //My extension : if type-name is function then follow is compound-statement
        ( type-name ) compound-statement

        */
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {


        if (first_of_type_name_ahead(ctx)) // aqui preciso ver se nao eh primary
        {
            assert(false); // este caso esta pegando lá dentro deo cast expression.
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL)
                throw;


            assert(ctx->current != NULL);
            p_expression_node->first_token = ctx->current;
            if (parser_match_tk(ctx, '(') != 0)
                throw;

            p_expression_node->type_name = type_name(ctx);
            if (p_expression_node->type_name == NULL)
                throw;

            p_expression_node->type = make_type_using_declarator(ctx, p_expression_node->type_name->abstract_declarator);

            if (parser_match_tk(ctx, ')') != 0)
                throw;
            // printf("\n");
            // print_type(&p_expression_node->type);

            if (type_is_function(&p_expression_node->type))
            {
                p_expression_node->expression_type = POSTFIX_EXPRESSION_FUNCTION_LITERAL;
                p_expression_node->compound_statement = compound_statement(ctx);
                if (p_expression_node->compound_statement == NULL)
                    throw;

                p_expression_node->last_token = p_expression_node->compound_statement->last_token;
            }
            else
            {
                p_expression_node->expression_type = POSTFIX_EXPRESSION_COMPOUND_LITERAL;
                p_expression_node->braced_initializer = braced_initializer(ctx);
                if (p_expression_node->braced_initializer == NULL) throw;
                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    throw;
                }

                p_expression_node->last_token = ctx->current;
            }
        }
        else
        {
            p_expression_node = primary_expression(ctx);
            if (p_expression_node == NULL)
                throw;
        }

        p_expression_node = postfix_expression_tail(ctx, p_expression_node);
        if (p_expression_node == NULL)
            throw;
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }
    return p_expression_node;
}

bool is_first_of_compiler_function(struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return
        // traits
        ctx->current->type == TK_KEYWORD_IS_LVALUE ||
        ctx->current->type == TK_KEYWORD_IS_OWNER ||
        ctx->current->type == TK_KEYWORD_IS_CONST ||
        ctx->current->type == TK_KEYWORD_IS_POINTER ||
        ctx->current->type == TK_KEYWORD_IS_ARRAY ||
        ctx->current->type == TK_KEYWORD_IS_FUNCTION ||

        ctx->current->type == TK_KEYWORD_ASSERT ||

        ctx->current->type == TK_KEYWORD_IS_SCALAR ||
        ctx->current->type == TK_KEYWORD_IS_ARITHMETIC ||
        ctx->current->type == TK_KEYWORD_IS_FLOATING_POINT ||
        ctx->current->type == TK_KEYWORD_IS_INTEGRAL;
}

bool is_first_of_unary_expression(struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return first_of_postfix_expression(ctx) ||
        ctx->current->type == '++' ||
        ctx->current->type == '--' ||
        ctx->current->type == '&' ||
        ctx->current->type == '*' ||
        ctx->current->type == '+' ||
        ctx->current->type == '-' ||
        ctx->current->type == '~' ||
        ctx->current->type == '!' ||
        ctx->current->type == TK_KEYWORD_SIZEOF ||
        ctx->current->type == TK_KEYWORD__COUNTOF ||
        ctx->current->type == TK_KEYWORD__ALIGNOF ||
        is_first_of_compiler_function(ctx);
}

static int check_sizeof_argument(struct parser_ctx* ctx,
    const struct expression* p_expression,
    const struct type* const p_type)
{
    //sizeof(type)  p_expression is the sizeof expression
    //sizeof(expression) p_expression is expression


    enum type_category category = type_get_category(p_type);

    if (category == TYPE_CATEGORY_FUNCTION)
    {
        //In GCC returns 1

        //The sizeof operator shall not be applied to an expression that has function type or an incomplete type
    }
    else if (category == TYPE_CATEGORY_ITSELF &&
            p_type->type_specifier_flags & TYPE_SPECIFIER_STRUCT_OR_UNION)
    {
        assert(p_type->struct_or_union_specifier);

        struct struct_or_union_specifier* _Opt p_complete =
            get_complete_struct_or_union_specifier(p_type->struct_or_union_specifier);
        if (p_complete == NULL)
        {
            //The sizeof operator shall not be applied to an expression that has function type or an incomplete type
            compiler_diagnostic(C_ERROR_STRUCT_IS_INCOMPLETE,
                                       ctx,
                                       p_expression->first_token,
                                       NULL,
                                       "struct is incomplete type");
            return -1;
        }
    }
    else if (category == TYPE_CATEGORY_ARRAY)
    {
        if (type_is_vla(p_type))
        {
            return 0;
        }

        if (p_type->storage_class_specifier_flags & STORAGE_SPECIFIER_PARAMETER)
        {
            //GCC
            //<source>:4:21: warning: 'sizeof' on array function parameter 'a' will return size of 'int *' [-Wsizeof-array-argument]
            //CLANG
            //<source>:4:21: warning: sizeof on array function parameter will return size of 'int *' instead of 'int[]' [-Wsizeof-array-argument]

            compiler_diagnostic(W_SIZEOF_ARRAY_ARGUMENT,
                                        ctx,
                                        p_expression->first_token,
                                        NULL,
                                        "sizeof applied to array function parameter");

        }
    }

    return 0; //ok
}

struct expression* _Owner _Opt unary_expression(struct parser_ctx* ctx)
{
    /*
    unary-expression:
        postfix-expression
        ++ unary-expression
        -- unary-expression
        unary-operator cast-expression
        sizeof unary-expression
        sizeof ( type-name )
        _Countof unary-expression   //C2Y
        _Countof ( type-name )      //C2Y
        alignof ( type-name )
    */

    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        if (ctx->current->type == '++' || ctx->current->type == '--')
        {
            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->first_token = ctx->current;

            if (ctx->current->type == '++')
                new_expression->expression_type = UNARY_EXPRESSION_INCREMENT;
            else
                new_expression->expression_type = UNARY_EXPRESSION_DECREMENT;
            parser_match(ctx);
            if (ctx->current == NULL)
            {

                unexpected_end_of_file(ctx);

                expression_delete(new_expression);
                throw;
            }
            new_expression->right = unary_expression(ctx);
            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }

            new_expression->type = type_dup(&new_expression->right->type);
            p_expression_node = new_expression;
        }
        else if (ctx->current->type == '&' ||
                 ctx->current->type == '*' ||
                 ctx->current->type == '+' ||
                 ctx->current->type == '-' ||
                 ctx->current->type == '~' ||
                 ctx->current->type == '!')
        {

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->first_token = ctx->current;

            struct token* op_position = ctx->current; // marcar posicao
            enum token_type op = ctx->current->type;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(new_expression);
                throw;
            }
#if 0
            //visual studio formater is adding spaces..
            if (style_has_space(ctx->current))
            {
                compiler_diagnostic(W_STYLE, ctx, ctx->current, "don't use spaces");
            }
#endif

            new_expression->right = cast_expression(ctx);
            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }

            new_expression->last_token = new_expression->right->last_token;
            if (op == '!')
            {
                new_expression->expression_type = UNARY_EXPRESSION_NOT;
                if (!ctx->evaluation_is_disabled &&
                    object_has_constant_value(&new_expression->right->object))
                {
                    const bool v = object_to_bool(&new_expression->right->object);
                    new_expression->object = object_make_signed_int(!v);
                }
                new_expression->type = type_make_int_bool_like();
            }
            else if (op == '~')
            {
                if (!type_is_integer(&new_expression->right->type))
                {
                    compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_INTEGER,
                                               ctx,
                                               op_position,
                                               NULL,
                                               "requires integer type");

                    expression_delete(new_expression);
                    throw;
                }

                new_expression->expression_type = UNARY_EXPRESSION_BITNOT;

                /*
                The result of the ~ operator is the bitwise complement of its (promoted) operand (that is, each bit in
                the result is set if and only if the corresponding bit in the converted operand is not set). The integer
                promotions are performed on the operand, and the result has the promoted type. If the promoted
                type is an unsigned type, the expression ~E is equivalent to the maximum value representable in
                that type minus E.
                */
                struct type promoted = type_dup(&new_expression->right->type);
                type_integer_promotion(&promoted);
                new_expression->type = promoted;

                if (!ctx->evaluation_is_disabled &&
                  object_has_constant_value(&new_expression->right->object))
                {
                    enum object_value_type vt = type_to_object_type(&new_expression->type);
                    switch (vt)
                    {
                    case TYPE_SIGNED_INT:
                    {
                        signed int r = object_to_signed_int(&new_expression->right->object);
                        new_expression->object = object_make_signed_int(~r);
                    }
                    break;

                    case TYPE_UNSIGNED_INT:
                    {
                        unsigned int r = object_to_unsigned_int(&new_expression->right->object);
                        new_expression->object = object_make_unsigned_int(~r);
                    }
                    break;

                    case TYPE_SIGNED_LONG:
                    {
                        signed long r = object_to_signed_long(&new_expression->right->object);
                        new_expression->object = object_make_signed_long(~r);
                    }
                    break;
                    case TYPE_UNSIGNED_LONG:
                    {
                        unsigned long r = object_to_unsigned_long(&new_expression->right->object);
                        new_expression->object = object_make_unsigned_long(~r);
                    }
                    break;

                    case TYPE_SIGNED_LONG_LONG:
                    {
                        signed long long r = object_to_signed_long_long(&new_expression->right->object);
                        new_expression->object = object_make_signed_long_long(~r);
                    }
                    break;
                    case TYPE_UNSIGNED_LONG_LONG:
                    {
                        unsigned long long r = object_to_unsigned_long_long(&new_expression->right->object);
                        new_expression->object = object_make_unsigned_long_long(~r);
                    }
                    break;

                    case TYPE_SIGNED_SHORT:
                    case TYPE_UNSIGNED_SHORT:
                    case TYPE_SIGNED_CHAR:
                    case TYPE_UNSIGNED_CHAR:

                    case TYPE_BOOL:
                    case TYPE_FLOAT:
                    case TYPE_DOUBLE:
                    case TYPE_LONG_DOUBLE:
                        break;
                    };
                }
            }
            else if (op == '-' || op == '+')
            {
                if (op == '-')
                    new_expression->expression_type = UNARY_EXPRESSION_NEG;
                else
                    new_expression->expression_type = UNARY_EXPRESSION_PLUS;

                //promote
                new_expression->type = type_common(&new_expression->right->type, &new_expression->right->type);

                if (!ctx->evaluation_is_disabled &&
                    object_has_constant_value(&new_expression->right->object))
                {
                    enum object_value_type vt = type_to_object_type(&new_expression->type);
                    switch (vt)
                    {
                    case TYPE_SIGNED_INT:
                    {
                        const int a = object_to_signed_int(&new_expression->right->object);
                        if (op == '-')
                            new_expression->object = object_make_signed_int(-a);
                        else
                            new_expression->object = object_make_signed_int(+a);
                    }
                    break;

                    case TYPE_UNSIGNED_INT:
                    {
                        unsigned int a = object_to_unsigned_int(&new_expression->right->object);
                        if (op == '-')
                        {
                            //error C4146: unary minus operator applied to unsigned type, result still unsigned
                            new_expression->object = object_make_unsigned_int(-a);
                        }
                        else
                            new_expression->object = object_make_unsigned_int(+a);
                    }
                    break;

                    case TYPE_SIGNED_LONG:
                    {
                        const signed long a = object_to_signed_long(&new_expression->right->object);
                        if (op == '-')
                        {
                            //TODO check overflow
                            new_expression->object = object_make_signed_long(-a);
                        }
                        else
                            new_expression->object = object_make_signed_long(+a);
                    }
                    break;
                    case TYPE_UNSIGNED_LONG:
                    {
                        unsigned long a = object_to_unsigned_long(&new_expression->right->object);
                        if (op == '-')
                        {
                            //
                            //error C4146: unary minus operator applied to unsigned type, result still unsigned
                            new_expression->object = object_make_unsigned_long(-a);
                        }
                        else
                            new_expression->object = object_make_unsigned_long(+a);
                    }
                    break;

                    case TYPE_SIGNED_LONG_LONG:
                    {
                        signed long long a = object_to_signed_long_long(&new_expression->right->object);
                        if (op == '-')
                            new_expression->object = object_make_signed_long_long(-a);
                        else
                            new_expression->object = object_make_signed_long_long(+a);
                    }
                    break;

                    case TYPE_UNSIGNED_LONG_LONG:
                    {
                        unsigned long long a = object_to_unsigned_long_long(&new_expression->right->object);

                        if (op == '-')
                        {
                            //error C4146: unary minus operator applied to unsigned type, result still unsigned
                            new_expression->object = object_make_unsigned_long_long(-a);
                        }
                        else
                            new_expression->object = object_make_unsigned_long_long(+a);
                    }
                    break;

                    case TYPE_BOOL:
                    case TYPE_SIGNED_CHAR:
                    case TYPE_UNSIGNED_CHAR:
                    case TYPE_SIGNED_SHORT:
                    case TYPE_UNSIGNED_SHORT:
                        assert(false); //they are promoted
                        expression_delete(new_expression);
                        throw;
                        break;

                    case TYPE_FLOAT:
                    {
                        float a = object_to_float(&new_expression->right->object);
                        if (op == '-')
                            new_expression->object = object_make_float(-a);
                        else
                            new_expression->object = object_make_float(+a);
                    }
                    break;
                    case TYPE_DOUBLE:
                    {
                        double a = object_to_double(&new_expression->right->object);
                        if (op == '-')
                            new_expression->object = object_make_double(-a);
                        else
                            new_expression->object = object_make_double(+a);
                    }
                    break;
                    case TYPE_LONG_DOUBLE:
                    {
                        long double a = object_to_long_double(&new_expression->right->object);
                        if (op == '-')
                            new_expression->object = object_make_long_double(-a);
                        else
                            new_expression->object = object_make_long_double(+a);
                    }
                    break;
                    };
                }
                //'//'new_expression->type = type_dup(&new_expression->right->type);
                //type_integer_promotion(&new_expression->type);
            }
            else if (op == '*')
            {
                new_expression->expression_type = UNARY_EXPRESSION_CONTENT;
                // the result of the indirection(unary*) operator applied to a pointer to object

                if (!type_is_pointer_or_array(&new_expression->right->type))
                {
                    compiler_diagnostic(C_ERROR_INDIRECTION_REQUIRES_POINTER_OPERAND,
                                                ctx,
                                                op_position,
                        NULL,
                                                "indirection requires pointer operand");
                }
                if (type_is_pointer(&new_expression->right->type))
                {
                    new_expression->type = type_remove_pointer(&new_expression->right->type);
                }
                else
                {
                    compiler_diagnostic(W_ARRAY_INDIRECTION,
                                                ctx,
                                                op_position, NULL,
                                                "array indirection");
                    new_expression->type = get_array_item_type(&new_expression->right->type);
                }
            }
            else if (op == '&')
            {
                /*
                  The result of the unary & operator is a pointer to the object referred to by the lvalue-expression.
                  If the type of the lvalue-expression is " . . . ", the type of the result is
                  "pointer to . . ."
                */
                new_expression->expression_type = UNARY_EXPRESSION_ADDRESSOF;

                if (!expression_is_lvalue(new_expression->right))
                {
                    compiler_diagnostic(C_ERROR_ADDRESS_OF_REGISTER,
                                                ctx,
                                                new_expression->right->first_token,
                                                NULL,
                                                "lvalue required as unary '&' operand");
                }

                if (new_expression->right->type.storage_class_specifier_flags & STORAGE_SPECIFIER_REGISTER)
                {
                    const char* variable_name = "?";

                    if (new_expression->right->declarator &&
                        new_expression->right->declarator->name_opt)
                    {
                        variable_name = new_expression->right->declarator->name_opt->lexeme;
                    }

                    compiler_diagnostic(C_ERROR_ADDRESS_OF_REGISTER,
                                                ctx,
                                                new_expression->right->first_token,
                                                NULL,
                                                "address of register variable 'x' requested",
                                                variable_name);

                }

                new_expression->type = type_add_pointer(&new_expression->right->type, ctx->options.null_checks_enabled);
                new_expression->type.address_of = true;
            }
            else
            {
                expression_delete(new_expression);
                compiler_diagnostic(C_ERROR_INVALID_TOKEN,
                                            ctx,
                                            ctx->current,
                    NULL,
                                            "invalid token");
                throw;
            }
            p_expression_node = new_expression;
        }
        else if (ctx->current->type == TK_KEYWORD_SIZEOF)
        {
            const bool disable_evaluation_copy = ctx->evaluation_is_disabled;
            ctx->evaluation_is_disabled = true;
            // defer would be nice here...

            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->first_token = ctx->current;

            if (first_of_type_name_ahead(ctx))
            {
                new_expression->expression_type = UNARY_EXPRESSION_SIZEOF_TYPE;
                if (parser_match_tk(ctx, '(') != 0)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }
                new_expression->type_name = type_name(ctx);
                if (new_expression->type_name == NULL)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }

                new_expression->type = make_size_t_type();

                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    expression_delete(new_expression);
                    throw;
                }

                new_expression->last_token = ctx->current;

                if (parser_match_tk(ctx, ')') != 0)
                {
                    expression_delete(new_expression);
                    throw;
                }

                if (check_sizeof_argument(ctx, new_expression, &new_expression->type_name->type) != 0)
                {
                    //not fatal error
                    //fatal will be if someone need the sizeof at compile time
                    //but we don't have the object set here
                }
                else
                {
                    if (type_is_vla(&new_expression->type_name->abstract_declarator->type))
                    {
                        //not constant
                    }
                    else
                    {
                        size_t type_sizeof = 0;
                        if (type_get_sizeof(&new_expression->type_name->abstract_declarator->type, &type_sizeof) != 0)
                        {
                            throw;
                        }

                        new_expression->object = object_make_size_t(type_sizeof);
                    }
                }
            }
            else
            {
                new_expression->right = unary_expression(ctx);
                if (new_expression->right == NULL)
                {
                    /*restore*/
                    ctx->evaluation_is_disabled = disable_evaluation_copy;
                    expression_delete(new_expression);
                    throw;
                }

                new_expression->expression_type = UNARY_EXPRESSION_SIZEOF_EXPRESSION;

                if (check_sizeof_argument(ctx, new_expression->right, &new_expression->right->type) != 0)
                {
                    expression_delete(new_expression);
                    throw;
                }

                if (type_is_vla(&new_expression->right->type))
                {
                    //not constant
                }
                else
                {
                    size_t sz = 0;
                    if (type_get_sizeof(&new_expression->right->type, &sz) != 0)
                        throw;
                    new_expression->object = object_make_size_t(sz);
                }
            }

            type_destroy(&new_expression->type);
            new_expression->type = type_make_size_t();
            p_expression_node = new_expression;

            /*restore*/
            ctx->evaluation_is_disabled = disable_evaluation_copy;
        }
        else if (ctx->current->type == TK_KEYWORD__COUNTOF)//C2Y
        {
            // defer would be nice here...


            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->first_token = ctx->current;

            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(new_expression);
                throw;
            }

            new_expression->expression_type = UNARY_EXPRESSION_NELEMENTSOF_TYPE;

            if (first_of_type_name_ahead(ctx))
            {
                if (parser_match_tk(ctx, '(') != 0)
                {
                    expression_delete(new_expression);
                    throw;
                }
                new_expression->type_name = type_name(ctx);
                if (new_expression->type_name == NULL)
                {
                    expression_delete(new_expression);
                    throw;
                }

                new_expression->type = make_size_t_type();

                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    expression_delete(new_expression);
                    throw;
                }

                new_expression->last_token = ctx->current;

                if (parser_match_tk(ctx, ')') != 0)
                {
                    expression_delete(new_expression);
                    throw;
                }

                if (type_is_enum(&new_expression->type_name->abstract_declarator->type))
                {
                    const struct enum_specifier* _Opt p_enum_specifier =
                        get_complete_enum_specifier(new_expression->type_name->type.enum_specifier);
                    size_t nelements = 0;
                    if (p_enum_specifier)
                    {
                        struct enumerator* _Owner _Opt p =
                            p_enum_specifier->enumerator_list.head;
                        while (p)
                        {
                            nelements++;
                            p = p->next;
                        }
                    }
                    new_expression->object = object_make_size_t(nelements);
                }
                else if (type_is_array(&new_expression->type_name->abstract_declarator->type))
                {
                    size_t nelements = new_expression->type_name->abstract_declarator->type.num_of_elements;
                    if (nelements > 0)
                        new_expression->object = object_make_size_t(nelements);
                }
                else
                {
                    compiler_diagnostic(C_INVALID_ARGUMENT_NELEMENTSOF,
                                        ctx,
                                        new_expression->type_name->first_token,
                                        NULL,
                                        "argument of _Countof must be an array");

                    expression_delete(new_expression);
                    throw;
                }
            }
            else
            {
                if (parser_match_tk(ctx, '(') != 0)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }

                const bool disable_evaluation_copy = ctx->evaluation_is_disabled;
                ctx->evaluation_is_disabled = true;
                new_expression->right = unary_expression(ctx);
                /*restore*/
                ctx->evaluation_is_disabled = disable_evaluation_copy;

                if (new_expression->right == NULL)
                {
                    expression_delete(new_expression);
                    throw;
                }

                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    expression_delete(new_expression);
                    throw;
                }

                new_expression->last_token = ctx->current;

                if (parser_match_tk(ctx, ')') != 0)
                {
                    expression_delete(new_expression);
                    throw;
                }

                if (type_is_enum(&new_expression->right->type))
                {
                    const struct enum_specifier* _Opt p_enum_specifier =
                        get_complete_enum_specifier(new_expression->right->type.enum_specifier);
                    size_t nelements = 0;
                    if (p_enum_specifier)
                    {
                        struct enumerator* _Owner _Opt p =
                            p_enum_specifier->enumerator_list.head;
                        while (p)
                        {
                            nelements++;
                            p = p->next;
                        }
                    }
                    new_expression->object = object_make_size_t(nelements);

                }
                else if (type_is_array(&new_expression->right->type))
                {
                    size_t nelements = new_expression->right->type.num_of_elements;
                    if (nelements > 0)
                    {
                        new_expression->object = object_make_size_t(nelements);
                    }
                    else
                    {
                        //vla [n][2] but not vla[2][n]
                    }
                }
                else
                {
                    compiler_diagnostic(C_INVALID_ARGUMENT_NELEMENTSOF,
                        ctx,
                        new_expression->right->first_token,
                        NULL,
                        "argument of _Countof must be an array");

                    expression_delete(new_expression);
                    throw;
                }
            }

            type_destroy(&new_expression->type);
            new_expression->type = type_make_size_t();
            p_expression_node = new_expression;

        }

        else if (ctx->current->type == TK_KEYWORD_ASSERT)
        {
            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->expression_type = UNARY_EXPRESSION_ASSERT;
            new_expression->first_token = ctx->current;

            parser_match(ctx);

            if (ctx->current == NULL || parser_match_tk(ctx, '(') != 0)
            {
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }
            new_expression->right = expression(ctx);

            if (parser_match_tk(ctx, ')') != 0)
            {
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }
            return new_expression;
        }
        else if (ctx->current->type == TK_KEYWORD__ALIGNOF)
        {
            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->expression_type = UNARY_EXPRESSION_ALIGNOF;
            new_expression->first_token = ctx->current;
            parser_match(ctx);

            if (ctx->current == NULL || parser_match_tk(ctx, '(') != 0)
            {
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }
            new_expression->type_name = type_name(ctx);
            if (parser_match_tk(ctx, ')') != 0)
            {
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }

            if (!ctx->evaluation_is_disabled)
            {
                new_expression->object = object_make_size_t(type_get_alignof(&new_expression->type_name->type));
            }
            new_expression->type = type_make_int();

            assert(ctx->previous != NULL);
            new_expression->last_token = ctx->previous;

            p_expression_node = new_expression;
        }
        else if (
            ctx->current->type == TK_KEYWORD_IS_LVALUE ||
            ctx->current->type == TK_KEYWORD_IS_OWNER ||
            ctx->current->type == TK_KEYWORD_IS_CONST ||
            ctx->current->type == TK_KEYWORD_IS_POINTER ||
            ctx->current->type == TK_KEYWORD_IS_ARRAY ||
            ctx->current->type == TK_KEYWORD_IS_FUNCTION ||
            ctx->current->type == TK_KEYWORD_IS_ARITHMETIC ||
            ctx->current->type == TK_KEYWORD_IS_SCALAR ||
            ctx->current->type == TK_KEYWORD_IS_FLOATING_POINT ||
            ctx->current->type == TK_KEYWORD_IS_INTEGRAL)
        {
            const bool disable_evaluation_copy = ctx->evaluation_is_disabled;
            ctx->evaluation_is_disabled = true;


            struct token* traits_token = ctx->current;

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->first_token = ctx->current;
            new_expression->expression_type = UNARY_EXPRESSION_TRAITS;

            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }

            struct type* _Opt p_type = NULL;
            if (first_of_type_name_ahead(ctx))
            {
                if (parser_match_tk(ctx, '(') != 0)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }
                new_expression->type_name = type_name(ctx);
                if (new_expression->type_name == NULL)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }

                if (ctx->current == NULL)
                {

                    unexpected_end_of_file(ctx);

                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }

                new_expression->last_token = ctx->current;
                if (parser_match_tk(ctx, ')') != 0)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }
                p_type = &new_expression->type_name->abstract_declarator->type;
            }
            else
            {

                new_expression->right = unary_expression(ctx);
                if (new_expression->right == NULL)
                {
                    /*restore*/
                    ctx->evaluation_is_disabled = disable_evaluation_copy;
                    expression_delete(new_expression);
                    throw;
                }

                p_type = &new_expression->right->type;
                if (ctx->previous == NULL)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }

                new_expression->last_token = ctx->previous;
            }

            switch (traits_token->type)
            {
            case TK_KEYWORD_IS_LVALUE:
                assert(new_expression->right != NULL);
                new_expression->object = object_make_signed_int(expression_is_lvalue(new_expression->right));
                break;

            case TK_KEYWORD_IS_CONST:
                new_expression->object = object_make_signed_int(type_is_const(p_type));
                break;
            case TK_KEYWORD_IS_OWNER:
                new_expression->object = object_make_signed_int(type_is_owner(p_type));
                break;

            case TK_KEYWORD_IS_POINTER:
                new_expression->object = object_make_signed_int(type_is_pointer(p_type));

                break;
            case TK_KEYWORD_IS_FUNCTION:
                new_expression->object = object_make_signed_int(type_is_function(p_type));

                break;
            case TK_KEYWORD_IS_ARRAY:
                new_expression->object = object_make_signed_int(type_is_array(p_type));

                break;
            case TK_KEYWORD_IS_ARITHMETIC:
                new_expression->object = object_make_signed_int(type_is_arithmetic(p_type));

                break;
            case TK_KEYWORD_IS_SCALAR:
                new_expression->object = object_make_signed_int(type_is_scalar(p_type));

                break;
            case TK_KEYWORD_IS_FLOATING_POINT:
                new_expression->object = object_make_signed_int(type_is_floating_point(p_type));

                break;
            case TK_KEYWORD_IS_INTEGRAL:
                new_expression->object = object_make_signed_int(type_is_integer(p_type));

                break;

            default:
                assert(false);
            }

            new_expression->type = type_make_int_bool_like();
            p_expression_node = new_expression;
            /*restore*/
            ctx->evaluation_is_disabled = disable_evaluation_copy;
        }
        else // if (is_first_of_primary_expression(ctx))
        {
            p_expression_node = postfix_expression(ctx);
            if (p_expression_node == NULL)
                throw;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

struct expression* _Owner _Opt cast_expression(struct parser_ctx* ctx)
{
    /*
     cast-expression:
      unary-expression
      ( type-name ) cast-expression


      ( type-name ) //<- extension void value
    */

    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        if (first_of_type_name_ahead(ctx))
        {
            p_expression_node = calloc(1, sizeof * p_expression_node);
            if (p_expression_node == NULL)
                throw;

            p_expression_node->first_token = ctx->current;
            p_expression_node->expression_type = CAST_EXPRESSION;
            if (parser_match_tk(ctx, '(') != 0)
                throw;

            p_expression_node->type_name = type_name(ctx);
            if (p_expression_node->type_name == NULL)
            {
                expression_delete(p_expression_node);
                p_expression_node = NULL;
                throw;
            }

            p_expression_node->type = type_dup(&p_expression_node->type_name->type);


            if (parser_match_tk(ctx, ')') != 0)
                throw;

            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            if (ctx->current->type == '{')
            {
                // Thinking it was a cast expression was a mistake... 
                // because the { appeared then it is a compound literal which is a postfix.
                struct expression* _Owner _Opt new_expression = postfix_expression_type_name(ctx, p_expression_node->type_name);
                p_expression_node->type_name = NULL; // MOVED

                if (new_expression == NULL) throw;

                expression_delete(p_expression_node);
                p_expression_node = new_expression;
            }
            else if (is_first_of_unary_expression(ctx))
            {
                p_expression_node->left = cast_expression(ctx);
                if (p_expression_node->left == NULL)
                {
                    expression_delete(p_expression_node);
                    p_expression_node = NULL;
                    throw;
                }

                if (type_is_floating_point(&p_expression_node->type) &&
                    type_is_pointer(&p_expression_node->left->type))
                {
                    compiler_diagnostic(C_ERROR_POINTER_TO_FLOATING_TYPE,
                        ctx,
                        p_expression_node->first_token,
                        NULL,
                        "pointer type cannot be converted to any floating type");
                }
                else if (type_is_pointer(&p_expression_node->type) &&
                         type_is_floating_point(&p_expression_node->left->type))
                {
                    compiler_diagnostic(C_ERROR_FLOATING_TYPE_TO_POINTER,
                     ctx,
                     p_expression_node->first_token,
                     NULL,
                     "A floating type cannot be converted to any pointer type");
                }
                else if (type_is_nullptr_t(&p_expression_node->left->type))
                {
                    if (type_is_void(&p_expression_node->type) ||
                        type_is_bool(&p_expression_node->type) ||
                        type_is_pointer(&p_expression_node->type))
                    {
                        /*
                          The type nullptr_t shall not be converted to any type other than
                          void, bool or a pointer type
                        */
                    }
                    else
                    {
                        compiler_diagnostic(C_ERROR_NULLPTR_CAST_ERROR,
                        ctx,
                        p_expression_node->first_token,
                        NULL,
                        "cannot cast nullptr_t to this type");
                    }
                }
                else if (type_is_nullptr_t(&p_expression_node->type))
                {
                    /*
                      If the target type is nullptr_t, the cast expression shall
                      be a null pointer constant or have type nullptr_t.
                    */

                    if (expression_is_null_pointer_constant(p_expression_node->left) ||
                        type_is_nullptr_t(&p_expression_node->left->type))
                    {
                        //ok
                    }
                    else
                    {
                        compiler_diagnostic(C_ERROR_NULLPTR_CAST_ERROR,
                        ctx,
                        p_expression_node->left->first_token,
                        NULL,
                        "cannot cast this expression to nullptr_t");
                    }
                }

                if (p_expression_node->left->type.storage_class_specifier_flags & STORAGE_SPECIFIER_FUNCTION_RETURN &&
                    type_is_owner(&p_expression_node->left->type))
                {
                    if (!type_is_owner(&p_expression_node->type))
                    {
                        if (type_is_pointer(&p_expression_node->left->type))
                        {
                            //(int*) malloc(1)
                            compiler_diagnostic(W_OWNERSHIP_DISCARDING_OWNER, ctx, p_expression_node->first_token, NULL, "discarding _Owner pointer");
                        }
                        else
                        {
                            compiler_diagnostic(W_OWNERSHIP_DISCARDING_OWNER, ctx, p_expression_node->first_token, NULL, "discarding _Owner");
                        }
                    }
                }

                type_destroy(&p_expression_node->type);
                p_expression_node->type = make_type_using_declarator(ctx, p_expression_node->type_name->abstract_declarator);

                if (!ctx->evaluation_is_disabled &&
                    object_has_constant_value(&p_expression_node->left->object))
                {
                    enum object_value_type vt = type_to_object_type(&p_expression_node->type);

                    p_expression_node->object =
                        object_cast(vt, &p_expression_node->left->object);

                }



                p_expression_node->type.storage_class_specifier_flags =
                    p_expression_node->left->type.storage_class_specifier_flags;
            }
            else
            {
                compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "expected expression");
            }
        }
        else if (is_first_of_unary_expression(ctx))
        {
            p_expression_node = unary_expression(ctx);
            if (p_expression_node == NULL)
            {
                throw;
            }
        }
        else
        {
            compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "expected expression");
            assert(p_expression_node == NULL);
            throw;
        }

        if (ctx->current == NULL || ctx->previous == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        p_expression_node->last_token = ctx->previous;
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}


NODISCARD
errno_t execute_arithmetic(const struct parser_ctx* ctx,
                      const struct expression* new_expression,
                      int op,
                      struct object* result)
{

    assert(!ctx->evaluation_is_disabled);

    struct type common_type = { 0 };

    try
    {
        if (new_expression->left == NULL || new_expression->right == NULL)
        {
            assert(false);
            throw;
        }

        struct object value = { 0 };
        switch (op)
        {
        case '+':
        case '-':

        case '*':
        case '/':
        case '%':
            //
        case '>':
        case '<':
        case '>=':
        case '<=':
            //
        case '==':
        case '!=':
            break;
        default:
            assert(false);
            throw;
        }

        //Each of the operands shall have arithmetic type
        if (!type_is_arithmetic(&new_expression->left->type))
        {
            compiler_diagnostic(C_ERROR_LEFT_IS_NOT_INTEGER, ctx, ctx->current, NULL, "left type must be an arithmetic type");
            throw;
        }

        if (!type_is_arithmetic(&new_expression->right->type))
        {
            compiler_diagnostic(C_ERROR_LEFT_IS_NOT_INTEGER, ctx, ctx->current, NULL, "right type must be an arithmetic type");
            throw;
        }


        if (!ctx->evaluation_is_disabled &&
            object_has_constant_value(&new_expression->left->object) &&
            object_has_constant_value(&new_expression->right->object))
        {

            const struct marker m =
            {
                .p_token_begin = new_expression->left->first_token,
                .p_token_end = new_expression->right->last_token
            };

            common_type = type_common(&new_expression->left->type, &new_expression->right->type);

            enum object_value_type vt = type_to_object_type(&common_type);
            switch (vt)
            {
            case TYPE_SIGNED_INT:
            {
                const int a = object_to_signed_int(&new_expression->left->object);
                const int b = object_to_signed_int(&new_expression->right->object);

                if (op == '+')
                {
                    const int computed_result = a + b;
                    signed long long exact_result;
                    if (signed_long_long_add(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        assert(false);
                    }
                    value = object_make_signed_int(computed_result);
                }
                else if (op == '-')
                {
                    const int computed_result = a - b;
                    signed long long exact_result;
                    if (signed_long_long_sub(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        assert(false);
                    }
                    value = object_make_signed_int(computed_result);
                }
                else if (op == '*')
                {
                    const int computed_result = a * b;
                    signed long long exact_result;
                    if (signed_long_long_mul(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        assert(false);
                    }
                    value = object_make_signed_int(computed_result);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else if (a == INT_MIN && b == -1)
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, new_expression->right->first_token, NULL, "integer overflow");
                        value = object_make_signed_int(INT_MIN); //this is what other compiler are doing
                    }
                    else
                        value = object_make_signed_int(a / b);
                }
                else if (op == '%')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else
                        value = object_make_signed_int(a % b);
                }
                //////////
                else if (op == '>')
                {
                    value = object_make_signed_int(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_int(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_int(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_int(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_int(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_int(a != b);
                }
            }
            break;

            case TYPE_UNSIGNED_INT:
            {
                unsigned int a = object_to_unsigned_int(&new_expression->left->object);
                unsigned int b = object_to_unsigned_int(&new_expression->right->object);

                if (op == '+')
                {
                    const unsigned int computed_result = a + b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_add(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        assert(false);
                    }
                    value = object_make_signed_int(computed_result);
                }
                else if (op == '-')
                {
                    const unsigned int computed_result = a - b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_sub(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        assert(false);
                    }
                    value = object_make_signed_int(computed_result);
                }
                else if (op == '*')
                {
                    const unsigned int computed_result = a * b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_mul(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        assert(false);
                    }
                    value = object_make_signed_int(computed_result);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else
                        value = object_make_unsigned_int(a / b);
                }
                else if (op == '%')
                {
                    if (b == 0)
                    {
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                        throw;
                    }

                    value = object_make_unsigned_int(a % b);
                }
                //////////                
                else if (op == '>')
                {
                    value = object_make_signed_int(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_int(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_int(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_int(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_int(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_int(a != b);
                }

            }
            break;

            case TYPE_SIGNED_LONG:
            {
                const signed long a = object_to_signed_long(&new_expression->left->object);
                const signed long b = object_to_signed_long(&new_expression->right->object);

                if (op == '+')
                {
                    const signed long computed_result = a + b;
                    signed long long exact_result;
                    if (signed_long_long_add(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }
                    value = object_make_signed_long(computed_result);
                }
                else if (op == '-')
                {
                    const signed long computed_result = a - b;
                    signed long long exact_result;
                    if (signed_long_long_sub(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }
                    value = object_make_signed_long(computed_result);
                }
                else if (op == '*')
                {
                    const signed long computed_result = a * b;
                    signed long long exact_result;
                    if (signed_long_long_mul(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }
                    value = object_make_signed_long(computed_result);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else if (a == LONG_MIN && b == -1)
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, new_expression->right->first_token, NULL, "integer overflow");
                        value = object_make_signed_long(LONG_MIN); //this is what compilers are doing
                    }
                    else
                        value = object_make_signed_long(a / b);
                }
                else if (op == '%')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else
                        value = object_make_signed_long(a % b);
                }
                //////////
                else if (op == '>')
                {
                    value = object_make_signed_long(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_long(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_long(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_long(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_long(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_long(a != b);
                }
            }
            break;

            case TYPE_UNSIGNED_LONG:
            {
                unsigned long a = object_to_unsigned_long(&new_expression->left->object);
                unsigned long b = object_to_unsigned_long(&new_expression->right->object);

                if (op == '+')
                {
                    const unsigned long computed_result = a + b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_add(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }
                    value = object_make_unsigned_long(computed_result);
                }
                else if (op == '-')
                {
                    const unsigned long computed_result = a - b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_sub(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }
                    value = object_make_unsigned_long(computed_result);
                }
                else if (op == '*')
                {
                    const unsigned long computed_result = a * b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_mul(&exact_result, a, b))
                    {
                        if (computed_result != exact_result)
                        {
                            compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%d'. Exactly result is '%lld'.", computed_result, exact_result);
                        }
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }

                    value = object_make_unsigned_long(computed_result);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else
                        value = object_make_unsigned_long(a / b);
                }
                else if (op == '%')
                {
                    if (b == 0)
                    {
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                        throw;
                    }

                    value = object_make_unsigned_long(a % b);
                }
                //////////                
                else if (op == '>')
                {
                    value = object_make_unsigned_long(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_unsigned_long(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_unsigned_long(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_unsigned_long(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_unsigned_long(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_unsigned_long(a != b);
                }

            }
            break;

            case TYPE_SIGNED_LONG_LONG:
            {
                long long a = object_to_signed_long_long(&new_expression->left->object);
                long long b = object_to_signed_long_long(&new_expression->right->object);

                if (op == '+')
                {
                    const long long computed_result = a + b;
                    signed long long exact_result;
                    if (!signed_long_long_add(&exact_result, a, b))
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%dll'. ", computed_result);
                    }
                    value = object_make_signed_long_long(computed_result);
                }
                else if (op == '-')
                {
                    const long long computed_result = a - b;
                    signed long long exact_result;
                    if (!signed_long_long_sub(&exact_result, a, b))
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%dll'.", computed_result);
                    }
                    value = object_make_signed_long_long(computed_result);
                }
                else if (op == '*')
                {
                    const long long computed_result = a * b;
                    signed long long exact_result;
                    if (!signed_long_long_mul(&exact_result, a, b))
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow results in '%dll", computed_result);
                    }
                    value = object_make_signed_long_long(computed_result);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else if (a == LLONG_MIN && b == -1)
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, new_expression->right->first_token, NULL, "integer overflow");
                        value = object_make_signed_long_long(LLONG_MIN); //this is what compilers are doing
                    }
                    else
                        value = object_make_signed_long_long(a / b);
                }
                else if (op == '%')
                {

                    if (b == 0)
                    {
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                        throw;
                    }

                    value = object_make_signed_long_long(a % b);
                }
                //////////                
                else if (op == '>')
                {
                    value = object_make_signed_int(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_int(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_int(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_int(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_int(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_int(a != b);
                }
            }
            break;

            case TYPE_UNSIGNED_LONG_LONG:
            {
                unsigned long long a = object_to_unsigned_long_long(&new_expression->left->object);
                unsigned long long b = object_to_unsigned_long_long(&new_expression->right->object);


                if (op == '+')
                {
                    //const unsigned long long computed_result = a + b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_add(&exact_result, a, b))
                    {
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }

                    value = object_make_unsigned_long_long(a + b);
                }
                else if (op == '-')
                {
                    //const unsigned long long computed_result = a - b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_sub(&exact_result, a, b))
                    {
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }

                    value = object_make_unsigned_long_long(a - b);
                }
                else if (op == '*')
                {
                    //const unsigned long long computed_result = a * b;
                    unsigned long long exact_result;
                    if (unsigned_long_long_mul(&exact_result, a, b))
                    {
                    }
                    else
                    {
                        compiler_diagnostic(W_INTEGER_OVERFLOW, ctx, NULL, &m, "integer overflow");
                    }
                    value = object_make_unsigned_long_long(a * b);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");


                    value = object_make_unsigned_long_long(a / b);
                }
                else if (op == '%')
                {
                    if (b == 0)
                    {
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                        throw;
                    }
                    value = object_make_unsigned_long_long(a % b);
                }
                //////////
                //////////                
                else if (op == '>')
                {
                    value = object_make_signed_int(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_int(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_int(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_int(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_int(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_int(a != b);
                }

            }
            break;

            case TYPE_BOOL:
            case TYPE_SIGNED_CHAR:
            case TYPE_UNSIGNED_CHAR:
            case TYPE_SIGNED_SHORT:
            case TYPE_UNSIGNED_SHORT:
                assert(false); //they are promoted
                throw;
                break;

            case TYPE_FLOAT:
            {
                float a = object_to_float(&new_expression->left->object);
                float b = object_to_float(&new_expression->right->object);


                if (op == '+')
                {
                    value = object_make_float(a + b);
                }
                else if (op == '-')
                {
                    value = object_make_float(a - b);
                }
                else if (op == '*')
                {
                    value = object_make_float(a * b);
                }
                else if (op == '/')
                {
                    if (b == 0)
                    {
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    }

                    value = object_make_float(a / b);
                }
                else if (op == '%')
                {
                    //error C2296: '%': not valid as left operand has type 'float'
                    compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "'%': not valid as left operand has type 'float'");
                    throw;
                    //r = a % b;
                }
                //////////                
                else if (op == '>')
                {
                    value = object_make_signed_int(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_int(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_int(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_int(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_int(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_int(a != b);
                }

            }
            break;
            case TYPE_DOUBLE:
            {
                double a = object_to_double(&new_expression->left->object);
                double b = object_to_double(&new_expression->right->object);

                if (op == '+')
                {
                    value = object_make_double(a + b);
                }
                else if (op == '-')
                {
                    value = object_make_double(a - b);
                }
                else if (op == '*')
                {
                    value = object_make_double(a * b);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else
                        value = object_make_double(a / b);
                }
                else if (op == '%')
                {
                    //value = object_make_double(r);
                    //error C2296: '%': not valid as left operand has type 'float'
                    compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "'%': not valid as left operand has type 'float'");
                    throw;
                    //r = a % b;
                }
                //////////                
                else if (op == '>')
                {
                    value = object_make_signed_int(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_int(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_int(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_int(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_int(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_int(a != b);
                }

            }
            break;
            case TYPE_LONG_DOUBLE:
            {
                long double a = object_to_long_double(&new_expression->left->object);
                long double b = object_to_long_double(&new_expression->right->object);

                if (op == '+')
                {
                    value = object_make_long_double(a + b);
                }
                else if (op == '-')
                {
                    value = object_make_long_double(a - b);
                }
                else if (op == '*')
                {
                    value = object_make_long_double(a * b);
                }
                else if (op == '/')
                {
                    if (b == 0)
                        compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "division by zero");
                    else
                        value = object_make_long_double(a / b);
                }
                else if (op == '%')
                {
                    //error C2296: '%': not valid as left operand has type 'float'
                    //value = object_make_long_double(a % b);
                    compiler_diagnostic(W_DIVIZION_BY_ZERO, ctx, new_expression->right->first_token, NULL, "'%': not valid as left operand has type 'float'");
                    //r = a % b;
                    throw;
                }
                //////////                
                else if (op == '>')
                {
                    value = object_make_signed_int(a > b);
                }
                else if (op == '<')
                {
                    value = object_make_signed_int(a < b);
                }
                else if (op == '>=')
                {
                    value = object_make_signed_int(a >= b);
                }
                else if (op == '<=')
                {
                    value = object_make_signed_int(a <= b);
                }
                //
                else if (op == '==')
                {
                    value = object_make_signed_int(a == b);
                }
                else if (op == '!=')
                {
                    value = object_make_signed_int(a != b);
                }

            }
            break;

            };


        }

        type_destroy(&common_type);
        *result = value;
        return 0;//ok
    }
    catch
    {
    }

    type_destroy(&common_type);

    struct object empty = { 0 };
    *result = empty;
    return 1; //error
}

struct expression* _Owner _Opt multiplicative_expression(struct parser_ctx* ctx)
{

    /*
     multiplicative-expression:
    cast-expression
    multiplicative-expression * cast-expression
    multiplicative-expression / cast-expression
    multiplicative-expression % cast-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        p_expression_node = cast_expression(ctx);
        if (p_expression_node == NULL)
        {
            throw;
        }

        while (ctx->current != NULL &&
               (ctx->current->type == '*' ||
                   ctx->current->type == '/' ||
                   ctx->current->type == '%'))
        {
            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
            {
                expression_delete(p_expression_node);
                p_expression_node = NULL;
                throw;
            }

            new_expression->first_token = ctx->current;
            const enum token_type op = ctx->current->type;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(new_expression);
                throw;
            }

            switch (op)
            {
            case '*':
                new_expression->expression_type = MULTIPLICATIVE_EXPRESSION_MULT;
                break;
            case '/':
                new_expression->expression_type = MULTIPLICATIVE_EXPRESSION_DIV;
                break;
            case '%':
                new_expression->expression_type = MULTIPLICATIVE_EXPRESSION_MOD;
                break;
            default:
                assert(false);
                break;
            }


            new_expression->left = p_expression_node;
            p_expression_node = NULL; // MOVED

            new_expression->right = cast_expression(ctx);

            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }
            new_expression->last_token = new_expression->right->last_token;


            if (op == '%')
            {
                /*The operands of the % operator shall have integer type*/
                if (!type_is_integer(&new_expression->left->type))
                {
                    compiler_diagnostic(C_ERROR_LEFT_IS_NOT_INTEGER,
                        ctx,
                        new_expression->left->first_token,
                        NULL,
                        "left is not an integer type");
                }

                if (!type_is_integer(&new_expression->right->type))
                {
                    compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_INTEGER,
                        ctx,
                        new_expression->right->first_token,
                        NULL,
                        "right is not an integer type");
                }
            }
            else
            {
                /*Each of the operands shall have arithmetic type.*/
                if (!type_is_arithmetic(&new_expression->left->type))
                {
                    compiler_diagnostic(C_ERROR_LEFT_IS_NOT_ARITHMETIC,
                        ctx,
                        new_expression->left->first_token,
                        NULL,
                        "left is not an arithmetic type");
                }

                if (!type_is_arithmetic(&new_expression->right->type))
                {
                    compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_ARITHMETIC,
                        ctx,
                        new_expression->right->first_token,
                        NULL,
                        "right is not an arithmetic type");
                }
            }
            new_expression->type = type_common(&new_expression->left->type, &new_expression->right->type);

            if (!ctx->evaluation_is_disabled && execute_arithmetic(ctx, new_expression, op, &new_expression->object) != 0)
            {
                expression_delete(new_expression);
                throw;
            }

            p_expression_node = new_expression;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }
    return p_expression_node;
}


struct expression* _Owner _Opt additive_expression(struct parser_ctx* ctx)
{
    /*
     additive-expression:
    multiplicative-expression
    additive-expression + multiplicative-expression
    additive-expression - multiplicative-expression
    */

    struct expression* _Owner _Opt p_expression_node = NULL;

    try
    {
        p_expression_node = multiplicative_expression(ctx);
        if (p_expression_node == NULL)
        {
            throw;
        }

        while (ctx->current != NULL &&
               (ctx->current->type == '+' ||
                   ctx->current->type == '-'))
        {
            struct token* operator_position = ctx->current;

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
            {
                compiler_diagnostic(C_ERROR_OUT_OF_MEM, ctx, ctx->current, NULL, "out of mem");
                throw;
            }

            new_expression->first_token = ctx->current;

            enum token_type op = ctx->current->type;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }

            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/


            new_expression->right = multiplicative_expression(ctx);
            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }

            new_expression->last_token = new_expression->right->last_token;



            if (!type_is_scalar(&new_expression->left->type) && !type_is_array(&new_expression->left->type))
            {
                compiler_diagnostic(C_ERROR_LEFT_IS_NOT_SCALAR, ctx, operator_position, NULL, "left operator is not scalar");
            }
            if (!type_is_scalar(&new_expression->right->type) && !type_is_array(&new_expression->right->type))
            {
                compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_SCALAR, ctx, operator_position, NULL, "right operator is not scalar");
            }

            const bool b_left_is_arithmetic = type_is_arithmetic(&new_expression->left->type);
            const bool b_right_is_arithmetic = type_is_arithmetic(&new_expression->right->type);

            const enum type_category left_category = type_get_category(&new_expression->left->type);
            const enum type_category right_category = type_get_category(&new_expression->right->type);

            if (op == '+')
            {
                new_expression->expression_type = ADDITIVE_EXPRESSION_PLUS;

                /*
                 For addition, either both operands shall have arithmetic type,
                 or one operand shall be a pointer to a complete object type and
                 the other shall have integer type. (Incrementing is equivalent to adding 1.)
                */
                if (b_left_is_arithmetic && b_right_is_arithmetic)
                {
                    new_expression->type = type_common(&new_expression->left->type, &new_expression->right->type);

                    if (!ctx->evaluation_is_disabled && execute_arithmetic(ctx, new_expression, op, &new_expression->object) != 0)
                    {
                        expression_delete(new_expression);
                        throw;
                    }
                }
                else
                {
                    if (left_category == TYPE_CATEGORY_POINTER || left_category == TYPE_CATEGORY_ARRAY /* || left_category == TYPE_CATEGORY_FUNCTION*/)
                    {
                        if (type_is_integer(&new_expression->right->type))
                        {
                            if (left_category == TYPE_CATEGORY_ARRAY)
                            {
                                struct type t = get_array_item_type(&new_expression->left->type);
                                new_expression->type = type_add_pointer(&t, ctx->options.null_checks_enabled);
                                //new_expression->type.type_qualifier_flags |= not_nu;
                                type_destroy(&t);
                            }
                            else
                            {
                                new_expression->type = type_dup(&new_expression->left->type);
                            }
                        }
                        else
                        {
                            compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_INTEGER, ctx, ctx->current, NULL, "expected integer type on right");
                        }
                    }
                    else if (right_category == TYPE_CATEGORY_POINTER || right_category == TYPE_CATEGORY_ARRAY /*|| right_category == TYPE_CATEGORY_FUNCTION*/)
                    {
                        if (type_is_integer(&new_expression->left->type))
                        {
                            if (right_category == TYPE_CATEGORY_ARRAY)
                            {
                                new_expression->type = get_array_item_type(&new_expression->right->type);
                            }
                            else
                            {
                                new_expression->type = type_dup(&new_expression->right->type);
                            }
                        }
                        else
                        {
                            compiler_diagnostic(C_ERROR_LEFT_IS_NOT_INTEGER, ctx, ctx->current, NULL, "expected integer type on left");
                        }
                    }
                    else
                    {
                        compiler_diagnostic(C_ERROR_INVALID_TYPE, ctx, ctx->current, NULL, "invalid types additive expression");
                    }
                }
            }
            else if (op == '-')
            {
                new_expression->expression_type = ADDITIVE_EXPRESSION_MINUS;

                /*
                 For subtraction, one of the following shall hold:
                    — both operands have arithmetic type;
                    — both operands are pointers to qualified or unqualified versions of compatible complete object
                    types; or
                    — the left operand is a pointer to a complete object type and the right operand has integer type.
                    (Decrementing is equivalent to subtracting 1.)
                */
                if (b_left_is_arithmetic && b_right_is_arithmetic)
                {
                    new_expression->type = type_common(&new_expression->left->type, &new_expression->right->type);

                    if (!ctx->evaluation_is_disabled && execute_arithmetic(ctx, new_expression, op, &new_expression->object) != 0)
                    {
                        expression_delete(new_expression);
                        throw;
                    }
                }
                else
                {
                    if (left_category == TYPE_CATEGORY_POINTER || left_category == TYPE_CATEGORY_ARRAY)
                    {
                        if (right_category == TYPE_CATEGORY_POINTER || right_category == TYPE_CATEGORY_ARRAY)
                        {
                            // — both operands are pointers to qualified or unqualified versions of compatible complete object
                            //  types;
                            struct type t1 = type_lvalue_conversion(&new_expression->left->type, ctx->options.null_checks_enabled);
                            struct type t2 = type_lvalue_conversion(&new_expression->right->type, ctx->options.null_checks_enabled);

                            if (!type_is_same(&t1, &t2, false))
                            {
                                compiler_diagnostic(C_ERROR_INCOMPATIBLE_POINTER_TYPES, ctx, ctx->current, NULL, "incompatible pointer types");
                            }

                            new_expression->type = type_make_int();
                            type_destroy(&t1);
                            type_destroy(&t2);
                        }
                        else
                        {
                            if (type_is_integer(&new_expression->right->type))
                            {
                                //- the left operand is a pointer to a complete object typeand the right operand has integer type.
                                new_expression->type = type_dup(&new_expression->left->type);
                            }
                            else
                            {
                                compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_INTEGER, ctx, ctx->current, NULL, "right must be integer type");
                            }
                        }
                    }
                    else
                    {
                        compiler_diagnostic(C_ERROR_INVALID_TYPE, ctx, ctx->current, NULL, "invalid types for operator -");
                    }
                }
            }

            p_expression_node = new_expression;
            new_expression = NULL; /*MOVED*/
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

struct expression* _Owner _Opt shift_expression(struct parser_ctx* ctx)
{
    /*
     shift-expression:
       additive-expression
       shift-expression << additive-expression
       shift-expression >> additive-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        p_expression_node = additive_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '>>' ||
                   ctx->current->type == '<<'))
        {
            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL) throw;

            new_expression->first_token = ctx->current;

            enum token_type op = ctx->current->type;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }

            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            new_expression->right = multiplicative_expression(ctx);
            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }

            new_expression->last_token = new_expression->right->last_token;

            if (op == '>>')
            {
                new_expression->expression_type = SHIFT_EXPRESSION_RIGHT;
            }
            else if (op == '<<')
            {
                new_expression->expression_type = SHIFT_EXPRESSION_LEFT;
            }
            if (execute_bitwise_operator(ctx, new_expression, op) != 0)
            {
                expression_delete(new_expression);
                throw;
            }


            p_expression_node = new_expression;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

struct expression* _Owner _Opt relational_expression(struct parser_ctx* ctx)
{
    /*
    relational-expression:
        shift-expression
        relational-expression < shift-expression
        relational-expression > shift-expression
        relational-expression <= shift-expression
        relational-expression >= shift-expression
    */

    struct expression* _Owner _Opt p_expression_node = NULL;
    struct expression* _Owner _Opt new_expression = NULL;
    try
    {
        p_expression_node = shift_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '>' ||
                   ctx->current->type == '<' ||
                   ctx->current->type == '>=' ||
                   ctx->current->type == '<='))
        {
            assert(new_expression == NULL);
            new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
            {
                throw;
            }
            new_expression->first_token = ctx->current;
            enum token_type op = ctx->current->type;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            new_expression->right = shift_expression(ctx);
            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }

            new_expression->last_token = new_expression->right->last_token;
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(new_expression);
                new_expression = NULL;
                throw;
            }

            check_comparison(ctx,
              new_expression->left,
              new_expression->right,
              ctx->current);

            if (op == '>')
            {
                new_expression->expression_type = RELATIONAL_EXPRESSION_BIGGER_THAN;
            }
            else if (op == '<')
            {
                new_expression->expression_type = RELATIONAL_EXPRESSION_LESS_THAN;
            }
            else if (op == '>=')
            {
                new_expression->expression_type = RELATIONAL_EXPRESSION_BIGGER_OR_EQUAL_THAN;
            }
            else if (op == '<=')
            {
                new_expression->expression_type = RELATIONAL_EXPRESSION_LESS_OR_EQUAL_THAN;
            }

            //Each of the operands shall have arithmetic type
            if (type_is_arithmetic(&new_expression->left->type) &&
                type_is_arithmetic(&new_expression->right->type))
            {
                new_expression->type = type_common(&new_expression->left->type, &new_expression->right->type);

                if (!ctx->evaluation_is_disabled && execute_arithmetic(ctx, new_expression, op, &new_expression->object) != 0)
                {
                    expression_delete(new_expression);
                    new_expression = NULL;
                    throw;
                }
            }

            type_destroy(&new_expression->type);
            new_expression->type = type_make_int_bool_like();

            p_expression_node = new_expression;
            new_expression = NULL; /*MOVED*/
        }
    }
    catch
    {
        expression_delete(new_expression);
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

void check_diferent_enuns(struct parser_ctx* ctx,
                                 const struct token* operator_token,
                                 const struct expression* left,
                                 const struct expression* right,
                                 const char* message)
{
    if (left->type.type_specifier_flags & TYPE_SPECIFIER_ENUM &&
        right->type.type_specifier_flags & TYPE_SPECIFIER_ENUM)
    {
        assert(left->type.enum_specifier);
        assert(right->type.enum_specifier);

        if (get_complete_enum_specifier(left->type.enum_specifier) !=
            get_complete_enum_specifier(right->type.enum_specifier))
        {
            assert(left->type.enum_specifier != NULL);
            assert(right->type.enum_specifier != NULL);

            const char* lefttag = "";
            if (left->type.enum_specifier->tag_token)
                lefttag = left->type.enum_specifier->tag_token->lexeme;

            const char* righttag = "";
            if (right->type.enum_specifier->tag_token)
                righttag = right->type.enum_specifier->tag_token->lexeme;

            char finalmessage[200] = { 0 };
            snprintf(finalmessage,
                sizeof finalmessage,
                "%s (enum %s, enum %s)",
                message,
                lefttag,
                righttag);

            compiler_diagnostic(W_ENUN_CONVERSION,
                                        ctx,
                                        operator_token, NULL,
                                        finalmessage,
                                        lefttag,
                                        righttag);
        }
    }
}
void expression_evaluate_equal_not_equal(const struct expression* left,
                                         const struct expression* right,
                                         struct expression* result,
                                         int op,
                                         bool disabled)
{
    assert(op == '==' || op == '!=');
    //result->object =
        //object_op(&left->object, &right->object, op);
}

struct expression* _Owner _Opt equality_expression(struct parser_ctx* ctx)
{
    /*
      equality-expression:
       relational-expression
       equality-expression == relational-expression
       equality-expression != relational-expression
    */
    /*
    * Equality operators
    One of the following shall hold:
    — both operands have arithmetic type;
    — both operands are pointers to qualified or unqualified versions of compatible types;
    — one operand is a pointer to an object type and the other is a pointer to a qualified or unqualified
    version of void; or
    — one operand is a pointer and the other is a null pointer constant.
    */
    struct expression* _Owner _Opt  p_expression_node = NULL;
    struct expression* _Owner _Opt new_expression = NULL;

    try
    {
        p_expression_node = relational_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '==' ||
                   ctx->current->type == '!='))
        {
            assert(new_expression == NULL);
            new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
                throw;

            new_expression->first_token = ctx->current;

            struct token* operator_token = ctx->current;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            if (operator_token->type == '==')
                new_expression->expression_type = EQUALITY_EXPRESSION_EQUAL;
            else
                new_expression->expression_type = EQUALITY_EXPRESSION_NOT_EQUAL;

            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            new_expression->right = relational_expression(ctx);
            if (new_expression->right == NULL)
                throw;

            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            check_comparison(ctx,
              new_expression->left,
              new_expression->right,
              ctx->current);

            new_expression->last_token = new_expression->right->last_token;
            new_expression->first_token = operator_token;

            //TODO missing general checks described in standard
            //like pointer, nullptr etc...
            if (type_is_arithmetic(&new_expression->left->type) &&
                type_is_arithmetic(&new_expression->right->type))
            {
                if (!ctx->evaluation_is_disabled && execute_arithmetic(ctx, new_expression, operator_token->type, &new_expression->object) != 0)
                {
                    throw;
                }
            }

            new_expression->type = type_make_int_bool_like();
            p_expression_node = new_expression;
            new_expression = NULL; /*MOVED*/
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    expression_delete(new_expression);
    return p_expression_node;
}

struct expression* _Owner _Opt and_expression(struct parser_ctx* ctx)
{
    /*
     AND-expression:
      equality-expression
      AND-expression & equality-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    struct expression* _Owner _Opt new_expression = NULL;

    try
    {
        p_expression_node = equality_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL && ctx->current->type == '&')
        {
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            assert(new_expression == NULL);
            new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
                throw;

            new_expression->first_token = ctx->current;
            new_expression->expression_type = AND_EXPRESSION;
            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            new_expression->right = equality_expression(ctx);
            if (new_expression->right == NULL)
                throw;

            new_expression->last_token = new_expression->right->last_token;

            if (execute_bitwise_operator(ctx, new_expression, '&') != 0)
                throw;

            p_expression_node = new_expression;
            new_expression = NULL; /*MOVED*/
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    expression_delete(new_expression);
    return p_expression_node;
}

struct expression* _Owner _Opt  exclusive_or_expression(struct parser_ctx* ctx)
{
    /*
     exclusive-OR-expression:
      AND-expression
      exclusive-OR-expression ^ AND-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    struct expression* _Owner _Opt new_expression = NULL;

    try
    {
        p_expression_node = and_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '^'))
        {
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            assert(new_expression == NULL);
            new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
                throw;

            new_expression->first_token = ctx->current;
            new_expression->expression_type = EXCLUSIVE_OR_EXPRESSION;
            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            new_expression->right = and_expression(ctx);
            if (new_expression->right == NULL)
                throw;

            new_expression->last_token = new_expression->right->last_token;

            if (execute_bitwise_operator(ctx, new_expression, '^') != 0)
                throw;

            p_expression_node = new_expression;
            new_expression = NULL;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    expression_delete(new_expression);
    return p_expression_node;
}


NODISCARD
static errno_t execute_bitwise_operator(struct parser_ctx* ctx, struct expression* new_expression, int op)
{
    try
    {
        switch (op)
        {
        case '&':
        case '^':
        case '|':
        case '>>':
        case '<<':
            break;
        default:
            assert(false);
            throw;
        }


        //Each of the operands shall have integer type.
        if (!type_is_integer(&new_expression->left->type))
        {
            compiler_diagnostic(C_ERROR_LEFT_IS_NOT_INTEGER, ctx, ctx->current, NULL, "left type must be an integer type");
            throw;
        }

        if (!type_is_integer(&new_expression->right->type))
        {
            compiler_diagnostic(C_ERROR_LEFT_IS_NOT_INTEGER, ctx, ctx->current, NULL, "right type must be an integer type");
            throw;
        }

        type_destroy(&new_expression->type);
        new_expression->type = type_common(&new_expression->left->type, &new_expression->right->type);

        if (!ctx->evaluation_is_disabled &&
            object_has_constant_value(&new_expression->left->object) &&
            object_has_constant_value(&new_expression->right->object))
        {
            enum object_value_type vt = type_to_object_type(&new_expression->type);
            switch (vt)
            {
            case TYPE_SIGNED_INT:
            {
                int a = object_to_signed_int(&new_expression->left->object);
                int b = object_to_signed_int(&new_expression->right->object);

                int r = 0;
                if (op == '|')
                    r = a | b;
                else if (op == '^')
                    r = a ^ b;
                else if (op == '&')
                    r = a & b;
                //
                else if (op == '>>')
                    r = a >> b;
                else if (op == '<<')
                    r = a << b;

                object_destroy(&new_expression->object);
                new_expression->object = object_make_signed_int(r);
            }
            break;
            case TYPE_UNSIGNED_INT:
            {
                unsigned int a = object_to_unsigned_int(&new_expression->left->object);
                unsigned int b = object_to_unsigned_int(&new_expression->right->object);
                int r = 0;
                if (op == '|')
                    r = a | b;
                else if (op == '^')
                    r = a ^ b;
                else if (op == '&')
                    r = a & b;
                //
                else if (op == '>>')
                    r = a >> b;
                else if (op == '<<')
                    r = a << b;

                object_destroy(&new_expression->object);
                new_expression->object = object_make_unsigned_int(r);
            }
            break;

            case TYPE_SIGNED_LONG:
            {
                signed long a = object_to_signed_long(&new_expression->left->object);
                signed long b = object_to_signed_long(&new_expression->right->object);

                int r = 0;
                if (op == '|')
                    r = a | b;
                else if (op == '^')
                    r = a ^ b;
                else if (op == '&')
                    r = a & b;
                //
                else if (op == '>>')
                    r = a >> b;
                else if (op == '<<')
                    r = a << b;

                object_destroy(&new_expression->object);
                new_expression->object = object_make_signed_long(r);
            }
            break;

            case TYPE_UNSIGNED_LONG:
            {
                unsigned long a = object_to_unsigned_long(&new_expression->left->object);
                unsigned long b = object_to_unsigned_long(&new_expression->right->object);
                int r = 0;
                if (op == '|')
                    r = a | b;
                else if (op == '^')
                    r = a ^ b;
                else if (op == '&')
                    r = a & b;
                //
                else if (op == '>>')
                    r = a >> b;
                else if (op == '<<')
                    r = a << b;

                object_destroy(&new_expression->object);
                new_expression->object = object_make_unsigned_long(r);
            }
            break;


            case TYPE_SIGNED_LONG_LONG:
            {
                signed long long a = object_to_signed_long_long(&new_expression->left->object);
                signed long long b = object_to_signed_long_long(&new_expression->right->object);
                signed long long r = 0;
                if (op == '|')
                    r = a | b;
                else if (op == '^')
                    r = a ^ b;
                else if (op == '&')
                    r = a & b;
                //
                else if (op == '>>')
                    r = a >> b;
                else if (op == '<<')
                    r = a << b;

                object_destroy(&new_expression->object);
                new_expression->object = object_make_signed_long_long(r);

            }
            break;

            case TYPE_UNSIGNED_LONG_LONG:
            {
                unsigned long long a = object_to_unsigned_long_long(&new_expression->left->object);
                unsigned long long b = object_to_unsigned_long_long(&new_expression->right->object);
                unsigned long long r = 0;
                if (op == '|')
                    r = a | b;
                else if (op == '^')
                    r = a ^ b;
                else if (op == '&')
                    r = a & b;
                //
                else if (op == '>>')
                    r = a >> b;
                else if (op == '<<')
                    r = a << b;

                object_destroy(&new_expression->object);
                new_expression->object = object_make_unsigned_long_long(r);
            }
            break;

            case TYPE_BOOL:
            case TYPE_SIGNED_CHAR:
            case TYPE_UNSIGNED_CHAR:
            case TYPE_SIGNED_SHORT:
            case TYPE_UNSIGNED_SHORT:
                assert(false); //they are promoted
                throw;
                break;


            case TYPE_FLOAT:
            case TYPE_DOUBLE:
            case TYPE_LONG_DOUBLE:
                assert(false); //works for integers only
                throw;
                break;
            };
        }
        return 0;//ok
    }
    catch
    {
    }
    return 1; //error
}

struct expression* _Owner _Opt inclusive_or_expression(struct parser_ctx* ctx)
{
    /*
    inclusive-OR-expression:
        exclusive-OR-expression
        inclusive-OR-expression | exclusive-OR-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        p_expression_node = exclusive_or_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '|'))
        {
            struct token* operator_token = ctx->current;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
                throw;


            new_expression->first_token = ctx->current;
            new_expression->expression_type = INCLUSIVE_OR_EXPRESSION;
            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            new_expression->right = exclusive_or_expression(ctx);
            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }

            check_diferent_enuns(ctx,
                                operator_token,
                                new_expression->left,
                                new_expression->right,
                                "operator '|' between enumerations of different types.");

            new_expression->last_token = new_expression->right->last_token;

            if (execute_bitwise_operator(ctx, new_expression, '|') != 0)
            {
                expression_delete(new_expression);
                throw;
            }

            p_expression_node = new_expression;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

struct expression* _Owner _Opt logical_and_expression(struct parser_ctx* ctx)
{
    /*
    logical-AND-expression:
     inclusive-OR-expression
     logical-AND-expression && inclusive-OR-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        p_expression_node = inclusive_or_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '&&'))
        {
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
                throw;

            new_expression->first_token = ctx->current;
            new_expression->expression_type = LOGICAL_AND_EXPRESSION;
            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            bool right_evaluation_is_disabled = false;

            if (object_has_constant_value(&new_expression->left->object))
            {
                if (!object_to_bool(&new_expression->left->object))
                {
                    right_evaluation_is_disabled = true;
                }
            }

            const bool old_evaluation_is_disabled = ctx->evaluation_is_disabled;
            ctx->evaluation_is_disabled = right_evaluation_is_disabled;

            new_expression->right = inclusive_or_expression(ctx);

            ctx->evaluation_is_disabled = old_evaluation_is_disabled; //restore

            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }
            new_expression->last_token = new_expression->right->last_token;

            if (!ctx->evaluation_is_disabled)
            {
                if (object_has_constant_value(&new_expression->left->object))
                {
                    bool a = object_to_bool(&new_expression->left->object);
                    if (a == 0)
                    {
                        // 0 && something
                        new_expression->object = object_make_signed_int(0);
                    }
                    else
                    {
                        // 1 && something
                        if (object_has_constant_value(&new_expression->right->object))
                        {
                            bool b = object_to_bool(&new_expression->right->object);
                            new_expression->object = object_make_signed_int(a && b);
                        }
                    }
                }
            }

            //Each of the operands shall have scalar type
            if (!type_is_scalar(&new_expression->left->type))
            {
                expression_delete(new_expression);
                compiler_diagnostic(C_ERROR_LEFT_IS_NOT_SCALAR, ctx, ctx->current, NULL, "left type is not scalar for or expression");
                throw;
            }

            if (!type_is_scalar(&new_expression->right->type))
            {
                expression_delete(new_expression);
                compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_SCALAR, ctx, ctx->current, NULL, "right type is not scalar for or expression");
                throw;
            }

            //The result has type int
            new_expression->type = type_make_int_bool_like();

            p_expression_node = new_expression;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

struct expression* _Owner _Opt logical_or_expression(struct parser_ctx* ctx)
{
    /*
      logical-OR-expression:
       logical-AND-expression
       logical-OR-expression || logical-AND-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {


        p_expression_node = logical_and_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '||'))
        {
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
                throw;

            new_expression->first_token = ctx->current;
            new_expression->expression_type = LOGICAL_OR_EXPRESSION;
            new_expression->left = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            bool right_evaluation_is_disabled = false;

            if (object_has_constant_value(&new_expression->left->object))
            {
                if (object_to_bool(&new_expression->left->object))
                {
                    /*
                      If the first operand compares unequal to 0, the second operand is not evaluated.
                    */
                    right_evaluation_is_disabled = true;
                }
            }

            const bool old_evaluation_is_disabled = ctx->evaluation_is_disabled;
            ctx->evaluation_is_disabled = right_evaluation_is_disabled;

            new_expression->right = logical_and_expression(ctx);

            ctx->evaluation_is_disabled = old_evaluation_is_disabled; //restore

            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }

            new_expression->last_token = new_expression->right->last_token;

            if (!ctx->evaluation_is_disabled)
            {
                if (object_has_constant_value(&new_expression->left->object))
                {
                    bool a = object_to_bool(&new_expression->left->object);
                    if (a == 1)
                    {
                        // 1 || something
                        new_expression->object = object_make_signed_int(1);
                    }
                    else
                    {
                        // 0 || something
                        if (object_has_constant_value(&new_expression->right->object))
                        {
                            bool b = object_to_bool(&new_expression->right->object);
                            new_expression->object = object_make_signed_int(a || b);
                        }
                    }
                }
            }


            //Each of the operands shall have scalar type
            if (!type_is_scalar(&new_expression->left->type))
            {
                expression_delete(new_expression);
                compiler_diagnostic(C_ERROR_LEFT_IS_NOT_SCALAR, ctx, ctx->current, NULL, "left type is not scalar for or expression");
                throw;
            }

            if (!type_is_scalar(&new_expression->right->type))
            {
                expression_delete(new_expression);
                compiler_diagnostic(C_ERROR_RIGHT_IS_NOT_SCALAR, ctx, ctx->current, NULL, "right type is not scalar for or expression");
                throw;
            }

            //The result has type int
            new_expression->type = type_make_int_bool_like();

            p_expression_node = new_expression;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

struct expression* _Owner _Opt assignment_expression(struct parser_ctx* ctx)
{
    /*
    assignment-expression:
       conditional-expression
       unary-expression assignment-operator assignment-expression
       */
       /*
          assignment-operator: one of
          = *= /= %= += -= <<= >>= &= ^= |=
       */
       // aqui eh duvidoso mas conditional faz a unary tb.
       // a diferenca q nao eh qualquer expressao
       // que pode ser de atribuicao
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        p_expression_node = conditional_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        assert(p_expression_node->expression_type != EXPRESSION_TYPE_INVALID);

        while (ctx->current != NULL &&
               (ctx->current->type == '=' ||
                   ctx->current->type == '*=' ||
                   ctx->current->type == '/=' ||
                   ctx->current->type == '%=' ||
                   ctx->current->type == '+=' ||
                   ctx->current->type == '-=' ||
                   ctx->current->type == '<<=' ||
                   ctx->current->type == '>>=' ||
                   ctx->current->type == '&=' ||
                   ctx->current->type == '^=' ||
                   ctx->current->type == '|='))
        {

            const struct token* const op_token = ctx->current;
            parser_match(ctx);
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            struct expression* _Owner _Opt new_expression = calloc(1, sizeof * new_expression);
            if (new_expression == NULL)
                throw;

            new_expression->first_token = ctx->current;

            switch (op_token->type)
            {
            case '=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_ASSIGN;
                break;
            case '*=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_MULTI_ASSIGN;
                break;
            case '/=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_DIV_ASSIGN;
                break;
            case '%=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_MOD_ASSIGN;
                break;
            case '+=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_PLUS_ASSIGN;
                break;
            case '-=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_MINUS_ASSIGN;
                break;
            case '<<=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_SHIFT_LEFT_ASSIGN;
                break;
            case '>>=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_SHIFT_RIGHT_ASSIGN;
                break;
            case '&=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_AND_ASSIGN;
                break;
            case '^=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_NOT_ASSIGN;
                break;
            case '|=':
                new_expression->expression_type = ASSIGNMENT_EXPRESSION_OR_ASSIGN;
                break;
            default:
                assert(false);
                break;
            }


            new_expression->left = p_expression_node;
            p_expression_node = NULL; // MOVED

            const struct marker left_operand_marker = {
                      .p_token_begin = new_expression->left->first_token,
                      .p_token_end = new_expression->left->last_token,
            };

            if (type_is_function(&new_expression->left->type))
            {
                compiler_diagnostic(C_ERROR_ASSIGNMENT_OF_FUNCTION,
                    ctx,
                    NULL,
                    &left_operand_marker,
                    "assignment of function");
            }
            else if (type_is_array(&new_expression->left->type))
            {
                if (new_expression->left->type.storage_class_specifier_flags & STORAGE_SPECIFIER_PARAMETER)
                {
                    /*
                      assignment of array parameter
                    */
                    compiler_diagnostic(W_ASSIGNMENT_OF_ARRAY_PARAMETER,
                        ctx,
                        NULL,
                        &left_operand_marker,
                        "assignment to array parameter");
                }
                else
                {
                    compiler_diagnostic(C_ERROR_ASSIGNMENT_TO_EXPRESSION_WITH_ARRAY_TYPE,
                        ctx,
                        NULL,
                        &left_operand_marker,
                        "assignment to expression with array type");
                }
            }

            if (type_is_const(&new_expression->left->type))
            {
                compiler_diagnostic(C_ERROR_ASSIGNMENT_OF_READ_ONLY_OBJECT,
                    ctx,
                    NULL,
                    &left_operand_marker,
                    "assignment of read-only object");
            }


            if (!expression_is_lvalue(new_expression->left))
            {
                compiler_diagnostic(C_ERROR_OPERATOR_NEEDS_LVALUE,
                                            ctx,
                                            NULL,
                                            &left_operand_marker,
                                            "lvalue required as left operand of assignment");
            }

            new_expression->right = assignment_expression(ctx);
            if (new_expression->right == NULL)
            {
                expression_delete(new_expression);
                throw;
            }

            if (op_token->type == '=')
            {
                check_assigment(ctx, &new_expression->left->type, new_expression->right, ASSIGMENT_TYPE_OBJECTS);
            }

            new_expression->last_token = new_expression->right->last_token;

            new_expression->type = type_dup(&new_expression->left->type);

            new_expression->type.storage_class_specifier_flags &= ~STORAGE_SPECIFIER_FUNCTION_RETURN;
            new_expression->type.storage_class_specifier_flags &= ~STORAGE_SPECIFIER_FUNCTION_RETURN_NODISCARD;

            check_diferent_enuns(ctx,
                op_token,
                new_expression->left,
                new_expression->right,
                "assignment of different enums.");

            new_expression->left->is_assignment_expression = true;
            if (new_expression->left->left)
                new_expression->left->left->is_assignment_expression = true;
            p_expression_node = new_expression;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    return p_expression_node;
}

void argument_expression_list_push(struct argument_expression_list* list, struct argument_expression* _Owner pitem)
{
    if (list->head == NULL)
    {
        list->head = pitem;
    }
    else
    {
        assert(list->tail != NULL);
        assert(list->tail->next == NULL);
        list->tail->next = pitem;
    }
    list->tail = pitem;
}

void argument_expression_list_destroy(_Dtor struct argument_expression_list* p)
{
    struct argument_expression* _Owner _Opt item = p->head;
    while (item)
    {
        struct argument_expression* _Owner _Opt next = item->next;
        item->next = NULL;
        argument_expression_delete(item);
        item = next;
    }
}

bool expression_is_malloc(const struct expression* p)
{
    if (p->expression_type == POSTFIX_FUNCTION_CALL &&
        p->left &&
        p->left->declarator &&
        p->left->declarator->name_opt)
    {
        if (strcmp(p->left->declarator->name_opt->lexeme, "malloc") == 0)
            return true;
    }
    return false;
}

bool expression_is_calloc(const struct expression* p)
{
    if (p->expression_type == POSTFIX_FUNCTION_CALL &&
        p->left &&
        p->left->declarator &&
        p->left->declarator->name_opt)
    {
        if (strcmp(p->left->declarator->name_opt->lexeme, "calloc") == 0)
            return true;
    }
    return false;
}

void expression_delete(struct expression* _Owner _Opt p)
{
    if (p)
    {
        expression_delete(p->condition_expr);
        compound_statement_delete(p->compound_statement);
        type_name_delete(p->type_name);

        expression_delete(p->right);
        expression_delete(p->left);
        braced_initializer_delete(p->braced_initializer);
        generic_selection_delete(p->generic_selection);
        type_destroy(&p->type);
        argument_expression_list_destroy(&p->argument_expression_list);

        //explodindo
        //object_destroy(&p->object);

        free(p);
    }
}
struct expression* _Owner _Opt expression(struct parser_ctx* ctx)
{
    /*expression:
      assignment-expression
      expression, assignment-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        p_expression_node = assignment_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        if (ctx->current->type == ',')
        {
            while (ctx->current->type == ',')
            {
                parser_match(ctx);
                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    throw;
                }

                struct expression* _Owner _Opt p_expression_node_new = calloc(1, sizeof * p_expression_node_new);
                if (p_expression_node_new == NULL)
                    throw;

                p_expression_node_new->first_token = ctx->current;
                p_expression_node_new->expression_type = EXPRESSION_EXPRESSION;
                p_expression_node_new->left = p_expression_node;
                p_expression_node = NULL; /*MOVED*/

                p_expression_node_new->right = expression(ctx);
                if (p_expression_node_new->right == NULL)
                {
                    expression_delete(p_expression_node_new);
                    throw;
                }
                p_expression_node_new->left->last_token = p_expression_node_new->right->last_token;

                p_expression_node = p_expression_node_new;

                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    throw;
                }
            }

            /*same type of the last expression*/
            type_destroy(&p_expression_node->type);
            p_expression_node->type = type_dup(&p_expression_node->right->type);
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }
    return p_expression_node;
}

bool is_first_of_conditional_expression(struct parser_ctx* ctx)
{
    return is_first_of_unary_expression(ctx) ||
        is_first_of_primary_expression(ctx);
}

bool expression_is_one(const struct expression* expression)
{
    if (expression->expression_type == PRIMARY_EXPRESSION_NUMBER)
    {
        return (object_has_constant_value(&expression->object) &&
            object_to_signed_int(&expression->object) == 1);
    }
    return false;
}

bool expression_is_zero(const struct expression* expression)
{
    if (expression->expression_type == PRIMARY_EXPRESSION_NUMBER)
    {
        return (object_has_constant_value(&expression->object) &&
            object_to_signed_int(&expression->object) == 0);
    }
    return false;
}

bool expression_is_null_pointer_constant(const struct expression* expression)
{
    /*
      An integer constant expression with the value 0,
      such an expression cast to type void *, or the
      predefined constant nullptr is called a null pointer constant.57)
    */
    if (type_is_integer(&expression->type) &&
        object_has_constant_value(&expression->object) &&
        object_to_signed_int(&expression->object) == 0)
    {
        return true;
    }
    if (type_is_void_ptr(&expression->type) &&
        object_has_constant_value(&expression->object) &&
        object_to_signed_int(&expression->object) == 0)
    {
        return true;
    }

    if (type_is_nullptr_t(&expression->type))
    {
        return true;
    }

    return false;
}

struct expression* _Owner _Opt conditional_expression(struct parser_ctx* ctx)
{
    /*
      conditional-expression:
      logical-OR-expression
      logical-OR-expression ? expression : conditional-expression
    */
    struct expression* _Owner _Opt p_expression_node = NULL;
    struct type left_type = { 0 };
    struct type right_type = { 0 };
    try
    {
        p_expression_node = logical_or_expression(ctx);
        if (p_expression_node == NULL)
            throw;

        if (ctx->current && ctx->current->type == '?')
        {
            struct expression* _Owner _Opt p_conditional_expression = calloc(1, sizeof(struct expression));
            if (p_conditional_expression == NULL) throw;


            p_conditional_expression->first_token = ctx->current;
            p_conditional_expression->expression_type = CONDITIONAL_EXPRESSION;
            p_conditional_expression->condition_expr = p_expression_node;
            p_expression_node = NULL; /*MOVED*/

            parser_match(ctx); //?
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(p_conditional_expression);
                throw;
            }

            bool constant_expression_is_true = false;
            bool has_constant_expression = false;

            if (object_has_constant_value(&p_conditional_expression->condition_expr->object))
            {
                has_constant_expression = true;
                if (object_to_bool(&p_conditional_expression->condition_expr->object))
                {
                    constant_expression_is_true = true;
                }
            }

            const bool old_evaluation_is_disabled = ctx->evaluation_is_disabled;
            ctx->evaluation_is_disabled = has_constant_expression && !constant_expression_is_true;

            struct expression* _Owner _Opt p_left = expression(ctx);

            //restore original state (before throw)
            ctx->evaluation_is_disabled = old_evaluation_is_disabled;


            if (p_left == NULL)
            {
                expression_delete(p_conditional_expression);
                throw;
            }
            p_conditional_expression->left = p_left;


            if (parser_match_tk(ctx, TK_COLON) != 0)
            {
                unexpected_end_of_file(ctx);
                expression_delete(p_conditional_expression);
                throw;
            }

            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                expression_delete(p_conditional_expression);
                throw;
            }

            ctx->evaluation_is_disabled = has_constant_expression && constant_expression_is_true;

            struct expression* _Owner _Opt p_right = conditional_expression(ctx);
            //restore original state (before throw)
            ctx->evaluation_is_disabled = old_evaluation_is_disabled;

            if (p_right == NULL)
            {
                expression_delete(p_conditional_expression);
                throw;
            }
            p_conditional_expression->right = p_right;

            if (object_has_constant_value(&p_conditional_expression->condition_expr->object))
            {
                if (object_to_bool(&p_conditional_expression->condition_expr->object))
                {
                    p_conditional_expression->object = object_make_reference(&p_conditional_expression->left->object);
                }
                else
                {
                    p_conditional_expression->object = object_make_reference(&p_conditional_expression->right->object);
                }
            }

            if (expression_is_subjected_to_lvalue_conversion(p_conditional_expression->left))
            {
                left_type = type_lvalue_conversion(&p_conditional_expression->left->type, ctx->options.null_checks_enabled);
            }
            else
            {
                left_type = type_dup(&p_conditional_expression->left->type);
            }

            if (expression_is_subjected_to_lvalue_conversion(p_conditional_expression->right))
            {
                right_type = type_lvalue_conversion(&p_conditional_expression->right->type, ctx->options.null_checks_enabled);
            }
            else
            {
                right_type = type_dup(&p_conditional_expression->right->type);
            }

            /*The first operand shall have scalar type*/
            if (!type_is_scalar(&p_conditional_expression->condition_expr->type))
            {
                compiler_diagnostic(C_ERROR_CONDITION_MUST_HAVE_SCALAR_TYPE, ctx, ctx->current, NULL, "condition must have scalar type");
            }
            else if (type_is_arithmetic(&left_type) &&
                     type_is_arithmetic(&right_type))
            {
                /*
                 *  both operands have arithmetic type;
                */
                type_destroy(&p_conditional_expression->type);
                p_conditional_expression->type = type_common(&left_type, &right_type);
            }
            else if (type_is_struct_or_union(&left_type) && type_is_struct_or_union(&right_type))
            {
                /*
                 *  both operands have compatible structure or union type;
                 */
                if (!type_is_same(&left_type, &right_type, true))
                {
                    compiler_diagnostic(C_ERROR_INCOMPATIBLE_TYPES, ctx, p_conditional_expression->condition_expr->first_token, NULL, "incompatible types");
                }
                type_swap(&p_conditional_expression->type, &right_type);
            }
            else if (type_is_void(&left_type) && type_is_void(&right_type))
            {
                /*
                 *  both operands have void type;
                 */
                type_swap(&p_conditional_expression->type, &right_type);
            }
            else if (type_is_nullptr_t(&left_type) && type_is_nullptr_t(&right_type))
            {
                /* both operands have nullptr_t type;*/
                type_swap(&p_conditional_expression->type, &right_type);
            }
            else if (type_is_pointer(&left_type))
            {
                /*
                — one operand is a pointer and the other is a null pointer constant or has type nullptr_t; or
                    — one operand is a pointer to an object type and the other is a pointer to a qualified or unqualified
                    version of void
            */
                if (expression_is_null_pointer_constant(p_conditional_expression->right) ||
                    type_is_nullptr_t(&right_type) ||
                    type_is_void_ptr(&right_type))
                {
                    type_swap(&p_conditional_expression->type, &left_type);
                }
                else if (type_is_pointer(&right_type))
                {
                    if (type_is_nullptr_t(&left_type) || type_is_void_ptr(&left_type))
                    {
                        type_swap(&p_conditional_expression->type, &left_type);
                    }
                    else if (!type_is_same(&left_type, &right_type, false))
                    {
                        // type_print(&left_type);
                        // type_print(&right_type);
                        compiler_diagnostic(C_ERROR_INCOMPATIBLE_TYPES, ctx, ctx->current, NULL, "incompatible types");
                    }
                    else
                    {
                        type_swap(&p_conditional_expression->type, &right_type);
                    }
                }
                else
                {
                    compiler_diagnostic(C_ERROR_INCOMPATIBLE_TYPES, ctx, p_conditional_expression->condition_expr->first_token, NULL, "incompatible types");
                }
            }
            else if (type_is_pointer(&right_type))
            {
                if (expression_is_null_pointer_constant(p_conditional_expression->left) ||
                    type_is_nullptr_t(&left_type) ||
                    type_is_void_ptr(&left_type))
                {
                    type_swap(&p_conditional_expression->type, &right_type);
                }
                else if (type_is_pointer(&left_type))
                {
                    if (type_is_nullptr_t(&left_type) || type_is_void_ptr(&left_type))
                    {
                        type_swap(&p_conditional_expression->type, &right_type);
                    }
                    else if (!type_is_same(&left_type, &right_type, false))
                    {
                        compiler_diagnostic(C_ERROR_INCOMPATIBLE_TYPES, ctx, p_conditional_expression->condition_expr->first_token, NULL, "incompatible types");
                    }
                    else
                    {
                        type_swap(&p_conditional_expression->type, &right_type);
                    }
                }
                else
                {
                    compiler_diagnostic(C_ERROR_INCOMPATIBLE_TYPES, ctx, p_conditional_expression->condition_expr->first_token, NULL, "incompatible types");
                }
            }
            else
            {
                compiler_diagnostic(C_ERROR_INCOMPATIBLE_TYPES, ctx, p_conditional_expression->condition_expr->first_token, NULL, "incompatible types??");
                assert(false);
            }
            p_expression_node = p_conditional_expression;
        }
    }
    catch
    {
        expression_delete(p_expression_node);
        p_expression_node = NULL;
    }

    type_destroy(&left_type);
    type_destroy(&right_type);

    return p_expression_node;
}

struct expression* _Owner _Opt constant_expression(struct parser_ctx* ctx, bool show_error_if_not_constant)
{
    struct expression* _Owner _Opt p_expression = conditional_expression(ctx);

    if (show_error_if_not_constant &&
        p_expression &&
        !object_has_constant_value(&p_expression->object))
    {
        compiler_diagnostic(C_ERROR_EXPECTED_CONSTANT_EXPRESSION, ctx, ctx->current, NULL, "expected constant expression");
    }

    return p_expression;
}

bool expression_get_variables(const struct expression* expr, int n, struct object* variables[/*n*/])
{
    int count = 0;
    switch (expr->expression_type)
    {

    case EXPRESSION_TYPE_INVALID:  break;

    case PRIMARY_EXPRESSION_ENUMERATOR:  break;
    case PRIMARY_EXPRESSION_DECLARATOR:
        if (!object_has_constant_value(&expr->object))
        {
            if (count < n)
            {
                variables[count] = object_get_non_const_referenced(&expr->object);
                count++;
            }

        }
        break;

    case PRIMARY_EXPRESSION_STRING_LITERAL:  break;
    case PRIMARY_EXPRESSION__FUNC__:  break; /*predefined identifier __func__ */
    case PRIMARY_EXPRESSION_CHAR_LITERAL:  break;
    case PRIMARY_EXPRESSION_PREDEFINED_CONSTANT:  break; /*true false*/
    case PRIMARY_EXPRESSION_GENERIC:  break;
    case PRIMARY_EXPRESSION_NUMBER:  break;

    case PRIMARY_EXPRESSION_PARENTESIS:
        count += expression_get_variables(expr->right, n, variables);
        break;

    case POSTFIX_EXPRESSION_FUNCTION_LITERAL:  break;
    case POSTFIX_EXPRESSION_COMPOUND_LITERAL:  break;

    case POSTFIX_FUNCTION_CALL:  break; // ( ) 
    case POSTFIX_ARRAY:  break; // [ ]
    case POSTFIX_DOT:  break; // .
    case POSTFIX_ARROW:  break; // .
    case POSTFIX_INCREMENT:  break;
    case POSTFIX_DECREMENT:  break;


    case UNARY_EXPRESSION_SIZEOF_EXPRESSION:  break;
    case UNARY_EXPRESSION_SIZEOF_TYPE:  break;
    case UNARY_EXPRESSION_NELEMENTSOF_TYPE:  break;

    case UNARY_EXPRESSION_TRAITS:  break;
    case UNARY_EXPRESSION_IS_SAME:  break;
    case UNARY_DECLARATOR_ATTRIBUTE_EXPR:  break;
    case UNARY_EXPRESSION_ALIGNOF:  break;
    case UNARY_EXPRESSION_ASSERT:  break;

    case UNARY_EXPRESSION_INCREMENT:  break;
    case UNARY_EXPRESSION_DECREMENT:  break;

    case UNARY_EXPRESSION_NOT:  break;
    case UNARY_EXPRESSION_BITNOT:  break;
    case UNARY_EXPRESSION_NEG:  break;
    case UNARY_EXPRESSION_PLUS:  break;
    case UNARY_EXPRESSION_CONTENT:  break;
    case UNARY_EXPRESSION_ADDRESSOF:  break;

    case CAST_EXPRESSION:  break;

    case MULTIPLICATIVE_EXPRESSION_MULT:
    case MULTIPLICATIVE_EXPRESSION_DIV:
    case MULTIPLICATIVE_EXPRESSION_MOD:
        count += expression_get_variables(expr->left, n, variables);
        count += expression_get_variables(expr->right, n, variables);
        break;

    case ADDITIVE_EXPRESSION_PLUS:
    case ADDITIVE_EXPRESSION_MINUS:
        count += expression_get_variables(expr->left, n, variables);
        count += expression_get_variables(expr->right, n, variables);
        break;


    case SHIFT_EXPRESSION_RIGHT:
    case SHIFT_EXPRESSION_LEFT:

    case RELATIONAL_EXPRESSION_BIGGER_THAN:
    case RELATIONAL_EXPRESSION_LESS_THAN:
    case RELATIONAL_EXPRESSION_BIGGER_OR_EQUAL_THAN:
    case RELATIONAL_EXPRESSION_LESS_OR_EQUAL_THAN:
    case EQUALITY_EXPRESSION_EQUAL:
    case EQUALITY_EXPRESSION_NOT_EQUAL:
        count += expression_get_variables(expr->left, n, variables);
        count += expression_get_variables(expr->right, n, variables);
        break;

    case AND_EXPRESSION:  break;
    case EXCLUSIVE_OR_EXPRESSION:  break;
    case INCLUSIVE_OR_EXPRESSION:  break;

    case LOGICAL_OR_EXPRESSION:
    case LOGICAL_AND_EXPRESSION:
        count += expression_get_variables(expr->left, n, variables);
        count += expression_get_variables(expr->right, n, variables);
        break; //&&

    case ASSIGNMENT_EXPRESSION_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_PLUS_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_MINUS_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_MULTI_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_DIV_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_MOD_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_SHIFT_LEFT_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_SHIFT_RIGHT_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_AND_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_OR_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_NOT_ASSIGN:  break;


    case EXPRESSION_EXPRESSION:  break;

    case CONDITIONAL_EXPRESSION:  break;
    }
    return count;
}

bool expression_is_lvalue(const struct expression* expr)
{
    // https://en.cppreference.com/w/c/language/value_category

    switch (expr->expression_type)
    {
    case PRIMARY_EXPRESSION_DECLARATOR:
    case PRIMARY_EXPRESSION__FUNC__:
    case PRIMARY_EXPRESSION_STRING_LITERAL:
    case POSTFIX_ARRAY:
    case POSTFIX_ARROW:
    case POSTFIX_EXPRESSION_COMPOUND_LITERAL:
    case UNARY_EXPRESSION_CONTENT:
        return true;
    default:
        break;
    }

    if (expr->expression_type == PRIMARY_EXPRESSION_PARENTESIS &&
        expr->right)
    {
        return expression_is_lvalue(expr->right);
    }
    else if (expr->expression_type == POSTFIX_DOT &&
             expr->left)
    {
        return expression_is_lvalue(expr->left);
    }

    return false;
}


/*
 * Returns true if the type of expression is subjected to type_lvalue_conversion
 */
bool expression_is_subjected_to_lvalue_conversion(const struct expression* expression)
{

    switch (expression->expression_type)
    {
    case UNARY_EXPRESSION_ADDRESSOF:
    case UNARY_EXPRESSION_INCREMENT:
    case UNARY_EXPRESSION_DECREMENT:
    case POSTFIX_INCREMENT:
    case POSTFIX_DECREMENT:
        return false;
    default:
        if (expression->type.storage_class_specifier_flags & STORAGE_SPECIFIER_PARAMETER)
            return true;
    }

    return true;
}

void check_comparison(struct parser_ctx* ctx,
    struct expression* p_a_expression,
    struct expression* p_b_expression,
    const struct token* op_token)
{
    //TODO more checks unsigned < 0

    struct type* p_a_type = &p_a_expression->type;
    struct type* p_b_type = &p_b_expression->type;

    if (type_is_pointer(p_a_type) && type_is_integer(p_b_type))
    {
        if (expression_is_zero(p_b_expression))
        {
            // p == 0
            //style warning
        }
        else
        {
            //array functions..
            compiler_diagnostic(W_ENUN_CONVERSION,
                                        ctx,
                                        op_token, NULL,
                                        "comparison between pointer and integer");
        }
    }

    check_diferent_enuns(ctx,
                         op_token,
                         p_a_expression,
                         p_b_expression,
                         "comparing different enums.");
}

void check_assigment(struct parser_ctx* ctx,
    const struct type* p_a_type, /*this is not expression because function parameters*/
    const struct expression* p_b_expression,
    enum assigment_type assignment_type)
{
    const struct type* const p_b_type = &p_b_expression->type;

    const bool is_null_pointer_constant = expression_is_null_pointer_constant(p_b_expression);

    if (type_is_pointer(p_a_type))
    {
        if (!type_is_nullptr_t(p_b_type) &&
            !type_is_pointer_or_array(p_b_type) &&
            !type_is_function(p_b_type))
        {
            if (is_null_pointer_constant)
            {
                if (p_b_expression->expression_type == PRIMARY_EXPRESSION_NUMBER)
                {
                    // This is the only exception.
                    // p = 0;
                    compiler_diagnostic(W_STYLE, ctx, p_b_expression->first_token, NULL, "use NULL instead of 0");
                }
                else
                {
                    //Everything else is unusual
                    // p = false;
                    // p = 1-1;
                    // p = '\0';
                    compiler_diagnostic(W_UNSUAL_NULL_POINTER_CONSTANT, ctx, p_b_expression->first_token, NULL, "unusual expression/type used as null pointer constant");
                }
            }
            else
            {
                compiler_diagnostic(C_ERROR_INT_TO_POINTER, ctx, p_b_expression->first_token, NULL, "non-pointer to pointer");
            }
        }
    }

    if (type_is_bool(p_a_type) && type_is_nullptr_t(p_b_type))
    {
        struct marker marker = {
        .p_token_begin = p_b_expression->first_token,
        .p_token_end = p_b_expression->first_token
        };

        compiler_diagnostic(W_NULL_CONVERTION,
        ctx,
        NULL,
        &marker,
        "implicit conversion of nullptr constant to 'bool'");
    }

    struct type b_type_lvalue = { 0 };

    if (expression_is_subjected_to_lvalue_conversion(p_b_expression))
    {
        b_type_lvalue = type_lvalue_conversion(p_b_type, ctx->options.null_checks_enabled);
    }
    else
    {
        b_type_lvalue = type_dup(p_b_type);
    }


    if (type_is_owner(p_a_type) && !type_is_owner(&p_b_expression->type))
    {
        if (!is_null_pointer_constant)
        {
            compiler_diagnostic(W_OWNERSHIP_NON_OWNER_TO_OWNER_ASSIGN, ctx, p_b_expression->first_token, NULL, "cannot assign a non-owner to owner");
            type_destroy(&b_type_lvalue);
            //type_destroy(&t2);
            return;
        }
    }

    if (!type_is_owner(p_a_type) && type_is_owner_or_pointer_to_dtor(&p_b_expression->type))
    {
        if (p_b_expression->type.storage_class_specifier_flags & STORAGE_SPECIFIER_FUNCTION_RETURN)
        {
            compiler_diagnostic(W_OWNERSHIP_USING_TEMPORARY_OWNER,
                ctx,
                p_b_expression->first_token, NULL,
                "cannot assign a temporary owner to non-owner object.");
            type_destroy(&b_type_lvalue);
            //type_destroy(&t2);
            return;
        }
    }

    if (assignment_type == ASSIGMENT_TYPE_RETURN)
    {
        if (!type_is_owner(p_a_type) && type_is_owner_or_pointer_to_dtor(&p_b_expression->type))
        {
            if (p_b_expression->type.storage_class_specifier_flags & STORAGE_SPECIFIER_AUTOMATIC_STORAGE)
            {
                compiler_diagnostic(C_ERROR_RETURN_LOCAL_OWNER_TO_NON_OWNER,
                    ctx,
                    p_b_expression->first_token, NULL,
                    "cannot return a automatic storage duration _Owner to non-owner");
                type_destroy(&b_type_lvalue);
                // type_destroy(&t2);
                return;
            }
        }
    }
#if 1
    /*
    TODO REMOVE THiS
    #pragma safety enable

      #include <stdlib.h>

      struct X {
       struct Y * _Owner p;
      };

      void x_destroy(_Dtor struct X * p);

      void f(struct X * x) {
        x_destroy(x);
        static_debug(*x);
      }
    */
    if (type_is_pointed_dtor(p_a_type) && type_is_pointer(p_a_type))
    {
        if (type_is_owner(p_b_type))
        {
        }
        else if (!p_b_type->address_of)
        {
            compiler_diagnostic(W_MUST_USE_ADDRESSOF,
                       ctx,
                       p_b_expression->first_token, NULL,
                       "source expression of _Dtor must be addressof");
        }
    }
#endif


    if (type_is_pointer(p_a_type) &&
        !type_is_opt(p_a_type, ctx->options.null_checks_enabled) &&
        is_null_pointer_constant)
    {

        compiler_diagnostic(W_FLOW_NULLABLE_TO_NON_NULLABLE,
            ctx,
            p_b_expression->first_token, NULL,
            "cannot convert a null pointer constant to non-nullable pointer");

        type_destroy(&b_type_lvalue);
        //type_destroy(&t2);

        return;

    }



    /*
       less generic tests are first
    */
    if (type_is_enum(p_b_type) && type_is_enum(p_a_type))
    {
        if (!type_is_same(p_b_type, p_a_type, false))
        {
            compiler_diagnostic(W_INCOMPATIBLE_ENUN_TYPES, ctx,
                p_b_expression->first_token, NULL,
                " incompatible types ");
        }


        type_destroy(&b_type_lvalue);
        // type_destroy(&t2);
        return;
    }

    if (type_is_arithmetic(p_b_type) && type_is_arithmetic(p_a_type))
    {

        type_destroy(&b_type_lvalue);
        //type_destroy(&t2);
        return;
    }

    if (is_null_pointer_constant && type_is_pointer(p_a_type))
    {
        //TODO void F(int * [[_Opt]] p)
        // F(0) when passing null we will check if the parameter
        //have the anotation [[_Opt]]

        /*can be converted to any type*/

        type_destroy(&b_type_lvalue);
        //type_destroy(&t2);
        return;
    }

    if (is_null_pointer_constant && type_is_array(p_a_type))
    {
        compiler_diagnostic(W_FLOW_NON_NULL,
            ctx,
            p_b_expression->first_token, NULL,
            " passing null as array");


        type_destroy(&b_type_lvalue);
        //type_destroy(&t2);
        return;
    }

    /*
       We have two pointers or pointer/array combination
    */
    if (type_is_pointer_or_array(p_b_type) && type_is_pointer_or_array(p_a_type))
    {
        if (type_is_void_ptr(p_b_type))
        {
            /*void pointer can be converted to any type*/

            type_destroy(&b_type_lvalue);
            //type_destroy(&t2);
            return;
        }

        if (type_is_void_ptr(p_a_type))
        {
            /*any pointer can be converted to void* */

            type_destroy(&b_type_lvalue);
            // type_destroy(&t2);
            return;
        }


        //TODO  lvalue

        struct type a_type_lvalue = { 0 };

        if (type_is_array(p_a_type))
        {
            if (assignment_type == ASSIGMENT_TYPE_PARAMETER)
            {
                size_t parameter_array_size = p_a_type->num_of_elements;
                if (type_is_array(p_b_type))
                {
                    size_t argument_array_size = p_b_type->num_of_elements;
                    if (parameter_array_size != 0 &&
                        argument_array_size < parameter_array_size)
                    {
                        compiler_diagnostic(C_ERROR_ARGUMENT_SIZE_SMALLER_THAN_PARAMETER_SIZE, ctx,
                            p_b_expression->first_token, NULL,
                            " argument of size [%d] is smaller than parameter of size [%d]", argument_array_size, parameter_array_size);
                    }
                }
                else if (is_null_pointer_constant || type_is_nullptr_t(p_b_type))
                {
                    compiler_diagnostic(W_PASSING_NULL_AS_ARRAY, ctx,
                        p_b_expression->first_token, NULL,
                        " passing null as array");
                }
            }
            a_type_lvalue = type_lvalue_conversion(p_a_type, ctx->options.null_checks_enabled);
        }
        else
        {
            a_type_lvalue = type_dup(p_a_type);
        }



        if (!type_is_same(&b_type_lvalue, &a_type_lvalue, false))
        {
            type_print(&b_type_lvalue);
            type_print(&a_type_lvalue);

            compiler_diagnostic(W_ERROR_INCOMPATIBLE_TYPES, ctx,
                p_b_expression->first_token, NULL,
                " incompatible types");
        }


        if (assignment_type == ASSIGMENT_TYPE_PARAMETER)
        {
            if (type_is_pointer(&b_type_lvalue) && type_is_pointer(&a_type_lvalue))
            {
                //parameter pointer do non const
                //argument const.
                struct type b_pointed_type_lvalue = type_remove_pointer(&b_type_lvalue);
                struct type a_lvalue_pointed_type = type_remove_pointer(&a_type_lvalue);
                if (type_is_const(&b_pointed_type_lvalue) && !type_is_const(&a_lvalue_pointed_type))
                {
                    compiler_diagnostic(W_DISCARDED_QUALIFIERS, ctx,
                        p_b_expression->first_token, NULL,
                        " discarding const at argument ");
                }
                type_destroy(&b_pointed_type_lvalue);
                type_destroy(&a_lvalue_pointed_type);
            }
        }
        else
        {
            if (type_is_pointer(p_a_type) && type_is_pointer(&b_type_lvalue))
            {
                struct type b_pointed_type = type_remove_pointer(&b_type_lvalue);
                struct type a_pointed_type = type_remove_pointer(p_a_type);
                if (type_is_const(&b_pointed_type) && !type_is_const(&a_pointed_type))
                {
                    compiler_diagnostic(W_DISCARDED_QUALIFIERS, ctx,
                        p_b_expression->first_token, NULL,
                        " discarding const");
                }
                type_destroy(&b_pointed_type);
                type_destroy(&a_pointed_type);
            }
        }

        //return true;
        type_destroy(&a_type_lvalue);
    }

    if (!type_is_same(p_a_type, &b_type_lvalue, false))
    {
        // compiler_diagnostic(C_ERROR_INCOMPATIBLE_TYPES,
        //     ctx,
        //       p_b_expression->first_token, 
        //       NULL,
        //       " incompatible types ");
    }


    type_destroy(&b_type_lvalue);


}

struct object expression_eval(struct expression* p_expression)
{
    struct object result = { 0 };

    switch (p_expression->expression_type)
    {
    case EXPRESSION_TYPE_INVALID: break;

    case PRIMARY_EXPRESSION_ENUMERATOR:
    case PRIMARY_EXPRESSION_DECLARATOR:
        result = object_dup(&p_expression->object);
        break;

    case PRIMARY_EXPRESSION_STRING_LITERAL:  break;
    case PRIMARY_EXPRESSION__FUNC__:  break; /*predefined identifier __func__ */

    case PRIMARY_EXPRESSION_CHAR_LITERAL:
    case PRIMARY_EXPRESSION_PREDEFINED_CONSTANT:
        result = object_dup(&p_expression->object);
        break;

    case PRIMARY_EXPRESSION_GENERIC:  break;
    case PRIMARY_EXPRESSION_NUMBER:
        result = object_dup(&p_expression->object);
        break;


    case PRIMARY_EXPRESSION_PARENTESIS:
        result = expression_eval(p_expression->right);
        break;

    case POSTFIX_EXPRESSION_FUNCTION_LITERAL:  break;
    case POSTFIX_EXPRESSION_COMPOUND_LITERAL:  break;

    case POSTFIX_FUNCTION_CALL:  break; // ( ) 
    case POSTFIX_ARRAY:  break; // [ ]
    case POSTFIX_DOT:  break; // .
    case POSTFIX_ARROW:  break; // .
    case POSTFIX_INCREMENT:  break;
    case POSTFIX_DECREMENT:  break;


    case UNARY_EXPRESSION_SIZEOF_EXPRESSION:  break;
    case UNARY_EXPRESSION_SIZEOF_TYPE:  break;
    case UNARY_EXPRESSION_NELEMENTSOF_TYPE:  break;

    case UNARY_EXPRESSION_TRAITS:  break;
    case UNARY_EXPRESSION_IS_SAME:  break;
    case UNARY_DECLARATOR_ATTRIBUTE_EXPR:  break;
    case UNARY_EXPRESSION_ALIGNOF:  break;
    case UNARY_EXPRESSION_ASSERT:  break;

    case UNARY_EXPRESSION_INCREMENT:  break;
    case UNARY_EXPRESSION_DECREMENT:  break;

    case UNARY_EXPRESSION_NOT:  break;
    case UNARY_EXPRESSION_BITNOT:  break;
    case UNARY_EXPRESSION_NEG:  break;
    case UNARY_EXPRESSION_PLUS:  break;
    case UNARY_EXPRESSION_CONTENT:  break;
    case UNARY_EXPRESSION_ADDRESSOF:  break;

    case CAST_EXPRESSION:  break;

    case MULTIPLICATIVE_EXPRESSION_MULT:  break;
    case MULTIPLICATIVE_EXPRESSION_DIV:  break;
    case MULTIPLICATIVE_EXPRESSION_MOD:  break;

    case ADDITIVE_EXPRESSION_PLUS:
    {
        struct object a = expression_eval(p_expression->left);
        if (object_has_constant_value(&a))
        {
            struct object b = expression_eval(p_expression->right);
            if (object_has_constant_value(&b))
            {
                result = object_add(&a, &b);
            }
        }
    }
    break;
    case ADDITIVE_EXPRESSION_MINUS:
    {
        struct object a = expression_eval(p_expression->left);
        if (object_has_constant_value(&a))
        {
            struct object b = expression_eval(p_expression->right);
            if (object_has_constant_value(&b))
            {
                result = object_sub(&a, &b);
            }
        }
    }
    break;

    case SHIFT_EXPRESSION_RIGHT:  break;
    case SHIFT_EXPRESSION_LEFT:  break;

    case RELATIONAL_EXPRESSION_BIGGER_THAN:  break;
    case RELATIONAL_EXPRESSION_LESS_THAN:  break;
    case RELATIONAL_EXPRESSION_BIGGER_OR_EQUAL_THAN:  break;
    case RELATIONAL_EXPRESSION_LESS_OR_EQUAL_THAN:  break;

    case EQUALITY_EXPRESSION_EQUAL:
        if (object_equal(&p_expression->left->object, &p_expression->right->object))
        {
            result = object_make_signed_int(1);
        }
        else
        {
            result = object_make_signed_int(0);
        }
        break;


    case EQUALITY_EXPRESSION_NOT_EQUAL:
        if (object_not_equal(&p_expression->left->object, &p_expression->right->object))
        {
            result = object_make_signed_int(1);
        }
        else
        {
            result = object_make_signed_int(0);
        }
        break;


    case AND_EXPRESSION:  break;
    case EXCLUSIVE_OR_EXPRESSION:  break;
    case INCLUSIVE_OR_EXPRESSION:  break;

    case LOGICAL_OR_EXPRESSION:
    {
        struct object a = expression_eval(p_expression->left);
        if (object_has_constant_value(&a))
        {
            bool r1 = object_to_bool(&a);
            if (r1)
            {
                result = object_make_signed_int(1);
            }
            else
            {
                struct object b = expression_eval(p_expression->right);
                if (object_has_constant_value(&b))
                {
                    bool r2 = object_to_bool(&b);
                    if (r2)
                    {
                        result = object_make_signed_int(r2);
                    }
                }
            }
        }
    }
    break;  //||

    case LOGICAL_AND_EXPRESSION:  break; //&&

    case ASSIGNMENT_EXPRESSION_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_PLUS_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_MINUS_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_MULTI_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_DIV_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_MOD_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_SHIFT_LEFT_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_SHIFT_RIGHT_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_AND_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_OR_ASSIGN:  break;
    case ASSIGNMENT_EXPRESSION_NOT_ASSIGN:  break;


    case EXPRESSION_EXPRESSION:  break;

    case CONDITIONAL_EXPRESSION:  break;

    }
    return result;
}

/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

#pragma safety enable

/*
  For performance reasons we will separate expression from preprocessor from compiler.
*/



#include <locale.h>

#ifdef _WIN32
#endif

#if defined _MSC_VER && !defined __POCC__
#endif

/*context expressions preprocessor*/
struct pre_expression_ctx
{
    /*all preprocessor expressions are calculated with long long*/
    long long value;
};

static void pre_postfix_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_cast_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_multiplicative_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_unary_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_additive_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_shift_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_relational_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_equality_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_and_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_exclusive_or_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_inclusive_or_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_logical_and_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_logical_or_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_conditional_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);
static void pre_conditional_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx);

//TODO share this with parser!
/*
 * preprocessor uses long long
 */
static int ppnumber_to_longlong(struct preprocessor_ctx* ctx, struct token* token, long long* result)
{
    /*copy removing the separators*/
    // um dos maiores buffer necessarios seria 128 bits binario...
    // 0xb1'1'1....
    int c = 0;
    char buffer[128 * 2 + 4] = { 0 };
    const char* s = token->lexeme;
    while (*s)
    {
        if (*s != '\'')
        {
            buffer[c] = *s;
            c++;
        }
        s++;
    }

    char errormsg[100];
    char suffix[4] = { 0 };
    const enum token_type type = parse_number(token->lexeme, suffix, errormsg);
    if (type == TK_NONE)
    {
        preprocessor_diagnostic(
            C_INVALID_TOKEN,
            ctx,
            token,
            "%s",
            errormsg);
        return 0;
    }
    struct object  cv = { 0 };
    switch (type)
    {
    case TK_COMPILER_DECIMAL_CONSTANT:
    case TK_COMPILER_OCTAL_CONSTANT:
    case TK_COMPILER_HEXADECIMAL_CONSTANT:
    case TK_COMPILER_BINARY_CONSTANT:
    {
        unsigned long long value = 0;
        switch (type)
        {
        case TK_COMPILER_DECIMAL_CONSTANT:
            value = strtoull(buffer, NULL, 10);
            break;
        case TK_COMPILER_OCTAL_CONSTANT:
            value = strtoull(buffer + 1, NULL, 8);
            break;
        case TK_COMPILER_HEXADECIMAL_CONSTANT:
            value = strtoull(buffer + 2, NULL, 16);
            break;
        case TK_COMPILER_BINARY_CONSTANT:
            value = strtoull(buffer + 2, NULL, 2);
            break;
        default:
            break;
        }

        if (value == ULLONG_MAX && errno == ERANGE)
        {
            //compiler_diagnostic(
            //C_ERROR_LITERAL_OVERFLOW,
            //ctx,
            //token,
            //NULL,
            //"integer literal is too large to be represented in any integer type");
        }

        if (suffix[0] == 'U')
        {
            /*fixing the type that fits the size*/
            if (value <= UINT_MAX && suffix[1] != 'L')
            {
                cv = object_make_unsigned_int((unsigned int)value);

            }
            else if (value <= ULONG_MAX && suffix[2] != 'L')
            {
                cv = object_make_unsigned_long((unsigned long)value);
            }
            else //if (value <= ULLONG_MAX)
            {
                cv = object_make_unsigned_long_long((unsigned long long)value);
            }
        }
        else
        {
            /*fixing the type that fits the size*/
            if (value <= INT_MAX && suffix[0] != 'L')
            {
                cv = object_make_signed_int((int)value);
            }
            else if (value <= LONG_MAX && suffix[1] != 'L' /*!= LL*/)
            {
                cv = object_make_signed_long((long)value);
            }
            else if (value <= LLONG_MAX)
            {
                cv = object_make_signed_long_long((long long)value);
            }
            else
            {
                cv = object_make_signed_long_long(value);
            }
        }

    }
    break;

    case TK_COMPILER_DECIMAL_FLOATING_CONSTANT:
    case TK_COMPILER_HEXADECIMAL_FLOATING_CONSTANT:
        //error
        break;

    default:
        assert(false);
    }

    *result = object_to_signed_long_long(&cv);

    return 0;
}

/*
  ctx->current and pre_match are used only in preprocessor constant expressions
  (the preprocessor itself uses concept of removing from one list and adding
  into another so the head of the input list is the current.
  We could use the same concept here removing current.
*/
static struct token* _Opt pre_match(struct preprocessor_ctx* ctx)
{
    if (ctx->current == NULL)
        return NULL;

    ctx->current = ctx->current->next;

    while (ctx->current && token_is_blank(ctx->current))
    {
        ctx->current = ctx->current->next;
    }

    return ctx->current;
}

//TODO share this with parser
static struct object char_constant_to_value(const char* s, char error_message[/*sz*/], int error_message_sz_bytes)
{
    error_message[0] = '\0';

    const unsigned char* _Opt p = (const unsigned char*)s;

    try
    {
        if (p[0] == 'u' && p[1] == '8')
        {
            p++;
            p++;
            p++;

            // A UTF-8 character constant has type char8_t.

            unsigned int c = 0;
            p = utf8_decode(p, &c);
            if (p == NULL)
            {
                throw;
            }

            if (c == '\\')
            {
                p = escape_sequences_decode_opt(p, &c);
                if (p == NULL)
                {
                    throw;
                }
            }

            if (*p != '\'')
            {
                snprintf(error_message, error_message_sz_bytes, "Unicode character literals may not contain multiple characters.");
            }

            if (c > 0x80)
            {
                snprintf(error_message, error_message_sz_bytes, "Character too large for enclosing character literal type.");
            }

            return object_make_wchar_t((wchar_t)c);//, ctx->evaluation_is_disabled);
        }
        else if (p[0] == 'u')
        {
            p++;
            p++;

            // A UTF-16 character constant has type char16_t which is an unsigned integer types defined in the <uchar.h> header

            unsigned int c = 0;
            p = utf8_decode(p, &c);
            if (p == NULL)
            {
                throw;
            }

            if (c == '\\')
            {
                p = escape_sequences_decode_opt(p, &c);
                if (p == NULL)
                {
                    throw;
                }
            }

            if (*p != '\'')
            {
                snprintf(error_message, error_message_sz_bytes, "Unicode character literals may not contain multiple characters.");
            }

            if (c > USHRT_MAX)
            {
                snprintf(error_message, error_message_sz_bytes, "Character too large for enclosing character literal type.");
            }

            return object_make_wchar_t((wchar_t)c);
        }
        else if (p[0] == 'U')
        {
            p++;
            p++;

            // A UTF-16 character constant has type char16_t which is an unsigned integer types defined in the <uchar.h> header

            unsigned int c = 0;
            p = utf8_decode(p, &c);
            if (p == NULL)
            {
                throw;
            }

            if (c == '\\')
            {
                p = escape_sequences_decode_opt(p, &c);

                if (p == NULL)
                {
                    throw;
                }
            }

            if (*p != '\'')
            {
                snprintf(error_message, error_message_sz_bytes, "Unicode character literals may not contain multiple characters.");
            }

            if (c > UINT_MAX)
            {
                snprintf(error_message, error_message_sz_bytes, "Character too large for enclosing character literal type.");
            }

            return object_make_wchar_t((wchar_t)c);
        }
        else if (p[0] == 'L')
        {
            // A wchar_t character constant is prefixed by the letter L
            p++;
            p++;

            /*
             wchar_t character constant prefixed by the letter L has type wchar_t, an integer type defined in
             the <stddef.h> header. The value of a wchar_t character constant containing a single multibyte
             character that maps to a single member of the extended execution character set is the wide character
             corresponding to that multibyte character in the implementation-defined wide literal encoding
             (6.2.9). The value of a wchar_t character constant containing more than one multibyte character or a
             single multibyte character that maps to multiple members of the extended execution character set,
             or containing a multibyte character or escape sequence not represented in the extended execution
             character set, is implementation-defined.
            */
            long long value = 0;
            while (*p != '\'')
            {
                unsigned int c = 0;
                p = utf8_decode(p, &c);
                if (p == NULL)
                {
                    throw;
                }
                if (c == '\\')
                {
                    p = escape_sequences_decode_opt(p, &c);
                    if (p == NULL)
                        throw;
                }
       
                // TODO \u
                value = value * 256 + c;
#ifdef _WIN32
                if (value > USHRT_MAX)
                {
                    snprintf(error_message, error_message_sz_bytes, "character constant too long for its type");
                    break;
                }
#else
                if (value > UINT_MAX)
                {
                    snprintf(error_message, error_message_sz_bytes, "character constant too long for its type");
                    break;
                }
#endif
            }

            return object_make_wchar_t((wchar_t)value);
        }
        else
        {
            p++;

            /*
              An integer character constant has type int. The value of an integer character constant containing
              a single character that maps to a single value in the literal encoding (6.2.9) is the numerical value
              of the representation of the mapped character in the literal encoding interpreted as an integer.
              The value of an integer character constant containing more than one character (e.g., ’ab’), or
              containing a character or escape sequence that does not map to a single value in the literal encoding,
              is implementation-defined. If an integer character constant contains a single character or escape
              sequence, its value is the one that results when an object with type char whose value is that of the
              single character or escape sequence is converted to type int.
            */
            long long value = 0;
            while (*p != '\'')
            {
                unsigned int c = 0;
                p = utf8_decode(p, &c);
                if (p == NULL)
                {
                    throw;
                }

                if (c == '\\')
                {
                    p = escape_sequences_decode_opt(p, &c);
                    if (p == NULL)
                        throw;
                }


                if (c < 0x80)
                {
                    value = value * 256 + c;
                }
                else
                {
                    value = c;
                }
                if (value > INT_MAX)
                {
                    snprintf(error_message, error_message_sz_bytes, "character constant too long for its type");
                    break;
                }
            }
            return object_make_wchar_t((wchar_t)value);
        }
    }
    catch
    {
    }

    struct object empty = { 0 };
    return empty;
}

static void pre_primary_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
     primary-expression:
      identifier
      constant
      string-literal
      ( expression )
      generic-selection
    */
    try
    {
        if (ctx->current == NULL)
        {
            pre_unexpected_end_of_file(ctx->input_list.tail, ctx);
            throw;
        }

        if (ctx->current->type == TK_CHAR_CONSTANT)
        {
            const char* p = ctx->current->lexeme + 1;
            char errmsg[200] = { 0 };
            struct object v = char_constant_to_value(p, errmsg, sizeof errmsg);
            if (errmsg[0] != '\0')
            {
                preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, "%s", errmsg);
            }
            ectx->value = object_to_signed_long_long(&v);
            pre_match(ctx);
            object_destroy(&v);
        }
        else if (ctx->current->type == TK_PPNUMBER)
        {
            ppnumber_to_longlong(ctx, ctx->current, &ectx->value);
            pre_match(ctx);
        }
        else if (ctx->current->type == '(')
        {
            pre_match(ctx);
            pre_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;
            if (ctx->current && ctx->current->type != ')')
            {
                preprocessor_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, "expected )");
                throw;
            }
            pre_match(ctx);
        }
        else
        {
            preprocessor_diagnostic(C_ERROR_TOKEN_NOT_VALID_IN_PREPROCESSOR_EXPRESSIONS,
                                              ctx,
                                              ctx->current,
                                              "token '%s' is not valid in preprocessor expressions",
                                              ctx->current->lexeme);
            throw;
        }
    }
    catch
    {
    }
}

static void pre_postfix_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
      postfix-expression:
        primary-expression
        postfix-expression [ expression ]
        postfix-expression ( argument-expression-list_opt)
        postfix-expression . identifier
        postfix-expression -> identifier
        postfix-expression ++
        postfix-expression --
        ( type-name ) { initializer-ctx }
        ( type-name ) { initializer-ctx , }

        //My extension : if type-name is function then follow is compound-statement
        ( type-name ) compound-statement

        */
    try
    {
        pre_primary_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;
    }
    catch
    {
    }
}

static void pre_unary_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
    unary-expression:
      postfix-expression
      ++ unary-expression
      -- unary-expression

      one of (& * + - ~ !) cast-expression

      sizeof unary-expression
      sizeof ( type-name )
      _Alignof ( type-name )
      */
    try
    {
        if (ctx->current && (ctx->current->type == '++' || ctx->current->type == '--'))
        {
            preprocessor_diagnostic(C_ERROR_TOKEN_NOT_VALID_IN_PREPROCESSOR_EXPRESSIONS,
                                              ctx,
                                              ctx->current,
                                              "token '%s' is not valid in preprocessor expressions",
                                              ctx->current->lexeme);
            throw;
        }
        else if (ctx->current != NULL &&
                 (ctx->current->type == '&' || ctx->current->type == '*' || ctx->current->type == '+' || ctx->current->type == '-' || ctx->current->type == '~' || ctx->current->type == '!'))
        {
            const struct token* const p_old = ctx->current;
            enum token_type op = ctx->current->type;
            pre_match(ctx);
            pre_cast_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            if (op == '!')
                ectx->value = !ectx->value;
            else if (op == '~')
                ectx->value = ~ectx->value;
            else if (op == '-')
                ectx->value = -ectx->value;
            else if (op == '+')
                ectx->value = +ectx->value;
            else if (op == '*')
            {
                preprocessor_diagnostic(C_ERROR_TOKEN_NOT_VALID_IN_PREPROCESSOR_EXPRESSIONS, ctx, p_old, "token '%s' is not valid in preprocessor expressions", p_old->lexeme);
            }
            else if (op == '&')
            {
                preprocessor_diagnostic(C_ERROR_TOKEN_NOT_VALID_IN_PREPROCESSOR_EXPRESSIONS, ctx, p_old, "token '%s' is not valid in preprocessor expressions", p_old->lexeme);
            }
            else
            {
                preprocessor_diagnostic(C_ERROR_TOKEN_NOT_VALID_IN_PREPROCESSOR_EXPRESSIONS, ctx, p_old, "token '%s' is not valid in preprocessor expressions", p_old->lexeme);
            }
        }
        else
        {
            pre_postfix_expression(ctx, ectx);
        }
    }
    catch
    {
    }
}

static void pre_cast_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
     cast-expression:
      unary-expression
      ( type-name ) cast-expression
    */
    pre_unary_expression(ctx, ectx);
}

static void pre_multiplicative_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
     multiplicative-expression:
    cast-expression
    multiplicative-expression * cast-expression
    multiplicative-expression / cast-expression
    multiplicative-expression % cast-expression
    */
    try
    {
        pre_cast_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '*' ||
                   ctx->current->type == '/' ||
                   ctx->current->type == '%'))
        {
            struct token* op_token = ctx->current;
            enum token_type op = ctx->current->type;
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_cast_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            if (op == '*')
            {
                ectx->value = (left_value * ectx->value);
            }
            else if (op == '/')
            {
                if (ectx->value == 0)
                {
                    preprocessor_diagnostic(C_PRE_DIVISION_BY_ZERO, ctx, op_token, "division by zero");
                    throw;
                }
                else
                {
                    ectx->value = (left_value / ectx->value);
                }
            }
            else if (op == '%')
            {
                ectx->value = (left_value % ectx->value);
            }
        }
    }
    catch
    {
    }
}

static void pre_additive_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
     additive-expression:
       multiplicative-expression
       additive-expression + multiplicative-expression
       additive-expression - multiplicative-expression
    */
    try
    {
        pre_multiplicative_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '+' ||
                   ctx->current->type == '-'))
        {
            const struct token* p_op_token = ctx->current;
            pre_match(ctx);
            if (ctx->current == NULL)
            {
                pre_unexpected_end_of_file(ctx->input_list.tail, ctx);
                throw;
            }
            long long left_value = ectx->value;
            pre_multiplicative_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            if (p_op_token->type == '+')
            {
                ectx->value = left_value + ectx->value;
            }
            else if (p_op_token->type == '-')
            {
                ectx->value = left_value - ectx->value;
            }
            else
            {
                throw;
            }
        }
    }
    catch
    {
    }
}

static void pre_shift_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
     shift-expression:
       additive-expression
       shift-expression << additive-expression
       shift-expression >> additive-expression
    */
    try
    {
        pre_additive_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '>>' ||
                   ctx->current->type == '<<'))
        {
            enum token_type op = ctx->current->type;
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_multiplicative_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            if (op == '>>')
            {
                ectx->value = left_value >> ectx->value;
            }
            else if (op == '<<')
            {
                ectx->value = left_value << ectx->value;
            }
        }
    }
    catch
    {
    }
}

static void pre_relational_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
    relational-expression:
      shift-expression
      relational-expression < shift-expression
      relational-expression > shift-expression
      relational-expression <= shift-expression
      relational-expression >= shift-expression
    */
    try
    {
        pre_shift_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '>' ||
                   ctx->current->type == '<' ||
                   ctx->current->type == '>=' ||
                   ctx->current->type == '<='))
        {
            enum token_type op = ctx->current->type;
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_shift_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            if (op == '>')
            {
                ectx->value = left_value > ectx->value;
            }
            else if (op == '<')
            {
                ectx->value = left_value < ectx->value;
            }
            else if (op == '>=')
            {
                ectx->value = left_value >= ectx->value;
            }
            else if (op == '<=')
            {
                ectx->value = left_value <= ectx->value;
            }
        }
    }
    catch
    {
    }
}

static void pre_equality_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
      equality-expression:
        relational-expression
        equality-expression == relational-expression
        equality-expression != relational-expression
    */

    /*
    * Equality operators
    One of the following shall hold:
    — both operands have arithmetic type;
    — both operands are pointers to qualified or unqualified versions of compatible types;
    — one operand is a pointer to an object type and the other is a pointer to a qualified or unqualified
    version of void; or
    — one operand is a pointer and the other is a null pointer constant.
    */
    try
    {
        pre_relational_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '==' ||
                   ctx->current->type == '!='))
        {
            enum token_type op = ctx->current->type;
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_multiplicative_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            if (op == '==')
            {
                ectx->value = left_value == ectx->value;
            }
            else if (op == '!=')
            {
                ectx->value = left_value != ectx->value;
            }
        }
    }
    catch
    {
    }
}

static void pre_and_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
     AND-expression:
      equality-expression
      AND-expression & equality-expression
    */
    try
    {
        pre_equality_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;
        while (ctx->current != NULL &&
               (ctx->current->type == '&'))
        {
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_equality_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;
            ectx->value = left_value & ectx->value;
        }
    }
    catch
    {
    }
}

static void pre_exclusive_or_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
     exclusive-OR-expression:
      AND-expression
     exclusive-OR-expression ^ AND-expression
    */
    try
    {
        pre_and_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '^'))
        {
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_and_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;
            ectx->value = left_value ^ ectx->value;
        }
    }
    catch
    {
    }
}

static void pre_inclusive_or_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
    inclusive-OR-expression:
    exclusive-OR-expression
    inclusive-OR-expression | exclusive-OR-expression
    */
    try
    {
        pre_exclusive_or_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '|'))
        {
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_exclusive_or_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            ectx->value = left_value | ectx->value;
        }
    }
    catch
    {
    }
}

static void pre_logical_and_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
    logical-AND-expression:
     inclusive-OR-expression
     logical-AND-expression && inclusive-OR-expression
    */
    try
    {
        pre_inclusive_or_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '&&'))
        {
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_inclusive_or_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            ectx->value = left_value && ectx->value;
        }
    }
    catch
    {
    }
}

static void pre_logical_or_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
      logical-OR-expression:
       logical-AND-expression
       logical-OR-expression || logical-AND-expression
    */
    try
    {
        pre_logical_and_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '||'))
        {
            pre_match(ctx);
            long long left_value = ectx->value;
            pre_logical_and_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;

            ectx->value = left_value || ectx->value;
        }
    }
    catch
    {
    }
}

static void pre_assignment_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
    assignment-expression:
       conditional-expression
       unary-expression assignment-operator assignment-expression
       */
       /*
          assignment-operator: one of
          = *= /= %= += -= <<= >>= &= ^= |=
       */
       // here it is doubtful but conditional does unary too. 
       // the difference is that it is not just any expression 
       // which can be an assignment
    try
    {
        pre_conditional_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current != NULL &&
               (ctx->current->type == '=' ||
                   ctx->current->type == '*=' ||
                   ctx->current->type == '/=' ||
                   ctx->current->type == '+=' ||
                   ctx->current->type == '-=' ||
                   ctx->current->type == '<<=' ||
                   ctx->current->type == '>>=' ||
                   ctx->current->type == '&=' ||
                   ctx->current->type == '^=' ||
                   ctx->current->type == '|='))
        {
            preprocessor_diagnostic(C_ERROR_TOKEN_NOT_VALID_IN_PREPROCESSOR_EXPRESSIONS, ctx, ctx->current, "token '%s' is not valid in preprocessor expressions", ctx->current->lexeme);
            throw;
        }
    }
    catch
    {
    }
}

static void pre_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*expression:
      assignment-expression
      expression, assignment-expression
    */
    try
    {
        pre_assignment_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        while (ctx->current && ctx->current->type == ',')
        {
            pre_match(ctx);
            pre_expression(ctx, ectx);
            if (ctx->n_errors > 0)
                throw;
        }
    }
    catch
    {
    }
}

static void pre_conditional_expression(struct preprocessor_ctx* ctx, struct pre_expression_ctx* ectx)
{
    /*
      conditional-expression:
      logical-OR-expression
      logical-OR-expression ? expression : conditional-expression
    */
    try
    {
        pre_logical_or_expression(ctx, ectx);
        if (ctx->n_errors > 0)
            throw;

        if (ctx->current && ctx->current->type == '?')
        {
            pre_match(ctx);
            if (ectx->value)
            {
                pre_expression(ctx, ectx);
                if (ctx->n_errors > 0)
                    throw;

                pre_match(ctx); //:
                struct pre_expression_ctx temp = { 0 };
                pre_conditional_expression(ctx, &temp);
                if (ctx->n_errors > 0)
                    throw;
            }
            else
            {
                struct pre_expression_ctx temp = { 0 };
                pre_expression(ctx, &temp);
                if (ctx->n_errors > 0)
                    throw;

                pre_match(ctx); //:
                pre_conditional_expression(ctx, ectx);
                if (ctx->n_errors > 0)
                    throw;
            }
        }
    }
    catch
    {
    }
}

int pre_constant_expression(struct preprocessor_ctx* ctx, long long* pvalue)
{
    struct pre_expression_ctx ectx = { 0 };
    pre_conditional_expression(ctx, &ectx);
    *pvalue = ectx.value;
    return ctx->n_errors > 0;
}



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/

#pragma safety enable




/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
*/
 
//#pragma once

struct flow_visit_ctx;

extern unsigned int s_visit_number; //creates a unique number


enum flow_state
{
    /*
       Not applicable. The state cannot be used.
       struct...
       TODO we need empty state when object does not exist
       and the merge it is not a problem
       merge with nothing then
    */
    FLOW_OBJECT_STATE_NOT_APPLICABLE = 0,

    FLOW_OBJECT_STATE_UNINITIALIZED = 1 << 0,


    /*
      The only reason we have null and zero is because
      of non pointer references -1 for instance can be the "null"
    */
    FLOW_OBJECT_STATE_NULL = 1 << 1,
    FLOW_OBJECT_STATE_NOT_NULL = 1 << 2,

    //means not-null moved at same time
    FLOW_OBJECT_STATE_MOVED = 1 << 3,

    FLOW_OBJECT_STATE_ZERO = 1 << 5,
    FLOW_OBJECT_STATE_NOT_ZERO = 1 << 6,

    FLOW_OBJECT_STATE_LIFE_TIME_ENDED = 1 << 7,
    
    FLOW_OBJECT_STATE_DANGLING = 1 << 8
};


struct flow_objects
{
    struct flow_object* _Owner* _Owner _Opt data;
    int size;
    int capacity;
};

void flow_objects_clear(struct flow_objects* p);
void flow_objects_destroy(_Dtor struct flow_objects* p);
int flow_objects_push_back(struct flow_objects* p, struct flow_object* _Owner p_object);
const struct flow_object* _Opt flow_objects_find(const struct flow_objects* p, const struct flow_object* p_object);


struct flow_objects_view 
{
    struct flow_object** _Owner _Opt data;
    int size;
    int capacity;
};

void objects_view_destroy(_Dtor struct flow_objects_view* p);
int objects_view_push_back(struct flow_objects_view* p, struct flow_object* p_object);
bool objects_view_find(const struct flow_objects_view* p, const struct flow_object* p_object);
void objects_view_copy(struct flow_objects_view* dest, const struct flow_objects_view* source);
void objects_view_merge(struct flow_objects_view* dest, const struct flow_objects_view* source);
void objects_view_clear(struct flow_objects_view* p);


struct flow_object_state
{
    const char* dbg_name;
    int state_number;

    struct flow_object* _Opt pointed;
    enum flow_state state;
    struct flow_objects_view alternatives;
    struct flow_object_state* _Owner _Opt next;
};

void flow_object_state_copy(struct flow_object_state* to, const struct flow_object_state* from);
void flow_object_state_delete(struct flow_object_state* _Owner _Opt p);


/*
  Used in flow analysis to represent the object instance
*/
struct flow_object
{
    //used to avoid infinite recursion
    unsigned int visit_number;

    struct flow_object* _Opt parent;

    /*object are the result of expressions or they are declarators*/
    const struct declarator* _Opt p_declarator_origin;
    const struct expression* _Opt p_expression_origin;

    struct flow_objects_view members;

    struct flow_object_state current;

    int id; //helps debugging
    bool is_temporary;
};

void flow_object_set_is_moved(struct flow_object* p_object);
void flow_object_set_can_be_uninitialized(struct flow_object* p_object);
void flow_object_set_is_unitialized(struct flow_object* p_object);
void flow_object_update_current(struct flow_object* p);
void flow_object_set_current_state_to_can_be_null(struct flow_object* p);
void flow_object_set_current_state_to_is_null(struct flow_object* p);

int flow_object_add_state(struct flow_object* p, struct flow_object_state* _Owner pnew);

bool flow_object_is_zero_or_null(const struct flow_object* p_object);

bool flow_object_is_not_null(const struct flow_object* p);
bool flow_object_can_be_not_null_or_moved(const struct flow_object* p);

bool flow_object_is_null(const struct flow_object* p);
bool flow_object_can_be_null(const struct flow_object* p);
bool flow_object_can_be_moved(const struct flow_object* p);
bool flow_object_can_be_zero(const struct flow_object* p);



bool flow_object_is_not_zero(const struct flow_object* p);
bool flow_object_is_zero(const struct flow_object* p);

bool flow_object_is_uninitialized(const struct flow_object* p);
bool flow_object_can_be_uninitialized(const struct flow_object* p);

bool flow_object_can_have_its_lifetime_ended(const struct flow_object* p);

void flow_object_print_state(struct flow_object* p);

void object_set_pointer(struct flow_object* p_object, struct flow_object* p_object2);

void flow_object_destroy(_Dtor struct flow_object* p);
void flow_object_delete(struct flow_object* _Owner _Opt p);
void flow_object_swap(struct flow_object* a, struct flow_object* b);
void print_object_line(struct flow_object* p_object, int cols);
void print_object_state_to_str(enum flow_state e, char str[], int sz);

struct declarator;
struct flow_object* _Opt make_flow_object(struct flow_visit_ctx* ctx,
                                     struct type* p_type,
                                     const struct declarator* _Opt p_declarator_opt,
                                     const struct expression* _Opt p_expression_origin);

void flow_object_add_new_state_as_a_copy_of_current_state(struct flow_object* object, const char* name, int state_number);
struct token* _Opt flow_object_get_token(const struct flow_object* object);
void flow_object_remove_state(struct flow_object* object, int state_number);


int flow_object_restore_current_state_from(struct flow_object* object, int state_number);

void flow_object_merge_state(struct flow_object* pdest, struct flow_object* object1, struct flow_object* object2);


struct flow_visit_ctx;
struct token;


void print_flow_object(struct type* p_type, struct flow_object* p_object, bool short_version);

struct marker;

void flow_check_assignment(struct flow_visit_ctx* ctx,
    const struct token* error_position,
    const struct marker* p_a_marker,
    const struct marker* p_b_marker,
    enum  assigment_type assigment_type,
    bool check_uninitialized_b,
    bool a_type_is_view,
    bool a_type_is_nullable,
    struct type* p_a_type, struct flow_object* p_a_object,
    struct type* p_b_type, struct flow_object* p_b_object,
    bool * _Opt set_argument_to_unkown);

void flow_object_set_end_of_lifetime(struct type* p_type, struct flow_object* p_object);
void flow_object_set_zero(struct type* p_type, struct flow_object* p_object);
void flow_object_set_uninitialized(struct type* p_type, struct flow_object* p_object);
void flow_object_set_moved(struct type* p_type, struct flow_object* p_object);

void flow_object_set_unknown(struct type* p_type, bool t_is_nullable, struct flow_object* p_object, bool nullable_enabled);


void checked_read_object(struct flow_visit_ctx* ctx,
    struct type* p_type,
    bool is_nullable,
    struct flow_object* p_object,
    const struct token* _Opt position_token,
    const struct marker* _Opt p_marker,
    bool check_pointed_object);


void flow_end_of_block_visit(struct flow_visit_ctx* ctx,
    struct type* p_type,
    bool type_is_view,
    struct flow_object* p_object,
    const struct token* position_token,
    const char* previous_names);


bool flow_object_is_expansible(const struct flow_object* _Opt p_object);
void flow_object_expand_pointer(struct flow_visit_ctx* ctx, struct type* p_type, struct flow_object* p_object);
void flow_object_push_states_from(const struct flow_object* p_object_from, struct flow_object* p_object_to);

struct flow_object* _Opt expression_get_flow_object(struct flow_visit_ctx* ctx, struct expression* p_expression, bool nullable_enabled);


struct label_state
{
    const char * label_name;
    int state_number;
};

struct flow_visit_ctx
{
    struct secondary_block* _Opt catch_secondary_block_opt;

    struct parser_ctx *ctx;
    _View struct ast ast;    
 
    struct type* _Opt p_return_type;
    int parameter_list;
    
    int state_number_generator;
    bool expression_is_not_evaluated; //true when is expression for sizeof, missing state_set, typeof
    bool inside_assert;
    bool inside_contract;

    /*avoid messages like always something, because in loop the same expression is visited in diferent states*/
    bool inside_loop;

    int throw_join_state; /*state where throws are joined*/
    int break_join_state; /*state where breaks are joined*/
    int initial_state;    /*used to keep the original state*/

    struct flow_objects arena;

    struct label_state labels[100]; //max 100 labels in a function (case not included)
    int labels_size;
};

void flow_visit_ctx_destroy(_Dtor struct flow_visit_ctx* p);
void flow_start_visit_declaration(struct flow_visit_ctx* ctx, struct declaration* p_declaration);



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
 *
 *  The objective of this visit is to build the "defer list" on AST
 *  The defer list is the list of items that will go out of scope.
 *  Each item can point to a declarator or defer.
 *  It is complicated algorithm we make it ready to use on AST
*/

//#pragma once

struct defer_visit_ctx
{
    struct secondary_block* _Opt catch_secondary_block_opt;
    struct parser_ctx *ctx;
    _View struct ast ast;    
    struct defer_defer_scope* _Owner _Opt tail_block;
    int parameter_list;
};

void defer_visit_ctx_destroy(_Dtor struct defer_visit_ctx* p);
void defer_start_visit_declaration(struct defer_visit_ctx* ctx, struct declaration* p_declaration);


#ifdef _WIN32
#endif


/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake 
*/

//#pragma once

#define CAKE_VERSION "0.10.42"





#if defined _MSC_VER && !defined __POCC__
#endif



/*
 *  This file is part of cake compiler
 *  https://github.com/thradams/cake
 *  This visit generates a new and preprocessed C89 code from the AST
 */
 
//#pragma once

struct d_visit_ctx
{
    int indentation;

    /*used to create unique variable names inside functions*/
    int locals_count;

    int extern_count;

    int tag_name_count;
    struct hash_map tag_names;
    struct hash_map structs_map;
    struct hash_map function_map;
    
    /*
       static local are placed in the global scope on-demand.
       This map tell us if some declarator was already exported.
    */
    struct hash_map static_declarators;

    struct osstream local_declarators;
    struct osstream add_this_before;
    struct osstream add_this_before_external_decl;
    bool is_local;
    struct osstream data_types;
    struct osstream function_types;    
    bool zero_mem_used;
    bool memcpy_used;
    /*
    * Points to the function we're in. Or null in file scope.
    */
    struct declarator* _Opt p_current_function_opt;

    struct break_reference
    {
      struct selection_statement * _Opt p_selection_statement;
      struct iteration_statement * _Opt p_iteration_statement;
    } break_reference;

    bool is__func__predefined_identifier_added;

    _View struct ast ast;    
};

void d_visit(struct d_visit_ctx* ctx, struct osstream* oss);
void d_visit_ctx_destroy( _Dtor struct d_visit_ctx* ctx);


#ifdef PATH_MAX
#define MYMAX_PATH PATH_MAX // Linux uses it in realpath
#else
#define MYMAX_PATH MAX_PATH
#endif

NODISCARD
int initializer_init_new(struct parser_ctx* ctx,
                       struct type* p_type, /*in (in/out for arrays [])*/
                        struct object* object, /*in (in/out for arrays [])*/
                        struct initializer* initializer, /*rtocar para initializer item??*/
                        bool is_constant,
                        bool requires_constant_initialization);

struct defer_statement* _Owner _Opt defer_statement(struct parser_ctx* ctx);

void defer_statement_delete(struct defer_statement* _Owner _Opt p)
{
    if (p)
    {
        secondary_block_delete(p->secondary_block);
        free(p);
    }
}

static int s_anonymous_struct_count = 0;

///////////////////////////////////////////////////////////////////////////////
void naming_convention_struct_tag(struct parser_ctx* ctx, struct token* token);
void naming_convention_enum_tag(struct parser_ctx* ctx, struct token* token);
void naming_convention_function(struct parser_ctx* ctx, struct token* token);
void naming_convention_enumerator(struct parser_ctx* ctx, struct token* token);
void naming_convention_struct_member(struct parser_ctx* ctx, struct token* token, struct type* type);
void naming_convention_parameter(struct parser_ctx* ctx, struct token* token, struct type* type);
void naming_convention_global_var(struct parser_ctx* ctx, struct token* token, struct type* type, enum storage_class_specifier_flags storage);
void naming_convention_local_var(struct parser_ctx* ctx, struct token* token, struct type* type);

///////////////////////////////////////////////////////////////////////////////

static bool parser_is_diagnostic_enabled(const struct parser_ctx* ctx, enum diagnostic_id w)
{
    if (w > W_NOTE)
        return true;

    return ((ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].errors & w) != 0) ||
        ((ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings & w) != 0) ||
        ((ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].notes & w) != 0);
}

static void check_open_brace_style(struct parser_ctx* ctx, struct token* token)
{
    // token points to {

    if (token->level == 0 &&
        !(token->flags & TK_FLAG_MACRO_EXPANDED) &&
        token->type == '{' &&
        token->prev &&
        parser_is_diagnostic_enabled(ctx, W_STYLE))
    {
        if (ctx->options.style == STYLE_CAKE)
        {
            if (token->prev->type == TK_BLANKS &&
                token->prev->prev &&
                token->prev->prev->type == TK_NEWLINE)
            {
            }
            else
            {
                compiler_diagnostic(W_STYLE, ctx, token, NULL, "not following correct brace style {");
            }
        }
    }
}

static void check_close_brace_style(struct parser_ctx* ctx, struct token* token)
{
    // token points to {

    if (token->level == 0 &&
        !(token->flags & TK_FLAG_MACRO_EXPANDED) &&
        token->type == '}' &&
        token->prev &&
        token->prev->prev &&
        parser_is_diagnostic_enabled(ctx, W_STYLE))
    {
        if (ctx->options.style == STYLE_CAKE)
        {
            if (token->prev->type == TK_BLANKS &&
                token->prev->prev->type == TK_NEWLINE)
            {
            }
            else
            {
                compiler_diagnostic(W_STYLE, ctx, token, NULL, "not following correct close brace style }");
            }
        }
    }
}

static void check_func_open_brace_style(struct parser_ctx* ctx, struct token* token)
{
    // token points to {

    if (token->level == 0 &&
        !(token->flags & TK_FLAG_MACRO_EXPANDED) &&
        token->type == '{' &&
        token->prev &&
        parser_is_diagnostic_enabled(ctx, W_STYLE))
    {
        if (ctx->options.style == STYLE_CAKE)
        {
            if (token->prev->type == TK_NEWLINE)
            {
            }
            else
            {
                compiler_diagnostic(W_STYLE, ctx, token, NULL, "not following correct brace style {");
            }
        }
    }
}
/*
static void check_func_close_brace_style(struct parser_ctx* ctx, struct token* token)
{
    //token points to {

    if (token->level == 0 &&
        !(token->flags & TK_FLAG_MACRO_EXPANDED) &&
        token->type == '}' &&
        parser_is_warning_enabled(ctx, W_STYLE))
    {
        if (ctx->options.style == STYLE_CAKE)
        {
            if (token->prev->prev->type == TK_NEWLINE)
            {
            }
            else
            {
                compiler_diagnostic(W_STYLE, ctx, token, "not following correct close brace style }");
            }
        }
    }
}
*/


void scope_destroy(_Dtor struct scope* p)
{
    hashmap_destroy(&p->tags);
    hashmap_destroy(&p->variables);
}

void scope_list_push(struct scope_list* list, struct scope* pnew)
{
    if (list->tail)
        pnew->scope_level = list->tail->scope_level + 1;

    if (list->head == NULL)
    {
        list->head = pnew;
        list->tail = pnew;
        // pnew->prev = list->tail;
    }
    else
    {
        assert(list->tail != NULL);
        pnew->previous = list->tail;
        list->tail->next = pnew;
        list->tail = pnew;
    }
}

void scope_list_pop(struct scope_list* list)
{

    if (list->head == NULL)
        return;
    assert(list->tail != NULL);
    struct scope* p = list->tail;
    if (list->head == list->tail)
    {
        list->head = NULL;
        list->tail = NULL;
    }
    else
    {
        list->tail = list->tail->previous;
        if (list->tail == list->head)
        {
            assert(list->tail != NULL);
            list->tail->next = NULL;
            list->tail->previous = NULL;
        }
    }
    p->next = NULL;
    p->previous = NULL;
}

void parser_ctx_destroy(_Opt _Dtor struct parser_ctx* ctx)
{
    label_list_clear(&ctx->label_list);
    assert(ctx->label_list.head == NULL);
    assert(ctx->label_list.tail == NULL);

    if (ctx->sarif_file)
    {
        fclose(ctx->sarif_file);
    }
}

static void stringfy(const char* input, char* json_str_message, int output_size)
{
    json_str_message[0] = '\0'; //out

    int k = 0;
    while (*input != '\0')
    {
        if (*input == '\"')
        {
            if (k < output_size)
                json_str_message[k] = '\\';
            k++;
            if (k < output_size)
                json_str_message[k] = '"';
            k++;
            input++;
        }
        else if (*input == '\n')
        {
            if (k < output_size)
                json_str_message[k] = '\\';
            k++;
            if (k < output_size)
                json_str_message[k] = 'n';
            k++;
            input++;
        }
        else
        {
            if (k < output_size)
                json_str_message[k] = *input;
            k++;
            input++;
        }
    }
    if (k < output_size)
        json_str_message[k] = '\0';
    else
        json_str_message[output_size - 1] = '\0';
}

_Bool compiler_diagnostic(enum diagnostic_id w,
    const struct parser_ctx* ctx,
    const struct token* _Opt p_token_opt,
    const struct marker* _Opt p_marker_temp,
    const char* fmt, ...)
{
    bool included_file_location = false;
    struct marker marker = { 0 };
    if (p_marker_temp == NULL)
    {
        if (p_token_opt == NULL) return false;

        marker.file = p_token_opt->token_origin->lexeme;
        marker.line = p_token_opt->line;
        marker.start_col = p_token_opt->col;
        marker.end_col = p_token_opt->col;
        marker.p_token_caret = p_token_opt;
        included_file_location = p_token_opt->level > 0;
    }
    else
    {
        //assert(p_token_opt == NULL);
        marker = *p_marker_temp;
        if (marker.p_token_caret)
            p_token_opt = marker.p_token_caret;
        else if (marker.p_token_begin)
            p_token_opt = marker.p_token_begin;

        if (p_token_opt == NULL) return false;
        marker.file = p_token_opt->token_origin->lexeme;
        included_file_location = p_token_opt->level > 0;

        marker.line = p_token_opt->line;
        marker.start_col = p_token_opt->col;
        marker.end_col = p_token_opt->col;
    }

    bool is_error = false;
    bool is_warning = false;
    bool is_note = false;

    if (is_diagnostic_configurable(w))
    {
        is_error =
            (ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].errors & (1ULL << w)) != 0;

        is_warning =
            (ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings & (1ULL << w)) != 0;

        is_note =
            ((ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].notes & (1ULL << w)) != 0);
    }
    else
    {
        is_note = is_diagnostic_note(w);
        is_error = is_diagnostic_error(w);
        is_warning = is_diagnostic_warning(w);
    }

    if (is_error)
    {
        ctx->p_report->error_count++;
    }
    else if (is_warning)
    {
        /*warnings inside headers are ignored*/
        if (included_file_location)
        {
            return false;
        }

        ctx->p_report->warnings_count++;
    }
    else if (is_note)
    {
        /*notes inside headers are ignored*/
        if (included_file_location)
        {
            return false;
        }

        if (w != W_LOCATION)
            ctx->p_report->info_count++;
    }
    else
    {
        return false;
    }

    if (w != W_LOCATION)
    {
        //index 0 is the most recent
        ctx->p_report->last_diagnostics_ids[1] = ctx->p_report->last_diagnostics_ids[0];
        ctx->p_report->last_diagnostics_ids[0] = w;
    }

    const char* func_name = "module";
    if (ctx->p_current_function_opt)
    {
        if (ctx->p_current_function_opt->name_opt)
            func_name = ctx->p_current_function_opt->name_opt->lexeme;
        else
            func_name = "unnamed";
    }

    char buffer[200] = { 0 };

    char diagnostic_name[100] = { 0 };
    get_warning_name(w, sizeof diagnostic_name, diagnostic_name);




    print_position(marker.file, marker.line, marker.start_col, ctx->options.visual_studio_ouput_format);

#pragma CAKE diagnostic push
#pragma CAKE diagnostic ignored "-Wnullable-to-non-nullable"
#pragma CAKE diagnostic ignored "-Wanalyzer-null-dereference"

    va_list args = { 0 };
    va_start(args, fmt);
    /*int n =*/vsnprintf(buffer, sizeof(buffer), fmt, args);
    va_end(args);

#pragma CAKE diagnostic pop

    //bool show_warning_name = w < W_NOTE && w != W_LOCATION;


    if (ctx->options.visual_studio_ouput_format)
    {
        if (is_error)
            printf("error: ");
        else if (is_warning)
            printf("warning: ");
        else if (is_note)
            printf("note: ");

        printf("%s", buffer);

        printf(" [%s]\n", diagnostic_name);
    }
    else
    {
        if (is_error)
        {
            printf(LIGHTRED "error: " WHITE "%s [" LIGHTRED "%s" WHITE "]\n" RESET, buffer, diagnostic_name);
        }
        else if (is_warning)
        {
            printf(LIGHTMAGENTA "warning: " WHITE "%s [" LIGHTMAGENTA "%s" WHITE "]\n" RESET, buffer, diagnostic_name);
        }
        else if (is_note)
        {
            if (w == W_LOCATION)
                printf(LIGHTCYAN "note: " WHITE "%s\n" RESET, buffer);
            else
                printf(LIGHTCYAN "note: " WHITE "%s [" LIGHTCYAN "%s" WHITE "]\n" RESET, buffer, diagnostic_name);
        }
    }

    print_line_and_token(&marker, ctx->options.visual_studio_ouput_format);


    if (ctx->sarif_file)
    {

        char json_str_message[200] = { 0 };
        stringfy(buffer, json_str_message, sizeof json_str_message);

        if (ctx->sarif_entries > 0)
        {
            fprintf(ctx->sarif_file, "   ,\n");
        }

        ((struct parser_ctx*)ctx)->sarif_entries++;

        fprintf(ctx->sarif_file, "   {\n");
        fprintf(ctx->sarif_file, "     \"ruleId\":\"%s\",\n", diagnostic_name);

        if (is_error)
            fprintf(ctx->sarif_file, "     \"level\":\"error\",\n");
        else if (is_warning)
            fprintf(ctx->sarif_file, "     \"level\":\"warning\",\n");
        else if (is_note)
            fprintf(ctx->sarif_file, "     \"level\":\"note\",\n");

        fprintf(ctx->sarif_file, "     \"message\": {\n");
        fprintf(ctx->sarif_file, "            \"text\": \"%s\"\n", json_str_message);
        fprintf(ctx->sarif_file, "      },\n");
        fprintf(ctx->sarif_file, "      \"locations\": [\n");
        fprintf(ctx->sarif_file, "       {\n");

        fprintf(ctx->sarif_file, "       \"physicalLocation\": {\n");

        fprintf(ctx->sarif_file, "             \"artifactLocation\": {\n");
        fprintf(ctx->sarif_file, "                 \"uri\": \"file:///%s\"\n", marker.file);
        fprintf(ctx->sarif_file, "              },\n");

        fprintf(ctx->sarif_file, "              \"region\": {\n");
        fprintf(ctx->sarif_file, "                  \"startLine\": %d,\n", marker.line);
        fprintf(ctx->sarif_file, "                  \"startColumn\": %d,\n", marker.start_col);
        fprintf(ctx->sarif_file, "                  \"endLine\": %d,\n", marker.line);
        fprintf(ctx->sarif_file, "                  \"endColumn\": %d\n", marker.end_col);
        fprintf(ctx->sarif_file, "               }\n");
        fprintf(ctx->sarif_file, "         },\n");

        fprintf(ctx->sarif_file, "         \"logicalLocations\": [\n");
        fprintf(ctx->sarif_file, "          {\n");

        fprintf(ctx->sarif_file, "              \"fullyQualifiedName\": \"%s\",\n", func_name);
        fprintf(ctx->sarif_file, "              \"decoratedName\": \"%s\",\n", func_name);

        fprintf(ctx->sarif_file, "              \"kind\": \"%s\"\n", "function");
        fprintf(ctx->sarif_file, "          }\n");

        fprintf(ctx->sarif_file, "         ]\n");

        fprintf(ctx->sarif_file, "       }\n");
        fprintf(ctx->sarif_file, "     ]\n");

        fprintf(ctx->sarif_file, "   }\n");
    }

    return 1;
}

void print_scope(struct scope_list* e)
{
    printf("--- begin of scope---\n");
    struct scope* _Opt p = e->head;
    int level = 0;
    while (p)
    {
        if (p->variables.table)
        {
            for (int i = 0; i < p->variables.capacity; i++)
            {
                if (p->variables.table[i])
                {
                    for (int k = 0; k < level; k++)
                        printf(" ");
                    printf("%s\n", p->variables.table[i]->key);
                }
            }

            for (int i = 0; i < p->tags.capacity; i++)
            {
                if (p->tags.table[i])
                {
                    for (int k = 0; k < level; k++)
                        printf(" ");
                    printf("tag %s\n", p->tags.table[i]->key);
                }
            }
        }

        level++;
        p = p->next;
    }
    printf("--- end of scope---\n");
}


bool first_of_function_specifier(const struct parser_ctx* ctx)
{
    struct token* _Opt token = ctx->current;

    if (token == NULL)
        return false;

    return token->type == TK_KEYWORD_INLINE ||
        token->type == TK_KEYWORD__NORETURN;
}

bool first_of_enum_specifier_token(const struct token* token)
{
    return token->type == TK_KEYWORD_ENUM;
}

bool first_of_enum_specifier(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return first_of_enum_specifier_token(ctx->current);
}


bool first_of_alignment_specifier(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;
    return ctx->current->type == TK_KEYWORD__ALIGNAS;
}

bool first_of_atomic_type_specifier(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    /*
      If the _Atomic keyword is immediately followed by a left parenthesis, it is interpreted
      as a type specifier (with a type name), not as a type qualifier.
    */

    if (ctx->current->type == TK_KEYWORD__ATOMIC)
    {
        struct token* _Opt ahead = parser_look_ahead(ctx);
        if (ahead != NULL)
        {
            return ahead->type == '(';
        }
    }
    return false;
}

bool first_of_storage_class_specifier(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_KEYWORD_TYPEDEF ||
        ctx->current->type == TK_KEYWORD_CONSTEXPR ||
        ctx->current->type == TK_KEYWORD_EXTERN ||
        ctx->current->type == TK_KEYWORD_STATIC ||
        ctx->current->type == TK_KEYWORD__THREAD_LOCAL ||
        ctx->current->type == TK_KEYWORD_AUTO ||
        ctx->current->type == TK_KEYWORD_REGISTER;
}

bool first_of_struct_or_union_token(const struct token* token)
{
    return token->type == TK_KEYWORD_STRUCT || token->type == TK_KEYWORD_UNION;
}

bool first_of_struct_or_union(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return first_of_struct_or_union_token(ctx->current);
}

bool first_of_type_qualifier_token(const struct token* p_token)
{

    return p_token->type == TK_KEYWORD_CONST ||
        p_token->type == TK_KEYWORD_RESTRICT ||
        p_token->type == TK_KEYWORD_VOLATILE ||
        p_token->type == TK_KEYWORD__ATOMIC ||

        //MSVC
        p_token->type == TK_KEYWORD_MSVC__PTR32 ||
        p_token->type == TK_KEYWORD_MSVC__PTR64 ||

        /*extensions*/
        p_token->type == TK_KEYWORD__CTOR ||
        p_token->type == TK_KEYWORD_CAKE_OWNER ||
        p_token->type == TK_KEYWORD__DTOR ||
        p_token->type == TK_KEYWORD_CAKE_VIEW ||
        p_token->type == TK_KEYWORD_CAKE_OPT;

    //__fastcall
    //__stdcall
}

bool first_of_type_qualifier(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return first_of_type_qualifier_token(ctx->current);
}

struct map_entry* _Opt find_tag(struct parser_ctx* ctx, const char* lexeme)
{
    struct scope* _Opt scope = ctx->scopes.tail;
    while (scope)
    {
        struct map_entry* _Opt p_entry = hashmap_find(&scope->tags, lexeme);
        if (p_entry)
        {
            return p_entry;
        }
        scope = scope->previous;
    }
    return NULL;
}

struct map_entry* _Opt find_variables(const struct parser_ctx* ctx, const char* lexeme, struct scope* _Opt* _Opt ppscope_opt)
{
    if (ppscope_opt != NULL)
        *ppscope_opt = NULL; // out

    struct scope* _Opt scope = ctx->scopes.tail;
    while (scope)
    {
        struct map_entry* _Opt p_entry = hashmap_find(&scope->variables, lexeme);
        if (p_entry)
        {
            if (ppscope_opt)
                *ppscope_opt = scope;
            return p_entry;
        }
        scope = scope->previous;
    }
    return NULL;
}

struct enum_specifier* _Opt find_enum_specifier(struct parser_ctx* ctx, const char* lexeme)
{
    struct enum_specifier* _Opt best = NULL;
    struct scope* _Opt scope = ctx->scopes.tail;
    while (scope)
    {
        struct map_entry* _Opt p_entry = hashmap_find(&scope->tags, lexeme);
        if (p_entry &&
            p_entry->type == TAG_TYPE_ENUN_SPECIFIER)
        {
            assert(p_entry->data.p_enum_specifier != NULL);

            best = p_entry->data.p_enum_specifier;
            if (best->enumerator_list.head != NULL)
                return best; // OK bem completo
            else
            {
                // it's not complete let's keep going up
            }
        }
        scope = scope->previous;
    }
    return best; // mesmo que nao seja tao completo vamos retornar.
}

struct struct_or_union_specifier* _Opt find_struct_or_union_specifier(const struct parser_ctx* ctx, const char* lexeme)
{
    struct struct_or_union_specifier* _Opt p = NULL;
    struct scope* _Opt scope = ctx->scopes.tail;
    while (scope)
    {
        struct map_entry* _Opt p_entry = hashmap_find(&scope->tags, lexeme);
        if (p_entry &&
            p_entry->type == TAG_TYPE_STRUCT_OR_UNION_SPECIFIER)
        {
            assert(p_entry->data.p_struct_or_union_specifier != NULL);
            p = p_entry->data.p_struct_or_union_specifier;
            break;
        }
        scope = scope->previous;
    }
    return p;
}

struct declarator* _Opt find_declarator(const struct parser_ctx* ctx, const char* lexeme, struct scope** _Opt ppscope_opt)
{
    struct map_entry* _Opt p_entry = find_variables(ctx, lexeme, ppscope_opt);

    if (p_entry)
    {
        if (p_entry->type == TAG_TYPE_INIT_DECLARATOR)
        {
            assert(p_entry->data.p_init_declarator != NULL);
            struct init_declarator* p_init_declarator = p_entry->data.p_init_declarator;
            return (struct declarator*)p_init_declarator->p_declarator;
        }
        else if (p_entry->type == TAG_TYPE_DECLARATOR)
        {
            return p_entry->data.p_declarator;
        }
    }

    return NULL;
}

struct enumerator* _Opt find_enumerator(const struct parser_ctx* ctx, const char* lexeme, struct scope** _Opt ppscope_opt)
{
    struct map_entry* _Opt p_entry = find_variables(ctx, lexeme, ppscope_opt);

    if (p_entry && p_entry->type == TAG_TYPE_ENUMERATOR)
        return p_entry->data.p_enumerator;

    return NULL;
}

bool first_of_typedef_name(const struct parser_ctx* ctx, struct token* p_token)
{

    if (p_token->type != TK_IDENTIFIER)
    {
        // no need to check
        return false;
    }
    if (p_token->flags & TK_FLAG_IDENTIFIER_IS_TYPEDEF)
    {
        // it has already been verified that it is a typedef
        return true;
    }
    if (p_token->flags & TK_FLAG_IDENTIFIER_IS_NOT_TYPEDEF)
    {
        // it has already been verified that it is NOT a typedef
        return false;
    }

    struct declarator* _Opt p_declarator = find_declarator(ctx, p_token->lexeme, NULL);

    if (p_declarator &&
        p_declarator->declaration_specifiers &&
        (p_declarator->declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_TYPEDEF))
    {
        p_declarator->num_uses++;
        p_token->flags |= TK_FLAG_IDENTIFIER_IS_TYPEDEF;
        return true;
    }
    else
    {
        p_token->flags |= TK_FLAG_IDENTIFIER_IS_NOT_TYPEDEF;
    }
    return false;
}

bool first_of_type_specifier(const struct parser_ctx* ctx);
bool first_of_type_specifier_token(const struct parser_ctx* ctx, struct token* token);

bool first_of_type_name_ahead(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    if (ctx->current->type != '(')
        return false;

    struct token* _Opt token_ahead = parser_look_ahead(ctx);

    if (token_ahead == NULL)
        return false;

    return first_of_type_specifier_token(ctx, token_ahead) ||
        first_of_type_qualifier_token(token_ahead);
}

bool first_of_type_name(const struct parser_ctx* ctx)
{
    return first_of_type_specifier(ctx) || first_of_type_qualifier(ctx);
}

bool first_of_type_specifier_token(const struct parser_ctx* ctx, struct token* p_token)
{
    return p_token->type == TK_KEYWORD_VOID ||
        p_token->type == TK_KEYWORD_CHAR ||
        p_token->type == TK_KEYWORD_SHORT ||
        p_token->type == TK_KEYWORD_INT ||
        p_token->type == TK_KEYWORD_LONG ||

        // microsoft extension
        p_token->type == TK_KEYWORD_MSVC__INT8 ||
        p_token->type == TK_KEYWORD_MSVC__INT16 ||
        p_token->type == TK_KEYWORD_MSVC__INT32 ||
        p_token->type == TK_KEYWORD_MSVC__INT64 ||
        p_token->type == TK_KEYWORD_MSVC__DECLSPEC ||

        // end microsoft

        p_token->type == TK_KEYWORD_FLOAT ||
        p_token->type == TK_KEYWORD_DOUBLE ||
        p_token->type == TK_KEYWORD_SIGNED ||
        p_token->type == TK_KEYWORD_UNSIGNED ||
        p_token->type == TK_KEYWORD__BITINT ||
        p_token->type == TK_KEYWORD__BOOL ||
        p_token->type == TK_KEYWORD__COMPLEX ||
        p_token->type == TK_KEYWORD__DECIMAL32 ||
        p_token->type == TK_KEYWORD__DECIMAL64 ||
        p_token->type == TK_KEYWORD__DECIMAL128 ||
        p_token->type == TK_KEYWORD_TYPEOF ||        // C23
        p_token->type == TK_KEYWORD_TYPEOF_UNQUAL || // C23
        first_of_atomic_type_specifier(ctx) ||
        first_of_struct_or_union_token(p_token) ||
        first_of_enum_specifier_token(p_token) ||
        first_of_typedef_name(ctx, p_token);
}

bool first_of_type_specifier(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;
    return first_of_type_specifier_token(ctx, ctx->current);
}

bool first_of_type_specifier_qualifier(const struct parser_ctx* ctx)
{
    return first_of_type_specifier(ctx) ||
        first_of_type_qualifier(ctx) ||
        first_of_alignment_specifier(ctx);
}

bool first_of_compound_statement(const struct parser_ctx* ctx)
{
    return ctx->current != NULL && ctx->current->type == '{';
}

bool first_of_jump_statement(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_KEYWORD_GOTO ||
        ctx->current->type == TK_KEYWORD_CONTINUE ||
        ctx->current->type == TK_KEYWORD_BREAK ||
        ctx->current->type == TK_KEYWORD_RETURN ||
        ctx->current->type == TK_KEYWORD_CAKE_THROW /*extension*/;
}

bool first_of_selection_statement(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_KEYWORD_IF ||
        ctx->current->type == TK_KEYWORD_SWITCH;
}

bool first_of_iteration_statement(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_KEYWORD_WHILE ||
        ctx->current->type == TK_KEYWORD_DO ||
        ctx->current->type == TK_KEYWORD_FOR;
}

bool first_of_label(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    if (ctx->current->type == TK_IDENTIFIER)
    {
        struct token* _Opt next = parser_look_ahead(ctx);
        return next && next->type == ':';
    }
    else if (ctx->current->type == TK_KEYWORD_CASE)
    {
        return true;
    }
    else if (ctx->current->type == TK_KEYWORD_DEFAULT)
    {
        return true;
    }

    return false;
}

bool first_of_declaration_specifier(const struct parser_ctx* ctx)
{
    /*
    declaration-specifier:
    storage-class-specifier
    type-specifier-qualifier
    function-specifier
    */
    return first_of_storage_class_specifier(ctx) ||
        first_of_function_specifier(ctx) ||
        first_of_type_specifier_qualifier(ctx);
}

bool first_of_pragma_declaration(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_PRAGMA;
}

bool first_of_static_assert_declaration(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == TK_KEYWORD__STATIC_ASSERT ||
        ctx->current->type == TK_KEYWORD_CAKE_STATIC_DEBUG ||
        ctx->current->type == TK_KEYWORD_CAKE_STATIC_DEBUG_EX ||
        ctx->current->type == TK_KEYWORD_STATIC_STATE ||
        ctx->current->type == TK_KEYWORD_STATIC_SET;
}

bool first_of_attribute_specifier(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    if (ctx->current->type != '[')
    {
        return false;
    }
    struct token* _Opt p_token = parser_look_ahead(ctx);
    return p_token != NULL && p_token->type == '[';
}

bool first_of_labeled_statement(const struct parser_ctx* ctx)
{
    return first_of_label(ctx);
}

bool first_of_designator(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return false;

    return ctx->current->type == '[' || ctx->current->type == '.';
}

struct token* _Opt previous_parser_token(const struct token* token)
{
    if (token->prev == NULL)
    {
        return NULL;
    }

    struct token* _Opt prev = token->prev;
    while (prev && !(prev->flags & TK_FLAG_FINAL))
    {
        prev = prev->prev;
    }

    return prev;
}

enum token_type is_keyword(const char* text)
{
    switch (text[0])
    {
    case 'a':
        if (strcmp("alignof", text) == 0)
            return TK_KEYWORD__ALIGNOF;
        if (strcmp("auto", text) == 0)
            return TK_KEYWORD_AUTO;
        if (strcmp("alignas", text) == 0)
            return TK_KEYWORD__ALIGNAS; /*C23 alternate spelling _Alignas*/
        if (strcmp("alignof", text) == 0)
            return TK_KEYWORD__ALIGNAS; /*C23 alternate spelling _Alignof*/
        if (strcmp("assert", text) == 0)
            return TK_KEYWORD_ASSERT; /*extension*/
        break;

    case 'b':
        if (strcmp("break", text) == 0)
            return TK_KEYWORD_BREAK;
        if (strcmp("bool", text) == 0)
            return TK_KEYWORD__BOOL; /*C23 alternate spelling _Bool*/
        break;

    case 'c':
        if (strcmp("case", text) == 0)
            return TK_KEYWORD_CASE;
        if (strcmp("char", text) == 0)
            return TK_KEYWORD_CHAR;
        if (strcmp("const", text) == 0)
            return TK_KEYWORD_CONST;
        if (strcmp("constexpr", text) == 0)
            return TK_KEYWORD_CONSTEXPR;
        if (strcmp("continue", text) == 0)
            return TK_KEYWORD_CONTINUE;
        if (strcmp("catch", text) == 0)
            return TK_KEYWORD_CAKE_CATCH;
        break;

    case 'd':
        if (strcmp("default", text) == 0)
            return TK_KEYWORD_DEFAULT;
        if (strcmp("do", text) == 0)
            return TK_KEYWORD_DO;
        if (strcmp("defer", text) == 0)
            return TK_KEYWORD_DEFER;
        if (strcmp("double", text) == 0)
            return TK_KEYWORD_DOUBLE;
        break;

    case 'e':
        if (strcmp("else", text) == 0)
            return TK_KEYWORD_ELSE;
        if (strcmp("enum", text) == 0)
            return TK_KEYWORD_ENUM;
        if (strcmp("extern", text) == 0)
            return TK_KEYWORD_EXTERN;
        break;

    case 'f':
        if (strcmp("float", text) == 0)
            return TK_KEYWORD_FLOAT;
        if (strcmp("for", text) == 0)
            return TK_KEYWORD_FOR;
        if (strcmp("false", text) == 0)
            return TK_KEYWORD_FALSE;
        break;

    case 'g':
        if (strcmp("goto", text) == 0)
            return TK_KEYWORD_GOTO;
        break;

    case 'i':
        if (strcmp("if", text) == 0)
            return TK_KEYWORD_IF;
        if (strcmp("inline", text) == 0)
            return TK_KEYWORD_INLINE;
        if (strcmp("int", text) == 0)
            return TK_KEYWORD_INT;
        break;

    case 'n':
        if (strcmp("nullptr", text) == 0)
            return TK_KEYWORD_NULLPTR;
        break;

    case 'l':
        if (strcmp("long", text) == 0)
            return TK_KEYWORD_LONG;
        break;

    case 'r':
        if (strcmp("register", text) == 0)
            return TK_KEYWORD_REGISTER;
        if (strcmp("restrict", text) == 0)
            return TK_KEYWORD_RESTRICT;
        if (strcmp("return", text) == 0)
            return TK_KEYWORD_RETURN;
        break;

    case 's':
        if (strcmp("short", text) == 0)
            return TK_KEYWORD_SHORT;
        if (strcmp("signed", text) == 0)
            return TK_KEYWORD_SIGNED;
        if (strcmp("sizeof", text) == 0)
            return TK_KEYWORD_SIZEOF;
        if (strcmp("static", text) == 0)
            return TK_KEYWORD_STATIC;
        if (strcmp("struct", text) == 0)
            return TK_KEYWORD_STRUCT;
        if (strcmp("switch", text) == 0)
            return TK_KEYWORD_SWITCH;
        if (strcmp("static_assert", text) == 0)
            return TK_KEYWORD__STATIC_ASSERT; /*C23 alternate spelling _Static_assert*/
        if (strcmp("static_debug", text) == 0)
            return TK_KEYWORD_CAKE_STATIC_DEBUG;
        if (strcmp("static_debug_ex", text) == 0)
            return TK_KEYWORD_CAKE_STATIC_DEBUG_EX;
        if (strcmp("static_state", text) == 0)
            return TK_KEYWORD_STATIC_STATE;
        if (strcmp("static_set", text) == 0)
            return TK_KEYWORD_STATIC_SET;
        break;

    case 't':
        if (strcmp("typedef", text) == 0)
            return TK_KEYWORD_TYPEDEF;
        if (strcmp("typeof", text) == 0)
            return TK_KEYWORD_TYPEOF; /*C23*/
        if (strcmp("typeof_unqual", text) == 0)
            return TK_KEYWORD_TYPEOF_UNQUAL; /*C23*/
        if (strcmp("true", text) == 0)
            return TK_KEYWORD_TRUE; /*C23*/
        if (strcmp("thread_local", text) == 0)
            return TK_KEYWORD__THREAD_LOCAL; /*C23 alternate spelling _Thread_local*/
        if (strcmp("try", text) == 0)
            return TK_KEYWORD_CAKE_TRY;
        if (strcmp("throw", text) == 0)
            return TK_KEYWORD_CAKE_THROW;
        break;

    case 'u':
        if (strcmp("union", text) == 0)
            return TK_KEYWORD_UNION;
        if (strcmp("unsigned", text) == 0)
            return TK_KEYWORD_UNSIGNED;
        break;

    case 'v':
        if (strcmp("void", text) == 0)
            return TK_KEYWORD_VOID;
        if (strcmp("volatile", text) == 0)
            return TK_KEYWORD_VOLATILE;
        break;

    case 'w':
        if (strcmp("while", text) == 0)
            return TK_KEYWORD_WHILE;
        break;
    case '_':

        /*ownership*/
        if (strcmp("_Ctor", text) == 0)
            return TK_KEYWORD__CTOR; /*extension*/
        if (strcmp("_Owner", text) == 0)
            return TK_KEYWORD_CAKE_OWNER; /*extension*/
        if (strcmp("_Dtor", text) == 0)
            return TK_KEYWORD__DTOR; /*extension*/
        if (strcmp("_Opt", text) == 0)
            return TK_KEYWORD_CAKE_OPT; /*extension*/

        if (strcmp("_View", text) == 0)
            return TK_KEYWORD_CAKE_VIEW; /*extension*/

        if (strcmp("_Countof", text) == 0)
            return TK_KEYWORD__COUNTOF; /*C2Y*/

        /*TRAITS EXTENSION*/
        if (strcmp("_is_lvalue", text) == 0)
            return TK_KEYWORD_IS_LVALUE;
        if (strcmp("_is_const", text) == 0)
            return TK_KEYWORD_IS_CONST;
        if (strcmp("_is_owner", text) == 0)
            return TK_KEYWORD_IS_OWNER;
        if (strcmp("_is_pointer", text) == 0)
            return TK_KEYWORD_IS_POINTER;
        if (strcmp("_is_array", text) == 0)
            return TK_KEYWORD_IS_ARRAY;
        if (strcmp("_is_function", text) == 0)
            return TK_KEYWORD_IS_FUNCTION;
        if (strcmp("_is_arithmetic", text) == 0)
            return TK_KEYWORD_IS_ARITHMETIC;
        if (strcmp("_is_floating_point", text) == 0)
            return TK_KEYWORD_IS_FLOATING_POINT;
        if (strcmp("_is_integral", text) == 0)
            return TK_KEYWORD_IS_INTEGRAL;
        if (strcmp("_is_scalar", text) == 0)
            return TK_KEYWORD_IS_SCALAR;
        /*TRAITS EXTENSION*/

        if (strcmp("_Alignof", text) == 0)
            return TK_KEYWORD__ALIGNOF;
        if (strcmp("_Alignas", text) == 0)
            return TK_KEYWORD__ALIGNAS;
        if (strcmp("_Atomic", text) == 0)
            return TK_KEYWORD__ATOMIC;

        if (strcmp("__ptr32", text) == 0)
            return TK_KEYWORD_MSVC__PTR32;
        if (strcmp("__ptr64", text) == 0)
            return TK_KEYWORD_MSVC__PTR64;


        if (strcmp("_Bool", text) == 0)
            return TK_KEYWORD__BOOL;
        if (strcmp("_Complex", text) == 0)
            return TK_KEYWORD__COMPLEX;
        if (strcmp("_Decimal32", text) == 0)
            return TK_KEYWORD__DECIMAL32;
        if (strcmp("_Decimal64", text) == 0)
            return TK_KEYWORD__DECIMAL64;
        if (strcmp("_Decimal128", text) == 0)
            return TK_KEYWORD__DECIMAL128;
        if (strcmp("_Generic", text) == 0)
            return TK_KEYWORD__GENERIC;
        if (strcmp("_Imaginary", text) == 0)
            return TK_KEYWORD__IMAGINARY;
        if (strcmp("_Noreturn", text) == 0)
            return TK_KEYWORD__NORETURN; /*_Noreturn deprecated C23*/
        if (strcmp("_Static_assert", text) == 0)
            return TK_KEYWORD__STATIC_ASSERT;
        if (strcmp("_Thread_local", text) == 0)
            return TK_KEYWORD__THREAD_LOCAL;
        if (strcmp("_BitInt", text) == 0)
            return TK_KEYWORD__BITINT; /*(C23)*/
        if (strcmp("__typeof__", text) == 0)
            return TK_KEYWORD_TYPEOF; /*(C23)*/
#ifdef  _MSC_VER
        // begin microsoft
        if (strcmp("__int8", text) == 0)
            return TK_KEYWORD_MSVC__INT8;
        if (strcmp("__int16", text) == 0)
            return TK_KEYWORD_MSVC__INT16;
        if (strcmp("__int32", text) == 0)
            return TK_KEYWORD_MSVC__INT32;
        if (strcmp("__int64", text) == 0)
            return TK_KEYWORD_MSVC__INT64;
        if (strcmp("__forceinline", text) == 0)
            return TK_KEYWORD_INLINE;
        if (strcmp("__inline", text) == 0)
            return TK_KEYWORD_INLINE;
        if (strcmp("_asm", text) == 0 || strcmp("__asm", text) == 0)
            return TK_KEYWORD__ASM;
        if (strcmp("__stdcall", text) == 0 || strcmp("_stdcall", text) == 0)
            return TK_KEYWORD_MSVC__STDCALL;
        if (strcmp("__cdecl", text) == 0)
            return TK_KEYWORD_MSVC__CDECL;
        if (strcmp("__fastcall", text) == 0)
            return TK_KEYWORD_MSVC__FASTCALL;
        if (strcmp("__alignof", text) == 0)
            return TK_KEYWORD__ALIGNOF;
        if (strcmp("__restrict", text) == 0)
            return TK_KEYWORD_RESTRICT;
        if (strcmp("__declspec", text) == 0)
            return TK_KEYWORD_MSVC__DECLSPEC;
#endif
        break;
    default:
        break;
    }

    return 0;
}


static void token_promote(const struct parser_ctx* ctx, struct token* token)
{
    if (token->type == TK_IDENTIFIER_RECURSIVE_MACRO)
    {
        // talvez desse para remover antesisso..
        // assim que sai do tetris
        // virou passado
        token->type = TK_IDENTIFIER; /*nao precisamos mais disso*/
    }

    if (token->type == TK_IDENTIFIER)
    {
        enum token_type t = is_keyword(token->lexeme);
        if (t != TK_NONE)
            token->type = t;
    }
    else if (token->type == TK_PPNUMBER)
    {
        char errormsg[100] = { 0 };
        char suffix[4] = { 0 };
        token->type = parse_number(token->lexeme, suffix, errormsg);
        if (token->type == TK_NONE)
        {
            compiler_diagnostic(C_INVALID_TOKEN, ctx, token, NULL, errormsg);
        }
    }
}

struct token* _Opt parser_look_ahead(const struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return NULL;

    struct token* _Opt p = ctx->current->next;
    while (p && !(p->flags & TK_FLAG_FINAL))
    {
        p = p->next;
    }

    if (p)
    {
        token_promote(ctx, p);
    }

    return p;
}


static struct token* _Opt pragma_match(const struct token* p_current)
{
    struct token* _Opt p_token = p_current->next;
    while (p_token && p_token->type == TK_BLANKS)
    {
        p_token = p_token->next;
    }
    return p_token;
}

static void pragma_skip_blanks(struct parser_ctx* ctx)
{
    while (ctx->current && ctx->current->type == TK_BLANKS)
    {
        ctx->current = ctx->current->next;
    }
}

/*
 * Some pragmas needs to be handled by the compiler
 */
static void parse_pragma(struct parser_ctx* ctx, struct token* token)
{
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        if (ctx->current->type == TK_PRAGMA)
        {
            ctx->current = ctx->current->next;
            pragma_skip_blanks(ctx);

            if (ctx->current &&
                (strcmp(ctx->current->lexeme, "CAKE") == 0 ||
                    strcmp(ctx->current->lexeme, "cake") == 0))
            {
                ctx->current = ctx->current->next;
                pragma_skip_blanks(ctx);
            }

            if (ctx->current && strcmp(ctx->current->lexeme, "nullchecks") == 0)
            {
                ctx->current = ctx->current->next;
                pragma_skip_blanks(ctx);

                // This is not working because this information needs to be in the AST. 
                // because it is used in a second step.
                bool onoff = false;
                if (ctx->current && strcmp(ctx->current->lexeme, "ON") == 0)
                {
                    onoff = true;
                }
                else if (ctx->current && strcmp(ctx->current->lexeme, "OFF") == 0)
                {
                    onoff = false;
                }
                else
                {
                    compiler_diagnostic(C_ERROR_PRAGMA_ERROR, ctx, ctx->current, NULL, "nullchecks pragma needs to use ON OFF");
                }
                ctx->options.null_checks_enabled = onoff;
            }

            if (ctx->current && strcmp(ctx->current->lexeme, "diagnostic") == 0)
            {
                ctx->current = ctx->current->next;
                pragma_skip_blanks(ctx);

                if (ctx->current && strcmp(ctx->current->lexeme, "push") == 0)
                {
                    // #pragma GCC diagnostic push
                    if (ctx->options.diagnostic_stack.top_index <
                        sizeof(ctx->options.diagnostic_stack) / sizeof(ctx->options.diagnostic_stack.stack[0]))
                    {
                        ctx->options.diagnostic_stack.top_index++;
                        ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index] =
                            ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index - 1];
                    }
                    ctx->current = ctx->current->next;
                    pragma_skip_blanks(ctx);
                }
                else if (ctx->current && strcmp(ctx->current->lexeme, "pop") == 0)
                {
                    // #pragma CAKE diagnostic pop
                    if (ctx->options.diagnostic_stack.top_index > 0)
                    {
                        ctx->options.diagnostic_stack.top_index--;
                    }
                    ctx->current = ctx->current->next;
                    pragma_skip_blanks(ctx);
                }
                else if (ctx->current &&
                    (strcmp(ctx->current->lexeme, "error") == 0 ||
                        strcmp(ctx->current->lexeme, "warning") == 0 ||
                        strcmp(ctx->current->lexeme, "note") == 0 ||
                        strcmp(ctx->current->lexeme, "ignored") == 0))
                {
                    const bool is_error = strcmp(ctx->current->lexeme, "error") == 0;
                    const bool is_warning = strcmp(ctx->current->lexeme, "warning") == 0;
                    const bool is_note = strcmp(ctx->current->lexeme, "note") == 0;

                    ctx->current = ctx->current->next;
                    pragma_skip_blanks(ctx);

                    if (ctx->current && ctx->current->type == TK_STRING_LITERAL)
                    {
                        unsigned long long w = get_warning_bit_mask(ctx->current->lexeme + 1 /*+ 2*/);

                        ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].errors &= ~w;
                        ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].notes &= ~w;
                        ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings &= ~w;

                        if (is_error)
                            ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].errors |= w;
                        else if (is_warning)
                            ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].warnings |= w;
                        else if (is_note)
                            ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index].notes |= w;
                    }
                }
                else if (ctx->current &&
                    (strcmp(ctx->current->lexeme, "check") == 0))
                {
                    // TODO better name .  Ack. : means ‘alarm acknowledged’ ?
                    ctx->current = ctx->current->next;
                    pragma_skip_blanks(ctx);

                    if (ctx->current && ctx->current->type == TK_STRING_LITERAL)
                    {
                        enum diagnostic_id id = get_warning(ctx->current->lexeme + 1 + 2);
                        bool found = false;
                        for (int i = 0;
                             i < (int)(sizeof(ctx->p_report->last_diagnostics_ids) / sizeof(ctx->p_report->last_diagnostics_ids[0]));
                             i++)
                        {
                            if (ctx->p_report->last_diagnostics_ids[i] == 0) break;

                            if (ctx->p_report->last_diagnostics_ids[i] == id)
                            {
                                found = true;
                                // lets remove this error/warning/info from the final report.

                                int t =
                                    get_diagnostic_type(&ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index],
                                        id);
                                if (t == 3)
                                    ctx->p_report->error_count--;
                                else if (t == 2)
                                    ctx->p_report->warnings_count--;
                                else if (t == 1)
                                    ctx->p_report->info_count--;

                                break;
                            }
                        }

                        if (!found)
                        {
                            compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "pragma check failed");
                        }
                    }
                }
                else
                {
                    compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "unknown pragma");
                }
            }
        }
    }
    catch
    {
    }
}

static void parser_skip_blanks(struct parser_ctx* ctx)
{
    while (ctx->current && !(ctx->current->flags & TK_FLAG_FINAL))
    {
        if (ctx->current->type == TK_PRAGMA)
        {
            /*only active block have TK_PRAGMA*/
            parse_pragma(ctx, ctx->current);
        }

        if (ctx->current)
            ctx->current = ctx->current->next;
    }

    if (ctx->current)
    {
        token_promote(ctx, ctx->current); // transform to parser token
    }
}

void parser_match(struct parser_ctx* ctx)
{
    if (ctx->current == NULL)
        return;

    ctx->previous = ctx->current;
    ctx->current = ctx->current->next;
    parser_skip_blanks(ctx);
}

void unexpected_end_of_file(struct parser_ctx* ctx)
{
    compiler_diagnostic(C_ERROR_UNEXPECTED_TOKEN, ctx, ctx->input_list.tail, NULL, "unexpected end of file");
}

NODISCARD
int parser_match_tk(struct parser_ctx* ctx, enum token_type type)
{
    int error = 0;
    if (ctx->current != NULL)
    {
        if (ctx->current->type != type)
        {
            compiler_diagnostic(C_ERROR_UNEXPECTED_TOKEN, ctx, ctx->current, NULL, "expected %s", get_token_name(type));
            error = 1;
        }

        ctx->previous = ctx->current;
        ctx->current = ctx->current->next;
        parser_skip_blanks(ctx);
    }
    else
    {
        compiler_diagnostic(C_ERROR_UNEXPECTED_TOKEN, ctx, ctx->input_list.tail, NULL, "unexpected end of file after");
        error = 1;
    }

    return error;
}

void print_declaration_specifiers(struct osstream* ss, struct declaration_specifiers* p_declaration_specifiers)
{
    bool first = true;
    print_type_qualifier_flags(ss, &first, p_declaration_specifiers->type_qualifier_flags);

    if (p_declaration_specifiers->enum_specifier)
    {

        if (p_declaration_specifiers->enum_specifier->tag_token)
        {
            ss_fprintf(ss, "enum %s", p_declaration_specifiers->enum_specifier->tag_token->lexeme);
        }
        else
        {
            assert(false);
        }
    }
    else if (p_declaration_specifiers->struct_or_union_specifier)
    {
        ss_fprintf(ss, "struct %s", p_declaration_specifiers->struct_or_union_specifier->tag_name);
    }
    else if (p_declaration_specifiers->typedef_declarator)
    {
        if (p_declaration_specifiers->typedef_declarator->name_opt)
            print_item(ss, &first, p_declaration_specifiers->typedef_declarator->name_opt->lexeme);
    }
    else
    {
        print_type_specifier_flags(ss, &first, p_declaration_specifiers->type_specifier_flags);
    }
}

bool type_specifier_is_integer(enum type_specifier_flags flags)
{
    if ((flags & TYPE_SPECIFIER_CHAR) ||
        (flags & TYPE_SPECIFIER_SHORT) ||
        (flags & TYPE_SPECIFIER_INT) ||
        (flags & TYPE_SPECIFIER_LONG) ||
        (flags & TYPE_SPECIFIER_INT) ||
        (flags & TYPE_SPECIFIER_INT8) ||
        (flags & TYPE_SPECIFIER_INT16) ||
        (flags & TYPE_SPECIFIER_INT32) ||
        (flags & TYPE_SPECIFIER_INT64) ||
        (flags & TYPE_SPECIFIER_LONG_LONG))
    {
        return true;
    }
    return false;
}

int final_specifier(struct parser_ctx* ctx, enum type_specifier_flags* flags)
{
    if (((*flags) & TYPE_SPECIFIER_UNSIGNED) ||
        ((*flags) & TYPE_SPECIFIER_SIGNED))
    {
        //TODO leave as it is..and check at comparison 
        if (!type_specifier_is_integer(*flags))
        {
            // if you didn't specify anything, it becomes integer
            (*flags) |= TYPE_SPECIFIER_INT;
        }
    }

    return 0;
}

int add_specifier(struct parser_ctx* ctx,
    enum type_specifier_flags* flags,
    enum type_specifier_flags new_flag)
{
    /*
        transform the sequence of two longs
        in
        TYPE_SPECIFIER_LONG_LONG
    */
    if (new_flag & TYPE_SPECIFIER_LONG) // adding a long
    {
        if ((*flags) & TYPE_SPECIFIER_LONG_LONG) // ja tinha long long
        {
            compiler_diagnostic(C_ERROR_CANNOT_COMBINE_WITH_PREVIOUS_LONG_LONG, ctx, ctx->current, NULL, "cannot combine with previous 'long long' declaration specifier");
            return 1;
        }
        else if ((*flags) & TYPE_SPECIFIER_LONG) // ja tinha um long
        {
            (*flags) = (*flags) & ~TYPE_SPECIFIER_LONG;
            (*flags) |= TYPE_SPECIFIER_LONG_LONG;
        }
        else // nao tinha nenhum long
        {
            (*flags) = (*flags) & ~TYPE_SPECIFIER_INT;
            (*flags) |= TYPE_SPECIFIER_LONG;
        }
    }
    else
    {
        (*flags) |= new_flag;
    }

    //Following 6.7.2 we check possible combinations
    switch ((unsigned int)*flags)
    {
    case TYPE_SPECIFIER_NONE:  //void
    case TYPE_SPECIFIER_VOID:  //void
    case TYPE_SPECIFIER_CHAR:  //char
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_CHAR:  //signed char
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_CHAR:  //unsigned char
    case TYPE_SPECIFIER_SHORT:  //short
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_SHORT:  //signed short
    case TYPE_SPECIFIER_SHORT | TYPE_SPECIFIER_INT:  //short int
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_SHORT | TYPE_SPECIFIER_INT:  //signed short int
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_SHORT:  //unsigned short 
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_SHORT | TYPE_SPECIFIER_INT:  //unsigned short int
    case TYPE_SPECIFIER_INT:  //int
    case TYPE_SPECIFIER_SIGNED:  //signed
    case TYPE_SPECIFIER_INT | TYPE_SPECIFIER_SIGNED:  //int signed
    case TYPE_SPECIFIER_UNSIGNED:  //signed
    case TYPE_SPECIFIER_INT | TYPE_SPECIFIER_UNSIGNED:  //int unsigned
    case TYPE_SPECIFIER_LONG:  //long
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_LONG:  //signed long
    case TYPE_SPECIFIER_LONG | TYPE_SPECIFIER_INT:  //long int
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_LONG | TYPE_SPECIFIER_INT:  //signed long int
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_LONG:  //unsigned long
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_LONG | TYPE_SPECIFIER_INT:  //unsigned long int
    case TYPE_SPECIFIER_LONG_LONG:  //long long
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_LONG_LONG:  //signed long long
    case TYPE_SPECIFIER_LONG_LONG | TYPE_SPECIFIER_INT:  //long long int
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_LONG_LONG | TYPE_SPECIFIER_INT:  //signed long long
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_LONG_LONG:  //unsigned long long
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_LONG_LONG | TYPE_SPECIFIER_INT:  //unsigned long long int
        // _BitInt constant-expression, or signed _BitInt constant-expression        
        // unsigned _BitInt constant-expression
    case TYPE_SPECIFIER_FLOAT:  //float
    case TYPE_SPECIFIER_DOUBLE:  //double
    case TYPE_SPECIFIER_LONG | TYPE_SPECIFIER_DOUBLE:  //long double
    case TYPE_SPECIFIER_DECIMAL32:  //_Decimal32
    case TYPE_SPECIFIER_DECIMAL64:  //_Decimal64
    case TYPE_SPECIFIER_DECIMAL128:  //_Decimal128
    case TYPE_SPECIFIER_BOOL:  //bool
    case TYPE_SPECIFIER_COMPLEX | TYPE_SPECIFIER_FLOAT:  //complex float
    case TYPE_SPECIFIER_COMPLEX | TYPE_SPECIFIER_DOUBLE:  //complex double
    case TYPE_SPECIFIER_LONG | TYPE_SPECIFIER_COMPLEX | TYPE_SPECIFIER_DOUBLE:  //complex long double        
    case TYPE_SPECIFIER_ATOMIC:  //complex long double
    case TYPE_SPECIFIER_STRUCT_OR_UNION:  //complex long double
    case TYPE_SPECIFIER_ENUM:  //complex long double
    case TYPE_SPECIFIER_TYPEOF:  //typeof        
    case TYPE_SPECIFIER_TYPEDEF:

    case TYPE_SPECIFIER_INT8:
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT8:

    case TYPE_SPECIFIER_INT16:
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT16:
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_INT16:
    case TYPE_SPECIFIER_INT32:
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT32:
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_INT32:
    case TYPE_SPECIFIER_INT64:
    case TYPE_SPECIFIER_UNSIGNED | TYPE_SPECIFIER_INT64:
    case TYPE_SPECIFIER_SIGNED | TYPE_SPECIFIER_INT64:
        //VALID
        break;
    default:
        compiler_diagnostic(C_ERROR_TWO_OR_MORE_SPECIFIERS, ctx, ctx->current, NULL, "incompatible specifiers");
        return 1;
    }

    return 0;
}

void declaration_specifiers_delete(struct declaration_specifiers* _Owner _Opt p)
{
    if (p)
    {
        attribute_specifier_sequence_delete(p->p_attribute_specifier_sequence_opt);

        struct declaration_specifier* _Owner _Opt item = p->head;
        while (item)
        {
            struct declaration_specifier* _Owner _Opt next = item->next;
            item->next = NULL;
            declaration_specifier_delete(item);
            item = next;
        }
        free(p);
    }
}

void declaration_specifiers_add(struct declaration_specifiers* list, struct declaration_specifier* _Owner p_item)
{
    if (list->head == NULL)
    {
        list->head = p_item;
    }
    else
    {
        assert(list->tail != NULL);
        assert(list->tail->next == NULL);
        list->tail->next = p_item;
    }
    list->tail = p_item;
}

struct declaration_specifiers* _Owner _Opt declaration_specifiers(struct parser_ctx* ctx,
    enum storage_class_specifier_flags default_storage_flag)
{
    /*
        declaration-specifiers:
          declaration-specifier attribute-specifier-sequence_opt
          declaration-specifier declaration-specifiers
    */

    /*
     Ao fazer parser do segundo o X ja existe mas ele nao deve ser usado
     typedef char X;
     typedef char X;
    */

    if (ctx->current == NULL)
        return NULL;

    _Opt struct declaration_specifiers* _Owner _Opt p_declaration_specifiers = calloc(1, sizeof(struct declaration_specifiers));

    try
    {
        if (p_declaration_specifiers == NULL)
            throw;

        p_declaration_specifiers->first_token = ctx->current;

        while (first_of_declaration_specifier(ctx))
        {
            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            if (ctx->current->flags & TK_FLAG_IDENTIFIER_IS_TYPEDEF)
            {
                if (p_declaration_specifiers->type_specifier_flags != TYPE_SPECIFIER_NONE)
                {
                    // typedef tem que aparecer sozinho
                    // exemplo Socket eh nome e nao typdef
                    // typedef int Socket;
                    // struct X {int Socket;};
                    break;
                }
            }

            struct declaration_specifier* _Owner _Opt p_declaration_specifier = declaration_specifier(ctx);
            if (p_declaration_specifier == NULL) throw;

            if (p_declaration_specifier->type_specifier_qualifier)
            {
                if (p_declaration_specifier->type_specifier_qualifier->type_specifier)
                {
                    if (add_specifier(ctx,
                        &p_declaration_specifiers->type_specifier_flags,
                        p_declaration_specifier->type_specifier_qualifier->type_specifier->flags) != 0)
                    {
                        //not fatal error
                    }

                    if (p_declaration_specifier->type_specifier_qualifier->type_specifier->struct_or_union_specifier)
                    {
                        p_declaration_specifiers->struct_or_union_specifier = p_declaration_specifier->type_specifier_qualifier->type_specifier->struct_or_union_specifier;
                    }
                    else if (p_declaration_specifier->type_specifier_qualifier->type_specifier->enum_specifier)
                    {
                        p_declaration_specifiers->enum_specifier = p_declaration_specifier->type_specifier_qualifier->type_specifier->enum_specifier;
                    }
                    else if (p_declaration_specifier->type_specifier_qualifier->type_specifier->typeof_specifier)
                    {
                        p_declaration_specifiers->typeof_specifier = p_declaration_specifier->type_specifier_qualifier->type_specifier->typeof_specifier;
                    }

                    else if (p_declaration_specifier->type_specifier_qualifier->type_specifier->msvc_declspec)
                    {
                        p_declaration_specifiers->msvc_declspec_flags =
                            p_declaration_specifier->type_specifier_qualifier->type_specifier->msvc_declspec->flags;

                    }
                    else if (p_declaration_specifier->type_specifier_qualifier->type_specifier->token->type == TK_IDENTIFIER)
                    {
                        p_declaration_specifiers->typedef_declarator =
                            find_declarator(ctx,
                                p_declaration_specifier->type_specifier_qualifier->type_specifier->token->lexeme,
                                NULL);

                        // p_declaration_specifiers->typedef_declarator = p_declaration_specifier->type_specifier_qualifier->pType_specifier->token->lexeme;
                    }
                }
                else if (p_declaration_specifier->type_specifier_qualifier->alignment_specifier)
                {
                    p_declaration_specifiers->alignment_specifier_flags =
                        p_declaration_specifier->type_specifier_qualifier->alignment_specifier->flags;

                }
                else if (p_declaration_specifier->type_specifier_qualifier->type_qualifier)
                {
                    p_declaration_specifiers->type_qualifier_flags |= p_declaration_specifier->type_specifier_qualifier->type_qualifier->flags;
                }
            }
            else if (p_declaration_specifier->storage_class_specifier)
            {
                p_declaration_specifiers->storage_class_specifier_flags |= p_declaration_specifier->storage_class_specifier->flags;
            }
            else if (p_declaration_specifier->function_specifier)
            {
                p_declaration_specifiers->function_specifier_flags |= p_declaration_specifier->function_specifier->flags;
            }
            else if (p_declaration_specifier->alignment_specifier)
            {
                p_declaration_specifiers->alignment_specifier_flags |= p_declaration_specifier->alignment_specifier->flags;
            }

            declaration_specifiers_add(p_declaration_specifiers, p_declaration_specifier);

            assert(p_declaration_specifiers->p_attribute_specifier_sequence_opt == NULL);
            p_declaration_specifiers->p_attribute_specifier_sequence_opt = attribute_specifier_sequence_opt(ctx);

            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            if (ctx->current->type == TK_IDENTIFIER &&
                p_declaration_specifiers->type_specifier_flags != TYPE_SPECIFIER_NONE)
            {
                // typedef nao pode aparecer com outro especifier
                // entao ja tem tem algo e vier identifier signfica que acabou
                // exemplo
                /*
                 typedef char X;
                 typedef char X;
                */
                break;
            }
        }

        struct token* _Opt prev = previous_parser_token(ctx->current);
        if (prev == NULL)
            throw;

        p_declaration_specifiers->last_token = prev;

        // int main() { static int i; } // i is not automatic
        final_specifier(ctx, &p_declaration_specifiers->type_specifier_flags);

        p_declaration_specifiers->storage_class_specifier_flags |= default_storage_flag;

        if (p_declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_STATIC)
        {
            //
            p_declaration_specifiers->storage_class_specifier_flags &= ~STORAGE_SPECIFIER_AUTOMATIC_STORAGE;
        }
    }
    catch
    {
        declaration_specifiers_delete(p_declaration_specifiers);
        p_declaration_specifiers = NULL;
    }
    return p_declaration_specifiers;
}

struct declaration* _Owner _Opt declaration_core(struct parser_ctx* ctx,
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt /*SINK*/,
    bool can_be_function_definition,
    bool* is_function_definition,
    enum storage_class_specifier_flags default_storage_class_specifier_flags,
    bool without_semicolon)
{
    /*
                                  declaration-specifiers init-declarator-list_opt ;
     attribute-specifier-sequence declaration-specifiers init-declarator-list ;
     static_assert-declaration
     attribute-declaration
  */

    struct declaration* _Owner _Opt p_declaration = NULL;

    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        p_declaration = calloc(1, sizeof(struct declaration));
        if (p_declaration == NULL)
        {
            throw;
        }

        p_declaration->p_attribute_specifier_sequence_opt = p_attribute_specifier_sequence_opt;
        p_attribute_specifier_sequence_opt = NULL; /*MOVED*/

        p_declaration->first_token = ctx->current;

        if (ctx->current->type == ';')
        {
            p_declaration->last_token = ctx->current;
            parser_match(ctx);
            // empty declaration
            return p_declaration;
        }

        if (first_of_static_assert_declaration(ctx))
        {
            p_declaration->static_assert_declaration = static_assert_declaration(ctx);
        }
        else if (first_of_pragma_declaration(ctx))
        {
            p_declaration->pragma_declaration = pragma_declaration(ctx);
        }
        else
        {

            if (first_of_declaration_specifier(ctx))
            {
                p_declaration->declaration_specifiers = declaration_specifiers(ctx, default_storage_class_specifier_flags);
                if (p_declaration->declaration_specifiers == NULL) throw;

                if (p_declaration->p_attribute_specifier_sequence_opt)
                {
                    p_declaration->declaration_specifiers->attributes_flags =
                        p_declaration->p_attribute_specifier_sequence_opt->attributes_flags;
                }

                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    throw;
                }

                if (ctx->current->type != ';')
                {
                    p_declaration->init_declarator_list = init_declarator_list(ctx,
                        p_declaration->declaration_specifiers);

                    if (p_declaration->init_declarator_list.head == NULL)
                        throw;
                }

                if (ctx->current == NULL)
                {
                    unexpected_end_of_file(ctx);
                    throw;
                }

                p_declaration->last_token = ctx->current;

                if (ctx->current->type == '{')
                {
                    if (can_be_function_definition)
                        *is_function_definition = true;
                }
#if EXPERIMENTAL_CONTRACTS
                else if (ctx->current->type == TK_KEYWORD_TRUE ||
                         ctx->current->type == TK_KEYWORD_FALSE ||
                         ctx->current->type == TK_IDENTIFIER)
                {
                    if (can_be_function_definition)
                        *is_function_definition = true;
                }
#endif
                else
                {
                    if (!without_semicolon && parser_match_tk(ctx, ';') != 0)
                        throw;
                }
            }
            else
            {
                if (ctx->current->type == TK_IDENTIFIER)
                {
                    compiler_diagnostic(C_ERROR_INVALID_TYPE, ctx, ctx->current, NULL, "invalid type '%s'", ctx->current->lexeme);
                }
                else
                {
                    compiler_diagnostic(C_ERROR_EXPECTED_DECLARATION, ctx, ctx->current, NULL, "expected declaration not '%s'", ctx->current->lexeme);
                }
                parser_match(ctx); // we need to go ahead
            }
        }
    }
    catch
    {
        declaration_delete(p_declaration);
        p_declaration = NULL;
    }

    attribute_specifier_sequence_delete(p_attribute_specifier_sequence_opt);

    return p_declaration;
}

struct declaration* _Owner _Opt declaration(struct parser_ctx* ctx,
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt,
    enum storage_class_specifier_flags storage_specifier_flags,
    bool extern_declaration);

struct declaration* _Owner _Opt function_definition_or_declaration(struct parser_ctx* ctx)
{
    return declaration(ctx, NULL, STORAGE_SPECIFIER_NONE, true);
    /*
     function-definition:
        attribute-specifier-sequence _Opt declaration-specifiers declarator function-body
    */

    /*
      declaration:
        declaration-specifiers                              init-declarator-list opt ;
        attribute-specifier-sequence declaration-specifiers init-declarator-list ;
        static_assert-declaration
        attribute-declaration
    */

}

struct simple_declaration* _Owner _Opt simple_declaration(struct parser_ctx* ctx,
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt,
    bool ignore_semicolon)
{

    if (ctx->current == NULL)
    {
        unexpected_end_of_file(ctx);
        attribute_specifier_sequence_delete(p_attribute_specifier_sequence_opt);
        return NULL;
    }

    enum storage_class_specifier_flags storage_specifier_flags = STORAGE_SPECIFIER_AUTOMATIC_STORAGE;
    /*
      simple-declaration:
      declaration-specifiers init-declarator-list _Opt ;
      attribute-specifier-sequence declaration-specifiers init-declarator-list ;
    */

    struct simple_declaration* _Owner _Opt p_simple_declaration = calloc(1, sizeof(struct simple_declaration));

    try
    {
        if (p_simple_declaration == NULL)
            throw;

        p_simple_declaration->first_token = ctx->current;

        p_simple_declaration->p_attribute_specifier_sequence_opt = p_attribute_specifier_sequence_opt;
        p_attribute_specifier_sequence_opt = NULL; /*MOVED*/

        struct declaration_specifiers* _Owner _Opt ptemp = declaration_specifiers(ctx, storage_specifier_flags);
        if (ptemp == NULL) throw;

        p_simple_declaration->p_declaration_specifiers = ptemp;

        if (p_simple_declaration->p_attribute_specifier_sequence_opt)
        {
            p_simple_declaration->p_declaration_specifiers->attributes_flags =
                p_simple_declaration->p_attribute_specifier_sequence_opt->attributes_flags;
        }

        p_simple_declaration->init_declarator_list = init_declarator_list(ctx, p_simple_declaration->p_declaration_specifiers);

        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        struct token* _Opt prev = previous_parser_token(ctx->current);
        if (prev == NULL) throw;

        p_simple_declaration->last_token = prev;

        if (!ignore_semicolon && parser_match_tk(ctx, ';') != 0) throw;
    }
    catch
    {
        simple_declaration_delete(p_simple_declaration);
        p_simple_declaration = NULL;
    }

    attribute_specifier_sequence_delete(p_attribute_specifier_sequence_opt);

    return p_simple_declaration;
}

static void check_unused_parameters(struct parser_ctx* ctx, struct parameter_list* parameter_list)
{
    struct parameter_declaration* _Opt parameter = NULL;
    parameter = parameter_list->head;

    while (parameter)
    {
        if (!type_is_maybe_unused(&parameter->declarator->type) &&
            parameter->declarator &&
            parameter->declarator->num_uses == 0)
        {
            if (parameter->declarator->name_opt &&
                parameter->declarator->name_opt->level == 0 /*direct source*/
                )
            {
                compiler_diagnostic(W_UNUSED_PARAMETER,
                    ctx,
                    parameter->declarator->name_opt, NULL,
                    "'%s': unreferenced formal parameter",
                    parameter->declarator->name_opt->lexeme);
            }
        }
        parameter = parameter->next;
    }
}

struct declaration* _Owner _Opt declaration(struct parser_ctx* ctx,
    struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt00,
    enum storage_class_specifier_flags storage_specifier_flags,
    bool extern_declaration)
{
    struct declaration* _Owner _Opt p_declaration = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        struct attribute_specifier_sequence* _Owner _Opt p_attribute_specifier_sequence_opt =
            attribute_specifier_sequence_opt(ctx);

        bool is_function_definition = false;

        p_declaration = declaration_core(ctx, p_attribute_specifier_sequence_opt, true, &is_function_definition, storage_specifier_flags, false);
        if (p_declaration == NULL)
            throw;

        if (is_function_definition)
        {

            if (p_declaration->init_declarator_list.head == NULL ||
                p_declaration->init_declarator_list.head->p_declarator->direct_declarator == NULL ||
                p_declaration->init_declarator_list.head->p_declarator->direct_declarator->function_declarator == NULL)
            {
                compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "unexpected");
                throw; //unexpected
            }

            struct declarator* p_declarator =
                p_declaration->init_declarator_list.head->p_declarator;


            //ctx->p_current_function_opt = p_declaration->init_declarator_list.head->p_declarator;

            /*
                scope of parameters is the inner declarator

                void (*f(int i))(void) {
                    i = 1;
                    return 0;
                }
            */

            assert(p_declaration->init_declarator_list.head != NULL); //because functions definitions have names

            struct declarator* inner = p_declaration->init_declarator_list.head->p_declarator;
            for (;;)
            {
                if (inner->direct_declarator &&
                    inner->direct_declarator->function_declarator &&
                    inner->direct_declarator->function_declarator->direct_declarator &&
                    inner->direct_declarator->function_declarator->direct_declarator->declarator)
                {
                    inner = inner->direct_declarator->function_declarator->direct_declarator->declarator;
                }
                else
                    break;
            }


            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;
            }

            check_func_open_brace_style(ctx, ctx->current);

            if (ctx->current == NULL)
            {
                unexpected_end_of_file(ctx);
                throw;

            }
            struct diagnostic before_function_diagnostics = ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index];
#if EXPERIMENTAL_CONTRACTS


            if (ctx->current->type == TK_KEYWORD_TRUE ||
                ctx->current->type == TK_KEYWORD_FALSE ||
                ctx->current->type == TK_IDENTIFIER)
            {
                for (;;)
                {
                    if (ctx->current == NULL)
                    {
                        unexpected_end_of_file(ctx);
                        throw;
                    }

                    enum token_type type = ctx->current->type;
                    if (type != TK_KEYWORD_TRUE &&
                        type != TK_KEYWORD_FALSE &&
                        type != TK_IDENTIFIER)
                    {
                        throw;
                    }
                    parser_match(ctx); //true
                    parser_match(ctx); //(

                    if (type != TK_KEYWORD_FALSE)
                    {
                        assert(p_declarator->p_expression_true == NULL);
                        p_declarator->p_expression_true = expression(ctx);
                    }
                    else
                    {
                        assert(p_declarator->p_expression_false == NULL);
                        p_declarator->p_expression_false = expression(ctx);
                    }
                    parser_match(ctx); //)

                    if (ctx->current == NULL)
                    {
                        unexpected_end_of_file(ctx);
                        throw;
                    }

                    if (ctx->current->type != ',')
                        break;

                    parser_match(ctx); //)
                }
            }
#endif
            struct declarator* _Opt p_current_function_opt = ctx->p_current_function_opt;
            ctx->p_current_function_opt = p_declarator;


            struct scope* parameters_scope = &inner->direct_declarator->function_declarator->parameters_scope;
            scope_list_push(&ctx->scopes, parameters_scope);

            struct scope* p_current_function_scope_opt = ctx->p_current_function_scope_opt;
            ctx->p_current_function_scope_opt = ctx->scopes.tail;

            struct compound_statement* _Owner _Opt p_function_body = function_body(ctx);

            ctx->p_current_function_scope_opt = p_current_function_scope_opt; //restore
            ctx->p_current_function_opt = p_current_function_opt; //restore
            scope_list_pop(&ctx->scopes);

            if (p_function_body == NULL)
                throw;

            assert(p_declaration->function_body == NULL);
            p_declaration->function_body = p_function_body;
            p_declaration->init_declarator_list.head->p_declarator->function_body = p_declaration->function_body;

            if (p_declaration->init_declarator_list.head &&
                p_declaration->init_declarator_list.head->p_declarator->direct_declarator &&
                p_declaration->init_declarator_list.head->p_declarator->direct_declarator->function_declarator &&
                p_declaration->init_declarator_list.head->p_declarator->direct_declarator->function_declarator->parameter_type_list_opt &&
                p_declaration->init_declarator_list.head->p_declarator->direct_declarator->function_declarator->parameter_type_list_opt->parameter_list)
            {
                check_unused_parameters(ctx, p_declaration->init_declarator_list.head->p_declarator->direct_declarator->function_declarator->parameter_type_list_opt->parameter_list);
            }


            if (extern_declaration)
            {
                struct defer_visit_ctx ctx2 = { .ctx = ctx };
                defer_start_visit_declaration(&ctx2, p_declaration);
                defer_visit_ctx_destroy(&ctx2);

                if (ctx->options.flow_analysis)
                {
                    /*
                     Now we have the full function AST let´s visit to Analise
                     jumps
                    */

                    /*we are going to visit the function again.. lets put the same diagnostic state*/
                    ctx->options.diagnostic_stack.stack[ctx->options.diagnostic_stack.top_index] = before_function_diagnostics;

                    struct flow_visit_ctx ctx3 = { 0 };
                    ctx3.ctx = ctx;
                    flow_start_visit_declaration(&ctx3, p_declaration);
                    flow_visit_ctx_destroy(&ctx3);
                }
            }

        }
        else
        {
            if (ctx->options.flow_analysis && extern_declaration)
            {
                _Opt struct flow_visit_ctx ctx2 = { 0 };
                ctx2.ctx = ctx;
                flow_start_visit_declaration(&ctx2, p_declaration);
                flow_visit_ctx_destroy(&ctx2);
            }
        }
    }
    catch
    {
        declaration_delete(p_declaration);
        p_declaration = NULL;
    }

    return p_declaration;

    //bool is_function_definition = false;
    //return declaration_core(ctx, p_attribute_specifier_sequence_opt, false, &is_function_definition, storage_specifier_flags, false);
}

//(6.7) declaration-specifiers:
// declaration-specifier attribute-specifier-sequenceopt
// declaration-specifier declaration-specifiers

void declaration_specifier_delete(struct declaration_specifier* _Owner _Opt p)
{
    if (p)
    {
        free(p->function_specifier);
        type_specifier_qualifier_delete(p->type_specifier_qualifier);
        free(p->storage_class_specifier);
        assert(p->next == NULL);
        free(p);
    }
}

struct declaration_specifier* _Owner _Opt declaration_specifier(struct parser_ctx* ctx)
{
    //    storage-class-specifier
    //    type-specifier-qualifier
    //    function-specifier

    struct declaration_specifier* _Owner _Opt p_declaration_specifier = NULL;
    try
    {
        p_declaration_specifier = calloc(1, sizeof * p_declaration_specifier);
        if (p_declaration_specifier == NULL)
            throw;

        if (first_of_storage_class_specifier(ctx))
        {
            p_declaration_specifier->storage_class_specifier = storage_class_specifier(ctx);
            if (p_declaration_specifier->storage_class_specifier == NULL) throw;
        }
        else if (first_of_type_specifier_qualifier(ctx))
        {
            p_declaration_specifier->type_specifier_qualifier = type_specifier_qualifier(ctx);
            if (p_declaration_specifier->type_specifier_qualifier == NULL) throw;
        }
        else if (first_of_function_specifier(ctx))
        {
            p_declaration_specifier->function_specifier = function_specifier(ctx);
            if (p_declaration_specifier->function_specifier == NULL) throw;
        }
        else
        {
            compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "unexpected");
        }
    }
    catch
    {
        declaration_specifier_delete(p_declaration_specifier);
        p_declaration_specifier = NULL;
    }

    return p_declaration_specifier;
}

struct init_declarator* _Owner init_declarator_add_ref(struct init_declarator* p)
{
    p->has_shared_ownership = true;
    return (struct init_declarator* _Owner)p;
}


void init_declarator_sink(struct init_declarator* _Owner _Opt p) {}

void init_declarator_delete(struct init_declarator* _Owner _Opt p)
{
    if (p)
    {
        if (p->has_shared_ownership)
        {
            p->has_shared_ownership = false;
            init_declarator_sink(p);
            return;
        }

        initializer_delete(p->initializer);
        declarator_delete(p->p_declarator);
        assert(p->next == NULL);
        free(p);
    }
}


struct init_declarator* _Owner _Opt init_declarator(struct parser_ctx* ctx,
    struct declaration_specifiers* p_declaration_specifiers)
{
    /*
     init-declarator:
       declarator
       declarator = initializer
    */

    struct init_declarator* _Owner _Opt p_init_declarator = NULL;
    try
    {
        p_init_declarator = calloc(1, sizeof(struct init_declarator));
        if (p_init_declarator == NULL)
            throw;

        struct token* _Opt tkname = NULL;

        {
            struct declarator* _Owner _Opt p_temp_declarator = declarator(ctx,
                NULL,
                p_declaration_specifiers,
                false,
                &tkname);
            if (p_temp_declarator == NULL) throw;
            p_init_declarator->p_declarator = p_temp_declarator;
        }

        if (tkname == NULL)
        {
            compiler_diagnostic(C_ERROR_UNEXPECTED, ctx, ctx->current, NULL, "init declarator must have a name");
            throw;
        }

        p_init_declarator->p_declarator->declaration_specifiers = p_declaration_specifiers;
        p_init_declarator->p_declarator->name_opt = tkname;

        if (p_init_declarator->p_declarator->declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_AUTO)
        {
            /*
              auto requires we find the type after initializer
            */
        }
        else
        {
            assert(p_init_declarator->p_declarator->type.type_specifier_flags == 0);
            p_init_declarator->p_declarator->type = make_type_using_declarator(ctx, p_init_declarator->p_declarator);
        }

        assert(p_init_declarator->p_declarator->declaration_specifiers != NULL);

        assert(ctx->scopes.tail != NULL);

        /*
          Checking naming conventions
        */
        if (ctx->scopes.tail->scope_level == 0)
        {
            naming_convention_global_var(ctx,
                tkname,
                &p_init_declarator->p_declarator->type,
                p_init_declarator->p_declarator->declaration_specifiers->storage_class_specifier_flags);
        }

        /////////////////////////////////////////////////////////////////////////////
        const char* name = p_init_declarator->p_declarator->name_opt->lexeme;
        struct scope* _Opt out_scope = NULL;
        struct declarator* _Opt previous = find_declarator(ctx, name, &out_scope);
        if (previous)
        {
            assert(out_scope != NULL);
            assert(ctx->scopes.tail != NULL);

            if (out_scope->scope_level == ctx->scopes.tail->scope_level)
            {
                if (out_scope->scope_level == 0)
                {
                    /*
                    __C_ASSERT__ is failing..maybe because __builtin_offsetof is not implemented
                    */
                    if (strcmp(name, "__C_ASSERT__") != 0)
                    {
                        //TODO type_is_same needs changes see #164
                        if (!type_is_same(&previous->type, &p_init_declarator->p_declarator->type, false))
                        {
                            struct osstream ss = { 0 };
                            print_type_no_names(&ss, &previous->type);

                            compiler_diagnostic(
                                C_ERROR_REDECLARATION,
                                ctx,
                                ctx->current,
                                NULL,
                                "conflicting types for '%s' (%s)", name, ss.c_str);

                            ss_clear(&ss);
                            print_type_no_names(&ss, &p_init_declarator->p_declarator->type);

                            compiler_diagnostic(C_ERROR_REDECLARATION,
                                ctx,
                                previous->name_opt,
                                NULL,
                                "previous declaration (%s)", ss.c_str);
                            ss_close(&ss);
                        }
                    }
                }
                else
                {
                    compiler_diagnostic(C_ERROR_REDECLARATION, ctx, ctx->current, NULL, "redeclaration");
                    compiler_diagnostic(W_NOTE, ctx, previous->name_opt, NULL, "previous declaration");
                }
            }
            else
            {
                struct hash_item_set item = { 0 };
                item.p_init_declarator = init_declarator_add_ref(p_init_declarator);
                hashmap_set(&ctx->scopes.tail->variables, name, &item);
                hash_item_set_destroy(&item);

                /*global scope no warning...*/
                if (out_scope->scope_level != 0)
                {
                    /*but redeclaration at function scope we show warning*/
                    if (compiler_diagnostic(W_DECLARATOR_HIDE, ctx, p_init_declarator->p_declarator->first_token_opt, NULL, "declaration of '%s' hides previous declaration", name))
                    {
                        compiler_diagnostic(W_NOTE, ctx, previous->first_token_opt, NULL, "previous declaration is here");
                    }
                }
            }
        }
        else
        {
            /*first time we see this declarator*/
            struct hash_item_set item = { 0 };
            item.p_init_declarator = init_declarator_add_ref(p_init_declarator);
            hashmap_set(&ctx->scopes.tail->variables, name, &item);
            hash_item_set_destroy(&item);
        }
        /////////////////////////////////////////////////////////////////////////////


        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        if (ctx->current->type == '=')
        {
            const bool requires_constant_initialization =
                (ctx->p_current_function_opt == NULL) ||
                (p_declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_STATIC);

            parser_match(ctx);

            assert(p_init_declarator->initializer == NULL);
            p_init_declarator->initializer = initializer(ctx);

            if (p_init_declarator->initializer == NULL)
            {
                throw;
            }

            if (p_init_declarator->initializer->braced_initializer)
            {
                if (p_init_declarator->p_declarator->declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_AUTO)
                {
                    compiler_diagnostic(C_ERROR_AUTO_NEEDS_SINGLE_DECLARATOR, ctx, p_init_declarator->p_declarator->first_token_opt, NULL, "'auto' requires a plain identifier");
                    throw;
                }


                int er = make_object(&p_init_declarator->p_declarator->type, &p_init_declarator->p_declarator->object);
                if (er != 0)
                {
                    compiler_diagnostic(C_ERROR_STRUCT_IS_INCOMPLETE, ctx, p_init_declarator->p_declarator->first_token_opt, NULL, "incomplete struct/union type");
                    throw;
                }


                
                const bool is_constant = 
                    type_is_const_or_constexpr(&p_init_declarator->p_declarator->type);

                if (initializer_init_new(ctx,
                    &p_init_declarator->p_declarator->type,
                    &p_init_declarator->p_declarator->object,
                    p_init_declarator->initializer,
                    is_constant,
                    requires_constant_initialization) != 0)
                {
                    throw;
                }

                p_init_declarator->p_declarator->object.type.num_of_elements =
                    p_init_declarator->p_declarator->type.num_of_elements;
                //fixing the name of members?
            }
            else if (p_init_declarator->initializer->assignment_expression)
            {
                if (type_is_array(&p_init_declarator->p_declarator->type))
                {
                    const unsigned long long array_size_elements = p_init_declarator->p_declarator->type.num_of_elements;
                    if (array_size_elements == 0)
                    {
                        p_init_declarator->p_declarator->type.num_of_elements =
                            p_init_declarator->initializer->assignment_expression->type.num_of_elements;
                    }
                    else
                    {
                        if (p_init_declarator->initializer->assignment_expression->type.num_of_elements > array_size_elements)
                        {
                            if (p_init_declarator->p_declarator->first_token_opt)
                            {
                                compiler_diagnostic(W_ARRAY_SIZE, ctx, p_init_declarator->p_declarator->first_token_opt, NULL, "initializer for array is too long");
                            }
                        }
                    }
                }

                /*
                  Fixing the type of auto declarator
                */
                assert(p_init_declarator->p_declarator->declaration_specifiers != NULL);

                if (p_init_declarator->p_declarator->declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_AUTO)
                {

                    if (p_init_declarator->p_declarator->direct_declarator &&
                        (p_init_declarator->p_declarator->direct_declarator->array_declarator != NULL ||
                            p_init_declarator->p_declarator->direct_declarator->function_declarator != NULL))
                    {
                        compiler_diagnostic(C_ERROR_AUTO_NEEDS_SINGLE_DECLARATOR, ctx, p_init_declarator->p_declarator->first_token_opt, NULL, "'auto' requires a plain identifier");
                        throw;
                    }

                    if (p_init_declarator->p_declarator->pointer != NULL)
                    {
                        compiler_diagnostic(C_ERROR_AUTO_NEEDS_SINGLE_DECLARATOR, ctx, p_init_declarator->p_declarator->first_token_opt, NULL, "'auto' requires a plain identifier");
                    }

                    struct type t = { 0 };

                    if (p_init_declarator->initializer->assignment_expression->expression_type == UNARY_EXPRESSION_ADDRESSOF)
                    {
                        t = type_dup(&p_init_declarator->initializer->assignment_expression->type);
                    }
                    else
                    {
                        struct type t2 = type_lvalue_conversion(&p_init_declarator->initializer->assignment_expression->type, ctx->options.null_checks_enabled);
                        type_swap(&t2, &t);
                        type_destroy(&t2);
                    }

                    type_remove_names(&t);
                    assert(t.name_opt == NULL);
                    t.name_opt = strdup(p_init_declarator->p_declarator->name_opt->lexeme);

                    type_set_qualifiers_using_declarator(&t, p_init_declarator->p_declarator);

                    type_visit_to_mark_anonymous(&t);
                    type_swap(&p_init_declarator->p_declarator->type, &t);
                    type_destroy(&t);
                }

                check_assigment(ctx, &p_init_declarator->p_declarator->type, p_init_declarator->initializer->assignment_expression, ASSIGMENT_TYPE_INIT);

                const char* name2 = p_init_declarator->p_declarator->name_opt ?
                    p_init_declarator->p_declarator->name_opt->lexeme : "";

                int er = make_object_with_name(&p_init_declarator->p_declarator->type,
                    &p_init_declarator->p_declarator->object, name2);

                if (er != 0)
                {
                    throw;
                }

                const bool is_constant = 
                    type_is_const_or_constexpr(&p_init_declarator->p_declarator->type);

                
                if (initializer_init_new(ctx,
                    &p_init_declarator->p_declarator->type,
                    &p_init_declarator->p_declarator->object,
                    p_init_declarator->initializer,
                    is_constant,
                    requires_constant_initialization) != 0)
                {
                    throw;
                }
                //object_print_to_debug(&p_init_declarator->p_declarator->object);
            }
        }
        else
        {
            if (p_init_declarator->p_declarator->type.category != TYPE_CATEGORY_FUNCTION &&
                !(p_init_declarator->p_declarator->type.storage_class_specifier_flags & STORAGE_SPECIFIER_TYPEDEF))
            {
                const char* name2 = p_init_declarator->p_declarator->name_opt ?
                    p_init_declarator->p_declarator->name_opt->lexeme : "";

                int er = make_object_with_name(&p_init_declarator->p_declarator->type,
                    &p_init_declarator->p_declarator->object,
                    name2);

                if (er != 0)
                {
                    if (p_init_declarator->p_declarator->declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_EXTERN)
                    {
                        //extern struct X x;
                    }
                    else
                    {
                        compiler_diagnostic(C_ERROR_STRUCT_IS_INCOMPLETE, ctx, p_init_declarator->p_declarator->first_token_opt, NULL, "incomplete struct/union type");
                        throw;
                    }
                }

                if (type_is_const(&p_init_declarator->p_declarator->type))
                {
                    if (p_declaration_specifiers->storage_class_specifier_flags & STORAGE_SPECIFIER_TYPEDEF)
                    {
                        //no warning on typedefs
                    }
                    else
                    {
                        compiler_diagnostic(W_CONST_NOT_INITIALIZED,
                            ctx,
                            p_init_declarator->p_declarator->first_token_opt, NULL,
                            "const object should be initialized");
                    }
                }
            }
        }

        /*
           checking usage of [static ] other than in function arguments
        */
        if (p_init_declarator->p_declarator)
        {
            if (type_is_array(&p_init_declarator->p_declarator->type))
                if (p_init_declarator->p_declarator->type.type_qualifier_flags != 0 ||
                    p_init_declarator->p_declarator->type.has_static_array_size)
                {
                    if (p_init_declarator->p_declarator->first_token_opt)
                    {
                        compiler_diagnostic(C_ERROR_STATIC_OR_TYPE_QUALIFIERS_NOT_ALLOWED_IN_NON_PARAMETER,
                            ctx,
                            p_init_declarator->p_declarator->first_token_opt, NULL,
                            "static or type qualifiers are not allowed in non-parameter array declarator");
                    }
                    else if (p_init_declarator->initializer)
                    {
                        compiler_diagnostic(C_ERROR_STATIC_OR_TYPE_QUALIFIERS_NOT_ALLOWED_IN_NON_PARAMETER,
                        ctx,
                        p_init_declarator->initializer->first_token, NULL,
                        "static or type qualifiers are not allowed in non-parameter array declarator");
                    }
                }

            if (!type_is_pointer(&p_init_declarator->p_declarator->type) &&
                p_init_declarator->p_declarator->type.type_qualifier_flags & TYPE_QUALIFIER_DTOR)
            {
                if (p_init_declarator->p_declarator->first_token_opt)
                {
                    compiler_diagnostic(C_ERROR_OBJ_OWNER_CAN_BE_USED_ONLY_IN_POINTER,
                        ctx,
                        p_init_declarator->p_declarator->first_token_opt, NULL,
                        "_Dtor qualifier can only be used with pointers");
                }
                else if (p_init_declarator->initializer)
                {
                    compiler_diagnostic(C_ERROR_OBJ_OWNER_CAN_BE_USED_ONLY_IN_POINTER,
                    ctx,
                    p_init_declarator->initializer->first_token, NULL,
                    "_Dtor qualifier can only be used with pointers");
                }
            }
        }

        if (
            !(p_init_declarator->p_declarator->type.storage_class_specifier_flags & STORAGE_SPECIFIER_TYPEDEF) &&
            !type_is_function(&p_init_declarator->p_declarator->type))
        {

            if (type_is_vla(&p_init_declarator->p_declarator->type))
            {
            }
            else if (type_is_function(&p_init_declarator->p_declarator->type))
            {
                compiler_diagnostic(C_ERROR_STORAGE_SIZE,
                  ctx,
                  p_init_declarator->p_declarator->name_opt, NULL,
                  "invalid application of 'sizeof' to a function type");
            }
            else
            {
                size_t sz = 0;
                enum sizeof_error size_result = type_get_sizeof(&p_init_declarator->p_declarator->type, &sz);

                if (size_result == ESIZEOF_NONE)
                {
                    //ok
                }
                else if (size_result == ESIZEOF_INCOMPLETE)
                {
                    if (p_init_declarator->p_declarator->type.storage_class_specifier_flags & STORAGE_SPECIFIER_EXTERN)
                    {
                        //this is not a problem for extern variables
                    }
                    else
                    {
                        // clang warning: array 'c' assumed to have one element
                        // gcc "error: storage size of '%s' isn't known"
                        compiler_diagnostic(C_ERROR_STORAGE_SIZE,
                            ctx,
                            p_init_declarator->p_declarator->name_opt, NULL,
                            "storage size of '%s' isn't known",
                            p_init_declarator->p_declarator->name_opt->lexeme);
                    }
                }
                else if (size_result == ESIZEOF_OVERLOW)
                {
                    compiler_diagnostic(C_ERROR_STORAGE_SIZE,
                            ctx,
                            p_init_declarator->p_declarator->name_opt, NULL,
                            "sizeof '%s' is too large",
                            p_init_declarator->p_declarator->name_opt->lexeme);
                }
                else
                {
                    compiler_diagnostic(C_ERROR_STORAGE_SIZE,
                        ctx,
                        p_init_declarator->p_declarator->name_opt, NULL,
                        "storage size of '%s' isn't known",
                        p_init_declarator->p_declarator->name_opt->lexeme);
                }
            }
        }
    }
    catch
    {
        init_declarator_delete(p_init_declarator);
        p_init_declarator = NULL;
    }

    return p_init_declarator;
}

void init_declarator_list_add(struct init_declarator_list* list, struct init_declarator* _Owner p_item)
{
    if (list->head == NULL)
    {
        list->head = p_item;
    }
    else
    {
        assert(list->tail != NULL);
        assert(list->tail->next == NULL);
        list->tail->next = p_item;
    }
    list->tail = p_item;
}

void init_declarator_list_destroy(_Dtor struct init_declarator_list* p)
{
    struct init_declarator* _Owner _Opt item = p->head;
    while (item)
    {
        struct init_declarator* _Owner _Opt next = item->next;
        item->next = NULL;
        init_declarator_delete(item);
        item = next;
    }
}

struct init_declarator_list init_declarator_list(struct parser_ctx* ctx,
    struct declaration_specifiers* p_declaration_specifiers)
{
    /*
    init-declarator-list:
      init-declarator
      init-declarator-list , init-declarator
    */
    struct init_declarator_list init_declarator_list = { 0 };
    struct init_declarator* _Owner _Opt p_init_declarator = NULL;

    try
    {
        p_init_declarator = init_declarator(ctx, p_declaration_specifiers);

        if (p_init_declarator == NULL)
            throw;

        init_declarator_list_add(&init_declarator_list, p_init_declarator);
        p_init_declarator = NULL; /*MOVED*/

        while (ctx->current != NULL && ctx->current->type == ',')
        {
            parser_match(ctx);
            p_init_declarator = init_declarator(ctx, p_declaration_specifiers);
            if (p_init_declarator == NULL)
                throw;

            init_declarator_list_add(&init_declarator_list, p_init_declarator);
            p_init_declarator = NULL; /*MOVED*/
        }
    }
    catch
    {
    }

    return init_declarator_list;
}

void storage_class_specifier_delete(struct storage_class_specifier* _Owner _Opt p)
{
    if (p)
    {
        free(p);
    }
}

struct storage_class_specifier* _Owner _Opt storage_class_specifier(struct parser_ctx* ctx)
{
    struct storage_class_specifier* _Owner _Opt p_storage_class_specifier = NULL;
    try
    {
        if (ctx->current == NULL)
        {
            unexpected_end_of_file(ctx);
            throw;
        }

        p_storage_class_specifier = calloc(1, sizeof(struct storage_class_specifier));
        if (p_storage_class_specifier == NULL)
            throw;

        p_storage_class_specifier->token = ctx->current;
        switch (ctx->current->type)
        {
        case TK_KEYWORD_TYPEDEF:
            p_storage_class_specifier->flags = STORAGE_SPECIFIER_TYPEDEF;
            break;
        case TK_KEYWORD_EXTERN:
            p_storage_class_specifier->flags = STORAGE_SPECIFIER_EXTERN;
            break;
        case TK_KEYWORD_CONSTEXPR:

            p_storage_class_specifier->flags = STORAGE_SPECIFIER_CONSTEXPR;
            if (ctx->scopes.tail && ctx->scopes.tail->scope_level == 0)
                p_storage_class_specifier->flags |= STORAGE_SPECIFIER_CONSTEXPR_STATIC;
            break;
        case TK_KEYWORD_STATIC:
            p_storage_class_specifier->flags = STORAGE_SPECIFIER_STATIC;
            break;
        case TK_KEYWORD__THREAD_LOCAL:
            p_storage_class_specifier->flags = STORAGE_SPECIFIER_THREAD_LOCAL;
            break;
        case TK_KEYWORD_AUTO:
            p_storage_class_specifier->flags = STORAGE_SPECIFIER_AUTO;
            break;
        case TK_KEYWORD_REGISTER:
            p_storage_class_specifier->flags = STORAGE_SPECIFIER_REGISTER;
            break;
        default:
            assert(false);
        }

        /*
         TODO
         thread_local may appear with static or extern,
         auto may appear with all the others except typedef138), and
         constexpr may appear with auto, register, or static.
        */

        parser_match(ctx);
    }
    catch
    {
        storage_class_specifier_delete(p_storage_class_specifier);
        p_storage_class_specifier = NULL;
    }

    return p_storage_class_specifier;
}

struct typeof_specifier_argument* _Owner _Opt typeof_specifier_argument(struct parser_ctx* ctx)
{
    struct typeof_specifier_argument* _Owner _Opt new_typeof_specifier_argument = NULL;
    try
    {
        new_typeof_specifier_argument = calloc(1, sizeof(struct typeof_specifier_argument));
        if (new_typeof_specifier_argument == NULL)
            throw;

        if (first_of_type_name(ctx))
        {
            new_typeof_specifier_argument->type_name = type_name(ctx);
        }
        else
        {
            const bool disable_evaluation_copy = ctx->evaluation_is_disabled;
            ctx->evaluation_is_disabled = true;
            new_typeof_specifier_argument->expression = expression(ctx);
            /*restore*/
            ctx->evaluation_is_disabled = disable_evaluation_copy;

            if (new_typeof_specifier_argument->expression == NULL)
                throw;

            // declarator_type_clear_name(new_typeof_specifier_argument->expression->type.declarator_type);
        }
    }
    catch
    {
        typeof_spe