#include <iostream>
#include <string>
#include <cstdlib>
#include <csignal>
#include "tinyfiledialogs.h"

#ifdef _WIN32
    #define FFMPEG_CAPTURE_COMMAND_AUDIO_OFF "ffmpeg -f gdigrab -framerate %d -i desktop -video_size %s -c:v libx264 -pix_fmt yuv420p \"%s\""
    // System sound loopback device
    #define WINDOWS_SYSTEM_SOUND "-f dshow -i audio=\"virtual-audio-capturer\""
    // Default mic device
    #define WINDOWS_MIC_INPUT "-f dshow -i audio=\"Microphone (Realtek(R) Audio)\""
#else
    // Linux X11 grab command, audio off by default:
    #define FFMPEG_CAPTURE_COMMAND_AUDIO_OFF "ffmpeg -f x11grab -framerate %d -video_size %s -i :0.0 -c:v libx264 -pix_fmt yuv420p \"%s\""
    // PulseAudio system sound (monitor) device
    #define LINUX_SYSTEM_SOUND "-f pulse -i default"
    // PulseAudio mic input device (default mic)
    #define LINUX_MIC_INPUT "-f pulse -i default"
#endif

volatile sig_atomic_t stopRecording = 0;

void handleSignal(int signal) {
    stopRecording = 1;
    std::cout << "\nStopping recording...\n";
}

std::string getResolution(int choice) {
    switch (choice) {
        case 1: return "640x480";   // 480p
        case 2: return "1280x720";  // 720p
        case 3: return "1920x1080"; // 1080p
        default: return "1280x720";
    }
}

int main() {
    signal(SIGINT, handleSignal);

    std::cout << "Select Resolution:\n";
    std::cout << "1. 480p\n2. 720p\n3. 1080p\n> ";
    int resChoice;
    std::cin >> resChoice;
    std::string resolution = getResolution(resChoice);

    // 60 fps Only
    int fps = 60;
    std::cout << "Frame rate fixed to 60 FPS.\n";

    // Audio options
    std::cout << "Audio input options:\n";
    std::cout << "0. No audio\n";
    std::cout << "1. PC system sound\n";
    std::cout << "2. External microphone\n";
    std::cout << "> ";
    int audioChoice;
    std::cin >> audioChoice;

    if (audioChoice < 0 || audioChoice > 2) {
        std::cerr << "Invalid audio choice. No audio will be recorded.\n";
        audioChoice = 0;
    }

    const char* filters[] = { "*.mp4" };
    const char* savePath = tinyfd_saveFileDialog(
        "Save Recording As",
        "recording.mp4",
        1,
        filters,
        "MP4 files"
    );

    if (!savePath) {
        std::cerr << "No file selected. Exiting.\n";
        return 1;
    }

    std::string ffmpegCommand;

#ifdef _WIN32
    // Windows build

    // Base video input (gdigrab desktop)
    std::string videoPart = " -f gdigrab -framerate " + std::to_string(fps) + " -video_size " + resolution + " -i desktop";

    if (audioChoice == 0) {
        // No audio
        ffmpegCommand = "ffmpeg" + videoPart + " -c:v libx264 -pix_fmt yuv420p \"" + std::string(savePath) + "\"";
    } else if (audioChoice == 1) {
        // PC system sound (loopback)
        // virtual audio capturer (Might be required)
        ffmpegCommand = "ffmpeg -f gdigrab -framerate " + std::to_string(fps) + " -video_size " + resolution + " -i desktop -f dshow -i audio=\"virtual-audio-capturer\" -c:v libx264 -pix_fmt yuv420p -c:a aac \"" + std::string(savePath) + "\"";
    } else if (audioChoice == 2) {
        // External microphone
        ffmpegCommand = "ffmpeg -f gdigrab -framerate " + std::to_string(fps) + " -video_size " + resolution + " -i desktop -f dshow -i audio=\"Microphone (Realtek(R) Audio)\" -c:v libx264 -pix_fmt yuv420p -c:a aac \"" + std::string(savePath) + "\"";
    }

#else
    // Linux build

    // Base video input (x11grab)
    std::string videoPart = "-f x11grab -framerate " + std::to_string(fps) + " -video_size " + resolution + " -i :0.0";

    if (audioChoice == 0) {
        // No audio
        ffmpegCommand = "ffmpeg " + videoPart + " -c:v libx264 -pix_fmt yuv420p \"" + std::string(savePath) + "\"";
    } else if (audioChoice == 1) {
        // PC system sound
        // Default PulseAudio monitor device is 'default'
        ffmpegCommand = "ffmpeg " + videoPart + " -f pulse -i default -c:v libx264 -pix_fmt yuv420p -c:a aac \"" + std::string(savePath) + "\"";
    } else if (audioChoice == 2) {
        // External mic input - default pulse mic device
        ffmpegCommand = "ffmpeg " + videoPart + " -f pulse -i default -c:v libx264 -pix_fmt yuv420p -c:a aac \"" + std::string(savePath) + "\"";
    }
#endif

    std::cout << "Running command:\n" << ffmpegCommand << "\n";
    std::cout << "Starting screen recording... Press Ctrl+C to stop.\n";

    int ret = std::system(ffmpegCommand.c_str());

    if (ret == 0) {
        std::cout << "Recording completed successfully.\n";
    } else if (ret == 130) {
        std::cout << "Recording stopped by user (SIGINT).\n";
    } else {
        std::cerr << "Recording failed. Exit code: " << ret << "\n";
        return 1;
    }

    std::cout << "Recording saved to: " << savePath << "\n";
    return 0;
}

//Enjoy hehe
