// This file is part of the FidelityFX SDK.
//
// Copyright (C) 2025 Advanced Micro Devices, Inc.
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and /or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#pragma once

#include "../render/buffer.h"
#include "../render/commandlist.h"
#include "../render/device.h"
#include "../render/gpuresource.h"
#include "../render/pipelineobject.h"
#include "../render/texture.h"

// Temporary
#include "../../../../FidelityFX/api/internal/ffx_internal_types.h"
#include "../../../../FidelityFX/api/internal/ffx_error.h"
#include "../../../../FidelityFX/api/internal/ffx_interface.h"
// Temporary

#include "../render/dx12/gpuresource_dx12.h"
#include "../../../../FidelityFX/api/include/dx12/ffx_api_dx12.h"

namespace SDKWrapper
{
static inline FfxApiResource ffxGetResourceApi(const cauldron::GPUResource* cauldronResource,
    uint32_t            state = FFX_API_RESOURCE_STATE_COMPUTE_READ,
    uint32_t            additionalUsages = 0)
{
    ID3D12Resource* pDX12Resource = cauldronResource ? const_cast<ID3D12Resource*>(cauldronResource->GetImpl()->DX12Resource()) : nullptr;
    FfxApiResource apiRes = ffxApiGetResourceDX12(pDX12Resource, state, additionalUsages);
    // If this is a buffer, and a stride was specified, preserve it
    if (cauldronResource && cauldronResource->IsBuffer() && cauldronResource->GetBufferResource() && cauldronResource->GetBufferResource()->GetDesc().Stride)
        apiRes.description.stride = cauldronResource->GetBufferResource()->GetDesc().Stride;
    return apiRes;

    cauldron::CauldronCritical(L"Unsupported API or Platform for FFX Validation Remap");
    return FfxApiResource();   // Error
}

//////////////////////////////////////////////////////////////////////////
// FFX to Framework conversion functions

static cauldron::ResourceFormat GetFrameworkSurfaceFormat(FfxApiSurfaceFormat format)
{
    switch (format)
    {
    case FFX_API_SURFACE_FORMAT_UNKNOWN:
        return cauldron::ResourceFormat::Unknown;
    case FFX_API_SURFACE_FORMAT_R32G32B32A32_TYPELESS:
        return cauldron::ResourceFormat::RGBA32_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R32G32B32A32_FLOAT:
        return cauldron::ResourceFormat::RGBA32_FLOAT;
    case FFX_API_SURFACE_FORMAT_R16G16B16A16_FLOAT:
        return cauldron::ResourceFormat::RGBA16_FLOAT;
    case FFX_API_SURFACE_FORMAT_R32G32_FLOAT:
        return cauldron::ResourceFormat::RG32_FLOAT;
    case FFX_API_SURFACE_FORMAT_R32_UINT:
        return cauldron::ResourceFormat::R32_UINT;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_TYPELESS:
        return cauldron::ResourceFormat::RGBA8_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_UNORM:
        return cauldron::ResourceFormat::RGBA8_UNORM;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_SNORM:
        return cauldron::ResourceFormat::RGBA8_SNORM;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_SRGB:
        return cauldron::ResourceFormat::RGBA8_SRGB;
    case FFX_API_SURFACE_FORMAT_B8G8R8A8_TYPELESS:
        return cauldron::ResourceFormat::BGRA8_TYPELESS;
    case FFX_API_SURFACE_FORMAT_B8G8R8A8_UNORM:
        return cauldron::ResourceFormat::BGRA8_UNORM;
    case FFX_API_SURFACE_FORMAT_B8G8R8A8_SRGB:
        return cauldron::ResourceFormat::BGRA8_SRGB;
    case FFX_API_SURFACE_FORMAT_R11G11B10_FLOAT:
        return cauldron::ResourceFormat::RG11B10_FLOAT;
    case FFX_API_SURFACE_FORMAT_R9G9B9E5_SHAREDEXP:
        return cauldron::ResourceFormat::RGB9E5_SHAREDEXP;
    case FFX_API_SURFACE_FORMAT_R16G16_FLOAT:
        return cauldron::ResourceFormat::RG16_FLOAT;
    case FFX_API_SURFACE_FORMAT_R16G16_UINT:
        return cauldron::ResourceFormat::RG16_UINT;
    case FFX_API_SURFACE_FORMAT_R16G16_SINT:
        return cauldron::ResourceFormat::RG16_SINT;
    case FFX_API_SURFACE_FORMAT_R16_FLOAT:
        return cauldron::ResourceFormat::R16_FLOAT;
    case FFX_API_SURFACE_FORMAT_R16_UINT:
        return cauldron::ResourceFormat::R16_UINT;
    case FFX_API_SURFACE_FORMAT_R16_UNORM:
        return cauldron::ResourceFormat::R16_UNORM;
    case FFX_API_SURFACE_FORMAT_R16_SNORM:
        return cauldron::ResourceFormat::R16_SNORM;
    case FFX_API_SURFACE_FORMAT_R8_UNORM:
        return cauldron::ResourceFormat::R8_UNORM;
    case FFX_API_SURFACE_FORMAT_R8_SNORM:
        return cauldron::ResourceFormat::R8_SNORM;
    case FFX_API_SURFACE_FORMAT_R8_UINT:
        return cauldron::ResourceFormat::R8_UINT;
    case FFX_API_SURFACE_FORMAT_R8G8_UNORM:
        return cauldron::ResourceFormat::RG8_UNORM;
    case FFX_API_SURFACE_FORMAT_R32_FLOAT:
        return cauldron::ResourceFormat::R32_FLOAT;
    case FFX_API_SURFACE_FORMAT_R10G10B10A2_UNORM:
        return cauldron::ResourceFormat::RGB10A2_UNORM;
    case FFX_API_SURFACE_FORMAT_R10G10B10A2_TYPELESS:
        return cauldron::ResourceFormat::RGB10A2_TYPELESS;
    default:
        cauldron::CauldronCritical(L"FFXInterface: Framework: Unsupported format requested. Please implement.");
        return cauldron::ResourceFormat::Unknown;
    }
}

static cauldron::ResourceFlags GetFrameworkResourceFlags(FfxApiResourceUsage flags)
{
    cauldron::ResourceFlags cauldronResourceFlags = cauldron::ResourceFlags::None;

    if (flags & FFX_API_RESOURCE_USAGE_RENDERTARGET)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowRenderTarget;

    if (flags & FFX_API_RESOURCE_USAGE_DEPTHTARGET)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowDepthStencil;

    if (flags & FFX_API_RESOURCE_USAGE_UAV)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowUnorderedAccess;

    if (flags & FFX_API_RESOURCE_USAGE_INDIRECT)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowIndirect;

    return cauldronResourceFlags;
}

inline cauldron::ResourceState GetFrameworkState(FfxApiResourceState state)
{
    switch (state)
    {
    case FFX_API_RESOURCE_STATE_UNORDERED_ACCESS:
        return cauldron::ResourceState::UnorderedAccess;
    case FFX_API_RESOURCE_STATE_COMPUTE_READ:
        return cauldron::ResourceState::NonPixelShaderResource;
    case FFX_API_RESOURCE_STATE_PIXEL_READ:
        return cauldron::ResourceState::PixelShaderResource;
    case FFX_API_RESOURCE_STATE_PIXEL_COMPUTE_READ:
        return cauldron::ResourceState(cauldron::ResourceState::NonPixelShaderResource | cauldron::ResourceState::PixelShaderResource);
    case FFX_API_RESOURCE_STATE_COPY_SRC:
        return cauldron::ResourceState::CopySource;
    case FFX_API_RESOURCE_STATE_COPY_DEST:
        return cauldron::ResourceState::CopyDest;
    case FFX_API_RESOURCE_STATE_GENERIC_READ:
        return (cauldron::ResourceState)((uint32_t)cauldron::ResourceState::NonPixelShaderResource | (uint32_t)cauldron::ResourceState::PixelShaderResource);
    case FFX_API_RESOURCE_STATE_INDIRECT_ARGUMENT:
        return cauldron::ResourceState::IndirectArgument;
    case FFX_API_RESOURCE_STATE_PRESENT:
        return cauldron::ResourceState::Present;
    case FFX_API_RESOURCE_STATE_RENDER_TARGET:
        return cauldron::ResourceState::RenderTargetResource;
    case FFX_API_RESOURCE_STATE_DEPTH_ATTACHMENT:
        return cauldron::ResourceState::DepthWrite;
    default:
        cauldron::CauldronCritical(L"FFXInterface: Cauldron: Unsupported resource state requested. Please implement.");
        return cauldron::ResourceState::CommonResource;
    }
}

inline cauldron::TextureDesc GetFrameworkTextureDescription(const FfxApiResourceDescription& desc)
{
    cauldron::ResourceFormat format = GetFrameworkSurfaceFormat((FfxApiSurfaceFormat)desc.format);
    cauldron::ResourceFlags  flags  = GetFrameworkResourceFlags((FfxApiResourceUsage)desc.usage);
    switch (desc.type)
    {
    case FFX_API_RESOURCE_TYPE_TEXTURE1D:
        return cauldron::TextureDesc::Tex1D(L"", format, desc.width, 1, desc.mipCount, flags);
    case FFX_API_RESOURCE_TYPE_TEXTURE2D:
        return cauldron::TextureDesc::Tex2D(L"", format, desc.width, desc.height, 1, desc.mipCount, flags);
    case FFX_API_RESOURCE_TYPE_TEXTURE_CUBE:
        return cauldron::TextureDesc::TexCube(L"", format, desc.width, desc.height, 1, desc.mipCount, flags);
    case FFX_API_RESOURCE_TYPE_TEXTURE3D:
        return cauldron::TextureDesc::Tex3D(L"", format, desc.width, desc.height, desc.depth, desc.mipCount, flags);
    default:
        cauldron::CauldronCritical(L"Description should be a texture.");
        return cauldron::TextureDesc();
    }
}

inline cauldron::BufferDesc GetFrameworkBufferDescription(const FfxApiResourceDescription& desc)
{
    if (desc.type == FFX_API_RESOURCE_TYPE_BUFFER)
    {
        cauldron::ResourceFlags flags = GetFrameworkResourceFlags((FfxApiResourceUsage)desc.usage);
        return cauldron::BufferDesc::Data(L"", desc.size, desc.stride, desc.alignment, flags);
    }
    else
    {
        cauldron::CauldronCritical(L"Description should be a texture.");
        return cauldron::BufferDesc();
    }
}

static cauldron::ResourceFormat GetFrameworkSurfaceFormatApi(uint32_t format)
{
    switch (format)
    {
    case FFX_API_SURFACE_FORMAT_UNKNOWN:
        return cauldron::ResourceFormat::Unknown;
    case FFX_API_SURFACE_FORMAT_R32G32B32A32_TYPELESS:
        return cauldron::ResourceFormat::RGBA32_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R32G32B32A32_FLOAT:
        return cauldron::ResourceFormat::RGBA32_FLOAT;
    case FFX_API_SURFACE_FORMAT_R32G32B32_FLOAT:
        return cauldron::ResourceFormat::RGB32_FLOAT;
    case FFX_API_SURFACE_FORMAT_R16G16B16A16_TYPELESS:
        return cauldron::ResourceFormat::RGBA16_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R16G16B16A16_FLOAT:
        return cauldron::ResourceFormat::RGBA16_FLOAT;
    case FFX_API_SURFACE_FORMAT_R32G32_TYPELESS:
        return cauldron::ResourceFormat::RG32_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R32G32_FLOAT:
        return cauldron::ResourceFormat::RG32_FLOAT;
    case FFX_API_SURFACE_FORMAT_R32_UINT:
        return cauldron::ResourceFormat::R32_UINT;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_TYPELESS:
        return cauldron::ResourceFormat::RGBA8_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_UNORM:
        return cauldron::ResourceFormat::RGBA8_UNORM;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_SNORM:
        return cauldron::ResourceFormat::RGBA8_SNORM;
    case FFX_API_SURFACE_FORMAT_R8G8B8A8_SRGB:
        return cauldron::ResourceFormat::RGBA8_SRGB;
    case FFX_API_SURFACE_FORMAT_B8G8R8A8_TYPELESS:
        return cauldron::ResourceFormat::BGRA8_TYPELESS;
    case FFX_API_SURFACE_FORMAT_B8G8R8A8_UNORM:
        return cauldron::ResourceFormat::BGRA8_UNORM;
    case FFX_API_SURFACE_FORMAT_B8G8R8A8_SRGB:
        return cauldron::ResourceFormat::BGRA8_SRGB;
    case FFX_API_SURFACE_FORMAT_R11G11B10_FLOAT:
        return cauldron::ResourceFormat::RG11B10_FLOAT;
    case FFX_API_SURFACE_FORMAT_R9G9B9E5_SHAREDEXP:
        return cauldron::ResourceFormat::RGB9E5_SHAREDEXP;
    case FFX_API_SURFACE_FORMAT_R16G16_TYPELESS:
        return cauldron::ResourceFormat::RG16_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R16G16_FLOAT:
        return cauldron::ResourceFormat::RG16_FLOAT;
    case FFX_API_SURFACE_FORMAT_R16G16_UINT:
        return cauldron::ResourceFormat::RG16_UINT;
    case FFX_API_SURFACE_FORMAT_R16G16_SINT:
        return cauldron::ResourceFormat::RG16_SINT;
    case FFX_API_SURFACE_FORMAT_R16_TYPELESS:
        return cauldron::ResourceFormat::R16_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R16_FLOAT:
        return cauldron::ResourceFormat::R16_FLOAT;
    case FFX_API_SURFACE_FORMAT_R16_UINT:
        return cauldron::ResourceFormat::R16_UINT;
    case FFX_API_SURFACE_FORMAT_R16_UNORM:
        return cauldron::ResourceFormat::R16_UNORM;
    case FFX_API_SURFACE_FORMAT_R16_SNORM:
        return cauldron::ResourceFormat::R16_SNORM;
    case FFX_API_SURFACE_FORMAT_R8_TYPELESS:
        return cauldron::ResourceFormat::R8_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R8_UNORM:
        return cauldron::ResourceFormat::R8_UNORM;
    case FFX_API_SURFACE_FORMAT_R8_SNORM:
        return cauldron::ResourceFormat::R8_SNORM;
    case FFX_API_SURFACE_FORMAT_R8_UINT:
        return cauldron::ResourceFormat::R8_UINT;
    case FFX_API_SURFACE_FORMAT_R8G8_TYPELESS:
        return cauldron::ResourceFormat::RG8_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R8G8_UNORM:
        return cauldron::ResourceFormat::RG8_UNORM;
    case FFX_API_SURFACE_FORMAT_R32_TYPELESS:
        return cauldron::ResourceFormat::R32_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R32_FLOAT:
        return cauldron::ResourceFormat::R32_FLOAT;
    case FFX_API_SURFACE_FORMAT_R10G10B10A2_TYPELESS:
        return cauldron::ResourceFormat::RGB10A2_TYPELESS;
    case FFX_API_SURFACE_FORMAT_R10G10B10A2_UNORM:
        return cauldron::ResourceFormat::RGB10A2_UNORM;
    default:
        cauldron::CauldronCritical(L"FFXInterface: Framework: Unsupported format requested. Please implement.");
        return cauldron::ResourceFormat::Unknown;
    }
}

static cauldron::ResourceFlags GetFrameworkResourceFlagsApi(uint32_t flags)
{
    cauldron::ResourceFlags cauldronResourceFlags = cauldron::ResourceFlags::None;

    if (flags & FFX_API_RESOURCE_USAGE_RENDERTARGET)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowRenderTarget;

    if (flags & FFX_API_RESOURCE_USAGE_DEPTHTARGET)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowDepthStencil;

    if (flags & FFX_API_RESOURCE_USAGE_UAV)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowUnorderedAccess;

    if (flags & FFX_API_RESOURCE_USAGE_INDIRECT)
        cauldronResourceFlags |= cauldron::ResourceFlags::AllowIndirect;

    return cauldronResourceFlags;
}

//////////////////////////////////////////////////////////////////////////
// Framework to FFX conversion functions

inline FfxApiSurfaceFormat GetFfxSurfaceFormat(cauldron::ResourceFormat format)
{
    switch (format)
    {
    case (cauldron::ResourceFormat::RGBA32_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R32G32B32A32_TYPELESS;
    case (cauldron::ResourceFormat::RGBA32_UINT):
        return FFX_API_SURFACE_FORMAT_R32G32B32A32_UINT;
    case (cauldron::ResourceFormat::RGBA32_FLOAT):
        return FFX_API_SURFACE_FORMAT_R32G32B32A32_FLOAT;
    case (cauldron::ResourceFormat::RGBA16_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R16G16B16A16_TYPELESS;
    case (cauldron::ResourceFormat::RGBA16_FLOAT):
        return FFX_API_SURFACE_FORMAT_R16G16B16A16_FLOAT;
    case (cauldron::ResourceFormat::RGB32_FLOAT):
        return FFX_API_SURFACE_FORMAT_R32G32B32_FLOAT;
    case (cauldron::ResourceFormat::RG32_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R32G32_TYPELESS;
    case (cauldron::ResourceFormat::RG32_FLOAT):
        return FFX_API_SURFACE_FORMAT_R32G32_FLOAT;
    case (cauldron::ResourceFormat::R8_UINT):
        return FFX_API_SURFACE_FORMAT_R8_UINT;
    case (cauldron::ResourceFormat::R32_UINT):
        return FFX_API_SURFACE_FORMAT_R32_UINT;
    case (cauldron::ResourceFormat::RGBA8_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R8G8B8A8_TYPELESS;
    case (cauldron::ResourceFormat::RGBA8_UNORM):
        return FFX_API_SURFACE_FORMAT_R8G8B8A8_UNORM;
    case (cauldron::ResourceFormat::RGBA8_SNORM):
        return FFX_API_SURFACE_FORMAT_R8G8B8A8_SNORM;
    case (cauldron::ResourceFormat::RGBA8_SRGB):
        return FFX_API_SURFACE_FORMAT_R8G8B8A8_SRGB;
    case (cauldron::ResourceFormat::BGRA8_TYPELESS):
        return FFX_API_SURFACE_FORMAT_B8G8R8A8_TYPELESS;
    case (cauldron::ResourceFormat::BGRA8_UNORM):
        return FFX_API_SURFACE_FORMAT_B8G8R8A8_UNORM;
    case (cauldron::ResourceFormat::BGRA8_SRGB):
        return FFX_API_SURFACE_FORMAT_B8G8R8A8_SRGB;
    case (cauldron::ResourceFormat::RG11B10_FLOAT):
        return FFX_API_SURFACE_FORMAT_R11G11B10_FLOAT;
    case (cauldron::ResourceFormat::RGB9E5_SHAREDEXP):
        return FFX_API_SURFACE_FORMAT_R9G9B9E5_SHAREDEXP;
    case (cauldron::ResourceFormat::RGB10A2_UNORM):
        return FFX_API_SURFACE_FORMAT_R10G10B10A2_UNORM;
    case (cauldron::ResourceFormat::RGB10A2_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R10G10B10A2_TYPELESS;
    case (cauldron::ResourceFormat::RG16_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R16G16_TYPELESS;
    case (cauldron::ResourceFormat::RG16_FLOAT):
        return FFX_API_SURFACE_FORMAT_R16G16_FLOAT;
    case (cauldron::ResourceFormat::RG16_UINT):
        return FFX_API_SURFACE_FORMAT_R16G16_UINT;
    case (cauldron::ResourceFormat::RG16_SINT):
        return FFX_API_SURFACE_FORMAT_R16G16_SINT;
    case (cauldron::ResourceFormat::R16_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R16_TYPELESS;
    case (cauldron::ResourceFormat::R16_FLOAT):
        return FFX_API_SURFACE_FORMAT_R16_FLOAT;
    case (cauldron::ResourceFormat::R16_UINT):
        return FFX_API_SURFACE_FORMAT_R16_UINT;
    case (cauldron::ResourceFormat::R16_UNORM):
        return FFX_API_SURFACE_FORMAT_R16_UNORM;
    case (cauldron::ResourceFormat::R16_SNORM):
        return FFX_API_SURFACE_FORMAT_R16_SNORM;
    case (cauldron::ResourceFormat::R8_TYPELESS):
        return FFX_API_SURFACE_FORMAT_R8_TYPELESS;
    case (cauldron::ResourceFormat::R8_UNORM):
        return FFX_API_SURFACE_FORMAT_R8_UNORM;
    case (cauldron::ResourceFormat::R8_SNORM):
        return FFX_API_SURFACE_FORMAT_R8_SNORM;
    case cauldron::ResourceFormat::RG8_TYPELESS:
        return FFX_API_SURFACE_FORMAT_R8G8_TYPELESS;
    case cauldron::ResourceFormat::RG8_UNORM:
        return FFX_API_SURFACE_FORMAT_R8G8_UNORM;
    case cauldron::ResourceFormat::RG8_UINT:
        return FFX_API_SURFACE_FORMAT_R8G8_UINT;
    case cauldron::ResourceFormat::R32_TYPELESS:
        return FFX_API_SURFACE_FORMAT_R32_TYPELESS;
    case cauldron::ResourceFormat::R32_FLOAT:
    case cauldron::ResourceFormat::D32_FLOAT:
        return FFX_API_SURFACE_FORMAT_R32_FLOAT;
    case (cauldron::ResourceFormat::Unknown):
        return FFX_API_SURFACE_FORMAT_UNKNOWN;
    default:
        cauldron::CauldronCritical(L"ValidationRemap: Unsupported format requested. Please implement.");
        return FFX_API_SURFACE_FORMAT_UNKNOWN;
    }
}

inline FfxApiResourceDescription GetFfxResourceDescription(const cauldron::GPUResource* pResource, FfxApiResourceUsage additionalUsages)
{
    FfxApiResourceDescription resourceDescription = {};

    // This is valid
    if (!pResource)
        return resourceDescription;

    if (pResource->IsBuffer())
    {
        const cauldron::BufferDesc& bufDesc = pResource->GetBufferResource()->GetDesc();

        resourceDescription.flags  = FFX_API_RESOURCE_FLAGS_NONE;
        resourceDescription.usage  = FFX_API_RESOURCE_USAGE_UAV;
        resourceDescription.width  = bufDesc.Size;
        resourceDescription.height = bufDesc.Stride;
        resourceDescription.format = GetFfxSurfaceFormat(cauldron::ResourceFormat::Unknown);

        // Does not apply to buffers
        resourceDescription.depth    = 0;
        resourceDescription.mipCount = 0;

        // Set the type
        resourceDescription.type = FFX_API_RESOURCE_TYPE_BUFFER;
    }
    else
    {
        const cauldron::TextureDesc& texDesc = pResource->GetTextureResource()->GetDesc();

        // Set flags properly for resource registration
        resourceDescription.flags = FFX_API_RESOURCE_FLAGS_NONE;
        resourceDescription.usage = IsDepth(texDesc.Format) ? FFX_API_RESOURCE_USAGE_DEPTHTARGET : FFX_API_RESOURCE_USAGE_READ_ONLY;
        if (static_cast<bool>(texDesc.Flags & cauldron::ResourceFlags::AllowUnorderedAccess))
            resourceDescription.usage = (FfxApiResourceUsage)(resourceDescription.usage | FFX_API_RESOURCE_USAGE_UAV);

        resourceDescription.width    = texDesc.Width;
        resourceDescription.height   = texDesc.Height;
        resourceDescription.depth    = texDesc.DepthOrArraySize;
        resourceDescription.mipCount = texDesc.MipLevels;
        resourceDescription.format   = GetFfxSurfaceFormat(texDesc.Format);

        resourceDescription.usage = (FfxApiResourceUsage)(resourceDescription.usage | additionalUsages);

        switch (texDesc.Dimension)
        {
        case cauldron::TextureDimension::Texture1D:
            resourceDescription.type = FFX_API_RESOURCE_TYPE_TEXTURE1D;
            break;
        case cauldron::TextureDimension::Texture2D:
            resourceDescription.type = FFX_API_RESOURCE_TYPE_TEXTURE2D;
            break;
        case cauldron::TextureDimension::CubeMap:
            // 2D array access to cube map resources
            if (FFX_CONTAINS_FLAG(resourceDescription.usage, FFX_API_RESOURCE_USAGE_ARRAYVIEW))
                resourceDescription.type = FFX_API_RESOURCE_TYPE_TEXTURE2D;
            else
                resourceDescription.type = FFX_API_RESOURCE_TYPE_TEXTURE_CUBE;
            break;
        case cauldron::TextureDimension::Texture3D:
            resourceDescription.type = FFX_API_RESOURCE_TYPE_TEXTURE3D;
            break;
        default:
            cauldron::CauldronCritical(L"FFXInterface: Cauldron: Unsupported texture dimension requested. Please implement.");
            break;
        }
    }

    return resourceDescription;
}

static FfxApiBackbufferTransferFunction GetFfxApiBackbufferTransferFunction(const DisplayMode displayMode)
{
    switch (displayMode)
    {
    case DisplayMode::DISPLAYMODE_LDR:
        return FfxApiBackbufferTransferFunction::FFX_API_BACKBUFFER_TRANSFER_FUNCTION_SRGB;
    case DisplayMode::DISPLAYMODE_HDR10_2084:
        return FfxApiBackbufferTransferFunction::FFX_API_BACKBUFFER_TRANSFER_FUNCTION_PQ;
    case DisplayMode::DISPLAYMODE_HDR10_SCRGB:
        return FfxApiBackbufferTransferFunction::FFX_API_BACKBUFFER_TRANSFER_FUNCTION_SCRGB;
    default:
        cauldron::CauldronCritical(L"FFXInterface: Cauldron: Unsupported display mode requested. Please implement.");
        return FfxApiBackbufferTransferFunction::FFX_API_BACKBUFFER_TRANSFER_FUNCTION_SRGB;
    }
}

} // namespace SDKWrapper
