"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.killProcess = exports.ProcessWrapper = exports.StartError = void 0;
const child_process_1 = require("child_process");
const os_1 = __importDefault(require("os"));
const child_process_2 = __importDefault(require("child_process"));
const path_1 = __importDefault(require("path"));
const events_1 = __importDefault(require("events"));
const iconv_lite_1 = __importDefault(require("iconv-lite"));
class StartError extends Error {
}
exports.StartError = StartError;
class ProcessWrapper extends events_1.default {
    constructor(file, args, cwd, timeout = 0, code = "utf-8", option = {}) {
        super();
        this.file = file;
        this.args = args;
        this.cwd = cwd;
        this.timeout = timeout;
        this.code = code;
        this.option = option;
        this.errMsg = {
            timeoutErr: "task timeout!",
            exitErr: "task error!",
            startErr: "task start error!"
        };
    }
    setErrMsg(errMsg) {
        this.errMsg = errMsg;
    }
    start() {
        return new Promise((resolve, reject) => {
            let timeTask;
            const subProcess = child_process_2.default.spawn(this.file, this.args, {
                stdio: "pipe",
                windowsHide: true,
                cwd: path_1.default.normalize(this.cwd),
                ...this.option
            });
            this.process = subProcess;
            this.pid = subProcess.pid;
            this.emit("start", subProcess.pid);
            if (!subProcess || !subProcess.pid)
                return reject(new Error(this.errMsg.startErr));
            subProcess.stdout.on("data", (text) => this.emit("data", iconv_lite_1.default.decode(text, this.code)));
            subProcess.stderr.on("data", (text) => this.emit("data", iconv_lite_1.default.decode(text, this.code)));
            subProcess.on("exit", (code) => {
                try {
                    this.emit("exit", code);
                    this.destroy();
                }
                catch (error) { }
                if (timeTask)
                    clearTimeout(timeTask);
                if (code !== 0)
                    return reject(new Error(this.errMsg.exitErr));
                return resolve(true);
            });
            // timeout, terminate the task
            if (this.timeout) {
                timeTask = setTimeout(() => {
                    if ((subProcess === null || subProcess === void 0 ? void 0 : subProcess.pid) && !subProcess.exitCode && subProcess.exitCode !== 0) {
                        killProcess(subProcess.pid, subProcess);
                        reject(new Error(this.errMsg.timeoutErr));
                    }
                    else {
                        reject(new Error(this.errMsg.exitErr));
                    }
                }, 1000 * this.timeout);
            }
        });
    }
    getPid() {
        var _a;
        return (_a = this.process) === null || _a === void 0 ? void 0 : _a.pid;
    }
    write(data) {
        var _a, _b;
        return (_b = (_a = this.process) === null || _a === void 0 ? void 0 : _a.stdin) === null || _b === void 0 ? void 0 : _b.write(iconv_lite_1.default.encode(data, this.code));
    }
    kill() {
        var _a, _b;
        if ((_a = this.process) === null || _a === void 0 ? void 0 : _a.pid)
            killProcess((_b = this.process) === null || _b === void 0 ? void 0 : _b.pid, this.process);
    }
    status() {
        var _a;
        return !!((_a = this.process) === null || _a === void 0 ? void 0 : _a.exitCode);
    }
    exitCode() {
        var _a;
        return (_a = this.process) === null || _a === void 0 ? void 0 : _a.exitCode;
    }
    async destroy() {
        var _a, _b, _c, _d, _e, _f, _g;
        try {
            for (const n of this.eventNames())
                this.removeAllListeners(n);
            if ((_a = this.process) === null || _a === void 0 ? void 0 : _a.stdout)
                for (const eventName of this.process.stdout.eventNames())
                    this.process.stdout.removeAllListeners(eventName);
            if ((_b = this.process) === null || _b === void 0 ? void 0 : _b.stderr)
                for (const eventName of this.process.stderr.eventNames())
                    this.process.stderr.removeAllListeners(eventName);
            if (this.process)
                for (const eventName of this.process.eventNames())
                    this.process.removeAllListeners(eventName);
            (_d = (_c = this.process) === null || _c === void 0 ? void 0 : _c.stdout) === null || _d === void 0 ? void 0 : _d.destroy();
            (_f = (_e = this.process) === null || _e === void 0 ? void 0 : _e.stderr) === null || _f === void 0 ? void 0 : _f.destroy();
            if (((_g = this.process) === null || _g === void 0 ? void 0 : _g.exitCode) === null) {
                this.process.kill("SIGTERM");
                this.process.kill("SIGKILL");
            }
        }
        catch (error) {
            console.log("[ProcessWrapper destroy() Error]", error);
        }
        finally {
            this.process = undefined;
        }
    }
}
exports.ProcessWrapper = ProcessWrapper;
function killProcess(pid, process, signal) {
    try {
        if (os_1.default.platform() === "win32") {
            (0, child_process_1.execSync)(`taskkill /PID ${pid} /T /F`);
            return true;
        }
        if (os_1.default.platform() === "linux") {
            (0, child_process_1.execSync)(`kill -s 9 ${pid}`);
            return true;
        }
    }
    catch (err) {
        return signal ? process.kill(signal) : process.kill("SIGKILL");
    }
    return signal ? process.kill(signal) : process.kill("SIGKILL");
}
exports.killProcess = killProcess;
//# sourceMappingURL=process_tools.js.map