/**
 * @file
 * @brief https://pubs.opengroup.org/onlinepubs/7908799/xcurses/term.h.html
 *
 * @date 1.09.22
 * @author Aleksey Zhmulin
 */

#ifndef COMPAT_POSIX_TERM_H_
#define COMPAT_POSIX_TERM_H_

#include <endian.h>
#include <stdbool.h>
#include <stdint.h>

struct termtype {
	char *term_names;
	bool *flags;
	int16_t *numbers;
	int16_t *strings;
	char *str_table;
};

#ifdef TERM_PRIV_H_
#undef TERMINAL
#define TERMINAL struct terminal
TERMINAL;
#else
typedef struct terminal {
	struct termtype type;
} TERMINAL;
#endif /* TERM_PRIV_H_ */

extern TERMINAL *cur_term;

#define CUR_TERM_FLAGS(n)   cur_term->type.flags[n]
#define CUR_TERM_NUMBERS(n) le16toh(cur_term->type.numbers[n])
#define CUR_TERM_STRINGS(n) \
	(cur_term->type.str_table + le16toh(cur_term->type.strings[n]))

#define TINFO_AUTO_LEFT_MARGIN          CUR_TERM_FLAGS(0)
#define TINFO_AUTO_RIGHT_MARGIN         CUR_TERM_FLAGS(1)
#define TINFO_NO_ESC_CTLC               CUR_TERM_FLAGS(2)
#define TINFO_CEOL_STANDOUT_GLITCH      CUR_TERM_FLAGS(3)
#define TINFO_EAT_NEWLINE_GLITCH        CUR_TERM_FLAGS(4)
#define TINFO_ERASE_OVERSTRIKE          CUR_TERM_FLAGS(5)
#define TINFO_GENERIC_TYPE              CUR_TERM_FLAGS(6)
#define TINFO_HARD_COPY                 CUR_TERM_FLAGS(7)
#define TINFO_HAS_META_KEY              CUR_TERM_FLAGS(8)
#define TINFO_HAS_STATUS_LINE           CUR_TERM_FLAGS(9)
#define TINFO_INSERT_NULL_GLITCH        CUR_TERM_FLAGS(10)
#define TINFO_MEMORY_ABOVE              CUR_TERM_FLAGS(11)
#define TINFO_MEMORY_BELOW              CUR_TERM_FLAGS(12)
#define TINFO_MOVE_INSERT_MODE          CUR_TERM_FLAGS(13)
#define TINFO_MOVE_STANDOUT_MODE        CUR_TERM_FLAGS(14)
#define TINFO_OVER_STRIKE               CUR_TERM_FLAGS(15)
#define TINFO_STATUS_LINE_ESC_OK        CUR_TERM_FLAGS(16)
#define TINFO_DEST_TABS_MAGIC_SMSO      CUR_TERM_FLAGS(17)
#define TINFO_TILDE_GLITCH              CUR_TERM_FLAGS(18)
#define TINFO_TRANSPARENT_UNDERLINE     CUR_TERM_FLAGS(19)
#define TINFO_XON_XOFF                  CUR_TERM_FLAGS(20)
#define TINFO_NEEDS_XON_XOFF            CUR_TERM_FLAGS(21)
#define TINFO_PRTR_SILENT               CUR_TERM_FLAGS(22)
#define TINFO_HARD_CURSOR               CUR_TERM_FLAGS(23)
#define TINFO_NON_REV_RMCUP             CUR_TERM_FLAGS(24)
#define TINFO_NO_PAD_CHAR               CUR_TERM_FLAGS(25)
#define TINFO_NON_DEST_SCROLL_REGION    CUR_TERM_FLAGS(26)
#define TINFO_CAN_CHANGE                CUR_TERM_FLAGS(27)
#define TINFO_BACK_COLOR_ERASE          CUR_TERM_FLAGS(28)
#define TINFO_HUE_LIGHTNESS_SATURATION  CUR_TERM_FLAGS(29)
#define TINFO_COL_ADDR_GLITCH           CUR_TERM_FLAGS(30)
#define TINFO_CR_CANCELS_MICRO_MODE     CUR_TERM_FLAGS(31)
#define TINFO_HAS_PRINT_WHEEL           CUR_TERM_FLAGS(32)
#define TINFO_ROW_ADDR_GLITCH           CUR_TERM_FLAGS(33)
#define TINFO_SEMI_AUTO_RIGHT_MARGIN    CUR_TERM_FLAGS(34)
#define TINFO_CPI_CHANGES_RES           CUR_TERM_FLAGS(35)
#define TINFO_LPI_CHANGES_RES           CUR_TERM_FLAGS(36)
#define TINFO_COLUMNS                   CUR_TERM_NUMBERS(0)
#define TINFO_INIT_TABS                 CUR_TERM_NUMBERS(1)
#define TINFO_LINES                     CUR_TERM_NUMBERS(2)
#define TINFO_LINES_OF_MEMORY           CUR_TERM_NUMBERS(3)
#define TINFO_MAGIC_COOKIE_GLITCH       CUR_TERM_NUMBERS(4)
#define TINFO_PADDING_BAUD_RATE         CUR_TERM_NUMBERS(5)
#define TINFO_VIRTUAL_TERMINAL          CUR_TERM_NUMBERS(6)
#define TINFO_WIDTH_STATUS_LINE         CUR_TERM_NUMBERS(7)
#define TINFO_NUM_LABELS                CUR_TERM_NUMBERS(8)
#define TINFO_LABEL_HEIGHT              CUR_TERM_NUMBERS(9)
#define TINFO_LABEL_WIDTH               CUR_TERM_NUMBERS(10)
#define TINFO_MAX_ATTRIBUTES            CUR_TERM_NUMBERS(11)
#define TINFO_MAXIMUM_WINDOWS           CUR_TERM_NUMBERS(12)
#define TINFO_MAX_COLORS                CUR_TERM_NUMBERS(13)
#define TINFO_MAX_PAIRS                 CUR_TERM_NUMBERS(14)
#define TINFO_NO_COLOR_VIDEO            CUR_TERM_NUMBERS(15)
#define TINFO_BUFFER_CAPACITY           CUR_TERM_NUMBERS(16)
#define TINFO_DOT_VERT_SPACING          CUR_TERM_NUMBERS(17)
#define TINFO_DOT_HORZ_SPACING          CUR_TERM_NUMBERS(18)
#define TINFO_MAX_MICRO_ADDRESS         CUR_TERM_NUMBERS(19)
#define TINFO_MAX_MICRO_JUMP            CUR_TERM_NUMBERS(20)
#define TINFO_MICRO_COL_SIZE            CUR_TERM_NUMBERS(21)
#define TINFO_MICRO_LINE_SIZE           CUR_TERM_NUMBERS(22)
#define TINFO_NUMBER_OF_PINS            CUR_TERM_NUMBERS(23)
#define TINFO_OUTPUT_RES_CHAR           CUR_TERM_NUMBERS(24)
#define TINFO_OUTPUT_RES_LINE           CUR_TERM_NUMBERS(25)
#define TINFO_OUTPUT_RES_HORZ_INCH      CUR_TERM_NUMBERS(26)
#define TINFO_OUTPUT_RES_VERT_INCH      CUR_TERM_NUMBERS(27)
#define TINFO_PRINT_RATE                CUR_TERM_NUMBERS(28)
#define TINFO_WIDE_CHAR_SIZE            CUR_TERM_NUMBERS(29)
#define TINFO_BUTTONS                   CUR_TERM_NUMBERS(30)
#define TINFO_BIT_IMAGE_ENTWINING       CUR_TERM_NUMBERS(31)
#define TINFO_BIT_IMAGE_TYPE            CUR_TERM_NUMBERS(32)
#define TINFO_BACK_TAB                  CUR_TERM_STRINGS(0)
#define TINFO_BELL                      CUR_TERM_STRINGS(1)
#define TINFO_CARRIAGE_RETURN           CUR_TERM_STRINGS(2)
#define TINFO_CHANGE_SCROLL_REGION      CUR_TERM_STRINGS(3)
#define TINFO_CLEAR_ALL_TABS            CUR_TERM_STRINGS(4)
#define TINFO_CLEAR_SCREEN              CUR_TERM_STRINGS(5)
#define TINFO_CLR_EOL                   CUR_TERM_STRINGS(6)
#define TINFO_CLR_EOS                   CUR_TERM_STRINGS(7)
#define TINFO_COLUMN_ADDRESS            CUR_TERM_STRINGS(8)
#define TINFO_COMMAND_CHARACTER         CUR_TERM_STRINGS(9)
#define TINFO_CURSOR_ADDRESS            CUR_TERM_STRINGS(10)
#define TINFO_CURSOR_DOWN               CUR_TERM_STRINGS(11)
#define TINFO_CURSOR_HOME               CUR_TERM_STRINGS(12)
#define TINFO_CURSOR_INVISIBLE          CUR_TERM_STRINGS(13)
#define TINFO_CURSOR_LEFT               CUR_TERM_STRINGS(14)
#define TINFO_CURSOR_MEM_ADDRESS        CUR_TERM_STRINGS(15)
#define TINFO_CURSOR_NORMAL             CUR_TERM_STRINGS(16)
#define TINFO_CURSOR_RIGHT              CUR_TERM_STRINGS(17)
#define TINFO_CURSOR_TO_LL              CUR_TERM_STRINGS(18)
#define TINFO_CURSOR_UP                 CUR_TERM_STRINGS(19)
#define TINFO_CURSOR_VISIBLE            CUR_TERM_STRINGS(20)
#define TINFO_DELETE_CHARACTER          CUR_TERM_STRINGS(21)
#define TINFO_DELETE_LINE               CUR_TERM_STRINGS(22)
#define TINFO_DIS_STATUS_LINE           CUR_TERM_STRINGS(23)
#define TINFO_DOWN_HALF_LINE            CUR_TERM_STRINGS(24)
#define TINFO_ENTER_ALT_CHARSET_MODE    CUR_TERM_STRINGS(25)
#define TINFO_ENTER_BLINK_MODE          CUR_TERM_STRINGS(26)
#define TINFO_ENTER_BOLD_MODE           CUR_TERM_STRINGS(27)
#define TINFO_ENTER_CA_MODE             CUR_TERM_STRINGS(28)
#define TINFO_ENTER_DELETE_MODE         CUR_TERM_STRINGS(29)
#define TINFO_ENTER_DIM_MODE            CUR_TERM_STRINGS(30)
#define TINFO_ENTER_INSERT_MODE         CUR_TERM_STRINGS(31)
#define TINFO_ENTER_SECURE_MODE         CUR_TERM_STRINGS(32)
#define TINFO_ENTER_PROTECTED_MODE      CUR_TERM_STRINGS(33)
#define TINFO_ENTER_REVERSE_MODE        CUR_TERM_STRINGS(34)
#define TINFO_ENTER_STANDOUT_MODE       CUR_TERM_STRINGS(35)
#define TINFO_ENTER_UNDERLINE_MODE      CUR_TERM_STRINGS(36)
#define TINFO_ERASE_CHARS               CUR_TERM_STRINGS(37)
#define TINFO_EXIT_ALT_CHARSET_MODE     CUR_TERM_STRINGS(38)
#define TINFO_EXIT_ATTRIBUTE_MODE       CUR_TERM_STRINGS(39)
#define TINFO_EXIT_CA_MODE              CUR_TERM_STRINGS(40)
#define TINFO_EXIT_DELETE_MODE          CUR_TERM_STRINGS(41)
#define TINFO_EXIT_INSERT_MODE          CUR_TERM_STRINGS(42)
#define TINFO_EXIT_STANDOUT_MODE        CUR_TERM_STRINGS(43)
#define TINFO_EXIT_UNDERLINE_MODE       CUR_TERM_STRINGS(44)
#define TINFO_FLASH_SCREEN              CUR_TERM_STRINGS(45)
#define TINFO_FORM_FEED                 CUR_TERM_STRINGS(46)
#define TINFO_FROM_STATUS_LINE          CUR_TERM_STRINGS(47)
#define TINFO_INIT_1STRING              CUR_TERM_STRINGS(48)
#define TINFO_INIT_2STRING              CUR_TERM_STRINGS(49)
#define TINFO_INIT_3STRING              CUR_TERM_STRINGS(50)
#define TINFO_INIT_FILE                 CUR_TERM_STRINGS(51)
#define TINFO_INSERT_CHARACTER          CUR_TERM_STRINGS(52)
#define TINFO_INSERT_LINE               CUR_TERM_STRINGS(53)
#define TINFO_INSERT_PADDING            CUR_TERM_STRINGS(54)
#define TINFO_KEY_BACKSPACE             CUR_TERM_STRINGS(55)
#define TINFO_KEY_CATAB                 CUR_TERM_STRINGS(56)
#define TINFO_KEY_CLEAR                 CUR_TERM_STRINGS(57)
#define TINFO_KEY_CTAB                  CUR_TERM_STRINGS(58)
#define TINFO_KEY_DC                    CUR_TERM_STRINGS(59)
#define TINFO_KEY_DL                    CUR_TERM_STRINGS(60)
#define TINFO_KEY_DOWN                  CUR_TERM_STRINGS(61)
#define TINFO_KEY_EIC                   CUR_TERM_STRINGS(62)
#define TINFO_KEY_EOL                   CUR_TERM_STRINGS(63)
#define TINFO_KEY_EOS                   CUR_TERM_STRINGS(64)
#define TINFO_KEY_F0                    CUR_TERM_STRINGS(65)
#define TINFO_KEY_F1                    CUR_TERM_STRINGS(66)
#define TINFO_KEY_F10                   CUR_TERM_STRINGS(67)
#define TINFO_KEY_F2                    CUR_TERM_STRINGS(68)
#define TINFO_KEY_F3                    CUR_TERM_STRINGS(69)
#define TINFO_KEY_F4                    CUR_TERM_STRINGS(70)
#define TINFO_KEY_F5                    CUR_TERM_STRINGS(71)
#define TINFO_KEY_F6                    CUR_TERM_STRINGS(72)
#define TINFO_KEY_F7                    CUR_TERM_STRINGS(73)
#define TINFO_KEY_F8                    CUR_TERM_STRINGS(74)
#define TINFO_KEY_F9                    CUR_TERM_STRINGS(75)
#define TINFO_KEY_HOME                  CUR_TERM_STRINGS(76)
#define TINFO_KEY_IC                    CUR_TERM_STRINGS(77)
#define TINFO_KEY_IL                    CUR_TERM_STRINGS(78)
#define TINFO_KEY_LEFT                  CUR_TERM_STRINGS(79)
#define TINFO_KEY_LL                    CUR_TERM_STRINGS(80)
#define TINFO_KEY_NPAGE                 CUR_TERM_STRINGS(81)
#define TINFO_KEY_PPAGE                 CUR_TERM_STRINGS(82)
#define TINFO_KEY_RIGHT                 CUR_TERM_STRINGS(83)
#define TINFO_KEY_SF                    CUR_TERM_STRINGS(84)
#define TINFO_KEY_SR                    CUR_TERM_STRINGS(85)
#define TINFO_KEY_STAB                  CUR_TERM_STRINGS(86)
#define TINFO_KEY_UP                    CUR_TERM_STRINGS(87)
#define TINFO_KEYPAD_LOCAL              CUR_TERM_STRINGS(88)
#define TINFO_KEYPAD_XMIT               CUR_TERM_STRINGS(89)
#define TINFO_LAB_F0                    CUR_TERM_STRINGS(90)
#define TINFO_LAB_F1                    CUR_TERM_STRINGS(91)
#define TINFO_LAB_F10                   CUR_TERM_STRINGS(92)
#define TINFO_LAB_F2                    CUR_TERM_STRINGS(93)
#define TINFO_LAB_F3                    CUR_TERM_STRINGS(94)
#define TINFO_LAB_F4                    CUR_TERM_STRINGS(95)
#define TINFO_LAB_F5                    CUR_TERM_STRINGS(96)
#define TINFO_LAB_F6                    CUR_TERM_STRINGS(97)
#define TINFO_LAB_F7                    CUR_TERM_STRINGS(98)
#define TINFO_LAB_F8                    CUR_TERM_STRINGS(99)
#define TINFO_LAB_F9                    CUR_TERM_STRINGS(100)
#define TINFO_META_OFF                  CUR_TERM_STRINGS(101)
#define TINFO_META_ON                   CUR_TERM_STRINGS(102)
#define TINFO_NEWLINE                   CUR_TERM_STRINGS(103)
#define TINFO_PAD_CHAR                  CUR_TERM_STRINGS(104)
#define TINFO_PARM_DCH                  CUR_TERM_STRINGS(105)
#define TINFO_PARM_DELETE_LINE          CUR_TERM_STRINGS(106)
#define TINFO_PARM_DOWN_CURSOR          CUR_TERM_STRINGS(107)
#define TINFO_PARM_ICH                  CUR_TERM_STRINGS(108)
#define TINFO_PARM_INDEX                CUR_TERM_STRINGS(109)
#define TINFO_PARM_INSERT_LINE          CUR_TERM_STRINGS(110)
#define TINFO_PARM_LEFT_CURSOR          CUR_TERM_STRINGS(111)
#define TINFO_PARM_RIGHT_CURSOR         CUR_TERM_STRINGS(112)
#define TINFO_PARM_RINDEX               CUR_TERM_STRINGS(113)
#define TINFO_PARM_UP_CURSOR            CUR_TERM_STRINGS(114)
#define TINFO_PKEY_KEY                  CUR_TERM_STRINGS(115)
#define TINFO_PKEY_LOCAL                CUR_TERM_STRINGS(116)
#define TINFO_PKEY_XMIT                 CUR_TERM_STRINGS(117)
#define TINFO_PRINT_SCREEN              CUR_TERM_STRINGS(118)
#define TINFO_PRTR_OFF                  CUR_TERM_STRINGS(119)
#define TINFO_PRTR_ON                   CUR_TERM_STRINGS(120)
#define TINFO_REPEAT_CHAR               CUR_TERM_STRINGS(121)
#define TINFO_RESET_1STRING             CUR_TERM_STRINGS(122)
#define TINFO_RESET_2STRING             CUR_TERM_STRINGS(123)
#define TINFO_RESET_3STRING             CUR_TERM_STRINGS(124)
#define TINFO_RESET_FILE                CUR_TERM_STRINGS(125)
#define TINFO_RESTORE_CURSOR            CUR_TERM_STRINGS(126)
#define TINFO_ROW_ADDRESS               CUR_TERM_STRINGS(127)
#define TINFO_SAVE_CURSOR               CUR_TERM_STRINGS(128)
#define TINFO_SCROLL_FORWARD            CUR_TERM_STRINGS(129)
#define TINFO_SCROLL_REVERSE            CUR_TERM_STRINGS(130)
#define TINFO_SET_ATTRIBUTES            CUR_TERM_STRINGS(131)
#define TINFO_SET_TAB                   CUR_TERM_STRINGS(132)
#define TINFO_SET_WINDOW                CUR_TERM_STRINGS(133)
#define TINFO_TAB                       CUR_TERM_STRINGS(134)
#define TINFO_TO_STATUS_LINE            CUR_TERM_STRINGS(135)
#define TINFO_UNDERLINE_CHAR            CUR_TERM_STRINGS(136)
#define TINFO_UP_HALF_LINE              CUR_TERM_STRINGS(137)
#define TINFO_INIT_PROG                 CUR_TERM_STRINGS(138)
#define TINFO_KEY_A1                    CUR_TERM_STRINGS(139)
#define TINFO_KEY_A3                    CUR_TERM_STRINGS(140)
#define TINFO_KEY_B2                    CUR_TERM_STRINGS(141)
#define TINFO_KEY_C1                    CUR_TERM_STRINGS(142)
#define TINFO_KEY_C3                    CUR_TERM_STRINGS(143)
#define TINFO_PRTR_NON                  CUR_TERM_STRINGS(144)
#define TINFO_CHAR_PADDING              CUR_TERM_STRINGS(145)
#define TINFO_ACS_CHARS                 CUR_TERM_STRINGS(146)
#define TINFO_PLAB_NORM                 CUR_TERM_STRINGS(147)
#define TINFO_KEY_BTAB                  CUR_TERM_STRINGS(148)
#define TINFO_ENTER_XON_MODE            CUR_TERM_STRINGS(149)
#define TINFO_EXIT_XON_MODE             CUR_TERM_STRINGS(150)
#define TINFO_ENTER_AM_MODE             CUR_TERM_STRINGS(151)
#define TINFO_EXIT_AM_MODE              CUR_TERM_STRINGS(152)
#define TINFO_XON_CHARACTER             CUR_TERM_STRINGS(153)
#define TINFO_XOFF_CHARACTER            CUR_TERM_STRINGS(154)
#define TINFO_ENA_ACS                   CUR_TERM_STRINGS(155)
#define TINFO_LABEL_ON                  CUR_TERM_STRINGS(156)
#define TINFO_LABEL_OFF                 CUR_TERM_STRINGS(157)
#define TINFO_KEY_BEG                   CUR_TERM_STRINGS(158)
#define TINFO_KEY_CANCEL                CUR_TERM_STRINGS(159)
#define TINFO_KEY_CLOSE                 CUR_TERM_STRINGS(160)
#define TINFO_KEY_COMMAND               CUR_TERM_STRINGS(161)
#define TINFO_KEY_COPY                  CUR_TERM_STRINGS(162)
#define TINFO_KEY_CREATE                CUR_TERM_STRINGS(163)
#define TINFO_KEY_END                   CUR_TERM_STRINGS(164)
#define TINFO_KEY_ENTER                 CUR_TERM_STRINGS(165)
#define TINFO_KEY_EXIT                  CUR_TERM_STRINGS(166)
#define TINFO_KEY_FIND                  CUR_TERM_STRINGS(167)
#define TINFO_KEY_HELP                  CUR_TERM_STRINGS(168)
#define TINFO_KEY_MARK                  CUR_TERM_STRINGS(169)
#define TINFO_KEY_MESSAGE               CUR_TERM_STRINGS(170)
#define TINFO_KEY_MOVE                  CUR_TERM_STRINGS(171)
#define TINFO_KEY_NEXT                  CUR_TERM_STRINGS(172)
#define TINFO_KEY_OPEN                  CUR_TERM_STRINGS(173)
#define TINFO_KEY_OPTIONS               CUR_TERM_STRINGS(174)
#define TINFO_KEY_PREVIOUS              CUR_TERM_STRINGS(175)
#define TINFO_KEY_PRINT                 CUR_TERM_STRINGS(176)
#define TINFO_KEY_REDO                  CUR_TERM_STRINGS(177)
#define TINFO_KEY_REFERENCE             CUR_TERM_STRINGS(178)
#define TINFO_KEY_REFRESH               CUR_TERM_STRINGS(179)
#define TINFO_KEY_REPLACE               CUR_TERM_STRINGS(180)
#define TINFO_KEY_RESTART               CUR_TERM_STRINGS(181)
#define TINFO_KEY_RESUME                CUR_TERM_STRINGS(182)
#define TINFO_KEY_SAVE                  CUR_TERM_STRINGS(183)
#define TINFO_KEY_SUSPEND               CUR_TERM_STRINGS(184)
#define TINFO_KEY_UNDO                  CUR_TERM_STRINGS(185)
#define TINFO_KEY_SBEG                  CUR_TERM_STRINGS(186)
#define TINFO_KEY_SCANCEL               CUR_TERM_STRINGS(187)
#define TINFO_KEY_SCOMMAND              CUR_TERM_STRINGS(188)
#define TINFO_KEY_SCOPY                 CUR_TERM_STRINGS(189)
#define TINFO_KEY_SCREATE               CUR_TERM_STRINGS(190)
#define TINFO_KEY_SDC                   CUR_TERM_STRINGS(191)
#define TINFO_KEY_SDL                   CUR_TERM_STRINGS(192)
#define TINFO_KEY_SELECT                CUR_TERM_STRINGS(193)
#define TINFO_KEY_SEND                  CUR_TERM_STRINGS(194)
#define TINFO_KEY_SEOL                  CUR_TERM_STRINGS(195)
#define TINFO_KEY_SEXIT                 CUR_TERM_STRINGS(196)
#define TINFO_KEY_SFIND                 CUR_TERM_STRINGS(197)
#define TINFO_KEY_SHELP                 CUR_TERM_STRINGS(198)
#define TINFO_KEY_SHOME                 CUR_TERM_STRINGS(199)
#define TINFO_KEY_SIC                   CUR_TERM_STRINGS(200)
#define TINFO_KEY_SLEFT                 CUR_TERM_STRINGS(201)
#define TINFO_KEY_SMESSAGE              CUR_TERM_STRINGS(202)
#define TINFO_KEY_SMOVE                 CUR_TERM_STRINGS(203)
#define TINFO_KEY_SNEXT                 CUR_TERM_STRINGS(204)
#define TINFO_KEY_SOPTIONS              CUR_TERM_STRINGS(205)
#define TINFO_KEY_SPREVIOUS             CUR_TERM_STRINGS(206)
#define TINFO_KEY_SPRINT                CUR_TERM_STRINGS(207)
#define TINFO_KEY_SREDO                 CUR_TERM_STRINGS(208)
#define TINFO_KEY_SREPLACE              CUR_TERM_STRINGS(209)
#define TINFO_KEY_SRIGHT                CUR_TERM_STRINGS(210)
#define TINFO_KEY_SRSUME                CUR_TERM_STRINGS(211)
#define TINFO_KEY_SSAVE                 CUR_TERM_STRINGS(212)
#define TINFO_KEY_SSUSPEND              CUR_TERM_STRINGS(213)
#define TINFO_KEY_SUNDO                 CUR_TERM_STRINGS(214)
#define TINFO_REQ_FOR_INPUT             CUR_TERM_STRINGS(215)
#define TINFO_KEY_F11                   CUR_TERM_STRINGS(216)
#define TINFO_KEY_F12                   CUR_TERM_STRINGS(217)
#define TINFO_KEY_F13                   CUR_TERM_STRINGS(218)
#define TINFO_KEY_F14                   CUR_TERM_STRINGS(219)
#define TINFO_KEY_F15                   CUR_TERM_STRINGS(220)
#define TINFO_KEY_F16                   CUR_TERM_STRINGS(221)
#define TINFO_KEY_F17                   CUR_TERM_STRINGS(222)
#define TINFO_KEY_F18                   CUR_TERM_STRINGS(223)
#define TINFO_KEY_F19                   CUR_TERM_STRINGS(224)
#define TINFO_KEY_F20                   CUR_TERM_STRINGS(225)
#define TINFO_KEY_F21                   CUR_TERM_STRINGS(226)
#define TINFO_KEY_F22                   CUR_TERM_STRINGS(227)
#define TINFO_KEY_F23                   CUR_TERM_STRINGS(228)
#define TINFO_KEY_F24                   CUR_TERM_STRINGS(229)
#define TINFO_KEY_F25                   CUR_TERM_STRINGS(230)
#define TINFO_KEY_F26                   CUR_TERM_STRINGS(231)
#define TINFO_KEY_F27                   CUR_TERM_STRINGS(232)
#define TINFO_KEY_F28                   CUR_TERM_STRINGS(233)
#define TINFO_KEY_F29                   CUR_TERM_STRINGS(234)
#define TINFO_KEY_F30                   CUR_TERM_STRINGS(235)
#define TINFO_KEY_F31                   CUR_TERM_STRINGS(236)
#define TINFO_KEY_F32                   CUR_TERM_STRINGS(237)
#define TINFO_KEY_F33                   CUR_TERM_STRINGS(238)
#define TINFO_KEY_F34                   CUR_TERM_STRINGS(239)
#define TINFO_KEY_F35                   CUR_TERM_STRINGS(240)
#define TINFO_KEY_F36                   CUR_TERM_STRINGS(241)
#define TINFO_KEY_F37                   CUR_TERM_STRINGS(242)
#define TINFO_KEY_F38                   CUR_TERM_STRINGS(243)
#define TINFO_KEY_F39                   CUR_TERM_STRINGS(244)
#define TINFO_KEY_F40                   CUR_TERM_STRINGS(245)
#define TINFO_KEY_F41                   CUR_TERM_STRINGS(246)
#define TINFO_KEY_F42                   CUR_TERM_STRINGS(247)
#define TINFO_KEY_F43                   CUR_TERM_STRINGS(248)
#define TINFO_KEY_F44                   CUR_TERM_STRINGS(249)
#define TINFO_KEY_F45                   CUR_TERM_STRINGS(250)
#define TINFO_KEY_F46                   CUR_TERM_STRINGS(251)
#define TINFO_KEY_F47                   CUR_TERM_STRINGS(252)
#define TINFO_KEY_F48                   CUR_TERM_STRINGS(253)
#define TINFO_KEY_F49                   CUR_TERM_STRINGS(254)
#define TINFO_KEY_F50                   CUR_TERM_STRINGS(255)
#define TINFO_KEY_F51                   CUR_TERM_STRINGS(256)
#define TINFO_KEY_F52                   CUR_TERM_STRINGS(257)
#define TINFO_KEY_F53                   CUR_TERM_STRINGS(258)
#define TINFO_KEY_F54                   CUR_TERM_STRINGS(259)
#define TINFO_KEY_F55                   CUR_TERM_STRINGS(260)
#define TINFO_KEY_F56                   CUR_TERM_STRINGS(261)
#define TINFO_KEY_F57                   CUR_TERM_STRINGS(262)
#define TINFO_KEY_F58                   CUR_TERM_STRINGS(263)
#define TINFO_KEY_F59                   CUR_TERM_STRINGS(264)
#define TINFO_KEY_F60                   CUR_TERM_STRINGS(265)
#define TINFO_KEY_F61                   CUR_TERM_STRINGS(266)
#define TINFO_KEY_F62                   CUR_TERM_STRINGS(267)
#define TINFO_KEY_F63                   CUR_TERM_STRINGS(268)
#define TINFO_CLR_BOL                   CUR_TERM_STRINGS(269)
#define TINFO_CLEAR_MARGINS             CUR_TERM_STRINGS(270)
#define TINFO_SET_LEFT_MARGIN           CUR_TERM_STRINGS(271)
#define TINFO_SET_RIGHT_MARGIN          CUR_TERM_STRINGS(272)
#define TINFO_LABEL_FORMAT              CUR_TERM_STRINGS(273)
#define TINFO_SET_CLOCK                 CUR_TERM_STRINGS(274)
#define TINFO_DISPLAY_CLOCK             CUR_TERM_STRINGS(275)
#define TINFO_REMOVE_CLOCK              CUR_TERM_STRINGS(276)
#define TINFO_CREATE_WINDOW             CUR_TERM_STRINGS(277)
#define TINFO_GOTO_WINDOW               CUR_TERM_STRINGS(278)
#define TINFO_HANGUP                    CUR_TERM_STRINGS(279)
#define TINFO_DIAL_PHONE                CUR_TERM_STRINGS(280)
#define TINFO_QUICK_DIAL                CUR_TERM_STRINGS(281)
#define TINFO_TONE                      CUR_TERM_STRINGS(282)
#define TINFO_PULSE                     CUR_TERM_STRINGS(283)
#define TINFO_FLASH_HOOK                CUR_TERM_STRINGS(284)
#define TINFO_FIXED_PAUSE               CUR_TERM_STRINGS(285)
#define TINFO_WAIT_TONE                 CUR_TERM_STRINGS(286)
#define TINFO_USER0                     CUR_TERM_STRINGS(287)
#define TINFO_USER1                     CUR_TERM_STRINGS(288)
#define TINFO_USER2                     CUR_TERM_STRINGS(289)
#define TINFO_USER3                     CUR_TERM_STRINGS(290)
#define TINFO_USER4                     CUR_TERM_STRINGS(291)
#define TINFO_USER5                     CUR_TERM_STRINGS(292)
#define TINFO_USER6                     CUR_TERM_STRINGS(293)
#define TINFO_USER7                     CUR_TERM_STRINGS(294)
#define TINFO_USER8                     CUR_TERM_STRINGS(295)
#define TINFO_USER9                     CUR_TERM_STRINGS(296)
#define TINFO_ORIG_PAIR                 CUR_TERM_STRINGS(297)
#define TINFO_ORIG_COLORS               CUR_TERM_STRINGS(298)
#define TINFO_INITIALIZE_COLOR          CUR_TERM_STRINGS(299)
#define TINFO_INITIALIZE_PAIR           CUR_TERM_STRINGS(300)
#define TINFO_SET_COLOR_PAIR            CUR_TERM_STRINGS(301)
#define TINFO_SET_FOREGROUND            CUR_TERM_STRINGS(302)
#define TINFO_SET_BACKGROUND            CUR_TERM_STRINGS(303)
#define TINFO_CHANGE_CHAR_PITCH         CUR_TERM_STRINGS(304)
#define TINFO_CHANGE_LINE_PITCH         CUR_TERM_STRINGS(305)
#define TINFO_CHANGE_RES_HORZ           CUR_TERM_STRINGS(306)
#define TINFO_CHANGE_RES_VERT           CUR_TERM_STRINGS(307)
#define TINFO_DEFINE_CHAR               CUR_TERM_STRINGS(308)
#define TINFO_ENTER_DOUBLEWIDE_MODE     CUR_TERM_STRINGS(309)
#define TINFO_ENTER_DRAFT_QUALITY       CUR_TERM_STRINGS(310)
#define TINFO_ENTER_ITALICS_MODE        CUR_TERM_STRINGS(311)
#define TINFO_ENTER_LEFTWARD_MODE       CUR_TERM_STRINGS(312)
#define TINFO_ENTER_MICRO_MODE          CUR_TERM_STRINGS(313)
#define TINFO_ENTER_NEAR_LETTER_QUALITy CUR_TERM_STRINGS(314)
#define TINFO_ENTER_NORMAL_QUALITY      CUR_TERM_STRINGS(315)
#define TINFO_ENTER_SHADOW_MODE         CUR_TERM_STRINGS(316)
#define TINFO_ENTER_SUBSCRIPT_MODE      CUR_TERM_STRINGS(317)
#define TINFO_ENTER_SUPERSCRIPT_MODE    CUR_TERM_STRINGS(318)
#define TINFO_ENTER_UPWARD_MODE         CUR_TERM_STRINGS(319)
#define TINFO_EXIT_DOUBLEWIDE_MODE      CUR_TERM_STRINGS(320)
#define TINFO_EXIT_ITALICS_MODE         CUR_TERM_STRINGS(321)
#define TINFO_EXIT_LEFTWARD_MODE        CUR_TERM_STRINGS(322)
#define TINFO_EXIT_MICRO_MODE           CUR_TERM_STRINGS(323)
#define TINFO_EXIT_SHADOW_MODE          CUR_TERM_STRINGS(324)
#define TINFO_EXIT_SUBSCRIPT_MODE       CUR_TERM_STRINGS(325)
#define TINFO_EXIT_SUPERSCRIPT_MODE     CUR_TERM_STRINGS(326)
#define TINFO_EXIT_UPWARD_MODE          CUR_TERM_STRINGS(327)
#define TINFO_MICRO_COLUMN_ADDRESS      CUR_TERM_STRINGS(328)
#define TINFO_MICRO_DOWN                CUR_TERM_STRINGS(329)
#define TINFO_MICRO_LEFT                CUR_TERM_STRINGS(330)
#define TINFO_MICRO_RIGHT               CUR_TERM_STRINGS(331)
#define TINFO_MICRO_ROW_ADDRESS         CUR_TERM_STRINGS(332)
#define TINFO_MICRO_UP                  CUR_TERM_STRINGS(333)
#define TINFO_ORDER_OF_PINS             CUR_TERM_STRINGS(334)
#define TINFO_PARM_DOWN_MICRO           CUR_TERM_STRINGS(335)
#define TINFO_PARM_LEFT_MICRO           CUR_TERM_STRINGS(336)
#define TINFO_PARM_RIGHT_MICRO          CUR_TERM_STRINGS(337)
#define TINFO_PARM_UP_MICRO             CUR_TERM_STRINGS(338)
#define TINFO_SELECT_CHAR_SET           CUR_TERM_STRINGS(339)
#define TINFO_SET_BOTTOM_MARGIN         CUR_TERM_STRINGS(340)
#define TINFO_SET_BOTTOM_MARGIN_PARM    CUR_TERM_STRINGS(341)
#define TINFO_SET_LEFT_MARGIN_PARM      CUR_TERM_STRINGS(342)
#define TINFO_SET_RIGHT_MARGIN_PARM     CUR_TERM_STRINGS(343)
#define TINFO_SET_TOP_MARGIN            CUR_TERM_STRINGS(344)
#define TINFO_SET_TOP_MARGIN_PARM       CUR_TERM_STRINGS(345)
#define TINFO_START_BIT_IMAGE           CUR_TERM_STRINGS(346)
#define TINFO_START_CHAR_SET_DEF        CUR_TERM_STRINGS(347)
#define TINFO_STOP_BIT_IMAGE            CUR_TERM_STRINGS(348)
#define TINFO_STOP_CHAR_SET_DEF         CUR_TERM_STRINGS(349)
#define TINFO_SUBSCRIPT_CHARACTERS      CUR_TERM_STRINGS(350)
#define TINFO_SUPERSCRIPT_CHARACTERS    CUR_TERM_STRINGS(351)
#define TINFO_THESE_CAUSE_CR            CUR_TERM_STRINGS(352)
#define TINFO_ZERO_MOTION               CUR_TERM_STRINGS(353)
#define TINFO_CHAR_SET_NAMES            CUR_TERM_STRINGS(354)
#define TINFO_KEY_MOUSE                 CUR_TERM_STRINGS(355)
#define TINFO_MOUSE_INFO                CUR_TERM_STRINGS(356)
#define TINFO_REQ_MOUSE_POS             CUR_TERM_STRINGS(357)
#define TINFO_GET_MOUSE                 CUR_TERM_STRINGS(358)
#define TINFO_SET_A_FOREGROUND          CUR_TERM_STRINGS(359)
#define TINFO_SET_A_BACKGROUND          CUR_TERM_STRINGS(360)
#define TINFO_PKEY_PLAB                 CUR_TERM_STRINGS(361)
#define TINFO_DEVICE_TYPE               CUR_TERM_STRINGS(362)
#define TINFO_CODE_SET_INIT             CUR_TERM_STRINGS(363)
#define TINFO_SET0_DES_SEQ              CUR_TERM_STRINGS(364)
#define TINFO_SET1_DES_SEQ              CUR_TERM_STRINGS(365)
#define TINFO_SET2_DES_SEQ              CUR_TERM_STRINGS(366)
#define TINFO_SET3_DES_SEQ              CUR_TERM_STRINGS(367)
#define TINFO_SET_LR_MARGIN             CUR_TERM_STRINGS(368)
#define TINFO_SET_TB_MARGIN             CUR_TERM_STRINGS(369)
#define TINFO_BIT_IMAGE_REPEAT          CUR_TERM_STRINGS(370)
#define TINFO_BIT_IMAGE_NEWLINE         CUR_TERM_STRINGS(371)
#define TINFO_BIT_IMAGE_CARRIAGE_RETURn CUR_TERM_STRINGS(372)
#define TINFO_COLOR_NAMES               CUR_TERM_STRINGS(373)
#define TINFO_DEFINE_BIT_IMAGE_REGION   CUR_TERM_STRINGS(374)
#define TINFO_END_BIT_IMAGE_REGION      CUR_TERM_STRINGS(375)
#define TINFO_SET_COLOR_BAND            CUR_TERM_STRINGS(376)
#define TINFO_SET_PAGE_LENGTH           CUR_TERM_STRINGS(377)
#define TINFO_DISPLAY_PC_CHAR           CUR_TERM_STRINGS(378)
#define TINFO_ENTER_PC_CHARSET_MODE     CUR_TERM_STRINGS(379)
#define TINFO_EXIT_PC_CHARSET_MODE      CUR_TERM_STRINGS(380)
#define TINFO_ENTER_SCANCODE_MODE       CUR_TERM_STRINGS(381)
#define TINFO_EXIT_SCANCODE_MODE        CUR_TERM_STRINGS(382)
#define TINFO_PC_TINFO_OPTIONS          CUR_TERM_STRINGS(383)
#define TINFO_SCANCODE_ESCAPE           CUR_TERM_STRINGS(384)
#define TINFO_ALT_SCANCODE_ESC          CUR_TERM_STRINGS(385)
#define TINFO_ENTER_HORIZONTAL_HL_MODE  CUR_TERM_STRINGS(386)
#define TINFO_ENTER_LEFT_HL_MODE        CUR_TERM_STRINGS(387)
#define TINFO_ENTER_LOW_HL_MODE         CUR_TERM_STRINGS(388)
#define TINFO_ENTER_RIGHT_HL_MODE       CUR_TERM_STRINGS(389)
#define TINFO_ENTER_TOP_HL_MODE         CUR_TERM_STRINGS(390)
#define TINFO_ENTER_VERTICAL_HL_MODE    CUR_TERM_STRINGS(391)
#define TINFO_SET_A_ATTRIBUTES          CUR_TERM_STRINGS(392)
#define TINFO_SET_PGLEN_INCH            CUR_TERM_STRINGS(393)

extern int setupterm(char *term, int fildes, int *errret);
extern TERMINAL *set_curterm(TERMINAL *nterm);
extern int del_curterm(TERMINAL *oterm);

#define setterm(term) setupterm(term, 1, NULL)

extern char *tparm(char *str, long p1, long p2, long p3, long p4, long p5,
    long p6, long p7, long p8, long p9);

extern int tputs(char *str, int affcnt, int (*putc)(int));
extern int putp(char *str);

#endif /* COMPAT_POSIX_TERM_H_ */
