#!/bin/bash

TEMPDIR="out"
UNCRUSTIFY_BASE=$(dirname "$0")

verbose="true"

while getopts :qh opt; do
	case $opt in
	q)
		verbose=""
		;;
	h)
		echo "run_uncrustify.sh [options] [<git diff args>]"
		echo ""
		echo "Check codestyle of a change. The change is generated by passing"
		echo "<git diff args> to git diff. Like:"
		echo ""
		echo "run_uncrustify.sh                      # check working tree"
		echo "run_uncrustify.sh --cached             # check index"
		echo "run_uncrustify.sh origin/master...HEAD # check current branch"
		echo ""
		echo "Options:"
		echo "-q : show only file names with codestyle issues, suppress"
		echo "     printing suggested fix"
		echo "-h : shows this message"
		exit 1
		;;
	?)
		break
		;;

	esac
done

# the rest will go to git diff directly
shift $(($OPTIND - 1))
diffargs=( "$@" )

tmpundiff=$TEMPDIR/diff
mkdir -p $tmpundiff

git diff --no-prefix "${diffargs[@]}" -- 'src/*.[ch]' | awk -v outpref="$tmpundiff" -f "$UNCRUSTIFY_BASE"/undiff.awk

result=0

if [ $verbose ]; then
	fixdiffarg="--color" # output is not suitable for parsing anyway
else
	fixdiffarg="--quiet"
fi

for f in $(cd $tmpundiff; find . -type f); do
	from=$tmpundiff/$f
	to=$TEMPDIR/fixed/$f
	mkdir -p "$(dirname "$to")"

	uncrustify --frag -c "$UNCRUSTIFY_BASE"/uncrustify_cfg.ini -f "$from" > "$to" 2>/dev/null
	if [ ! -s "$to" ]; then
		continue
	fi


	git diff $fixdiffarg --no-index -- "$from" "$to" | tail -n +3
	if [ "${PIPESTATUS[0]}" != 0 ]; then
		[ $verbose ] || echo "Codestyle issue: $f"
		result=1
	fi
done

rm -rf $TEMPDIR
exit $result
