// Created on: 2011-08-01
// Created by: Sergey ZERCHANINOV
// Copyright (c) 2011-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <OpenGl_CappingAlgo.hxx>
#include <OpenGl_GlCore11.hxx>
#include <OpenGl_ClippingIterator.hxx>
#include <OpenGl_ShaderManager.hxx>
#include <OpenGl_ShaderProgram.hxx>
#include <OpenGl_StructureShadow.hxx>
#include <OpenGl_Vec.hxx>
#include <OpenGl_View.hxx>
#include <OpenGl_Workspace.hxx>

IMPLEMENT_STANDARD_RTTIEXT(OpenGl_Structure, Graphic3d_CStructure)

//=================================================================================================

void OpenGl_Structure::renderBoundingBox(const occ::handle<OpenGl_Workspace>& theWorkspace) const
{
  if (!myBndBox.IsValid())
  {
    return;
  }

  const occ::handle<OpenGl_Context>&   aCtx = theWorkspace->GetGlContext();
  const occ::handle<OpenGl_TextureSet> aPrevTexture =
    aCtx->BindTextures(occ::handle<OpenGl_TextureSet>(), occ::handle<OpenGl_ShaderProgram>());
  const Graphic3d_ZLayerSettings& aLayer = myGraphicDriver->ZLayerSettings(myZLayer);
  const NCollection_Vec3<double>  aMoveVec =
    myTrsfPers.IsNull() && !aLayer.OriginTransformation().IsNull()
       ? -NCollection_Vec3<double>(aLayer.Origin().X(), aLayer.Origin().Y(), aLayer.Origin().Z())
       : NCollection_Vec3<double>(0.0, 0.0, 0.0);
  if (aCtx->core20fwd != nullptr && aCtx->ShaderManager()->BindBoundBoxProgram())
  {
    const NCollection_Vec3<double> aCenter = myBndBox.Center() + aMoveVec;
    const NCollection_Vec3<double> aSize   = myBndBox.Size();
    aCtx->ActiveProgram()->SetUniform(
      aCtx,
      "occBBoxCenter",
      NCollection_Vec3<float>((float)aCenter.x(), (float)aCenter.y(), (float)aCenter.z()));
    aCtx->ActiveProgram()->SetUniform(
      aCtx,
      "occBBoxSize",
      NCollection_Vec3<float>((float)aSize.x(), (float)aSize.y(), (float)aSize.z()));
    aCtx->SetColor4fv(theWorkspace->InteriorColor());

    const occ::handle<OpenGl_VertexBuffer>& aBoundBoxVertBuffer =
      aCtx->ShaderManager()->BoundBoxVertBuffer();
    aBoundBoxVertBuffer->BindAttribute(aCtx, Graphic3d_TOA_POS);
    aCtx->core20fwd->glDrawArrays(GL_LINES, 0, aBoundBoxVertBuffer->GetElemsNb());
    aBoundBoxVertBuffer->UnbindAttribute(aCtx, Graphic3d_TOA_POS);
  }
  else if (aCtx->core11ffp != nullptr)
  {
    const NCollection_Vec3<double> aMind = myBndBox.CornerMin() + aMoveVec;
    const NCollection_Vec3<double> aMaxd = myBndBox.CornerMax() + aMoveVec;
    const NCollection_Vec3<float>  aMin((float)aMind.x(), (float)aMind.y(), (float)aMind.z());
    const NCollection_Vec3<float>  aMax((float)aMaxd.x(), (float)aMaxd.y(), (float)aMaxd.z());
    const NCollection_Vec3<float>  aVerts[16] = {
      NCollection_Vec3<float>(aMin.x(), aMin.y(), aMin.z()),
      NCollection_Vec3<float>(aMin.x(), aMin.y(), aMax.z()),
      NCollection_Vec3<float>(aMin.x(), aMax.y(), aMax.z()),
      NCollection_Vec3<float>(aMin.x(), aMax.y(), aMin.z()),
      NCollection_Vec3<float>(aMin.x(), aMin.y(), aMin.z()),
      NCollection_Vec3<float>(aMax.x(), aMin.y(), aMin.z()),
      NCollection_Vec3<float>(aMax.x(), aMin.y(), aMax.z()),
      NCollection_Vec3<float>(aMax.x(), aMax.y(), aMax.z()),
      NCollection_Vec3<float>(aMax.x(), aMax.y(), aMin.z()),
      NCollection_Vec3<float>(aMax.x(), aMin.y(), aMin.z()),
      NCollection_Vec3<float>(aMax.x(), aMax.y(), aMin.z()),
      NCollection_Vec3<float>(aMin.x(), aMax.y(), aMin.z()),
      NCollection_Vec3<float>(aMin.x(), aMax.y(), aMax.z()),
      NCollection_Vec3<float>(aMax.x(), aMax.y(), aMax.z()),
      NCollection_Vec3<float>(aMax.x(), aMin.y(), aMax.z()),
      NCollection_Vec3<float>(aMin.x(), aMin.y(), aMax.z())};

    aCtx->ShaderManager()->BindLineProgram(occ::handle<OpenGl_TextureSet>(),
                                           Aspect_TOL_SOLID,
                                           Graphic3d_TypeOfShadingModel_Unlit,
                                           Graphic3d_AlphaMode_Opaque,
                                           false,
                                           occ::handle<OpenGl_ShaderProgram>());
    aCtx->SetColor4fv(theWorkspace->InteriorColor());
    aCtx->core11fwd->glDisable(GL_LIGHTING);
    aCtx->core11ffp->glEnableClientState(GL_VERTEX_ARRAY);
    aCtx->core11ffp->glVertexPointer(3, GL_FLOAT, 0, aVerts[0].GetData());
    aCtx->core11fwd->glDrawArrays(GL_LINE_STRIP, 0, 16);
    aCtx->core11ffp->glDisableClientState(GL_VERTEX_ARRAY);
  }
  aCtx->BindTextures(aPrevTexture, occ::handle<OpenGl_ShaderProgram>());
}

//=================================================================================================

OpenGl_Structure::OpenGl_Structure(const occ::handle<Graphic3d_StructureManager>& theManager)
    : Graphic3d_CStructure(theManager),
      myInstancedStructure(nullptr),
      myIsRaytracable(false),
      myModificationState(0),
      myIsMirrored(false)
{
  updateLayerTransformation();
}

//=================================================================================================

OpenGl_Structure::~OpenGl_Structure()
{
  Release(occ::handle<OpenGl_Context>());
}

//=================================================================================================

void OpenGl_Structure::SetZLayer(const Graphic3d_ZLayerId theLayerIndex)
{
  Graphic3d_CStructure::SetZLayer(theLayerIndex);
  updateLayerTransformation();
}

//=================================================================================================

void OpenGl_Structure::SetTransformation(const occ::handle<TopLoc_Datum3D>& theTrsf)
{
  myTrsf       = theTrsf;
  myIsMirrored = false;
  if (!myTrsf.IsNull())
  {
    // Determinant of transform matrix less then 0 means that mirror transform applied.
    const gp_Trsf& aTrsf = myTrsf->Transformation();
    const double   aDet =
      aTrsf.Value(1, 1)
        * (aTrsf.Value(2, 2) * aTrsf.Value(3, 3) - aTrsf.Value(3, 2) * aTrsf.Value(2, 3))
      - aTrsf.Value(1, 2)
          * (aTrsf.Value(2, 1) * aTrsf.Value(3, 3) - aTrsf.Value(3, 1) * aTrsf.Value(2, 3))
      + aTrsf.Value(1, 3)
          * (aTrsf.Value(2, 1) * aTrsf.Value(3, 2) - aTrsf.Value(3, 1) * aTrsf.Value(2, 2));
    myIsMirrored = aDet < 0.0;
  }

  updateLayerTransformation();
  if (IsRaytracable())
  {
    ++myModificationState;
  }
}

//=================================================================================================

void OpenGl_Structure::SetTransformPersistence(
  const occ::handle<Graphic3d_TransformPers>& theTrsfPers)
{
  if ((myTrsfPers.IsNull() || theTrsfPers.IsNull()) && myTrsfPers != theTrsfPers)
  {
    ++myModificationState;
  }
  myTrsfPers = theTrsfPers;
  updateLayerTransformation();
}

//=================================================================================================

void OpenGl_Structure::updateLayerTransformation()
{
  gp_Trsf aRenderTrsf;
  if (!myTrsf.IsNull())
  {
    aRenderTrsf = myTrsf->Trsf();
  }

  const Graphic3d_ZLayerSettings& aLayer = myGraphicDriver->ZLayerSettings(myZLayer);
  if (!aLayer.OriginTransformation().IsNull() && myTrsfPers.IsNull())
  {
    aRenderTrsf.SetTranslationPart(aRenderTrsf.TranslationPart() - aLayer.Origin());
  }
  aRenderTrsf.GetMat4(myRenderTrsf);
}

//=================================================================================================

void OpenGl_Structure::GraphicHighlight(
  const occ::handle<Graphic3d_PresentationAttributes>& theStyle)
{
  myHighlightStyle = theStyle;
  highlight        = 1;
}

//=================================================================================================

void OpenGl_Structure::GraphicUnhighlight()
{
  highlight = 0;
  myHighlightStyle.Nullify();
}

//=================================================================================================

void OpenGl_Structure::OnVisibilityChanged()
{
  if (IsRaytracable())
  {
    ++myModificationState;
  }
}

//=================================================================================================

bool OpenGl_Structure::IsRaytracable() const
{
  if (!myGroups.IsEmpty() && myIsRaytracable && myTrsfPers.IsNull())
  {
    return true;
  }

  return myInstancedStructure != nullptr && myInstancedStructure->IsRaytracable();
}

//=================================================================================================

void OpenGl_Structure::UpdateStateIfRaytracable(const bool toCheck) const
{
  myIsRaytracable = !toCheck;
  if (!myIsRaytracable)
  {
    for (OpenGl_Structure::GroupIterator anIter(myGroups); anIter.More(); anIter.Next())
    {
      if (anIter.Value()->IsRaytracable())
      {
        myIsRaytracable = true;
        break;
      }
    }
  }

  if (IsRaytracable())
  {
    ++myModificationState;
  }
}

//=================================================================================================

void OpenGl_Structure::Connect(Graphic3d_CStructure& theStructure)
{
  OpenGl_Structure* aStruct = static_cast<OpenGl_Structure*>(&theStructure);

  Standard_ASSERT_RAISE(myInstancedStructure == nullptr || myInstancedStructure == aStruct,
                        "Error! Instanced structure is already defined");

  myInstancedStructure = aStruct;

  if (aStruct->IsRaytracable())
  {
    UpdateStateIfRaytracable(false);
  }
}

//=================================================================================================

void OpenGl_Structure::Disconnect(Graphic3d_CStructure& theStructure)
{
  OpenGl_Structure* aStruct = static_cast<OpenGl_Structure*>(&theStructure);

  if (myInstancedStructure == aStruct)
  {
    myInstancedStructure = nullptr;

    if (aStruct->IsRaytracable())
    {
      UpdateStateIfRaytracable();
    }
  }
}

//=================================================================================================

occ::handle<Graphic3d_Group> OpenGl_Structure::NewGroup(
  const occ::handle<Graphic3d_Structure>& theStruct)
{
  occ::handle<OpenGl_Group> aGroup = new OpenGl_Group(theStruct);
  myGroups.Append(aGroup);
  return aGroup;
}

//=================================================================================================

void OpenGl_Structure::RemoveGroup(const occ::handle<Graphic3d_Group>& theGroup)
{
  if (theGroup.IsNull())
  {
    return;
  }

  for (NCollection_Sequence<occ::handle<Graphic3d_Group>>::Iterator aGroupIter(myGroups);
       aGroupIter.More();
       aGroupIter.Next())
  {
    // Check for the given group
    if (aGroupIter.Value() == theGroup)
    {
      const bool wasRaytracable = static_cast<const OpenGl_Group&>(*theGroup).IsRaytracable();

      theGroup->Clear(false);

      if (wasRaytracable)
      {
        UpdateStateIfRaytracable();
      }

      myGroups.Remove(aGroupIter);
      return;
    }
  }
}

//=================================================================================================

void OpenGl_Structure::Clear()
{
  Clear(GlDriver()->GetSharedContext());
}

//=================================================================================================

void OpenGl_Structure::Clear(const occ::handle<OpenGl_Context>& theGlCtx)
{
  bool aRaytracableGroupDeleted(false);

  // Release groups
  for (OpenGl_Structure::GroupIterator aGroupIter(myGroups); aGroupIter.More(); aGroupIter.Next())
  {
    aRaytracableGroupDeleted |= aGroupIter.Value()->IsRaytracable();

    // Delete objects
    aGroupIter.ChangeValue()->Release(theGlCtx);
  }
  myGroups.Clear();

  if (aRaytracableGroupDeleted)
  {
    myIsRaytracable = false;
  }

  Is2dText       = false;
  IsForHighlight = false;
}

//=================================================================================================

void OpenGl_Structure::renderGeometry(const occ::handle<OpenGl_Workspace>& theWorkspace,
                                      bool&                                theHasClosed) const
{
  if (myInstancedStructure != nullptr)
  {
    myInstancedStructure->renderGeometry(theWorkspace, theHasClosed);
  }

  bool                               anOldCastShadows = false;
  const occ::handle<OpenGl_Context>& aCtx             = theWorkspace->GetGlContext();
  for (OpenGl_Structure::GroupIterator aGroupIter(myGroups); aGroupIter.More(); aGroupIter.Next())
  {
    const OpenGl_Group* aGroup = aGroupIter.Value();

    const gp_Trsf& aTrsf = aGroup->Transformation();
    if (aTrsf.Form() != gp_Identity)
    {
      applyTransformation(aCtx, aTrsf, true);
    }

    const occ::handle<Graphic3d_TransformPers>& aTrsfPers = aGroup->TransformPersistence();
    if (!aTrsfPers.IsNull())
    {
      applyPersistence(aCtx, aTrsfPers, true, anOldCastShadows);
      aCtx->ApplyModelViewMatrix();
    }

    theHasClosed = theHasClosed || aGroup->IsClosed();
    aGroup->Render(theWorkspace);

    if (!aTrsfPers.IsNull())
    {
      revertPersistence(aCtx, aTrsfPers, true, anOldCastShadows);
      aCtx->ApplyModelViewMatrix();
    }

    if (aTrsf.Form() != gp_Identity)
    {
      applyTransformation(aCtx, aTrsf, false);
    }
  }
}

//=================================================================================================

void OpenGl_Structure::applyTransformation(const occ::handle<OpenGl_Context>& theContext,
                                           const gp_Trsf&                     theTrsf,
                                           const bool                         toEnable) const
{
  if (toEnable)
  {
    NCollection_Mat4<float> aTrsf;
    theTrsf.GetMat4(aTrsf);
    theContext->ModelWorldState.Push();
    NCollection_Mat4<float>& aModelWorld = theContext->ModelWorldState.ChangeCurrent();
    aModelWorld                          = aModelWorld * aTrsf;
    theContext->ApplyModelViewMatrix();
  }
  else
  {
    theContext->ModelWorldState.Pop();
    theContext->ApplyModelViewMatrix();
  }
}

//=================================================================================================

void OpenGl_Structure::Render(const occ::handle<OpenGl_Workspace>& theWorkspace) const
{
  // Process the structure only if visible
  if (!visible)
  {
    return;
  }

  const occ::handle<OpenGl_Context>& aCtx = theWorkspace->GetGlContext();

  // Render named status
  if (highlight && !myHighlightStyle.IsNull() && myHighlightStyle->Method() != Aspect_TOHM_BOUNDBOX)
  {
    theWorkspace->SetHighlightStyle(myHighlightStyle);
  }

  // Apply local transformation
  aCtx->ModelWorldState.Push();
  NCollection_Mat4<float>& aModelWorld = aCtx->ModelWorldState.ChangeCurrent();
  aModelWorld                          = myRenderTrsf;

  const bool anOldGlNormalize = aCtx->IsGlNormalizeEnabled();

  // detect scale transform
  if (aCtx->core11ffp != nullptr && !myTrsf.IsNull())
  {
    const double aScale = myTrsf->Trsf().ScaleFactor();
    if (std::abs(aScale - 1.0) > Precision::Confusion())
    {
      aCtx->SetGlNormalizeEnabled(true);
    }
  }

  bool anOldCastShadows = false;
#ifdef GL_DEPTH_CLAMP
  bool toRestoreDepthClamp = false;
#endif
  if (!myTrsfPers.IsNull())
  {
    applyPersistence(aCtx, myTrsfPers, false, anOldCastShadows);

#ifdef GL_DEPTH_CLAMP
    if (myTrsfPers->Mode() == Graphic3d_TMF_CameraPers && aCtx->arbDepthClamp)
    {
      toRestoreDepthClamp = true;
      aCtx->core11fwd->glEnable(GL_DEPTH_CLAMP);
    }
#endif
  }

  // Take into account transform persistence
  aCtx->ApplyModelViewMatrix();

  // remember aspects
  const OpenGl_Aspects* aPrevAspectFace = theWorkspace->Aspects();

  // Apply correction for mirror transform
  if (myIsMirrored)
  {
    aCtx->core11fwd->glFrontFace(GL_CW);
  }

  // Collect clipping planes of structure scope
  aCtx->ChangeClipping().SetLocalPlanes(myClipPlanes);

  // True if structure is fully clipped
  bool isClipped   = false;
  bool hasDisabled = false;
  if (aCtx->Clipping().IsClippingOrCappingOn())
  {
    const Graphic3d_BndBox3d& aBBox = BoundingBox();
    if (!myClipPlanes.IsNull() && myClipPlanes->ToOverrideGlobal())
    {
      aCtx->ChangeClipping().DisableGlobal();
      hasDisabled = aCtx->Clipping().HasDisabled();
    }
    else if (!myTrsfPers.IsNull())
    {
      if (myTrsfPers->IsZoomOrRotate())
      {
        // Zoom/rotate persistence object lives in two worlds at the same time.
        // Global clipping planes can not be trivially applied without being converted
        // into local space of transformation persistence object.
        // As more simple alternative - just clip entire object by its anchor point defined in the
        // world space.
        const gp_Pnt anAnchor = myTrsfPers->AnchorPoint();
        for (OpenGl_ClippingIterator aPlaneIt(aCtx->Clipping());
             aPlaneIt.More() && aPlaneIt.IsGlobal();
             aPlaneIt.Next())
        {
          const occ::handle<Graphic3d_ClipPlane>& aPlane = aPlaneIt.Value();
          if (!aPlane->IsOn())
          {
            continue;
          }

          // check for clipping
          const NCollection_Vec4<double> aCheckPnt(anAnchor.X(), anAnchor.Y(), anAnchor.Z(), 1.0);
          if (aPlane->ProbePoint(aCheckPnt) == Graphic3d_ClipState_Out)
          {
            isClipped = true;
            break;
          }
        }
      }

      aCtx->ChangeClipping().DisableGlobal();
      hasDisabled = aCtx->Clipping().HasDisabled();
    }

    // Set of clipping planes that do not intersect the structure,
    // and thus can be disabled to improve rendering performance
    if (aBBox.IsValid() && myTrsfPers.IsNull())
    {
      for (OpenGl_ClippingIterator aPlaneIt(aCtx->Clipping()); aPlaneIt.More(); aPlaneIt.Next())
      {
        const occ::handle<Graphic3d_ClipPlane>& aPlane = aPlaneIt.Value();
        if (aPlaneIt.IsDisabled())
        {
          continue;
        }

        const Graphic3d_ClipState aBoxState = aPlane->ProbeBox(aBBox);
        if (aBoxState == Graphic3d_ClipState_In)
        {
          aCtx->ChangeClipping().SetEnabled(aPlaneIt, false);
          hasDisabled = true;
        }
        else if (aBoxState == Graphic3d_ClipState_Out && myBndBoxClipCheck)
        {
          isClipped = true;
          break;
        }
      }
    }

    if ((!myClipPlanes.IsNull() && !myClipPlanes->IsEmpty()) || hasDisabled)
    {
      // Set OCCT state uniform variables
      aCtx->ShaderManager()->UpdateClippingState();
    }
  }

  // Render groups
  bool hasClosedPrims = false;
  if (!isClipped)
  {
    renderGeometry(theWorkspace, hasClosedPrims);
  }

  // Reset correction for mirror transform
  if (myIsMirrored)
  {
    aCtx->core11fwd->glFrontFace(GL_CCW);
  }

  // Render capping for structure groups
  if (hasClosedPrims && aCtx->Clipping().IsCappingOn())
  {
    OpenGl_CappingAlgo::RenderCapping(theWorkspace, *this);
  }

  // Revert structure clippings
  if (hasDisabled)
  {
    // enable planes that were previously disabled
    aCtx->ChangeClipping().RestoreDisabled();
  }
  aCtx->ChangeClipping().SetLocalPlanes(occ::handle<Graphic3d_SequenceOfHClipPlane>());
  if ((!myClipPlanes.IsNull() && !myClipPlanes->IsEmpty()) || hasDisabled)
  {
    // Set OCCT state uniform variables
    aCtx->ShaderManager()->RevertClippingState();
  }

  // Restore local transformation
  aCtx->ModelWorldState.Pop();
  aCtx->SetGlNormalizeEnabled(anOldGlNormalize);

  // Restore aspects
  theWorkspace->SetAspects(aPrevAspectFace);

  // Apply highlight box
  if (!isClipped && !myHighlightStyle.IsNull()
      && myHighlightStyle->Method() == Aspect_TOHM_BOUNDBOX)
  {
    aCtx->ApplyModelViewMatrix();
    theWorkspace->SetHighlightStyle(myHighlightStyle);
    renderBoundingBox(theWorkspace);
  }

  if (!myTrsfPers.IsNull())
  {
    revertPersistence(aCtx, myTrsfPers, false, anOldCastShadows);
#ifdef GL_DEPTH_CLAMP
    if (toRestoreDepthClamp)
    {
      aCtx->core11fwd->glDisable(GL_DEPTH_CLAMP);
    }
#endif
  }

  // Restore named status
  theWorkspace->SetHighlightStyle(occ::handle<Graphic3d_PresentationAttributes>());
}

//=================================================================================================

void OpenGl_Structure::Release(const occ::handle<OpenGl_Context>& theGlCtx)
{
  // Release groups
  Clear(theGlCtx);
  myHighlightStyle.Nullify();
}

//=================================================================================================

void OpenGl_Structure::ReleaseGlResources(const occ::handle<OpenGl_Context>& theGlCtx)
{
  for (OpenGl_Structure::GroupIterator aGroupIter(myGroups); aGroupIter.More(); aGroupIter.Next())
  {
    aGroupIter.ChangeValue()->Release(theGlCtx);
  }
}

//=================================================================================================

occ::handle<Graphic3d_CStructure> OpenGl_Structure::ShadowLink(
  const occ::handle<Graphic3d_StructureManager>& theManager) const
{
  return new OpenGl_StructureShadow(theManager, this);
}

//=================================================================================================

void OpenGl_Structure::applyPersistence(const occ::handle<OpenGl_Context>&          theCtx,
                                        const occ::handle<Graphic3d_TransformPers>& theTrsfPers,
                                        const bool                                  theIsLocal,
                                        bool& theOldCastShadows) const
{
  // temporarily disable shadows on non-3d objects
  theOldCastShadows = theCtx->ShaderManager()->SetCastShadows(false);

  theCtx->WorldViewState.Push();
  NCollection_Mat4<float>& aWorldView = theCtx->WorldViewState.ChangeCurrent();

  if (theIsLocal && theTrsfPers->IsZoomOrRotate())
  {
    // move anchor point to presentation location
    theCtx->ModelWorldState.Push();
    NCollection_Mat4<float>& aModelWorld   = theCtx->ModelWorldState.ChangeCurrent();
    gp_Pnt                   aStartPnt     = theTrsfPers->AnchorPoint();
    NCollection_Vec4<float>  anAnchorPoint = aModelWorld
                                            * NCollection_Vec4<float>((float)aStartPnt.X(),
                                                                      (float)aStartPnt.Y(),
                                                                      (float)aStartPnt.Z(),
                                                                      1.0f);
    // clang-format off
    aModelWorld.SetColumn (3, NCollection_Vec4<float> (NCollection_Vec3<float> (0.0), 1.0)); // reset translation part
    // clang-format on
    aStartPnt.SetCoord(anAnchorPoint.x(), anAnchorPoint.y(), anAnchorPoint.z());

    theTrsfPers->Apply(theCtx->Camera(),
                       theCtx->ProjectionState.Current(),
                       aWorldView,
                       theCtx->VirtualViewport()[2],
                       theCtx->VirtualViewport()[3],
                       &aStartPnt);
  }
  else
  {
    theTrsfPers->Apply(theCtx->Camera(),
                       theCtx->ProjectionState.Current(),
                       aWorldView,
                       theCtx->VirtualViewport()[2],
                       theCtx->VirtualViewport()[3]);
  }

  if (!theCtx->IsGlNormalizeEnabled() && theCtx->core11ffp != nullptr)
  {
    const double aScale = Graphic3d_TransformUtils::ScaleFactor(aWorldView);
    if (std::abs(aScale - 1.0) > Precision::Confusion())
    {
      theCtx->SetGlNormalizeEnabled(true);
    }
  }
}

//=================================================================================================

void OpenGl_Structure::revertPersistence(const occ::handle<OpenGl_Context>&          theCtx,
                                         const occ::handle<Graphic3d_TransformPers>& theTrsfPers,
                                         const bool                                  theIsLocal,
                                         const bool theOldCastShadows) const
{
  if (theIsLocal && theTrsfPers->IsZoomOrRotate())
  {
    theCtx->ModelWorldState.Pop();
  }
  theCtx->WorldViewState.Pop();
  theCtx->ShaderManager()->SetCastShadows(theOldCastShadows);
}

//=================================================================================================

void OpenGl_Structure::DumpJson(Standard_OStream& theOStream, int theDepth) const
{
  OCCT_DUMP_TRANSIENT_CLASS_BEGIN(theOStream)

  OCCT_DUMP_BASE_CLASS(theOStream, theDepth, Graphic3d_CStructure)

  OCCT_DUMP_FIELD_VALUE_POINTER(theOStream, myInstancedStructure)

  OCCT_DUMP_FIELD_VALUE_NUMERICAL(theOStream, myIsRaytracable)
  OCCT_DUMP_FIELD_VALUE_NUMERICAL(theOStream, myModificationState)
  OCCT_DUMP_FIELD_VALUE_NUMERICAL(theOStream, myIsMirrored)
}
