// Created on: 1996-12-30
// Created by: Stagiaire Mary FABIEN
// Copyright (c) 1996-1999 Matra Datavision
// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#ifndef _BRepTools_GTrsfModification_HeaderFile
#define _BRepTools_GTrsfModification_HeaderFile

#include <Standard.hxx>
#include <Standard_Type.hxx>

#include <gp_GTrsf.hxx>
#include <BRepTools_Modification.hxx>
#include <GeomAbs_Shape.hxx>
class TopoDS_Face;
class Geom_Surface;
class TopLoc_Location;
class TopoDS_Edge;
class Geom_Curve;
class TopoDS_Vertex;
class gp_Pnt;
class Geom2d_Curve;

//! Defines a modification of the geometry by a GTrsf
//! from gp. All methods return True and transform the
//! geometry.
class BRepTools_GTrsfModification : public BRepTools_Modification
{

public:
  Standard_EXPORT BRepTools_GTrsfModification(const gp_GTrsf& T);

  //! Gives an access on the GTrsf.
  Standard_EXPORT gp_GTrsf& GTrsf();

  //! Returns true if the face <F> has been
  //! modified. In this case, <S> is the new geometric
  //! support of the face, <L> the new location,<Tol>
  //! the new tolerance.<RevWires> has to be set to
  //! true when the modification reverses the
  //! normal of the surface. (the wires have to be
  //! reversed). <RevFace> has to be set to
  //! true if the orientation of the modified
  //! face changes in the shells which contain it.
  //! Here, <RevFace> will return true if the
  //! -- gp_Trsf is negative.
  Standard_EXPORT bool NewSurface(const TopoDS_Face&         F,
                                  occ::handle<Geom_Surface>& S,
                                  TopLoc_Location&           L,
                                  double&                    Tol,
                                  bool&                      RevWires,
                                  bool&                      RevFace) override;

  //! Returns true if the edge <E> has been
  //! modified. In this case, <C> is the new geometric
  //! support of the edge, <L> the new location, <Tol>
  //! the new tolerance. Otherwise, returns
  //! false, and <C>, <L>, <Tol> are not
  //! significant.
  Standard_EXPORT bool NewCurve(const TopoDS_Edge&       E,
                                occ::handle<Geom_Curve>& C,
                                TopLoc_Location&         L,
                                double&                  Tol) override;

  //! Returns true if the vertex <V> has been
  //! modified. In this case, <P> is the new geometric
  //! support of the vertex, <Tol> the new tolerance.
  //! Otherwise, returns false, and <P>, <Tol>
  //! are not significant.
  Standard_EXPORT bool NewPoint(const TopoDS_Vertex& V, gp_Pnt& P, double& Tol) override;

  //! Returns true if the edge <E> has a new
  //! curve on surface on the face <F>.In this case, <C>
  //! is the new geometric support of the edge, <L> the
  //! new location, <Tol> the new tolerance.
  //! Otherwise, returns false, and <C>, <L>,
  //! <Tol> are not significant.
  Standard_EXPORT bool NewCurve2d(const TopoDS_Edge&         E,
                                  const TopoDS_Face&         F,
                                  const TopoDS_Edge&         NewE,
                                  const TopoDS_Face&         NewF,
                                  occ::handle<Geom2d_Curve>& C,
                                  double&                    Tol) override;

  //! Returns true if the Vertex <V> has a new
  //! parameter on the edge <E>. In this case, <P> is
  //! the parameter, <Tol> the new tolerance.
  //! Otherwise, returns false, and <P>, <Tol>
  //! are not significant.
  Standard_EXPORT bool NewParameter(const TopoDS_Vertex& V,
                                    const TopoDS_Edge&   E,
                                    double&              P,
                                    double&              Tol) override;

  //! Returns the continuity of <NewE> between <NewF1>
  //! and <NewF2>.
  //!
  //! <NewE> is the new edge created from <E>. <NewF1>
  //! (resp. <NewF2>) is the new face created from <F1>
  //! (resp. <F2>).
  Standard_EXPORT GeomAbs_Shape Continuity(const TopoDS_Edge& E,
                                           const TopoDS_Face& F1,
                                           const TopoDS_Face& F2,
                                           const TopoDS_Edge& NewE,
                                           const TopoDS_Face& NewF1,
                                           const TopoDS_Face& NewF2) override;

  //! Returns true if the face has been modified according to changed triangulation.
  //! If the face has been modified:
  //! - theTri is a new triangulation on the face
  Standard_EXPORT bool NewTriangulation(const TopoDS_Face&               theFace,
                                        occ::handle<Poly_Triangulation>& theTri) override;

  //! Returns true if the edge has been modified according to changed polygon.
  //! If the edge has been modified:
  //! - thePoly is a new polygon
  Standard_EXPORT bool NewPolygon(const TopoDS_Edge&           theEdge,
                                  occ::handle<Poly_Polygon3D>& thePoly) override;

  //! Returns true if the edge has been modified according to changed polygon on triangulation.
  //! If the edge has been modified:
  //! - thePoly is a new polygon on triangulation
  Standard_EXPORT bool NewPolygonOnTriangulation(
    const TopoDS_Edge&                        theEdge,
    const TopoDS_Face&                        theFace,
    occ::handle<Poly_PolygonOnTriangulation>& thePoly) override;

  DEFINE_STANDARD_RTTIEXT(BRepTools_GTrsfModification, BRepTools_Modification)

private:
  gp_GTrsf myGTrsf;
  double   myGScale;
};

#endif // _BRepTools_GTrsfModification_HeaderFile
