// Created on: 1996-12-16
// Created by: Bruno DUMORTIER
// Copyright (c) 1996-1999 Matra Datavision
// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#ifndef _BiTgte_Blend_HeaderFile
#define _BiTgte_Blend_HeaderFile

#include <Standard.hxx>
#include <Standard_DefineAlloc.hxx>
#include <Standard_Handle.hxx>

#include <TopoDS_Shape.hxx>
#include <NCollection_List.hxx>
#include <TopTools_ShapeMapHasher.hxx>
#include <NCollection_DataMap.hxx>
#include <NCollection_IndexedMap.hxx>
#include <BRepOffset_Analyse.hxx>
#include <BRepOffset_Offset.hxx>
#include <BRepAlgo_Image.hxx>
#include <Standard_Integer.hxx>
#include <NCollection_Array1.hxx>
#include <NCollection_HArray1.hxx>
#include <BiTgte_ContactType.hxx>
#include <Bnd_Box.hxx>
class BRepAlgo_AsDes;
class TopoDS_Face;
class TopoDS_Edge;
class Geom_Surface;
class Geom_Curve;
class Geom2d_Curve;
class BRepOffset_Offset;
class BRepOffset_Inter3d;

//! Root class
class BiTgte_Blend
{
public:
  DEFINE_STANDARD_ALLOC

  Standard_EXPORT BiTgte_Blend();

  //! <S>: Shape to be rounded
  //! <Radius>: radius of the fillet
  //! <Tol>: Tol3d used in approximations
  //! <NUBS>: if true, generate only NUBS surfaces,
  //! if false, generate analytical surfaces if possible
  Standard_EXPORT BiTgte_Blend(const TopoDS_Shape& S,
                               const double        Radius,
                               const double        Tol,
                               const bool          NUBS);

  Standard_EXPORT void Init(const TopoDS_Shape& S,
                            const double        Radius,
                            const double        Tol,
                            const bool          NUBS);

  //! Clear all the Fields.
  Standard_EXPORT void Clear();

  //! Set two faces of <myShape> on which the Sphere
  //! must roll.
  Standard_EXPORT void SetFaces(const TopoDS_Face& F1, const TopoDS_Face& F2);

  //! Set an edge of <myShape> to be rounded.
  Standard_EXPORT void SetEdge(const TopoDS_Edge& Edge);

  //! Set a face on which the fillet must stop.
  Standard_EXPORT void SetStoppingFace(const TopoDS_Face& Face);

  //! Compute the generated surfaces.
  //! If <BuildShape> is true, compute the resulting Shape.
  //! If false, only the blending surfaces are computed.
  Standard_EXPORT void Perform(const bool BuildShape = true);

  Standard_EXPORT bool IsDone() const;

  //! returns the result
  Standard_EXPORT const TopoDS_Shape& Shape() const;

  //! returns the Number of generated surfaces.
  Standard_EXPORT int NbSurfaces() const;

  //! returns the surface of range Index
  Standard_EXPORT occ::handle<Geom_Surface> Surface(const int Index) const;

  //! returns the surface of range Index
  Standard_EXPORT const TopoDS_Face& Face(const int Index) const;

  //! set in <LC> all the center lines
  Standard_EXPORT void CenterLines(NCollection_List<TopoDS_Shape>& LC) const;

  //! returns the surface generated by the centerline.
  //! <CenterLine> may be
  //! - an edge  : generate a pipe.
  //! - a vertex : generate a sphere.
  //! Warning: returns a Null Handle if <CenterLine> generates
  //! no surface.
  Standard_EXPORT occ::handle<Geom_Surface> Surface(const TopoDS_Shape& CenterLine) const;

  //! returns the face generated by the centerline.
  //! <CenterLine> may be
  //! - an edge  : generate a pipe.
  //! - a vertex : generate a sphere.
  //! Warning: returns a Null Shape if <CenterLine> generates
  //! no surface.
  Standard_EXPORT const TopoDS_Face& Face(const TopoDS_Shape& CenterLine) const;

  //! returns the type of contact
  Standard_EXPORT BiTgte_ContactType ContactType(const int Index) const;

  //! gives the first support shape relative to
  //! SurfaceFillet(Index);
  Standard_EXPORT const TopoDS_Shape& SupportShape1(const int Index) const;

  //! gives the second support shape relative to
  //! SurfaceFillet(Index);
  Standard_EXPORT const TopoDS_Shape& SupportShape2(const int Index) const;

  //! gives the 3d curve of SurfaceFillet(Index)
  //! on SupportShape1(Index)
  Standard_EXPORT occ::handle<Geom_Curve> CurveOnShape1(const int Index) const;

  //! gives the 3d curve of SurfaceFillet(Index)
  //! on SupportShape2(Index)
  Standard_EXPORT occ::handle<Geom_Curve> CurveOnShape2(const int Index) const;

  //! gives the PCurve associated to CurvOnShape1(Index)
  //! on the support face
  //! Warning: returns a Null Handle if SupportShape1 is not a Face
  Standard_EXPORT occ::handle<Geom2d_Curve> PCurveOnFace1(const int Index) const;

  //! gives the PCurve associated to CurveOnShape1(Index)
  //! on the Fillet
  Standard_EXPORT occ::handle<Geom2d_Curve> PCurve1OnFillet(const int Index) const;

  //! gives the PCurve associated to CurveOnShape2(Index)
  //! on the support face
  //! Warning: returns a Null Handle if SupportShape2 is not a Face
  Standard_EXPORT occ::handle<Geom2d_Curve> PCurveOnFace2(const int Index) const;

  //! gives the PCurve associated to CurveOnShape2(Index)
  //! on the fillet
  Standard_EXPORT occ::handle<Geom2d_Curve> PCurve2OnFillet(const int Index) const;

  Standard_EXPORT int NbBranches();

  //! Set in <From>,<To> the indices of the faces of
  //! the branche <Index>.
  //!
  //! i.e: Branche<Index> = Face(From) + Face(From+1) + ..+ Face(To)
  Standard_EXPORT void IndicesOfBranche(const int Index, int& From, int& To) const;

  //! Computes the center lines
  Standard_EXPORT void ComputeCenters();

private:
  //! Perform the generated surfaces.
  Standard_EXPORT void ComputeSurfaces();

  //! Build the resulting shape
  //! All the faces must be computed
  Standard_EXPORT void ComputeShape();

  //! Computes the intersections with <Face> and all the
  //! OffsetFaces stored in <myMapSF>. Returns <True>
  //! if an intersections ends on a boundary of a Face.
  Standard_EXPORT bool Intersect(
    const TopoDS_Shape&                                                        Init,
    const TopoDS_Face&                                                         Face,
    const NCollection_DataMap<TopoDS_Shape, Bnd_Box, TopTools_ShapeMapHasher>& MapSBox,
    const BRepOffset_Offset&                                                   OF1,
    BRepOffset_Inter3d&                                                        Inter);

  double       myRadius;
  double       myTol;
  bool         myNubs;
  TopoDS_Shape myShape;
  TopoDS_Shape myResult;
  bool         myBuildShape;
  NCollection_IndexedDataMap<TopoDS_Shape, NCollection_List<TopoDS_Shape>, TopTools_ShapeMapHasher>
    myAncestors;
  NCollection_DataMap<
    TopoDS_Shape,
    NCollection_DataMap<TopoDS_Shape, NCollection_List<TopoDS_Shape>, TopTools_ShapeMapHasher>,
    TopTools_ShapeMapHasher>
    myCreated;
  NCollection_DataMap<TopoDS_Shape, NCollection_List<TopoDS_Shape>, TopTools_ShapeMapHasher>
                                                                                myCutEdges;
  NCollection_IndexedMap<TopoDS_Shape, TopTools_ShapeMapHasher>                 myFaces;
  NCollection_IndexedMap<TopoDS_Shape, TopTools_ShapeMapHasher>                 myEdges;
  NCollection_Map<TopoDS_Shape, TopTools_ShapeMapHasher>                        myStopFaces;
  BRepOffset_Analyse                                                            myAnalyse;
  NCollection_IndexedMap<TopoDS_Shape, TopTools_ShapeMapHasher>                 myCenters;
  NCollection_DataMap<TopoDS_Shape, BRepOffset_Offset, TopTools_ShapeMapHasher> myMapSF;
  BRepAlgo_Image                                                                myInitOffsetFace;
  BRepAlgo_Image                                                                myImage;
  BRepAlgo_Image                                                                myImageOffset;
  occ::handle<BRepAlgo_AsDes>                                                   myAsDes;
  int                                                                           myNbBranches;
  occ::handle<NCollection_HArray1<int>>                                         myIndices;
  bool                                                                          myDone;
};

#endif // _BiTgte_Blend_HeaderFile
