// Created on: 2011-06-02
// Created by: Oleg AGASHIN
// Copyright (c) 2011-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <BRepMesh_VertexTool.hxx>
#include <Precision.hxx>

IMPLEMENT_STANDARD_RTTIEXT(BRepMesh_VertexTool, Standard_Transient)

//=================================================================================================

NCollection_CellFilter_Action BRepMesh_VertexInspector::Inspect(const int theTarget)
{
  const BRepMesh_Vertex& aVertex = myVertices->Value(theTarget - 1);
  if (aVertex.Movability() == BRepMesh_Deleted)
  {
    myDelNodes.Append(theTarget);
    return CellFilter_Purge;
  }

  gp_XY aVec = (myPoint - aVertex.Coord());
  bool  inTol;
  if (std::abs(myTolerance[1]) < Precision::Confusion())
  {
    inTol = aVec.SquareModulus() < myTolerance[0];
  }
  else
  {
    inTol = ((aVec.X() * aVec.X()) < myTolerance[0]) && ((aVec.Y() * aVec.Y()) < myTolerance[1]);
  }

  if (inTol)
  {
    const double aSqDist = aVec.SquareModulus();
    if (aSqDist < myMinSqDist)
    {
      myMinSqDist = aSqDist;
      myIndex     = theTarget;
    }
  }

  return CellFilter_Keep;
}

//=================================================================================================

BRepMesh_VertexTool::BRepMesh_VertexTool(const occ::handle<NCollection_IncAllocator>& theAllocator)
    : myAllocator(theAllocator),
      myCellFilter(0., myAllocator),
      mySelector(myAllocator)
{
  constexpr double aTol = Precision::Confusion();
  SetCellSize(aTol + 0.05 * aTol);
  SetTolerance(aTol, aTol);
}

//=================================================================================================

int BRepMesh_VertexTool::Add(const BRepMesh_Vertex& theVertex, const bool isForceAdd)
{
  int aIndex = isForceAdd ? 0 : FindIndex(theVertex);
  if (aIndex == 0)
  {
    aIndex = mySelector.Add(theVertex);

    gp_XY aMinPnt, aMaxPnt;
    expandPoint(theVertex.Coord(), aMinPnt, aMaxPnt);
    myCellFilter.Add(aIndex, aMinPnt, aMaxPnt);
  }
  return aIndex;
}

//=================================================================================================

void BRepMesh_VertexTool::DeleteVertex(const int theIndex)
{
  BRepMesh_Vertex& aV = mySelector.GetVertex(theIndex);

  gp_XY aMinPnt, aMaxPnt;
  expandPoint(aV.Coord(), aMinPnt, aMaxPnt);

  myCellFilter.Remove(theIndex, aMinPnt, aMaxPnt);
  mySelector.Delete(theIndex);
}

//=================================================================================================

void BRepMesh_VertexTool::Substitute(const int theIndex, const BRepMesh_Vertex& theVertex)
{
  BRepMesh_Vertex& aV = mySelector.GetVertex(theIndex);

  gp_XY aMinPnt, aMaxPnt;
  expandPoint(aV.Coord(), aMinPnt, aMaxPnt);

  myCellFilter.Remove(theIndex, aMinPnt, aMaxPnt);

  aV = theVertex;
  expandPoint(aV.Coord(), aMinPnt, aMaxPnt);
  myCellFilter.Add(theIndex, aMinPnt, aMaxPnt);
}

//=================================================================================================

void BRepMesh_VertexTool::Statistics(Standard_OStream& theStream) const
{
  theStream << "\nStructure Statistics\n---------------\n\n";
  theStream << "This structure has " << mySelector.NbVertices() << " Nodes\n\n";
}
